/************************************************************************
 * Copyright (c) 2005, 2010 Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *    Viacheslav Rybalov, Intel - Initial API and implementation
 *    Vsevolod Sandomirskiy, OC Systems - support variable length options 
 *
 * $Id: ACCollector.cpp,v 1.34 2010/12/29 18:58:43 jcayne Exp $  
 ************************************************************************/

#include <stdlib.h>

#include "ACCollector.h"


#ifndef _WIN32
	#include <unistd.h>
#endif

using namespace std;
using namespace Martini::ACCollector;

/////////////////////////////////////////////////////////////////

Martini::OSA::IThreadSync* lockObject = Martini::OSA::CreateThreadSync();

/////////////////////////////////////////////////////////////////

extern "C" API_EXPORT int ACCollector_Init(ACC_ENV* env, bool isControlled)
{
    return ACCollector::GetInstance().Init(env, isControlled);
}

extern "C" API_EXPORT int ACCollector_SendData(char *s, unsigned short length)
{
    return ACCollector::GetInstance().sendData_(s, length);
}

extern "C" API_EXPORT int ACCollector_SendVMAgentInitializedCommand()
{
    return ACCollector::GetInstance().SendVMAgentInitializedCommand();
}

/////////////////////////////////////////////////////////////////

// implementation util

// caller must free output name and value
static
void parseOption( const char* param_name, const char* param_value,
				  char*& name, char*& value )
{
	char NAME_START[] = "<name>";
    char NAME_END[] = "</name>";
    char VALUE_START[] = "<value>";
    char VALUE_END[] = "</value>";
            	
    #define SSIZE(str) (sizeof(str) - 1)            	
	#define BAD_OPTION(msg) { LOG_ERROR(msg); return; }
            	
    // reset output
    name = 0;
    value = 0;
            	
	// parse string            	
    size_t len = strlen(param_value);
    if( len < SSIZE(NAME_START) + SSIZE(NAME_END) + 
    		  SSIZE(VALUE_START) + SSIZE(VALUE_END) )
    	BAD_OPTION("value too short");
            		
	const char* ns = strstr( param_value, NAME_START);
    if( !ns ) BAD_OPTION("invalid syntax");  
    ns += SSIZE(NAME_START);
            	          		
	const char* ne = strstr( ns, NAME_END);
    if( !ne ) BAD_OPTION("invalid syntax");
            	
    const char* vs = strstr( ne, VALUE_START);
    if( !vs ) BAD_OPTION("invalid syntax");
    vs += SSIZE(VALUE_START);
            	
    /* string must end with </value> */
    const char* ve = param_value + len - SSIZE(VALUE_END); 
    if( ve <= vs ) BAD_OPTION("invalid syntax");
    if( strstr( ve, VALUE_END ) != ve  ) 
    	BAD_OPTION("invalid syntax: option doesn't end with \"</value>\"");
            	
    /* name */
    len = ne - ns;
    name = (char*)malloc( len + 1 );
	if( name == 0 )
		return;
    strncpy( name, ns, len );
    name[ len ] = 0;
            	
    /* value */
    len = ve - vs;
    value = (char*)malloc( len + 1 );
	if( value == 0 )
		return;
    strncpy( value, vs, len );
    value[ len ] = 0;
    
    return;
}

/////////////////////////////////////////////////////////////////

#ifdef MVS
	static char* profilerName () {
		char * profileName = "org.eclipse.tptp.jvmti";
		char * nativeBuffer;
		Martini::OSA::unicode2native(&nativeBuffer, profileName, strlen(profileName));
		return nativeBuffer;
	}
	static char* profileName = profilerName();
#else
	static char* profileName = "org.eclipse.tptp.jvmti";
#endif

ACCollector::ACCollector() : BaseCollectorImpl(profileName), BaseAgentImpl(profileName)
{    
    m_dataListenerID = 0; 
    m_AgentID = 0;
    m_isWaiting = true;
    m_isTracing = false;
    m_ACC_env = 0;
}

ACCollector::~ACCollector()
{
}

ACCollector& 
ACCollector::GetInstance()
{
    static ACCollector acCollector;
    return acCollector;
}

int 
ACCollector::run(CmdBlock* cmdBlock)
{
    if (m_isTracing) {
        SendWrongPhaseError(cmdBlock->getContextID());
        return 0;
    }
    
    JNIEnv* unusedEnv;
    bool    unusedBool;
    m_ACC_env->openSession( &unusedEnv, &unusedBool );	// to reduce number of attachments to JVM, bugzilla 201408  

    if (m_isEnabledDataCollection) {
        m_ACC_env->Attach(); 
        m_isWaiting = false;
        m_ACC_env->StartTracing();
        m_isTracing = true;
    }
    else {
        m_ACC_env->Attach(); 
        m_isWaiting = false;
        m_isTracing = true;
    }
//    if (m_ACC_env->IsAgentInitialized()) {
//        SendVMAgentInitializedCommand();
//    }

    m_ACC_env->closeSession(); 

    return BaseCollectorImpl::run(cmdBlock);
}

int 
ACCollector::stop(CmdBlock* cmdBlock)
{
    if (!m_isTracing) {
        SendWrongPhaseError(cmdBlock->getContextID());
        return 0;
    }
    if (!isAgentInitialized(cmdBlock->getContextID())) {
        return 0;
    }
    m_ACC_env->StopTracing();
    m_ACC_env->Detach();
    m_isTracing = false;
    return BaseCollectorImpl::stop(cmdBlock);
}

int 
ACCollector::resume(CmdBlock* cmdBlock)
{
    if (!m_isTracing) {
        SendWrongPhaseError(cmdBlock->getContextID());
        return 0;
    }

    m_ACC_env->StartTracing(); 

    return BaseCollectorImpl::resume(cmdBlock);
}

int 
ACCollector::pause(CmdBlock* cmdBlock) {
    if (!m_isTracing) {
        SendWrongPhaseError(cmdBlock->getContextID());
        return 0;
    }
    if (!isAgentInitialized(cmdBlock->getContextID())) {
        return 0;
    }

    m_ACC_env->StopTracing();

    return BaseCollectorImpl::pause(cmdBlock);
}

int 
ACCollector::establishDataPath(int dataConnectionID, int dataChanneltype, int destinationID, int contextID)
{
    return BaseCollectorImpl::establishDataPath(dataConnectionID, dataChanneltype, destinationID, contextID);
}

int 
ACCollector::releaseDataPath(int dataConnectionID)
{
    return BaseCollectorImpl::releaseDataPath(dataConnectionID);
}

#ifdef MVS
CmdBlock* ACCollector::native2unicodeCmdBlock(CmdBlock* cmdBlock)
{
		CmdBlock* cmd = new CmdBlock();

		char* nativeIID = cmdBlock->getIID();
		char* interfaceName = 0;

		// Convert IID, store it in interfaceName
		Martini::OSA::native2unicode(&interfaceName, nativeIID, strlen(nativeIID));

		char* nativeCmdName = cmdBlock->getCommandName();
		char* unicodeCmdName = 0;
		// Convert Command Name
		Martini::OSA::native2unicode(&unicodeCmdName, nativeCmdName, strlen(nativeCmdName));

		// Set the integer values, plus those we've converted above
		cmd->setSourceID(cmdBlock->getSourceID());
		cmd->setDestID(cmdBlock->getDestID());
		cmd->setContextID(cmdBlock->getContextID());
		cmd->setIID(interfaceName);
		cmd->setCommandName(unicodeCmdName);

		// When setCommandName(...)/setIID(...) are called, a copy is made inside the setter, so it is ok to free them here.
		if(unicodeCmdName != 0)
			free(unicodeCmdName);

		if(interfaceName != 0)
			free(interfaceName);

		// Convert the parameeters in the cmdBlock's parameter list
//		tptp_list_t* unicodeParamList = (tptp_list_t*)malloc(sizeof(tptp_list_t));
//		tptp_list_init(unicodeParamList);
		tptp_list_t* paramList = cmdBlock->getParamList();
        _tptp_node_t* node = paramList->head;
        for (int i = 0; i < paramList->count; i++) {
            tptp_param_t* param = (tptp_param_t*)(node->data);

            node = node->next;

        	char* unicodeName = 0;
        	char* unicodeValue = 0;

        	Martini::OSA::native2unicode(&unicodeName, param->name, strlen(param->name));
        	Martini::OSA::native2unicode(&unicodeValue, param->value, strlen(param->value));

        	tptp_param_t* unicodeParam = initParamT(unicodeName, unicodeValue);
        	//tptp_list_add(unicodeParamList, unicodeParam );

        	tptp_list_add(cmd->getParamList(), unicodeParam );

        	// initParamT copies the string values, so it is ok to free them here
        	if(unicodeName != 0)
        		free(unicodeName);

        	if(unicodeValue != 0)
        		free(unicodeValue);
		}
		//cmd->setParamList(unicodeParamList);

		return cmd;
}
#endif

ACCollector::Agentlet* 
ACCollector::lookupAgentlet( JNIEnv* jni, const char* iid, const char* message )
{
	// Try the lookup
	std::string key = std::string(iid) + message;
	std::map<std::string,Agentlet*>::iterator match = m_agentlets.find( key );
	if( m_agentlets.end() != match )
		return (*match).second;

	// No match, query the system property
	jclass           sysCls = jni->FindClass("java/lang/System");
	jmethodID getPropertyID = jni->GetStaticMethodID(sysCls, "getProperty",
	                                                 "(Ljava/lang/String;)Ljava/lang/String;");
	
	std::ostringstream propertyNameBuf;
	propertyNameBuf
		<< ACCOLLECTOR_AGENTLET_EXT << ":"
		<< iid << "#" << message;
	
	std::string propertyName = propertyNameBuf.str();
	jstring jpropertyName = jni->NewStringUTF(propertyName.c_str());
	jstring jagentletName = (jstring)jni->CallStaticObjectMethod(sysCls,
	                                                             getPropertyID,
	                                                             jpropertyName);
	if( !jagentletName )
		return NULL;

	// Replace '.'s with '/'s
	const char* agentletName = jni->GetStringUTFChars(jagentletName,
	                                                  NULL);
	char* jvmAgentletName = new char[strlen(agentletName)+1];
	strcpy( jvmAgentletName, agentletName );
	while( NULL != strchr(jvmAgentletName, '.') )
		*(strchr(jvmAgentletName, '.')) = '/';
	
	jclass agentletCls = jni->FindClass(jvmAgentletName);

	jni->ReleaseStringUTFChars(jagentletName, agentletName);
	delete[] jvmAgentletName;

	if( !agentletCls ) 
		return NULL;

	jmethodID handlerID = jni->GetStaticMethodID(agentletCls, 
	                                             ACCOLLECTOR_AGENTLET_HANDLER,
	                                             ACCOLLECTOR_AGENTLET_HANDLER_SIGNATURE);

	if( !handlerID )
		return NULL;


	Agentlet* agentlet = new Agentlet( iid, message, agentletCls, handlerID );
	m_agentlets.insert( std::pair<std::string,Agentlet*>(key, agentlet) );

	return agentlet;

}

jobject
ACCollector::buildParameterMap( JNIEnv* jni, tptp_list_t* paramList ) {

	// Allocate parameter list
	jclass mapCls = jni->FindClass("java/util/HashMap");
	jmethodID mapCid = jni->GetMethodID(mapCls, "<init>", "()V");
	jmethodID mapPut = jni->GetMethodID(mapCls, "put", "(Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");

	if( !mapCls || !mapCid || !mapPut )
		return NULL;

	jobject jparams  = jni->NewObject(mapCls, mapCid);
	if( !jparams )
		return NULL;

	// Build up the parameter list
	_tptp_node_t* node = paramList->head;
	for (int i=0; i<paramList->count; i++, node=node->next) {

		tptp_param_t* param = (tptp_param_t*)(node->data);
		jstring name = jni->NewStringUTF( param->name );
		jstring value = jni->NewStringUTF( param->value );
		
		jni->CallObjectMethod( jparams, mapPut, name, value );
		
	}

	return jparams;

}

const char* 
ACCollector::dispatchAgentlet( CmdBlock* cmdBlock ) {

	char* result = NULL;

	JNIEnv* jni;
	bool    attached;

	m_ACC_env->openSession(&jni, &attached);
	if( !jni )
		return NULL;

	Agentlet* agentlet = lookupAgentlet( jni, cmdBlock->getIID(), cmdBlock->getCommandName() );
	if( !agentlet ) {
		m_ACC_env->closeSession();
		return NULL;
	}

	jstring jiid = jni->NewStringUTF( cmdBlock->getIID() );
	jstring jcmd = jni->NewStringUTF( cmdBlock->getCommandName() );
	jobject jparams = buildParameterMap( jni, cmdBlock->getParamList() );
	if( !jiid || !jcmd || !jparams ) {
		m_ACC_env->closeSession();
		return NULL;
	}

	jstring jresult = (jstring)jni->CallStaticObjectMethod(agentlet->cls, 
	                                                       agentlet->handler,
	                                                       jiid,
	                                                       jcmd, 
	                                                       jparams);

	if( jresult ) {
		const char* resultTmp = jni->GetStringUTFChars(jresult, NULL);
		result = new char[ strlen(resultTmp)+1 ];
		strcpy( result, resultTmp );
		jni->ReleaseStringUTFChars( jresult, resultTmp );
	}

	m_ACC_env->closeSession();
	return result;

}

int 
ACCollector::processCommand(CmdBlock* cmdBlock)
{
    LOG_TRACE("AC command: SourceID:" << cmdBlock->getSourceID() << " DestID:" << cmdBlock->getDestID() << " getContextID:" << cmdBlock->getContextID() << " CommandName:" << cmdBlock->getCommandName() << " IID:" << cmdBlock->getIID() << std::endl);

    // Pass the incoming command to the base class so that
    // it can take care of commands common to all collectors
    // and all agents.
    
    int handled = 0;
    handled = BaseCollectorImpl::processCommand(cmdBlock);

	#ifdef MVS
		cmdBlock = native2unicodeCmdBlock(cmdBlock);
	#endif

    if (cmdBlock->getSourceID() != 1) {
    	if (isEqualString(cmdBlock->getCommandName(), "establishDataPath")) {
        	m_dataListenerID = cmdBlock->getSourceID();
		}
    }
    m_AgentID = cmdBlock->getDestID();

    // If the base class handled the command, we are done.
    // 0 indicates command was handled, otherwise -1.
    if (handled == 0) {
        return handled; 
    }

    // See if this command is part of our interface.
    char* iid = cmdBlock->getIID();
    if (strncmp(iid, "org.eclipse.tptp.jvmti", strlen("org.eclipse.tptp.jvmti")) != 0) {
        sendErrorCommand(cmdBlock->getSourceID(),
            cmdBlock->getContextID(),
            TPTP_INTERFACE_UNKNOWN,
            cmdBlock->getIID());
        return -1;
    }

    char* cmdName = cmdBlock->getCommandName();
    // Handle the commands for our interface.
    if (isEqualString(cmdName, "applyFilters")) {
        char classPattern[250];
        char methodPattern[250];
        char mode[100];

        tptp_list_t* paramList = cmdBlock->getParamList();
        _tptp_node_t* node = paramList->head;
        for (int i = 0; i < paramList->count; i++) {
            tptp_param_t* param = (tptp_param_t*)(node->data);
            node = node->next;
            if (strcmp("Filter", param->name) == 0) {
                LOG_TRACE("Filter: " << param->value);
                if (sscanf(param->value, "%s %s %s\n", classPattern, methodPattern, mode) != EOF) {
                    m_ACC_env->AddFilter(iid, classPattern, methodPattern, mode);
                } else {
                    //TODO
                }
            }
        }
        m_ACC_env->ApplyFilters();
//        if(m_isTracing) {
//            m_ACC_env->Stop();
//            m_ACC_env->Detach();
//            m_ACC_env->Attach();
//            m_ACC_env->Start();
//        }
            
    } else if (isEqualString(cmdName, "applyOptions")) {
    	// TODO: this code doens't limit the option size; 
    	// add some limitation for security purposes?
        tptp_list_t* paramList = cmdBlock->getParamList();
        _tptp_node_t* node = paramList->head;
        for (int i = 0; i < paramList->count; i++) {
            tptp_param_t* param = (tptp_param_t*)(node->data);
            node = node->next;
            
            if (param->name && param->value && 
            	strcmp("option", param->name) == 0) {
           		
           		char* name;
            	char* value;
            	parseOption( param->name, param->value, name, value );
            		            		
            	// TODO Add check for agg profiler options and iid
            		
            	// name or value null means string not formed properly
            	if( name && value ) {            				
    				LOG_TRACE("Options: name='" << name 
    					<< "' value='" << value << "'" << std::endl);
    				m_ACC_env->SetProfileOption(iid, name, value);                               
            	}	
				free( name );
    			free( value ); 
			}
		}
		// Check if the Heap Instance Data collection has been enabled and
		// update the option.
        if(m_ACC_env->isHeapInstanceDataEnabled()) {
        	m_ACC_env->setHeapInstanceDataEnabled();
        }

    } else if (isEqualString(cmdName, "AnalyseHeap")) {
        m_ACC_env->AnalyseHeap();
    } else if (isEqualString(cmdName, "RunGC")) {
    	m_ACC_env->RunGC();
    } else if (isEqualString(cmdName, "ObjectDataCollection")) {
    	// Received the command to run Object instance data collection, analyze the
    	// Object and send the name value pairs back to the client.
    	
    	if(m_ACC_env->isHeapInstanceDataEnabled()) {
			// Set the max size for the returned result value to be 128 KB = (131072 * 1 byte).
			// Matches the value set in DataManager.
			const int RESULT_VALUE_MAX_SIZE = 131072;

			// Buffer for the header surrounding the command.
			const int CMD_HEADER_BUFFER_SIZE = 400;

			// Get the parameters from the received command
			tptp_list_t* paramList = cmdBlock->getParamList();
			_tptp_node_t* node = paramList->head;

			for (int i = 0; i < paramList->count; i++) {
				tptp_param_t* param = (tptp_param_t*)(node->data);
				node = node->next;

				// The TPTP ID of the Object from which to obtain the instance values
				if (param->name && param->value &&
					strcmp("TId", param->name) == 0) {

					char *TIdValues = (char *)malloc(strlen(param->value)+1);
					strcpy(TIdValues, param->value);
					LOG_TRACE("ObjectDataCollection: TIDs='" << TIdValues << "'" << std::endl);

					// Iterate through the comma separated TIds passed.
					char *currentTId = strtok(TIdValues, ",");
					while(currentTId != NULL) {

						// Capture the output to be returned in the command through the AC to the client
						// Obtains the values of the data for this Object to be returned to the client.
						Martini::ACCollector::MHeapObjectAnalysis * heapObjectAnalysis = (MHeapObjectAnalysis *)malloc(sizeof(MHeapObjectAnalysis));
						heapObjectAnalysis->TId = currentTId;
						heapObjectAnalysis->resultValue = (char *)malloc(RESULT_VALUE_MAX_SIZE);
						// Send the custom command to request the information about the passed Object.
						m_ACC_env->RunHeapObjectInstanceCollection(heapObjectAnalysis);

						// Create a String to hold the command and add a buffer for the header
						char * cmd = (char *)malloc(strlen((char *)heapObjectAnalysis->resultValue) + CMD_HEADER_BUFFER_SIZE);
						// Create the return command to send to the client
						sprintf( cmd, "<Cmd src=\"%d\" dest=\"%d\" ctxt=\"%d\"> "\
										   "<objInstData iid=\"org.eclipse.tptp.jvmti\"><tid>%s</tid><values>%s</values></objInstData></Cmd>",
										   cmdBlock->getDestID(), cmdBlock->getSourceID(), cmdBlock->getContextID(), currentTId, (char *)heapObjectAnalysis->resultValue);

						LOG_TRACE("ObjectDataCollection: about to show TID value pair" << std::endl);
						LOG_TRACE("ObjectDataCollection: TID='" << currentTId
									<< "' value='" << (char *)heapObjectAnalysis->resultValue << "'" << std::endl);
						LOG_TRACE("ObjectDataCollection: TID value pair displayed" << std::endl);
						#ifdef MVS
							char* nativeBuffer = 0;
							Martini::OSA::unicode2native(&nativeBuffer, cmd, strlen(cmd));
							BaseAgentImpl::sendCommand(nativeBuffer);

							if(nativeBuffer != 0)
								free(nativeBuffer);
						#else
							BaseAgentImpl::sendCommand(cmd);
						#endif

						free(heapObjectAnalysis->resultValue);
						free(heapObjectAnalysis);
						free(cmd);

						currentTId = strtok(NULL, ",");
					}
					free(TIdValues);
				}
			}
    	}
    } else if (isEqualString(cmdName, "CollectData")) {
        // TODO Add check iid (.CGProf)
        m_ACC_env->CollectAggData();
    } else if (isEqualString(cmdName, "EnableDataCollection")) {
		m_isEnabledDataCollection = true;
		m_ACC_env->EnableDataCollection();
    } else if (isEqualString(cmdName, "getMetadata")) {
        char* profilerName = m_ACC_env->GetProfiler();
        char buf[256];
        sprintf(buf, "<Cmd src=\"%d\" dest=\"%d\" ctxt=\"%d\"><getMetadata iid=\"org.eclipse.tptp.jvmti\"><name>profilerName</name><value>%s</value></getMetadata></Cmd>", cmdBlock->getDestID(), cmdBlock->getSourceID(), cmdBlock->getContextID(), profilerName);
        #ifdef MVS
        	char* nativeBuffer = 0;
        	Martini::OSA::unicode2native(&nativeBuffer, buf, strlen(buf));

        	BaseAgentImpl::sendCommand(nativeBuffer);

        	if(nativeBuffer != 0)
        		free(nativeBuffer);
        #else
        	BaseAgentImpl::sendCommand(buf);
        #endif
    } else if (isEqualString(cmdName, "terminateAgent")) {
        exit(0);
	} else if (isEqualString(cmdName, "setOutputFormat")) {
        tptp_list_t* paramList = cmdBlock->getParamList();
        _tptp_node_t* node = paramList->head;
        for (int i = 0; i < paramList->count; i++) {
            tptp_param_t* param = (tptp_param_t*)(node->data);
            node = node->next;
            
            if (param->name && param->value && 
            	strcmp("format", param->name) == 0) {
           		
				char cmd[BUFFER_LENGTH];
    			LOG_TRACE("OutputFormat requested: name='" << param->name 
    				<< "' value='" << param->value << "'");
    			char *enabledFormat = m_ACC_env->SetOutputFormat(param->value);
				sprintf( cmd, "<Cmd src=\"%d\" dest=\"%d\" ctxt=\"%d\"> "\
								   "<enabledOutputFormat iid=\"org.eclipse.tptp.jvmti\"><format>%s</format></enabledOutputFormat></Cmd>",
								   cmdBlock->getDestID(), cmdBlock->getSourceID(), cmdBlock->getContextID(), enabledFormat);
				#ifdef MVS
					char* nativeBuffer = 0;
					Martini::OSA::unicode2native(&nativeBuffer, cmd, strlen(cmd));
					BaseAgentImpl::sendCommand(nativeBuffer);

					if(nativeBuffer != 0)
						free(nativeBuffer);
				#else
					BaseAgentImpl::sendCommand(cmd);
				#endif
			}
        }
    } else {

	    // Try dispatching to extensions
	    const char* response = dispatchAgentlet(cmdBlock);
	    if( response ) {
		    
		    // If response is the empty string, the command was handled
		    // but no response is needed.
		    if( strlen(response) != 0 ) {
			    char cmd[BUFFER_LENGTH];
			    
			    sprintf(cmd, "<Cmd src=\"%d\" dest=\"%d\" ctxt=\"%d\">%s</Cmd>",
			            cmdBlock->getDestID(), cmdBlock->getSourceID(), 
			            cmdBlock->getContextID(), response);
			    
				#ifdef MVS
					char* nativeBuffer = 0;
					Martini::OSA::unicode2native(&nativeBuffer, cmd, strlen(cmd));
					BaseAgentImpl::sendCommand(nativeBuffer);

					if(nativeBuffer != 0)
						free(nativeBuffer);
				#else
					BaseAgentImpl::sendCommand(cmd);
				#endif
		    }
		    delete[] response;
		    
	    } else {
		    // This command is not in our interface and no custom handlers
		    // were found
		    sendErrorCommand(cmdBlock->getSourceID(),
		                     cmdBlock->getContextID(),
		                     TPTP_CMD_UNKNOWN,
		                     cmdName);
		    return -1; 
	    }
    }

	#ifdef MVS
		// Our native2unicodeCmdBlock cloned and converted all the values inside cmdBlock, so we need to free that here.
		// The caller of this function (BaseAgent) will free the original CmdBlock structure.
		delete(cmdBlock);
	#endif

    return 0;
}

int 
ACCollector::receiveData(int sourceID, char buffer[], int bytesRead, DIME_HEADER_PTR_T dimeHeader)
{

    // Display the incoming data, skipping past the header info.
    int headerLength = dimeHeader->id_length + dimeHeader->options_length + dimeHeader->type_length;
    LOG_TRACE("::: Received data: " << buffer + headerLength);
    return 0;
}

int 
ACCollector::Init(ACC_ENV* env, bool isControlled)
{
    if (env == 0) {
        return -1;
    }
    m_ACC_env = env;

    // Tell the Agent Controller this is an agent and establish its personal
    // command channel with the Agent Controller.
    // A command handler thread is started which listens on this channel and
    // calls the processCommand() method of this agent for all commands coming
    // through this channel.
    
    char* cargv[10];
    //No command line args hence the env to be picked up.
    this->processCommandLine(0, cargv);
    
    this->asyncRegisterAgent();
    if (isControlled) {
        while(m_isWaiting) {
		#ifdef _AIX
            usleep((100 * 1000)); // 100 milliseconds
		#else
            Sleep(100);
		#endif
        };
    }
    
    #ifdef MVS
    	// ACCollector does not use the waitForTermination() base agent call, so we delete this semaphore immediately
    	tptp_deleteSemaphore(this->getTerminationSemaphore());
    #endif
    
    return 0;
}

bool 
ACCollector::isAgentInitialized(int contextID)
{
    if (!m_ACC_env->IsAgentInitialized()) {
        LOG_TRACE("Sending Error command: TPTP_SYS_UNSPECIFIED_ERR 'Agent is not initialized yet'");
        sendErrorCommand(m_dataListenerID, contextID, TPTP_SYS_UNSPECIFIED_ERR, "Agent is not initialized yet");
        return false;
    }
    return true;
}

int 
ACCollector::SendVMAgentInitializedCommand()
{
//    if (m_isTracing) {
//        char buf[256];
//        sprintf(buf, "<Cmd src=\"%d\" dest=\"%d\" ctxt=\"%d\"><agentInitialized iid=\"org.eclipse.tptp.jvmti\"></agentInitialized></Cmd>", m_AgentID, m_dataListenerID, -1);
//        LOG_TRACE("Sending command: " << buf);
//        return BaseAgentImpl::sendCommand(buf);
//    }
//    return -1;
    return 0;
}

void 
ACCollector::SendWrongPhaseError(int contextID)
{
    LOG_TRACE("Sending Error command: TPTP_SYS_UNSPECIFIED_ERR 'Wrong Phase'");
    sendErrorCommand(m_dataListenerID, contextID, TPTP_SYS_UNSPECIFIED_ERR, "Wrong Phase");
}
