/************************************************************************
 * Copyright (c) 2006, 2010 Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *    Viacheslav Rybalov, Intel - Initial API and implementation
 *
 * $Id: AttachEvent.cpp,v 1.21 2010/08/17 16:29:59 jwest Exp $ 
 ************************************************************************/

#include "AttachEvent.h"
#include <string.h>

using namespace Martini::BaseProf;
using namespace Martini::MPI;
using namespace Martini::JPIAgent;

CECAttachEvent::CECAttachEvent(){}
CECAttachEvent::~CECAttachEvent(){}

/*
 * Init - initializes internal data and registers for Attach event    
 */
TResult 
CECAttachEvent::Init(CProfEnv* profEnv)
{
    m_pProfEnv = profEnv;
    TResult res = profEnv->m_pMpiApi->RegisterEvent(profEnv->m_clientId, *this);
    return res;
}

/*
 * HandleEvent - callback function for Attach event    
 */
void 
CECAttachEvent::HandleEvent()
{
    if (!m_pProfEnv->m_bVMInitDone) {
        return;
    }
    
    m_pProfEnv->DeleteAggCallGraphData();
    m_pProfEnv->DeleteStoredThreadEvents();

    // Add for bug 194081
    m_pProfEnv->DeleteMethodsData();
    m_pProfEnv->m_pShadowStackTracker->TrackECAttach();
    

    if (m_pProfEnv->IsSupportedEG(EG_CALL_GRAPH))
    {
    	// Call graph profiler skips everything after the 'return'
        TResult res = m_pProfEnv->EnableSupportedEG();
        if (MRTE_FAILED(res))
        {
            if (MRTE_ERROR_PHASE_FAILURE == res) 
            {
                LOG_DIE("Attach failed (too early)");
            } 
            else
            {
                LOG_DIE("Attach failed");
            }
        }
        
        return;
    }
    // Add for bug 194081

    const int MAX_THREADS = 100;
    const int MAX_FRAMES = 100;
    TResult res; 

    SThreadInfoArray threadInfoArray;
    threadInfoArray.uiSize = MAX_THREADS; //TODO check it!!!!
    threadInfoArray.pEntries = (SThreadInfoArrayEntry*)malloc(sizeof(SThreadInfoArrayEntry) * threadInfoArray.uiSize);
    memset(threadInfoArray.pEntries, 0, sizeof(SThreadInfoArrayEntry) * threadInfoArray.uiSize);

    // If monitor contention is supported (e.g. Thread Analysis w/ Contention Analysis - JGW)
    if (m_pProfEnv->IsSupportedEG(EG_MONITOR)) {
        unsigned int i;
        for (i = 0; i < MAX_THREADS; i++) {
            threadInfoArray.pEntries[i].threadInfo.vmOffsetStack.uiSize = MAX_FRAMES;
            threadInfoArray.pEntries[i].threadInfo.vmOffsetStack.pStackEntries = new SStackEntry[MAX_FRAMES];
        }

        // Retrieve the states, statuses, and info, of all threads that are currently running
        // and store in threadInfoArray
        res = m_pProfEnv->m_pMpiApi->GetAllThreadsInfo(m_pProfEnv->m_clientId, MAX_FRAMES, DR_THREAD_INFO | DR_THREAD_STATE | DR_CURRENT_MONITOR | DR_OBJECT_ID | DR_VM_RELATIVE_STACK_TRACE, &threadInfoArray);
        LOG_ASSERT(res == MRTE_RESULT_OK);

        for (i = 0; i < threadInfoArray.uiActualSize; i++) {
        	// For each thread.....

            m_pProfEnv->CheckObjectId(threadInfoArray.pEntries[i].threadInfo.threadObjectId, threadInfoArray.pEntries[i].threadId);


			if (m_pProfEnv->m_enableDataCollection) {
	            m_pProfEnv->ec_env->PrintThreadStartElement(threadInfoArray.pEntries[i].threadId, threadInfoArray.pEntries[i].threadInfo.threadObjectId, &(threadInfoArray.pEntries[i].threadInfo));
			}
			else {
				m_pProfEnv->StoreThreadEvent(threadInfoArray.pEntries[i].threadId, THREAD_START_EVENT, &(threadInfoArray.pEntries[i].threadInfo), 0, threadInfoArray.pEntries[i].threadInfo.threadObjectId);
			}

			// If the current thread is sleeping...
            if (threadInfoArray.pEntries[i].threadInfo.state == TS_SLEEPING) {
                SMonitorWaitEventData data; 
                data.objectId = -1;
                data.threadId = threadInfoArray.pEntries[i].threadId;
                data.timeoutMillis = 0; //Not available

                // Associate the current time with the time out value for the current thread
                m_pProfEnv->AddTimeOut(m_pProfEnv->ec_env->GetTime(), threadInfoArray.pEntries[i].threadId);

                // Print the monitor wait element (either with or without the corresponding stack trace)
                if (m_pProfEnv->ec_env->isStackInfoNormal()) {
                    SStackTrace_ stackTrace;
                    m_pProfEnv->ConvertStackTrace(&(threadInfoArray.pEntries[i].threadInfo), &stackTrace);
                    m_pProfEnv->ec_env->PrintMonitorWaitElement(&data, &stackTrace);
                    m_pProfEnv->FreeStackTrace(&stackTrace);
                } else {
                    m_pProfEnv->ec_env->PrintMonitorWaitElement(&data, 0);
                }

				if (m_pProfEnv->m_enableDataCollection) {
					if (m_pProfEnv->ec_env->isStackInfoNormal()) {
						SStackTrace_ stackTrace;
						m_pProfEnv->ConvertStackTrace(&(threadInfoArray.pEntries[i].threadInfo), &stackTrace);
						m_pProfEnv->ec_env->PrintMonitorWaitElement(&data, &stackTrace);
						m_pProfEnv->FreeStackTrace(&stackTrace);
					} else {
						m_pProfEnv->ec_env->PrintMonitorWaitElement(&data, 0);
					}

				} else {
					// ... if data collection is not enabled, then just store the event
						m_pProfEnv->StoreThreadEvent(threadInfoArray.pEntries[i].threadId, MONITOR_WAIT_EVENT, &data, 0, threadInfoArray.pEntries[i].threadInfo.threadObjectId);
				}

			// If the current thread is blocked...
            } else if  (threadInfoArray.pEntries[i].threadInfo.state == TS_BLOCKED) {

            	// Get the info for the object monitor that this thread is currently blocked on,
            	// to determine which thread currently owns it, which is used below.
                SObjectDataRequest objectDataRequest;
                res = m_pProfEnv->m_pMpiApi->GetObjectInfo(m_pProfEnv->m_clientId, threadInfoArray.pEntries[i].threadInfo.currentMonitorId, DR_MONITOR_OWNER_THREAD_ID, &objectDataRequest);
                LOG_ASSERT(res == MRTE_RESULT_OK);

                // Generate the data element, to pass to the printer (or store)
                SContendedMonitorEnterEventData data;
                data.objectId = threadInfoArray.pEntries[i].threadInfo.currentMonitorId;
                data.ownerThreadId = objectDataRequest.monOwnerThreadId;
                data.threadId = threadInfoArray.pEntries[i].threadId;

				if (threadInfoArray.pEntries[i].threadInfo.currentMonitorId != 0) {
					m_pProfEnv->CheckObjectId(data.objectId, data.threadId);
				}

				// Print the monitor contended enter element (and also the stack trace, if appropriate)
				if (m_pProfEnv->m_enableDataCollection) {
					if (m_pProfEnv->ec_env->isStackInfoNormal()) {
						SStackTrace_ stackTrace;
						m_pProfEnv->ConvertStackTrace(&(threadInfoArray.pEntries[i].threadInfo), &stackTrace);
						m_pProfEnv->ec_env->PrintMonitorContendedEnterElement(&data, &stackTrace);
						m_pProfEnv->FreeStackTrace(&stackTrace);
					} else {
						m_pProfEnv->ec_env->PrintMonitorContendedEnterElement(&data, 0);
					}
				}
				else {
					m_pProfEnv->StoreThreadEvent(threadInfoArray.pEntries[i].threadId, CONTENDED_MONITOR_ENTER_EVENT, &data, 0, threadInfoArray.pEntries[i].threadInfo.threadObjectId);
				}


			// If the current thread is waiting....
            } else if  (threadInfoArray.pEntries[i].threadInfo.state == TS_WAITING) {

            	// Generate the wait event data, to pass to the printer below (or to store)
                SMonitorWaitEventData data; 
                data.objectId = threadInfoArray.pEntries[i].threadInfo.currentMonitorId;
                data.threadId = threadInfoArray.pEntries[i].threadId;
                data.timeoutMillis = 0;  //Not available
                m_pProfEnv->AddTimeOut(m_pProfEnv->ec_env->GetTime(), threadInfoArray.pEntries[i].threadId);

				if (threadInfoArray.pEntries[i].threadInfo.currentMonitorId != 0) {
			        m_pProfEnv->CheckObjectId(data.objectId, threadInfoArray.pEntries[i].threadId);
				}

				// Print the monitor wait element (with or without stack as needed)
				if (m_pProfEnv->m_enableDataCollection) {
					if (m_pProfEnv->ec_env->isStackInfoNormal()) {
						SStackTrace_ stackTrace = {0};
						m_pProfEnv->ConvertStackTrace(&(threadInfoArray.pEntries[i].threadInfo), &stackTrace);
						m_pProfEnv->ec_env->PrintMonitorWaitElement(&data, &stackTrace);
						m_pProfEnv->FreeStackTrace(&stackTrace);
					} else {
						m_pProfEnv->ec_env->PrintMonitorWaitElement(&data, 0);
					}
				} else {
					m_pProfEnv->StoreThreadEvent(threadInfoArray.pEntries[i].threadId, MONITOR_WAIT_EVENT, &data, 0, threadInfoArray.pEntries[i].threadInfo.threadObjectId);
				}

            }
        }
        for (i = 0; i < MAX_THREADS; i++) {
            delete [](threadInfoArray.pEntries[i].threadInfo.vmOffsetStack.pStackEntries);
        }
    } else {
    	// Get the information for all the threads
        res = m_pProfEnv->m_pMpiApi->GetAllThreadsInfo(m_pProfEnv->m_clientId, 0, DR_THREAD_INFO, &threadInfoArray);
        LOG_ASSERT(res == MRTE_RESULT_OK);

        // Print (or store) thread start elements for each
        for (unsigned int i = 0; i < threadInfoArray.uiActualSize; i++) {
			if (m_pProfEnv->m_enableDataCollection) {
	            m_pProfEnv->ec_env->PrintThreadStartElement(threadInfoArray.pEntries[i].threadId, 0, &(threadInfoArray.pEntries[i].threadInfo)); //TODO Investigate
			}
			else {
				m_pProfEnv->StoreThreadEvent(threadInfoArray.pEntries[i].threadId, THREAD_START_EVENT, &(threadInfoArray.pEntries[i].threadInfo), 0, threadInfoArray.pEntries[i].threadInfo.threadObjectId);
			}
        }
    }
    free(threadInfoArray.pEntries);

    res = m_pProfEnv->EnableSupportedEG();
    if (MRTE_FAILED(res)) {
        if (MRTE_ERROR_PHASE_FAILURE == res) {
            LOG_DIE("Attach failed (too early)");
        } else {
            LOG_DIE("Attach failed");
        }
    }
}
