/************************************************************************
 * Copyright (c) 2006, 2010 Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *    Viacheslav Rybalov, Intel - Initial API and implementation
 *
 * $Id: ProfEnv.cpp,v 1.45 2010/12/07 00:35:43 jwest Exp $ 
 ************************************************************************/
#ifndef LIN
#pragma warning(disable:4786) // Add for bug 194081 to disable VC compiler warning when use STL
#endif

#include <algorithm>
#include <string.h>

#include "ProfEnv.h"
#include "log.h"

using namespace Martini::BaseProf;
using namespace Martini::MPI;
using namespace Martini::OSA;
using namespace Martini::JPIAgent;
using namespace std;  // Add for bug 194081

CProfEnv::CProfEnv() : m_pClassSData(NULL), m_pMethodSData(NULL), m_pObjectSData(NULL),
	m_pMonitorTimeOuts(NULL), m_pSThreadEventsData(NULL), m_pSMethodLeaveData_(NULL), m_pStackSet(NULL),
	m_bVMInitDone(false), m_profilerIsActive(false)
{
    m_pClassSDataLockObject = CreateThreadSync();
    m_pMethodSDataLockObject = CreateThreadSync();
    m_pObjectSDataLockObject = CreateThreadSync();
    m_pMonitorTimeOutsLockObject = CreateThreadSync();
    m_pSThreadEventsDataLockObject = CreateThreadSync();

    lockClassDefObject = CreateThreadSync();
    lockObjectCheckObject = CreateThreadSync();
    m_pSSMethodLeaveData_LockObject = CreateThreadSync();
    
    //Aggregated
    lockAggObject = CreateThreadSync();
    lockGetThreadInfoObject = CreateThreadSync();
}

CProfEnv::~CProfEnv()
{
    Flush();
    m_pClassSDataLockObject->Destroy();
    m_pMethodSDataLockObject->Destroy();
    m_pObjectSDataLockObject->Destroy();
    m_pMonitorTimeOutsLockObject->Destroy();
    lockClassDefObject->Destroy();
    lockObjectCheckObject->Destroy();
    m_pSThreadEventsDataLockObject->Destroy();
    m_pSSMethodLeaveData_LockObject->Destroy(); 
    lockGetThreadInfoObject->Destroy(); 
    
    // Add for bug 194081
    m_pShadowStackTracker = NULL;
    // Add for bug 194081
}

void 
CProfEnv::Flush()
{
    m_pClassSDataLockObject->Enter();
    if (m_pClassSData != 0) {
        delete m_pClassSData;
    }
    m_pClassSDataLockObject->Leave();
    m_pObjectSDataLockObject->Enter();
    if (m_pObjectSData != 0) {
        delete m_pObjectSData;
    }
    m_pObjectSDataLockObject->Leave();
    m_pMonitorTimeOutsLockObject->Enter();
    if (m_pMonitorTimeOuts != 0) {
        delete m_pMonitorTimeOuts;
    }
    m_pMonitorTimeOutsLockObject->Leave();
    lockAggObject->Enter();
    if (m_pStackSet != 0) {
        delete m_pStackSet;
    }
    lockAggObject->Leave();
    m_pSThreadEventsDataLockObject->Enter();
    if (m_pSThreadEventsData != 0) {
        delete m_pSThreadEventsData;
    }
    m_pSThreadEventsDataLockObject->Leave();
    m_pSSMethodLeaveData_LockObject->Enter();
    if (m_pSMethodLeaveData_ != 0) {
        delete m_pSMethodLeaveData_;
    }
    m_pSSMethodLeaveData_LockObject->Leave();
    m_pMethodSDataLockObject->Enter();
    if (m_pMethodSData != 0) {
        delete m_pMethodSData;
    }
    m_pMethodSDataLockObject->Leave();
    
    // Add for bug 194081
    if (m_pShadowStackTracker)
    {
        delete m_pShadowStackTracker;
        m_pShadowStackTracker = NULL;
    }
    // Add for bug 194081
}

/*
 * Init - initializes internal data and registers 
 */
TResult 
CProfEnv::Init(const char* libName)
{
    Martini::OSA::ILibraryLoader* libraryLoader;
    // Modified for bug 241984 and bug 226572
	//libraryLoader = CreateLibraryLoader(libName, 0);
    libraryLoader = LoadBistroLibrary(libName);
    // Modified for bug 241984 and bug 226572

    if (libraryLoader == 0) {
        return MRTE_ERROR_OSA_FAILURE;
    }
    GetEC_Env_t ecpEnv = (GetEC_Env_t)(libraryLoader->GetEntry("GetEC_Env"));
    if (ecpEnv == 0) {
        return MRTE_ERROR_OSA_FAILURE;
    }
    libraryLoader->Destroy();
    (*ecpEnv)(profName, &ec_env);
    
    // Add for bug 194081
	m_pShadowStackTracker = new CCGShadowStackTracker(*this);
	if ( (NULL == m_pShadowStackTracker) ||
	    !(m_pShadowStackTracker->ValidateCGShadowStackTracker()) )
	{
	    return MRTE_ERROR_FAIL;
	}
	// Add for bug 194081
	
    return MRTE_RESULT_OSA_PLACE_HOLDER;
}

void 
CProfEnv::AddNewClassData(TId classId, SClassInfo* pSClassInfo, bool isPrinted)
{
    m_pClassSDataLockObject->Enter();
    if (m_pClassSData == 0) {
        m_pClassSData = new CClassDefDataSet(classId, pSClassInfo, isPrinted);
    } else {
        m_pClassSData->addNewData(classId, pSClassInfo, isPrinted);    
    }
    m_pClassSDataLockObject->Leave();
}

SClassInfo* 
CProfEnv::GetClassData(TId classId)
{
    SClassInfo* res;
    m_pClassSDataLockObject->Enter();
    if (m_pClassSData == 0) {
        res = 0;
    } else {
        res = (SClassInfo*)m_pClassSData->GetData(classId);    
    }
    m_pClassSDataLockObject->Leave();
    return res;
}

/*bool 
CProfEnv::IsClassPrinted(TId classId)
{
    bool ret;
    m_pClassSDataLockObject->Enter();
    if (m_pClassSData == 0) {
        ret = false;
    } else {
        ret = m_pClassSData->IsClassPrinted(classId, ec_env);
    }
    m_pClassSDataLockObject->Leave();
    return ret;
}
*/
bool 
CProfEnv::IsClassStored(TId classId)
{
    bool ret;
    m_pClassSDataLockObject->Enter();
    if (m_pClassSData == 0) {
        ret = false;
    } else {
        ret = m_pClassSData->IsClassStored(classId);
    }
    m_pClassSDataLockObject->Leave();
    return ret;
}


void
CProfEnv::AddNewObjectData(SHeapEventData* pHeapEventData, bool isPrinted)
{
    m_pObjectSDataLockObject->Enter();
    if (m_pObjectSData == 0) {
        m_pObjectSData = new CObjectDataSet(pHeapEventData->objectId, pHeapEventData, isPrinted);
    } else {
        m_pObjectSData->addNewData(pHeapEventData->objectId, pHeapEventData, isPrinted);    
    }
    m_pObjectSDataLockObject->Leave();
}


bool 
CProfEnv::IsObjectStored(TId objectId)
{
    bool ret;
    m_pObjectSDataLockObject->Enter();
    if (m_pObjectSData == 0) {
        ret = false;
    } else {
        ret = m_pObjectSData->IsObjectStored(objectId);
    }
    m_pObjectSDataLockObject->Leave();
    return ret;
}

// Add for bug 194081
void 
CProfEnv::DeleteMethodsData()
{
    if (m_pMethodSData) 
    {
        delete m_pMethodSData;
        m_pMethodSData = NULL;
    }
}
// Add for bug 194081

void 
CProfEnv::AddNewMethodData(SMethodInfo* pSMethodInfo, TId methodId, bool isPrinted)
{
    m_pMethodSDataLockObject->Enter();
    if (m_pMethodSData == 0) {
        m_pMethodSData = new CMethodDefDataSet(methodId, pSMethodInfo, isPrinted);
    } else {
        m_pMethodSData->addNewData(methodId, pSMethodInfo, isPrinted);    
    }
    m_pMethodSDataLockObject->Leave();
}

SMethodInfo* 
CProfEnv::GetMethodData(TId methodId)
{
    SMethodInfo* res;
    m_pMethodSDataLockObject->Enter();
    if (m_pMethodSData == 0) {
        res = 0;
    } else {
        res = (SMethodInfo*)m_pMethodSData->GetData(methodId);    
    }
    m_pMethodSDataLockObject->Leave();
    return res;
}

bool 
CProfEnv::IsMethodStored(TId methodId)
{
    bool ret;
    m_pMethodSDataLockObject->Enter();
    if (m_pMethodSData == 0) {
        ret = false;
    } else {
        ret = m_pMethodSData->IsMethodStored(methodId);
    }
    m_pMethodSDataLockObject->Leave();
    return ret;
}


void 
CProfEnv::SendCollectedData() 
{
    m_pClassSDataLockObject->Enter();
    if (m_pClassSData != 0) {
        m_pClassSData->PrintClassDefElements(ec_env);
    }
    m_pClassSDataLockObject->Leave();
    m_pMethodSDataLockObject->Enter();
    if (m_pMethodSData != 0) {
        m_pMethodSData->PrintMethodDefElements(ec_env);
    }
    m_pMethodSDataLockObject->Leave();
    m_pObjectSDataLockObject->Enter();
    if (m_pObjectSData != 0) {
        m_pObjectSData->PrintObjAllocElements(this);
    }
    m_pObjectSDataLockObject->Leave();
}


void 
CProfEnv::CleanPrintFlags() 
{
    m_pClassSDataLockObject->Enter();
    if (m_pClassSData != 0) {
        m_pClassSData->CleanPrintFlags();
    }
    m_pClassSDataLockObject->Leave();
    m_pMethodSDataLockObject->Enter();
    if (m_pMethodSData != 0) {
        m_pMethodSData->CleanPrintFlags();
    }
    m_pMethodSDataLockObject->Leave();
    m_pObjectSDataLockObject->Enter();
    if (m_pObjectSData != 0) {
        m_pObjectSData->CleanPrintFlags();
    }
    m_pObjectSDataLockObject->Leave();
}

void 
CProfEnv::CheckClassId(TId classId) {
    bool isPrint = false;

    lockClassDefObject->Enter();
    if (IsClassStored(classId)) {
        lockClassDefObject->Leave();
        return;    
    }

	SClassInfo* classInfo = new SClassInfo;
    TResult iRes = m_pMpiApi->GetClassInfo(m_clientId, classId, 
        DR_JAVA_NATIVE_CLASS_NAME | DR_SOURCE_FILE_NAME, classInfo);
    LOG_ASSERT(iRes == MRTE_RESULT_OK);
    LOG_ASSERT((classInfo->validData & DR_JAVA_NATIVE_CLASS_NAME) != 0);
    LOG_ASSERT((classInfo->validData & DR_SOURCE_FILE_NAME) != 0);

    if (m_profilerIsActive) {
        isPrint = true;
    }

    AddNewClassData(classId, classInfo, isPrint);

    if (isPrint) {
        ec_env->PrintClassDefElement(classId, classInfo);
    }

    lockClassDefObject->Leave();
}

void 
CProfEnv::CheckObjectId(TId objectId, TId threadId) {
    lockObjectCheckObject->Enter();
    if (IsObjectStored(objectId)) {
        lockObjectCheckObject->Leave();
        return;    
    }
    
    SObjectDataRequest objectData;
    TResult iRes = m_pMpiApi->GetObjectInfo(m_clientId, objectId, 
        DR_OBJECT_INFO, &objectData);
    LOG_ASSERT(iRes == MRTE_RESULT_OK);

    // Object can be alredy garbage collected, so GetObjectInfo can't
    // provide information about it.
    if ((objectData.validData & DR_OBJECT_INFO) != DR_OBJECT_INFO) {
        lockObjectCheckObject->Leave();
        return;
    }

    SObjectInfo* objectInfo = new SObjectInfo;
    *objectInfo = objectData.objectInfo;

    SHeapEventData* heapEventData = new SHeapEventData;
    heapEventData->objectId = objectId;
    heapEventData->pObjectInfo = objectInfo;
    heapEventData->threadId = threadId; // TODO ThreadId is not correct
    heapEventData->validData = DR_OBJECT_ID | DR_THREAD_ID | DR_OBJECT_INFO;

    CheckClassId(objectData.objectInfo.classId);

    bool isPrint = false;
    if (m_profilerIsActive) {
        isPrint = true;
    }

    AddNewObjectData(heapEventData, isPrint);

    if (isPrint) {
        ec_env->PrintObjAllocElement(heapEventData, (U64)-1);
    }

    lockObjectCheckObject->Leave();
}

// Add for bug 194081. Filter additional data referring to CanInstrumentClass of CGAdaptor
bool CProfEnv::IsExcludedClassMethod(const char* szClassName, const char* szMethodName)
{
    // Can not get which classes can not be instrumented from MPI in CGAdaptor, so repeat them
    static char* ExcludedClassPatterns[] = 
    {
        "org.eclipse.tptp.martini.",
        "java.lang.object",
        "sun.reflect.Generated",
        "jrockit.",
        "java.lang.J9VM",
        "com.Intel.VTune.VTuneAPI",
        "org.eclipse.hyades.collection.profiler",
        "CGLIB$$"
    };

    // Check if the class appear in the excluded class patterns list
    unsigned int patternArraySize = 
        sizeof(ExcludedClassPatterns)/sizeof(*ExcludedClassPatterns);
    
    for (unsigned int i = 0; i < patternArraySize; ++i)
    {
        char *szExcludedPattern = ExcludedClassPatterns[i];
        if (strstr(szClassName, szExcludedPattern) != NULL)
        {
            return true;
        }
    }

    return (ec_env->IsExcluded(szClassName, szMethodName));
}
// Add for bug 194081. Filter additional data referring to CanInstrumentClass

/** Given a class string, for instance, 'java/util/HashMap<K,V>',
 * if the class contains generics (the <K,V> component) then
 * remove them, otherwise, leave the string untouched. In either case, the
 * result is stored in pNewClassName. 0 is returned on success, otherwise
 * -1 is returned. */
int CProfEnv::RemoveGenericsFromClassName(const char *pClassName, char *pNewClassName) {
	if(pClassName == 0) {
		return -1;
	}

	if(strstr(pClassName, "<") != NULL && strstr(pClassName, ">") != NULL) {

		// Copy the current class name into the new class name
		strcpy(pNewClassName, pClassName);
		char * firstLT = strstr(pNewClassName, "<");

		// Terminate at the first LT, in the new class name
		firstLT[0] = 0;
	} else {
		strcpy(pNewClassName, pClassName);
	}

	return 0;
}

void 
CProfEnv::CheckMethodId(TId methodId) {
    // Add for bug 194081 to filter additional data after reattach
    SMethodInfo methodClassInfo = {0};
    TResult iRes = m_pMpiApi->GetMethodInfo(m_clientId, 
        methodId, DR_CLASS_NAME | DR_METHOD_NAME, &methodClassInfo);
    
    LOG_ASSERT(iRes == MRTE_RESULT_OK);
    LOG_ASSERT((methodClassInfo.validData & DR_CLASS_NAME) != 0);
    LOG_ASSERT((methodClassInfo.validData & DR_METHOD_NAME) != 0);

	char classNameWithoutGenerics[65536]; // Max Length of fully qualified path
	
	// The user-provided filters coming from the workbench do not take into account generics on the class names,
	// for instance, java.util.HashMap<K, V>, so we temporarily strip them when doing our filter comparison.
	if(methodClassInfo.szClassName != NULL) {
		if(RemoveGenericsFromClassName(methodClassInfo.szClassName, classNameWithoutGenerics) == -1) {
			// If the call fails for any reason, then return
			return;
		}
	} else {
		// If the original class is NULL, theres nothing we can do (so may as well exclude by returning)
		return;
	}

    if (IsExcludedClassMethod(classNameWithoutGenerics, methodClassInfo.szMethodName))
    {
        return;
    }
    // Add for bug 194081  to filter additional data after reattach
    
    bool isPrint = false;

    m_pMethodSDataLockObject->Enter();
    if (IsMethodStored(methodId)) {
        m_pMethodSDataLockObject->Leave();
        return;    
    }

    SMethodInfo* methodInfo = new SMethodInfo();
    if (NULL == methodInfo)
    {
        m_pMethodSDataLockObject->Leave();
        return;
    }

	// The allocated memory will be freed when methodData is not used in MethodDataSet
    methodInfo->managedToSrcLineMap.uiSize = 1;
    methodInfo->managedToSrcLineMap.entries = new SLineNumberTableEntry[methodInfo->managedToSrcLineMap.uiSize];
    
    iRes = m_pMpiApi->GetMethodInfo(m_clientId, methodId,
        DR_JAVA_NATIVE_METHOD_NAME | DR_JAVA_NATIVE_METHOD_SIGNATURE | DR_METHOD_NAME | DR_METHOD_SIGNATURE | 
        DR_JAVA_NATIVE_CLASS_NAME | DR_CLASS_NAME | DR_METHOD_LINE_NUMBERS | DR_CLASS_ID | DR_MANAGED_TO_SRC_LINE_MAP, methodInfo);
    if (iRes == MRTE_ERROR_BUFFER_TOO_SHORT) {
        delete [](methodInfo->managedToSrcLineMap.entries);
        methodInfo->managedToSrcLineMap.uiSize = methodInfo->managedToSrcLineMap.uiActualSize;
        methodInfo->managedToSrcLineMap.entries = new SLineNumberTableEntry[methodInfo->managedToSrcLineMap.uiSize];
        iRes = m_pMpiApi->GetMethodInfo(m_clientId, methodId,
            DR_JAVA_NATIVE_METHOD_NAME | DR_JAVA_NATIVE_METHOD_SIGNATURE | DR_METHOD_NAME | DR_METHOD_SIGNATURE | 
            DR_JAVA_NATIVE_CLASS_NAME | DR_CLASS_NAME | DR_METHOD_LINE_NUMBERS | DR_CLASS_ID | DR_MANAGED_TO_SRC_LINE_MAP, methodInfo);
    }
    
    LOG_ASSERT(iRes == MRTE_RESULT_OK);
    LOG_ASSERT((methodInfo->validData & DR_JAVA_NATIVE_METHOD_NAME) != 0);
    LOG_ASSERT((methodInfo->validData & DR_JAVA_NATIVE_CLASS_NAME) != 0);
    LOG_ASSERT((methodInfo->validData & DR_CLASS_NAME) != 0);
    LOG_ASSERT((methodInfo->validData & DR_JAVA_NATIVE_METHOD_SIGNATURE) != 0);
    LOG_ASSERT((methodInfo->validData & DR_METHOD_NAME) != 0);
    LOG_ASSERT((methodInfo->validData & DR_METHOD_SIGNATURE) != 0);
    LOG_ASSERT((methodInfo->validData & DR_METHOD_LINE_NUMBERS) != 0);
    LOG_ASSERT((methodInfo->validData & DR_CLASS_ID) != 0);

    if (m_profilerIsActive) {
        isPrint = true;
    }

    CheckClassId(methodInfo->classId);

    AddNewMethodData(methodInfo, methodId, isPrint);
    if (isPrint) {
        ec_env->PrintMethodDefElement(methodId, methodInfo);
    }
    m_pMethodSDataLockObject->Leave();
}


/** Associate the given time with the given thread (AFAIK timeOut param is the current time - jgw),
 * and adds it to the monitor timeout list */
void 
CProfEnv::AddTimeOut(U64 timeOut, TId threadId)
{
    TimeMillis* timeOutStruct = static_cast<TimeMillis*>(malloc(sizeof(TimeMillis)));
    timeOutStruct->timeoutMillis = timeOut;
    m_pMonitorTimeOutsLockObject->Enter();
    if (m_pMonitorTimeOuts == 0) {
        m_pMonitorTimeOuts = new CTIdSet(threadId, timeOutStruct);
    } else {
        m_pMonitorTimeOuts->addNewData(threadId, timeOutStruct);    
    }
    m_pMonitorTimeOutsLockObject->Leave();
}

/** Get the associated time for the given thread */
U64 
CProfEnv::GetTimeOut(TId threadId)
{
    TimeMillis* timeOutStruct;
    m_pMonitorTimeOutsLockObject->Enter();
    LOG_ASSERT(m_pMonitorTimeOuts != 0);
    timeOutStruct = static_cast<TimeMillis*>(m_pMonitorTimeOuts->GetData(threadId));
    m_pMonitorTimeOutsLockObject->Leave();
    return timeOutStruct->timeoutMillis;
}

//============ Aggregated CG profiler ==========================================
CStackHead* CProfEnv::GetThreadAggCallTree(TId threadId)
{
    CStackHead* pCallTree = m_pStackSet;
    while (NULL != pCallTree)
    {
        if( pCallTree->IsSameThreadStack(threadId) ) break;
        pCallTree = pCallTree->NextCallTree();
    }
    
    return pCallTree;
}

CStackHead* CProfEnv::AddThreadAggCallTree(TId threadId)
{
    // Check whether thread call tree exists or not
    lockAggObject->Enter();
    CStackHead* pCallTreeIter = GetThreadAggCallTree(threadId);

    if (NULL == pCallTreeIter)
    {
        pCallTreeIter = new CStackHead(ec_env, threadId, m_pStackSet);
        if (NULL == pCallTreeIter)
        {
            LOG_DIE("New memory for Agg Call graph failed.")
        }
        
        m_pStackSet = pCallTreeIter;
    }
    
    lockAggObject->Leave();

    return pCallTreeIter;
}
	

// Aggregated
void 
CProfEnv::AddNewStackEntry(TId threadId, TId methodId, U64 uiCpuNanos, bool is_active)
{
    CStackHead* pCallTree = AddThreadAggCallTree(threadId);
    pCallTree->AddStackEntry(methodId, uiCpuNanos, is_active);
}

// Aggregated
void 
CProfEnv::FinalizeStackEntry(TId threadId, U64 uiCpuNanos)
{
    lockAggObject->Enter();
    CStackHead* pCallTree = GetThreadAggCallTree(threadId);
    lockAggObject->Leave();
    
    if (NULL != pCallTree)
    {
        pCallTree->FinalizeStackEntry(uiCpuNanos);
    }
}

// Aggregated
void 
CProfEnv::PrintAggCallGraph() {
    lockAggObject->Enter();
    if (m_pStackSet != 0) {
        m_pStackSet->PrintAggCallGraph();
    }
    lockAggObject->Leave();
}


void CProfEnv::DeleteAggCallGraphData() {
    lockAggObject->Enter();
    if (m_pStackSet != NULL) {
    	delete m_pStackSet;
    	m_pStackSet = NULL;
    }
    lockAggObject->Leave();
}

//======================= Stack ============================================

int 
CProfEnv::GetLineNumber(U64 location, SLineNumberTable* lineNumberTable) {
    int lineNumber = -1;
    if ((NULL == lineNumberTable) || (-1 == location)) {
        return lineNumber;
    }

    // Find the nearest line offset in line number table to location.
    for (unsigned int k = 0; (k < lineNumberTable->uiActualSize) && 
    	                      (location >= lineNumberTable->entries[k].uiOffset); k++) {
        lineNumber = lineNumberTable->entries[k].uiLineNumber;
    }
    
    return lineNumber;
}

int 
CProfEnv::GetLineNumber(U64 location, TId methodId) {

    SMethodInfo* methodInfo = GetMethodData(methodId);
    if ((methodInfo != 0) && ((methodInfo->validData & DR_MANAGED_TO_SRC_LINE_MAP) != 0)) {
        return GetLineNumber(location, &(methodInfo->managedToSrcLineMap));
    } else {
        return -1;
    }

}

void 
CProfEnv::GetStackTrace(TId threadId, SStackTrace_* stackTrace)
{
    SThreadInfo threadInfo ={0};
    int size = 0;
    int maxDepth = ec_env->getMaxStackDepth();
    TResult retVal;
    
    do {
    	if(maxDepth >= 0){
    		size = maxDepth;
    	} else {
    		size += 100;
    	}
    	
        threadInfo.vmOffsetStack.uiSize = size;
        threadInfo.vmOffsetStack.pStackEntries = (SStackEntry*)malloc(sizeof(SStackEntry) * threadInfo.vmOffsetStack.uiSize);
        retVal = GetThreadInfo(m_clientId, threadId, DR_VM_RELATIVE_STACK_TRACE, &threadInfo);
        if (retVal == MRTE_ERROR_PHASE_FAILURE) {
            stackTrace->uiSize = 0;
            stackTrace->pStackEntries = 0;
            free(threadInfo.vmOffsetStack.pStackEntries);
            return;
        } else if (retVal == MRTE_ERROR_BUFFER_TOO_SHORT) {
        	if(size == maxDepth)
        		break;
            free(threadInfo.vmOffsetStack.pStackEntries);
            continue;
        } else if (MRTE_FAILED(retVal)) {
            stackTrace->uiSize = 0;
            stackTrace->pStackEntries = 0;
            free(threadInfo.vmOffsetStack.pStackEntries);
            LOG_ERROR("GetThreadInfo failed: " << retVal);
            return;
        }
    } while (retVal == MRTE_ERROR_BUFFER_TOO_SHORT);

    ConvertStackTrace(&threadInfo, stackTrace);
    free(threadInfo.vmOffsetStack.pStackEntries);
}

void 
CProfEnv::ConvertStackTrace(SThreadInfo* threadInfo, SStackTrace_* stackTrace)
{
    stackTrace->pStackEntries = (SStackEntry_*)malloc(sizeof(SStackEntry_) * threadInfo->vmOffsetStack.uiActualSize);
    if (NULL == stackTrace->pStackEntries)
    {
        stackTrace->uiSize = 0;
        stackTrace->pStackEntries = 0;
        return;
    }

    stackTrace->uiSize = threadInfo->vmOffsetStack.uiActualSize;
    memset(stackTrace->pStackEntries, 0, sizeof(SStackEntry_) * stackTrace->uiSize);
    
    for (unsigned int i = 0; i < stackTrace->uiSize; i++) {
        stackTrace->pStackEntries[i].methodId = threadInfo->vmOffsetStack.pStackEntries[i].methodId;
        SMethodInfo* methodInfo = GetMethodData(stackTrace->pStackEntries[i].methodId);
        if (NULL != methodInfo)
        {
            stackTrace->pStackEntries[i].methodName = 
            	(char*)malloc(strlen(methodInfo->szMethodName) + 
            	strlen(methodInfo->szMethodSig) + 
            	strlen(methodInfo->szClassName) + 3);  // +3 for two separators '.' and ' ', and for end of string

            if (NULL != stackTrace->pStackEntries[i].methodName)
            {
                sprintf(stackTrace->pStackEntries[i].methodName, "%s.%s %s", 
            	    methodInfo->szClassName, methodInfo->szMethodName, methodInfo->szMethodSig);
            }
            
            stackTrace->pStackEntries[i].lineNumber = 
            	GetLineNumber(threadInfo->vmOffsetStack.pStackEntries[i].location, 
            	                &(methodInfo->managedToSrcLineMap));
        }
    }
}

void 
CProfEnv::FreeStackTrace(SStackTrace_* stackTrace)
{
    if (NULL == stackTrace || stackTrace->pStackEntries == NULL) {
        return;
    }
    for (unsigned int i = 0; i < stackTrace->uiSize; i++) {
    	if (NULL != stackTrace->pStackEntries[i].methodName)
    	{
    	    free(stackTrace->pStackEntries[i].methodName);
    	    stackTrace->pStackEntries[i].methodName = NULL;
    	}
    }
    free(stackTrace->pStackEntries);
    stackTrace->pStackEntries = NULL;
    stackTrace->uiSize = 0;
}

//==================================================================================
void 
CProfEnv::StoreThreadEvent(TId threadId, EventType type, void* data, U64 timeout, TId objectId)
{
    SThreadEventsData* evdata = NULL;

    if (m_pSThreadEventsData != 0) {
            evdata = (SThreadEventsData*)m_pSThreadEventsData->GetData(threadId);    
    }
    if (evdata == NULL) {
        evdata = new SThreadEventsData;
        memset(evdata, 0, sizeof(SThreadEventsData));
        evdata->isNewThread = true;
        evdata->threadId = threadId;

        if (type != THREAD_START_EVENT) {
            TResult iRes = GetThreadInfo(m_clientId, threadId, 
                    DR_THREAD_INFO, &(evdata->threadInfo));
            if (iRes != MRTE_RESULT_OK) {
                LOG_TRACE("Thread start event (GetThreadInfo wrong phase)");
                return;
            }
        }

    }

    evdata->type = type;
    evdata->timeout = timeout;
    switch (type) {
        case CONTENDED_MONITOR_ENTERED_EVENT:
            evdata->data.cedData = *((SContendedMonitorEnteredEventData*)(data));
            break;
        case CONTENDED_MONITOR_ENTER_EVENT:
            evdata->data.cerData = *((SContendedMonitorEnterEventData*)(data));
            break;
        case MONITOR_WAITED_EVENT:
            evdata->data.wedData = *((SMonitorWaitedEventData*)(data));
            break;
        case MONITOR_WAIT_EVENT:
            evdata->data.wetData = *((SMonitorWaitEventData*)(data));
            break;
        case THREAD_END_EVENT:
            evdata->data.tevData = *((SThreadEventData*)(data));
            break;
        case THREAD_START_EVENT:
            evdata->isNewThread = true;
            evdata->threadInfo = *((SThreadInfo*)(data));
            evdata->objectId = objectId;
            break;
    }

    m_pSThreadEventsDataLockObject->Enter();
    if (m_pSThreadEventsData == 0) {
        m_pSThreadEventsData = new CThreadEventsSet(threadId, evdata);
    } else {
        m_pSThreadEventsData->addNewData(threadId, evdata);    
    }
    m_pSThreadEventsDataLockObject->Leave();
}

void 
CProfEnv::PrintStoredThreadEvents()
{
    if (m_pSThreadEventsData != 0) {
        m_pSThreadEventsData->PrintStoredThreadEvents(this);
        delete m_pSThreadEventsData;
        m_pSThreadEventsData = 0;
    }
}

void
CProfEnv::DeleteStoredThreadEvents()
{
    if (m_pSThreadEventsData == 0) {
        return;
    }
    delete m_pSThreadEventsData;
    m_pSThreadEventsData = 0;
}

//================================================================================================
void 
CProfEnv::AddMethodLeaveData(TId threadId, TId methodId, TId classId, unsigned long ticket, U64 cpuTime)
{
    SMethodLeaveData_* methodLeaveData = new SMethodLeaveData_;
    methodLeaveData->threadId_ = threadId;
    methodLeaveData->methodId_ = methodId;
    methodLeaveData->classId_ = classId;
    methodLeaveData->ticket_ = ticket;
    methodLeaveData->cpuTime_ = cpuTime;
    m_pSSMethodLeaveData_LockObject->Enter();
    if (m_pSMethodLeaveData_ == 0) {
        m_pSMethodLeaveData_ = new CMethodLeaveDataSet(ticket, methodLeaveData);
    } else {
        m_pSMethodLeaveData_->addNewData(ticket, methodLeaveData);    
    }
    m_pSSMethodLeaveData_LockObject->Leave();
}


void 
CProfEnv::PrintMethodLeaveData() 
{
    m_pSSMethodLeaveData_LockObject->Enter();
    if (m_pSMethodLeaveData_ != 0) {
        m_pSMethodLeaveData_->PrintMethodLeaveElements(ec_env);
        delete m_pSMethodLeaveData_;
        m_pSMethodLeaveData_ = 0;
    }
    m_pSSMethodLeaveData_LockObject->Leave();
}

//================================================================================================
// Fix for 184834 [ThreadProf] crash when New Method event notification arrives after a method id is reported in a stack trace
TResult 
CProfEnv::GetThreadInfo(TId clientId, TId threadId, BitSet requestedDataTypes, SThreadInfo *pData)
{
    LOG_ASSERT(m_pMpiApi != 0);
    lockGetThreadInfoObject->Enter();
    TResult res = m_pMpiApi->GetThreadInfo(clientId, threadId, requestedDataTypes, pData);
    lockGetThreadInfoObject->Leave();
    return res;
}

TResult
CProfEnv::AddSupportedEG(EEventGroup group) 
{
    if (!IsSupportedEG(group)) {
        m_implementedEG.push_back(group);
    }
    return MRTE_RESULT_OK;
}

TResult 
CProfEnv::DelSupportedEG(EEventGroup group) 
{
    // Since each same EG appears just once, delete the first exist EG element
    vector<EEventGroup>::iterator pos = 
        find(m_implementedEG.begin(), m_implementedEG.end(), group);
    
    if (pos != m_implementedEG.end()) {
        m_implementedEG.erase(pos);
    }
    
    return MRTE_RESULT_OK;
}

bool
CProfEnv::IsSupportedEG(EEventGroup group) 
{
    return (m_implementedEG.end() != 
        find(m_implementedEG.begin(), m_implementedEG.end(), group));
}

TResult CProfEnv::EnableSupportedEG() {    
	return SetSupportedEG(&IMpi::EnableEventGroup);
}

TResult CProfEnv::DisableSupportedEG() {
    return SetSupportedEG(&IMpi::DisableEventGroup);
}

TResult CProfEnv::SetSupportedEG(SET_EG_FUNC pSetEGFunc)
{
	vector<EEventGroup>::iterator pos = m_implementedEG.begin();
	while ( pos != m_implementedEG.end() )
	{
	    TResult retVal = (m_pMpiApi->*pSetEGFunc)(m_clientId, *pos);	    
	    if (MRTE_FAILED(retVal)) {
            return retVal;
        }

	    ++pos;
	}
     
    return MRTE_RESULT_OK;
}

//==================================================================
// Add for bug 194081
CCGShadowStackTracker::CCGShadowStackTracker(CProfEnv& profEnv)
    : m_refProfEnv(profEnv)
{
    m_lockVMStackDataMap = CreateThreadSync();
}

CCGShadowStackTracker::~CCGShadowStackTracker()
{
    if (m_lockVMStackDataMap)
    {
        m_lockVMStackDataMap->Destroy();
    }
}

bool CCGShadowStackTracker::isNeedTrack()
{
    return ( 
            (m_refProfEnv.IsSupportedEG(EG_CALL_GRAPH)) && 
            !((m_refProfEnv.ec_env)->isStandAlone()) 
           );
}

bool CCGShadowStackTracker::ValidateCGShadowStackTracker()
{
    if (!isNeedTrack())
    {
        return true;
    }
    
    // You can add additional validate condition in this function
    return (NULL != m_lockVMStackDataMap);
}

void CCGShadowStackTracker::TrackVMShutdown()
{    
    if (!isNeedTrack())
    {
        return;
    }
    
    ReportLeftStackTrackDataMethodsLeave();
}

TResult CCGShadowStackTracker::InitNewThreadInfoArray(
    SThreadInfoArray& threadInfoArray, 
    int MAX_THREADS, int MAX_FRAMES)
{
    DestroyThreadInfoArray(threadInfoArray);
            
    threadInfoArray.pEntries = 
   	(SThreadInfoArrayEntry*)malloc(sizeof(SThreadInfoArrayEntry) * MAX_THREADS);
    if (NULL == threadInfoArray.pEntries)
    { 
        return MRTE_ERROR_OUT_OF_MEMORY;
    }

    threadInfoArray.uiSize = MAX_THREADS;
    memset(threadInfoArray.pEntries, 0, sizeof(SThreadInfoArrayEntry) * MAX_THREADS);
    
    unsigned int i = 0;
    for (i = 0; i < MAX_THREADS; i++)
    {
        threadInfoArray.pEntries[i].threadInfo.vmOffsetStack.uiSize = MAX_FRAMES;
        threadInfoArray.pEntries[i].threadInfo.vmOffsetStack.pStackEntries = new SStackEntry[MAX_FRAMES];

        if (NULL == threadInfoArray.pEntries[i].threadInfo.vmOffsetStack.pStackEntries)
        {
            DestroyThreadInfoArray(threadInfoArray);
            return MRTE_ERROR_OUT_OF_MEMORY;
        }

        memset(threadInfoArray.pEntries[i].threadInfo.vmOffsetStack.pStackEntries, 0, 
              	sizeof(SStackEntry) * MAX_FRAMES);
    }

    return MRTE_RESULT_OK;
}

void CCGShadowStackTracker::DestroyThreadInfoArray(
    SThreadInfoArray& threadInfoArray)
{
    if (NULL != threadInfoArray.pEntries)
    {
        for (unsigned int i = 0; i < threadInfoArray.uiSize; i++)
        {
            if (NULL != threadInfoArray.pEntries[i].threadInfo.vmOffsetStack.pStackEntries)
            {
                delete [](threadInfoArray.pEntries[i].threadInfo.vmOffsetStack.pStackEntries);
            }
        }

        free(threadInfoArray.pEntries);
    }
            
    threadInfoArray.uiSize = 0;
    threadInfoArray.uiActualSize = 0;
}

TResult CCGShadowStackTracker::GetInitialShadowStackTraces(
    SThreadInfoArray& threadInfoArray)
{
    // Suspend all VM threads and Get all threads stack traces
    TResult res = (m_refProfEnv.m_pMpiApi)->SuspendVM();
    if (MRTE_FAILED(res))
    {
        LOG_ERROR("Can not suspend VM when try to get shadow stack sanpshot")
        return res;
    } 
    
    // TODO: This fixed thread num and frame num is a limit, referring to original AttachEvent.cpp
    int MAX_THREADS = 100;
    int MAX_FRAMES = 100;
        
    res = InitNewThreadInfoArray(threadInfoArray, MAX_THREADS, MAX_FRAMES);
    if (MRTE_FAILED(res))
    {
        (m_refProfEnv.m_pMpiApi)->ResumeVM();
        return res;
    }

    res = (m_refProfEnv.m_pMpiApi)->GetAllThreadsInfo(m_refProfEnv.m_clientId, MAX_FRAMES, 
        DR_VM_RELATIVE_STACK_TRACE, &threadInfoArray);
    if (MRTE_FAILED(res))
    {
        (m_refProfEnv.m_pMpiApi)->ResumeVM();
        DestroyThreadInfoArray(threadInfoArray);
    }
    
    return res;
}

void CCGShadowStackTracker::ReportThreadsStart(SThreadInfoArray& threadInfoArray)
{    
    for (unsigned int threadIdx = 0; threadIdx < threadInfoArray.uiActualSize; threadIdx++)
    {
        SThreadInfoArrayEntry* pThread = &(threadInfoArray.pEntries[threadIdx]);
        
        // Clear outdated data
        m_refProfEnv.m_Tickets.ReleaseTicket(pThread->threadId);
        m_refProfEnv.m_Tickets.DeleteThread(pThread->threadId);
        
        // Add new data info
        m_refProfEnv.m_Tickets.AddNewThread(pThread->threadId);
                
        m_refProfEnv.GetThreadInfo(m_refProfEnv.m_clientId, pThread->threadId, 
            DR_THREAD_INFO | DR_THREAD_CPU_TIME, &(pThread->threadInfo));
        
        if (m_refProfEnv.m_profilerIsActive)
        {
            (m_refProfEnv.ec_env)->PrintThreadStartElement(pThread->threadId, 0, 
                &(pThread->threadInfo));
        }
        else
        {
            m_refProfEnv.StoreThreadEvent(pThread->threadId, THREAD_START_EVENT, 
                &(pThread->threadInfo), 0);            
        }
    }
}

void CCGShadowStackTracker::ReportMethodEnter(TId threadId, SThreadInfo* pThreadInfo, TId methodId)
{
    // Get classId according to methodId
    SMethodInfo methodInfo = {0};
    TResult iRes = (m_refProfEnv.m_pMpiApi)->GetMethodInfo(m_refProfEnv.m_clientId, 
        methodId, DR_CLASS_ID | DR_CLASS_NAME | DR_METHOD_NAME, &methodInfo);
    LOG_ASSERT(iRes == MRTE_RESULT_OK);
    LOG_ASSERT((methodInfo.validData & DR_CLASS_ID) != 0);
    LOG_ASSERT((methodInfo.validData & DR_CLASS_NAME) != 0);
    LOG_ASSERT((methodInfo.validData & DR_METHOD_NAME) != 0);

    if (m_refProfEnv.IsExcludedClassMethod(methodInfo.szClassName, methodInfo.szMethodName))
    {
        return;
    }
    
    m_refProfEnv.CheckMethodId(methodId);

    if ((m_refProfEnv.ec_env)->isCGExecDetails()) {
        unsigned long ticket = m_refProfEnv.m_Tickets.GetNewTicket(threadId, 
            pThreadInfo->uiCpuNanos, true); // m_refProfEnv.m_profilerIsActive = true
        unsigned long stackDepth = m_refProfEnv.m_Tickets.GetStackDepth(threadId);
        (m_refProfEnv.ec_env)->PrintMethodEntryElement(threadId, methodId,
            methodInfo.classId, ticket, stackDepth);
    }
    else
    {
        m_refProfEnv.AddNewStackEntry(threadId, methodId, pThreadInfo->uiCpuNanos, m_refProfEnv.m_profilerIsActive);
    }
}

void CCGShadowStackTracker::ReportMethodsEnter(SThreadInfoArray& threadInfoArray)
{    
    for (unsigned int threadIdx = 0; threadIdx < threadInfoArray.uiActualSize; threadIdx++)
    {
        TId threadId = threadInfoArray.pEntries[threadIdx].threadId;
        SThreadInfo* pThreadInfo = &(threadInfoArray.pEntries[threadIdx].threadInfo);
        
        for (int stackTraceIdx = (pThreadInfo->vmOffsetStack).uiActualSize - 1; stackTraceIdx >= 0 ; 
            stackTraceIdx--)
        {
            // Get methodId from stack trace info
            SStackEntry* pStackEntry = &(pThreadInfo->vmOffsetStack.pStackEntries[stackTraceIdx]);
            
            ReportMethodEnter(threadId, pThreadInfo, pStackEntry->methodId);
        }
    }
}

void CCGShadowStackTracker::SaveAllShadowStacksMethods(SThreadInfoArray& threadInfoArray)
{
    for (unsigned int threadIdx = 0; threadIdx < threadInfoArray.uiActualSize; threadIdx++)
    {
        SThreadInfoArrayEntry* pThread = &(threadInfoArray.pEntries[threadIdx]);
        CVMStackTrackData vmStacks;
        vmStacks.threadId = pThread->threadId;
        unsigned int stackTraceSize = 
            (pThread->threadInfo).vmOffsetStack.uiActualSize;

        for (int stackTraceIdx = stackTraceSize - 1; stackTraceIdx >= 0; stackTraceIdx--)
        {
            vmStacks.shadowStack.push(
                (pThread->threadInfo).vmOffsetStack.pStackEntries[stackTraceIdx].methodId);
        }

        m_VMStacktraceDataMap.insert(
          std::pair<MPI::TId, CVMStackTrackData>(vmStacks.threadId, vmStacks));
    }
}

void CCGShadowStackTracker::TrackECAttach()
{    
    if (!isNeedTrack())
    {
        return;
    }
    
    SThreadInfoArray threadInfoArray = {0};
    TResult res = GetInitialShadowStackTraces(threadInfoArray);
    if (MRTE_FAILED(res))
    {
        LOG_ERROR("Can not get VM thread stack traces info.")
        return;
    }
    
    // Report related events and get initial shadow stacks
    ReportThreadsStart(threadInfoArray);

    ReportMethodsEnter(threadInfoArray);
    
    SaveAllShadowStacksMethods(threadInfoArray);
    
    DestroyThreadInfoArray(threadInfoArray);
    (m_refProfEnv.m_pMpiApi)->ResumeVM();
}

void CCGShadowStackTracker::TrackECDetach()
{
    if (!isNeedTrack())
    {
        return;
    }
    
    ReportLeftStackTrackDataMethodsLeave();
}

void CCGShadowStackTracker::ReportMethodLeave(TId threadId, TId methodId, 
    TTimeStamp uiCpuNanos)
{
    SMethodInfo methodInfo;
    TResult iRes = (m_refProfEnv.m_pMpiApi)->GetMethodInfo(m_refProfEnv.m_clientId, 
        methodId, DR_CLASS_ID | DR_CLASS_NAME | DR_METHOD_NAME, &methodInfo);
    LOG_ASSERT(iRes == MRTE_RESULT_OK);
    LOG_ASSERT((methodInfo.validData & DR_CLASS_ID) != 0);
    LOG_ASSERT((methodInfo.validData & DR_CLASS_NAME) != 0);
    LOG_ASSERT((methodInfo.validData & DR_METHOD_NAME) != 0);

    if (m_refProfEnv.IsExcludedClassMethod(methodInfo.szClassName, methodInfo.szMethodName))
    {
        return;
    }
    
    if ((m_refProfEnv.ec_env)->isCGExecDetails()) {
    	// Workaround for bug 293756
    	U64 cpuTime = 0;  // uiCpuNanos - m_refProfEnv.m_Tickets.GetCpuTime(threadId);
        unsigned long ticket = m_refProfEnv.m_Tickets.ReleaseTicket(threadId);
        (m_refProfEnv.ec_env)->PrintMethodExitElement(threadId, methodId, 
            methodInfo.classId, ticket, cpuTime); 
    }
    else
    {
    	// Workaround for bug 293756
    	//m_refProfEnv.FinalizeStackEntry(threadId, uiCpuNanos);
    	m_refProfEnv.FinalizeStackEntry(threadId, 0);

    }
}

void CCGShadowStackTracker::UpdateShadowStack(stack< TId >& tmpShadowStack,
    stack< TId >& shadowStack, const SMethodEnterEventData& data)
{ 
    // Because normal stack is empty now, the tmpShadowStack must less or equal shadowStack
    while (tmpShadowStack.size() < shadowStack.size())
    {// Geneate method leave events for shadow stack method
        TId methodId = shadowStack.top();
        shadowStack.pop();
        ReportMethodLeave(data.threadId, methodId, data.uiCpuNanos);
    }
}

void CCGShadowStackTracker::TrackMethodEnter(SMethodEnterEventData &data)
{
    if (!isNeedTrack())
    {
        return;
    }
    
    m_lockVMStackDataMap->Enter();
    map< MPI::TId, CVMStackTrackData >::iterator pos = m_VMStacktraceDataMap.find(data.threadId);
    
    if (pos == m_VMStacktraceDataMap.end())
    {// Enter a new thread after attach, it has no shadow stack record and can be handled normally
        m_lockVMStackDataMap->Leave();
        return;
    }
    
    CVMStackTrackData& refVmStacks = pos->second;
    
    if (refVmStacks.normalStack.empty())
    {
        SStackTrace_ stackTrace = {0};
        
        m_refProfEnv.GetStackTrace(data.threadId, &stackTrace);
        
        std::stack< MPI::TId > tmpShadowStack;
        // Ignore the new entered method, CGProxy.EarlyMethodEnter and CGProxy.MethodEnter on the stack top
        const int ignoredMethodNum = 3;
        for (int stackTraceIdx = stackTrace.uiSize - 1; stackTraceIdx > ignoredMethodNum - 1; stackTraceIdx--)
        {
            tmpShadowStack.push(stackTrace.pStackEntries[stackTraceIdx].methodId);
        }
        
        UpdateShadowStack(tmpShadowStack, refVmStacks.shadowStack, data);

        m_refProfEnv.FreeStackTrace(&stackTrace);
    }
    
    refVmStacks.normalStack.push(data.methodId);
    m_lockVMStackDataMap->Leave();
}

// return value: If continue to handle the MethodLeave events
bool CCGShadowStackTracker::TrackMethodLeave(SMethodLeaveEventData &data)
{
    if (!isNeedTrack())
    {
        return true;
    }   
    
    m_lockVMStackDataMap->Enter();
    map< MPI::TId, CVMStackTrackData >::iterator pos = m_VMStacktraceDataMap.find(data.threadId);
    
    if (pos == m_VMStacktraceDataMap.end())
    {// Enter a new thread after attach, it has no shadow stack record and can be handled normally
        m_lockVMStackDataMap->Leave();
        return true;
    }
    
    CVMStackTrackData& refVmStacks = pos->second;
    
    if (refVmStacks.normalStack.empty())
    {// Method leave notification is generated by instrumet before reattach, ignore it
        m_lockVMStackDataMap->Leave();
        return false;
    }
    else
    {
        refVmStacks.normalStack.pop();
        m_lockVMStackDataMap->Leave();
        return true;
    }
}

void CCGShadowStackTracker::TrackThreadEnd(SThreadEventData &data)
{
    if (!isNeedTrack())
    {
        return;
    }

    m_lockVMStackDataMap->Enter();
    map< MPI::TId, CVMStackTrackData >::iterator pos = m_VMStacktraceDataMap.find(data.threadId);
    
    if (pos == m_VMStacktraceDataMap.end())
    {// Enter a new thread after attach, it has no shadow stack record and can be handled normally
        m_lockVMStackDataMap->Leave();
        return;
    }
    
    CVMStackTrackData& refVmStacks = pos->second;
    TTimeStamp cpuNanos = GetThreadCPUTime(data.threadId);
    
    // refVmStacks.shadowStack report method leave events
    while (!refVmStacks.shadowStack.empty())
    {
        TId methodId = refVmStacks.shadowStack.top();
        refVmStacks.shadowStack.pop();
        ReportMethodLeave(data.threadId, methodId, cpuNanos);
    }
    m_lockVMStackDataMap->Leave();
}

void CCGShadowStackTracker::ReportLeftStackTrackDataMethodsLeave()
{
    m_lockVMStackDataMap->Enter();
    
    map< MPI::TId, CVMStackTrackData >::iterator pos = m_VMStacktraceDataMap.begin();
    for(; pos != m_VMStacktraceDataMap.end(); pos++)
    {
        CVMStackTrackData& refVmStacks = pos->second;
        TTimeStamp cpuNanos = GetThreadCPUTime(pos->first);
        
        while (!refVmStacks.normalStack.empty())
        {
            TId methodId = refVmStacks.normalStack.top();
            refVmStacks.normalStack.pop();
            ReportMethodLeave(pos->first, methodId, cpuNanos);
        }
        
        while (!refVmStacks.shadowStack.empty())
        {
            TId methodId = refVmStacks.shadowStack.top();
            refVmStacks.shadowStack.pop();
            ReportMethodLeave(pos->first, methodId, cpuNanos);
        }
    }
    m_lockVMStackDataMap->Leave();
}
// Add for bug 194081

TTimeStamp CCGShadowStackTracker::GetThreadCPUTime(TId threadId)
{
    SThreadInfo threadInfo;
    m_refProfEnv.GetThreadInfo(m_refProfEnv.m_clientId, threadId, 
        DR_THREAD_CPU_TIME, &threadInfo);
    return threadInfo.uiCpuNanos;
}
