/************************************************************************
 * Copyright (c) 2006, 2008 Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *    Viacheslav Rybalov, Intel - Initial API and implementation
 *
 * $Id: TIdSet.h,v 1.21 2008/06/16 21:02:25 jkubasta Exp $ 
 ************************************************************************/

#ifndef _TID_SET_H_
#define _TID_SET_H_

#include "MpiAPI.h"
#include "OSA.h"
#include "EC_Env.h"
#include "log.h"
#include <stdlib.h>

namespace Martini { namespace BaseProf {

    class CProfEnv;

    class CTIdSet {
    public:
        CTIdSet(MPI::TId tId_Key, void* pData) {
            m_TId_Key = tId_Key;
            m_pData = pData;
            m_isPrinted = true;
            m_less = 0;
            m_lessDepth = 0;
            m_more = 0;
            m_moreDepth = 0;
        }

        CTIdSet(MPI::TId tId_Key, void* pData, bool isPrinted) {
            m_TId_Key = tId_Key;
            m_pData = pData;
            m_isPrinted = isPrinted;
            m_less = 0;
            m_lessDepth = 0;
            m_more = 0;
            m_moreDepth = 0;
        }


        virtual ~CTIdSet() {
            if (m_pData != 0) {
                free(m_pData);
            }
            if (m_less != 0) {
                delete m_less;
            }
            if (m_more != 0) {
                delete m_more;
            }
        }

        int GetCurrentDepth() {
            return m_lessDepth > m_moreDepth ? m_lessDepth : m_moreDepth;
        }

        void BalanceSubTrees() {
            if (m_less != 0) {
                if ((m_less->m_lessDepth - m_less->m_moreDepth) > 1) {
                    CTIdSet* tmp = m_less;
                    m_less = tmp->m_less;
                    tmp->m_less = m_less->m_more;
                    tmp->m_lessDepth = tmp->m_less ? tmp->m_less->GetCurrentDepth() + 1 : 0;
                    m_less->m_more = tmp;
                    m_less->m_moreDepth = m_less->m_more->GetCurrentDepth() + 1;
                } else if ((m_less->m_moreDepth - m_less->m_lessDepth) > 1) {
                    CTIdSet* tmp = m_less;
                    m_less = tmp->m_more;
                    tmp->m_more = m_less->m_less;
                    tmp->m_moreDepth = tmp->m_more ? tmp->m_more->GetCurrentDepth() + 1 : 0;
                    m_less->m_less = tmp;
                    m_less->m_lessDepth = m_less->m_less->GetCurrentDepth() + 1;
                }
                m_lessDepth = m_less->GetCurrentDepth() + 1;
            }
            if (m_more != 0) {
                if ((m_more->m_lessDepth - m_more->m_moreDepth) > 1) {
                    CTIdSet* tmp = m_more;
                    m_more = tmp->m_less;
                    tmp->m_less = m_more->m_more;
                    tmp->m_lessDepth = tmp->m_less ? tmp->m_less->GetCurrentDepth() + 1 : 0;
                    m_more->m_more = tmp;
                    m_more->m_moreDepth = m_more->m_more->GetCurrentDepth() + 1;
                } else if ((m_more->m_moreDepth - m_more->m_lessDepth) > 1) {
                    CTIdSet* tmp = m_more;
                    m_more = tmp->m_more;
                    tmp->m_more = m_more->m_less;
                    tmp->m_moreDepth = tmp->m_more ? tmp->m_more->GetCurrentDepth() + 1 : 0;
                    m_more->m_less = tmp;
                    m_more->m_lessDepth = m_more->m_less->GetCurrentDepth() + 1;
                }
                m_moreDepth = m_more->GetCurrentDepth() + 1;
            }
        }

        int addNewData(MPI::TId tId_Key, void* pData) {
            return addNewData(tId_Key, pData, true);
        }

        int addNewData(MPI::TId tId_Key, void* pData, bool isPrinted) {
            if (m_TId_Key == tId_Key) {
/*  quick patch for bug 187006 ([CGProf] Crash on multi-core platforms when printing <methodDef> element)
    it may cause memory leaks while bug 168531 (JVMTI CG profiler. Martini produces multiple NewMethod events) is not fixed */
//                free(m_pData);
                m_pData = pData;
                m_isPrinted = isPrinted;
                return GetCurrentDepth();
            } else if (tId_Key < m_TId_Key) {
                if (m_less == 0) {
                    m_less = new CTIdSet(tId_Key, pData);
                    m_less->m_isPrinted = isPrinted;
                    m_lessDepth = 1;
                    return 1;
                }
                m_lessDepth = m_less->addNewData(tId_Key, pData, isPrinted) + 1;
            } else {
                if (m_more == 0) {
                    m_more = new CTIdSet(tId_Key, pData);
                    m_more->m_isPrinted = isPrinted;
                    m_moreDepth = 1;
                    return 1;
                }
                m_moreDepth = m_more->addNewData(tId_Key, pData, isPrinted) + 1;
            }
            BalanceSubTrees();
            return GetCurrentDepth();
        }

        bool IsIdAdded(MPI::TId tId_Key) {
            if (m_TId_Key == tId_Key) {
                return true;
            } else if (tId_Key < m_TId_Key) {
                if (m_less != 0) {
                    return m_less->IsIdAdded(tId_Key);
                }
            } else {
                if (m_more != 0) {
                    return m_more->IsIdAdded(tId_Key);
                }
            }
             return false;
        }

        void CleanPrintFlags() {
            m_isPrinted = false;
            if (m_less != 0) {
                m_less->CleanPrintFlags() ;
            }
            if (m_more != 0) {
                m_more->CleanPrintFlags() ;
            }
        }

        void* GetData(MPI::TId tId_Key) {
            if (m_TId_Key == tId_Key) {
                return m_pData;
            } else if (tId_Key < m_TId_Key) {
                if (m_less != 0) {
                    return m_less->GetData(tId_Key);
                }
            } else {
                if (m_more != 0) {
                    return m_more->GetData(tId_Key);
                }
            }
            return 0;
        }

    protected:
        MPI::TId m_TId_Key;
        void* m_pData;
        bool m_isPrinted;
        CTIdSet* m_less;
        int m_lessDepth;
        CTIdSet* m_more;
        int m_moreDepth;
    };

//==============================================================================

    class CMethodDefDataSet : public CTIdSet {
    public:
        CMethodDefDataSet(MPI::TId tId_Key, void* pData, bool isPrinted) : CTIdSet (tId_Key, pData, isPrinted) {};
        void PrintMethodDefElements(JPIAgent::EC_Env* ec_env);
        //bool IsMethodPrinted(MPI::TId tId_Key, JPIAgent::EC_Env* ec_env);
        bool IsMethodStored(MPI::TId tId_Key);
    };

    class CClassDefDataSet : public CTIdSet {
    public:
        CClassDefDataSet(MPI::TId tId_Key, void* pData, bool isPrinted) : CTIdSet (tId_Key, pData, isPrinted) {};
        void PrintClassDefElements(JPIAgent::EC_Env* ec_env);
        //bool IsClassPrinted(MPI::TId tId_Key, JPIAgent::EC_Env* ec_env);
        bool IsClassStored(MPI::TId tId_Key);
    };

    class CObjectDataSet : public CTIdSet {
    public:
        CObjectDataSet(MPI::TId tId_Key, void* pData, bool isPrinted) : CTIdSet (tId_Key, pData, isPrinted) {};
        void PrintObjAllocElements(CProfEnv* env);
        bool IsObjectStored(MPI::TId tId_Key);
    };

    class CMethodLeaveDataSet : public CTIdSet {
    public:
        CMethodLeaveDataSet(MPI::TId tId_Key, void* pData) : CTIdSet (tId_Key, pData) {};
        void PrintMethodLeaveElements(JPIAgent::EC_Env* ec_env);
    };

    class CThreadEventsSet : public CTIdSet {
    public:
        CThreadEventsSet(MPI::TId tId_Key, void* pData) : CTIdSet (tId_Key, pData) {};
        void PrintStoredThreadEvents(CProfEnv* env);
    };

} /*namespace Martini*/ } /*namespace BaseProf*/

#endif // _TID_SET_H_
