/************************************************************************
 * Copyright (c) 2006, 2011 Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *    Viacheslav Rybalov, Intel - Initial API and implementation
 *    Jonathan West, IBM - Rewrote aggregated stats portion
 *
 * $Id: Tickets.cpp,v 1.9 2011/02/03 19:17:11 jwest Exp $ 
 ************************************************************************/

#include "Tickets.h"
#include "log.h"

using namespace Martini::BaseProf;
using namespace Martini::MPI;
using namespace Martini::OSA;


// ======= Aggregated Code =============================================

//======= CStackEntry ============================================================================

CStackEntry::~CStackEntry() {


	m_pParentStack->SetNumEntries(m_pParentStack->GetNumEntries()-1);

	// Call the destructor on all the elements of the map (clear does not do this for pointers, so we do it ourselves)
	std::map<TId, CStackEntry*>::iterator it;
    for ( it = m_pInvokedChildren->begin() ; it != m_pInvokedChildren->end(); it++ ) {
    	CStackEntry* entry = (*it).second;
    	delete entry;
    }

    // Clear the map itself
	m_pInvokedChildren->clear();

	delete(m_pInvokedChildren);

}


/** This method is called when a Java method has exited. This has the effect of calculating data
 * for this exiting method event. The call of this function will then pop it off the stack. */
CStackEntry* CStackEntry::FinalizeStackEntry(U64 uiCpuNanos) {

	// Sanity check the call, we shouldn't finalize on StackHead
	if(IsStackHead()) {
		LOG_TRACE("FinalizeStackEntry called on stack head");
		return this;
	}

	// Modified isActive condition for bug 284502. We need update track methods info
    //if(isActive){

	// Update Count
	m_count++;

    // Update Base Time
    U64 cumulativeTime = m_ec_env->GetTime() - m_EnterTime;
    U64 baseTime = cumulativeTime - m_InvokeTime;
    m_baseTime += baseTime;

    // Update Base CPU TIme
    U64 cumulativeCpuTime = uiCpuNanos - m_EnterCpuTime;
    U64 baseCpuTime = cumulativeCpuTime - m_InvokeCpuTime;
    m_baseCPUTime += baseCpuTime;

    // Invoke time is the time that it took to invoke methods that you called (eg child methods that were at one point on the stack below you)
    if (m_pParentEntry != 0 && !m_pParentEntry->IsStackHead()) {
    	m_pParentEntry->IncreaseInvokeTime(cumulativeTime, cumulativeCpuTime);
    }
    if (baseTime > m_maxTime) {
        m_maxTime = baseTime;
    }
    if (baseTime < m_minTime) {
        m_minTime = baseTime;
    }

    m_updated = true;

	// Ensure that every parent of this entry has update set to true
	CStackEntry *parent = m_pParentEntry;
	while(parent != NULL && parent->m_updated == false) {
		parent->m_updated = true;
		parent = parent->m_pParentEntry;
	}

    isActive = false;

    return m_pParentEntry;

}

/** Called by FinalizeStackEntry on the parent caller of a method on the stack*/
void CStackEntry::IncreaseInvokeTime(U64 invokeTime, U64 invokeCpuTime) {
    m_InvokeTime += invokeTime;
    m_InvokeCpuTime += invokeCpuTime;
}

/** Print the given stack entry (and the methods it calls) */
void CStackEntry::PrintStack(TId threadId) {
	// If this entry (or any of its children) hasn't been updated, then no new data is available,  so return
	if(m_updated == false) return;

	// Don't print stack head, it should contain no stats data
	if(!this->IsStackHead()) {
		m_ec_env->PrintAgMethodEntryElement(threadId, m_MethodId, m_baseTime, m_minTime, m_maxTime, m_baseCPUTime, m_count);
	}

	// Call printstack for each of the child calls
	std::map<TId, CStackEntry*>::iterator it;
	for ( it = m_pInvokedChildren->begin() ; it != m_pInvokedChildren->end(); it++ ) {
		CStackEntry* entry = (*it).second;
		entry->PrintStack(threadId);
	}

	// Don't print stack head
	if(!this->IsStackHead()) {
		m_ec_env->PrintAgMethodExitElement(threadId, m_MethodId);

		// reset all counts and stats, as they have been reported
		m_count = 0;

		m_baseTime = 0;
		m_baseCPUTime = 0;
		m_maxTime = 0;

		#ifdef _WIN32
			m_minTime = 0xFFFFFFFFFFFFFFFF;
		#else
			m_minTime = 0xFFFFFFFFFFFFFFFFULL;
		#endif

	}

	// Reset updated for all printed entries
	m_updated = false;

}

/** Delete all the child methods in the subtree that DO NOT match the given stackEntryId method id,
 * and then call the parent of the current entry, with the method id of the current entry.
 *
 * This method should only be called on the entry that is at the top of the stack, with parameter 0.
 * The purpose of this method is then to ensure that all items are removed and freed from the tree
 * _except_ those that are still required by the method currently executing on the tree and its parents. */
void CStackEntry::DeleteTreeNonMatchingStackEntry(TId stackEntryId) {

	// Erase all child methods except the one specified by the param (or if param is 0 all will be deleted)
	std::map<TId, CStackEntry*>::iterator it;
	for ( it = m_pInvokedChildren->begin() ; it != m_pInvokedChildren->end();  ) {
		CStackEntry* entry = (*it).second;

		if(stackEntryId == 0 || entry->m_MethodId != stackEntryId) {
			m_pInvokedChildren->erase(it++);
			delete entry;
		} else {
			it++;
		}
	}

	// Call this entry's parent, with the id of this entry (and thus we move up the tree)
	if(!this->IsStackHead() && m_pParentEntry != NULL) {
    	m_pParentEntry->DeleteTreeNonMatchingStackEntry(m_MethodId);
    }
}

/** This function is called by CStackHead::AddStackEntry(), and is called when a Java method is
 * entered. This function adds the given method to the top of the stack, and is called only on the
 * entry that is currently at the top of the stack. */
CStackEntry* CStackEntry::AddStackEntry(TId methodId, U64 uiCpuNanos, bool is_active) {

	/** If the given entry is already in our map, because it was called previously, use it*/
	CStackEntry *entry;
	std::map<TId, CStackEntry*>::iterator it = m_pInvokedChildren->find(methodId);

	if(it == m_pInvokedChildren->end()) {

		// Note that this will in fact make stackhead the parent of the top most entries in the tree. This is acceptable.
		entry = new CStackEntry(m_pParentStack, this, m_ec_env, methodId, uiCpuNanos, is_active);
		m_pInvokedChildren->insert(std::pair<TId, CStackEntry *>(methodId, entry));

		m_pParentStack->SetNumEntries(m_pParentStack->GetNumEntries()+1);

	} else {
		entry = it->second;

		entry->m_EnterTime = m_ec_env->GetTime();
        entry->m_InvokeTime = 0;
        entry->m_EnterCpuTime = uiCpuNanos;
        entry->m_InvokeCpuTime = 0;
        entry->isActive = is_active;

	}

	return entry;
}


//======= CStackHead  ============================================================================


/** Called when a Java method is entered. */
void CStackHead::AddStackEntry(TId methodId, U64 uiCpuNanos, bool is_active) {
	lockThreadObject->Enter();

	if(m_pCurrEntry == NULL) {
		// If curr entry isn't defined yet, add to stack head
		m_pCurrEntry = m_pStackHead->AddStackEntry(methodId, uiCpuNanos, is_active);
	} else {
		// otherwise, add to whatever is currently at the top of the stack (most recently entered method)
		m_pCurrEntry = m_pCurrEntry->AddStackEntry(methodId, uiCpuNanos, is_active);
	}

    lockThreadObject->Leave();
}

/** Called when a Java method is exited, the top of the stack is assumed to be the exiting method*/
void CStackHead::FinalizeStackEntry(U64 uiCpuNanos) {
	lockThreadObject->Enter();

	// Finalize the top of the stack, and replace the top with its parent
	if(m_pCurrEntry != 0 && !m_pCurrEntry->IsStackHead()) {
		m_pCurrEntry = m_pCurrEntry->FinalizeStackEntry(uiCpuNanos);
	}

	// If current entry is stack head, then reset it to 0, as a precaution
	if(m_pCurrEntry != 0 && m_pCurrEntry->IsStackHead()) {
		m_pCurrEntry = 0;
	}

    lockThreadObject->Leave();
}

/** Flush the current method data in the tree, and reset/free the data as needed */
void CStackHead::PrintAggCallGraph() {

	// Recursively call the other stacks in the list first
    if (m_pNext != 0) {
        m_pNext->PrintAggCallGraph();
    }

    lockThreadObject->Enter();

    m_pStackHead->PrintStack(m_ThreadId);

	if(m_pCurrEntry != NULL) {
	    // The purpose of this call is to ensure that all items are removed and freed from the tree
	    // _except_ those that are still required by the method currently executing on the tree and its parents.
		m_pCurrEntry->DeleteTreeNonMatchingStackEntry(0);
	}

    lockThreadObject->Leave();
}


// ======= Non-aggregated Code =============================================

//======= CTicketStack ====================================================================================

void 
CTicketStack::PrintMethodEntryElements(Martini::JPIAgent::EC_Env* ec_env)
{
    if (m_next != 0) {
        m_next->PrintMethodEntryElements(ec_env);
    }
    if (m_isActive) {
        return;
    }
    m_isActive = true;
    ec_env->PrintMethodEntryElement(methodInvocationData.threadId_, methodInvocationData.methodId_, 
        methodInvocationData.classId_, methodInvocationData.ticket_, methodInvocationData.stackDepth_);
}

//======================= CThreadSet ==========================================
CThreadSet::CThreadSet(TId threadId, CThreadSet** previous) {
    m_threadId = threadId;
    m_next = 0;
    m_previous = previous;
    m_pTicketStack = 0;
    m_stackDepth = 0;
}

CThreadSet::~CThreadSet() {
    if (m_next != 0) {
        delete m_next;
    }
    if (m_pTicketStack != 0) {
        delete m_pTicketStack;
    }
}

void 
CThreadSet::AddNewThreadId(TId threadId) {
    if (m_threadId == threadId) {
        return;
    }
    if (m_next == 0) {
        m_next = new CThreadSet(threadId, &m_next);
        return;
    }
    m_next->AddNewThreadId(threadId);
}

void 
CThreadSet::DeleteThread(TId threadId) {
    if (m_threadId == threadId) {
        *m_previous = m_next;
        if (m_next != 0) {
            m_next->m_previous = m_previous;
        }
        m_next = 0;
        delete this;
    } else {
        if (NULL != m_next) {
            m_next->DeleteThread(threadId);
        }
    }
}

void 
CThreadSet::AddNewTicket(unsigned long ticket, TId threadId, U64 cpuTime, bool isActive) {
    if (m_threadId == threadId) {
        m_pTicketStack = new CTicketStack(ticket, cpuTime, isActive, m_pTicketStack, &m_pTicketStack);
        m_stackDepth++;
    } else {
        if (NULL != m_next) {
            m_next->AddNewTicket(ticket, threadId, cpuTime, isActive);
        }
    }
}

U64 
CThreadSet::GetCpuTime(TId threadId) {
    if (m_threadId == threadId) {
        if (m_pTicketStack != 0) {
            return m_pTicketStack->GetCpuTime();
        }
    } else {
        if (NULL != m_next) {
            return m_next->GetCpuTime(threadId);
        }
    }
    return static_cast<U64>(-1);
}

unsigned long 
CThreadSet::ReleaseTicket(TId threadId) {
    if (m_threadId == threadId) {
        m_stackDepth--;
        if (m_pTicketStack != 0) {
            return m_pTicketStack->ReleaseTicket();
        }
    } else {
        if (NULL != m_next) {
            return m_next->ReleaseTicket(threadId);
        }
    }
    return static_cast<unsigned long>(-1);
}

unsigned long
CThreadSet::GetStackDepth(TId threadId)
{
    if (m_threadId == threadId) {
        return m_stackDepth;
    } else {
        if (NULL != m_next) {
            return m_next->GetStackDepth(threadId);
        }
    }
    return static_cast<unsigned long>(-1);
}

void 
CThreadSet::AddMethodEnterData(unsigned long ticket, TId threadId, TId methodId, TId classId, U64 cpuTime) {
    if (m_threadId == threadId) {
        m_pTicketStack = new CTicketStack(ticket, cpuTime, false, m_pTicketStack, &m_pTicketStack);
        m_stackDepth++;
        m_pTicketStack->AddMethodEnterData(threadId, methodId, classId, m_stackDepth);
    } else {
        if (NULL != m_next) {
            m_next->AddMethodEnterData(ticket, threadId, methodId, classId, cpuTime);
        }
    }
}

void 
CThreadSet::PrintMethodEntryElements(Martini::JPIAgent::EC_Env* ec_env)
{
    if (m_pTicketStack != 0) {
        m_pTicketStack->PrintMethodEntryElements(ec_env);
    }
    if (m_next != 0) {
        m_next->PrintMethodEntryElements(ec_env);
    }

}

//======================= CTickets ==========================================
CTickets::CTickets(void)
{
    m_lockObject = Martini::OSA::CreateThreadSync();
    m_pThreadSet = 0;
    m_currentTicket = 0;
}

CTickets::~CTickets(void)
{
    m_lockObject->Enter();
    if (m_pThreadSet != 0) {
        delete m_pThreadSet;
    }
    m_lockObject->Leave();
    m_lockObject->Destroy();
}

void 
CTickets::AddNewThread(TId threadId) 
{
    m_lockObject->Enter();
    if (m_pThreadSet == 0) {
        m_pThreadSet = new CThreadSet(threadId, &m_pThreadSet); 
    } else {
        m_pThreadSet->AddNewThreadId(threadId);
    }
    m_lockObject->Leave();
}

void 
CTickets::DeleteThread(TId threadId)
{
    m_lockObject->Enter();
    m_pThreadSet->DeleteThread(threadId);
    m_lockObject->Leave();
}

unsigned long 
CTickets::GetNewTicket(TId threadId, U64 cpuTime, bool isActive)
{
    m_lockObject->Enter();
    AddNewThread(threadId);
    m_pThreadSet->AddNewTicket(m_currentTicket, threadId, cpuTime, isActive);
    unsigned long ret = m_currentTicket++;
    m_lockObject->Leave();
    return ret;
}

U64 
CTickets::GetCpuTime(TId threadId)
{
    U64 ret;
    m_lockObject->Enter();
    if (m_pThreadSet != 0) {
        ret = m_pThreadSet->GetCpuTime(threadId);
    } else {
        ret = static_cast<unsigned long>(-1);
    }
    m_lockObject->Leave();
    return ret;
}

unsigned long
CTickets::ReleaseTicket(TId threadId)
{
    unsigned long ret;
    m_lockObject->Enter();
    if (m_pThreadSet != 0) {
        ret = m_pThreadSet->ReleaseTicket(threadId);
    } else {
        ret = static_cast<unsigned long>(-1);
    }
    m_lockObject->Leave();
    return ret;
}

unsigned long
CTickets::GetStackDepth(TId threadId)
{
    unsigned long ret;
    m_lockObject->Enter();
    if (m_pThreadSet != 0) {
        ret = m_pThreadSet->GetStackDepth(threadId);
    } else {
        ret = static_cast<unsigned long>(-1);
    }
    m_lockObject->Leave();
    return ret;
}

void 
CTickets::AddMethodEnterData(TId threadId, TId methodId, TId classId, U64 cpuTime)
{
    m_lockObject->Enter();
    AddNewThread(threadId);
    m_pThreadSet->AddMethodEnterData(m_currentTicket, threadId, methodId, classId, cpuTime);
    m_currentTicket++;
    m_lockObject->Leave();
}

void 
CTickets::PrintMethodEntryElements(Martini::JPIAgent::EC_Env* ec_env)
{
    m_lockObject->Enter();
    if (m_pThreadSet != 0) {
        m_pThreadSet->PrintMethodEntryElements(ec_env);
    }
    m_lockObject->Leave();
}
