/************************************************************************
 * Copyright (c) 2006, 2011 Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *    Viacheslav Rybalov, Intel - Initial API and implementation
 *    Jonathan West, IBM - Rewrote aggregated stats portion
 *    
 * $Id: Tickets.h,v 1.17 2011/02/04 21:58:27 jwest Exp $
 ************************************************************************/

#ifndef _TICKETS_H_
#define _TICKETS_H_

#include <stack>
#include <map>

#include "MpiAPI.h"
#include "OSA.h" 
#include "InStructs.h"
#include "EC_Env.h"



namespace Martini { namespace BaseProf {

//======= Non-Aggregated CG profiler ==================================================================
//======= CTicketStack ================================================================================

    class CTicketStack {
    public:
        CTicketStack(unsigned long ticket, U64 cpuTime, bool isActive, CTicketStack* next, CTicketStack** previous) {
            m_ticket = ticket;
            m_enterCpuTime = cpuTime;
            m_isActive  = isActive; 
            m_next = next;
            m_previous = previous;
        }

        ~CTicketStack() {
            if (m_next != 0) {
                delete m_next;
            }
        }

        U64 GetCpuTime() {
            return m_enterCpuTime;
        }

        unsigned long ReleaseTicket() {
            /*TODO*/
            (*m_previous) = m_next;
            m_next = 0;
            unsigned long ticket = (m_isActive) ? m_ticket : static_cast<unsigned long>(-1);
            delete this;
            return ticket;
        }

        void AddMethodEnterData(MPI::TId threadId, MPI::TId methodId, MPI::TId classId, unsigned long stackDepth) {
            methodInvocationData.classId_ = classId;
            methodInvocationData.methodId_ = methodId;
            methodInvocationData.stackDepth_ = stackDepth;
            methodInvocationData.threadId_ = threadId;
            methodInvocationData.ticket_ = m_ticket;
        }

        void PrintMethodEntryElements(JPIAgent::EC_Env* ec_env);

    private:
        unsigned long m_ticket;
        U64 m_enterCpuTime;
        bool m_isActive;
        JPIAgent::SMethodEnterData_ methodInvocationData; //used in pause mode
        CTicketStack* m_next;
        CTicketStack** m_previous;
    };

//======= CThreadSet ====================================================================================

    class CThreadSet {
    public:
        CThreadSet(MPI::TId threadId, CThreadSet** previous);
        ~CThreadSet();

        void AddNewThreadId(MPI::TId threadId);
        void DeleteThread(MPI::TId threadId);

        void AddNewTicket(unsigned long ticket, MPI::TId threadId, U64 cpuTime, bool isActive);
        unsigned long ReleaseTicket(MPI::TId threadId);
        U64 GetCpuTime(MPI::TId threadId);
        unsigned long GetStackDepth(MPI::TId threadId);
        void AddMethodEnterData(unsigned long ticket, MPI::TId threadId, MPI::TId methodId, MPI::TId classId, U64 cpuTime);
        void PrintMethodEntryElements(JPIAgent::EC_Env* ec_env);

    private:
        MPI::TId m_threadId;
        CThreadSet* m_next;
        CThreadSet** m_previous;
        CTicketStack* m_pTicketStack;
        unsigned long m_stackDepth;
    };

//======= CTickets ====================================================================================

    class CTickets
    {
    public:
        CTickets(void);
        ~CTickets(void);

        void AddNewThread(MPI::TId threadId);
        void DeleteThread(MPI::TId threadId);

        unsigned long GetNewTicket(MPI::TId threadId, U64 cpuTime, bool isActive);
        unsigned long ReleaseTicket(MPI::TId threadId);
        U64 GetCpuTime(MPI::TId threadId);
        unsigned long GetStackDepth(MPI::TId threadId);
        void AddMethodEnterData(MPI::TId threadId, MPI::TId methodId, MPI::TId classId, U64 cpuTime);
        void PrintMethodEntryElements(JPIAgent::EC_Env* ec_env);

    private:
        unsigned long m_currentTicket;
        CThreadSet* m_pThreadSet;
        OSA::IThreadSync* m_lockObject;
    };

//==============================================================================
//============ Aggregated CG profiler ==========================================


/*
    The aggregated statistics mode works by building a tree of method calls for each thread.

    In addition to the tree, it also maintains a pointer which acts as a stack, keeping
    track of the current method being executed for a given thread (the method at the
    top of the stack). Using the tree, along with this stack pointer, the algorithm is
    able to maintain aggregated stack call stats.

    Each CStackHead corresponds to the method calls for a single thread. The thread
    that a particular CStackHead corresponds to is found in its m_ThreadId member variable.

    The methods at the top of the tree will be those that were invoked early in the stack.
    For instance, a single-threaded application that begins with a main() will have
    main() at the top of the tree, with the children of that node being all of methods
    that main() calls directly.

     [A]
     |  \
    [B] [C]
     | \   \
    [D][E]  [F]

    Here, A represents a main(...) call. B and C are methods called by main. D and E would
    be methods called by B, and F would be a method Called by C.

    Multiple nodes with the same method ID will occur in the tree. This is expected. These
    nodes should be separate instances. Statistics will be reported separately for each of these.

	(Note: I am leaving out a single entry above A, here, that is the head of the stack. More
	 on this below)

    ----

    The purpose of aggregated mode is to aggregate method calls in the call stack, while
    maintaining invoked and invoked-by relationships of the methods.

    Each stack entry contains a methodId, statistical data, and a map of the child calls
    for that method (the methods that the stack entry's method calls). Each stack entry
    contains its own map, and it does not share the map or its contents with any other
    stack entry's map. All maps are separate and self-contained.

    Here is the algorithm. There are additional details besides what is listed here, but
    they can be found in the program source.

    Method Enter (method id):
    - Search the current entry on the top of the stack (the mostly recently entered method)
      for the given method id. If found, update the entry statistics for this method.
      If not found, create a new entry as a child entry of the entry at the top of the
      stack, and update the enter time of the new entry.
    - The above entry (the created or found entry) becomes the new top of the stack.

    Method Exit (method id):
    - Pop the top of the stack, but leave the entry on the tree
    - Using the enter time of the popped entry, calculate the base time for the method, and
      increase the number method calls.
    - Update the 'isUpdated' field for the popped off entry, as well as the isUpdated field
      for all direct and indirect parents of that entry in the tree.

    Print Stack:
    - Beginning at the top of the tree, recursively visit each of the methods and print
      the values for the entry
    - Only enter nodes with isUpdated set to true. If a node has this value as false, then
      the entire subtree from that point contains no new information.
    - Once an entry has been entered, set its calls to 0, but leave the other information
      alone (enter time, for instance)
    - Delete all entries on the tree except those still needed by the method currently
      at the top of the stack (as it has been entered, but not exited yet), and its parents

    Additionally, a special CStackEntry called 'stack head' will always be the top entry
    in the tree. It can be distinguished by its method id being 0, and its parent being
    a null reference. It contains references to the methods that the thread begins
    executing on. This is not to be confused with the name of the class CStackHead.
*/

	class CStackHead;

   /** CStackEntry is only referenced in CStackHead. */
    class CStackEntry {
    public:
        CStackEntry(CStackHead *parentStack, CStackEntry * parentEntry, JPIAgent::EC_Env* ec_env, MPI::TId methodId, U64 uiCpuNanos,
        	bool is_active = true) : m_pParentStack(parentStack), m_pParentEntry(parentEntry), m_ec_env(ec_env), m_MethodId(methodId),
        	                          m_baseTime(0), m_baseCPUTime(0),
        	                          #ifdef _WIN32
        	                          m_minTime(0xFFFFFFFFFFFFFFFF),
        	                          #else
        	                          m_minTime(0xFFFFFFFFFFFFFFFFULL),  // unsigned long long 
        	                          #endif
        	                          m_maxTime(0), m_count(0), m_InvokeTime(0), m_EnterCpuTime(uiCpuNanos),
        	                          m_InvokeCpuTime(0), isActive(is_active), m_updated(false)

        {
            m_EnterTime = m_ec_env->GetTime();
            m_pInvokedChildren = new std::map<MPI::TId, CStackEntry *>();
        }

        ~CStackEntry();

        CStackEntry* AddStackEntry(MPI::TId methodId, U64 uiCpuNanos, bool is_active);

        CStackEntry* FinalizeStackEntry(U64 uiCpuNanos);

        void IncreaseInvokeTime(U64 invokeTime, U64 invokeCpuTime);

        void PrintStack(MPI::TId threadId);

        bool IsStackHead() { return m_pParentEntry == NULL && m_MethodId == 0; }

		void DeleteTreeNonMatchingStackEntry(MPI::TId stackEntryId);

    private:
        JPIAgent::EC_Env* m_ec_env;

        MPI::TId m_MethodId;

        // Call Statistics
        U64 m_baseTime;
        U64 m_baseCPUTime;
        U64 m_minTime; 
        U64 m_maxTime; 
        U64 m_count;

		U64 m_EnterTime;
        U64 m_InvokeTime;
        U64 m_EnterCpuTime;
        U64 m_InvokeCpuTime;
        bool isActive;

        /** Whether or not this stack entry, or any entry in its sub-tree, has been updated since the last data flush */
        bool m_updated;

        /** "child" methods directly called by this method */
        std::map<MPI::TId /* method id */, CStackEntry * /* entry */> * m_pInvokedChildren;

        /** The parent of this entry in the call tree*/
        CStackEntry * m_pParentEntry;

        /** The stack head associated with this entry */
        CStackHead * m_pParentStack;

    };

    class CStackHead {
    public:
        CStackHead(JPIAgent::EC_Env* ec_env, MPI::TId threadId, CStackHead* nextCallTree) : m_ec_env(ec_env),
        	m_ThreadId(threadId), m_pNext(nextCallTree), m_pCurrEntry(NULL)
        {
        	lockThreadObject = OSA::CreateThreadSync();

        	m_pStackHead = new CStackEntry(this, 0, m_ec_env, 0, 0, true);

        	m_NumEntries = 0;

        }

        ~CStackHead() {
        	delete(m_pStackHead);

            lockThreadObject->Destroy();
        }

        bool IsSameThreadStack(MPI::TId threadId) const { return threadId == m_ThreadId; }
        CStackHead* NextCallTree() const { return m_pNext; }

        void AddStackEntry(MPI::TId methodId, U64 uiCpuNanos, bool is_active);

        void FinalizeStackEntry(U64 uiCpuNanos);

        void PrintAggCallGraph();

        int GetNumEntries() { return m_NumEntries; }
        void SetNumEntries(int x) { m_NumEntries = x; }

    private:
        JPIAgent::EC_Env* m_ec_env;
        MPI::TId m_ThreadId;

        /** Set by constructor, only used in printAggCallGraph */
        CStackHead* m_pNext;

        /** Entry at the top of the tree */
        CStackEntry * m_pStackHead;

        /** The entry that is currently at the top of the stack. This is the method that is currently
         * executing on this thread (most recently entered method) */
        CStackEntry * m_pCurrEntry;

        OSA::IThreadSync* lockThreadObject;

        /** Total entries in the tree */
        int m_NumEntries;

    };

} /*namespace Martini*/ } /*namespace BaseProf*/

#endif // _TICKETS_H_
