/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation, Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    IBM Corporation - Initial API and implementation
 *    Viacheslav Rybalov, Intel - Initial API and implementation
 *
 * $Id: ECJvmtiAgent.cpp,v 1.20 2010/01/13 19:04:54 jwest Exp $ 
 ***********************************************************************/

#include "ECJvmtiAgent.h"
#include "ECDefs.h"
#include "PrintWrapper.h"
#include "OSA.h"

using namespace Martini::ExternalControl;
using namespace Martini::JPIAgent;
using namespace Martini::OSA;
 
/////////////////////////////////////////////////////////////////

Martini::OSA::IThreadSync* lockObject = Martini::OSA::CreateThreadSync();

/////////////////////////////////////////////////////////////////
// Exported functions

extern "C" EC_EXPORT TResult EC_Init(char **szWorkingDir, const char *szOptions)
{
    return CECJvmtiAgent::GetInstance().EC_Init(szWorkingDir, szOptions);
}

extern "C" EC_EXPORT int RegisterMartiniCallback(ECommand command, 
                                                 UCallback callback)
{
    return CECJvmtiAgent::GetInstance().RegisterMartiniCallback(command, callback);
}

extern "C" EC_EXPORT void MartiniMessage(char* szMsg, ESeverityLevel level)
{
    CECJvmtiAgent::GetInstance().MartiniMessage(szMsg, level);
}

extern "C" EC_EXPORT void MartiniProcessShutdown()
{
    CECJvmtiAgent::GetInstance().MartiniProcessShutdown();
}

/////////////////////////////////////////////////////////////////
// CECJvmtiAgent implementation

CECJvmtiAgent::CECJvmtiAgent() : 
	m_Options(),
	m_Filters(),
	m_Print(new CPrintWrapper(&m_Options, &m_Filters)),
	m_ECWrapper(m_Print, &m_Filters, &m_Options),
	m_Callbacks()
{
}

CECJvmtiAgent::~CECJvmtiAgent() 
{
	if (m_Print != NULL) delete(m_Print);
}

CECJvmtiAgent& CECJvmtiAgent::GetInstance()
{
    static CECJvmtiAgent ecJvmtiAgent;
    return ecJvmtiAgent;
}

int 
CECJvmtiAgent::RegisterMartiniCallback(ECommand command, 
                                           UCallback callback)
{
    bool static isInitACCollectorInvoked = false;
    int res = m_Callbacks.RegisterMartiniCallback_(command, callback);
    if (res == 1) {
        if ((command == CMD_START) && m_Options.isStandAlone() && !m_Options.isProfilerApiEnabled()) {
			m_Print->start();
            m_Callbacks.Start();
        }
        
        if (!isInitACCollectorInvoked && m_Callbacks.ProfCallbacksAvailable()) {
            m_ECWrapper.RegisterMartiniCallbacks(&m_Callbacks);
 			
 			if (!m_Options.isStandAlone()) {
            	m_ECWrapper.InitACCollector();
            }
            
            isInitACCollectorInvoked = true;
        }
    }
    return res;
}

void 
CECJvmtiAgent::MartiniMessage(char* szMsg, ESeverityLevel level)
{
    if (level == SL_FATAL) {
        LOG_DIE("EC JPIAgent received message (FATAL ERROR): " << szMsg);
    }
    LOG_TRACE("EC JPIAgent received message: " << szMsg);
}

void 
CECJvmtiAgent::MartiniProcessShutdown()
{
    LOG_TRACE("EC JPIAgent received Martini Shutdown notification");
}

TResult 
CECJvmtiAgent::EC_Init(char **szWorkingDir, const char *szOptions)
{
    LOG_TRACE("EC options: " << szOptions);
    m_Options.InitializeJvmtiAgentOptionsDefaults();

//    determineTicksPerMicrosecond(&bogus);

	/* Configure debug dumping XML data - available only if binary format is enabled */
	char xmlDumpFileName[MAX_PATH];
	unsigned int xmlDumpFileNameSize;

	TResult dump2xml = GetEnvironmentVar("TPTP_DUMP_XML", xmlDumpFileName, MAX_PATH, &xmlDumpFileNameSize);
	if (dump2xml == MRTE_RESULT_OK) {
		m_Print->setXmlDumpFile(xmlDumpFileName);
	} else if (dump2xml != MRTE_RESULT_FALSE) {
		MartiniMessage("Invalid TPTP_DUMP_XML file path", SL_FATAL);
	}

    /* Turn IO off until we need to start tracing */
	m_Print->suspend();

    /* Determine start time from here */
    jvmtiAgent_collectStartTimeInformation();

    /* Make a global copy of the options for the agent to print later */
    if (szOptions) {
        m_Options.m_jvmtiAgent_Options.invocationOptions = (char*)jvmtiAgent_Calloc(strlen(szOptions) + 1);
        strcpy(m_Options.m_jvmtiAgent_Options.invocationOptions, szOptions);
    } else {
        m_Options.m_jvmtiAgent_Options.invocationOptions = "";
    }    

    /* Process the command line options, continue initialization only if this is successful */
    if (m_Options.ProcessInvocationOptions(szOptions) != 0) {
        m_Options.printUsage();
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }

    if (m_Options.m_jvmtiAgent_Options.standalone) {
        m_Options.ProcessProfile(m_Options.m_jvmtiAgent_Options.profileFile);
        m_Filters.ProcessFilters(m_Options.m_jvmtiAgent_Options.filterFileName);
    }

    /* Now command-line options, as well as profile has been processed, check consistency */
    if (m_Options.CheckOptionsConsistency() != 0) {
        LOG_ERROR("Specified options are not consistent");
        m_Options.printUsage();
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }

	/* Configure printer */
	m_Print->setOutputFormat(m_Options.m_jvmtiAgent_Options.format);

    /* Turn IO off until we need to start tracing */
    m_Print->suspend();
    /* Do setup for Agent when being controlled by the "Agent Controller" */
    if (m_Options.isStandAlone())  {
        /* Standalone mode */
        /* Open the output file if we are doing file I/O */
		if ( strcmp(m_Options.m_jvmtiAgent_Options.outputFileName, "") != 0 ) {
			m_Print->InitializeStandaloneIO(m_Options.m_jvmtiAgent_Options.outputFileName);
		}
    } else {
#if defined(DEBUG) || defined(_DEBUG)
        m_Print->InitializeStandaloneIO(m_Options.m_jvmtiAgent_Options.outputFileName);
#endif
        //ECWrapper::InitACCollector(); May be here
    } 
    return MRTE_RESULT_OK;
}
