/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation, Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    IBM Corporation - Initial API and implementation
 *    Viacheslav Rybalov, Intel - Initial API and implementation
 *
 * $Id: ECWrapper.cpp,v 1.40 2010/12/15 16:49:36 mreid Exp $ 
 ***********************************************************************/

#include <jni.h>
#include "ECWrapper.h"
#include "Options.h"
#include "Filters.h"
#include "Print.h"
#include "Performance.h"
#include "log.h"

#include "LibraryLoader.h"  // Add for bug 226572

using namespace Martini::ExternalControl;
using namespace Martini::ACCollector;
using namespace Martini::MPI;
using namespace Martini::JPIAgent;

ECFunctions_ ECWrapper::m_ECFunctions;
EC_Env ECWrapper::m_EC_Env;
CPrintWrapper* ECWrapper::m_pCPrint = 0;
CFilters* ECWrapper::m_pCFilters = 0;
COptions* ECWrapper::m_pCOptions = 0;
CEC_Callbacks* ECWrapper::m_pCallbacks = 0;
char* ECWrapper::m_profName = 0;
bool ECWrapper::m_VMInitDone = false;
ACCollector_SendVMAgentInitialized_t ECWrapper::m_SendVMAgentInitializedCommand = 0;

ECWrapper::ECWrapper(CPrintWrapper* pCPrint, CFilters* pCFilters, COptions* pCOptions)
{
    m_pCPrint = pCPrint;
    m_pCFilters = pCFilters;
    m_pCOptions = pCOptions;
}

ECWrapper::~ECWrapper(void)
{
}

//=== ACC_ENV ===============================================================================================

static bool ACTracingStarted = false;
static bool appTracingStarted = false;
static bool tracingStarted = false;

static void startTracing();
static void start();
static void stopTracing();
static void stop();

extern "C" int isProfilerApiEnabled() {
	return (ECWrapper::m_pCOptions != NULL) &&	ECWrapper::m_pCOptions->isProfilerApiEnabled();
}

API_EXPORT void ACC_CALL ACStartTracing() {
    LOG_TRACE("ACStartTracing command!");
	if (ACTracingStarted) return;
	
	ACTracingStarted = true;
	startTracing();
}

extern "C" void appStartTracing() {
    LOG_TRACE("appStartTracing command!");
	if (appTracingStarted) return;
	
	appTracingStarted = true;
	startTracing();
}

/** START_TRACING  *************************************************************
* This is the entry point which must be called to start a trace. It takes
* care of emitting all the header information.
*/
static void startTracing() {
	if (tracingStarted) return;
	
	bool profilerApiEnabled = false;
	bool standaloneMode = false;
	
	if (ECWrapper::m_pCOptions != NULL) {
		profilerApiEnabled = ECWrapper::m_pCOptions->isProfilerApiEnabled();
		standaloneMode = ECWrapper::m_pCOptions->isStandAlone(); 
	}
	
	bool toTrace = (standaloneMode || ACTracingStarted) && (appTracingStarted || !profilerApiEnabled);

	if (!toTrace) return;
	
	start();
	tracingStarted = true;
}

static void start() {
    LOG_TRACE("Start command!");

   	LOG_ASSERT(ECWrapper::m_pCPrint != 0);
   	ECWrapper::m_pCPrint->start();
	
    LOG_ASSERT(ECWrapper::m_pCallbacks != 0);
    ECWrapper::m_pCallbacks->Start();
}

API_EXPORT void ACC_CALL ACStopTracing() {
    LOG_TRACE("ACStop command!");
    if (!ACTracingStarted) return;
    
    ACTracingStarted = false;
    stopTracing();
}

extern "C" void appStopTracing() {
    LOG_TRACE("appStop command!");
    if (!appTracingStarted) return;
    
    appTracingStarted = false;
    stopTracing();
}

/** STOP_TRACING  **************************************************************
* This is where a trace is stopped.
*/
static void stopTracing() {
	if (!tracingStarted) return;

	bool profilerApiEnabled = false;
	bool standaloneMode = false;
	
	if (ECWrapper::m_pCOptions != NULL) {
		profilerApiEnabled = ECWrapper::m_pCOptions->isProfilerApiEnabled();
		standaloneMode = ECWrapper::m_pCOptions->isStandAlone(); 
	}
	
	bool toTrace = (standaloneMode || ACTracingStarted) && (appTracingStarted || !profilerApiEnabled);
	if (toTrace) return;
	
	tracingStarted = false;
	stop();
}

static void stop() {
    LOG_TRACE("Stop command!");
	    
    LOG_ASSERT(ECWrapper::m_pCallbacks != 0);
    ECWrapper::m_pCallbacks->Stop();
   	LOG_ASSERT(ECWrapper::m_pCPrint != 0);
	ECWrapper::m_pCPrint->suspend();
}

extern "C" void printCustomElement(const char* str) {
    LOG_ASSERT(ECWrapper::m_pCPrint != 0);
    (ECWrapper::m_pCPrint)->printCustomElement(str);
}

API_EXPORT void ACC_CALL
Attach()
{
    LOG_TRACE("Attach command!");
    
   	LOG_ASSERT(ECWrapper::m_pCPrint != 0);
    ECWrapper::m_pCPrint->start();

    LOG_ASSERT(ECWrapper::m_pCallbacks != 0);
    ECWrapper::m_pCallbacks->Attach();
}

API_EXPORT void ACC_CALL
Detach()
{
    LOG_TRACE("Detach command!");
    LOG_ASSERT(ECWrapper::m_pCallbacks != 0);
    ECWrapper::m_pCallbacks->Detach();

   	LOG_ASSERT(ECWrapper::m_pCPrint != 0);
   	ECWrapper::m_pCPrint->stop();
}

API_EXPORT void ACC_CALL
AddFilter(char *iid, char *classPattern, char *methodPattern, char *filterMode)
{
    LOG_TRACE("AddFilter command!");
    LOG_ASSERT(ECWrapper::m_pCFilters != 0);
    ECWrapper::m_pCFilters->AddFilter(classPattern, methodPattern, strcmp("INCLUDE", filterMode) == 0 ? INCLUDE : EXCLUDE);
    //TODO iid is ignored while multiple profilers are not supported 
}

API_EXPORT void ACC_CALL
ApplyFilters()
{
    LOG_TRACE("ApplyFilters command!");
    LOG_ASSERT(ECWrapper::m_pCFilters != 0);
    ECWrapper::m_pCFilters->ApplyFilters();
}

API_EXPORT void ACC_CALL
SetProfileOption(char *iid, char *name, char *value)
{
    LOG_TRACE("SetProfileOption command!");
    LOG_ASSERT(ECWrapper::m_pCOptions != 0);
    ECWrapper::m_pCOptions->SetProfileOption(name, value);
    //TODO iid is ignored while multiple profilers are not supported 
}

API_EXPORT void ACC_CALL
AnalyseHeap()
{
    LOG_TRACE("AnalyseHeap command!");
    LOG_ASSERT(ECWrapper::m_pCallbacks != 0);
    ECWrapper::m_pCallbacks->CustomCommand(ANALYSE_HEAP_COMMAND, 0);
}

API_EXPORT void ACC_CALL
RunGC()
{
    LOG_TRACE("RunGC command!");
    LOG_ASSERT(ECWrapper::m_pCallbacks != 0);
    if (ECWrapper::m_pCallbacks != NULL) {
    	ECWrapper::m_pCallbacks->CustomCommand(RUN_GC_COMMAND, 0);
    }
}

extern "C" void runGC() {
	RunGC();
}

// Checks if heap instance data collection has been enabled.
API_EXPORT bool ACC_CALL isHeapInstanceDataEnabled() {
	return ECWrapper::m_pCOptions->isHeapInstanceDataEnabled();
}

// Sets the flag for heap instance data collection by sending the custom command
// so that Martini is aware of the option value.
API_EXPORT void ACC_CALL setHeapInstanceDataEnabled()
{
	bool isEnabled = ECWrapper::m_pCOptions->isHeapInstanceDataEnabled();

	if(isEnabled && ECWrapper::m_pCallbacks != NULL) {
		ECWrapper::m_pCallbacks->CustomCommand(SET_HEAP_OBJ_COLLECTION_CMD, &isEnabled);
	}
}

// Make the call to the custom command to run the heap instance data collection inside Martini.
API_EXPORT void ACC_CALL
RunHeapObjectInstanceCollection(Martini::ACCollector::MHeapObjectAnalysis *heapAnalysis)
{
    LOG_TRACE("Calling the RunHeapObjectInstanceCollection custom command.");
    LOG_ASSERT(ECWrapper::m_pCallbacks != 0);

    // Convert between THeapObjectAnalysis and MHeapObjectAnalysis so that the result
    // can be returned to the client.
    if (ECWrapper::m_pCallbacks != NULL) {
    	THeapObjectAnalysis jpiHeapAnalysis;
    	jpiHeapAnalysis.TId = heapAnalysis->TId;
    	jpiHeapAnalysis.resultValue = heapAnalysis->resultValue;

    	ECWrapper::m_pCallbacks->CustomCommand(RUN_HEAP_OBJ_DATA_COLLECTION_CMD, &jpiHeapAnalysis);
    }
}

API_EXPORT void ACC_CALL
CollectAggData()
{
    LOG_TRACE("CollectData command!");
    LOG_ASSERT(ECWrapper::m_pCallbacks != 0);
    ECWrapper::m_pCallbacks->CustomCommand(COLLECT_DATA, 0);
}

API_EXPORT void ACC_CALL
EnableDataCollection()
{
    LOG_TRACE("EnableDataCollection command!");
    LOG_ASSERT(ECWrapper::m_pCallbacks != 0);
    ECWrapper::m_pCallbacks->CustomCommand(ENABLE_DATA_COLLECTION, 0);
}

typedef TResult (*TJpiAttachCurrentThread) (JNIEnv **ppJNIEnv, bool *bAttached);
TJpiAttachCurrentThread pJpiAttachCurrentThread = NULL;

typedef void (*TJpiDetachCurrentThread)();
TJpiDetachCurrentThread pJpiDetachCurrentThread = NULL;

API_EXPORT void ACC_CALL
openSession( JNIEnv** pEnv, bool* bAttached) {
    LOG_TRACE("openSession command!");
    
    if (pJpiAttachCurrentThread != NULL) 
    	pJpiAttachCurrentThread(pEnv, bAttached);
    else {
	    *(pEnv) = NULL;
	    *(bAttached) = false;
    }
	    
}

API_EXPORT void ACC_CALL
closeSession() {
    LOG_TRACE("closeSession command!");
    if (pJpiDetachCurrentThread != NULL) pJpiDetachCurrentThread(); 
}

API_EXPORT char* ACC_CALL
GetProfiler()
{
    LOG_TRACE("GetProfiler command!");
    return ECWrapper::m_profName;
}

API_EXPORT bool ACC_CALL
IsAgentInitialized()
{
    LOG_TRACE("IsAgentInitialized request!");
    return ECWrapper::m_VMInitDone;
}

API_EXPORT char * ACC_CALL
SetOutputFormat(char *format)
{
    LOG_TRACE("SetOutputFormat command - " << format);
	if (strcmp(format, BINARY_TRACE_FORMAT) == 0) {
		ECWrapper::m_pCPrint->setOutputFormat(BinaryFormat);
		return format;
	} else if (strcmp(format, XML_TRACE_FORMAT) == 0) {
		ECWrapper::m_pCPrint->setOutputFormat(XmlFormat);
		return format;
	} 
	return XML_TRACE_FORMAT;
}

ACC_ENV*
ECWrapper::PrepareACCEnv() 
{
    static ACCFunctions_ functions;
    functions.StartTracing = &ACStartTracing;
    functions.StopTracing = &ACStopTracing;

    functions.Detach = &Detach;
    functions.Attach = &Attach;
    functions.AddFilter = &AddFilter;
    functions.ApplyFilters = &ApplyFilters;
    functions.SetProfileOption = &SetProfileOption;
    functions.AnalyseHeap = &AnalyseHeap;
    functions.RunGC = &RunGC;
    functions.RunHeapObjectInstanceCollection = &RunHeapObjectInstanceCollection;
    functions.isHeapInstanceDataEnabled = &isHeapInstanceDataEnabled;
    functions.setHeapInstanceDataEnabled = &setHeapInstanceDataEnabled;
    functions.CollectAggData = &CollectAggData;
    functions.EnableDataCollection = &EnableDataCollection;
    functions.GetProfiler = &GetProfiler;
    functions.openSession = &openSession;
    functions.closeSession = &closeSession;
    functions.IsAgentInitialized = &IsAgentInitialized;
	functions.SetOutputFormat = &SetOutputFormat;
    static ACC_ENV ac_env;
    ac_env.functions = &functions;
    return &ac_env;
}

void ECWrapper::RegisterMartiniCallbacks(CEC_Callbacks* callbacks) {
    m_pCallbacks = callbacks;
}

void 
ECWrapper::InitACCollector() {
    ACC_ENV* ac_env = PrepareACCEnv();

    Martini::OSA::ILibraryLoader* libraryLoader;
    // Modified for bug 226572
    libraryLoader = LoadBistroLibrary("ACCollector");
    if (libraryLoader == 0) {
        LOG_DIE("FATAL ERROR: JPIAgent can't load ACCollector");
    }
    m_pCPrint->setSendDataFunc((Martini::ACCollector::ACCollector_SendData_t)(libraryLoader->GetEntry("ACCollector_SendData")));
    if (m_pCPrint->getSendDataFunc() == 0) {
        LOG_DIE("FATAL ERROR: ACCollector is invalid");
    }

    m_SendVMAgentInitializedCommand = (ACCollector_SendVMAgentInitialized_t)(libraryLoader->GetEntry("ACCollector_SendVMAgentInitializedCommand"));

    ACCollector_Init_t acCollector_Init = (ACCollector_Init_t)(libraryLoader->GetEntry("ACCollector_Init"));
    if (acCollector_Init == 0) {
        LOG_DIE("FATAL ERROR: JPIAgent can't initialize ACCollector");
    }
    libraryLoader->Destroy();
    
    if (m_pCOptions->isEnabled()) BindJpiFunctions();
    
    (*acCollector_Init)(ac_env, m_pCOptions->isControlled());

    // Bug 332656 - Allow ACCollector to obtain a JNI handle when running in controlled mode
    if (m_pCOptions->isControlled()) 
	    BindJpiFunctions();
}

void ECWrapper::BindJpiFunctions() {
    Martini::OSA::ILibraryLoader* libraryLoader;
    // Modified for bug 226572
    libraryLoader = LoadBistroLibrary("JPI");
    if (libraryLoader == 0) {
    	LOG_TRACE("Unable to load JPI library");
    	return;
    }

    pJpiAttachCurrentThread = (TJpiAttachCurrentThread)(libraryLoader->GetEntry("JPI_AttachCurrentThread"));
    if (pJpiAttachCurrentThread == NULL) {
    	LOG_TRACE("Unable to get JPI_AttachCurrentThread entry point");
    }
    
    pJpiDetachCurrentThread = (TJpiDetachCurrentThread)(libraryLoader->GetEntry("JPI_DetachCurrentThread"));
    if (pJpiDetachCurrentThread == NULL) {
    	LOG_TRACE("Unable to get JPI_DetachCurrentThread entry point");
    } 
    
    libraryLoader->Destroy();
}

//=== EC_Env ================================================================================================

EC_EXPORT EC_ENV_Error ECCALL 
PrintThreadStartElement(EC_Env* env, TId threadId, TId objectId, SThreadInfo* threadInfo)
{
    LOG_ASSERT(ECWrapper::m_pCPrint != 0);
    (ECWrapper::m_pCPrint)->printThreadStartElement(threadId, objectId, threadInfo);
    return EC_ENV_ERROR_NONE;
}

EC_EXPORT EC_ENV_Error ECCALL 
PrintThreadEndElement(EC_Env* env, TId threadId)
{
    LOG_ASSERT(ECWrapper::m_pCPrint != 0);
    (ECWrapper::m_pCPrint)->printThreadEndElement(threadId);
    return EC_ENV_ERROR_NONE;
}

EC_EXPORT EC_ENV_Error ECCALL 
PrintRuntimeVMInitElement(EC_Env* env, TId threadId)
{
    LOG_ASSERT(ECWrapper::m_pCPrint != 0);
    (ECWrapper::m_pCPrint)->printJvmInitDoneElement(threadId);
    return EC_ENV_ERROR_NONE;
}

EC_EXPORT EC_ENV_Error ECCALL 
PrintRuntimeShutdownElement(EC_Env* env)
{
    LOG_ASSERT(ECWrapper::m_pCPrint != 0);
    (ECWrapper::m_pCPrint)->printJvmShutdownElement();
    return EC_ENV_ERROR_NONE;
}

EC_EXPORT EC_ENV_Error ECCALL 
PrintMethodDefElement(EC_Env* env, TId methodId, SMethodInfo* methodInfo)
{
    LOG_ASSERT(ECWrapper::m_pCPrint != 0);
    (ECWrapper::m_pCPrint)->printNewMethodElement(methodId, methodInfo);
    return EC_ENV_ERROR_NONE;
}

EC_EXPORT EC_ENV_Error ECCALL 
PrintClassDefElement(EC_Env* env, TId classId, SClassInfo* classInfo)
{
    LOG_ASSERT(ECWrapper::m_pCPrint != 0);
    (ECWrapper::m_pCPrint)->printNewClassElement(classId, classInfo);
    return EC_ENV_ERROR_NONE;
}

EC_EXPORT EC_ENV_Error ECCALL 
PrintMethodEntryElement(EC_Env* env, TId threadId, TId methodId, TId classId, unsigned long ticket, unsigned long stackDepth)
{
    LOG_ASSERT(ECWrapper::m_pCPrint != 0);
    (ECWrapper::m_pCPrint)->printMethodEntryElement(threadId, methodId, classId, ticket, stackDepth);
    return EC_ENV_ERROR_NONE;
}

EC_EXPORT EC_ENV_Error ECCALL 
PrintAgMethodEntryElement(EC_Env* env, TId threadId, TId methodId, U64 baseTime, U64 minTime, U64 maxTime, U64 baseCPUTime, U64 numCalls)
{
    LOG_ASSERT(ECWrapper::m_pCPrint != 0);
    (ECWrapper::m_pCPrint)->printAgMethodEntryElement(threadId, methodId, baseTime, minTime, maxTime, baseCPUTime, numCalls);
    return EC_ENV_ERROR_NONE;
}

EC_EXPORT EC_ENV_Error ECCALL 
PrintMethodExitElement(EC_Env* env, TId threadId, TId methodId, TId classId, unsigned long ticket, U64 cpuTime)
{
    LOG_ASSERT(ECWrapper::m_pCPrint != 0);
    (ECWrapper::m_pCPrint)->printMethodExitElement(threadId, methodId, classId, ticket, cpuTime);
    return EC_ENV_ERROR_NONE;
}

EC_EXPORT EC_ENV_Error ECCALL 
PrintAgMethodExitElement(EC_Env* env, TId threadId, TId methodId)
{
    LOG_ASSERT(ECWrapper::m_pCPrint != 0);
    (ECWrapper::m_pCPrint)->printAgMethodExitElement(threadId, methodId);
    return EC_ENV_ERROR_NONE;
}

EC_EXPORT EC_ENV_Error ECCALL 
PrintObjAllocElement(EC_Env* env, SHeapEventData* heapData, U64 lineNumber)
{
    LOG_ASSERT(ECWrapper::m_pCPrint != 0);
    (ECWrapper::m_pCPrint)->printObjAllocElement(heapData, lineNumber);
    return EC_ENV_ERROR_NONE;
}

EC_EXPORT EC_ENV_Error ECCALL 
PrintObjFreeElement(EC_Env* env, SHeapEventData* heapData)
{
    LOG_ASSERT(ECWrapper::m_pCPrint != 0);
    (ECWrapper::m_pCPrint)->printObjFreeElement(heapData);
    return EC_ENV_ERROR_NONE;
}

EC_EXPORT EC_ENV_Error ECCALL 
PrintGcStartElement(EC_Env* env)
{
    LOG_ASSERT(ECWrapper::m_pCPrint != 0);
    (ECWrapper::m_pCPrint)->printGcStartElement();
    return EC_ENV_ERROR_NONE;
}

EC_EXPORT EC_ENV_Error ECCALL 
PrintGcFinishElement(EC_Env* env)
{
    LOG_ASSERT(ECWrapper::m_pCPrint != 0);
    (ECWrapper::m_pCPrint)->printGcFinishElement();
    return EC_ENV_ERROR_NONE;
}

EC_EXPORT EC_ENV_Error ECCALL 
PrintHeapDumpDefElement(EC_Env* env, TId heapDumpId, char* heapDefName, U64 basetime)
{
    LOG_ASSERT(ECWrapper::m_pCPrint != 0);
    (ECWrapper::m_pCPrint)->printHDStartElement(heapDumpId, heapDefName, basetime);
    return EC_ENV_ERROR_NONE;
}

EC_EXPORT EC_ENV_Error ECCALL 
PrintGcRootElement(EC_Env* env, TId heapDumpIdRef, TId objIdRef, EObjectReferenceType type)
{
    LOG_ASSERT(ECWrapper::m_pCPrint != 0);
    (ECWrapper::m_pCPrint)->printGcRootElement(heapDumpIdRef, objIdRef, type);
    return EC_ENV_ERROR_NONE;
}

EC_EXPORT EC_ENV_Error ECCALL 
PrintObjectReferenceElement(EC_Env* env, TId heapDumpIdRef, SObjectReference* objRef)
{
    LOG_ASSERT(ECWrapper::m_pCPrint != 0);
    (ECWrapper::m_pCPrint)->printObjectReferenceElement(heapDumpIdRef, objRef);
    return EC_ENV_ERROR_NONE;
}

EC_EXPORT EC_ENV_Error ECCALL 
PrintMonitorWaitElement(EC_Env* env, SMonitorWaitEventData* monitorWaitData, SStackTrace_* stackTrace)
{
    LOG_ASSERT(ECWrapper::m_pCPrint != 0);
    (ECWrapper::m_pCPrint)->printMonitorWaitElement(monitorWaitData, stackTrace);
    return EC_ENV_ERROR_NONE;
}

EC_EXPORT EC_ENV_Error ECCALL 
PrintMonitorWaitedElement(EC_Env* env, SMonitorWaitedEventData* monitorWaitedData, U64 timeout, SStackTrace_* stackTrace)
{
    LOG_ASSERT(ECWrapper::m_pCPrint != 0);
    (ECWrapper::m_pCPrint)->printMonitorWaitedElement(monitorWaitedData, timeout, stackTrace);
    return EC_ENV_ERROR_NONE;
}

EC_EXPORT EC_ENV_Error ECCALL 
PrintMonitorContendedEnterElement(EC_Env* env, SContendedMonitorEnterEventData* conMonEnterData, SStackTrace_* stackTrace)
{
    LOG_ASSERT(ECWrapper::m_pCPrint != 0);
    (ECWrapper::m_pCPrint)->printMonitorContendedEnterElement(conMonEnterData, stackTrace);
    return EC_ENV_ERROR_NONE;
}

EC_EXPORT EC_ENV_Error ECCALL 
PrintMonitorContendedEnteredElement(EC_Env* env, SContendedMonitorEnteredEventData* conMonEnteredData, SStackTrace_* stackTrace)
{
    LOG_ASSERT(ECWrapper::m_pCPrint != 0);
    (ECWrapper::m_pCPrint)->printMonitorContendedEnteredElement(conMonEnteredData, stackTrace);
    return EC_ENV_ERROR_NONE;
}

EC_EXPORT EC_ENV_Error ECCALL 
printThreadInteractionElements(EC_Env* env, SThreadInteractionEventData* threadInteractionData, SStackTrace_* stackTrace)
{
    LOG_ASSERT(ECWrapper::m_pCPrint != 0);
    ECWrapper::m_pCPrint->printThreadInteractionElements(threadInteractionData, stackTrace);
    return EC_ENV_ERROR_NONE;
}

EC_EXPORT EC_ENV_Error ECCALL 
PrintCustomElement(EC_Env* env, const char* str)
{
    LOG_ASSERT(ECWrapper::m_pCPrint != 0);
    (ECWrapper::m_pCPrint)->printCustomElement(str);
    return EC_ENV_ERROR_NONE;
}

EC_EXPORT bool ECCALL
IsExcluded(EC_Env* env, const char *className, const char *methodName)
{
    LOG_ASSERT(ECWrapper::m_pCFilters != 0);
    Martini::JPIAgent::Filter* classFilter = ECWrapper::m_pCFilters->GetFilter(className, methodName);
    int res = ECWrapper::m_pCFilters->CheckMethodFilters(methodName, classFilter);
    if (res == EXCLUDE) {
        return true;
    }else {
        return false;
    } 
}

EC_EXPORT U64 ECCALL
GetTime_(EC_Env* env)
{
    return GetProfilerTime();
}

EC_EXPORT int ECCALL
SetProfileOption_(EC_Env* env, char* key, char* value)
{
    return ECWrapper::m_pCOptions->SetProfileOption(key, value);
}

EC_EXPORT bool ECCALL
isCGExecDetails(EC_Env* env)
{
    return ECWrapper::m_pCOptions->isCGExecDetails();
}

EC_EXPORT bool ECCALL
isCalcStackMap(EC_Env* env)
{
    return ECWrapper::m_pCOptions->isCalcStackMap();
}

EC_EXPORT bool ECCALL
isStackMemoryFree(EC_Env* env)
{
    return ECWrapper::m_pCOptions->isStackMemoryFree();
}

EC_EXPORT bool ECCALL
isStackInfoNormal(EC_Env* env)
{
    return ECWrapper::m_pCOptions->isStackInfoNormal();
}

EC_EXPORT bool ECCALL
isStandAlone(EC_Env* env)
{
    return ECWrapper::m_pCOptions->isStandAlone();
}

EC_EXPORT bool ECCALL
isControlled(EC_Env* env)
{
    return ECWrapper::m_pCOptions->isControlled();
}

EC_EXPORT bool ECCALL
isEnabled(EC_Env* env)
{
    return ECWrapper::m_pCOptions->isEnabled();
}

EC_EXPORT bool ECCALL
isAllocSitesSupported(EC_Env* env)
{
    return ECWrapper::m_pCOptions->isAllocSitesSupported();
}

EC_EXPORT bool ECCALL
isContentionAnalysisSupported(EC_Env* env)
{
    return ECWrapper::m_pCOptions->isContentionAnalysisSupported();
}

EC_EXPORT bool ECCALL
isHeapInstanceDataEnabled(EC_Env* env)
{
    return ECWrapper::m_pCOptions->isHeapInstanceDataEnabled();
}

EC_EXPORT const char* ECCALL
getUnknownOptionByName(EC_Env* env, const char* name)
{
    return ECWrapper::m_pCOptions->getUnknownOptionByName(name);
}

EC_EXPORT void ECCALL
VMInitDone(EC_Env* env)
{
    if (!ECWrapper::m_pCOptions->isStandAlone() && (ECWrapper::m_SendVMAgentInitializedCommand != 0)) {
        (*ECWrapper::m_SendVMAgentInitializedCommand)();
    }
    ECWrapper::m_VMInitDone = true;
}

EC_EXPORT int ECCALL
getMaxStackDepth(EC_Env* env)
{
    return ECWrapper::m_pCOptions->getMaxStackDepth();
}

extern "C" EC_EXPORT int /*ECCALL*/ GetEC_Env(char* profName, EC_Env** env)
{
    ECWrapper::m_profName = profName;

    ECWrapper::m_ECFunctions.PrintThreadStartElement = &PrintThreadStartElement;
    ECWrapper::m_ECFunctions.PrintThreadEndElement = &PrintThreadEndElement;
    ECWrapper::m_ECFunctions.PrintRuntimeShutdownElement = &PrintRuntimeShutdownElement;
    ECWrapper::m_ECFunctions.PrintRuntimeVMInitElement = &PrintRuntimeVMInitElement;
    ECWrapper::m_ECFunctions.PrintMethodDefElement = &PrintMethodDefElement;
    ECWrapper::m_ECFunctions.PrintMethodEntryElement = &PrintMethodEntryElement;
    ECWrapper::m_ECFunctions.PrintAgMethodEntryElement = &PrintAgMethodEntryElement;
    ECWrapper::m_ECFunctions.PrintMethodExitElement = &PrintMethodExitElement;
    ECWrapper::m_ECFunctions.PrintAgMethodExitElement = &PrintAgMethodExitElement;
    ECWrapper::m_ECFunctions.PrintClassDefElement = &PrintClassDefElement;
    ECWrapper::m_ECFunctions.PrintObjAllocElement = &PrintObjAllocElement;
    ECWrapper::m_ECFunctions.PrintObjFreeElement = &PrintObjFreeElement;
    ECWrapper::m_ECFunctions.PrintGcRootElement = &PrintGcRootElement;
    ECWrapper::m_ECFunctions.PrintGcStartElement = &PrintGcStartElement;
    ECWrapper::m_ECFunctions.PrintGcFinishElement = &PrintGcFinishElement;
    ECWrapper::m_ECFunctions.PrintHeapDumpDefElement = &PrintHeapDumpDefElement;
    ECWrapper::m_ECFunctions.PrintObjectReferenceElement = &PrintObjectReferenceElement;
    ECWrapper::m_ECFunctions.PrintMonitorWaitElement = &PrintMonitorWaitElement;
    ECWrapper::m_ECFunctions.PrintMonitorWaitedElement = &PrintMonitorWaitedElement;
    ECWrapper::m_ECFunctions.PrintMonitorContendedEnterElement = &PrintMonitorContendedEnterElement;
    ECWrapper::m_ECFunctions.PrintMonitorContendedEnteredElement = &PrintMonitorContendedEnteredElement;
    ECWrapper::m_ECFunctions.printThreadInteractionElements = &printThreadInteractionElements;
    ECWrapper::m_ECFunctions.PrintCustomElement = &PrintCustomElement;
    ECWrapper::m_ECFunctions.IsExcluded = &IsExcluded;
    ECWrapper::m_ECFunctions.GetTime = &GetTime_;
    ECWrapper::m_ECFunctions.SetProfileOption = &SetProfileOption_;
    ECWrapper::m_ECFunctions.isCGExecDetails = &isCGExecDetails;
    ECWrapper::m_ECFunctions.isStackMemoryFree = &isStackMemoryFree;
    ECWrapper::m_ECFunctions.isStackInfoNormal = &isStackInfoNormal;
    ECWrapper::m_ECFunctions.isStandAlone = &isStandAlone;
    ECWrapper::m_ECFunctions.isControlled = &isControlled;
    ECWrapper::m_ECFunctions.isEnabled = &isEnabled;
    ECWrapper::m_ECFunctions.isAllocSitesSupported = &isAllocSitesSupported;
    ECWrapper::m_ECFunctions.isContentionAnalysisSupported = &isContentionAnalysisSupported;
    ECWrapper::m_ECFunctions.isCalcStackMap = &isCalcStackMap;
    ECWrapper::m_ECFunctions.isHeapInstanceDataEnabled = &isHeapInstanceDataEnabled;
    ECWrapper::m_ECFunctions.getUnknownOptionByName = &getUnknownOptionByName;
    ECWrapper::m_ECFunctions.VMInitDone = &VMInitDone;
    ECWrapper::m_ECFunctions.getMaxStackDepth = &getMaxStackDepth;
    ECWrapper::m_EC_Env.functions = &ECWrapper::m_ECFunctions;
    *env = &ECWrapper::m_EC_Env;
    return 0;
}
