/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation, Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    IBM Corporation - Initial API and implementation
 *    Viacheslav Rybalov, Intel - Initial API and implementation
 *
 * $Id: Filters.cpp,v 1.14 2010/08/17 17:00:59 jwest Exp $ 
 ***********************************************************************/

#if defined __linux__
    #include <stdint.h>
#else
    #define INT32_MAX (2147483647)
#endif


#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include "Filters.h"

using namespace Martini::JPIAgent;

/******* Filter ********************************************/

/** str_fits_ptrn *********************************************************************************
  * This function returns a non-zero value if the string matches the pattern, or a zero.  The
  * input pointers can be empty srings, but cannot be 0.  All strings and patterns, including
  * the empty ones are handled properly.
  *
  * @param str          the string to match the pattern
  * @param ptrn         the pattern to match the string
  * @param ptrn_type    the type of the pattern (PREFIX OR SUFFIX)
  * @param ptrn_len     the length of the pattern
  * @returns    a non-zero value if the string fits into the pattern, or zero otherwise.
  */
int 
Filter::str_fits_ptrn(const char* str, const char* ptrn, const int ptrn_type, const int ptrn_len) {
    size_t off; /* the offset of the start of comparison in a prefix match */

    if (str == 0 || ptrn == 0) {
        return 0; 
    }

    /* A zero-length (empty) pattern matches all strings including the empty ones.  Otherwise, an exact
       match, prefix, or suffix match will be performed.  The first case is really not really a special
       case as it has already been coverted under the other three.  So, this case is really included
       for clarity purposes only.  */
    return !ptrn_len ||
           (ptrn_type == NONE && !strcmp(ptrn, str)) ||
           (ptrn_type == PREFIX && (off = strlen(str) - ptrn_len) >= 0 && !strcmp(ptrn, str + off)) ||
           (ptrn_type == SUFFIX && !strncmp(ptrn, str, ptrn_len));
}

/** match_mthd_fltr *******************************************************************************
  * This static function returns the first method filter from the class filter list, matching the method
  * name.
  *
  * @param mthd_fltr_lst        the method filter list from which a match is searched
  * @param mthd_fltr_lst_len    the length of the above list
  * @param mthd_name            the method name to match
  * @returns    the first matching method filter, or 0 if none is found.
  */
MethodFilter_t* 
Filter::match_mthd_fltr(MethodFilter_t *mthd_fltr_lst, const unsigned int mthd_fltr_lst_len, const char *mthd_name) {
    unsigned int i;             /* the loop control variable */
    MethodFilter_t *mthd_fltr;  /* a method filter from a class filter */

    /* Iterate through the method filter list to find the first matching method filter. */
    for (i = 0; i < mthd_fltr_lst_len; i++) {
        mthd_fltr = mthd_fltr_lst + i;
        if (str_fits_ptrn(mthd_name, mthd_fltr -> pattern, mthd_fltr -> genericPattern, mthd_fltr -> patternLength)) {
            return mthd_fltr;
        }
    }
    return 0;
}


/** match_cls_mthd_fltr ***************************************************************************
  * This static function returns the first class filter from the class filter list, matching the class
  * and method names.  If the method name supplied is an empty string, the first matching class
  * filter with a wildcard method filter will be picked.
  *
  * @param cls_fltr_lst     the class filter list from which a match is searched
  * @param cls_fltr_lst_len the length of the above list
  * @param cls_name         the class name to match
  * @param mthd_name        the method name to match
  * @returns    the first matching class filter, or 0 if none is found.
  */
Filter* 
Filter::match_cls_mthd_fltr(Filter *cls_fltr_lst, int cls_fltr_lst_len, const char *cls_name, const char *mthd_name) {
    int i;                      /* the loop control variable */
    Filter *cls_fltr;           /* a class filter from the filter list */
    MethodFilter_t *mthd_fltr;  /* a method filter from a class filter */
    const char *reg_cls_name;   /* the class name without JNI style array notation */ 

    /* bugzilla 70263 - remove the array notation from the front of a class name so we filter 
    array classes as well. For example, if the class name is '[[Ljava.lang.Thread', the
    following code will set reg_cls_name to 'java.lang.Thread'. */ 

    reg_cls_name = strrchr(cls_name,'['); /* returns the last occurence of '[' */ 
    if (reg_cls_name != 0 && reg_cls_name[1] == 'L') {
        reg_cls_name += 2; /* skip to the point after the 'L' */ 
    } else {
        reg_cls_name = cls_name; 
    }

    /* Iterate through the class filter list to find the first filter matching the class and method names. */
    for (i = 0; i < cls_fltr_lst_len; i++) {
        cls_fltr = cls_fltr_lst + i;
        if (!str_fits_ptrn(reg_cls_name, cls_fltr -> pattern, cls_fltr -> genericPattern, cls_fltr -> patternLength)) {
            continue;
        }
        mthd_fltr = match_mthd_fltr(cls_fltr -> methodDetails, cls_fltr -> methodDetailCount, mthd_name);
        if (mthd_fltr) {
            return cls_fltr;
        }
    }
    return 0;
}

/** jvmpiAgent_getClassFilterMode *****************************************************************
  * This static function determines whether a class is to be INCLUDED or EXCLUDED.  This can occur if
  * the class filter has a wildcard method filter, or if the class filter itself is 0, i.e.,
  * the class does not match any existing filter.
  *
  * @param cls_fltr the matching filter for this class
  * @returns    INCLUDE or EXCLUDE as is appropriate.
  */
enum FilterMode 
Filter::GetClassFilterMode(Filter *cls_fltr) {
    MethodFilter_t *mthd_fltr; /* the method filter for the default methods */

    mthd_fltr = cls_fltr ? match_mthd_fltr(cls_fltr -> methodDetails, cls_fltr -> methodDetailCount, "") : 0;
    return mthd_fltr ? mthd_fltr -> mode : INCLUDE;
}

/** get_cls_fltr **********************************************************************************
  * This static function returns a matching class filtering from the class filter list provided,
  * or a 0 pointer if none is found.  Matching filters must share the same filter pattern,
  * type, and length.
  *
  * @param cls_fltr_lst     the class filter list from which a match is searched
  * @param cls_fltr_lst_len the length of the above list
  * @param cls_ptrn         the class pattern to match
  * @param cls_ptrn_type    the type of the class pattern to match
  * @param cls_ptrn_len     the length of the class pattern to match
  */
Filter*
Filter::get_cls_fltr(Filter *cls_fltr_lst, unsigned int cls_fltr_lst_len,
                     char *cls_ptrn, enum GenericPattern cls_ptrn_type, unsigned int cls_ptrn_len) {
    unsigned int i; /* the loop control variable */
    Filter *cls_fltr; /* a class filter from the filter list */

    for (i = 0; i < cls_fltr_lst_len; i++) {

        cls_fltr = cls_fltr_lst + i;
        if (cls_fltr -> patternLength == cls_ptrn_len &&
            cls_fltr -> genericPattern == cls_ptrn_type &&
            !strcmp(cls_fltr -> pattern, cls_ptrn)) {
            return cls_fltr;
        }
    }
    return 0;
}

/** get_mthd_fltr *********************************************************************************
  * This static function returns a matching method filtering from the method filter list provided,
  * or a 0 pointer if none is found.  Matching filters must share the same filter pattern,
  * type, and length.
  *
  * @param mthd_fltr_lst        the method filter list from which a match is searched
  * @param mthd_fltr_lst_len    the length of the above list
  * @param mthd_ptrn            the method pattern to match
  * @param mthd_ptrn_type       the type of the method pattern to match
  * @param mthd_ptrn_len        the length of the method pattern to match
  */
MethodFilter_t*
Filter::get_mthd_fltr(MethodFilter_t *mthd_fltr_lst, unsigned int mthd_fltr_lst_len,
                              char *mthd_ptrn, enum GenericPattern mthd_ptrn_type, unsigned int mthd_ptrn_len) {
    unsigned int i; /* the loop control variable */
    MethodFilter_t *mthd_fltr; /* a method filter from the filter list */

    for (i = 0; i < mthd_fltr_lst_len; i++) {
        mthd_fltr = mthd_fltr_lst + i;
        if (mthd_fltr -> patternLength == mthd_ptrn_len &&
            mthd_fltr -> genericPattern == mthd_ptrn_type &&
            !strcmp(mthd_fltr -> pattern, mthd_ptrn)) {
            return mthd_fltr;
        }
    }
    return 0;
}

/******* CFilters ********************************************/

Filter* CFilters::m_Filters = 0;  /* Pointer to a list of active Filter elements 174190 made static */

int CFilters::m_filterCount = 0; /* Number of active Filter elements in m_Filters list */

int CFilters::m_filterListSize = FILTER_LIST_INCREMENT; /* Current size (in # of Filter elements) of the pending filter element buffer */

Filter *CFilters::m_filterListBuffer = 0; /* Pointer to a list of pending filter elements */

int CFilters::m_filterListBufferCount = 0; /* Number of pending filter elements in the pending list */

char CFilters::m_pathDelimiter;

CFilters::CFilters()
{
    InitializeFilters('.');
}

CFilters::~CFilters()
{
}

/*
 This function initializes the data structures used by the filter mechanism.
*/
void 
CFilters::InitializeFilters(char pathDelimiter)
{
    m_Filters = (Filter *)jvmtiAgent_Calloc(sizeof(Filter)*FILTER_LIST_INCREMENT);
    m_filterListBuffer = (Filter *)jvmtiAgent_Calloc(sizeof(Filter)*FILTER_LIST_INCREMENT);
    m_pathDelimiter = pathDelimiter;
}

/** jvmpiAgent_getFilter **************************************************************************
  * This function returns the first class filter that matches the class and method names.
  *
  * @param cls_name     the class name to match
  * @param mthd_name    the method name to match
  * @returns    the first matching class filter, or 0 if none is found.
  */
Filter* 
CFilters::GetFilter(const char *cls_name, const char *mthd_name) {
    return Filter::match_cls_mthd_fltr(m_Filters, m_filterCount, cls_name, mthd_name);
}

/** jvmpiAgent_checkMethodFilters *****************************************************************
  * This function determines whether a method is to be INCLUDED or EXCLUDED.  This can occur if
  * the class filter has a matching method filter, or if the class filter itself is 0, i.e.,
  * the class does not match any existing filter.
  *
  * @param mthd_name    the method name to match
  * @param cls_fltr     the matching filter for this class
  * @returns    INCLUDE or EXCLUDE as is appropriate.
  */
int 
CFilters::CheckMethodFilters(const char *mthd_name, Filter *cls_fltr) {
    MethodFilter_t *mthd_fltr; /* the method filter for the default methods */

    mthd_fltr = cls_fltr ? cls_fltr->match_mthd_fltr(cls_fltr -> methodDetails, cls_fltr -> methodDetailCount, mthd_name) : 0;
    return (mthd_fltr ? mthd_fltr -> mode : INCLUDE);
}


/*
 This function takes a filterpattern and determines if it is a prefix (leading "*"), suffix (trailing "*"), or
 exact match (no "*") type pattern.  It returns the type and adjusts the pointer to the filterPattern buffer
 to exclude the "*")
*/
enum GenericPattern 
CFilters::ParseFilterPattern(char ** filterPattern)
{
    enum GenericPattern genericPattern;

    char *p = strchr(*filterPattern, '*');
    if (p == 0) {
        genericPattern = NONE;
    } else if (p == *filterPattern) {
        genericPattern = PREFIX;
        (*filterPattern)++; /* Bump the buffer pointer past the "*" */
    } else if (p == (*filterPattern + strlen(*filterPattern) - 1)) {
        genericPattern = SUFFIX;
        *p = '\0'; /* Truncate the buffer to eliminate the "*" */
    } else {
        /* Error "*" found imbedded within pattern */
        genericPattern = NONE;
    }

    /*
    Translate the path delimiter character specified in the filterPattern to the path delimiter
    character used by the current JVM (this value was specified during filter initialization.
    */
    if (m_pathDelimiter == '.') {
        for (int i = strlen(*filterPattern) - 1; i >= 0; i--) {
            if ((*filterPattern)[i] == '/') (*filterPattern)[i] = '.';
        }
    } else if (m_pathDelimiter == '/') {
        for (int i = strlen(*filterPattern)-1; i >= 0; i--) {
            if ((*filterPattern)[i] == '.') {
                (*filterPattern)[i] = '/';
            }
        }
    }
    return genericPattern;

}

/** jvmpiAgent_addFilter **************************************************************************
  * This function appends a new filter using the class and method patterns with the filtering
  * mode.  The filters are saved in an ordered list, and the order in which they are inserted
  * makes a difference.  The filters will not be used until they are applied by calling
  * jvmpiAgent_ApplyFilters().  0, empty strings, and '*'s are all treated as wildcard
  * characters.
  *
  * @param cls_ptrn     the pattern to match a class
  * @param mthd_ptrn    the method pattern to match
  * @param mode         whether the method will be INCLUDED or EXCLUDED
  */
void 
CFilters::AddFilter(char *cls_ptrn, char *mthd_ptrn, enum FilterMode mode) {
    Filter *cls_fltr; /* the class filter to add or modify */
    MethodFilter_t *mthd_fltr; /* the method filter to add or modify */
    enum GenericPattern cls_ptrn_type, mthd_ptrn_type; /* the class and method pattern types */
    unsigned int cls_ptrn_len, mthd_ptrn_len; /* the class and method pattern lengths */

    /* Ensure that 0 pointers are replaced with empty strings to eliminate special cases. */
    if (cls_ptrn == 0) {
        cls_ptrn = "";
    }
    if (mthd_ptrn == 0) {
        mthd_ptrn = "";
    }

    /* Try to locate an existing class filter.  Create a new class filter if none found. */
    cls_ptrn_type = ParseFilterPattern(&cls_ptrn);
    cls_ptrn_len = strlen(cls_ptrn);
    cls_fltr = Filter::get_cls_fltr(m_filterListBuffer, m_filterListBufferCount, cls_ptrn, cls_ptrn_type, cls_ptrn_len);

    if (!cls_fltr) {
        /* Enlarge the filter buffer if necessary. */
        if (m_filterListBufferCount >= m_filterListSize) {
            m_filterListSize += FILTER_LIST_INCREMENT;
            m_filterListBuffer = (Filter *) realloc(m_filterListBuffer, sizeof(Filter)*m_filterListSize);
        }
        /* Create the class filter for this method. */
        cls_fltr = m_filterListBuffer + m_filterListBufferCount;
        cls_fltr -> genericPattern = cls_ptrn_type;
        cls_fltr -> patternLength = cls_ptrn_len;
        cls_fltr -> pattern = (char *) jvmtiAgent_Calloc(cls_ptrn_len + 1);
        strcpy(cls_fltr -> pattern, cls_ptrn);
        cls_fltr -> methodDetailCount = 0;
        m_filterListBufferCount++;
    } else if (cls_fltr -> methodDetailCount >= MAX_METHOD_FILTERS_PER_CLASS) {
        /* If there is no more room for the method filter, the filter will be dropped. */
        return;
    }

    /* If there already exists an identical filter, then the new filter will be ignored. */
    mthd_ptrn_type = ParseFilterPattern(&mthd_ptrn);
    mthd_ptrn_len = strlen(mthd_ptrn);
    mthd_fltr = Filter::get_mthd_fltr(cls_fltr -> methodDetails, cls_fltr -> methodDetailCount,
        mthd_ptrn, mthd_ptrn_type, mthd_ptrn_len);
    if (mthd_fltr) {
        return;
    }

    /* Insert the method filter into the class filter. */
    mthd_fltr = cls_fltr -> methodDetails + cls_fltr -> methodDetailCount;
    mthd_fltr -> genericPattern = mthd_ptrn_type;
    mthd_fltr -> patternLength = mthd_ptrn_len;
    strcpy((mthd_fltr -> pattern = (char *) jvmtiAgent_Calloc(mthd_ptrn_len + 1)), mthd_ptrn);
    mthd_fltr -> mode = mode;
    cls_fltr -> methodDetailCount++;
}


/* PR */
void 
CFilters::ChangeDelimiter()
{
    int i;
    char sunch = '.';
    char *tempFilterPattern;
    memcpy(m_filterListBuffer, m_Filters, m_filterCount * sizeof(Filter));
    for (i = 0; i < m_filterCount; i++) {
        int j = 0;
        tempFilterPattern = m_filterListBuffer[i].pattern;
        for (j = strlen(tempFilterPattern) - 1; j >= 0; j--) {
//            if(m_setPathDelimiter == sunch) {
                if(tempFilterPattern[j] == '/') {
                    (tempFilterPattern[j] = sunch);
                }
//            }
        }
        strcpy(m_filterListBuffer[i].pattern, tempFilterPattern);
    }
    ApplyFilters();
}
/* PR */

/*
 This function is used to replace the current set of filters with the pending set of filters.
*/

void 
CFilters::ApplyFilters()
{
    int i;
    if (!m_filterListBufferCount) return; /* Empty list so this may be an extaneous apply, so ignore it */

    for (i=0; i < m_filterCount; i++) { /* 173995 - replaced m_filterListBufferCount with m_filterCount for loop upper bound */
#ifndef _DEBUG
        free(m_Filters[i].pattern);
#endif
        m_Filters[i].pattern=0;
    }
    free(m_Filters);

    // Active filter list is sized to the current size of the pending filter list (buffer)
    m_Filters = (Filter *)jvmtiAgent_Calloc(sizeof(Filter)*m_filterListSize);

    // Copy all elements from the pending list to the active list
    memcpy(m_Filters, m_filterListBuffer, m_filterListBufferCount * sizeof(Filter));

    // Number of active elements is now set to the number of elements in the pending list
    m_filterCount = m_filterListBufferCount;

    // Clear the pending list
    m_filterListBufferCount = 0;
    memset(m_filterListBuffer, '\0', sizeof(Filter)*m_filterListSize);

    /* After the filter process, we need to set the tracing flags of the primitive classes */
    // jvmpiAgent_SetTracingFlagsOnPrimitives(); TODO ?????
}

#ifndef MVS
int 
CFilters::ProcessFilters(char * fileName) {
    FILE * filterFile = 0;
    char classPattern[250];
    char methodPattern[250];
    char mode[100];

    filterFile = fopen(fileName, "r");
    if (!filterFile) {
        // TODO: Add error message what filter file can't be open
        return -1;
    }

    while (fscanf(filterFile, "%s %s %s\n", classPattern, methodPattern, mode) != EOF) {
        AddFilter(classPattern, methodPattern, strcmp("INCLUDE", mode) == 0 ? INCLUDE : EXCLUDE);
    }
    fclose(filterFile);
    ApplyFilters();
    return 0;
}
#else

// z/OS specific filter read function

int CFilters::ProcessFilters(char * fileName) {
    FILE * filterFile = 0;
    char classPattern[250];
    char methodPattern[250];
    char mode[100];

    char readStr[65536];
    char *fgetResult;
    char *convStr;

    char *tokStr;

    const int CLASS_FIELD = 0;
    const int METHOD_FIELD = 1;
    const int MODE_FIELD = 2;

	int continueReading = 1; // Whether or not to continue the loop
	int firstField = 1; // Whether or not we have read the first field (of the file) yet
	int currField = 0; // The current field we are reading into (class / method / mode)

    filterFile = fopen(fileName, "r");
    if (!filterFile) {
        // TODO: Add error message what filter file can't be open
        return -1;
    }

	// Retrieve the entire file (64k limit, which is more than reasonable for this)
	fgetResult = fgets(readStr, 65530, filterFile);
	if(fgetResult == NULL /* && (feof(filterFile) || ferror(filterFile))*/  ) {
		return -1;
	}

	// Convert the entirety of the file to UTF
	Martini::OSA::native2unicode(&convStr, readStr, strlen(readStr));

	// Remove all CR/LFs
	for(int x = 0; x < strlen(convStr); x++ ) {
		if(convStr[x] == '\r') {
			convStr[x] = ' ';
		}

		if(convStr[x] == '\n') {
			convStr[x] = ' ';
		}
	}

	while(continueReading) {

		if(firstField) {

			// If this is the first field of the file, then strtok takes a non-null param
			tokStr = strtok(convStr, " ");
			firstField = 0;
			currField = METHOD_FIELD;

			if(tokStr != NULL) {
				strcpy(classPattern, tokStr);
			} else {
				continueReading = 0;
			}

		} else {

			if(tokStr != NULL) {
				// This is used for all string tokens beyond the first in the file
				tokStr = strtok(NULL, " ");

				if(tokStr == NULL) {
					continueReading = 0;
				} else {
					// Order is (loops): CLASS_FIELD -> METHOD_FIELD -> MODE_FIELD

					if(currField == CLASS_FIELD) {
						strcpy(classPattern, tokStr);
						currField = METHOD_FIELD;

					} else if(currField == METHOD_FIELD) {
						strcpy(methodPattern, tokStr);
						currField = MODE_FIELD;

					} else if(currField == MODE_FIELD) {
						strcpy(mode, tokStr);

						AddFilter(classPattern, methodPattern, strcmp("INCLUDE", mode) == 0 ? INCLUDE : EXCLUDE);
						currField = CLASS_FIELD;
					} else {
						continueReading = 0;
					}

				}

			} else {
				continueReading = 0;
			}

		}
	}

	if(convStr != NULL) {
		free(convStr);
	}

    fclose(filterFile);
    ApplyFilters();

    return 0;
}
#endif



void 
CFilters::PrintFilters(CPrint *p_pPrint)
{
    for (int i =0 ; i < m_filterCount; i++) {
        p_pPrint->printFilter(&m_Filters[i]);
    }
}
