/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation, Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    IBM Corporation - Initial API and implementation
 *    Viacheslav Rybalov, Intel - Initial API and implementation
 *
 * $Id: Filters.h,v 1.7 2010/04/08 20:34:17 jwest Exp $ 
 ***********************************************************************/

#ifndef __FILTERS_H__
#define __FILTERS_H__

#include "Print.h"

#define FILTER_LIST_INCREMENT 100
#define MAX_METHOD_FILTERS_PER_CLASS 5

namespace Martini { namespace JPIAgent {

enum FilterMode {EXCLUDE = 0, INCLUDE = 1};
enum GenericPattern {NONE, PREFIX, SUFFIX};

typedef struct {
    enum FilterMode mode; /* Include or exclude */
    enum GenericPattern genericPattern; /* Prefix or suffix */
    unsigned int patternLength;
    char * pattern;
} MethodFilter_t;

class Filter{
public:
    enum GenericPattern genericPattern;  /* Prefix or suffix (on the class filter) */
    unsigned int patternLength;
    char* pattern;

    // Additional method filters, for this class
    unsigned int methodDetailCount;
    MethodFilter_t methodDetails[MAX_METHOD_FILTERS_PER_CLASS];  /* Initial limit of specific methods per class */
 
    enum FilterMode GetClassFilterMode(Filter *cls_fltr);

    // Static functions

    /* The difference between the 'match' and 'get' methods is that the 'match' methods try to match a
     * class name/method name in the list, while the 'get' methods try to match a pattern/Generic Pattern/length
     * in the list.
     *  */

    /* Given a class and method name, function returns the first class filter in the list that matches */
    static Filter* match_cls_mthd_fltr(Filter *cls_fltr_lst, int cls_fltr_lst_len, const char *cls_name, const char *mthd_name);

    /* Given a method name, function returns the first method filter from the list that matches */
    static MethodFilter_t* match_mthd_fltr(MethodFilter_t *mthd_fltr_lst, const unsigned int mthd_fltr_lst_len, const char *mthd_name);

    /* Given a method filter list, and a method ptr/pattern/length combination, find the MethodFilter_t in the list that matches the combination */
    static MethodFilter_t* get_mthd_fltr(MethodFilter_t *mthd_fltr_lst, unsigned int mthd_fltr_lst_len,
                              char *mthd_ptrn, enum GenericPattern mthd_ptrn_type, unsigned int mthd_ptrn_len);

    /* Given a class filter list, and a class ptrn text/pattern type/length combination, find the matching Filter (class filter)*/
    static Filter* get_cls_fltr(Filter *cls_fltr_lst, unsigned int cls_fltr_lst_len,
                     char *cls_ptrn, enum GenericPattern cls_ptrn_type, unsigned int cls_ptrn_len);
protected:

    /* Generic function, that, given an str, will determine if the ptrn matches (pattern type is either PREFIX or SUFFIX */
    static int str_fits_ptrn(const char* str, const char* ptrn, const int ptrn_type, const int ptrn_len);
};

class CPrint;

class CFilters
{
public:

    CFilters();
    ~CFilters();

/*
 This function is used in standalone mode to read and parse a file containing a set of filter
 definitions.  It uses the jvmpiAgent_AddFilter and jvmpiAgent_ApplyFilters functions to
 actually set the active filter definitions for the agent.
*/
int ProcessFilters(char * fileName);

/** jvmpiAgent_checkMethodFilters *****************************************************************
  * This function determines whether a method is to be INCLUDED or EXCLUDED.  This can occur if
  * the class filter has a matching method filter, or if the class filter itself is 0, i.e.,
  * the class does not match any existing filter.
  *
  * @param mthd_name    the method name to match
  * @param cls_fltr     the matching filter for this class
  * @returns    INCLUDE or EXCLUDE as is appropriate.
  */
int CheckMethodFilters(const char *mthd_name, Filter *cls_fltr);

/** jvmpiAgent_getFilter **************************************************************************
  * This function returns the first class filtering that matches the class and method names.
  *
  * @param cls_name     the class name to match
  * @param mthd_name    the method name to match
  * @returns    the first matching class filter, or 0 if none is found.
  */
Filter* GetFilter(const char *cls_name, const char *mthd_name);


/*
 This function outputs the currently active filter definitions as elements in the trace.
*/
void PrintFilters(CPrint *p_pPrint);

/*
 This function is takes the filter definitions accumulated using the jvmpiAgent_AddFilter function,
 and replaces the current set of active filter definitions.  In addition, the accumlated filter definitions
 are cleared.
 Note: It is recommended that the JVM be suspended during the execution of this function in order
 to avoid accessing the filter definitions while they are in an indeterminate state.
*/
void ApplyFilters();

/** jvmpiAgent_addFilter **************************************************************************
  * This function appends a new filter using the class and method patterns with the filtering
  * mode.  The filters are saved in an ordered list, and the order in which they are inserted
  * makes a difference.  The filters will not be used until they are applied by calling
  * jvmpiAgent_ApplyFilters().  0, empty strings, and '*'s are all treated as wildcard
  * characters.
  *
  * @param cls_ptrn     the pattern to match a class
  * @param mthd_ptrn    the method pattern to match
  * @param mode         whether the method will be INCLUDED or EXCLUDED
  */
void AddFilter(char *classPattern, char *methodPattern, enum FilterMode mode);

/** jvmpiAgent_getClassFilterMode *****************************************************************
  * This function determines whether a class is to be INCLUDED or EXCLUDED.  This can occur if
  * the class filter has a wildcard method filter, or if the class filter itself is 0, i.e.,
  * the class does not match any existing filter.
  *
  * @param cls_fltr the matching filter for this class
  * @returns    INCLUDE or EXCLUDE as is appropriate.
  */

/* Initialize the Filter data structures */
void InitializeFilters(char pathChar);

void ChangeDelimiter(void);

protected:
    static Filter* m_Filters;  /* Pointer to a list of active Filter elements 174190 made static */
    static int m_filterCount; /* Number of active Filter elements in list */
    static int m_filterListSize; /* Current size (in # of Filter elements) of
                                                    the pending Filter element buffer */
    static Filter* m_filterListBuffer; /* Pointer to a list of pending Filter elements */
    static int m_filterListBufferCount; /* Number of pending Filter elements in list */

    static char m_pathDelimiter;

    enum GenericPattern ParseFilterPattern(char ** filterPattern);
}; // class CFilters

} /* namespace Martini */ } /*namespace JPIAgent*/

#endif
