/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation, Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    IBM Corporation - Initial API and implementation
 *    Viacheslav Rybalov, Intel - Initial API and implementation
 *
 * $Id: Options.cpp,v 1.38 2010/05/25 14:48:19 jcayne Exp $ 
 ***********************************************************************/

#include <string.h>
#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>

#ifdef MVS
	#include <strings.h>
#endif

#ifdef _WIN32
#include <io.h>
#include <direct.h>
#define MKDIR(dirname) mkdir(dirname)
#else
#define MKDIR(dirname) mkdir((dirname), S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH)
#endif

#include "Options.h"
#include "ECDefs.h"

using namespace Martini::JPIAgent;

///////////////////////////////////////////////////////////////////////////


#ifdef _WIN32
#define STRICOLL _stricoll        /* Case insensitive on Windows */
#else
//#define STRICOLL strcoll        /* Case sensitive on other platforms */
#define STRICOLL strcasecmp        /* Case insensitive on other platforms */
#endif

#define BOOLTOSTRING(b) (b ? "true" : "false")
static const char* tiAgentVersion= "@(#)JVMTI Agent version 1.0.0.1";  //TODO

static const char* DEFAULT_FILTER_FILENAME      = "filters.txt";
static const char* DEFAULT_PROCESS_NAME         = "org.eclipse.tptp.jvmti";//"JVMTI Profiling Agent";
static const char* DEFAULT_PROCESS_TYPE         = "Profiler";
static const char* DEFAULT_OUTPUT_FILENAME      = "trace.trcxml";
static const char* DEFAULT_OUTPUT_BIN_FILENAME  = "trace.trcbin";  // default binary trace file name
static const char* DEFAULT_PROFILE_FILE         = "jvmti.pro";
static const char* DEFAULT_WORK_DIR             = ".";
static const char* DEFAULT_FORMAT				= XML_TRACE_FORMAT;

Options COptions::m_jvmtiAgent_Options =     
{
         1                         /* Provide timestamps by default */
        ,0                         /* Standalone mode */
        ,0                         /* profilerApiEnabled */
        ,1                         /* Enabled mode */
        ,1                         /* Filters */
        ,1                         /* Options */
        ,0                         /* TraceIdrefs  */
        ,1                         /* TICKET (default) */
        ,0                         /* cpuTime */
        ,1 //TODO
        ,1                         /* Call graph profiler specific option. Modify default vaule as true for bug 293378 */
        ,1                         /* Call graph profiler specific option.*/
        ,0
        ,XmlFormat		           /* Default output format.*/
		,0						   /* ThreadProf option: contention analysis (off) */
        ,0                         /* Global option: StackMapTable calculation (off) */
        ,StackInfoNormal           /* Stack mode preference (default) */
        ,0                         /* HeapInstanceData option */
        ,0                         /* The invocation options from the command line */
        ,0                         /* Filter File  (to be initialized) */
        ,0                         /* Default process name (to be initialized) */
        ,0                         /* Default process type (to be initialized) */
        ,0                         /* Default trace file name (to be initialized) */
        ,0                         /* Default profile file name (to be initialized) */
        ,0                         /* Defualt working dir (to be initialized) */
        ,0                         /* currently no unknown options */ 
		,STACK_MAX_DEPTH_INFINITE
};

COptions::COptions()
{
}

COptions::~COptions()
{
}

void 
COptions::InitializeJvmtiAgentOptionsDefaults() {
    /* initialize filterFileName */ 
    m_jvmtiAgent_Options.filterFileName = (char *) jvmtiAgent_Calloc(strlen(DEFAULT_FILTER_FILENAME)+1);
    strcpy(m_jvmtiAgent_Options.filterFileName, DEFAULT_FILTER_FILENAME);

    /* initialize process name */ 
    m_jvmtiAgent_Options.processName = (char *)jvmtiAgent_Calloc(strlen(DEFAULT_PROCESS_NAME)+1); 
    strcpy(m_jvmtiAgent_Options.processName, DEFAULT_PROCESS_NAME); 

    /* initialize process type */
    m_jvmtiAgent_Options.processType = (char *)jvmtiAgent_Calloc(strlen(DEFAULT_PROCESS_TYPE)+1); 
    strcpy(m_jvmtiAgent_Options.processType, DEFAULT_PROCESS_TYPE); 

    /* initialize output file */ 
    m_jvmtiAgent_Options.outputFileName = (char *)jvmtiAgent_Calloc(strlen(DEFAULT_OUTPUT_FILENAME)+1);
    strcpy(m_jvmtiAgent_Options.outputFileName, DEFAULT_OUTPUT_FILENAME);

    /* intiialize the profile file */ 
    m_jvmtiAgent_Options.profileFile = (char *)jvmtiAgent_Calloc(strlen(DEFAULT_PROFILE_FILE)+1); 
    strcpy(m_jvmtiAgent_Options.profileFile, DEFAULT_PROFILE_FILE);

    /* initiailize the working dir */ 
    m_jvmtiAgent_Options.workDir = (char *)jvmtiAgent_Calloc(strlen(DEFAULT_WORK_DIR)+1); 
    strcpy(m_jvmtiAgent_Options.workDir, DEFAULT_WORK_DIR); 
}

/** PRINT_USAGE **********************************************************************
* Print the command line useage of the program.
*/
void 
COptions::printUsage()
{
    fprintf(stdout, "\n%s\n"
        "Usage:\n"
        "  -agentlib:JPIBootLoader=JPIAgent[:[help]|[<option>=<value>, ...]];<profiler>\n"
        "\n"
        "Where:\n"
        "  help\t\t\tDisplays this message\n"
        "  <option>=<value>\tCommand line options (see below)\n"
        "  <profiler>\t\tThe profiler to launch:\n"
        "\tCGProf\t\tExecution time analysis\n"
        "\tHeapProf\tMemory analysis\n"
        "\tThreadProf\tThread analysis\n"
        "\n"
        "Supported option names and values:\n"
        "  server=standalone|enabled|controlled\n" 
        "  api=true|false\tWhether to enable the Profiler API or not.\n"
        "  \t\t\tDefault is false.\n" 
        "  stackmap=true|false\tWhether to recalculate the StackMapTable attribute of\n"
        "  \t\t\tinstrumented methods.\n"
        "  \t\t\tUse only when experiencing verification errors in \n"
        "  \t\t\tJava 1.6+. Default is false.\n"
        "  file=<file>\t\tOutput file (default is %s)\n"
        "  \t\t\tOnly applicable when server=standalone\n"
        "  filters=<file>\tFilter definition file (default is %s)\n"
        "  \t\t\tOnly applicable when server=standalone\n"
        "  profiler=<file>\tProfiling options file (default is %s)\n"
        "  \t\t\tOnly applicable when server=standalone\n"
        "  format=binary|xml\tOutput trace format (default is %s)\n"
        "\n"
        "Examples\n"
        "\n"
        "Execution time analysis in standalone mode:\n"
        "java -agentlib:JPIBootLoader=JPIAgent:server=standalone,file=log.trcxml;CGProf <Java class file>\n"
        "\n"
        "Memory analysis in controlled mode:\n"
        "java -agentlib:JPIBootLoader=JPIAgent:server=controlled;HeapProf <Java class file>\n"
        "\n", tiAgentVersion, DEFAULT_OUTPUT_FILENAME, DEFAULT_FILTER_FILENAME, DEFAULT_PROFILE_FILE, DEFAULT_FORMAT);
    fflush(stdout); 
}

/** STRIP_LEADING_BLANKS **************************************************************
* Remove all spaces from the beginning of a buffer
*/
static char * stripLeadingBlanks(char * buffer)  //TODO Check alg!
{
    size_t len = strlen(buffer);
    for (size_t i = 0; i <= len; i++) {
        if (!isspace(buffer[i])) {
            return buffer + i;
        }
    }
    return buffer;
}

int
COptions::setBooleanOption(char * value, char * key, int defaultValue)
{
    if (STRICOLL(value, "true") == 0) {
        return 1;
    } else if(STRICOLL(value, "false") == 0) {
        return 0;
    } else {
        fprintf(stderr, "Invalid %s option value \"%s\"\n", key, value);
        fflush(stderr); 
        return defaultValue;
    }
}

int
COptions::setIntOption(char * value, char * key, int defaultValue)
{
	int depth = atoi(value);
	if(depth <= 0){
        fprintf(stderr, "Invalid %s option value \"%s\"\n", key, value);
        fflush(stderr); 
		return defaultValue;
	}

	return depth;
}

/* Piyush Agarwal 
Get the absolute directory path for the optheap working dir*/
void 
COptions::getWorkDir(char* buf)
{
    realpath(m_jvmtiAgent_Options.workDir, buf);
    buf += strlen(buf);
#ifdef _WIN32
    *buf = '\\'; /* file delim in Windows */
#else
    *buf = '/'; /* file delim in Unix */
#endif
    buf += 1; 
    *buf = '\0';

}
/** PRINT_OPTIONS *********************************************************************
*
*/
void
COptions::printOptions(CPrint *p_pPrint)
{
    p_pPrint->PrintOption("FILTERS", BOOLTOSTRING(m_jvmtiAgent_Options.filters));
    p_pPrint->PrintOption("OPTIONS", BOOLTOSTRING(m_jvmtiAgent_Options.options));
    p_pPrint->PrintOption("STACK_INFORMATION", m_jvmtiAgent_Options.stackInfo == StackInfoNone ? "none" : m_jvmtiAgent_Options.stackInfo == StackInfoNormal ? "normal" : "bad");
    p_pPrint->PrintOption("TICKET", BOOLTOSTRING(m_jvmtiAgent_Options.ticket));
    p_pPrint->PrintOption("TIMESTAMPS", BOOLTOSTRING(m_jvmtiAgent_Options.timestamp));
    p_pPrint->PrintOption("TRACE_IDREFS", BOOLTOSTRING(m_jvmtiAgent_Options.traceIdrefs));
    p_pPrint->PrintOption("CPU_TIME", BOOLTOSTRING(m_jvmtiAgent_Options.cpuTime));
//    p_pPrint->PrintOption("ALLOCSITES", BOOLTOSTRING(m_jvmtiAgent_Options.allocSites));

    {
        generic_option *cur_unknown = m_jvmtiAgent_Options.unknowns; 
        while (cur_unknown != 0) {
            p_pPrint->PrintOption(cur_unknown->key, cur_unknown->value);
            cur_unknown = cur_unknown->next; 
        }
    }
}

/**
* addUnknownOption - 60879 add a key,value pair that the piAgent doesn't care about
* as an 'unknown' to the m_jvmtiAgent_Options structure so that we can later echo
* it back to the UI. 
***/ 
void 
COptions::addUnknownOption(char *key, char *value) {
    /* m_jvmtiAgent_Options.unknowns is essentially a linked list. We only expect to encounter a dozen 
    or so unknown options, and moreover, we only do this processing once at start up. Hence the
    performance impact of not using a more sophisticated data structure is negligible.  */ 


    /* find a spot in the linked list where we can add the new unknown */ 
    generic_option *new_unknown = 0; 
    if (m_jvmtiAgent_Options.unknowns == 0) {
        m_jvmtiAgent_Options.unknowns = (generic_option *) malloc(sizeof(generic_option)); 
        memset(m_jvmtiAgent_Options.unknowns,0,sizeof(generic_option));
        new_unknown = m_jvmtiAgent_Options.unknowns; 
    } else {
        generic_option *cur_unknown = m_jvmtiAgent_Options.unknowns; 
        while (cur_unknown->next != 0) {
            cur_unknown = cur_unknown->next; 
        }
        new_unknown = (generic_option *) malloc(sizeof(generic_option)); 
        memset(new_unknown,0,sizeof(generic_option)); 
        cur_unknown->next = new_unknown; 
    }

    /* set the new unknown option */ 
    new_unknown->key = (char *) malloc(strlen(key)*sizeof(char)+1); 
    strcpy(new_unknown->key,key); 
    new_unknown->value = (char *) malloc(strlen(value)*sizeof(char)+1); 
    strcpy(new_unknown->value,value); 
}


/** SET_PROFILE_OPTION *******************************************************************
*
*/
int
COptions::SetProfileOption(char * key, char * value)
{

    if (key == 0 || value == 0) {
        return 0;
    }

    if (STRICOLL(key, "FILTERS") == 0) {
        m_jvmtiAgent_Options.filters = setBooleanOption(value, key, m_jvmtiAgent_Options.filters);
    } else if (STRICOLL(key, "OPTIONS") == 0) {
        m_jvmtiAgent_Options.options = setBooleanOption(value, key, m_jvmtiAgent_Options.options);
    } else if (STRICOLL(key, "STACK_INFORMATION") == 0) {
        if (STRICOLL(value, "none") == 0) {
            m_jvmtiAgent_Options.stackInfo = StackInfoNone;
        } else if (STRICOLL(value, "normal") == 0) {
            m_jvmtiAgent_Options.stackInfo = StackInfoNormal;
        } else {
            fprintf(stderr, "Invalid %s option value \"%s\"\n", key, value);
            fflush(stderr); 
            return -1;
        }
    } else if (STRICOLL(key, "TICKET") == 0) {
        m_jvmtiAgent_Options.ticket = setBooleanOption(value, key, m_jvmtiAgent_Options.ticket);
    } else if (STRICOLL(key, "TIMESTAMPS") == 0) {
        m_jvmtiAgent_Options.timestamp = setBooleanOption(value, key, m_jvmtiAgent_Options.timestamp);
    } else if (STRICOLL(key, "TRACE_IDREFS") == 0) {
        m_jvmtiAgent_Options.traceIdrefs = setBooleanOption(value, key, m_jvmtiAgent_Options.traceIdrefs);
    } else if (STRICOLL(key, "THREAD_CPU_TIME") == 0 || STRICOLL(key, "CPU_TIME") == 0 ) {
        m_jvmtiAgent_Options.cpuTime = setBooleanOption(value, key, m_jvmtiAgent_Options.cpuTime);
    } else if (STRICOLL(key, "EXECDETAILS") == 0) {
        m_jvmtiAgent_Options.cgExecDetails = setBooleanOption(value, key, m_jvmtiAgent_Options.cgExecDetails);
    } else if (STRICOLL(key, "MEMORYFREE") == 0) {
        m_jvmtiAgent_Options.cgAggStackMemoryFree = setBooleanOption(value, key, m_jvmtiAgent_Options.cgAggStackMemoryFree);
    } else if (STRICOLL(key, "ALLOCSITES") == 0) {
        m_jvmtiAgent_Options.allocSites = setBooleanOption(value, key, m_jvmtiAgent_Options.allocSites);
    } else if (STRICOLL(key, "CONTANALYSIS") == 0) {
        m_jvmtiAgent_Options.contAnalysis = setBooleanOption(value, key, m_jvmtiAgent_Options.contAnalysis);
    } else if (STRICOLL(key, "STACKMAP") == 0) {
        m_jvmtiAgent_Options.calcStackMap = setBooleanOption(value, key, m_jvmtiAgent_Options.calcStackMap);
    } else if (STRICOLL(key, "HEAPINSTANCEDATA") == 0) {
        m_jvmtiAgent_Options.heapInstanceData = setBooleanOption(value, key, m_jvmtiAgent_Options.heapInstanceData);
    } else if(STRICOLL(key, "MAXSTACKDEPTH") == 0) { 
    	m_jvmtiAgent_Options.maxStackDepth = setIntOption(value, key, m_jvmtiAgent_Options.maxStackDepth);
    } else {
        /* 60879 add the unknown profiling option to the 'unknowns' list so that we 
        can echo it back to the UI */ 
        addUnknownOption(key,value); 
    }
    
    return 0;
}


int 
COptions::ProcessProfile(char * fileName)
{
    int rc = 0, linelen;
    FILE * profileFile = 0;
    fpos_t filepos; 
    char *buffer; 
    char *strippedBuffer; 

    profileFile = fopen(fileName, "r");
    if (!profileFile) return -1;
    /* loop until we reach the end of file */ 
    while (feof(profileFile) == 0) {
        int len, i, ch;
        char *end;
        char *key;
        char *value; 

        /* save current file position */ 
        if (fgetpos(profileFile,&filepos) != 0) {
            return -1; 
        }

        /* determine the length of the current line */ 
        linelen = 0; 
        do {
            ch = fgetc(profileFile); 
            linelen++; 
        } while(ch != EOF && ch != '\n'); 
        /* allocate enough buffer space for the current line */ 
        buffer = (char *)malloc(linelen + 1); 
        /* rewind to the beginning of the line */ 
        if (fsetpos(profileFile,&filepos) != 0) {
            free(buffer);
            return -1;
        }

        /* read the line */ 
        if (!fgets(buffer,linelen+1,profileFile)) {
            /* loop termination condition */ 
            free(buffer); 
            break; 
        }
        /* Truncate the buffer at the start of a comment */

        end = strchr(buffer, '*');

        if (end) {
            *end = '\0';
        }

        strippedBuffer = stripLeadingBlanks(buffer);
        len = strlen(strippedBuffer);
        if (!len) {
            /* free the original buffer */
            free(buffer);
            continue;
        }

        /* Parse key */
        for (i = 0; i < len; i++) {
            if (isspace(strippedBuffer[i]) || strippedBuffer[i] == '=') {
                key = (char *)malloc(i+1);
                strncpy(key, strippedBuffer, i);
                key[i] = '\0';
                strippedBuffer = strippedBuffer+i;
                break;
            }
        }  
        strippedBuffer = stripLeadingBlanks(strippedBuffer);

        if (strippedBuffer[0] != '=') {
            fprintf(stderr, "Syntax error in profile file\n");
            fflush(stderr); 
            rc = -1;
            /* free the original buffer */ 
            free(buffer); 
            break;
        } else {
            strippedBuffer = strippedBuffer+1;
            strippedBuffer = stripLeadingBlanks(strippedBuffer);
            /* Parse value */
            for (i = 0; i <= len; i++) {
                if (isspace(strippedBuffer[i]) || strippedBuffer[i] == '\0') {
                    value = (char *)malloc(i+1); 
                    strncpy(value, strippedBuffer, i);
                    value[i] = '\0';
                    break;
                }
            }
            /*MW changed so that one bad option won't stop option processing */
            /*    if ( */
            SetProfileOption(key,value);
            /*    == -1)
            {
            rc = -1;
            break;
            }
            */ }

            free(buffer); 
            free(key);
            free(value); 
    }

    fclose(profileFile);
    return rc;
}

int COptions::parseOptions(char* buf) {
	int outputSpecified = 0;

	char* pnext = buf;
    while (pnext) {
    	char* pname = pnext;	// current tag
    	
    	pnext = strchr(pnext, ',');
    	if (pnext != NULL) {
    		*pnext++ = '\0';	// next tag
    	}
    	
    	char* pvalue = strchr(pname, '=');
    	if (pvalue != NULL) {
    		*pvalue++ = '\0';
    	}

        /* Specied output file */
        if (STRICOLL(pname, "file") == 0) {
            if (outputSpecified || pvalue == NULL) return -1;
            
            strcpyrealloc(&m_jvmtiAgent_Options.outputFileName, pvalue);
            outputSpecified = 1;
        } else if (STRICOLL(pname, "filters") == 0) {
            if (pvalue == NULL) return -1;
            
            strcpyrealloc(&m_jvmtiAgent_Options.filterFileName, pvalue);
        }
        /* Mode of operation */
        else if (STRICOLL(pname, "server") == 0) {
            if (pvalue == NULL) return -1; 

            /* If standalone */
            if (STRICOLL(pvalue,"standalone") == 0) {
                m_jvmtiAgent_Options.standalone = 1;
                m_jvmtiAgent_Options.enabled = 0;
            }
            /* If enabled */
            else if (STRICOLL(pvalue, "enabled") == 0) {
                m_jvmtiAgent_Options.standalone = 0;
                m_jvmtiAgent_Options.enabled = 1;
            }
            /* If controlled */
            else if (STRICOLL(pvalue, "controlled") == 0) {
                m_jvmtiAgent_Options.standalone = 0;
                m_jvmtiAgent_Options.enabled = 0;
            } else {
                return -1;
            }
        } else if (STRICOLL(pname, "api") == 0) {
            if (pvalue == NULL) return -1; 
        
            if (STRICOLL(pvalue, "true") == 0) {
            	m_jvmtiAgent_Options.profilerApiEnabled = 1;
            }
        } else if (STRICOLL(pname, "workDir") == 0) {
            if (pvalue == NULL) return -1;

            strcpyrealloc(&m_jvmtiAgent_Options.workDir, pvalue);

            /* create the directory */
            if (MKDIR(m_jvmtiAgent_Options.workDir) != 0 && errno != EEXIST) {
                /* couldn't create the dir and it doesn't exists either */
                perror(m_jvmtiAgent_Options.workDir);
                return -1;
            }

            /* end of workDir */
        } else if (STRICOLL(pname, "profile") == 0) {
            m_jvmtiAgent_Options.profileFile;
            if (pvalue == NULL) return -1;

            strcpyrealloc(&m_jvmtiAgent_Options.profileFile, buf);
        }
        /* Ouput format */
        else if (STRICOLL(pname, "format") == 0) {
            if (pvalue == NULL) return -1;

            /* If binary */
            if (STRICOLL(pvalue,"binary") == 0) {
                m_jvmtiAgent_Options.format = BinaryFormat;

                // if output trace file name is not set when format is binary,
                // using default binary format trace file name
                if (!outputSpecified) {
                    strcpyrealloc(&m_jvmtiAgent_Options.outputFileName,
                    		DEFAULT_OUTPUT_BIN_FILENAME);
                }
            }
            /* If XML */
            else if (STRICOLL(pvalue, "xml") == 0) {
                m_jvmtiAgent_Options.format = XmlFormat;
            } else {
				return -1;
            }
        }
		else if (STRICOLL(pname, "stackmap") == 0) {
            if (pvalue == NULL) return -1; 
        
            if (STRICOLL(pvalue, "true") == 0) {
            	m_jvmtiAgent_Options.calcStackMap = 1;
            }
        }
		else {
            if (pvalue == NULL) return -1;
            //fprintf(stderr, "Unknown profiler external control option %s=%s\n", pname, pvalue);
            return SetProfileOption(pname, pvalue);
        }
	}
	
	return 0;
}

/** PROCESS_INVOCATION_OPTIONS *******************************************
* Takes the command line parameters and populates the m_jvmtiAgent_Options
* with the correct values based upon the options specified.
* @param   optionString - the command line args
* @returns
*/

int COptions::ProcessInvocationOptions(const char *str) {
    char* buf;

    if (str == NULL) return 0;

    /* if "help" is specified print out the useage */
    if ((STRICOLL(str, "help")) == 0) {
        return -1;
    }

	buf = (char*) malloc(strlen(str) + 1);
	if (buf == NULL) {
    	fprintf(stderr, "Memory allocation error\n"); 
    	fflush(stderr);
    	return -1;
    } 

	strcpy(buf, str);
	 
    int r = parseOptions(buf);
    free(buf);
    
    if (r < 0) {
    	fprintf(stderr, "Illegal TIAgent option\n");
    	fflush(stderr);
    }
    
    return r;
}

/** CHECK_OPTIONS_CONSISTENCY ********************************************
* Checks whether there is any conflict or inconsistency among various
* options.
* @returns -1 if error, 0 if no error
*/
int 
COptions::CheckOptionsConsistency()
{
    return 0;
}

const char* 
COptions::getUnknownOptionByName( const char* name ) 
{
	generic_option* opt = m_jvmtiAgent_Options.unknowns;
    while( opt ) {
    	if( opt->key && STRICOLL( opt->key, name ) == 0 ) {
        	return opt->value;
		}
		opt = opt->next;
	}
    return 0;	
}
