/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation, Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    IBM Corporation - Initial API and implementation
 *    Viacheslav Rybalov, Intel - Initial API and implementation
 *
 * $Id: Options.h,v 1.24 2010/05/25 14:48:19 jcayne Exp $ 
 ***********************************************************************/

#ifndef __OPTIONS_H__
#define __OPTIONS_H__

#include "Print.h"

namespace Martini { namespace JPIAgent {

#define XML_TRACE_FORMAT "xml"
#define BINARY_TRACE_FORMAT "binary"

	enum TraceFormat { BinaryFormat, XmlFormat };

    enum StackInfo {StackInfoNone, StackInfoNormal};

    /* 60879 a structure used for storing options that the piAgent otherwise doesn't care about. 
    This is used mainly so that we can echo back these options to the UI. */ 

    typedef struct gen_option {
        char *key; 
        char *value;
        struct gen_option *next;  
    } generic_option; 

    /** These are the global options specified on the command line and/or profile of the agent at
    invocation time.
    */
    
    const int STACK_MAX_DEPTH_INFINITE = -1;
    
    typedef struct {
        unsigned int timestamp:1;                  /* Provide timestamps on all events (TRUE) */
        unsigned int standalone:1;                 /* Standalone mode  (FALSE) */
        unsigned int profilerApiEnabled:1;         /* isProfilerApiEnabled (FALSE) */
        unsigned int enabled:1;                    /* Enabled mode  ie. attachable (TRUE)  */
        unsigned int filters:1;                    /* Print FILTER Elements (TRUE) */
        unsigned int options:1;                    /* Print OPTIONS Elements (TRUE) */
        unsigned int traceIdrefs:1;                /* Print trace_idref attribute on each element  */
        unsigned int ticket:1;                     /* TICKET (TRUE) */
        unsigned int cpuTime:1;                    /* Are we emmiting thread cpuTime (FALSE) */
        unsigned int objAllocIsArray:1; //TODO
        unsigned int cgExecDetails:1;              /* Call graph profiler specific option. Is sggregated? (FALSE) */ 
        unsigned int cgAggStackMemoryFree:1;       /* Call graph profiler specific option. Memory free after flush (TRUE)*/ 
        unsigned int allocSites:1;                 /* Print allocsites information*/
        enum TraceFormat format;				   /* Binary or XML output format*/
        unsigned int contAnalysis:1;               /* Provide information for contention analysis*/
        unsigned int calcStackMap:1;               /* Enables recalculation of the StackMapTable attribute for Java class versions >= 50.0 */
        enum StackInfo stackInfo;                  /* Stack maintenance mode  (StackInfoContiguous)*/
        unsigned int heapInstanceData:1;		   /* Store heapInstanceData information*/
        char *invocationOptions;                   /* The invocation options from the command line */
        char *filterFileName;                      /* Name of the filters file (filters.txt)*/
        char *processName;                         /* Name of the process */
        char *processType;                         /* The type of process */
        char *outputFileName;                      /* Output file for trace */
        char *profileFile;                         /* Name of the profile file */
        char *workDir;                             /* Working directory (default is .) */
        generic_option *unknowns;                  /* space to store the options the piAgent doesn't care about, but can echo back to the UI */
        int maxStackDepth;							/* maximum stack depth */
    } Options;

    class CPrint;

    class COptions
    {
    public:

        // constructor
        COptions();
        // destructor
        ~COptions();

        /* The structure to hold the invocation options for this process */
        static Options m_jvmtiAgent_Options;

        /** PROCESS_INVOCATION_OPTIONS *******************************************
        * Takes the command line parameters and populates the _jvmpiAgent_Options
        * with the correct values based upon the options specified.
        * @param   optionString - the command line args
        * @returns
        */
        int ProcessInvocationOptions(const char *optionString);

        /** PROCESS_PROFILE  ******************************************************
        * Process the specified profile and set the profile options within the
        * agent.
        * @param    profileName - the name of the profile file.
        * @returns
        */
        int ProcessProfile(char *profileName);

        /** SET_PROFILE_OPTION ****************************************************
        * Sets the profile option specified by key to the value specified by
        * value.
        * @param     key - the option that will be set
        * @param   value - the value that will be associated with the key.
        * @returns
        */
        int SetProfileOption(char * key, char * value);

        /** CHECK_OPTIONS_CONSISTENCY ********************************************
        * Checks whether there is any conflict or inconsistency among various
        * options.
        * @returns -1 if error, 0 if no error
        */
        int CheckOptionsConsistency();

        /* initializeJvmpiAgentOptionsDefaults() - must be called at start up time */ 
        void InitializeJvmtiAgentOptionsDefaults();

        /** PRINT_OPTIONS ********************************************************
        * Prints the options
        */
        void printOptions(CPrint *p_pPrint);
        void printUsage();

        bool isStandAlone() {
            return (m_jvmtiAgent_Options.standalone == 1);
        }

        /* If the piAgent is running in a controlled mode */
        bool isControlled() {
            return (m_jvmtiAgent_Options.standalone == 0 &&  m_jvmtiAgent_Options.enabled == 0) ;
        }

        bool isEnabled() {
            return (m_jvmtiAgent_Options.enabled == 1) ;
        }

        bool isProfilerApiEnabled() {
            return (m_jvmtiAgent_Options.profilerApiEnabled == 1);
        }

        // is CG pfofiler aggregated
        bool isCGExecDetails() {
            // Modified for bug 194081
            //return (!isEnabled()) && (m_jvmtiAgent_Options.cgExecDetails == 1); //in enabled mode we support aggregated cg profiler only
            return (m_jvmtiAgent_Options.cgExecDetails == 1);
            // Modified for bug 194081
        }

        // if cg memory should be freed after snapshot then returns true
        bool isStackMemoryFree() {
            return (m_jvmtiAgent_Options.cgAggStackMemoryFree == 1)/* || isStandAlone()*/;
        }

        bool isStackInfoNormal() {
            return m_jvmtiAgent_Options.stackInfo == StackInfoNormal;
        }

        bool isAllocSitesSupported() {
            return m_jvmtiAgent_Options.allocSites == 1;
        }

        bool isContentionAnalysisSupported() {
            return m_jvmtiAgent_Options.contAnalysis == 1;
        }
        
        // Is StackMap calculation enabled for Java class file versions >= 50.0 (Java 6)
        bool isCalcStackMap() {
            return m_jvmtiAgent_Options.calcStackMap == 1;
        }

        // Is tracking of heap instance data enabled
        bool isHeapInstanceDataEnabled() {
			return m_jvmtiAgent_Options.heapInstanceData == 1;
		}

        const char* getUnknownOptionByName( const char* name );

        int getMaxStackDepth() {
        	return m_jvmtiAgent_Options.maxStackDepth;
        }
    private:
        /**
        * addUnknownOption - 60879 add a key,value pair that the piAgent doesn't care about
        * as an 'unknown' to the m_jvmtiAgent_Options structure so that we can later echo
        * it back to the UI. 
        ***/ 
        void addUnknownOption(char *key, char *value);

		int parseOptions(char* buf);

        /* Get the absolute directory path for the optheap working dir*/
        void getWorkDir(char* buf);

        int setBooleanOption(char * value, char * key, int defaultValue);
        
        int setIntOption(char * value, char * key, int defaultValue);
    }; // class COptions

} /* namespace Martini */ } /*namespace JPIAgent*/

#endif // __OPTIONS_H__
