/*******************************************************************************
* Copyright (c) 2005, 2009 IBM Corporation, Intel Corporation.
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
*
* Contributors:
*    IBM Corporation - Initial API and implementation
*    Viacheslav Rybalov, Intel - Initial API and implementation
*
* $Id: Performance.cpp,v 1.8 2009/08/26 15:21:07 jwest Exp $ 
***********************************************************************/

#include "OSA.h" 
#include "log.h" 

#ifdef MVS
	#include <time.h>
#endif

#ifdef _WIN32
  #include <sys/timeb.h>
#else
  #include <sys/time.h>
  #include <sys/timeb.h>
  #include <unistd.h>
#endif

#include <stdio.h>
#include "Performance.h"

#ifdef _WIN32                                    /* _WIN32 */
  #define TIMEB  struct _timeb
  #define FTIME(param)  _ftime(param)
#else                                            /* else */
  #define TIMEB    struct timeb
  #define FTIME(param)  ftime(param)
#endif                                           /* endif */

double       _startTimeAsDouble;
U64          _startTimeNanosec;

/** COLLECT_START_TIME_INFORMATION  ****************************************
* Initialization routine for future use.
* This routine records the time at which this function was called and uses
* this time to determine what the current time is.
*/
void jvmtiAgent_collectStartTimeInformation() 
{
    InitProfilerTimer();
#ifdef _WIN32
    TIMEB time;
    FTIME(&time);
    _startTimeAsDouble = time.time + time.millitm/1000.;
    _startTimeNanosec = (U64)time.time * 1000000000 + (U64)time.millitm * 1000000;
#else
    struct timeval tv;
    gettimeofday(&tv, NULL);
    _startTimeAsDouble = tv.tv_sec + tv.tv_usec/1000000.;
    _startTimeNanosec = (U64)tv.tv_sec * 1000000000 + (U64)tv.tv_usec * 1000; 
#endif
}

/** GET_PROCESS_START_TIME  ***************************************************
*  Returns the start time of the current process in nanoseconds.
*/
U64 jvmtiAgent_getProcessStartTime() 
{
#ifdef _WIN32

    FILETIME  start, end, kernal, user;
    U64 startTime, deltaTime;
    HANDLE handle;

    handle = OpenProcess(PROCESS_QUERY_INFORMATION,
        FALSE,
        /*ra_getProcessId()*/GetCurrentProcessId());  //TODO Check and fix it
    GetProcessTimes(handle,
        &start,    /* start time */
        &end,      /* dont want exit time */
        &kernal,   /* dont want kernal mode time */
        &user);    /* dont want user mode time */

    /* The filetime structure starts as of Janauary 1, 1601.  Subtract the passed time
    between this and January 1, 1970
    */
    startTime = ((Uint64)(start.dwHighDateTime)) << 32;
    startTime |= start.dwLowDateTime;

    deltaTime = ((Uint64)(0x19db1de)) << 32;
    deltaTime |= 0xd53e8000;

    startTime -= deltaTime;

    return (startTime * 100);
#else
   /* On Unix we don't know when the process was started.  We
    * will specify the time the trace was started.
    */
    return _startTimeNanosec;
#endif
}

/** GET_TIMEZONE  *************************************************************
* Returns the current timezone as an offset, in minutes, working westward from
* GMT.  ie.  GMT+5  would be 300 as there are 300 minutes in 5 hours.
*/
unsigned long jvmtiAgent_getTimezone()
{
    /* The first version here may work on all platforms. */
    TIMEB currenttime;
    FTIME(&currenttime);
    return currenttime.timezone;
}

/** InitTimer  **************************************************************/
/** GetTime    **************************************************************/

#ifdef _WIN32

double   _ticksPerNanoSecond;
__int64  startTimeAsTicks;
BOOL     _highResPerfAvailable = FALSE; /* high-resolution performance counter if facility available*/

int InitProfilerTimer()
{
    LARGE_INTEGER liHighResTicks; /* actual ticks per sec if high resolution performance counter available */
    if (QueryPerformanceFrequency(&liHighResTicks)) { 
        _highResPerfAvailable = TRUE;
        _ticksPerNanoSecond = (double)liHighResTicks.QuadPart/1000000000.;
		if (QueryPerformanceCounter(&liHighResTicks)) {
			startTimeAsTicks = liHighResTicks.QuadPart;
		} else {
            return -1;
        }		
    } else {

        HKEY handle;
        DWORD error;
        DWORD valueType;
        DWORD valueLength = 4;
        char *valueName = "~Mhz";
        char *key = "HARDWARE\\DESCRIPTION\\System\\CentralProcessor\\0";
        unsigned long cpuMhz;

        error = RegOpenKeyEx(HKEY_LOCAL_MACHINE,    /* Registery tree */
            key,                                    /* Key to look up */
            0,                                      /* Reserved, must be 0 */
            KEY_READ,                               /* READ mode */
            &handle);                               /* Handle to the key */
        if (error != ERROR_SUCCESS) {
            // Error opening CPU speed key 0x%x\n", error
            return -1;
        }

        error = RegQueryValueEx(handle,               /* Handle to the key */
            valueName,                              /* Value to query */
            NULL,                                   /* Reserved, must be NULL */
            &valueType,                             /* Type of value */
            (LPBYTE)&cpuMhz,                        /* The result of the query */
            &valueLength);                          /* Length of the data */


        if(error != ERROR_SUCCESS) {
            // Error getting CPU speed 0x%x\n", error
            return -1;
        }
        RegCloseKey(handle);
        _ticksPerNanoSecond = (double)cpuMhz/1000000000.;
        startTimeAsTicks = Martini::OSA::GetTimeStamp();
    }
    return MRTE_RESULT_OK;
}
    
U64 GetProfilerTime()
{
    __int64 timestamp;
	if (_highResPerfAvailable == TRUE) {
		LARGE_INTEGER liHighResPerfCount;		
		if (QueryPerformanceCounter(&liHighResPerfCount)) {
			timestamp = liHighResPerfCount.QuadPart;
		} else {
            return 0;
		}		
    } else {
        timestamp = Martini::OSA::GetTimeStamp();
    }
    return (U64)((timestamp - startTimeAsTicks)/_ticksPerNanoSecond); 
}

#else //_WIN32

U64 _startTime;  //start time in nanoseconds

int InitProfilerTimer()
{
	struct timeval tv;
	#ifndef MVS
		struct timezone tz;
		gettimeofday(&tv, &tz);
	#else
		gettimeofday(&tv, 0);
	#endif
	_startTime = (U64)tv.tv_sec * 1000000000;
	_startTime += (U64)tv.tv_usec * 1000;
    return MRTE_RESULT_OK;
}
    
U64 GetProfilerTime()
{
	struct timeval tv;
	#ifndef MVS
		struct timezone tz;
		gettimeofday(&tv, &tz);
	#else
		gettimeofday(&tv, 0);
	#endif
    U64 time;
	time = (U64)tv.tv_sec * 1000000000;
	time += (U64)tv.tv_usec * 1000;
    return time - _startTime; 
}

#endif
