/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation, Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    IBM Corporation - Initial API and implementation
 *    Viacheslav Rybalov, Intel - Initial API and implementation
 *
 * $Id: Print.cpp,v 1.29 2010/12/02 15:32:55 mreid Exp $ 
 ***********************************************************************/

#include "Print.h"
#include "Filters.h"
#include "Options.h"
#include "strings.h"
#include "log.h"
#include <string.h>
#include <errno.h>

#ifdef MVS
	#include <fcntl.h>
	#include <sys/time.h>
	#include <strings.h>
#endif

#ifdef __linux__
	#include <sys/time.h>
#endif

#include <sys/types.h>
#include <iostream>
#include <typeinfo>


using namespace Martini::MPI;
using namespace Martini::JPIAgent;

/************ CElementBuffer **************************************************/

CElementBuffer::CElementBuffer(bool is_binary): binary(is_binary)
{
    buffer = (char *)jvmtiAgent_Calloc(ORIGINAL_MESSAGE_BUFFER_SIZE);
    m_REALLOCATED_BUFFER_SIZE = ORIGINAL_MESSAGE_BUFFER_SIZE;
    offset = 0;
}

CElementBuffer::~CElementBuffer() 
{
    tptp_free(buffer);
}

void 
CElementBuffer::AddString(const char *src) 
{
    CheckMemorySize(strlen(src));
    strcpy(buffer + offset, src);
    offset += strlen(src);
}

void 
CElementBuffer::AddChar(const char chr) 
{
    CheckMemorySize(1);
    buffer[offset++] = chr;
}

void 
CElementBuffer::MemCpy(const void *src, size_t count) 
{
    CheckMemorySize(count);
    memcpy(buffer + offset, src, count);
    offset += count;
};

void *
CElementBuffer::AllocBuffer(size_t size)
{
    CheckMemorySize(size);
	void *result = buffer + offset;
	offset += size;

	return result;
};

/* REALLOCATE memory if the buffer size insufficient*******************************
 * Check buffer size and realloc memory to prevent 
 * buffer overflow.
 *
 */
void
CElementBuffer::CheckMemorySize(size_t in_value)
{    
    if(offset + in_value + 1 > m_REALLOCATED_BUFFER_SIZE){
        char *p = (char *)jvmtiAgent_Calloc((offset + in_value + ORIGINAL_MESSAGE_BUFFER_SIZE) * sizeof(char));
        if(p != 0 && buffer != 0){            
            memcpy(p, buffer, offset);
            p[offset] = '\0';
            free((void*)buffer);
            buffer = (char *)p;
            m_REALLOCATED_BUFFER_SIZE = offset + in_value + ORIGINAL_MESSAGE_BUFFER_SIZE;
        }
    }
}

/************ CPrint **************************************************/

CPrint::CPrint(COptions* pCOptions, CFilters* pCFilters)
{
    m_pOptions = pCOptions;
    m_pCFilters = pCFilters;
    m_pfSendData = 0;
    m_jvmtiAgent_outFile = RA_HANDLE_0;            /* The output file */
    m_jvmtiAgent_suspendIO = true;				   // suspended
    printingStarted = false;
	m_dumpMode = false;
//#ifndef _WIN32
    m_lockObject = Martini::OSA::CreateThreadSync();
//#endif
}

CPrint::CPrint():m_pOptions(NULL), 
				 m_pCFilters(NULL),
				 m_pfSendData(0),
				 m_jvmtiAgent_outFile(RA_HANDLE_0),
				 printingStarted(false),
				 m_dumpMode(false)
{
}

CPrint::~CPrint()
{
}


void
CPrint::setDumpMode(bool dumpMode)
{
	m_dumpMode = dumpMode;
}

void
CPrint::setSendDataFunc(Martini::ACCollector::ACCollector_SendData_t pfSendData)
{
	m_pfSendData = pfSendData;
}

Martini::ACCollector::ACCollector_SendData_t
CPrint::getSendDataFunc()
{
	return m_pfSendData;
}

void 
CPrint::InitializeStandaloneIO(const char *fileName) {
#ifdef _WIN32
    m_jvmtiAgent_outFile = CreateFile(fileName,
        GENERIC_WRITE,
        FILE_SHARE_READ,
        0,
        CREATE_ALWAYS,
        FILE_ATTRIBUTE_NORMAL
#ifdef _JVMPI__ASYNCRONOUS_IO
        | FILE_FLAG_OVERLAPPED
#endif
        ,
        0);

#else
    /* 199373 - added permissions */
    m_jvmtiAgent_outFile = open(fileName,
        O_WRONLY|O_CREAT|O_TRUNC,
        S_IRUSR | S_IWUSR | S_IRGRP );
#endif
    /* Check the file descriptor to ensure it is valid */

    if(m_jvmtiAgent_outFile
#ifdef _WIN32
        == INVALID_HANDLE_VALUE)
#else
        < 0)
#endif
    {
        LOG_DIE("*** Could not open profile output file " << fileName <<". Exiting JVM. ***");
    }

#ifdef _JVMPI__ASYNCRONOUS_IO
    int i;
    for(i=0; i < SEGMENT_COUNT; i++) {
        _ioBuffers[i].offset=0;
#ifdef _WIN32
        _ioBuffers[i].overlap.hEvent=CreateEvent(0, FALSE, FALSE, 0);
        _ioBuffers[i].overlap.Offset=0;
        _ioBuffers[i].overlap.OffsetHigh=0;
#else
        _ioBuffers[i].aiocb.aio_offset=0;                     /* file offset */
        _ioBuffers[i].aiocb.aio_buf=_ioBuffers[i].buffer;     /* i/o buffer */
        _ioBuffers[i].aiocb.aio_reqprio=0;                    /* request priority offset */
        /*_ioBuffers[i].aiocb.aio_sigevent= */                /* signal number and value */
        /*_ioBuffers[i].aiocb.aio_lio_opcode=LIO_WRITE; */    /* operation to be preformed */
#endif
        _ioBuffers[i].flushPending=FALSE;
    }
#endif /* _JVMPI__ASYNCRONOUS_IO */
}

void 
CPrint::CleanupStandaloneIO()
{
#ifdef _JVMPI__ASYNCRONOUS_IO
    int segment=getSegment();
    _ioBuffers[segment].flushPending=TRUE;
    releaseSegment(segment, _ioBuffers[segment].offset);
#endif /* _JVMPI__ASYNCRONOUS_IO */
    CLOSE_RA_HANDLE(m_jvmtiAgent_outFile);
}

/** Whether or not an I/O error has occured in jvmtiAgent_print(...). */
static bool ioErrorEncountered = false;

/** PRINT  **********************************************************************
* Transmit/print the raw data to the median.
* @param  buffer - the buffer to write to the median.
* @param  length - number of bytes to write
*/
void 
CPrint::jvmtiAgent_print(CElementBuffer *pElB) {
    char *s = pElB->buffer; 
    size_t length = pElB->offset;
#ifndef _WIN32
    ssize_t ioResult;
#endif

    if (!m_jvmtiAgent_suspendIO) {
        if (m_dumpMode ? false : !m_pOptions->isStandAlone()) {
            s[length] = 0; //TODO ????
			LOG_ASSERT(m_pfSendData != 0);
//            LOG_MESSAGE(">>>SendData: " << s);
			(m_pfSendData)(s, length);
#if defined(DEBUG) || defined(_DEBUG)
			if (!pElB->IsBinary()) {
				s[length++]='\n';
			}
			// Bug 328637 - Don't write to unopened file
			if( RA_HANDLE_0 != m_jvmtiAgent_outFile ) {
			
			#ifdef _WIN32
				DWORD currentBufferOffset;
				m_lockObject->Enter();
				WriteFile(m_jvmtiAgent_outFile, s, length, &currentBufferOffset, 0);
				m_lockObject->Leave();
			#else
				int currentBufferOffset;
				m_lockObject->Enter();
				#ifdef MVS
					char* nativeBuffer = 0;
					Martini::OSA::unicode2native(&nativeBuffer, s, strlen(s));
					ioResult = write(m_jvmtiAgent_outFile, nativeBuffer, strlen(nativeBuffer));
					if(nativeBuffer != 0)
						free(nativeBuffer);
				#else
					ioResult = write(m_jvmtiAgent_outFile, s, length);
				#endif
				m_lockObject->Leave();
			#endif
			
			}
#endif
        } else {
//            bool success = false;
			if (!pElB->IsBinary()) {
				s[length++]='\n';
			}
 
#ifndef _JVMPI__ASYNCRONOUS_IO

			// Bug 328637 - Don't write to unopened file
			if( RA_HANDLE_0 != m_jvmtiAgent_outFile ) {

			#ifdef _WIN32
				DWORD currentBufferOffset;
				m_lockObject->Enter();
				WriteFile(m_jvmtiAgent_outFile, s, length, &currentBufferOffset, 0);
				m_lockObject->Leave();
			#else
				int currentBufferOffset;
				m_lockObject->Enter();
				#ifdef MVS
					char* nativeBuffer = 0;
					Martini::OSA::unicode2native(&nativeBuffer, s, strlen(s));
					ioResult = write(m_jvmtiAgent_outFile, nativeBuffer, strlen(nativeBuffer));

					if(nativeBuffer != 0)
						free(nativeBuffer);
				#else
					ioResult = write(m_jvmtiAgent_outFile, s, length);
				#endif
				m_lockObject->Leave();
			#endif

			}

#else /*  _JVMPI__ASYNCRONOUS_IO  */

            ///* Always work with the same segment */
            //currentSegment=getSegment();

            ///* Grab some space within this segment */
            //currentBufferOffset=_ioBuffers[currentSegment].offset;
            //while(!ra_atomicCAS((int*)&_ioBuffers[currentSegment].offset, (int*)&currentBufferOffset, (int)(currentBufferOffset+length)));

            ///* If there is room in this segment copy our data in.  Otherwise mark
            //this buffer to be flushed and increment the segment counter */
            //if(currentBufferOffset+length<SEGMENT_SIZE)
            //{
            //    memcpy(&_ioBuffers[currentSegment].buffer[currentBufferOffset], s, length);
            //    success=TRUE;
            //}
            //else {
            //    /* printf("Flush pending\n"); */
            //    _ioBuffers[currentSegment].flushPending=TRUE;

            //    /* Increment the segment pointer */
            //    while(!ra_atomicCAS((int*)&_ioSegment, (int*)&currentSegment, (int)((currentSegment+1)%16)));
            //}

            //releaseSegment(currentSegment, currentBufferOffset);

            ///* If we failed to transfer our data we need to try again */
            //if(!success)
            //{
            //    jvmtiAgent_print(pElB, s, length);
            //}
#endif /* _JVMPI__ASYNCRONOUS_IO */

        }
        
#ifndef _WIN32
		// 	Catch errors in I/O in the Unixes
        if(ioResult == -1 && !ioErrorEncountered) {

			if(errno == EFBIG) {
				printf("ERROR: Unable to write additional data to the filesystem. The file has likely exceeded the system maximum.\n");
				printf("\n");
				#ifdef _AIX
				printf("\n");
				printf("On AIX, in order to write to files >2GB in size, the file system must support large files, ulimit must\n");
				printf("not restrict the file size, and the process must be a 64-bit process (which requires a 64-bit JVM).\n");
				printf("\n");
				#endif

			} else {
				printf("ERROR: Unable to write to output file.\n\n");
			}
			ioErrorEncountered = true;
        }
#endif

    }
}

/*
 This function ensures that the string name can be legally output as XML by translating any
 '<' or '>" characters to '-'.
*/
char* 
CPrint::FormatName(const char *name)
{
    if (name == 0) {
        return 0;
    }
    char* buf = reinterpret_cast<char*>(jvmtiAgent_Calloc(1 + strlen(name)));
    strcpy(buf, name);

    int i;
    if (buf) {
        for (i = strlen(buf) - 1; i >= 0; i--) {
            if (buf[i] == '<' || buf[i] == '>') {
                buf[i] = '-';
            }
        }
    }
    return buf;
}

#ifdef _WIN32

	#include <rpc.h>
	char* CPrint::GenerateUUID()
	{
	    UUID uuid;
	    unsigned char* stringUuid;
	    UuidCreate(&uuid);
	    UuidToString(&uuid, &stringUuid);
	    return reinterpret_cast<char*>(stringUuid); 
	    // TODO How about deallocation?
	}

#elif defined(_SOLARIS)
	#include <uuid/uuid.h>
	char* CPrint::GenerateUUID()
	{
	    uuid_t uuid;
	    char* stringUuid = new char[37]; //TOOD Check !!!
	    uuid_generate(uuid);
	    uuid_unparse(uuid, stringUuid);
	    return reinterpret_cast<char*>(stringUuid); 
	    // TODO How about deallocation?
	}

#else

// Integer -> ASCII Hex lookup table
	const char INTEGER_TO_ASCIIHEX[16] = { 48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 65, 66, 67, 68, 69, 70 };

	void formatUUIDString(char *inHexVals, char *outResultUUID) {
		char first[8+1];
		char second[4+1];
		char third[4+1];
		char fourth[4+1];
		char fifth[12+1];

		char * buffer = inHexVals;

		first[0] = 0; second[0] = 0; third[0] = 0; fourth[0] = 0; fifth[0] = 0;

		strncat(first, buffer, 8); buffer += 8;
		strncat(second, buffer, 4); buffer += 4;
		strncat(third, buffer, 4); buffer += 4;
		strncat(fourth, buffer, 4); buffer += 4;
		strncat(fifth, buffer, 12); buffer += 12;

		sprintf(outResultUUID, "%s-%s-%s-%s-%s", first, second, third, fourth, fifth);
	}

	char* CPrint::GenerateUUID()
	{
		static int firstTime = 1;
		static unsigned short seed[3];
		struct timeval tv;
		unsigned long seconds, microseconds;
		char buffer[128];
		char *result;
		
		result = (char *)malloc(36+8);
		memset(result, 0, 36+8);
		
		if(buffer) {
			/* Get a timestamp */
			gettimeofday(&tv, 0);
			seconds=tv.tv_sec;
			microseconds=tv.tv_usec;

			/* Seed our generator on the first pass */
			if(firstTime) {
				seed[0]=seconds;
				seed[1]=microseconds;
				seed[2]=0;
				seed48(seed);
				firstTime=0;
			}

			for(int x = 0; x < 32; x++) {
				buffer[x] = INTEGER_TO_ASCIIHEX[(int)(lrand48() % 16)];
			}
			
			// 4
			buffer[12] = '4';
			
			// 8, 9, A or B
			buffer[16] = INTEGER_TO_ASCIIHEX[(int)(lrand48() % 4)+8];
			
			buffer[32] = 0;

			formatUUIDString(buffer, result);		

			return result;

		}
		return NULL;
	}

#endif

char* 
CPrint::GetRATraceUUID()
{
    static char* UUID = 0;
  //  if (m_pOptions->isStandAlone()) {  // Commented out for workaronf for 139984 
        if (UUID == 0) {
            UUID = GenerateUUID();
        }
        return UUID;
//    } else {  // Commented out for workaronf for 139984 
        //TODO Fix it, get UUID from AC
//        return "";  // Commented out for workaronf for 139984 
//    }  // Commented out for workaronf for 139984 
}

char* 
CPrint::GetRAProcessUUID()
{
    static char* UUID = 0;
//    if (m_pOptions->isStandAlone()) {  // Commented out for workaronf for 139984 
        if (UUID == 0) {
            UUID = GenerateUUID();
        }
        return UUID;
//    } else {  // Commented out for workaronf for 139984 
        //TODO Fix it, get UUID from AC  // Workaronf for 139984 
//        return "";  // Commented out for workaronf for 139984 
//    }  // Commented out for workaronf for 139984 
}

char* 
CPrint::GetRANodeUUID()
{
    static char* UUID = 0;
    if (m_pOptions->isStandAlone()) {
        if (UUID == 0) {
//            UUID = GenerateUUID();
            UUID = "";
        }
        return UUID;
    } else {
        //TODO Fix it, get UUID from AC
        return "";
    }
}

char* 
CPrint::GetRAAgentUUID()
{
    static char* UUID = 0;
//    if (m_pOptions->isStandAlone()) {  // Commented out for workaronf for 139984 
        if (UUID == 0) {
            UUID = GenerateUUID();
        }
        return UUID;
//    } else {  // Commented out for workaronf for 139984 
        //TODO Fix it, get UUID from AC
//        return "";  // Commented out for workaronf for 139984 
//    }  // Commented out for workaronf for 139984 
}

PID 
CPrint::GetRAProcessId() {
#ifdef _WIN32
    return GetCurrentProcessId();
//#elif defined __linux__
//    return _pid;
#else
    return getpid();
#endif
    return 0;
}

char* 
CPrint::ConvertClassName(const char *inbuf)
{
    if ((inbuf[strlen(inbuf) - 1] != ';') || (inbuf[0] == '[')) {
        char* str = (char*)malloc(strlen(inbuf) + 1);
        return strcpy(str, inbuf);
    } else {
        char* str = (char*)malloc(strlen(inbuf) -1);
        strncpy(str, inbuf + 1, strlen(inbuf) - 2);
        str[strlen(inbuf) - 2] = 0;
        return str;
    }
}


char *
CPrint::GetHostname() 
{
	return "localhost";
}

char *
CPrint::GetIPAddress()
{
	return "127.0.0.1";
}

void CPrint::suspend() {
	m_jvmtiAgent_suspendIO = true;
}

void CPrint::resume() {
	m_jvmtiAgent_suspendIO = false;
}

void CPrint::start() {
	m_jvmtiAgent_suspendIO = false;
	
	if (!printingStarted) {
		printingStarted = true;
		PrintStartingFragments();
	}
}

void CPrint::stop() {
	if (printingStarted) {
		printTraceEndElement();
		printingStarted = false;
	}
		
	m_jvmtiAgent_suspendIO = true;
}

bool CPrint::isSuspended() {
	return m_jvmtiAgent_suspendIO;
}

bool CPrint::isStarted() {
	return printingStarted;
}
