/*******************************************************************************
 * Copyright (c) 2008, 2010 Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * Stanislav Polevic, Intel - Initial API and implementation
 *
 * $Id: PrintBinary.cpp,v 1.7 2010/05/11 16:32:21 jwest Exp $ 
 ***********************************************************************/

// PrintBinary.cpp: implementation of the CPrintBinary class.
//
//////////////////////////////////////////////////////////////////////

#include "Filters.h"
#include "Options.h"
#include "strings.h"
#include "log.h"
#include "PrintBinary.h"


#include <stdio.h>

using namespace Martini::MPI;
using namespace Martini::JPIAgent;

enum ThreadInteractionType { 
	EmptyThreadInteraction = 0, 
	CallIdNotifyAll = 1, 
	CallIdNotify = 2, 
	CallIdInterrupt = 3, 
	CallIdStart = 4,
	CallIdSleepStart = 5,
	CallIdSleepEnd = 6
};

static bool header_printed = false;

CPrintBinary::CPrintBinary(COptions* pCOptions, CFilters* pCFilters):CPrint(pCOptions, pCFilters)
{
}

CPrintBinary::~CPrintBinary()
{
}

void 
CPrintBinary::printFormatHeader(bool force_print = false)
{
	if (force_print || !header_printed) {
		CElementBuffer elb(true);
		printMessage(&format_header, &elb);
		header_printed = true;
	}
}

void 
CPrintBinary::PrintStartingFragments()
{
	printFormatHeader(true);

	PrintEncodingMessage();
	PrintFrequencyMessage();


    printNodeElement();
    printProcessCreateElement();
    printAgentCreateElement(m_pOptions->m_jvmtiAgent_Options.invocationOptions);
    printTraceStartElement(m_pOptions->m_jvmtiAgent_Options.invocationOptions);

    /* Print the filters and options only if the proper option is set */
    if (m_pOptions->m_jvmtiAgent_Options.filters) {
        m_pCFilters->PrintFilters(this);
    }
    if (m_pOptions->m_jvmtiAgent_Options.options) {
        m_pOptions->printOptions(this);
    }
}

void 
CPrintBinary::PrintEncodingMessage()
{
    CElementBuffer elb(true);

	CEncodingMessage encMsg;
	printMessage(&encMsg, &elb);
}

void 
CPrintBinary::PrintFrequencyMessage()
{
    CElementBuffer elb(true);

	CFreqMessage freqMsg;
	freqMsg.frequency = 0;
	printMessage(&freqMsg, &elb);
}

void 
CPrintBinary::printNodeElement()
{
    CElementBuffer elb(true);
	CNodeMessage msg;
    U64 time = jvmtiAgent_getProcessStartTime();

    msg.node_id = GetRANodeUUID();
    msg.hostname = GetHostname();
    msg.ip = GetIPAddress();
    msg.timezone = jvmtiAgent_getTimezone();
    if(m_pOptions->m_jvmtiAgent_Options.timestamp) {
        msg.timestamp = time;
    }

	printMessage(&msg, &elb);
}

void 
CPrintBinary::printProcessCreateElement()
{
    CElementBuffer elb(true);
	CProcessCreateMessage msg;
    U64 time = jvmtiAgent_getProcessStartTime();

    msg.process_id = GetRAProcessUUID();
	msg.pid = GetRAProcessId();
	msg.node_id_ref = GetRANodeUUID();
    if(m_pOptions->m_jvmtiAgent_Options.timestamp) {
        msg.timestamp = time;
    }

	printMessage(&msg, &elb);
}

void 
CPrintBinary::printAgentCreateElement(char *options)
{
    CElementBuffer elb(true);
	CAgentCreateMessage msg;

    msg.agent_id = GetRAAgentUUID();
    msg.version = AGENT_VERSION;
    msg.process_id_ref = GetRAProcessUUID();
    msg.name = m_pOptions->m_jvmtiAgent_Options.processName;
    msg.type = m_pOptions->m_jvmtiAgent_Options.processType;
    msg.params = options;
    if(m_pOptions->m_jvmtiAgent_Options.timestamp) {
        msg.timestamp = getTimestamp(0);
    }

	printMessage(&msg, &elb);
}

void 
CPrintBinary::printTraceStartElement(char *options)
{
    CElementBuffer elb(true);
	CTraceStartMessage msg;

    msg.trace_id = GetRATraceUUID();
    msg.agent_id_ref = GetRAAgentUUID();
    if(m_pOptions->m_jvmtiAgent_Options.timestamp) {
        msg.timestamp = getTimestamp();
    }

	printMessage(&msg, &elb);
}

void 
CPrintBinary::printTraceEndElement()
{
    CElementBuffer elb(true);
	CTraceEndMessage msg;

    /* Get the timestamp information if necessary */
    if(m_pOptions->m_jvmtiAgent_Options.timestamp) {
        msg.timestamp = getTimestamp();
    }


	printMessage(&msg, &elb);
}

void 
CPrintBinary::printJvmShutdownElement() {
    CElementBuffer elb(true);
	CRuntimeShutdownMessage msg;

    /* Get the timestamp information if necessary */
    if(m_pOptions->m_jvmtiAgent_Options.timestamp) {
        msg.timestamp = getTimestamp();
    }

	printMessage(&msg, &elb);

    /* Print the closing tag */
    printTraceEndElement();
    printAgentDestroyElement();
}

void 
CPrintBinary::printAgentDestroyElement()
{
    CElementBuffer elb(true);
	CAgentDestroyMessage msg;

    msg.agent_id_ref = GetRAAgentUUID();

    /* Get the timestamp information if necessary */
    if(m_pOptions->m_jvmtiAgent_Options.timestamp) {
        msg.timestamp = getTimestamp();
    }

	printMessage(&msg, &elb);
}

/** PRINT_METHOD  **********************************************************
* Handles the output of a METHOD element and attributes
*/
void 
CPrintBinary::printNewMethodElement(TId methodId, SMethodInfo* methodInfo)
{
	CElementBuffer elb(true);
	CMethodDefMessage msg;

    char* formatedName = FormatName(methodInfo->szJavaMethodName);
	msg.name = formatedName;
	msg.sig = const_cast<char*>(methodInfo->szJavaMethodSig);
    if ((methodInfo->validData & DR_METHOD_LINE_NUMBERS) != 0) {
        msg.start_line = methodInfo->methodLineNumbers.uiStart;
        msg.end_line = methodInfo->methodLineNumbers.uiEnd;
    }
    msg.method_id = methodId;
    msg.class_id_ref = methodInfo->classId;

	printMessage(&msg, &elb);

    free(formatedName);
}

/** PRINT_METHOD_ENTRY_EVENT  **********************************************
* Handles the output of a METHOD_ENTRY, METHOD_CALL element and attributes
*/
void 
CPrintBinary::printMethodEntryElement(TId threadId, TId methodId, TId classId, unsigned long ticket, unsigned long stackDepth) 
{
    CElementBuffer elb(true);
	CMethodEntryMessage msg;

	msg.thread_id_ref = threadId;

	msg.thread_id_ref = threadId;
    if(m_pOptions->m_jvmtiAgent_Options.timestamp) {
        msg.timestamp = getTimestamp();
    }
    
	msg.method_id_ref = methodId;
    msg.class_id_ref = classId;

    if(m_pOptions->m_jvmtiAgent_Options.ticket) {
        msg.ticket = ticket;
    }
    msg.stack_depth = stackDepth;

	printMessage(&msg, &elb);
}

/** PRINT_AG_METHOD_ENTRY_EVENT  **************************preagg/135437*******
  *
  * Handles the output of a AG_METHOD_ENTRY element and attributes
  */
void 
CPrintBinary::printAgMethodEntryElement(TId threadId, TId methodId, U64 baseTime, U64 minTime, U64 maxTime, U64 baseCPUTime, U64 numCalls) {
    CElementBuffer elb(true);
	CAGMethodEntryMessage msg;

    msg.thread_id_ref = threadId;
    msg.method_id_ref = methodId;
    if (numCalls > 0) {
        msg.base_time = baseTime;
        msg.min_time = minTime;
        msg.max_time = maxTime;   
        if(m_pOptions->m_jvmtiAgent_Options.cpuTime) {
            msg.base_cpu_time = getInterval(baseCPUTime); 
        }
    }
    msg.num_calls = numCalls;

	printMessage(&msg, &elb);
}

/** PRINT_METHOD_EXIT_EVENT  ***********************************************
* Prints the METHOD_EXIT, METHOD_RETURN element
*/
void
CPrintBinary::printMethodExitElement(TId threadId, TId methodId, TId classId, unsigned long ticket, U64 cpuTime)
{
    CElementBuffer elb(true);
	CMethodExitMessage msg;

    msg.thread_id_ref = threadId;
    msg.method_id_ref = methodId;
    msg.class_id_ref = classId;

    if(m_pOptions->m_jvmtiAgent_Options.cpuTime) {
        msg.thread_cpu_time = getInterval(cpuTime); 
    }
    if(m_pOptions->m_jvmtiAgent_Options.ticket) {
        msg.ticket = ticket;
    }

    /* Print the timestamp (and overhead) information if necessary */
    if(m_pOptions->m_jvmtiAgent_Options.timestamp) {
        msg.timestamp = getTimestamp();
    }

	printMessage(&msg, &elb);
}

/** PRINT_AG_METHOD_EXIT_ELEMENT **************************preAgg/135437*******
  * 
  * Prints the METHOD_EXIT element
  */
void 
CPrintBinary::printAgMethodExitElement(TId threadId, TId methodId) {
    CElementBuffer elb(true);
	CAGMethodExitMessage msg;

    msg.thread_id_ref = threadId;
    msg.method_id_ref = methodId;

	printMessage(&msg, &elb);
}

/** PRINT_THREAD_START_ELEMENT  ********************************************
*
*/
void
CPrintBinary::printThreadStartElement(TId threadId, TId objectId, SThreadInfo* threadInfo)
{
    CElementBuffer elb(true);
	CThreadStartMessage msg;
    char* threadName = 0;
    char* groupName = 0;
    char* parentGroupName = 0;

    msg.thread_id = threadId;
    /* Get the timestamp information if necessary */
    if(m_pOptions->m_jvmtiAgent_Options.timestamp) {
        msg.timestamp = getTimestamp();
    }
    if (objectId != 0) {
        msg.obj_id_ref = objectId;
    }
    if (threadInfo->validData && DR_THREAD_INFO) {
        threadName = FormatName(threadInfo->szName);
        msg.thread_name = threadName;
        groupName = FormatName(threadInfo->szGroupName);
        msg.group_name = groupName;
        parentGroupName = FormatName(threadInfo->szParentGroupName);
        msg.parent_group_name = parentGroupName;
    }

	printMessage(&msg, &elb);

    if (threadName != NULL) free(threadName);
    if (groupName != NULL) free(groupName);
    if (parentGroupName != NULL) free(parentGroupName);
}

/** PRINT_THREAD_END_ELEMENT  **********************************************
*/
void 
CPrintBinary::printThreadEndElement(TId threadId)
{
    CElementBuffer elb(true);
	CThreadEndMessage msg;

    msg.thread_id_ref = threadId;
    /* Get the timestamp information if necessary */
    if(m_pOptions->m_jvmtiAgent_Options.timestamp) {
        msg.timestamp = getTimestamp();
    }

	printMessage(&msg, &elb);
}

void 
CPrintBinary::printJvmInitDoneElement(TId threadId)
{
    CElementBuffer elb(true);
	CRuntimeInitDoneMessage msg;

    msg.thread_id_ref = threadId;
    /* Get the timestamp information if necessary */
    if(m_pOptions->m_jvmtiAgent_Options.timestamp) {
        msg.timestamp = getTimestamp();
    }

	printMessage(&msg, &elb);
}

/** PRINT_CLASS  ****************************************************************
  * Handles the output of the CLASS_LOAD element and attributes
  */
void
CPrintBinary::printNewClassElement(TId classId, SClassInfo* classInfo)
{
    CElementBuffer elb(true);
	CClassDefMessage msg;

    char* convertedName = ConvertClassName(classInfo->szJavaClassName);
    msg.name = convertedName;
    msg.source_name = const_cast<char*>(classInfo->szSourceFileName);
    msg.class_id = classId;
    if(m_pOptions->m_jvmtiAgent_Options.timestamp) {
        msg.timestamp = getTimestamp();
    }
	printMessage(&msg, &elb);

    free(convertedName);
}

/** PRINT_OBJ_ALLOC_ELEMENT  *****************************************************
  *
  */
void
CPrintBinary::printObjAllocElement(SHeapEventData* heapData, U64 lineNumber)
{
    CElementBuffer elb(true);
	CObjAllocMessage msg;

    msg.thread_id_ref = heapData->threadId;
    if(m_pOptions->m_jvmtiAgent_Options.timestamp) {
        msg.timestamp = getTimestamp();
    }
    /* Print the object identifier as well as the class identifier */
    msg.obj_id_ref = heapData->objectId;
    msg.class_id_ref = heapData->pObjectInfo->classId;
    if(m_pOptions->m_jvmtiAgent_Options.objAllocIsArray) {
        msg.is_array = '\0';
    }
    /* Print the size of this object */
    msg.size = heapData->pObjectInfo->uiSize;
    if (m_pOptions->isAllocSitesSupported() && ((heapData->validData & DR_METHOD_ID) != 0)) {
        msg.method_id = heapData->allocMethodId;
		msg.line = lineNumber;
    }

	printMessage(&msg, &elb);
}

/** PRINT_OBJ_FREE_ELEMENT  ************************************************
* Handles the output of OBJ_FREE.
*/
void
CPrintBinary::printObjFreeElement(SHeapEventData* heapData)
{
    CElementBuffer elb(true);
	CObjFreeMessage msg;

    if(m_pOptions->m_jvmtiAgent_Options.timestamp) {
        msg.timestamp = getTimestamp();
    }
    msg.obj_id_ref = heapData->objectId;
    msg.obj_age = heapData->objectAge;

	printMessage(&msg, &elb);
}

void 
CPrintBinary::printGcStartElement()
{
    CElementBuffer elb(true);
	CGCStartMessage msg;

    if(m_pOptions->m_jvmtiAgent_Options.timestamp) {
        msg.timestamp = getTimestamp();
    }

	printMessage(&msg, &elb);
}

void 
CPrintBinary::printGcFinishElement()
{
    CElementBuffer elb(true);
	CGCFinishMessage msg;

    if(m_pOptions->m_jvmtiAgent_Options.timestamp) {
        msg.timestamp = getTimestamp();
    }

	printMessage(&msg, &elb);
}

/* PR Object Ref */
void 
CPrintBinary::printHDStartElement(TId heapDumpId, char* heapDefName, U64 basetime)
{
    CElementBuffer elb(true);
	CHDStartMessage msg;

    msg.heap_dump_id_ref = heapDumpId;
    if(m_pOptions->m_jvmtiAgent_Options.timestamp) {
        msg.timestamp = getTimestamp();
    }
	msg.name = heapDefName;
	msg.base_time = basetime;

	printMessage(&msg, &elb);
}

/** PRINT_GC_ROOT_ELEMEMT  *******************************************************
*
*/
void
CPrintBinary::printGcRootElement(TId heapDumpIdRef, TId objIdRef, EObjectReferenceType type)
{
    CElementBuffer elb(true);
	CGCRootMessage msg;

    char* typeName = "Fix me";
    unsigned short current=0;

    if(type = OR_SYSTEM_CLASS) {
        msg.class_id_ref = objIdRef;
    } else {
        msg.obj_id_ref = objIdRef;
    }
    switch (type) {
        case OR_SYSTEM_CLASS:
            typeName = "SYSTEM_CLASS";  //TODO Check
            break;
        case OR_MONITOR:
            typeName = "MONITOR";  //TODO Check
            break;
        case OR_STACK_LOCAL:
            typeName = "STACK_LOCAL";  //TODO Check
            break;
        case OR_JAVA_JNI_LOCAL:
            typeName = "JNI_LOCAL";
            break;
        case OR_THREAD:
            typeName = "THREAD";  //TODO Check
            break;
        case OR_OTHER:
            typeName = "UNKNOWN";
            break;
    }
    msg.type = typeName;

	printMessage(&msg, &elb);
}

void 
CPrintBinary::printObjectReferenceElement(TId heapDumpIdRef, SObjectReference* objRef)
{
    CElementBuffer elb(true);
	CObjRefMessage msg;

    msg.src_id_ref = objRef->srcObjectId;
    msg.target_id_ref = objRef->targetObjectId;
	msg.heap_dump_id_ref = heapDumpIdRef;

	printMessage(&msg, &elb);
}

void
CPrintBinary::printMonitorWaitElement(SMonitorWaitEventData* monitorWaitData/*int isThreadSleep*/, SStackTrace_* stackTrace)
{
    CElementBuffer elb(true);
	CMonWaitMessage msg;

    msg.thread_id_ref = monitorWaitData->threadId;
    if(m_pOptions->m_jvmtiAgent_Options.timestamp) {
        msg.timestamp = getTimestamp();
    }
    msg.obj_id_ref = monitorWaitData->objectId;
    msg.timeout = monitorWaitData->timeoutMillis;

    if (stackTrace != NULL && m_pOptions->isStackInfoNormal()) {
		msg.stack_methods = static_cast<bf_string_t*>(malloc(sizeof(void *) * stackTrace->uiSize));
		msg.stack_lines = static_cast<bf_long_t*>(malloc(sizeof(bf_long_t) * stackTrace->uiSize));
        
		AppendAnnotations(msg.stack_methods, msg.stack_lines, &msg.stack_depth, stackTrace);
    }

	printMessage(&msg, &elb);

	if (msg.stack_methods != NULL) free(msg.stack_methods);
	if (msg.stack_lines != NULL) free(msg.stack_lines);
}

void
CPrintBinary::printMonitorWaitedElement(SMonitorWaitedEventData* monitorWaitedData, U64 timeout, SStackTrace_* stackTrace)
{
    CElementBuffer elb(true);
	CMonWaitedMessage msg;

    msg.thread_id_ref = monitorWaitedData->threadId;
    if(m_pOptions->m_jvmtiAgent_Options.timestamp) {
        msg.timestamp = getTimestamp();
    }

    msg.obj_id_ref = monitorWaitedData->objectId;
    msg.timeout = timeout/1000000;

    if (stackTrace != NULL && m_pOptions->isStackInfoNormal()) {
		msg.stack_methods = static_cast<bf_string_t*>(malloc(sizeof(void *) * stackTrace->uiSize));
		msg.stack_lines = static_cast<bf_long_t*>(malloc(sizeof(bf_long_t) * stackTrace->uiSize));
        
		AppendAnnotations(msg.stack_methods, msg.stack_lines, &msg.stack_depth, stackTrace);
    }

	printMessage(&msg, &elb);

	if (msg.stack_methods != NULL) free(msg.stack_methods);
	if (msg.stack_lines != NULL) free(msg.stack_lines);
}

/** PRINT_MONITOR_CONTENDED_ENTER_ELEMENT  ************************************************
  *
  * Print the <monContendedEnter> element to the trace indicating
  * JVMPI_EVENT_MONITOR_CONTENDED_ENTER occurred.
  *
  * args -
  *		objectHashEntry - the hash entry corresponding to the monitor object
  *		event - the MONITOR_WAITED event as reported by JVMPI
  *		thread_owner - the current thread that owns the monitor object
  *		timestamp - when the MONITOR_WAITED event occurred
  */
void
CPrintBinary::printMonitorContendedEnterElement(SContendedMonitorEnterEventData* conMonEnterData, SStackTrace_* stackTrace)

{
    CElementBuffer elb(true);
	CMonContendedEnterMessage msg;

    msg.thread_id_ref = conMonEnterData->threadId;
    if(m_pOptions->m_jvmtiAgent_Options.timestamp) {
        msg.timestamp = getTimestamp();
    }
    msg.obj_id_ref = conMonEnterData->objectId;

    /* print the thread owner attribute */
    if ((conMonEnterData->validData & DR_MONITOR_OWNER_THREAD_ID) != 0) {
        msg.thread_owner_id_ref = conMonEnterData->ownerThreadId;
    } else {
        msg.thread_owner_id_ref = -1;
    }

    if (stackTrace != NULL && m_pOptions->isStackInfoNormal()) {
		msg.stack_methods = static_cast<bf_string_t*>(malloc(sizeof(void *) * stackTrace->uiSize));
		msg.stack_lines = static_cast<bf_long_t*>(malloc(sizeof(bf_long_t) * stackTrace->uiSize));
        
		AppendAnnotations(msg.stack_methods, msg.stack_lines, &msg.stack_depth, stackTrace);
    }

	printMessage(&msg, &elb);

	if (msg.stack_methods != NULL) free(msg.stack_methods);
	if (msg.stack_lines != NULL) free(msg.stack_lines);
}

/** PRINT_MONITOR_CONTENDED_ENTERED_ELEMENT  ************************************************
  *
  * Print the <monContendedEntered> element to the trace
  */
void
CPrintBinary::printMonitorContendedEnteredElement(SContendedMonitorEnteredEventData* conMonEnteredData, SStackTrace_* stackTrace)
{
    CElementBuffer elb(true);
	CMonContendedEnteredMessage msg;

    msg.thread_id_ref = conMonEnteredData->threadId;
    if(m_pOptions->m_jvmtiAgent_Options.timestamp) {
        msg.timestamp = getTimestamp();
    }
    msg.obj_id_ref = conMonEnteredData->objectId;

    if (stackTrace != NULL && m_pOptions->isStackInfoNormal()) {
		msg.stack_methods = static_cast<bf_string_t*>(malloc(sizeof(void *) * stackTrace->uiSize));
		msg.stack_lines = static_cast<bf_long_t*>(malloc(sizeof(bf_long_t) * stackTrace->uiSize));
        
		AppendAnnotations(msg.stack_methods, msg.stack_lines, &msg.stack_depth, stackTrace);
    }

	printMessage(&msg, &elb);

	if (msg.stack_methods != NULL) free(msg.stack_methods);
	if (msg.stack_lines != NULL) free(msg.stack_lines);
}

void 
CPrintBinary::printThreadInteractionElements(SThreadInteractionEventData* threadInteractionData, SStackTrace_* stackTrace)
{
    CElementBuffer elb;

	CCallIdNotifyMessage *msg = NULL;
	switch (threadInteractionData->interactionType) {
	case CallIdNotifyAll:
	case CallIdNotify:
		msg = new CCallIdNotifyMessage();
		break;
	case CallIdInterrupt:
		msg = new CCallIdInterruptMessage();
		break;
	case CallIdStart:
		msg = new CCallIdStartMessage();
		break;

	case CallIdSleepStart:
		msg = new CCallIdThreadSleepStartMessage();
		break;

	case CallIdSleepEnd:
		msg = new CCallIdThreadSleepEndMessage();
		break;

	default:
		return;
	};

	msg->thread_id_ref = threadInteractionData->threadId;
    if(m_pOptions->m_jvmtiAgent_Options.timestamp) {
		msg->timestamp = getTimestamp();
    }

	msg->static_obj_id_ref = threadInteractionData->objectId;

    if (threadInteractionData->interactionType == IT_NOTIFY_ALL) {
		msg->notify_all = 1;
    }

	// Sleep Start and Sleep End do not use the stacktrace
    if (stackTrace != NULL && m_pOptions->isStackInfoNormal()
    		&& threadInteractionData->interactionType != CallIdSleepStart
    		&& threadInteractionData->interactionType != CallIdSleepEnd) {
		msg->stack_methods = static_cast<bf_string_t*>(malloc(sizeof(void *) * stackTrace->uiSize));
		msg->stack_lines = static_cast<bf_long_t*>(malloc(sizeof(bf_long_t) * stackTrace->uiSize));
		AppendAnnotations(msg->stack_methods, msg->stack_lines, &msg->stack_depth, stackTrace);
    }

	printMessage(msg, &elb);

	if (msg->stack_methods != NULL) free(msg->stack_methods);
	if (msg->stack_lines != NULL) free(msg->stack_lines);

	delete msg;
}

/** PRINT_OPTION  **********************************************************
* Outputs an OPTION element
*/
void
CPrintBinary::PrintOption(const char *key, const char *value)
{
    CElementBuffer elb(true);
	COptionMessage msg;

    msg.key = const_cast<char*>(key);
    msg.value = const_cast<char*>(value);

	printMessage(&msg, &elb);
}

/** PRINT_FILTER  **********************************************************
  * Outputs a FILTER element
  */
void
CPrintBinary::printFilter(Filter *filter) {
    CElementBuffer elb(true); 
    char *startPattern;
	CFilterMessage msg;
    
    startPattern = elb.buffer + elb.offset;
    msg.pattern = filter->pattern;

    elb.buffer[elb.offset] = '\0';

    /* RKD:  If the pattern contains any '/' replace these with '.' */
    while((startPattern = strchr(startPattern, '/'))) {
        *startPattern = (char)'.';
    }

    if (filter->GetClassFilterMode(filter) == INCLUDE) {
        msg.mode = INCLUDE_VALUE;
    } else {
        msg.mode = EXCLUDE_VALUE;
    }
    if (filter->genericPattern == NONE) {
        msg.gen_pattern = NONE_VALUE;
    }
    else if (filter->genericPattern == PREFIX) {
        msg.gen_pattern = PREFIX_VALUE;
    } else {
        msg.gen_pattern = SUFFIX_VALUE;
    }

    if (filter->methodDetailCount > 0) {
        unsigned int i;
        size_t offset = elb.offset;
        for(i = 0; i < filter->methodDetailCount; i++) {
            elb.offset = offset;
            msg.meth_pattern = filter->methodDetails[i].pattern;
            if (filter->methodDetails[i].mode == INCLUDE) {
                msg.meth_mode = INCLUDE_VALUE;
            } else {
                msg.meth_mode = EXCLUDE_VALUE;
            }
            if (filter->methodDetails[i].genericPattern == NONE) {
                msg.meth_gen_pattern = NONE_VALUE;
            } else if (filter->methodDetails[i].genericPattern == PREFIX) {
                msg.meth_gen_pattern = PREFIX_VALUE;
            } else {
                msg.meth_gen_pattern = SUFFIX_VALUE;
            }

			printMessage(&msg, &elb);
        }
    } else {
		printMessage(&msg, &elb);
    }
}


// This is included into another tag!!!
void 
CPrintBinary::AppendAnnotations(bf_string_t methods[], bf_long_t lines[], size_t *length, SStackTrace_* stackTrace)
{
    if (!m_pOptions->isStackInfoNormal()) {
        return;
    }
    if (stackTrace == 0) {
        return;
    }
	*length = stackTrace->uiSize;

	if (*length == 0) {
		return;
	}
    unsigned int i;
    for (i = 0; i < stackTrace->uiSize; i++) {
        methods[i] = stackTrace->pStackEntries[i].methodName;
    }
    for (i = 0; i < stackTrace->uiSize; i++) {
		lines[i] = stackTrace->pStackEntries[i].lineNumber;
    }
}

void 
CPrintBinary::printCustomElement(const char *str) 
{
    CElementBuffer elb(true);
	CCustomMessage msg;

    msg.body = const_cast<char*>(str);
    
	printMessage(&msg, &elb);
}

void 
CPrintBinary::printMessage(CBinaryMessage *msg, CElementBuffer *elb) 
{
	if (msg == NULL || elb == NULL) {
		return;
	}

	msg->message_len = msg->getLength();
	size_t length =  msg->message_len + msg->getHeaderLength();
	void *buffer = elb->AllocBuffer(length);
	
	(*msg) >> buffer;
    jvmtiAgent_print(elb);
}

U64 
CPrintBinary::getTimestamp(U64 time) 
{
	return (time + _startTimeNanosec);
}

U64
CPrintBinary::getInterval(U64 interval) 
{
	return (interval);
}
