/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation, Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    IBM Corporation - Initial API and implementation
 *    Viacheslav Rybalov, Intel - Initial API and implementation
 *
 * $Id: PrintXML.cpp,v 1.35 2010/08/17 16:29:59 jwest Exp $ 
 ***********************************************************************/

#include "PrintXML.h"
#include "Filters.h"
#include "Options.h"
#include "strings.h"
#include "log.h"
#include <stdio.h>
#include <string.h>

using namespace Martini::MPI;
using namespace Martini::JPIAgent;


CPrintXML::CPrintXML(COptions* pCOptions, CFilters* pCFilters):CPrint(pCOptions, pCFilters)
{}

CPrintXML::~CPrintXML()
{}

void 
CPrintXML::PrintStartingFragments()
{
    /* If we are in standalone print the required additional elements */
    if((m_pOptions->m_jvmtiAgent_Options.standalone/*||m_pOptions->m_jvmtiAgent_Options.profilerApiEnabled*/)) {
        printXMLStartElement();
        printStandaloneTraceTagOpen();
    }

    /* Print the trace header */
    printNodeElement();
    printProcessCreateElement();
    printAgentCreateElement(m_pOptions->m_jvmtiAgent_Options.invocationOptions);
    printTraceStartElement(m_pOptions->m_jvmtiAgent_Options.invocationOptions);

    /* Print the filters and options only if the proper option is set */
    if (m_pOptions->m_jvmtiAgent_Options.filters) {
        m_pCFilters->PrintFilters(this);
    }
    if (m_pOptions->m_jvmtiAgent_Options.options) {
        m_pOptions->printOptions(this);
    }
}

void 
CPrintXML::printXMLStartElement()
{
    CElementBuffer elb;
    char *data = "?xml version=\"1.0\"?>";
    AppendElementStart(&elb, data, strlen(data));
    jvmtiAgent_print(&elb);
}

void 
CPrintXML::printStandaloneTraceTagOpen()
{
    CElementBuffer elb;
    elb.AddString("<TRACE>");
    jvmtiAgent_print(&elb);
}

void 
CPrintXML::printStandaloneTraceTagClose()
{
    CElementBuffer elb;
    elb.AddString("</TRACE>");
    jvmtiAgent_print(&elb);
}

void 
CPrintXML::printNodeElement()
{
    CElementBuffer elb;
    U64 time = jvmtiAgent_getProcessStartTime();
    AppendElementStart(&elb, NODE_ELEMENT, STRLEN_NODE_ELEMENT);

    AppendStringAttribute(&elb, NODE_ID_ATTRIBUTE, STRLEN_NODE_ID_ATTRIBUTE, GetRANodeUUID());
    AppendHostNameAttribute(&elb);
    AppendIPAddressAttribute(&elb);
    AppendIntegerAttribute(&elb, TIMEZONE_ATTRIBUTE, STRLEN_TIMEZONE_ATTRIBUTE, jvmtiAgent_getTimezone());
    if(m_pOptions->m_jvmtiAgent_Options.timestamp) {
        AppendLabeledTimestamp(&elb, TIME_ATTRIBUTE, STRLEN_TIME_ATTRIBUTE, time, TRUE);
    }
    AppendElementEnd(&elb, 0, 0);
    jvmtiAgent_print(&elb);
}

void 
CPrintXML::printProcessCreateElement()
{
    CElementBuffer elb;
    U64 time = jvmtiAgent_getProcessStartTime();

    AppendElementStart(&elb, PROCESS_CREATE_ELEMENT, STRLEN_PROCESS_CREATE_ELEMENT);
    AppendStringAttribute(&elb, PROCESS_ID_ATTRIBUTE, STRLEN_PROCESS_ID_ATTRIBUTE, GetRAProcessUUID());
    AppendIntegerAttribute(&elb, PID_ATTRIBUTE, STRLEN_PID_ATTRIBUTE, GetRAProcessId());
    AppendStringAttribute(&elb, NODE_IDREF_ATTRIBUTE, STRLEN_NODE_IDREF_ATTRIBUTE, GetRANodeUUID());
    if(m_pOptions->m_jvmtiAgent_Options.timestamp) {
        AppendLabeledTimestamp(&elb, TIME_ATTRIBUTE, STRLEN_TIME_ATTRIBUTE, time, TRUE);
    }
    AppendElementEnd(&elb, 0, 0);
    jvmtiAgent_print(&elb);
}

void 
CPrintXML::printAgentCreateElement(char *options)
{
    CElementBuffer elb;
    AppendElementStart(&elb, AGENT_CREATE_ELEMENT, STRLEN_AGENT_CREATE_ELEMENT);
    AppendStringAttribute(&elb, AGENT_ID_ATTRIBUTE, STRLEN_AGENT_ID_ATTRIBUTE, GetRAAgentUUID());
    AppendStringAttribute(&elb, AGENT_VERSION_ATTRIBUTE, STRLEN_AGENT_VERSION_ATTRIBUTE, AGENT_VERSION);
    AppendStringAttribute(&elb, PROCESS_IDREF_ATTRIBUTE, STRLEN_PROCESS_IDREF_ATTRIBUTE, GetRAProcessUUID());
    AppendStringAttribute(&elb, AGENT_NAME_ATTRIBUTE, STRLEN_AGENT_NAME_ATTRIBUTE, m_pOptions->m_jvmtiAgent_Options.processName);
    AppendStringAttribute(&elb, AGENT_TYPE_ATTRIBUTE, STRLEN_AGENT_TYPE_ATTRIBUTE, m_pOptions->m_jvmtiAgent_Options.processType);
    AppendStringAttribute(&elb, AGENT_PARAMETERS_ATTRIBUTE, STRLEN_AGENT_PARAMETERS_ATTRIBUTE, options);
    if(m_pOptions->m_jvmtiAgent_Options.timestamp) {
        AppendTimeStamp(&elb, 0);
    }
    AppendElementEnd(&elb, 0, 0);
    jvmtiAgent_print(&elb);
}

void 
CPrintXML::printTraceStartElement(char *options)
{
    CElementBuffer elb;
    AppendElementStart(&elb, TRACE_START_ELEMENT, STRLEN_TRACE_START_ELEMENT);
    AppendStringAttribute(&elb, TRACE_ID_ATTRIBUTE, STRLEN_TRACE_ID_ATTRIBUTE, GetRATraceUUID());
    AppendStringAttribute(&elb, AGENT_IDREF_ATTRIBUTE, STRLEN_AGENT_IDREF_ATTRIBUTE, GetRAAgentUUID());
    if(m_pOptions->m_jvmtiAgent_Options.timestamp) {
        AppendCurrentTimeStamp(&elb);
    }
    AppendElementEnd(&elb, 0, 0);
    jvmtiAgent_print(&elb);
}

void 
CPrintXML::printTraceEndElement()
{
    CElementBuffer elb;
    AppendElementStart(&elb, TRACE_STOP_ELEMENT, STRLEN_TRACE_STOP_ELEMENT);
//    AppendTraceIdRefAttribute(&elb);

    /* Get the timestamp information if necessary */
    if(m_pOptions->m_jvmtiAgent_Options.timestamp) {
        AppendCurrentTimeStamp(&elb);
    }

    AppendElementEnd(&elb, 0, 0);
    jvmtiAgent_print(&elb);
}

void 
CPrintXML::printJvmShutdownElement() {
    CElementBuffer elb;

    AppendElementStart(&elb, JVM_SHUT_DOWN_ELEMENT, STRLEN_JVM_SHUT_DOWN_ELEMENT);
//    AppendEmptyEventHeader(&elb, /*threadId*/0, JVM_SHUT_DOWN_ELEMENT, STRLEN_JVM_SHUT_DOWN_ELEMENT); //TODO Fix thread id
    /* Get the timestamp information if necessary */
    if(m_pOptions->m_jvmtiAgent_Options.timestamp) {
        AppendCurrentTimeStamp(&elb);
    }
//    AppendTraceIdRefAttribute(&elb);
    AppendElementEnd(&elb, 0, 0);
    jvmtiAgent_print(&elb);
    /* If a global method count was asked for,  print it out */
//    if (m_pOptions->m_jvmtiAgent_Options.methodCounts) {
//            jvmpiAgent_ForAll(Method_t, jvmtiAgent_printMethodCount, pElB); //TODO
//    }

    /* Print the closing tag */
    printTraceEndElement();
    printAgentDestroyElement();
    if(m_pOptions->m_jvmtiAgent_Options.standalone) {
        printStandaloneTraceTagClose();
    }
}

void 
CPrintXML::printAgentDestroyElement()
{
    CElementBuffer elb;
    AppendElementStart(&elb, AGENT_DESTROY_ELEMENT, STRLEN_AGENT_DESTROY_ELEMENT);
    AppendStringAttribute(&elb, AGENT_IDREF_ATTRIBUTE, STRLEN_AGENT_IDREF_ATTRIBUTE, GetRAAgentUUID());

    /* Get the timestamp information if necessary */
    if(m_pOptions->m_jvmtiAgent_Options.timestamp) {
        AppendCurrentTimeStamp(&elb);
    }
    AppendElementEnd(&elb, 0, 0);
    jvmtiAgent_print(&elb);
}

/** PRINT_METHOD  **********************************************************
* Handles the output of a METHOD element and attributes
*/
void 
CPrintXML::printNewMethodElement(TId methodId, SMethodInfo* methodInfo)
{
    CElementBuffer elb;
    AppendElementStart(&elb, METHOD_ELEMENT, STRLEN_METHOD_ELEMENT);
    char* formatedName = FormatName(methodInfo->szJavaMethodName);
    AppendStringAttribute(&elb, NAME_ATTRIBUTE, STRLEN_NAME_ATTRIBUTE, formatedName);
    free(formatedName);
    AppendStringAttribute(&elb, SIGNATURE_ATTRIBUTE, STRLEN_SIGNATURE_ATTRIBUTE, methodInfo->szJavaMethodSig);
    if ((methodInfo->validData & DR_METHOD_LINE_NUMBERS) != 0) {
        AppendIntegerAttribute(&elb, START_LINENO_ATTRIBUTE, STRLEN_START_LINENO_ATTRIBUTE, methodInfo->methodLineNumbers.uiStart);
        AppendIntegerAttribute(&elb, END_LINENO_ATTRIBUTE, STRLEN_END_LINENO_ATTRIBUTE, methodInfo->methodLineNumbers.uiEnd);
    }
    AppendInteger64Attribute(&elb, STATIC_METHOD_ID_ATTRIBUTE, STRLEN_STATIC_METHOD_ID_ATTRIBUTE, methodId);
    AppendInteger64Attribute(&elb, STATIC_CLASS_IDREF_ATTRIBUTE, STRLEN_STATIC_CLASS_IDREF_ATTRIBUTE, methodInfo->classId);
//    AppendTraceIdRefAttribute(&elb);
    AppendElementEnd(&elb, 0, 0);
    jvmtiAgent_print(&elb);
}

/** PRINT_METHOD_ENTRY_EVENT  **********************************************
* Handles the output of a METHOD_ENTRY, METHOD_CALL element and attributes
*/
void 
CPrintXML::printMethodEntryElement(TId threadId, TId methodId, TId classId, unsigned long ticket, unsigned long stackDepth) 
{
    CElementBuffer elb;
    AppendEmptyEventHeader(&elb, threadId, METHOD_ENTRY_ATTRIBUTE, STRLEN_METHOD_ENTRY_ATTRIBUTE );

    if(m_pOptions->m_jvmtiAgent_Options.timestamp) {
        AppendCurrentTimeStamp(&elb);
    }
    AppendInteger64Attribute(&elb, STATIC_METHOD_IDREF_ATTRIBUTE, STRLEN_STATIC_METHOD_IDREF_ATTRIBUTE, methodId);
    AppendInteger64Attribute(&elb, STATIC_CLASS_IDREF_ATTRIBUTE, STRLEN_STATIC_CLASS_IDREF_ATTRIBUTE, classId);

    if(m_pOptions->m_jvmtiAgent_Options.ticket) {
        AppendTicketAttribute(&elb, ticket);
    }
    AppendStackDepthAttribute(&elb, stackDepth);

//    AppendTraceIdRefAttribute(&elb);
    AppendElementEnd(&elb, 0, 0);
    jvmtiAgent_print(&elb);
}

/** PRINT_AG_METHOD_ENTRY_EVENT  **************************preagg/135437*******
  *
  * Handles the output of a AG_METHOD_ENTRY element and attributes
  */
void 
CPrintXML::printAgMethodEntryElement(TId threadId, TId methodId, U64 baseTime, U64 minTime, U64 maxTime, U64 baseCPUTime, U64 numCalls) {

    CElementBuffer elb;
    AppendEmptyEventHeader(&elb, threadId, AG_METHOD_ENTRY_ELEMENT, STRLEN_AG_METHOD_ENTRY_ELEMENT);
    AppendInteger64Attribute(&elb, STATIC_METHOD_IDREF_ATTRIBUTE, STRLEN_STATIC_METHOD_IDREF_ATTRIBUTE, methodId);
    if (numCalls > 0) {
        AppendLabeledTimestamp(&elb, BASETIME1_ATTRIBUTE, STRLEN_BASETIME1_ATTRIBUTE, baseTime, TRUE);
        AppendLabeledTimestamp(&elb, MINTIME_ATTRIBUTE, STRLEN_MINTIME_ATTRIBUTE, minTime, TRUE);
        AppendLabeledTimestamp(&elb, MAXTIME_ATTRIBUTE, STRLEN_MAXTIME_ATTRIBUTE, maxTime, TRUE);   
        if(m_pOptions->m_jvmtiAgent_Options.cpuTime) {
            AppendThreadCPUTime(&elb, baseCPUTime); 
        }
    }
    AppendInteger64Attribute(&elb, COUNT_ATTRIBUTE, STRLEN_COUNT_ATTRIBUTE, numCalls);
    AppendElementEnd(&elb, 0, 0);
    jvmtiAgent_print(&elb);
}

/** PRINT_METHOD_EXIT_EVENT  ***********************************************
* Prints the METHOD_EXIT, METHOD_RETURN element
*/
void
CPrintXML::printMethodExitElement(TId threadId, TId methodId, TId classId, unsigned long ticket, U64 cpuTime)
{
    CElementBuffer elb;
    AppendEmptyEventHeader(&elb, threadId, METHOD_EXIT_ELEMENT, STRLEN_METHOD_EXIT_ELEMENT);
    AppendInteger64Attribute(&elb, STATIC_METHOD_IDREF_ATTRIBUTE, STRLEN_STATIC_METHOD_IDREF_ATTRIBUTE, methodId);
    AppendInteger64Attribute(&elb, STATIC_CLASS_IDREF_ATTRIBUTE, STRLEN_STATIC_CLASS_IDREF_ATTRIBUTE, classId);

    if(m_pOptions->m_jvmtiAgent_Options.cpuTime) {
        AppendThreadCPUTime(&elb, cpuTime); 
    }
    if(m_pOptions->m_jvmtiAgent_Options.ticket) {
        AppendTicketAttribute(&elb, ticket);
    }

//    AppendTraceIdRefAttribute(&elb);

    /* Print the timestamp (and overhead) information if necessary */
    if(m_pOptions->m_jvmtiAgent_Options.timestamp) {
        AppendCurrentTimeStamp(&elb);
    }

    AppendElementEnd(&elb, 0, 0);
    jvmtiAgent_print(&elb);
}

/** PRINT_AG_METHOD_EXIT_ELEMENT **************************preAgg/135437*******
  * 
  * Prints the METHOD_EXIT element
  */
void 
CPrintXML::printAgMethodExitElement(TId threadId, TId methodId) {

    CElementBuffer elb;
    AppendEmptyEventHeader(&elb, threadId, AG_METHOD_EXIT_ELEMENT, STRLEN_AG_METHOD_EXIT_ELEMENT);
    AppendInteger64Attribute(&elb, STATIC_METHOD_IDREF_ATTRIBUTE, STRLEN_STATIC_METHOD_IDREF_ATTRIBUTE, methodId);
    AppendElementEnd(&elb, 0, 0);
    jvmtiAgent_print(&elb);
}

/** PRINT_THREAD_START_ELEMENT  ********************************************
*
*/
void
CPrintXML::printThreadStartElement(TId threadId, TId objectId, SThreadInfo* threadInfo)
{
    CElementBuffer elb;

    AppendEmptyEventHeader1(&elb, threadId, THREAD_START_ELEMENT, STRLEN_THREAD_START_ELEMENT);
    /* Get the timestamp information if necessary */
    if(m_pOptions->m_jvmtiAgent_Options.timestamp) {
        AppendCurrentTimeStamp(&elb);
    }
    if (objectId != 0) {
        AppendInteger64Attribute(&elb, STATIC_OBJ_IDREF_ATTRIBUTE, STRLEN_STATIC_OBJ_IDREF_ATTRIBUTE, objectId);
    }
    if (threadInfo->validData && DR_THREAD_INFO) {
        char* formatedName = 0;
        formatedName = FormatName(threadInfo->szName);
        AppendStringAttribute(&elb, THREAD_NAME_ATTRIBUTE, STRLEN_THREAD_NAME_ATTRIBUTE, formatedName);
        free(formatedName);
        formatedName = FormatName(threadInfo->szGroupName);
        AppendStringAttribute(&elb, GROUP_NAME_ATTRIBUTE, STRLEN_GROUP_NAME_ATTRIBUTE, formatedName);
        free(formatedName);
        formatedName = FormatName(threadInfo->szParentGroupName);
        AppendStringAttribute(&elb, PARENT_NAME_ATTRIBUTE, STRLEN_PARENT_NAME_ATTRIBUTE, formatedName);
        free(formatedName);
    }
//    AppendTraceIdRefAttribute(&elb);
    AppendElementEnd(&elb, 0, 0);
    jvmtiAgent_print(&elb);
}

/** PRINT_THREAD_END_ELEMENT  **********************************************
*/
void 
CPrintXML::printThreadEndElement(TId threadId)
{
    CElementBuffer elb;
    AppendEmptyEventHeader(&elb, threadId, THREAD_END_ELEMENT, STRLEN_THREAD_END_ELEMENT);
    /* Get the timestamp information if necessary */
    if(m_pOptions->m_jvmtiAgent_Options.timestamp) {
        AppendCurrentTimeStamp(&elb);
    }

//    AppendTraceIdRefAttribute(&elb);
    AppendElementEnd(&elb, 0, 0);
    jvmtiAgent_print(&elb);
}

void 
CPrintXML::printJvmInitDoneElement(TId threadId)
{
    CElementBuffer elb;

    AppendEmptyEventHeader(&elb, threadId, JVM_INIT_DONE_ELEMENT, STRLEN_JVM_INIT_DONE_ELEMENT);
    /* Get the timestamp information if necessary */
    if(m_pOptions->m_jvmtiAgent_Options.timestamp) {
        AppendCurrentTimeStamp(&elb);
    }
//    AppendTraceIdRefAttribute(&elb);
    AppendElementEnd(&elb, 0, 0);
    jvmtiAgent_print(&elb);
}

/** PRINT_CLASS  ****************************************************************
  * Handles the output of the CLASS_LOAD element and attributes
  */
void
CPrintXML::printNewClassElement(TId classId, SClassInfo* classInfo)
{
    CElementBuffer elb;

    AppendElementStart(&elb, CLASS_LOAD_ELEMENT, STRLEN_CLASS_LOAD_ELEMENT);
    char* convertedName = ConvertClassName(classInfo->szJavaClassName);
    AppendStringAttribute(&elb, NAME_ATTRIBUTE, STRLEN_NAME_ATTRIBUTE, convertedName);
    free(convertedName);
    AppendStringAttribute(&elb, SOURCE_NAME_ATTRIBUTE, STRLEN_SOURCE_NAME_ATTRIBUTE, classInfo->szSourceFileName);
    AppendInteger64Attribute(&elb, STATIC_CLASS_ID_ATTRIBUTE, STRLEN_STATIC_CLASS_ID_ATTRIBUTE, classId);
    if(m_pOptions->m_jvmtiAgent_Options.timestamp) {
        AppendCurrentTimeStamp(&elb);
    }
//    AppendTraceIdRefAttribute(&elb);
    AppendElementEnd(&elb, 0, 0);
    jvmtiAgent_print(&elb);
}

/** PRINT_OBJ_ALLOC_ELEMENT  *****************************************************
  *
  */
void
CPrintXML::printObjAllocElement(SHeapEventData* heapData, U64 lineNumber)
{
    CElementBuffer elb;

    AppendEmptyEventHeader(&elb, heapData->threadId, OBJ_ALLOC_ELEMENT, STRLEN_OBJ_ALLOC_ELEMENT);
    if(m_pOptions->m_jvmtiAgent_Options.timestamp) {
        AppendCurrentTimeStamp(&elb);
    }
    /* Print the object identifier as well as the class identifier */
    AppendInteger64Attribute(&elb, STATIC_OBJ_ID_ATTRIBUTE, STRLEN_STATIC_OBJ_ID_ATTRIBUTE, heapData->objectId);
    AppendInteger64Attribute(&elb, STATIC_CLASS_IDREF_ATTRIBUTE, STRLEN_STATIC_CLASS_IDREF_ATTRIBUTE, heapData->pObjectInfo->classId);
    if(m_pOptions->m_jvmtiAgent_Options.objAllocIsArray) {
//        AppendIntegerAttribute(&elb, IS_ARRAY_ATTRIBUTE, STRLEN_IS_ARRAY_ATTRIBUTE, heapData->pObjectInfo->arrayType == AT_NONE ? 0 : 1); //workaround for 182684, 173861, 182706, 178049
        AppendIntegerAttribute(&elb, IS_ARRAY_ATTRIBUTE, STRLEN_IS_ARRAY_ATTRIBUTE, 0);
    }
    /* Print the size of this object */
    AppendInteger64Attribute(&elb, SIZE_ATTRIBUTE, STRLEN_SIZE_ATTRIBUTE, heapData->pObjectInfo->uiSize);
    if (m_pOptions->isAllocSitesSupported() && ((heapData->validData & DR_METHOD_ID) != 0)) {
        AppendInteger64Attribute(&elb, ALLOC_METHOD_ID_ATTRIBUTE, STRLEN_ALLOC_METHOD_ID_ATTRIBUTE, heapData->allocMethodId);
        if (lineNumber != -1) {
            AppendInteger64Attribute(&elb, ALLOC_LINE_NUMBER_ATTRIBUTE, STRLEN_ALLOC_LINE_NUMBER_ATTRIBUTE, lineNumber);
        } else {
            AppendStringAttribute(&elb, ALLOC_LINE_NUMBER_ATTRIBUTE, STRLEN_ALLOC_LINE_NUMBER_ATTRIBUTE, "-1");
        }
    }
//    AppendTraceIdRefAttribute(&elb);
    AppendElementEnd(&elb, 0, 0);
    jvmtiAgent_print(&elb);
}

/** PRINT_OBJ_FREE_ELEMENT  ************************************************
* Handles the output of OBJ_FREE.
*/
void
CPrintXML::printObjFreeElement(SHeapEventData* heapData)
{
    CElementBuffer elb;

//     if (!_jvmpiAgent_suspendIO && classHashEntry && (_jvmpiAgent_Options.gc != GcNone))   //TODO Check conditions
    AppendElementStart(&elb, OBJ_FREE_ELEMENT, STRLEN_OBJ_FREE_ELEMENT);
//    AppendEmptyEventHeader(&elb, heapData->threadId, OBJ_FREE_ELEMENT, STRLEN_OBJ_FREE_ELEMENT);  Check it
    if(m_pOptions->m_jvmtiAgent_Options.timestamp) {
        AppendCurrentTimeStamp(&elb);
    }
    AppendInteger64Attribute(&elb, STATIC_OBJ_IDREF_ATTRIBUTE, STRLEN_STATIC_OBJ_IDREF_ATTRIBUTE, heapData->objectId);
    AppendInteger64Attribute(&elb, OBJ_AGE_ATTRIBUTE, STRLEN_OBJ_AGE_ATTRIBUTE, heapData->objectAge);
//    AppendTraceIdRefAttribute(&elb);
    AppendElementEnd(&elb, 0, 0);
    jvmtiAgent_print(&elb);
}

void 
CPrintXML::printGcStartElement()
{
    CElementBuffer elb;

//    if (!_jvmpiAgent_suspendIO && _jvmpiAgent_Options.gc != GcNone) TODO Check it
	AppendElementStart(&elb, GC_START_ELEMENT, STRLEN_GC_START_ELEMENT);
    if(m_pOptions->m_jvmtiAgent_Options.timestamp) {
        AppendCurrentTimeStamp(&elb);
    }
//    AppendTraceIdRefAttribute(&elb);
    AppendElementEnd(&elb, 0, 0);
    jvmtiAgent_print(&elb);
}

void 
CPrintXML::printGcFinishElement()
{
    CElementBuffer elb;

//    if (!_jvmpiAgent_suspendIO && _jvmpiAgent_Options.gc != GcNone) TODO
	AppendElementStart(&elb, GC_FINISH_ELEMENT, STRLEN_GC_FINISH_ELEMENT);
    if(m_pOptions->m_jvmtiAgent_Options.timestamp) {
        AppendCurrentTimeStamp(&elb);
    }
//    AppendInteger64Attribute(&elb, USED_OBJECTS_ATTRIBUTE, STRLEN_USED_OBJECTS_ATTRIBUTE, 0);  //TODO
//    AppendInteger64Attribute(&elb, USED_OBJECT_SPACE_ATTRIBUTE, STRLEN_USED_OBJECT_SPACE_ATTRIBUTE, 0);   //TODO
//    AppendInteger64Attribute(&elb, TOTAL_OBJECT_SPACE_ATTRIBUTE, STRLEN_TOTAL_OBJECT_SPACE_ATTRIBUTE, 0);   //TODO
//    AppendTraceIdRefAttribute(&elb);
    AppendElementEnd(&elb, 0, 0);
    jvmtiAgent_print(&elb);
}

/* PR Object Ref */
void 
CPrintXML::printHDStartElement(TId heapDumpId, char* heapDefName, U64 basetime)
{
    CElementBuffer elb;

	AppendElementStart(&elb, HD_START_ELEMENT, STRLEN_HD_START_ELEMENT);
    AppendInteger64Attribute(&elb, HEAPDUMP_ID_ATTRIBUTE, STRLEN_HEAPDUMP_ID_ATTRIBUTE, heapDumpId);
    if(m_pOptions->m_jvmtiAgent_Options.timestamp) {
        AppendCurrentTimeStamp(&elb);
    }
	AppendStringAttribute(&elb, NAME_ATTRIBUTE, STRLEN_NAME_ATTRIBUTE, heapDefName);
	AppendLabeledTimestamp(&elb, BASETIME_ATTRIBUTE, STRLEN_BASETIME_ATTRIBUTE, basetime, FALSE);
//    AppendTraceIdRefAttribute(&elb);
    AppendElementEnd(&elb, 0, 0);
    jvmtiAgent_print(&elb);
}

/** PRINT_GC_ROOT_ELEMEMT  *******************************************************
*
*/
void
CPrintXML::printGcRootElement(TId heapDumpIdRef, TId objIdRef, EObjectReferenceType type)
{
    CElementBuffer elb;
    char* typeName = "Fix me";
//    if(!_jvmpiAgent_suspendIO && _jvmpiAgent_Options.gc != GcNone) TODO Check 
        unsigned short current=0;
    AppendElementStart(&elb, GC_ROOT_ELEMENT, STRLEN_GC_ROOT_ELEMENT);

    if(type = OR_SYSTEM_CLASS) {
        AppendInteger64Attribute(&elb, STATIC_CLASS_IDREF_ATTRIBUTE, STRLEN_STATIC_CLASS_IDREF_ATTRIBUTE, objIdRef);
    } else {
        AppendInteger64Attribute(&elb, STATIC_OBJ_IDREF_ATTRIBUTE, STRLEN_STATIC_OBJ_IDREF_ATTRIBUTE, objIdRef);
    }
    switch (type) {
        case OR_SYSTEM_CLASS:
            typeName = "SYSTEM_CLASS";  //TODO Check
            break;
        case OR_MONITOR:
            typeName = "MONITOR";  //TODO Check
            break;
        case OR_STACK_LOCAL:
            typeName = "STACK_LOCAL";  //TODO Check
            break;
        case OR_JAVA_JNI_LOCAL:
            typeName = "JNI_LOCAL";
            break;
        case OR_THREAD:
            typeName = "THREAD";  //TODO Check
            break;
        case OR_OTHER:
            typeName = "UNKNOWN";
            break;
    }
    AppendStringAttribute(&elb, GC_ROOT_TYPE_ATTRIBUTE, STRLEN_GC_ROOT_TYPE_ATTRIBUTE, typeName);
    AppendElementEnd(&elb, 0, 0);
    jvmtiAgent_print(&elb);
}

void 
CPrintXML::printObjectReferenceElement(TId heapDumpIdRef, SObjectReference* objRef)
{
    CElementBuffer elb;

	AppendElementStart(&elb, OBJECT_REFERENCE_ATTRIBUTE, STRLEN_OBJECT_REFERENCE_ATTRIBUTE);
    AppendInteger64Attribute(&elb, OBJECT_OWNER_ATTRIBUTE, STRLEN_OBJECT_OWNER_ATTRIBUTE, objRef->srcObjectId);
    AppendInteger64Attribute(&elb, OBJECT_TARGET_ATTRIBUTE, STRLEN_OBJECT_TARGET_ATTRIBUTE, objRef->targetObjectId);
    if (0/*isArray*/) {
//        AppendIntegerAttribute(&elb, OBJECT_ARRAY_INDEX_ATTRIBUTE, STRLEN_OBJECT_ARRAY_INDEX_ATTRIBUTE, 0); //TODO
    } else {
//	    AppendIntegerAttribute(&elb, FIELD_IDREF_ATTRIBUTE, STRLEN_FIELD_IDREF_ATTRIBUTE, 0); //TODO
    }
	AppendInteger64Attribute(&elb, HEAPDUMP_IDREF_ATTRIBUTE, STRLEN_HEAPDUMP_IDREF_ATTRIBUTE, heapDumpIdRef);
//	AppendIntegerAttribute(&elb, HEAPDUMP_FIRST_SEEN_ATTRIBUTE, STRLEN_HEAPDUMP_FIRST_SEEN_ATTRIBUTE, 0); //TODO ???
    AppendElementEnd(&elb, 0, 0);
    jvmtiAgent_print(&elb);
}

void
CPrintXML::printMonitorWaitElement(SMonitorWaitEventData* monitorWaitData/*int isThreadSleep*/, SStackTrace_* stackTrace)
{
    CElementBuffer elb;
    AppendEmptyEventHeader(&elb, monitorWaitData->threadId, MONITOR_WAIT_ELEMENT, STRLEN_MONITOR_WAIT_ELEMENT);
    if(m_pOptions->m_jvmtiAgent_Options.timestamp) {
        AppendCurrentTimeStamp(&elb);
    }
    if (monitorWaitData->objectId != -1) {
        AppendInteger64Attribute(&elb, STATIC_OBJ_IDREF_ATTRIBUTE, STRLEN_STATIC_OBJ_IDREF_ATTRIBUTE, monitorWaitData->objectId);
    } else {
        AppendStringAttribute(&elb, STATIC_OBJ_IDREF_ATTRIBUTE, STRLEN_STATIC_OBJ_IDREF_ATTRIBUTE, "-1");
    }
    AppendInteger64Attribute(&elb, TIMEOUT_ATTRIBUTE, STRLEN_TIMEOUT_ATTRIBUTE, monitorWaitData->timeoutMillis);
    //    AppendTraceIdRefAttribute(&elb);
    if (stackTrace != NULL && m_pOptions->isStackInfoNormal()) {
        AppendElementEnd2(&elb);
        AppendAnnotations(&elb, stackTrace);
        AppendClosingTag(&elb, MONITOR_WAIT_ELEMENT, STRLEN_MONITOR_WAIT_ELEMENT);
    } else {
        AppendElementEnd(&elb, 0, 0);
    }
    jvmtiAgent_print(&elb);
}

void
CPrintXML::printMonitorWaitedElement(SMonitorWaitedEventData* monitorWaitedData, U64 timeout, SStackTrace_* stackTrace)
{
    CElementBuffer elb;
    AppendEmptyEventHeader(&elb, monitorWaitedData->threadId, MONITOR_WAITED_ELEMENT, STRLEN_MONITOR_WAITED_ELEMENT);
    if(m_pOptions->m_jvmtiAgent_Options.timestamp) {
        AppendCurrentTimeStamp(&elb);
    }
    /* Print the object identifier. Object identifier is -1 if in Thread.sleep() */
    AppendInteger64Attribute(&elb, STATIC_OBJ_IDREF_ATTRIBUTE, STRLEN_STATIC_OBJ_IDREF_ATTRIBUTE,
        /*(isThreadSleep == 0 ?*/ monitorWaitedData->objectId /*: -1)*/);

    AppendInteger64Attribute(&elb, TIMEOUT_ATTRIBUTE, STRLEN_TIMEOUT_ATTRIBUTE, timeout/1000000);

    AppendIntegerAttribute(&elb, MONITOR_IS_TIMED_OUT_ATTRIBUTE, STRLEN_MONITOR_IS_TIMED_OUT_ATTRIBUTE, monitorWaitedData->isTimedOut);

    if (stackTrace != NULL && m_pOptions->isStackInfoNormal()) {
        AppendElementEnd2(&elb);
        AppendAnnotations(&elb, stackTrace);
        AppendClosingTag(&elb, MONITOR_WAITED_ELEMENT, STRLEN_MONITOR_WAITED_ELEMENT);
    } else {
        AppendElementEnd(&elb, 0, 0);
    }
    jvmtiAgent_print(&elb);
}

/** PRINT_MONITOR_CONTENDED_ENTER_ELEMENT  ************************************************
  *
  * Print the <monContendedEnter> element to the trace indicating
  * JVMPI_EVENT_MONITOR_CONTENDED_ENTER occurred.
  *
  * args -
  *		objectHashEntry - the hash entry corresponding to the monitor object
  *		event - the MONITOR_WAITED event as reported by JVMPI
  *		thread_owner - the current thread that owns the monitor object
  *		timestamp - when the MONITOR_WAITED event occurred
  */
void
CPrintXML::printMonitorContendedEnterElement(SContendedMonitorEnterEventData* conMonEnterData, SStackTrace_* stackTrace)

{
    CElementBuffer elb;
    AppendEmptyEventHeader(&elb, conMonEnterData->threadId, MONITOR_CONTENDED_ENTER_ELEMENT, STRLEN_MONITOR_CONTENDED_ENTER_ELEMENT);
    if(m_pOptions->m_jvmtiAgent_Options.timestamp) {
        AppendCurrentTimeStamp(&elb);
    }
    AppendInteger64Attribute(&elb, STATIC_OBJ_IDREF_ATTRIBUTE, STRLEN_STATIC_OBJ_IDREF_ATTRIBUTE, conMonEnterData->objectId);
    /* print the thread owner attribute */
    if ((conMonEnterData->validData & DR_MONITOR_OWNER_THREAD_ID) != 0) {
        AppendInteger64Attribute(&elb, THREAD_OWNER_ATTRIBUTE, STRLEN_THREAD_OWNER_ATTRIBUTE, conMonEnterData->ownerThreadId);
    } else {
        AppendStringAttribute(&elb, THREAD_OWNER_ATTRIBUTE, STRLEN_THREAD_OWNER_ATTRIBUTE, "<Unknown>");
    }
//    AppendTraceIdRefAttribute(&elb);
    if (stackTrace != NULL && m_pOptions->isStackInfoNormal()) {
        AppendElementEnd2(&elb);
        AppendAnnotations(&elb, stackTrace);
        AppendClosingTag(&elb, MONITOR_CONTENDED_ENTER_ELEMENT, STRLEN_MONITOR_CONTENDED_ENTER_ELEMENT);
    } else {
        AppendElementEnd(&elb, 0, 0);
    }
    jvmtiAgent_print(&elb);
}

/** PRINT_MONITOR_CONTENDED_ENTERED_ELEMENT  ************************************************
  *
  * Print the <monContendedEntered> element to the trace
  */
void
CPrintXML::printMonitorContendedEnteredElement(SContendedMonitorEnteredEventData* conMonEnteredData, SStackTrace_* stackTrace)
{
    CElementBuffer elb;
    AppendEmptyEventHeader(&elb, conMonEnteredData->threadId, MONITOR_CONTENDED_ENTERED_ELEMENT, STRLEN_MONITOR_CONTENDED_ENTERED_ELEMENT);
    if(m_pOptions->m_jvmtiAgent_Options.timestamp) {
        AppendCurrentTimeStamp(&elb);
    }
    AppendInteger64Attribute(&elb, STATIC_OBJ_IDREF_ATTRIBUTE, STRLEN_STATIC_OBJ_IDREF_ATTRIBUTE, conMonEnteredData->objectId);
//    AppendTraceIdRefAttribute(&elb);
    if (stackTrace != NULL && m_pOptions->isStackInfoNormal()) {
        AppendElementEnd2(&elb);
        AppendAnnotations(&elb, stackTrace);
        AppendClosingTag(&elb, MONITOR_CONTENDED_ENTERED_ELEMENT, STRLEN_MONITOR_CONTENDED_ENTERED_ELEMENT);
    } else {
        AppendElementEnd(&elb, 0, 0);
    }
    jvmtiAgent_print(&elb);
}

/** Print Thread interaction elements   ************************************************
  *
  * Print all element originated by ThreadInteraction to the trace 
  * TODO: clarify sintaxis of the element
  */
void
CPrintXML::printThreadInteractionElements(SThreadInteractionEventData* threadInteractionData, SStackTrace_* stackTrace)
{
    CElementBuffer elb;

    static const char * const strThreadInteractionElements[] = {
        NULL,
        MONITOR_NOTIFY_CALLED_ELEMENT,   // CALL_ID_NOTIFY_ALL
        MONITOR_NOTIFY_CALLED_ELEMENT,   // CALL_ID_NOTIFY
        THREAD_INTERRUPT_CALLED_ELEMENT, // CALL_ID_INTERRUPT
        THREAD_START_CALLED_ELEMENT,      // CALL_ID_START
        THREAD_SLEEP_START_ELEMENT,
        THREAD_SLEEP_END_ELEMENT
    };
    static const int strThreadInteractionElementsLen[] = {
        0,
        STRLEN_MONITOR_NOTIFY_CALLED_ELEMENT,   // CALL_ID_NOTIFY_ALL
        STRLEN_MONITOR_NOTIFY_CALLED_ELEMENT,   // CALL_ID_NOTIFY
        STRLEN_THREAD_INTERRUPT_CALLED_ELEMENT, // CALL_ID_INTERRUPT
        STRLEN_THREAD_START_CALLED_ELEMENT,      // CALL_ID_START
        STRLEN_THREAD_SLEEP_START_ELEMENT,
        STRLEN_THREAD_SLEEP_END_ELEMENT
    };


    const char* const elementName = strThreadInteractionElements[threadInteractionData->interactionType];
    const int elementNameLen = strThreadInteractionElementsLen[threadInteractionData->interactionType];
    bool interactionTypeIsSleep = FALSE;

    if(threadInteractionData->interactionType == IT_SLEEP_START || threadInteractionData->interactionType == IT_SLEEP_END) {
    	interactionTypeIsSleep = TRUE;
    }

    AppendEmptyEventHeader(&elb, threadInteractionData->threadId, elementName, elementNameLen);
    if(m_pOptions->m_jvmtiAgent_Options.timestamp) {
        AppendCurrentTimeStamp(&elb);
    }

    // If the interaction event is neither sleep start or sleep end
    if(!interactionTypeIsSleep) {
    	AppendInteger64Attribute(&elb, STATIC_OBJ_IDREF_ATTRIBUTE, STRLEN_STATIC_OBJ_IDREF_ATTRIBUTE, threadInteractionData->objectId);
    }

    if (threadInteractionData->interactionType == IT_NOTIFY_ALL) {
        AppendIntegerAttribute(&elb, MONITOR_IS_NOTIFYALL_ATTRIBUTE, STRLEN_MONITOR_IS_NOTIFYALL_ATTRIBUTE, 1);
    }
    if (threadInteractionData->interactionType == IT_NOTIFY) {
        AppendIntegerAttribute(&elb, MONITOR_IS_NOTIFYALL_ATTRIBUTE, STRLEN_MONITOR_IS_NOTIFYALL_ATTRIBUTE, 0);
    }
    //AppendTraceIdRefAttribute(&elb);

    if (stackTrace != NULL && m_pOptions->isStackInfoNormal() && !interactionTypeIsSleep) {
        AppendElementEnd2(&elb);
        AppendAnnotations(&elb, stackTrace);
        AppendClosingTag(&elb, elementName, elementNameLen);
    } else {
        AppendElementEnd(&elb, 0, 0);
    }
    jvmtiAgent_print(&elb);
}

void
CPrintXML::printCustomElement(const char *str)
{
    CElementBuffer elb;
    elb.AddString(str);
    jvmtiAgent_print(&elb);
}

/** PRINT_OPTION  **********************************************************
* Outputs an OPTION element
*/
void
CPrintXML::PrintOption(const char *key, const char *value)
{
    CElementBuffer elb;
    AppendElementStart(&elb, OPTION_ELEMENT, STRLEN_OPTION_ELEMENT);
    AppendStringAttribute(&elb, KEY_ATTRIBUTE, STRLEN_KEY_ATTRIBUTE, key);
    AppendStringAttribute(&elb, VALUE_ATTRIBUTE, STRLEN_VALUE_ATTRIBUTE, value);
    AppendElementEnd(&elb, 0, 0);
    jvmtiAgent_print(&elb);     
}

/** PRINT_FILTER  **********************************************************
  * Outputs a FILTER element
  */
void
CPrintXML::printFilter(Filter *filter) {
    CElementBuffer elb; 
    char *startPattern;
    
    AppendElementStart(&elb, FILTER_ELEMENT, STRLEN_FILTER_ELEMENT);
    startPattern = elb.buffer + elb.offset;
    AppendStringAttribute(&elb, PATTERN_ATTRIBUTE, STRLEN_PATTERN_ATTRIBUTE, filter->pattern);

    elb.buffer[elb.offset] = '\0';

    /* RKD:  If the pattern contains any '/' replace these with '.' */
    while((startPattern = strchr(startPattern, '/'))) {
        *startPattern = (char)'.';
    }

    if (filter->GetClassFilterMode(filter) == INCLUDE) {
        AppendStringAttribute(&elb, CLASS_MODE_ATTRIBUTE, STRLEN_CLASS_MODE_ATTRIBUTE, INCLUDE_VALUE);
    } else {
        AppendStringAttribute(&elb, CLASS_MODE_ATTRIBUTE, STRLEN_CLASS_MODE_ATTRIBUTE, EXCLUDE_VALUE);
    }
    if (filter->genericPattern == NONE) {
        AppendStringAttribute(&elb, GENERICPATTERN_ATTRIBUTE, STRLEN_GENERICPATTERN_ATTRIBUTE, NONE_VALUE);
    }
    else if (filter->genericPattern == PREFIX) {
        AppendStringAttribute(&elb, GENERICPATTERN_ATTRIBUTE, STRLEN_GENERICPATTERN_ATTRIBUTE, PREFIX_VALUE);
    } else {
        AppendStringAttribute(&elb, GENERICPATTERN_ATTRIBUTE, STRLEN_GENERICPATTERN_ATTRIBUTE, SUFFIX_VALUE);
    }

    if (filter->methodDetailCount > 0) {
        unsigned int i;
        size_t offset = elb.offset;
        for(i = 0; i < filter->methodDetailCount; i++) {
            elb.offset = offset;
            AppendStringAttribute(&elb, METHOD_PATTERN_ATTRIBUTE, STRLEN_METHOD_PATTERN_ATTRIBUTE, filter->methodDetails[i].pattern);
            if (filter->methodDetails[i].mode == INCLUDE) {
                AppendStringAttribute(&elb, METHOD_MODE_ATTRIBUTE, STRLEN_METHOD_MODE_ATTRIBUTE, INCLUDE_VALUE);
            } else {
                AppendStringAttribute(&elb, METHOD_MODE_ATTRIBUTE, STRLEN_METHOD_MODE_ATTRIBUTE, EXCLUDE_VALUE);
            }
            if (filter->methodDetails[i].genericPattern == NONE) {
                AppendStringAttribute(&elb, METHOD_GENERICPATTERN_ATTRIBUTE, STRLEN_METHOD_GENERICPATTERN_ATTRIBUTE, NONE_VALUE);
            } else if (filter->methodDetails[i].genericPattern == PREFIX) {
                AppendStringAttribute(&elb, METHOD_GENERICPATTERN_ATTRIBUTE, STRLEN_METHOD_GENERICPATTERN_ATTRIBUTE, PREFIX_VALUE);
            } else {
                AppendStringAttribute(&elb, METHOD_GENERICPATTERN_ATTRIBUTE, STRLEN_METHOD_GENERICPATTERN_ATTRIBUTE, SUFFIX_VALUE);
            }
            AppendElementEnd(&elb, 0, 0);
            jvmtiAgent_print(&elb);
        }
    } else {
//    AppendTraceIdRefAttribute(&elb);
        AppendElementEnd(&elb, 0, 0);
        jvmtiAgent_print(&elb);
    }
}

void 
CPrintXML::AppendAnnotations(CElementBuffer *pElB, SStackTrace_* stackTrace)
{
    if (!m_pOptions->isStackInfoNormal()) {
        return;
    }
    if (0 == stackTrace) {
        return;
    }
    unsigned int i;
    pElB->AddString("\n<annotation name=\"callStackDumpMethods\">");
    for (i = 0; i < stackTrace->uiSize; i++) {
        pElB->AddString("\n\t<value>");
        AppendString(pElB, stackTrace->pStackEntries[i].methodName);
        pElB->AddString("</value>");
    }
    pElB->AddString("\n</annotation>");
    pElB->AddString("\n<annotation name=\"callStackDumpLineNumbers\">");
    char buf[128];
    for (i = 0; i < stackTrace->uiSize; i++) {
        if (stackTrace->pStackEntries[i].lineNumber == -1) {
            pElB->AddString("\n\t<value>-1</value>");
        } else {
            int length = sprintf(buf, "\n\t<value>%u</value>", stackTrace->pStackEntries[i].lineNumber);
            pElB->MemCpy(buf, length);
        }
    }
    pElB->AddString("\n</annotation>");
}

/** INSERT_ELEMENT_START  ********************************************************
* Creates an XML entry tag and places it immediately following the data header
* of the provided buffer. DOES NO BOUNDS CHECKING.
* with the name specified. This function
* @param       elementName - The XML tage name
* @param elementNameLength - the length of the element name
* @param      buffer - The buffer to place the tag in (at the beginning)
* @returns        >0 - the next index in the buffer free for writing.
*                  0 -
*/
void
CPrintXML::AppendElementStart(CElementBuffer *pElB, const char *elementName,
                           size_t elementNameLength)
{ 
    pElB->AddChar('<');
    pElB->MemCpy(elementName, elementNameLength);
    return;
}

void
CPrintXML::AppendHostNameAttribute(CElementBuffer *pElB)
{
    AppendStringAttribute(pElB, HOSTNAME_ATTRIBUTE, STRLEN_HOSTNAME_ATTRIBUTE, GetHostname()); //TODO fix it! get from Agent framework
    return;
}

void 
CPrintXML::AppendIPAddressAttribute(CElementBuffer *pElB)
{
    AppendStringAttribute(pElB, IPADDRESS_ATTRIBUTE, STRLEN_IPADDRESS_ATTRIBUTE, GetIPAddress()); //TODO Fix it, get from AC
    return;
}

void
CPrintXML::AppendStringAttribute(CElementBuffer *pElB, const char *attributeName,
                                                 unsigned short attributeNameLength,
                                                 const char *in_value)
{
    pElB->AddChar(' ');
    pElB->MemCpy(attributeName, attributeNameLength);
    pElB->AddChar('=');
    pElB->AddChar('\"');
    AppendString(pElB, in_value);
    pElB->AddChar('\"');
    return;
}

void
CPrintXML::AppendString(CElementBuffer *pElB, const char *in_value)
{
    size_t valLength = in_value ? strlen(in_value) : 0;
    
    if(valLength > 0) {
        /* Strip invalid control characters from the string.  These will mess up the XML */
        const char *strippedValue = 0;
        if(StripControls((unsigned char *)in_value, (unsigned char**)&strippedValue)) {
            valLength = strlen(strippedValue);
        } else {
            strippedValue = in_value;
        }

        const char *value;
        if (Escape_for_xml(strippedValue, (char**)&value)) {    /* 50273 */
            /* we had to add escape sequences to input string */
            valLength = strlen(value);
        } else {
            /* nothing changed, use orig input, don't reset length */
            value = strippedValue;
        }

        pElB->MemCpy(value, valLength);

        if(strippedValue != in_value) {
            free((void*)strippedValue);
        }

        if (value != strippedValue) {    /* 50273 */
            /* pointing to buffer alloc'd by escape_for_xml */
            free((void*)value);
        }

    }
    return;
}

/** APPEND_INTEGER_ATTRIBUTE  **********************************************************
  */
void
CPrintXML::AppendIntegerAttribute(CElementBuffer *pElB, 
                               const char *attributeName,
                               unsigned short attributeNameLength,
                               int value)
{
#ifdef _WIN32
    pElB->AddChar(' ');
    pElB->MemCpy(attributeName, attributeNameLength);
    pElB->AddChar('=');
    pElB->AddChar('\"');
    /* We could optimize this conversion by writing assembley language
    and keepin track of the offset, instead we will convert using itoa
    and then add the length of this string to the offset */
    char buf[32];
    itoa(value, buf, 10);
    pElB->AddString(buf);
    pElB->AddChar('\"');
    return;
#else
    char buf[128];
    int length = sprintf(buf, " %s=\"%d\"", attributeName, value);
    pElB->MemCpy(buf, length);
    return;
#endif
}

/** APPEND_INTEGER_64_ATTRIBUTE  **********************************************************
 */
void
CPrintXML::AppendInteger64Attribute(CElementBuffer *pElB, 
                               const char *attributeName,
                               unsigned short attributeNameLength,
                               U64 value)
{
    char buf[128];
#ifdef _WIN32
    int length = sprintf(buf, " %s=\"%I64u\"", attributeName, value);
#else
    int length = sprintf(buf, " %s=\"%llu\"", attributeName, value);
#endif
    pElB->MemCpy(buf, length);
    return;
}


/** APPEND__LABELED_TIMESTAMP  ****************************************************
*
*/
void
CPrintXML::AppendLabeledTimestamp(CElementBuffer *pElB,
                               const char *attributeName,
                               unsigned short attributeNameLength,
                               U64 time,
                               BOOL asInterval) 
{
    U64 res_time = time;
    if (!asInterval) {
        res_time += _startTimeNanosec;
    }
    U64 quotient = res_time / 1000000000;
    U64 remainder = res_time % 1000000000;

    char buf[128];
#ifdef _WIN32
    int length = sprintf(buf, " %s=\"%I64u.%9.9I64u\"", attributeName, quotient, remainder);
#else
    int length = sprintf(buf, " %s=\"%llu.%9.9llu\"", attributeName, quotient, remainder);
#endif
    pElB->MemCpy(buf, length);
    return;
}

/** APPEND_TIMESTAMP  ****************************************************
  *
  */
void
CPrintXML::AppendTimeStamp(CElementBuffer *pElB, U64 time)
{
    AppendLabeledTimestamp(pElB, TIME_ATTRIBUTE, STRLEN_TIME_ATTRIBUTE, time, FALSE);
    return;
}

void 
CPrintXML::AppendCurrentTimeStamp(CElementBuffer *pElB)
{
    U64 timestamp = GetProfilerTime();
    AppendTimeStamp(pElB, timestamp);
    return;
}

void
CPrintXML::AppendOverhead(CElementBuffer *pElB, U64 time)
{
//    AppendLabeledTimestamp(pElB, OVERHEAD_ATTRIBUTE, STRLEN_OVERHEAD_ATTRIBUTE, time, TRUE);
    return;
}

/* Append the Thread CPU time. 

  This function appends the cumulative cpuTime into the trace. It assumes the time value
  passed in is in nanoseconds (the JVMPI method GetCurrentThreadCPUTime() returns
  values in units of nanoseconds). 

   @param buffer the buffer to write to
   @param offset the offset into the buffer where we're writing
   @param time the cumulative cpu time to put into the trace element (in nanoseconds) 
*/ 
 
void
CPrintXML::AppendThreadCPUTime(CElementBuffer *pElB, U64 time)
{

    AppendLabeledTimestamp(pElB, THREAD_CPU_TIME_ATTRIBUTE, STRLEN_THREAD_CPU_TIME_ATTRIBUTE, time, TRUE);
/*
    /* convert the time into a double value. This allows us to use our
    utility function double2string to properly output the timestamp * / 
#ifdef _WIN32
    double ts_double = ((double) (signed __int64)time) / ((double)1000000000); 
#else
    double ts_double = ((double)time) / ((double)1000000000); 
#endif 

    pElB->AddChar(' ');
    pElB->MemCpy(THREAD_CPU_TIME_ATTRIBUTE, STRLEN_THREAD_CPU_TIME_ATTRIBUTE);
    pElB->AddChar('=');
    pElB->AddChar('\"');
 /* convert the time to a string using helper method that is locale-insensitive * / 
    unsigned short offset = double2string(ts_double, pElB->buffer, pElB->offset, 9); 
    pElB->offset = offset;
    pElB->AddChar('\"');*/
    return;
}


//void
//CPrintXML::AppendString(CElementBuffer *pElB, const char *value)
//{
//    int length = strlen(value);
//    pElB->MemCpy(value, length + 1);
//    return;
//}

/** APPEND_ELEMENT_END  ********************************************************
* Creates an XML closing tag and places in at the end of the provided buffer.
* If the elementName is 0 the entire buffer is appended with the closing
* tag.  If it is non-0 the buffer is loaded with a closing tag with the
* specified name.  NOTE:  When a elementName is specified the buffer is trampled
* @param       elementName - The XML tag name, 0 for empty tag
* @param elementNameLength - length of the XML tag name, ignored if elementName=0
*/
void
CPrintXML::AppendElementEnd(CElementBuffer *pElB, const char *elementName,
                         unsigned short elementNameLength)
{
    if (elementName != 0) {
        pElB->MemCpy(elementName, elementNameLength);
    }
    pElB->MemCpy("/>", 2);
    return;
}

void
CPrintXML::AppendElementEnd2(CElementBuffer *pElB)
{
    pElB->AddChar('>');
    return;
}

void
CPrintXML::AppendClosingTag(CElementBuffer *pElB, const char *eventName,
                               unsigned short eventNameLength)
{
    pElB->MemCpy("\n</", 3);
    pElB->MemCpy(eventName, eventNameLength);
    pElB->AddChar('>');
    return;
}

/*
 * Used to start a new empty element
 */
void
CPrintXML::AppendEmptyEventHeader(CElementBuffer *pElB,
                               TId threadId,
                               const char *eventName,
                               unsigned short eventNameLength)
{
    AppendElementStart(pElB, eventName, eventNameLength);
    AppendIntegerAttribute(pElB, STATIC_THREAD_IDREF_ATTRIBUTE,
        STRLEN_STATIC_THREAD_IDREF_ATTRIBUTE, (int)(threadId));
    return;
}

/*
 * Used to start a new empty element
 */
void
CPrintXML::AppendEmptyEventHeader1(CElementBuffer *pElB,
                                   TId threadId,
                                   const char *eventName,
                                   unsigned short eventNameLength)
{
    AppendElementStart(pElB, eventName, eventNameLength);
    AppendIntegerAttribute(pElB, STATIC_THREAD_ID_ATTRIBUTE,
        STRLEN_STATIC_THREAD_ID_ATTRIBUTE, (int)(threadId));
    return;
}

void 
CPrintXML::AppendTicketAttribute(CElementBuffer *pElB, unsigned long ticket)
{
    AppendUnsignedLongAttribute(pElB, TICKET_ATTRIBUTE, STRLEN_TICKET_ATTRIBUTE, ticket);
    return;
}

void 
CPrintXML::AppendStackDepthAttribute(CElementBuffer *pElB, unsigned long stackDepth)
{
    AppendUnsignedLongAttribute(pElB, STACK_DEPTH_ATTRIBUTE, STRLEN_STACK_DEPTH_ATTRIBUTE, stackDepth);
    return;
}

//void
//CPrintXML::AppendTraceIdRefAttribute(CElementBuffer *pElB)
//{
//    if (m_pOptions->m_jvmtiAgent_Options.traceIdrefs) {
//        AppendStringAttribute(pElB, TRACE_IDREF_ATTRIBUTE, STRLEN_TRACE_IDREF_ATTRIBUTE, m_jvmtiAgent_trace_id);
//    }
//    return;
//}

void
CPrintXML::AppendUnsignedLongAttribute(CElementBuffer *pElB, const char *attributeName, 
            unsigned short attributeNameLength, unsigned long value)
{
#ifdef _WIN32
    pElB->AddChar(' ');
    pElB->MemCpy(attributeName, attributeNameLength);
    pElB->AddChar('=');
    pElB->AddChar('\"');
    /* We could optimize this conversion by writing assembley language
    and keepin track of the offset, instead we will convert using itoa
    and then add the length of this string to the offset */
    char buf[32];
    ultoa(value, buf, 10);
    pElB->AddString(buf);
    pElB->AddChar('\"');
#else
    char buf[256];
    int length = sprintf(buf, " %s=\"%lu\"", attributeName, value);
    pElB->MemCpy(buf, length);
#endif
}

//void
//CPrintXML::AppendListInteger(CElementBuffer *pElB, int value)
//{
//#ifdef _WIN32
//    /* Overwrite the previously appended space  */
//    buffer[offset-1] = ',';
//    ltoa(value, &buffer[offset], 10);
//    offset += strlen(&buffer[offset]);
//    buffer[offset++] = '\"';
//    return offset;
//#else
//    int length;
//    buffer[offset-1] = ',';
//    length = sprintf(&buffer[offset], "%d\"", value);
//    return offset + length;
//#endif
//}

/** APPEND_STRING_ATTRIBUTE  **********************************************************
  */

/* 50273
 * KC: Certain characters occurring in string attributes must be escaped
 * to make them palatabale to XML. For example, '>' better show up as
 * '&gt;'. The code below is in support of these conversions, on behalf
 * of jvmpiAgent_appendStringAttribute.
 */
#ifndef MAX
#define MAX(a,b) ((a) >= (b) ? (a) : (b))
#endif

#define MAX_ESCAPED_STRING_SIZE 7
typedef struct escape_seq
{
    char s[MAX_ESCAPED_STRING_SIZE];
    int  len;
} escape_seq_t;

/* ordering in escapees must match that in escape_str table */

const char escapees[] = "<>&\"'";
static const escape_seq_t escape_str[] =
  {
    {/* < */ "&lt;",   sizeof("&lt;") - 1},
    {/* > */ "&gt;",   sizeof("&gt;") - 1},
    {/* & */ "&amp;",  sizeof("&amp;") - 1},
    {/* " */ "&quot;", sizeof("&quot;") - 1},
    {/* ' */ "&apos;", sizeof("&apos;") - 1},
    {"", 0}
  };

/* To accomodate escape strings, we need to alloc some extra space.
 * Do it in chunks instead of one string at a time, to minimize
 * the allocation thrash. */
#define XML_ESCAPE_CHUNK_SIZE 128

typedef struct range {
    unsigned char lower; 
    unsigned char upper; 
} Range; 


/* This table of values is based on Table 3-6 "Well-formed UTF-8
Byte Sequences", available at 
http://www.unicode.org/versions/Unicode4.0.0/ch03.pdf

Entries with 0x00 as the upper part of the range are NOT part of
the expected byte sequence in a well-formed utf-8 byte sequence. 
For example, for utf8Table[1], the first character seen in the valid
byte sequence is between 0xC2 and 0xDF, and the next must be between 0x80
and 0xBF. Since utf8Table[1][2].upper is 0x00, the next character beyond this 
is considered to be part of the "next" byte sequence. 
*/ 

Range utf8Table[][4] = 
{ 
    { {0x00,0x7F},{0x00,0x00},{0x00,0x00}, {0x00,0x00} },
    { {0xC2,0xDF},{0x80,0xBF},{0x00,0x00}, {0x00,0x00} },
    { {0xE0,0xE0},{0xA0,0xBF},{0x80,0xBF}, {0x00,0x00} },
    { {0xE1,0xEC},{0x80,0xBF},{0x80,0xBF}, {0x00,0x00} },
    { {0xED,0xED},{0x80,0x9F},{0x80,0xBF}, {0x00,0x00} },
    { {0xEE,0xEF},{0x80,0xBF},{0x80,0xBF}, {0x00,0x00} },
    { {0xF0,0xF0},{0x90,0xBF},{0x80,0xBF}, {0x80,0xBF} },
    { {0xF1,0xF3},{0x80,0xBF},{0x80,0xBF}, {0x80,0xBF} },
    { {0xF4,0xF4},{0x80,0x8F},{0x80,0xBF}, {0x80,0xBF} }
}; 

/* stripControls(const char *inbuf, char **outBuf)
 
   Strip invalid UTF-8 characters. 

   This function strips invalid UTF-8 characters by verifying that the
   byte sequences within the buffer provided (inbuf) are valid according
   to the "Well-formed UTF-8 Byte Sequences" table. 

   @param inbuf the buffer to strip of invalid utf-8 characters
   @param outBuf the buffer into which to put the stripped version of inbuf
   @return outBuf if any characters were stripped from inbuf, 0 otherwise. 
*/ 
char* 
CPrintXML::StripControls(const unsigned char *inbuf, unsigned char **outBuf) 
{
    const unsigned char * end;
    unsigned char *offset=0;
    unsigned char *current=(unsigned char*)inbuf;
    end=inbuf+strlen((const char *)inbuf);
    while(current<end) {
        int idx, row = -1,     numCharsToStrip = 0, numCharsToPass = 1; 

        /* we first determine the row of the UTF-8 table that we are dealing 
        with. This will allow us to examine the byte sequence to see
        if the individual bytes fall within the expected range as laid out in 
        the table. */ 
        for (idx = 0; idx < 9; idx++) {
            if (*current >= utf8Table[idx][0].lower &&
                *current <= utf8Table[idx][0].upper) {
                    row = idx; 
                    break; 
                }
        }


        /* If we were unable to determine the row, then this character
        comprises an ill-formed byte-sequence and should be stripped */ 
        if (row == -1) {
            numCharsToStrip = 1; 
        }

        /* If we are dealing with the first row, then this is a standard ASCII
        character and we must check if there are any control characters 
        to be stripped */ 
        else if (row == 0) {
            if (*current < 0x20) {
                switch (*current) {
                    case 0x9: /* Horizontal TAB */ 
                    case 0xA: /* New line */ 
                    case 0xD: /* Carriage return */ 
                        break; /* do not strip the character */ 
                    default:
                        numCharsToStrip = 1; /* strip the character */
                        break;  
                }
            }
        }

        /* Otherwise verify that each byte in the byte-sequence
        falls within the required range as dictated by the utf-8 table */ 
        else {
            for (idx = 1; idx < 4 && current+idx < end &&  
                utf8Table[row][idx].upper != 0x00; idx++) {
                    if (*(current+idx) < utf8Table[row][idx].lower ||
                        *(current+idx) > utf8Table[row][idx].upper) {
                            /* We have detected an invalid byte in the
                            byte sequence. We can safely strip the 
                            bytes up to but not including the
                            current byte we are examining. */ 
                            numCharsToStrip = idx;
                            break;
                        }
                    else {
                        numCharsToPass++; 
                    }
                }
                /* If we fell off the end before the byte-sequence was expected
                to finish then we should strip the characters */ 
                if (current + idx >= end && idx < 4 
                    && utf8Table[row][idx].upper != 0x00) {
                        numCharsToStrip = idx; 
                    }

                    /* Code points U+FFFE and U+FFFF are not valid XML -- we need
                    the following special case to check for the UTF-8 encodings of these 
                    code points and strip them */ 
                    if (numCharsToStrip == 0 && row == 5 && idx == 3) {
                        /* check for code points U+FFFE or U+FFFF */ 
                        if ( (*current == 0xEF) && 
                            (*(current+1) == 0xBF) &&
                            ( (*(current+2) == 0xBE) || (*(current+2) == 0xBF) ) ) {
                                numCharsToStrip = 3; 
                            }
                    }
        }

        if (numCharsToStrip > 0) {
            /* if necessary create the new buffer into which we put 
            the stripped version of the original buffer */ 
            if(!offset) {
                offset=(unsigned char*)malloc(strlen((const char *)inbuf));
                memcpy(offset, inbuf, current-inbuf);
                *outBuf=offset;
                offset+=current-inbuf;
            }
            current += numCharsToStrip; 
        }
        else if(offset) {
            if (numCharsToPass == 1) {
                /* we don't bother with the overhead calling the memcpy
                function in the most frequent case */ 
                *offset++ = *current++; 
            }
            else {
                memcpy(offset,current,numCharsToPass); 
                offset += numCharsToPass; 
                current += numCharsToPass;
            }
        }
        else {
            current += numCharsToPass;
        }     
    }


    if(offset) {
        *offset='\0';
        return (char *)*outBuf;
    }
    return 0;

}

/* The basic idea is to locate and quickly copy the substrings
* within inbuf that do not require escaping, then do something more
* costly for handling the special characters. An assumption is made
* that special chars will be few and far between, so the code is
* optimized first for the case where there are no special chars
* and then for large substrings of "normal" chars. 
*/
char*
CPrintXML::Escape_for_xml(const char *inbuf, char **outbuf)
{
    size_t span;                   /* span of substr not requiring escape */
    char *op;                      /* current loc in output buffer */
    char *start_op;                /* beginning of output buffer */
    char *end_op;                  /* end of alloc'd output buffer */
    size_t out_len;                /* size of alloc'd output buffer */
    const char *ip;                /* current loc in inbuf */
    const char *end_ip;            /* ptr to terminating 0 in inbuf */
    size_t in_len = strlen(inbuf);


    /* Check the green path first */
    span = strcspn(inbuf, escapees);
    if (span == in_len) {
        /* No chars require escaping. We're outta here. */
        return 0;
    }

    /* There are one or more escapable characters. */
    ip = inbuf;
    end_ip = inbuf + in_len;

    out_len = in_len + XML_ESCAPE_CHUNK_SIZE;
    start_op = (char*) malloc(out_len);
    end_op = start_op + out_len;
    op = start_op;

    while (ip < end_ip)
    {
        int i;

        /* do we need more space in output buffer? */
        if (op + span + MAX_ESCAPED_STRING_SIZE >= end_op)
        {
            size_t used = op - start_op;

            out_len += MAX(span+MAX_ESCAPED_STRING_SIZE,
                XML_ESCAPE_CHUNK_SIZE);
            start_op = (char*) realloc(start_op, out_len);
            op = start_op + used;        /* reset curr ptr in new buffer */
            end_op = start_op + out_len;
        }

        /* handle unescaped portion with straight copy */
        if (span > 0) {
            memcpy(op, ip, span);
            op += span;
            ip += span;
        }

        /* handle char requiring escaping */
        for (i = 0; escape_str[i].len > 0; i++) {
            if ( *ip == escapees[i] ) {
                memcpy(op, escape_str[i].s, escape_str[i].len);
                op += escape_str[i].len;
                ip++;
                break;
            }
        }

        /* are there any more chars requiring escaping? */
        span = strcspn(ip, escapees);
    }
    *op = '\0';
    *outbuf = start_op;

    return start_op;
}
/*
*/

char* 
CPrintXML::ConvertClassName(const char *inbuf)
{
    if ((inbuf[strlen(inbuf) - 1] != ';') || (inbuf[0] == '[')) {
        char* str = (char*)malloc(strlen(inbuf) + 1);
        return strcpy(str, inbuf);
    } else {
        char* str = (char*)malloc(strlen(inbuf) -1);
        strncpy(str, inbuf + 1, strlen(inbuf) - 2);
        str[strlen(inbuf) - 2] = 0;
        return str;
    }
}
