/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation, Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Stanislav Polevic, Intel - Initial API and implementation
 *
 * $Id: binformat.cpp,v 1.5 2010/05/11 16:32:20 jwest Exp $ 
 ***********************************************************************/

#include "binformat.h"

	
using namespace Martini::MPI;
using namespace Martini::JPIAgent;

CFormatHeader Martini::JPIAgent::format_header(0, TBF, VERSION_MAJOR, VERSION_MINOR, PLATFORM, ENDIANNESS, DATA_OFFSET);


CBinaryMessage::CBinaryMessage(bf_short_t id):	message_id(id),
												message_len(0)
{
}

void * 
CBinaryMessage::operator>>(void *buf) 
{
	void *buffer = buf;

	buffer = write(buffer, message_id);
	buffer = write(buffer, static_cast<bf_int_t>(message_len));

	return (buffer);
}

size_t
CBinaryMessage::getHeaderLength() const
{
	return (getSize(message_id) +
			sizeof(message_len));
}

/**
 * Binary format header. First message in the stream.
 */
CFormatHeader::	CFormatHeader(bf_byte_t indent,
							bf_string_t magic,
							bf_byte_t version_major, 
							bf_byte_t version_minor, 
							bf_byte_t platform,
							bf_byte_t endianness,
							bf_int_t data_offset):
								CBinaryMessage(0),
								indent(indent),
								magic(magic),			   
								version_major(version_major),
								version_minor(version_minor),
								platform(platform),
								endianness(endianness),
								data_offset(data_offset)
{
}

void * 
CFormatHeader::operator>>(void *buf) 
{
	void *buffer = buf;

	buffer = write(buffer, indent);
	buffer = write(buffer, magic[0]);
	buffer = write(buffer, magic[1]);
	buffer = write(buffer, magic[2]);
	buffer = write(buffer, version_major);
	buffer = write(buffer, version_minor);
	buffer = write(buffer, platform);
	buffer = write(buffer, endianness);
	buffer = write(buffer, data_offset);

	return (buffer);
}

size_t 
CFormatHeader::getLength() const 
{
	return (getSize(indent) + 
			3 +
			getSize(version_major) + 
			getSize(version_minor) + 
			getSize(platform) +
			getSize(endianness) +
			getSize(data_offset));
}

size_t
CFormatHeader::getHeaderLength() const
{
	return (0);
}


// Encoding message
CEncodingMessage::CEncodingMessage(): CBinaryMessage(BF_ENCODING_ID), 
									  encoding(DEFAULT_ENCODING)
{
}

void * 
CEncodingMessage::operator>>(void *buf)
{
	void * buffer = CBinaryMessage::operator >>(buf);
	
	buffer = write(buffer, encoding);

	return (buffer);
}

size_t 
CEncodingMessage::getLength() const 
{
	return (getSize(encoding));
}


// Max qualified frequency message
CFreqMessage::CFreqMessage(): CBinaryMessage(BF_FREQ_ID), 
							  frequency(0)
{
}

void * 
CFreqMessage::operator>>(void *buf)
{
	void * buffer = CBinaryMessage::operator >>(buf);
	
	buffer = write(buffer, frequency);

	return (buffer);
}

size_t 
CFreqMessage::getLength() const 
{
	return (getSize(frequency));
}

// Node message
CNodeMessage::CNodeMessage(): CBinaryMessage(BF_NODE_ID), 
							  node_id(NULL),
							  hostname(NULL),
							  ip(NULL),
							  timezone(0),
							  timestamp(0)
{
}

void * 
CNodeMessage::operator>>(void *buf)
{
	void * buffer = CBinaryMessage::operator >>(buf);
	
	buffer = write(buffer, node_id);
	buffer = write(buffer, hostname);
	buffer = write(buffer, ip);
	buffer = write(buffer, timezone);
	buffer = write(buffer, timestamp);

	return (buffer);
}

size_t 
CNodeMessage::getLength() const 
{
	return (getSize(node_id) +
			getSize(hostname) +
			getSize(ip) +
			getSize(timezone) +
			getSize(timestamp));
}

// Process create message
CProcessCreateMessage::CProcessCreateMessage(): CBinaryMessage(BF_PROCESS_CREATE_ID),
												process_id(NULL),
												name(NULL),
												pid(0),
												node_id_ref(NULL),
												timestamp(0),
												exe(NULL)
{
}

void * 
CProcessCreateMessage::operator>>(void *buf) 
{
	void * buffer = CBinaryMessage::operator >>(buf);
	
	buffer = write(buffer, process_id);
	buffer = write(buffer, name);
	buffer = write(buffer, pid);
	buffer = write(buffer, node_id_ref);
	buffer = write(buffer, timestamp);
	buffer = write(buffer, exe);

	return (buffer);
}

size_t 
CProcessCreateMessage::getLength() const 
{
	return (getSize(process_id) +
			getSize(name) +
			getSize(pid) +
			getSize(node_id_ref) +
			getSize(timestamp) +
			getSize(exe));
}


// Agent create message
CAgentCreateMessage::CAgentCreateMessage(): CBinaryMessage(BF_AGENT_CREATE_ID),
											agent_id(NULL),
											process_id_ref(NULL),
											name(NULL),
											type(NULL),
											timestamp(0),
											params(NULL),
											version(NULL)
{
}

void * 
CAgentCreateMessage::operator>>(void *buf) 
{
	void * buffer = CBinaryMessage::operator >>(buf);

	buffer = write(buffer, agent_id);
	buffer = write(buffer, process_id_ref);
	buffer = write(buffer, name);
	buffer = write(buffer, type);
	buffer = write(buffer, timestamp);
	buffer = write(buffer, params);
	buffer = write(buffer, version);

	return (buffer);
}

size_t 
CAgentCreateMessage::getLength() const 
{
	return (getSize(agent_id) +
			getSize(process_id_ref) +
			getSize(name) +
			getSize(type) +
			getSize(timestamp) +
			getSize(params) +
			getSize(version));
}


// Agent destroy message
CAgentDestroyMessage::CAgentDestroyMessage():	CBinaryMessage(BF_AGENT_DESTROY_ID),
												agent_id_ref(NULL),
												timestamp(0)
{
}

void *
CAgentDestroyMessage::operator >>(void *buf) 
{
	void * buffer = CBinaryMessage::operator >>(buf);
	buffer = write(buffer, agent_id_ref);
	buffer = write(buffer, timestamp);

	return (buffer);
}

size_t
CAgentDestroyMessage::getLength() const 
{
	return (getSize(agent_id_ref) +
			getSize(timestamp));
}

// Trace start message
CTraceStartMessage::CTraceStartMessage():	CBinaryMessage(BF_TRACE_START_ID),
											trace_id(NULL),
											agent_id_ref(NULL),
											timestamp(0),
											col_value(NULL)
{
}

void * 
CTraceStartMessage::operator>>(void *buf) 
{
	void * buffer = CBinaryMessage::operator >>(buf);
	
	buffer = write(buffer, trace_id);
	buffer = write(buffer, agent_id_ref);
	buffer = write(buffer, timestamp);
	buffer = write(buffer, col_value);

	return (buffer);
}

size_t 
CTraceStartMessage::getLength() const 
{
	return (getSize(trace_id) +
			getSize(agent_id_ref) +
			getSize(timestamp) +
			getSize(col_value));
}


// Trace end message
CTraceEndMessage::CTraceEndMessage():	CBinaryMessage(BF_TRACE_END_ID),
										trace_id_ref(NULL),
										timestamp(0),
										col_value(NULL)
{
}

void * 
CTraceEndMessage::operator>>(void *buf) 
{
	void * buffer = CBinaryMessage::operator >>(buf);
	
	buffer = write(buffer, trace_id_ref);
	buffer = write(buffer, timestamp);
	buffer = write(buffer, col_value);

	return (buffer);
}

size_t 
CTraceEndMessage::getLength() const 
{
	return (getSize(trace_id_ref) +
			getSize(timestamp) +
			getSize(col_value));
}


// Filter message
CFilterMessage::CFilterMessage():	CBinaryMessage(BF_FILTER_ID),
									trace_id_ref(NULL),
									pattern(NULL),
									gen_pattern(NULL),
									mode(NULL),
									meth_pattern(NULL),
									meth_gen_pattern(NULL),
									meth_mode(NULL)
{
}

void * 
CFilterMessage::operator>>(void *buf) 
{
	void * buffer = CBinaryMessage::operator >>(buf);
	buffer = write(buffer, trace_id_ref);
	buffer = write(buffer, pattern);
	buffer = write(buffer, gen_pattern);
	buffer = write(buffer, mode);
	buffer = write(buffer, meth_pattern);
	buffer = write(buffer, meth_gen_pattern);
	buffer = write(buffer, meth_mode);

	return (buffer);
}

size_t 
CFilterMessage::getLength() const 
{
	return (getSize(trace_id_ref) +
			getSize(pattern) +
			getSize(gen_pattern) +
			getSize(mode) +
			getSize(meth_pattern) +
			getSize(meth_gen_pattern) +
			getSize(meth_mode));
}


// Option message
COptionMessage::COptionMessage():	CBinaryMessage(BF_OPTION_ID),
									trace_id_ref(NULL),
									key(NULL),
									value(NULL)
{
}

void * 
COptionMessage::operator>>(void *buf) 
{
	void * buffer = CBinaryMessage::operator >>(buf);
	
	buffer = write(buffer, trace_id_ref);
	buffer = write(buffer, key);
	buffer = write(buffer, value);

	return (buffer);
}

size_t 
COptionMessage::getLength() const 
{
	return (getSize(trace_id_ref) +
			getSize(key) +
			getSize(value));
}

// Thread start message
CThreadStartMessage::CThreadStartMessage():	CBinaryMessage(BF_THREAD_START_ID),
											trans_thread_id(0),
											thread_id(0),
											timestamp(0),
											group_name(NULL),
											parent_group_name(NULL),
											trans_obj_id_ref(0),
											obj_id_ref(0),
											thread_name(NULL),
											col_value(NULL),
											trace_id_ref(NULL)
{
}

void * 
CThreadStartMessage::operator>>(void *buf) 
{
	void * buffer = CBinaryMessage::operator >>(buf);
	
	buffer = write(buffer, trans_thread_id);
	buffer = write(buffer, thread_id);
	buffer = write(buffer, timestamp);
	buffer = write(buffer, group_name);
	buffer = write(buffer, parent_group_name);
	buffer = write(buffer, trans_obj_id_ref);
	buffer = write(buffer, obj_id_ref);
	buffer = write(buffer, thread_name);
	buffer = write(buffer, col_value);
	buffer = write(buffer, trace_id_ref);

	return (buffer);
}

size_t 
CThreadStartMessage::getLength() const 
{
	return (getSize(trans_thread_id) +
			getSize(thread_id) +
			getSize(timestamp) +
			getSize(group_name) +
			getSize(parent_group_name) +
			getSize(trans_obj_id_ref) +
			getSize(obj_id_ref) +
			getSize(thread_name) +
			getSize(col_value) +
			getSize(trace_id_ref));
}

// Thread end message
CThreadEndMessage::CThreadEndMessage(): CBinaryMessage(BF_THREAD_END_ID),
										trans_thread_id_ref(0),
										thread_id_ref(0),
										timestamp(0),
										col_value(NULL),
										trace_id_ref(NULL)
{
}

void * 
CThreadEndMessage::operator>>(void *buf) 
{
	void * buffer = CBinaryMessage::operator >>(buf);
	buffer = write(buffer, trans_thread_id_ref);
	buffer = write(buffer, thread_id_ref);
	buffer = write(buffer, timestamp);
	buffer = write(buffer, col_value);
	buffer = write(buffer, trace_id_ref);

	return (buffer);
}

size_t 
CThreadEndMessage::getLength() const 
{
	return (getSize(trans_thread_id_ref) +
			getSize(thread_id_ref) +
			getSize(timestamp) +
			getSize(col_value) +
			getSize(trace_id_ref));
}

// Class def message
CClassDefMessage::CClassDefMessage():	CBinaryMessage(BF_CLASS_DEF_ID),
										trans_thread_id_ref(0),
										thread_id_ref(0),
										timestamp(0),
										num_interfaces(0),
										interface_names(NULL),
										trans_class_id(0),
										class_id(0),
										source_name(NULL),
										class_loader(NULL),
										superclass(NULL),
										trans_obj_id_ref(0),
										obj_id_ref(0),
										name(NULL),
										access(NULL),
										num_static_fields(0),
										num_methods(0),
										num_instance_fields(0),
										col_value(NULL),
										trace_id_ref(NULL)
{
}

void * 
CClassDefMessage::operator>>(void *buf) 
{
	void * buffer = CBinaryMessage::operator >>(buf);
	buffer = write(buffer, trans_thread_id_ref);
	buffer = write(buffer, thread_id_ref);
	buffer = write(buffer, timestamp);
	buffer = write(buffer, num_interfaces);
	buffer = write(buffer, interface_names);
	buffer = write(buffer, trans_class_id);
	buffer = write(buffer, class_id);
	buffer = write(buffer, source_name);
	buffer = write(buffer, class_loader);
	buffer = write(buffer, superclass);
	buffer = write(buffer, trans_obj_id_ref);
	buffer = write(buffer, obj_id_ref);
	buffer = write(buffer, name);
	buffer = write(buffer, access);
	buffer = write(buffer, num_static_fields);
	buffer = write(buffer, num_methods);
	buffer = write(buffer, num_instance_fields);
	buffer = write(buffer, col_value);
	buffer = write(buffer, trace_id_ref);

	return (buffer);
}

size_t 
CClassDefMessage::getLength() const 
{
	return (getSize(trans_thread_id_ref) +
			getSize(thread_id_ref) +
			getSize(timestamp) +
			getSize(num_interfaces) +
			getSize(interface_names) +
			getSize(trans_class_id) +
			getSize(class_id) +
			getSize(source_name) +
			getSize(class_loader) +
			getSize(superclass) +
			getSize(trans_obj_id_ref) +
			getSize(obj_id_ref) +
			getSize(name) +
			getSize(access) +
			getSize(num_static_fields) +
			getSize(num_methods) +
			getSize(num_instance_fields) +
			getSize(col_value) +
			getSize(trace_id_ref));
}

// Method def message
CMethodDefMessage::CMethodDefMessage():	CBinaryMessage(BF_METHOD_DEF_ID),
										name(NULL),
										sig(NULL),
										visibility(NULL),
										is_native(0),
										is_abstract(0),
										is_static(0),
										is_sync(0),
										exceptions(NULL),
										start_line(0),
										end_line(0),
										sig_notation(NULL),
										trans_class_id_ref(0),
										class_id_ref(0),
										method_id(0),
										col_value(NULL),
										trace_id_ref(NULL)
{
}

void * 
CMethodDefMessage::operator>>(void *buf) 
{
	void * buffer = CBinaryMessage::operator >>(buf);
	buffer = write(buffer, name);
	buffer = write(buffer, sig);
	buffer = write(buffer, visibility);
	buffer = write(buffer, is_native);
	buffer = write(buffer, is_abstract);
	buffer = write(buffer, is_static);
	buffer = write(buffer, is_sync);
	buffer = write(buffer, exceptions);
	buffer = write(buffer, start_line);
	buffer = write(buffer, end_line);
	buffer = write(buffer, sig_notation);
	buffer = write(buffer, trans_class_id_ref);
	buffer = write(buffer, class_id_ref);
	buffer = write(buffer, method_id);
	buffer = write(buffer, col_value);
	buffer = write(buffer, trace_id_ref);

	return (buffer);
}

size_t 
CMethodDefMessage::getLength() const 
{
	return (getSize(name) +
			getSize(sig) +
			getSize(visibility) +
			getSize(is_native) +
			getSize(is_abstract) +
			getSize(is_static) +
			getSize(is_sync) +
			getSize(exceptions) +
			getSize(start_line) +
			getSize(end_line) +
			getSize(sig_notation) +
			getSize(trans_class_id_ref) +
			getSize(class_id_ref) +
			getSize(method_id) +
			getSize(col_value) +
			getSize(trace_id_ref));
}


// Obj alloc message
CObjAllocMessage::CObjAllocMessage():	CBinaryMessage(BF_OBJ_ALLOC_ID),
										trans_thread_id_ref(0),
										thread_id_ref(0),
										timestamp(0),
										is_array(0),
										trans_obj_id_ref(0),
										obj_id_ref(0),
										size(0),
										line(0),
										method_id(0),
										trans_class_id_ref(0),
										class_id_ref(0),
										context_data(NULL),
										col_value(NULL),
										trace_id_ref(NULL)
{
}

void * 
CObjAllocMessage::operator>>(void *buf) 
{
	void * buffer = CBinaryMessage::operator >>(buf);
	buffer = write(buffer, trans_thread_id_ref);
	buffer = write(buffer, thread_id_ref);
	buffer = write(buffer, timestamp);
	buffer = write(buffer, is_array);
	buffer = write(buffer, trans_obj_id_ref);
	buffer = write(buffer, obj_id_ref);
	buffer = write(buffer, size);
	buffer = write(buffer, line);
	buffer = write(buffer, method_id);
	buffer = write(buffer, trans_class_id_ref);
	buffer = write(buffer, class_id_ref);
	buffer = write(buffer, context_data);
	buffer = write(buffer, col_value);
	buffer = write(buffer, trace_id_ref);

	return (buffer);
}

size_t 
CObjAllocMessage::getLength() const 
{
	return (getSize(trans_thread_id_ref) +
			getSize(thread_id_ref) +
			getSize(timestamp) +
			getSize(is_array) +
			getSize(trans_obj_id_ref) +
			getSize(obj_id_ref) +
			getSize(size) +
			getSize(line) +
			getSize(method_id) +
			getSize(trans_class_id_ref) +
			getSize(class_id_ref) +
			getSize(context_data) +
			getSize(col_value) +
			getSize(trace_id_ref));
}

// Method entry message
CMethodEntryMessage::CMethodEntryMessage():	CBinaryMessage(BF_METHOD_ENTRY_ID),
										trans_thread_id_ref(0),
										thread_id_ref(0),
										timestamp(0),
										method_id_ref(0),
										ticket(0),
										trans_obj_id_ref(0),
										class_id_ref(0),
										thread_cpu_time(0),
										seq_counter(0),
										stack_depth(0),
										col_value(NULL),
										trace_id_ref(NULL)
{
}

void * 
CMethodEntryMessage::operator>>(void *buf) 
{
	void * buffer = CBinaryMessage::operator >>(buf);
	buffer = write(buffer, trans_thread_id_ref);
	buffer = write(buffer, thread_id_ref);
	buffer = write(buffer, timestamp);
	buffer = write(buffer, method_id_ref);
	buffer = write(buffer, ticket);
	buffer = write(buffer, trans_obj_id_ref);
	buffer = write(buffer, class_id_ref);
	buffer = write(buffer, thread_cpu_time);
	buffer = write(buffer, seq_counter);
	buffer = write(buffer, stack_depth);
	buffer = write(buffer, col_value);
	buffer = write(buffer, trace_id_ref);

	return (buffer);
}

size_t 
CMethodEntryMessage::getLength() const 
{
	return (getSize(trans_thread_id_ref) +
			getSize(thread_id_ref) +
			getSize(timestamp) +
			getSize(method_id_ref) +
			getSize(ticket) +
			getSize(trans_obj_id_ref) +
			getSize(class_id_ref) +
			getSize(thread_cpu_time) +
			getSize(seq_counter) +
			getSize(stack_depth) +
			getSize(col_value) +
			getSize(trace_id_ref));
}

// Method Exit message
CMethodExitMessage::CMethodExitMessage():	CBinaryMessage(BF_METHOD_EXIT_ID),
										trans_thread_id_ref(0),
										thread_id_ref(0),
										timestamp(0),
										ticket(0),
										thread_cpu_time(0),
										method_id_ref(0),
										trans_obj_id_ref(0),
										obj_id_ref(0),
										trans_class_id_ref(0),
										class_id_ref(0),
										seq_counter(0),
										col_value(NULL),
										trace_id_ref(NULL)
{
}

void * 
CMethodExitMessage::operator>>(void *buf) 
{
	void * buffer = CBinaryMessage::operator >>(buf);
	buffer = write(buffer, trans_thread_id_ref);
	buffer = write(buffer, thread_id_ref);
	buffer = write(buffer, timestamp);
	buffer = write(buffer, ticket);
	buffer = write(buffer, thread_cpu_time);
	buffer = write(buffer, method_id_ref);
	buffer = write(buffer, trans_obj_id_ref);
	buffer = write(buffer, obj_id_ref);
	buffer = write(buffer, trans_class_id_ref);
	buffer = write(buffer, class_id_ref);
	buffer = write(buffer, seq_counter);
	buffer = write(buffer, col_value);
	buffer = write(buffer, trace_id_ref);

	return (buffer);
}

size_t 
CMethodExitMessage::getLength() const 
{
	return (getSize(trans_thread_id_ref) +
			getSize(thread_id_ref) +
			getSize(timestamp) +
			getSize(ticket) +
			getSize(thread_cpu_time) +
			getSize(method_id_ref) +
			getSize(trans_obj_id_ref) +
			getSize(obj_id_ref) +
			getSize(trans_class_id_ref) +
			getSize(class_id_ref) +
			getSize(seq_counter) +
			getSize(col_value) +
			getSize(trace_id_ref));
}

// Invocation context message
CInvocationContextMessage::CInvocationContextMessage():	CBinaryMessage(BF_INVOCATION_CONTEXT_ID),
										node_id_ref(0),
										process_id_ref(0),
										agent_id_ref(0),
										thread_id_ref(0),
										ticket(0),
										seq_counter(0),
										col_value(NULL),
										trace_id_ref(NULL)
{
}

void * 
CInvocationContextMessage::operator>>(void *buf) 
{
	void * buffer = CBinaryMessage::operator >>(buf);
	buffer = write(buffer, node_id_ref);
	buffer = write(buffer, process_id_ref);
	buffer = write(buffer, agent_id_ref);
	buffer = write(buffer, thread_id_ref);
	buffer = write(buffer, ticket);
	buffer = write(buffer, seq_counter);
	buffer = write(buffer, col_value);
	buffer = write(buffer, trace_id_ref);

	return (buffer);
}

size_t 
CInvocationContextMessage::getLength() const 
{
	return (getSize(node_id_ref) +
			getSize(process_id_ref) +
			getSize(agent_id_ref) +
			getSize(thread_id_ref) +
			getSize(ticket) +
			getSize(seq_counter) +
			getSize(col_value) +
			getSize(trace_id_ref));
}

// Obj def message
CObjDefMessage::CObjDefMessage():	CBinaryMessage(BF_OBJ_DEF_ID),
										obj_id(0),
										is_array(0),
										size(0),
										class_id_ref(0),
										col_value(NULL),
										trace_id_ref(NULL)
{
}

void * 
CObjDefMessage::operator>>(void *buf) 
{
	void * buffer = CBinaryMessage::operator >>(buf);
	buffer = write(buffer, obj_id);
	buffer = write(buffer, is_array);
	buffer = write(buffer, size);
	buffer = write(buffer, class_id_ref);
	buffer = write(buffer, col_value);
	buffer = write(buffer, trace_id_ref);

	return (buffer);
}

size_t 
CObjDefMessage::getLength() const 
{
	return (getSize(obj_id) +
			getSize(is_array) +
			getSize(size) +
			getSize(class_id_ref) +
			getSize(col_value) +
			getSize(trace_id_ref));
}

// Value message
CValueMessage::CValueMessage():	CBinaryMessage(BF_VALUE_ID),
										name(NULL),
										obj_id_ref(0),
										ser_format(NULL),
										type(NULL),
										value(NULL),
										col_value(NULL),
										trace_id_ref(NULL)
{
}

void * 
CValueMessage::operator>>(void *buf) 
{
	void * buffer = CBinaryMessage::operator >>(buf);
	buffer = write(buffer, name);
	buffer = write(buffer, obj_id_ref);
	buffer = write(buffer, ser_format);
	buffer = write(buffer, type);
	buffer = write(buffer, value);
	buffer = write(buffer, col_value);
	buffer = write(buffer, trace_id_ref);

	return (buffer);
}

size_t 
CValueMessage::getLength() const 
{
	return (getSize(name) +
			getSize(obj_id_ref) +
			getSize(ser_format) +
			getSize(type) +
			getSize(value) +
			getSize(col_value) +
			getSize(trace_id_ref));
}

// Method count message
CMethodCountMessage::CMethodCountMessage():	CBinaryMessage(BF_METHOD_COUNT_ID),
										count(0),
										method_id_ref(0),
										col_value(NULL),
										trace_id_ref(NULL)
{
}

void * 
CMethodCountMessage::operator>>(void *buf) 
{
	void * buffer = CBinaryMessage::operator >>(buf);
	buffer = write(buffer, count);
	buffer = write(buffer, method_id_ref);
	buffer = write(buffer, col_value);
	buffer = write(buffer, trace_id_ref);

	return (buffer);
}

size_t 
CMethodCountMessage::getLength() const 
{
	return (getSize(count) +
			getSize(method_id_ref) +
			getSize(col_value) +
			getSize(trace_id_ref));
}

// Line message
CLineMessage::CLineMessage():	CBinaryMessage(BF_LINE_ID),
										trans_thread_id_ref(0),
										thread_id(0),
										timestamp(0),
										line(0),
										file(NULL),
										method_id_ref(0),
										ticket(0),
										thread_cpu_time(0),
										col_value(NULL),
										trace_id_ref(NULL)
{
}

void * 
CLineMessage::operator>>(void *buf) 
{
	void * buffer = CBinaryMessage::operator >>(buf);
	buffer = write(buffer, trans_thread_id_ref);
	buffer = write(buffer, thread_id);
	buffer = write(buffer, timestamp);
	buffer = write(buffer, line);
	buffer = write(buffer, file);
	buffer = write(buffer, method_id_ref);
	buffer = write(buffer, thread_cpu_time);
	buffer = write(buffer, col_value);
	buffer = write(buffer, trace_id_ref);

	return (buffer);
}

size_t 
CLineMessage::getLength() const 
{
	return (getSize(trans_thread_id_ref) +
			getSize(thread_id) +
			getSize(timestamp) +
			getSize(line) +
			getSize(file) +
			getSize(method_id_ref) +
			getSize(thread_cpu_time) +
			getSize(col_value) +
			getSize(trace_id_ref));
}

// GC start message
CGCStartMessage::CGCStartMessage():	CBinaryMessage(BF_GC_START_ID),
										trans_thread_id_ref(0),
										thread_id_ref(0),
										timestamp(0),
										col_value(NULL),
										trace_id_ref(NULL)
{
}

void * 
CGCStartMessage::operator>>(void *buf) 
{
	void * buffer = CBinaryMessage::operator >>(buf);
	buffer = write(buffer, trans_thread_id_ref);
	buffer = write(buffer, thread_id_ref);
	buffer = write(buffer, timestamp);
	buffer = write(buffer, col_value);
	buffer = write(buffer, trace_id_ref);

	return (buffer);
}

size_t 
CGCStartMessage::getLength() const 
{
	return (getSize(trans_thread_id_ref) +
			getSize(thread_id_ref) +
			getSize(timestamp) +
			getSize(col_value) +
			getSize(trace_id_ref));
}

// Obj Free message
CObjFreeMessage::CObjFreeMessage():	CBinaryMessage(BF_OBJ_FREE_ID),
										trans_thread_id_ref(0),
										thread_id_ref(0),
										timestamp(0),
										trans_obj_id_ref(0),
										obj_id_ref(0),
										obj_age(0),
										seq_counter(0),
										stack_depth(0),
										col_value(NULL),
										trace_id_ref(NULL)
{
}

void * 
CObjFreeMessage::operator>>(void *buf) 
{
	void * buffer = CBinaryMessage::operator >>(buf);
	buffer = write(buffer, trans_thread_id_ref);
	buffer = write(buffer, thread_id_ref);
	buffer = write(buffer, timestamp);
	buffer = write(buffer, trans_obj_id_ref);
	buffer = write(buffer, obj_id_ref);
	buffer = write(buffer, obj_age);
	buffer = write(buffer, seq_counter);
	buffer = write(buffer, stack_depth);
	buffer = write(buffer, col_value);
	buffer = write(buffer, trace_id_ref);

	return (buffer);
}

size_t 
CObjFreeMessage::getLength() const 
{
	return (getSize(trans_thread_id_ref) +
			getSize(thread_id_ref) +
			getSize(timestamp) +
			getSize(trans_obj_id_ref) +
			getSize(obj_id_ref) +
			getSize(obj_age) +
			getSize(seq_counter) +
			getSize(stack_depth) +
			getSize(col_value) +
			getSize(trace_id_ref));
}

// Class unload message
CClassUnloadMessage::CClassUnloadMessage():	CBinaryMessage(BF_CLASS_UNLOAD_ID),
										trans_thread_id_ref(0),
										thread_id_ref(0),
										timestamp(0),
										trans_class_id_ref(0),
										class_id_ref(0),
										col_value(NULL),
										trace_id_ref(NULL)
{
}

void * 
CClassUnloadMessage::operator>>(void *buf) 
{
	void * buffer = CBinaryMessage::operator >>(buf);
	buffer = write(buffer, trans_thread_id_ref);
	buffer = write(buffer, thread_id_ref);
	buffer = write(buffer, timestamp);
	buffer = write(buffer, trans_class_id_ref);
	buffer = write(buffer, class_id_ref);
	buffer = write(buffer, col_value);
	buffer = write(buffer, trace_id_ref);

	return (buffer);
}

size_t 
CClassUnloadMessage::getLength() const 
{
	return (getSize(trans_thread_id_ref) +
			getSize(thread_id_ref) +
			getSize(timestamp) +
			getSize(trans_class_id_ref) +
			getSize(class_id_ref) +
			getSize(col_value) +
			getSize(trace_id_ref));
}

// Obj move message
CObjMoveMessage::CObjMoveMessage():	CBinaryMessage(BF_OBJ_MOVE_ID),
										trans_thread_id_ref(0),
										thread_id_ref(0),
										timestamp(0),
										trans_obj_id_ref(0),
										obj_id_ref(0),
										new_obj_id(0),
										col_value(NULL),
										trace_id_ref(NULL)
{
}

void * 
CObjMoveMessage::operator>>(void *buf) 
{
	void * buffer = CBinaryMessage::operator >>(buf);
	buffer = write(buffer, trans_thread_id_ref);
	buffer = write(buffer, thread_id_ref);
	buffer = write(buffer, timestamp);
	buffer = write(buffer, trans_obj_id_ref);
	buffer = write(buffer, obj_id_ref);
	buffer = write(buffer, new_obj_id);
	buffer = write(buffer, col_value);
	buffer = write(buffer, trace_id_ref);

	return (buffer);
}

size_t 
CObjMoveMessage::getLength() const 
{
	return (getSize(trans_thread_id_ref) +
			getSize(thread_id_ref) +
			getSize(timestamp) +
			getSize(trans_obj_id_ref) +
			getSize(obj_id_ref) +
			getSize(new_obj_id) +
			getSize(col_value) +
			getSize(trace_id_ref));
}

// GC finish message
CGCFinishMessage::CGCFinishMessage():	CBinaryMessage(BF_GC_FINISH_ID),
										trans_thread_id_ref(0),
										thread_id_ref(0),
										timestamp(0),
										total_object_space(0),
										used_object_space(0),
										used_objects(0),
										col_value(NULL),
										trace_id_ref(NULL)
{
}

void * 
CGCFinishMessage::operator>>(void *buf) 
{
	void * buffer = CBinaryMessage::operator >>(buf);
	buffer = write(buffer, trans_thread_id_ref);
	buffer = write(buffer, thread_id_ref);
	buffer = write(buffer, timestamp);
	buffer = write(buffer, total_object_space);
	buffer = write(buffer, used_object_space);
	buffer = write(buffer, used_objects);
	buffer = write(buffer, col_value);
	buffer = write(buffer, trace_id_ref);

	return (buffer);
}

size_t 
CGCFinishMessage::getLength() const 
{
	return (getSize(trans_thread_id_ref) +
			getSize(thread_id_ref) +
			getSize(timestamp) +
			getSize(total_object_space) +
			getSize(used_object_space) +
			getSize(used_objects) +
			getSize(col_value) +
			getSize(trace_id_ref));
}

// Throw message
CThrowMessage::CThrowMessage():	CBinaryMessage(BF_THROW_ID),
										trans_thread_id_ref(0),
										thread_id_ref(0),
										timestamp(0),
										obj_id_ref(0),
										obj_handle(0),
										method_id_ref(0),
										ticket(0),
										col_value(NULL),
										trace_id_ref(NULL)
{
}

void * 
CThrowMessage::operator>>(void *buf) 
{
	void * buffer = CBinaryMessage::operator >>(buf);
	buffer = write(buffer, trans_thread_id_ref);
	buffer = write(buffer, thread_id_ref);
	buffer = write(buffer, timestamp);
	buffer = write(buffer, obj_id_ref);
	buffer = write(buffer, obj_handle);
	buffer = write(buffer, method_id_ref);
	buffer = write(buffer, ticket);
	buffer = write(buffer, col_value);
	buffer = write(buffer, trace_id_ref);

	return (buffer);
}

size_t 
CThrowMessage::getLength() const 
{
	return (getSize(trans_thread_id_ref) +
			getSize(thread_id_ref) +
			getSize(timestamp) +
			getSize(obj_id_ref) +
			getSize(obj_handle) +
			getSize(method_id_ref) +
			getSize(ticket) +
			getSize(col_value) +
			getSize(trace_id_ref));
}

// Catch message
CCatchMessage::CCatchMessage():	CBinaryMessage(BF_CATCH_ID),
										trans_thread_id_ref(0),
										thread_id_ref(0),
										timestamp(0),
										obj_id_ref(0),
										trans_obj_id_ref(0),
										obj_handle(0),
										method_id_ref(0),
										ticket(0),
										col_value(NULL),
										trace_id_ref(NULL)
{
}

void * 
CCatchMessage::operator>>(void *buf) 
{
	void * buffer = CBinaryMessage::operator >>(buf);
	buffer = write(buffer, trans_thread_id_ref);
	buffer = write(buffer, thread_id_ref);
	buffer = write(buffer, timestamp);
	buffer = write(buffer, obj_id_ref);
	buffer = write(buffer, trans_obj_id_ref);
	buffer = write(buffer, obj_handle);
	buffer = write(buffer, method_id_ref);
	buffer = write(buffer, ticket);
	buffer = write(buffer, col_value);
	buffer = write(buffer, trace_id_ref);

	return (buffer);
}

size_t 
CCatchMessage::getLength() const 
{
	return (getSize(trans_thread_id_ref) +
			getSize(thread_id_ref) +
			getSize(timestamp) +
			getSize(obj_id_ref) +
			getSize(trans_obj_id_ref) +
			getSize(obj_handle) +
			getSize(method_id_ref) +
			getSize(ticket) +
			getSize(col_value) +
			getSize(trace_id_ref));
}

// Runtime init done message
CRuntimeInitDoneMessage::CRuntimeInitDoneMessage(): CBinaryMessage(BF_RUNTIME_INIT_DONE_ID),
													trans_thread_id_ref(0),
													thread_id_ref(0),
													timestamp(0),
													col_value(NULL),
													trace_id_ref(NULL)
{
}

void * 
CRuntimeInitDoneMessage::operator>>(void *buf) 
{
	void * buffer = CBinaryMessage::operator >>(buf);
	buffer = write(buffer, trans_thread_id_ref);
	buffer = write(buffer, thread_id_ref);
	buffer = write(buffer, timestamp);
	buffer = write(buffer, col_value);
	buffer = write(buffer, trace_id_ref);

	return (buffer);
}

size_t 
CRuntimeInitDoneMessage::getLength() const 
{
	return (getSize(trans_thread_id_ref) +
			getSize(thread_id_ref) +
			getSize(timestamp) +
			getSize(col_value) +
			getSize(trace_id_ref));
}


// Runtime shutdown message
CRuntimeShutdownMessage::CRuntimeShutdownMessage(): CBinaryMessage(BF_RUNTIME_SHUTDOWN_ID),
													trans_thread_id_ref(0),
													thread_id_ref(0),
													timestamp(0),
													col_value(NULL),
													trace_id_ref(NULL)
{
}

void * 
CRuntimeShutdownMessage::operator>>(void *buf) 
{
	void * buffer = CBinaryMessage::operator >>(buf);
	buffer = write(buffer, trans_thread_id_ref);
	buffer = write(buffer, thread_id_ref);
	buffer = write(buffer, timestamp);
	buffer = write(buffer, col_value);
	buffer = write(buffer, trace_id_ref);

	return (buffer);
}

size_t 
CRuntimeShutdownMessage::getLength() const 
{
	return (getSize(trans_thread_id_ref) +
			getSize(thread_id_ref) +
			getSize(timestamp) +
			getSize(col_value) +
			getSize(trace_id_ref));
}

// Monitor Wait message
CMonWaitMessage::CMonWaitMessage(bf_short_t message_id):CBinaryMessage(message_id),
														thread_id_ref(0),
														timestamp(0),
														obj_id_ref(0),
														timeout(0),
														stack_depth(0),
														stack_methods(NULL),
														stack_lines(NULL)
{
}

void * 
CMonWaitMessage::operator>>(void *buf) 
{
	void * buffer = CBinaryMessage::operator >>(buf);
	buffer = write(buffer, thread_id_ref);
	buffer = write(buffer, timestamp);
	buffer = write(buffer, obj_id_ref);
	buffer = write(buffer, timeout);
	buffer = write(buffer, static_cast<bf_int_t>(stack_depth));
	buffer = write(buffer, stack_methods, stack_depth);
	buffer = write(buffer, stack_lines, stack_depth);

	return (buffer);
}

size_t 
CMonWaitMessage::getLength() const 
{
	return (getSize(thread_id_ref) +
			getSize(timestamp) +
			getSize(obj_id_ref) +
			getSize(timeout) +
			getSize(static_cast<bf_int_t>(stack_depth))) +
			getSize(stack_methods, stack_depth) +
			getSize(stack_lines, stack_depth);
}

// Monitor Waited message
CMonWaitedMessage::CMonWaitedMessage():CMonWaitMessage(BF_MON_WAITED_ID)
{
}

// Monitor Contended Enter message
CMonContendedEnterMessage::CMonContendedEnterMessage():	CBinaryMessage(BF_MON_CONTENDED_ENTER_ID),
														thread_id_ref(0),
														timestamp(0),
														obj_id_ref(0),
														thread_owner_id_ref(0),
														stack_depth(0),
														stack_methods(NULL),
														stack_lines(NULL)
{
}

void * 
CMonContendedEnterMessage::operator>>(void *buf) 
{
	void * buffer = CBinaryMessage::operator >>(buf);
	buffer = write(buffer, thread_id_ref);
	buffer = write(buffer, timestamp);
	buffer = write(buffer, obj_id_ref);
	buffer = write(buffer, thread_owner_id_ref);
	buffer = write(buffer, static_cast<bf_int_t>(stack_depth));
	buffer = write(buffer, stack_methods, stack_depth);
	buffer = write(buffer, stack_lines, stack_depth);

	return (buffer);
}

size_t 
CMonContendedEnterMessage::getLength() const 
{
	return (getSize(thread_id_ref) +
			getSize(timestamp) +
			getSize(obj_id_ref) +
			getSize(thread_owner_id_ref) +
			getSize(static_cast<bf_int_t>(stack_depth))) +
			getSize(stack_methods, stack_depth) +
			getSize(stack_lines, stack_depth);
}

// Monitor Contended Entered message
CMonContendedEnteredMessage::CMonContendedEnteredMessage():	CBinaryMessage(BF_MON_CONTENDED_ENTERED_ID),
														thread_id_ref(0),
														timestamp(0),
														obj_id_ref(0),
														stack_depth(0),
														stack_methods(NULL),
														stack_lines(NULL)
{
}

void * 
CMonContendedEnteredMessage::operator>>(void *buf) 
{
	void * buffer = CBinaryMessage::operator >>(buf);
	buffer = write(buffer, thread_id_ref);
	buffer = write(buffer, timestamp);
	buffer = write(buffer, obj_id_ref);
	buffer = write(buffer, static_cast<bf_int_t>(stack_depth));
	buffer = write(buffer, stack_methods, stack_depth);
	buffer = write(buffer, stack_lines, stack_depth);

	return (buffer);
}

size_t 
CMonContendedEnteredMessage::getLength() const 
{
	return (getSize(thread_id_ref) +
			getSize(timestamp) +
			getSize(obj_id_ref) +
			getSize(static_cast<bf_int_t>(stack_depth))) +
			getSize(stack_methods, stack_depth) +
			getSize(stack_lines, stack_depth);
}


// Aggregate method entry message
CAGMethodEntryMessage::CAGMethodEntryMessage():	CBinaryMessage(BF_AG_METHOD_ENTRY_ID),
												thread_id_ref(0),
												method_id_ref(0),
												base_time(0),
												min_time(0),
												max_time(0),
												base_cpu_time(0),
												num_calls(0)
{
}

void * 
CAGMethodEntryMessage::operator>>(void *buf) 
{
	void * buffer = CBinaryMessage::operator >>(buf);
	buffer = write(buffer, thread_id_ref);
	buffer = write(buffer, method_id_ref);
	buffer = write(buffer, base_time);
	buffer = write(buffer, min_time);
	buffer = write(buffer, max_time);
	buffer = write(buffer, base_cpu_time);
	buffer = write(buffer, num_calls);

	return (buffer);
}

size_t 
CAGMethodEntryMessage::getLength() const 
{
	return (getSize(thread_id_ref) +
			getSize(method_id_ref) +
			getSize(base_time) +
			getSize(min_time) +
			getSize(max_time) +
			getSize(base_cpu_time) +
			getSize(num_calls));
}

// Aggregate method exit message
CAGMethodExitMessage::CAGMethodExitMessage():	CBinaryMessage(BF_AG_METHOD_EXIT_ID),
												thread_id_ref(0),
												method_id_ref(0)
{
}

void * 
CAGMethodExitMessage::operator>>(void *buf) 
{
	void * buffer = CBinaryMessage::operator >>(buf);
	buffer = write(buffer, thread_id_ref);
	buffer = write(buffer, method_id_ref);

	return (buffer);
}

size_t 
CAGMethodExitMessage::getLength() const 
{
	return (getSize(thread_id_ref) +
			getSize(method_id_ref));
}

// Heap dump start message
CHDStartMessage::CHDStartMessage():	CBinaryMessage(BF_HEAP_DUMP_DEF_ID),
									heap_dump_id_ref(0),
									timestamp(0),
									name(NULL),
									base_time(0)
{
}

void * 
CHDStartMessage::operator>>(void *buf) 
{
	void * buffer = CBinaryMessage::operator >>(buf);
	buffer = write(buffer, heap_dump_id_ref);
	buffer = write(buffer, timestamp);
	buffer = write(buffer, name);
	buffer = write(buffer, base_time);

	return (buffer);
}

size_t 
CHDStartMessage::getLength() const 
{
	return (getSize(heap_dump_id_ref) +
			getSize(timestamp) +
			getSize(name) +
			getSize(base_time));
}

// GC root message
CGCRootMessage::CGCRootMessage():	CBinaryMessage(BF_GC_ROOT_ID),
									class_id_ref(0),
									obj_id_ref(0),
									type(NULL)
{
}

void * 
CGCRootMessage::operator>>(void *buf) 
{
	void * buffer = CBinaryMessage::operator >>(buf);
	buffer = write(buffer, class_id_ref);
	buffer = write(buffer, obj_id_ref);
	buffer = write(buffer, type);

	return (buffer);
}

size_t 
CGCRootMessage::getLength() const 
{
	return (getSize(class_id_ref) +
			getSize(obj_id_ref) +
			getSize(type));
}

// Object reference message
CObjRefMessage::CObjRefMessage():	CBinaryMessage(BF_OBJ_REF_ID),
									src_id_ref(0),
									target_id_ref(0),
									heap_dump_id_ref(0)
{
}

void * 
CObjRefMessage::operator>>(void *buf) 
{
	void * buffer = CBinaryMessage::operator >>(buf);
	buffer = write(buffer, src_id_ref);
	buffer = write(buffer, target_id_ref);
	buffer = write(buffer, heap_dump_id_ref);

	return (buffer);
}

size_t 
CObjRefMessage::getLength() const 
{
	return (getSize(src_id_ref) +
			getSize(target_id_ref) +
			getSize(heap_dump_id_ref));
}

// Custom message
CCustomMessage::CCustomMessage():	CBinaryMessage(BF_CUSTOM_ID),
									body(NULL)
{
}

void * 
CCustomMessage::operator>>(void *buf) 
{
	void * buffer = CBinaryMessage::operator >>(buf);
	buffer = write(buffer, body);

	return (buffer);
}

size_t 
CCustomMessage::getLength() const 
{
	return (getSize(body));
}

// Call ID notify message
CCallIdNotifyMessage::CCallIdNotifyMessage():	CBinaryMessage(BF_CALL_ID_NOTIFY_ID),
														thread_id_ref(0),
														timestamp(0),
														static_obj_id_ref(0),
														notify_all(0),
														stack_depth(0),
														stack_methods(NULL),
														stack_lines(NULL)
{
}

CCallIdNotifyMessage::CCallIdNotifyMessage(enum bf_message_ids msg_id):
														CBinaryMessage(msg_id),
														thread_id_ref(0),
														timestamp(0),
														static_obj_id_ref(0),
														notify_all(0),
														stack_depth(0),
														stack_methods(NULL),
														stack_lines(NULL)
{
}

void * 
CCallIdNotifyMessage::operator>>(void *buf) 
{
	void * buffer = CBinaryMessage::operator >>(buf);
	buffer = write(buffer, thread_id_ref);
	buffer = write(buffer, timestamp);
	buffer = write(buffer, static_obj_id_ref);
	buffer = write(buffer, notify_all);
	buffer = write(buffer, static_cast<bf_int_t>(stack_depth));
	buffer = write(buffer, stack_methods, stack_depth);
	buffer = write(buffer, stack_lines, stack_depth);

	return (buffer);
}

size_t 
CCallIdNotifyMessage::getLength() const 
{
	return (getSize(thread_id_ref) +
			getSize(timestamp) +
			getSize(static_obj_id_ref) +
			getSize(notify_all) +
			getSize(static_cast<bf_int_t>(stack_depth))) +
			getSize(stack_methods, stack_depth) +
			getSize(stack_lines, stack_depth);
}



// Call Id interrupt
CCallIdInterruptMessage::CCallIdInterruptMessage(): CCallIdNotifyMessage(BF_CALL_ID_INTERRUPT_ID)
{
}

// Call Id start
CCallIdStartMessage::CCallIdStartMessage(): CCallIdNotifyMessage(BF_CALL_ID_START_ID)
{
}

// Call ID sleep start
CCallIdThreadSleepStartMessage::CCallIdThreadSleepStartMessage():  CCallIdNotifyMessage(BF_CALL_ID_SLEEP_START_ID)
{
}

// Call ID sleep end
CCallIdThreadSleepEndMessage::CCallIdThreadSleepEndMessage() : CCallIdNotifyMessage(BF_CALL_ID_SLEEP_END_ID)
{
}


void *
CCallIdThreadSleepStartMessage::operator>>(void *buf)
{
	void * buffer = CBinaryMessage::operator >>(buf);
	buffer = write(buffer, thread_id_ref);
	buffer = write(buffer, timestamp);

	return (buffer);
}

size_t
CCallIdThreadSleepStartMessage::getLength() const
{
	return (getSize(thread_id_ref) +
			getSize(timestamp));
}

void *
CCallIdThreadSleepEndMessage::operator>>(void *buf)
{
	void * buffer = CBinaryMessage::operator >>(buf);
	buffer = write(buffer, thread_id_ref);
	buffer = write(buffer, timestamp);

	return (buffer);
}

size_t
CCallIdThreadSleepEndMessage::getLength() const
{
	return (getSize(thread_id_ref) +
			getSize(timestamp));
}
