/*******************************************************************************
 * Copyright (c) 2008, 2010 Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Stanislav Polevic, Intel - Initial API and implementation
 *
 * $Id: binformat.h,v 1.11 2010/05/11 16:32:19 jwest Exp $
 ***********************************************************************/

#ifndef __BINARY_FORMAT_H__
#define __BINARY_FORMAT_H__

#include <string.h>
#include "Print.h"

namespace Martini { namespace JPIAgent {

#define VERSION_MAJOR 1
#define VERSION_MINOR 0
#define TBF "TBF"

#if defined(__i386__)
	#define PLATFORM 0
	#define ENDIANNESS 0
	#define DATA_OFFSET 0
#elif defined (_AIX) && defined(_LP64)
	#define PLATFORM 1
	#define ENDIANNESS 1
	#define DATA_OFFSET 0
#elif defined (_AIX) && defined(_ILP32)
	#define PLATFORM 0
	#define ENDIANNESS 1
	#define DATA_OFFSET 0
#elif defined (_SOLARIS) && defined(_LP64)
	#define PLATFORM 1
	#define ENDIANNESS 1
	#define DATA_OFFSET 0
#elif defined (_SOLARIS) && defined(_ILP32)
	#define PLATFORM 0
	#define ENDIANNESS 1
	#define DATA_OFFSET 0
#elif defined(_SOLARISX86) && defined(_ILP32)
	#define PLATFORM 0
	#define ENDIANNESS 0
	#define DATA_OFFSET 0
#elif defined(_SOLARISX86) && defined(_LP64)
	#define PLATFORM 1
	#define ENDIANNESS 0
	#define DATA_OFFSET 0
#elif defined(__linux__) && defined(__s390x__)
	#define PLATFORM 1
	#define ENDIANNESS 1
	#define DATA_OFFSET 0
#elif defined(__linux__) && defined(__s390__)
	#define PLATFORM 0
	#define ENDIANNESS 1
	#define DATA_OFFSET 0
#elif defined (MVS) && defined(_LP64)
	#define PLATFORM 1
	#define ENDIANNESS 1
	#define DATA_OFFSET 0
#elif defined (MVS)
	#define PLATFORM 0
	#define ENDIANNESS 1
	#define DATA_OFFSET 0
#else
	#define PLATFORM 0
	#define ENDIANNESS 0
	#define DATA_OFFSET 0
#endif

#define DEFAULT_ENCODING "UTF-8"

// Field types
typedef char bf_byte_t;
typedef U16 bf_short_t;
typedef U32 bf_int_t;
typedef S64 bf_long_t;
typedef double bf_double_t;
typedef char* bf_string_t;
typedef size_t bf_size_t;

typedef bf_long_t bf_timestamp_t;
typedef bf_long_t bf_id_t;
typedef bf_long_t bf_idref_t;



inline size_t
getSize(bf_byte_t t) {
	return sizeof(t);
}

inline size_t
getSize(bf_short_t t) {
	return sizeof(t);
}

inline size_t
getSize(bf_int_t t) {
	return sizeof(t);
}

inline size_t
getSize(bf_long_t t) {
	return sizeof(t);
}

inline size_t
getSize(bf_double_t t) {
	return sizeof(t);
}

inline size_t
getSize(bf_string_t t) {
	return t != NULL ? strlen(t) + 1 : 1;
}

inline size_t
getSize(const bf_string_t t[], const size_t length) {
	if (t == NULL || length == 0) {
		return 1;
	}
	size_t size = 0;
	for (size_t i = 0; i < length; ++i) {
		if (NULL != t[i])
		{
		    size += strlen(t[i]) + 1;
		}
		else
		{
		    size += 1;
		}
	}
	return size;
}

inline size_t
getSize(const bf_long_t t[], const size_t length) {
	if (t == NULL || length == 0) {
		return getSize(static_cast<bf_long_t>(0));
	}
	return sizeof(bf_long_t) * length;
}

inline void *
write(void *buf, bf_byte_t data)
{
	size_t size = getSize(data);
	memcpy(buf, &data, size);
	return (static_cast<char*>(buf) + size);
}

inline void *
write(void *buf, bf_short_t data)
{
	size_t size = getSize(data);
	memcpy(buf, &data, size);
	return (static_cast<char*>(buf) + size);
}

inline void *
write(void *buf, bf_int_t data)
{
	size_t size = getSize(data);
	memcpy(buf, &data, size);
	return (static_cast<char*>(buf) + size);
}

inline void *
write(void *buf, bf_long_t data)
{
	size_t size = getSize(data);
	memcpy(buf, &data, size);
	return (static_cast<char*>(buf) + size);
}

inline void *
write(void *buf, bf_double_t data)
{
	size_t size = getSize(data);
	memcpy(buf, &data, size);
	return (static_cast<char*>(buf) + size);
}

inline void *
write(void *buf, bf_string_t data)
{
	if (data == NULL) {
		return write(buf, static_cast<bf_byte_t>(0));
	}

	strcpy(static_cast<char*>(buf), data);
	return (static_cast<char*>(buf) + getSize(data)); // TODO inefficient
}

inline void *
write(void *buf, bf_string_t data[], size_t length)
{
	if (data == NULL || length == 0) {
		return write(buf, static_cast<bf_byte_t>(0));
	}

	void *buffer = buf;
	for (size_t i = 0; i < length; ++i) {
		bf_string_t str = data[i];
		buffer = write(buffer, str);
	}

	return (buffer);
}

inline void *
write(void *buf, bf_long_t data[], size_t length)
{
	if (data == NULL || length == 0) {
		return write(buf, static_cast<bf_long_t>(0));
	}

	void *buffer = buf;
	for (size_t i = 0; i < length; ++i) {
		buffer = write(buffer, data[i]);
	}

	return (buffer);
}


// Binary format header
/*
 * Header length: 12 bytes
 * Magic indent: \0TBF 4 bytes
 * Current version: major/minor 2 bytes
 * Platform: 0 - 32 bit, 1 - 64 bit 1 byte
 * Endianness: 0 - big-endian, 1 - little-endian
 * Offset to data: pointer 4 bytes
 */

struct CBinaryMessage {

	bf_short_t message_id;
	// modified for bug 255844, on IA-64 size_t is 8 bytes not 4 bytes
	//size_t message_len;
	bf_int_t message_len;


	CBinaryMessage(bf_short_t id);

	virtual void *operator>>(void *buf);

	virtual size_t getLength() const = 0;

	virtual size_t getHeaderLength() const;
};

/**
 * Binary format header.
 */
struct CFormatHeader: public CBinaryMessage {
	bf_byte_t indent;
	bf_string_t magic;
	bf_byte_t version_major;
	bf_byte_t version_minor;
	bf_byte_t platform;
	bf_byte_t endianness;
	bf_int_t data_offset;

	CFormatHeader(bf_byte_t indent,
		bf_string_t magic,
		bf_byte_t version_major,
		bf_byte_t version_minor,
		bf_byte_t platform,
		bf_byte_t endianness,
		bf_int_t data_offset);

	virtual void *operator>>(void *buf);

	virtual size_t getLength() const;

	virtual size_t getHeaderLength() const;
};

// Format header
extern CFormatHeader format_header;

// Format messages ids
enum bf_message_ids {
	// 0 - 1000 System messages
	BF_ENCODING_ID = 1,
	BF_FREQ_ID = 2,

	// 1000 - 32767 Data messages
	BF_NODE_ID = 1001,
	BF_PROCESS_CREATE_ID = 1002,
	BF_AGENT_CREATE_ID = 1003,
	BF_AGENT_DESTROY_ID = 1004,
	BF_TRACE_START_ID = 1005,
	BF_TRACE_END_ID = 1006,
	BF_FILTER_ID = 1007,
	BF_OPTION_ID = 1008,
	BF_THREAD_START_ID = 1009,
	BF_THREAD_END_ID = 1010,
	BF_CLASS_DEF_ID = 1011,
	BF_METHOD_DEF_ID = 1012,
	BF_OBJ_ALLOC_ID = 1013,
	BF_METHOD_ENTRY_ID = 1014,
	BF_METHOD_EXIT_ID = 1015,
	BF_METHOD_CALL_ID = 1016,
	BF_METHOD_RETURN_ID = 1017,
	BF_INVOCATION_CONTEXT_ID = 1018,
	BF_OBJ_DEF_ID = 1019,
	BF_VALUE_ID = 1020,
	BF_METHOD_COUNT_ID = 1021,
	BF_LINE_ID = 1022,
	BF_GC_START_ID = 1023,
	BF_OBJ_FREE_ID = 1024,
	BF_CLASS_UNLOAD_ID = 1025,
	BF_OBJ_MOVE_ID = 1026,
	BF_GC_FINISH_ID = 1027,
	BF_THROW_ID = 1028,
	BF_CATCH_ID = 1029,
	BF_RUNTIME_INIT_DONE_ID = 1030,
	BF_RUNTIME_SHUTDOWN_ID = 1031,
	BF_MON_WAIT_ID = 1032,
	BF_MON_WAITED_ID = 1033,
	BF_MON_CONTENDED_ENTER_ID = 1034,
	BF_MON_CONTENDED_ENTERED_ID = 1035,
	// Messages, which are not listed in public DTD
	BF_AG_METHOD_ENTRY_ID = 1036,
	BF_AG_METHOD_EXIT_ID = 1037,
	BF_HEAP_DUMP_DEF_ID = 1038,
	BF_GC_ROOT_ID = 1039,
	BF_OBJ_REF_ID = 1040,
	BF_CUSTOM_ID = 1041,
	BF_CALL_ID_NOTIFY_ID = 1042,
	BF_CALL_ID_INTERRUPT_ID = 1043,
	BF_CALL_ID_START_ID = 1044,
	BF_CALL_ID_SLEEP_START_ID = 1045,
	BF_CALL_ID_SLEEP_END_ID = 1046
};

/**
 * System message.
 * Defines character encoding used by loader to parse incoming strings
 */
struct CEncodingMessage: public CBinaryMessage {
	bf_string_t encoding;

	CEncodingMessage();
	virtual void *operator>>(void *buf);
	virtual size_t getLength() const;
};

/**
 * System message.
 * Defines max qualified frequency used by loader to calculate timestamp from CPU ticks
 */
struct CFreqMessage: public CBinaryMessage {
	bf_long_t frequency;

	CFreqMessage();
	virtual void *operator>>(void *buf);
	virtual size_t getLength() const;
};


/**
 *  Node message
 * <!ATTLIST node
 * 	nodeId CDATA #REQUIRED
 *	hostname CDATA #REQUIRED
 *	ipaddress CDATA #REQUIRED
 *	timezone CDATA #IMPLIED
 *	time CDATA #IMPLIED>
 */
struct CNodeMessage: public CBinaryMessage {
	bf_string_t node_id;
	bf_string_t hostname;
	bf_string_t ip;
	bf_int_t timezone;
	bf_timestamp_t timestamp;

	CNodeMessage();
	virtual void *operator>>(void *buf);
	virtual size_t getLength() const;
};


/**
 *  ProcessCreate message
 * <!ATTLIST processCreate
 *	processId CDATA #REQUIRED
 *	name CDATA #IMPLIED
 *	pid CDATA #REQUIRED
 *	nodeIdRef CDATA #REQUIRED
 *	time CDATA #REQUIRED
 *	application_executable CDATA #IMPLIED>
 */
struct CProcessCreateMessage: public CBinaryMessage {
	bf_string_t process_id;
	bf_string_t name;
	bf_int_t pid;
	bf_string_t node_id_ref;
	bf_timestamp_t timestamp;
	bf_string_t exe;

	CProcessCreateMessage();
	virtual void *operator>>(void *buf);
	virtual size_t getLength() const;
};

/**
 * AgentCreate message
 * <!ATTLIST agentCreate
 *	agentId CDATA #REQUIRED
 *	processIdRef CDATA #REQUIRED
 *	agentName CDATA #REQUIRED
 *	agentType CDATA #REQUIRED
 *	time CDATA #REQUIRED
 *	agentParameters CDATA #IMPLIED
 *	version CDATA #IMPLIED
 */
struct CAgentCreateMessage: public CBinaryMessage {
	bf_string_t agent_id;
	bf_string_t process_id_ref;
	bf_string_t name;
	bf_string_t type;
	bf_timestamp_t timestamp;
	bf_string_t params;
	bf_string_t version;

	CAgentCreateMessage();
	virtual void *operator>>(void *buf);
	virtual size_t getLength() const;
};


/**
 * AgentDestroy message
 * <!ATTLIST agentDestroy
 *	agentIdRef CDATA #REQUIRED
 *	time CDATA #REQUIRED
 */
struct CAgentDestroyMessage: public CBinaryMessage {
	bf_string_t agent_id_ref;
	bf_timestamp_t timestamp;

	CAgentDestroyMessage();
	virtual void *operator>>(void *buf);
	virtual size_t getLength() const;
};


/**
 * TraceStart message
 * <!ATTLIST traceStart
 *	traceId CDATA #REQUIRED
 *	agentIdRef CDATA #REQUIRED
 *	time CDATA #REQUIRED
 *	collationValue CDATA #IMPLIED
 */
struct CTraceStartMessage: public CBinaryMessage {
	bf_string_t trace_id;
	bf_string_t agent_id_ref;
	bf_timestamp_t timestamp;
	bf_string_t col_value;

	CTraceStartMessage();
	virtual void *operator>>(void *buf);
	virtual size_t getLength() const;
};

/**
 * TraceEnd message
 * <!ATTLIST traceEnd
 *	traceIdRef CDATA #REQUIRED
 *	time CDATA #REQUIRED
 *	collationValue CDATA #IMPLIED
 */
struct CTraceEndMessage: public CBinaryMessage {
	bf_string_t trace_id_ref;
	bf_timestamp_t timestamp;
	bf_string_t col_value;

	CTraceEndMessage();
	virtual void *operator>>(void *buf);
	virtual size_t getLength() const;
};

/**
 * Filter message
 * <!ATTLIST filter
 *	pattern CDATA #REQUIRED
 *	genericPattern CDATA #REQUIRED
 *	mode CDATA #REQUIRED
 *	traceIdRef CDATA #IMPLIED
 */
struct CFilterMessage: public CBinaryMessage {
	bf_string_t trace_id_ref;
	bf_string_t pattern;
	bf_string_t gen_pattern;
	bf_string_t mode;
	bf_string_t meth_pattern;
	bf_string_t meth_gen_pattern;
	bf_string_t meth_mode;

	CFilterMessage();
	virtual void *operator>>(void *buf);
	virtual size_t getLength() const;
};

/**
 * Option message
 * <!ATTLIST option
 *	key CDATA #IMPLIED
 *	value CDATA #IMPLIED
 *	traceIdRef CDATA #IMPLIED
 */
struct COptionMessage: public CBinaryMessage {
	bf_string_t trace_id_ref;
	bf_string_t key;
	bf_string_t value;

	COptionMessage();
	virtual void *operator>>(void *buf);
	virtual size_t getLength() const;

};


/**
 * <!ATTLIST threadStart
 *	transientThreadId CDATA #IMPLIED
 *	threadId CDATA #IMPLIED
 *	time CDATA #IMPLIED
 *	groupName CDATA #IMPLIED
 *	parentName CDATA #IMPLIED
 *	transientObjIdRef CDATA #IMPLIED
 *	objIdRef CDATA #IMPLIED
 *	threadName CDATA #IMPLIED
 *	collationValue CDATA #IMPLIED
 *	traceIdRef CDATA #IMPLIED
 */
struct CThreadStartMessage: public CBinaryMessage {
	bf_id_t trans_thread_id;
	bf_id_t thread_id;
	bf_timestamp_t timestamp;
	bf_string_t group_name;
	bf_string_t parent_group_name;
	bf_idref_t trans_obj_id_ref;
	bf_idref_t obj_id_ref;
	bf_string_t thread_name;
	bf_string_t col_value;
	bf_string_t trace_id_ref;

	CThreadStartMessage();
	virtual void *operator>>(void *buf);
	virtual size_t getLength() const;
};

/**
 * <!ATTLIST threadEnd
 *	transientThreadIdRef CDATA #IMPLIED
 *	threadIdRef CDATA #IMPLIED
 *	time CDATA #IMPLIED
 *	collationValue CDATA #IMPLIED
 *	traceIdRef CDATA #IMPLIED
 */
struct CThreadEndMessage: public CBinaryMessage {
	bf_idref_t trans_thread_id_ref;
	bf_idref_t thread_id_ref;
	bf_timestamp_t timestamp;
	bf_string_t col_value;
	bf_string_t trace_id_ref;

	CThreadEndMessage();
	virtual void *operator>>(void *buf);
	virtual size_t getLength() const;
};

/**
 * <!ATTLIST classDef
 *	transientThreadIdRef CDATA #IMPLIED
 *	threadIdRef CDATA #IMPLIED
 *	time CDATA #IMPLIED
 *	numInterfaces CDATA #IMPLIED
 *	interfaceNames CDATA #IMPLIED
 *	transientClassId CDATA #IMPLIED
 *	classId CDATA #IMPLIED
 *	sourceName CDATA #IMPLIED
 *	classLoader CDATA #IMPLIED
 *	superclass CDATA #IMPLIED
 *	transcientObjIdRef CDATA #IMPLIED
 *	objIdRef CDATA #IMPLIED
 *	name CDATA #IMPLIED
 *	access CDATA #IMPLIED
 *	numStaticFields CDATA #IMPLIED
 *	numMethods CDATA #IMPLIED
 *	numInstanceFields CDATA #IMPLIED
 *	collationValue CDATA #IMPLIED
 *	traceIdRef CDATA #IMPLIED
 */
struct CClassDefMessage: public CBinaryMessage {
	bf_idref_t trans_thread_id_ref;
	bf_idref_t thread_id_ref;
	bf_timestamp_t timestamp;
	bf_int_t num_interfaces;
	bf_string_t interface_names;
	bf_id_t trans_class_id;
	bf_id_t class_id;
	bf_string_t source_name;
	bf_string_t class_loader;
	bf_string_t superclass;
	bf_idref_t trans_obj_id_ref;
	bf_idref_t obj_id_ref;
	bf_string_t name;
	bf_string_t access;
	bf_int_t num_static_fields;
	bf_int_t num_methods;
	bf_int_t num_instance_fields;
	bf_string_t col_value;
	bf_string_t trace_id_ref;

	CClassDefMessage();
	virtual void *operator>>(void *buf);
	virtual size_t getLength() const;
};

/**
 * <!ATTLIST methodDef
 *	name CDATA #IMPLIED
 *	signature CDATA #IMPLIED
 *	visibility CDATA #IMPLIED
 *	isNative CDATA #IMPLIED
 *	isAbstract CDATA #IMPLIED
 *	isStatic CDATA #IMPLIED
 *	isSynchronized CDATA #IMPLIED
 *	exceptions CDATA #IMPLIED
 *	startLineNumber CDATA #IMPLIED
 *	endLineNumber CDATA #IMPLIED
 *	signitureNotation CDATA #IMPLIED
 *	transcientClassIdRef CDATA #IMPLIED
 *	classIdRef CDATA #IMPLIED
 *	methodId CDATA #IMPLIED
 *	collationValue CDATA #IMPLIED
 *	traceIdRef CDATA #IMPLIED
 */
struct CMethodDefMessage: public CBinaryMessage {
	bf_string_t name;
	bf_string_t sig;
	bf_string_t visibility;
	bf_byte_t is_native;
	bf_byte_t is_abstract;
	bf_byte_t is_static;
	bf_byte_t is_sync;
	bf_string_t exceptions;
	bf_long_t start_line;
	bf_long_t end_line;
	bf_string_t sig_notation;
	bf_idref_t trans_class_id_ref;
	bf_idref_t class_id_ref;
	bf_id_t method_id;
	bf_string_t col_value;
	bf_string_t trace_id_ref;

	CMethodDefMessage();
	virtual void *operator>>(void *buf);
	virtual size_t getLength() const;
};


/**
 * <!ATTLIST objAlloc
 *	transientThreadIdRef CDATA #IMPLIED
 *	threadIdRef CDATA #IMPLIED
 *	time CDATA #IMPLIED
 *	isArray CDATA #IMPLIED
 *	transientObjId CDATA #IMPLIED
 *	objId CDATA #IMPLIED
 *	size CDATA #IMPLIED
 *	transientClassIdRef CDATA #IMPLIED
 *	classIdRef CDATA #IMPLIED
 *	contextData CDATA #IMPLIED
 *	collationValue CDATA #IMPLIED
 *	traceIdRef CDATA #IMPLIED
 */
struct CObjAllocMessage: public CBinaryMessage {
	bf_idref_t trans_thread_id_ref;
	bf_idref_t thread_id_ref;
	bf_timestamp_t timestamp;
	bf_byte_t is_array;
	bf_idref_t trans_obj_id_ref;
	bf_idref_t obj_id_ref;
	bf_long_t size;
	bf_long_t line;
	bf_id_t method_id;
	bf_idref_t trans_class_id_ref;
	bf_idref_t class_id_ref;
	bf_string_t context_data;
	bf_string_t col_value;
	bf_string_t trace_id_ref;

	CObjAllocMessage();
	virtual void *operator>>(void *buf);
	virtual size_t getLength() const;
};

/**
 * <!ATTLIST methodEntry
 * 	transientThreadIdRef CDATA #IMPLIED
 *	threadIdRef CDATA #IMPLIED
 *	time CDATA #IMPLIED
 *	methodIdRef CDATA #IMPLIED
 *	ticket CDATA #IMPLIED
 *	transientObjIdRef CDATA #IMPLIED
 *	objIdRef CDATA #IMPLIED
 *	transientClassIdRef CDATA #IMPLIED
 *	classIdRef CDATA #IMPLIED
 *	threadCpuTime CDATA #IMPLIED
 *	sequenceCounter CDATA #IMPLIED
 *	stackDepth CDATA #IMPLIED
 *	collationValue CDATA #IMPLIED
 *	traceIdRef CDATA #IMPLIED
 */
struct CMethodEntryMessage: public CBinaryMessage {
	bf_idref_t trans_thread_id_ref;
	bf_idref_t thread_id_ref;
	bf_timestamp_t timestamp;
	bf_idref_t method_id_ref;
	bf_int_t ticket;
	bf_idref_t trans_obj_id_ref;
	bf_idref_t class_id_ref;
	bf_timestamp_t thread_cpu_time;
	bf_long_t seq_counter;
	bf_long_t stack_depth;
	bf_string_t col_value;
	bf_string_t trace_id_ref;

	CMethodEntryMessage();
	virtual void *operator>>(void *buf);
	virtual size_t getLength() const;
};

/**
 * <!ATTLIST methodExit
 *	transientThreadIdRef CDATA #IMPLIED
 *	threadIdRef CDATA #IMPLIED
 *	time CDATA #IMPLIED
 *	ticket CDATA #IMPLIED
 *	threadCpuTime CDATA #IMPLIED
 *	methodIdRef CDATA #IMPLIED
 *	transientObjIdRef CDATA #IMPLIED
 *	objIdRef CDATA #IMPLIED
 *	transientClassIdRef CDATA #IMPLIED
 *	classIdRef CDATA #IMPLIED
 *	sequenceCounter CDATA #IMPLIED
 *	collationValue CDATA #IMPLIED
 *	traceIdRef CDATA #IMPLIED
 */
struct CMethodExitMessage: public CBinaryMessage {
	bf_idref_t trans_thread_id_ref;
	bf_idref_t thread_id_ref;
	bf_timestamp_t timestamp;
	bf_int_t ticket;
	bf_timestamp_t thread_cpu_time;
	bf_idref_t method_id_ref;
	bf_idref_t trans_obj_id_ref;
	bf_idref_t obj_id_ref;
	bf_idref_t trans_class_id_ref;
	bf_idref_t class_id_ref;
	bf_long_t seq_counter;
	bf_string_t col_value;
	bf_string_t trace_id_ref;

	CMethodExitMessage();
	virtual void *operator>>(void *buf);
	virtual size_t getLength() const;
};

/**
 * <!ATTLIST InvocationContext
 *	nodeIdRef CDATA #IMPLIED
 *	processIdRef CDATA #IMPLIED
 *	agentIdRef CDATA #IMPLIED
 *	threadIdRef CDATA #IMPLIED
 *	ticket CDATA #IMPLIED
 *	sequenceCounter CDATA #IMPLIED
 *	collationValue CDATA #IMPLIED
 *	traceIdRef CDATA #IMPLIED
 */
struct CInvocationContextMessage: public CBinaryMessage {
	bf_idref_t node_id_ref;
	bf_idref_t process_id_ref;
	bf_idref_t agent_id_ref;
	bf_idref_t thread_id_ref;
	bf_int_t ticket;
	bf_long_t seq_counter;
	bf_string_t col_value;
	bf_string_t trace_id_ref;

	CInvocationContextMessage();
	virtual void *operator>>(void *buf);
	virtual size_t getLength() const;
};

/**
 * <!ATTLIST objDef
 *	objId CDATA #REQUIRED
 *	isArray CDATA #IMPLIED
 *	size CDATA #IMPLIED
 *	classIdRef CDATA #IMPLIED
 *	collationValue CDATA #IMPLIED
 *	traceIdRef CDATA #IMPLIED
 */
struct CObjDefMessage: public CBinaryMessage {
	bf_id_t obj_id;
	bf_byte_t is_array;
	bf_long_t size;
	bf_idref_t class_id_ref;
	bf_string_t col_value;
	bf_string_t trace_id_ref;

	CObjDefMessage();
	virtual void *operator>>(void *buf);
	virtual size_t getLength() const;
};

/**
 * <!ATTLIST value
 *	name CDATA #IMPLIED
 *	objIdRef CDATA #IMPLIED
 *	serializationFormat CDATA #IMPLIED
 *	type CDATA #IMPLIED
 *	value CDATA #IMPLIED
 *	collationValue CDATA #IMPLIED
 *	traceIdRef CDATA #IMPLIED
 */
struct CValueMessage: public CBinaryMessage {
	bf_string_t name;
	bf_idref_t obj_id_ref;
	bf_string_t ser_format;
	bf_string_t type;
	bf_string_t value;
	bf_string_t col_value;
	bf_string_t trace_id_ref;

	CValueMessage();
	virtual void *operator>>(void *buf);
	virtual size_t getLength() const;

};


/**
 * <!ATTLIST methodCount
 *	count CDATA #IMPLIED
 *	methodIdRef CDATA #IMPLIED
 *	collationValue CDATA #IMPLIED
 *	traceIdRef CDATA #IMPLIED
 */
struct CMethodCountMessage: public CBinaryMessage {
	bf_long_t count;
	bf_idref_t method_id_ref;
	bf_string_t col_value;
	bf_string_t trace_id_ref;

	CMethodCountMessage();
	virtual void *operator>>(void *buf);
	virtual size_t getLength() const;
};

/**
 * <!ATTLIST line
 *	transientThreadIdRef CDATA #IMPLIED
 *	threadId CDATA #IMPLIED
 *	time CDATA #IMPLIED
 *	line_number CDATA #IMPLIED
 *	file_name CDATA #IMPLIED
 *	methodIdRef CDATA #IMPLIED
 *	ticket CDATA #IMPLIED
 *	threadCpuTime CDATA #IMPLIED
 *	collationValue CDATA #IMPLIED
 *	traceIdRef CDATA #IMPLIED
 */
struct CLineMessage: public CBinaryMessage {
	bf_idref_t trans_thread_id_ref;
	bf_id_t thread_id;
	bf_timestamp_t timestamp;
	bf_long_t line;
	bf_string_t file;
	bf_idref_t method_id_ref;
	bf_int_t ticket;
	bf_timestamp_t thread_cpu_time;
	bf_string_t col_value;
	bf_string_t trace_id_ref;

	CLineMessage();
	virtual void *operator>>(void *buf);
	virtual size_t getLength() const;
};

/**
 * <!ATTLIST gcStart
 *	transientThreadIdRef CDATA #IMPLIED
 *	threadIdRef CDATA #IMPLIED
 *	time CDATA #IMPLIED
 *	collationValue CDATA #IMPLIED
 *	traceIdRef CDATA #IMPLIED
 */
struct CGCStartMessage: public CBinaryMessage {
	bf_idref_t trans_thread_id_ref;
	bf_idref_t thread_id_ref;
	bf_timestamp_t timestamp;
	bf_string_t col_value;
	bf_string_t trace_id_ref;

	CGCStartMessage();
	virtual void *operator>>(void *buf);
	virtual size_t getLength() const;
};

/**
 * <!ATTLIST objFree
 *	transientThreadIdRef CDATA #IMPLIED
 *	threadIdRef CDATA #IMPLIED
 *	time CDATA #IMPLIED
 *	transientObjIdRef CDATA #IMPLIED
 *	objIdRef CDATA #IMPLIED
 *	sequenceCounter CDATA #IMPLIED
 *	collationValue CDATA #IMPLIED
 *	traceIdRef CDATA #IMPLIED
 */
struct CObjFreeMessage: public CBinaryMessage {
	bf_idref_t trans_thread_id_ref;
	bf_idref_t thread_id_ref;
	bf_timestamp_t timestamp;
	bf_idref_t trans_obj_id_ref;
	bf_idref_t obj_id_ref;
	bf_long_t obj_age;
	bf_long_t seq_counter;
	bf_long_t stack_depth;
	bf_string_t col_value;
	bf_string_t trace_id_ref;

	CObjFreeMessage();
	virtual void *operator>>(void *buf);
	virtual size_t getLength() const;
};

/**
 * <!ATTLIST classUnload
 *	transientThreadIdRef CDATA #IMPLIED
 *	threadIdRef CDATA #IMPLIED
 *	time CDATA #IMPLIED
 *	transientClassIdRef CDATA #IMPLIED
 *	classIdRef CDATA #IMPLIED
 *	collationValue CDATA #IMPLIED
 *	traceIdRef CDATA #IMPLIED
 */
struct CClassUnloadMessage: public CBinaryMessage {
	bf_idref_t trans_thread_id_ref;
	bf_idref_t thread_id_ref;
	bf_timestamp_t timestamp;
	bf_idref_t trans_class_id_ref;
	bf_idref_t class_id_ref;
	bf_string_t col_value;
	bf_string_t trace_id_ref;

	CClassUnloadMessage();
	virtual void *operator>>(void *buf);
	virtual size_t getLength() const;
};

/**
 * <!ATTLIST objMove
 *	transientThreadIdRef CDATA #IMPLIED
 *	threadIdRef CDATA #IMPLIED
 *	time CDATA #IMPLIED
 *	transientObjIdRef CDATA #IMPLIED
 *	objIdRef CDATA #IMPLIED
 *	newObjId CDATA #IMPLIED
 *	collationValue CDATA #IMPLIED
 *	traceIdRef CDATA #IMPLIED
 */
struct CObjMoveMessage: public CBinaryMessage {
	bf_idref_t trans_thread_id_ref;
	bf_idref_t thread_id_ref;
	bf_timestamp_t timestamp;
	bf_idref_t trans_obj_id_ref;
	bf_idref_t obj_id_ref;
	bf_id_t new_obj_id;
	bf_string_t col_value;
	bf_string_t trace_id_ref;

	CObjMoveMessage();
	virtual void *operator>>(void *buf);
	virtual size_t getLength() const;
};

/**
 * <!ATTLIST gcFinish
 *	transientThreadIdRef CDATA #IMPLIED
 *	threadIdRef CDATA #IMPLIED
 *	time CDATA #IMPLIED
 *	totalObjectSpace CDATA #IMPLIED
 *	usedObjectSpace CDATA #IMPLIED
 *	usedObjects CDATA #IMPLIED
 *	collationValue CDATA #IMPLIED
 *	traceIdRef CDATA #IMPLIED
 */
struct CGCFinishMessage: public CBinaryMessage {
	bf_idref_t trans_thread_id_ref;
	bf_idref_t thread_id_ref;
	bf_timestamp_t timestamp;
	bf_long_t total_object_space;
	bf_long_t used_object_space;
	bf_long_t used_objects;
	bf_string_t col_value;
	bf_string_t trace_id_ref;

	CGCFinishMessage();
	virtual void *operator>>(void *buf);
	virtual size_t getLength() const;
};


/**
 * <!ATTLIST throw
 *	transientThreadIdRef CDATA #IMPLIED
 *	threadId CDATA #IMPLIED
 *	time CDATA #IMPLIED
 *	objIdRef CDATA #IMPLIED
 *	objHandle CDATA #IMPLIED
 *	methodIdRef CDATA #IMPLIED
 *	ticket CDATA #IMPLIED
 *	collationValue CDATA #IMPLIED
 *	traceIdRef CDATA #IMPLIED
 */
struct CThrowMessage: public CBinaryMessage {
	bf_idref_t trans_thread_id_ref;
	bf_idref_t thread_id_ref;
	bf_timestamp_t timestamp;
	bf_idref_t obj_id_ref;
	bf_long_t obj_handle;
	bf_idref_t method_id_ref;
	bf_int_t ticket;
	bf_string_t col_value;
	bf_string_t trace_id_ref;

	CThrowMessage();
	virtual void *operator>>(void *buf);
	virtual size_t getLength() const;
};

/**
 * <!ATTLIST catch
 *	transientThreadIdRef CDATA #IMPLIED
 *	threadIdRef CDATA #IMPLIED
 *	time CDATA #IMPLIED
 *	objIdRef CDATA #IMPLIED
 *	transientObjIdRef CDATA #IMPLIED
 *	objHandle CDATA #IMPLIED
 *	methodIdRef CDATA #IMPLIED
 *	ticket CDATA #IMPLIED
 *	collationValue CDATA #IMPLIED
 *	traceIdRef CDATA #IMPLIED
 */
struct CCatchMessage: public CBinaryMessage {
	bf_idref_t trans_thread_id_ref;
	bf_idref_t thread_id_ref;
	bf_timestamp_t timestamp;
	bf_idref_t obj_id_ref;
	bf_idref_t trans_obj_id_ref;
	bf_long_t obj_handle;
	bf_idref_t method_id_ref;
	bf_int_t ticket;
	bf_string_t col_value;
	bf_string_t trace_id_ref;

	CCatchMessage();
	virtual void *operator>>(void *buf);
	virtual size_t getLength() const;
};

/**
 * <!ATTLIST runtimeInitDone
 *	transientThreadIdRef CDATA #IMPLIED
 *	threadIdRef CDATA #IMPLIED
 *	time CDATA #IMPLIED
 *	collationValue CDATA #IMPLIED
 *	traceIdRef CDATA #IMPLIED
 */
struct CRuntimeInitDoneMessage: public CBinaryMessage {
	bf_idref_t trans_thread_id_ref;
	bf_idref_t thread_id_ref;
	bf_timestamp_t timestamp;
	bf_string_t col_value;
	bf_string_t trace_id_ref;

	CRuntimeInitDoneMessage();
	virtual void *operator>>(void *buf);
	virtual size_t getLength() const;
};


/**
 * <!ATTLIST runtimeShutdown
 *	transientThreadIdRef CDATA #IMPLIED
 *	threadIdRef CDATA #IMPLIED
 *	time CDATA #IMPLIED
 *	collationValue CDATA #IMPLIED
 *	traceIdRef CDATA #IMPLIED
 */
struct CRuntimeShutdownMessage: public CBinaryMessage {
	bf_idref_t trans_thread_id_ref;
	bf_idref_t thread_id_ref;
	bf_timestamp_t timestamp;
	bf_string_t col_value;
	bf_string_t trace_id_ref;

	CRuntimeShutdownMessage();
	virtual void *operator>>(void *buf);
	virtual size_t getLength() const;
};

/**
 * <!ATTLIST monWait
 *	threadIdRef CDATA #IMPLIED
 *	time CDATA #IMPLIED
 *	objIdRef CDATA #IMPLIED
 *	timeout CDATA #IMPLIED
 */
struct CMonWaitMessage: public CBinaryMessage {
	bf_idref_t thread_id_ref;
	bf_timestamp_t timestamp;
	bf_idref_t obj_id_ref;
	bf_timestamp_t timeout;
	bf_size_t stack_depth; // Auxilliary
	bf_string_t *stack_methods;
	bf_long_t *stack_lines;

	CMonWaitMessage(bf_short_t message_id = BF_MON_WAIT_ID);
	virtual void *operator>>(void *buf);
	virtual size_t getLength() const;
};

/**
 * <!ATTLIST monWaited
 *	threadIdRef CDATA #IMPLIED
 *	time CDATA #IMPLIED
 *	objIdRef CDATA #IMPLIED
 *	timeout CDATA #IMPLIED
 */
struct CMonWaitedMessage: public CMonWaitMessage {
	CMonWaitedMessage();
};


/**
 * <!ATTLIST monContendedEnter
 *	threadIdRef CDATA #IMPLIED
 *	time CDATA #IMPLIED
 *	objIdRef CDATA #IMPLIED
 *	threadOwner CDATA #IMPLIED
 */
struct CMonContendedEnterMessage: public CBinaryMessage {
	bf_idref_t thread_id_ref;
	bf_timestamp_t timestamp;
	bf_idref_t obj_id_ref;
	bf_idref_t thread_owner_id_ref;
	bf_size_t stack_depth; // Auxilliary
	bf_string_t *stack_methods;
	bf_long_t *stack_lines;

	CMonContendedEnterMessage();
	virtual void *operator>>(void *buf);
	virtual size_t getLength() const;
};

/**
 * <!ATTLIST monContendedEntered
 * threadIdRef CDATA #IMPLIED
 *	time CDATA #IMPLIED
 *	objIdRef CDATA #IMPLIED
 */
struct CMonContendedEnteredMessage: public CBinaryMessage {
	bf_idref_t thread_id_ref;
	bf_timestamp_t timestamp;
	bf_idref_t obj_id_ref;
	bf_size_t stack_depth; // Auxilliary
	bf_string_t *stack_methods;
	bf_long_t *stack_lines;

	CMonContendedEnteredMessage();
	virtual void *operator>>(void *buf);
	virtual size_t getLength() const;
};

// Messages not listed in official DTD
// Aggregate method entry
struct CAGMethodEntryMessage: public CBinaryMessage {
	bf_idref_t thread_id_ref;
	bf_idref_t method_id_ref;
	bf_timestamp_t base_time;
	bf_timestamp_t min_time;
	bf_timestamp_t max_time;
	bf_timestamp_t base_cpu_time;
	bf_long_t num_calls;

	CAGMethodEntryMessage();
	virtual void *operator>>(void *buf);
	virtual size_t getLength() const;
};


// Aggregate method exit
struct CAGMethodExitMessage: public CBinaryMessage {
	bf_idref_t thread_id_ref;
	bf_idref_t method_id_ref;

	CAGMethodExitMessage();
	virtual void *operator>>(void *buf);
	virtual size_t getLength() const;
};

// Heap dump start message
struct CHDStartMessage: public CBinaryMessage {
	bf_idref_t heap_dump_id_ref;
	bf_timestamp_t timestamp;
	bf_string_t name;
	bf_timestamp_t base_time;

	CHDStartMessage();
	virtual void *operator>>(void *buf);
	virtual size_t getLength() const;
};

// GC root message
struct CGCRootMessage: public CBinaryMessage {
	bf_idref_t class_id_ref;
	bf_idref_t obj_id_ref;
	bf_string_t type;

	CGCRootMessage();
	virtual void *operator>>(void *buf);
	virtual size_t getLength() const;
};


// Object reference message
struct CObjRefMessage: public CBinaryMessage {
	bf_idref_t src_id_ref;
	bf_idref_t target_id_ref;
	bf_idref_t heap_dump_id_ref;

	CObjRefMessage();
	virtual void *operator>>(void *buf);
	virtual size_t getLength() const;
};

// Custom message
struct CCustomMessage: public CBinaryMessage {
	bf_string_t body;

	CCustomMessage();
	virtual void *operator>>(void *buf);
	virtual size_t getLength() const;
};

// Call Id notify all
struct CCallIdNotifyMessage: public CBinaryMessage {
	bf_idref_t thread_id_ref;
	bf_timestamp_t timestamp;
	bf_idref_t static_obj_id_ref;
	bf_byte_t notify_all;
	bf_size_t stack_depth; // Auxilliary
	bf_string_t *stack_methods;
	bf_long_t *stack_lines;

	CCallIdNotifyMessage();
	virtual void *operator>>(void *buf);
	virtual size_t getLength() const;

protected:
	CCallIdNotifyMessage(enum bf_message_ids msg_id);
};


// Call id interrupt
struct CCallIdInterruptMessage: public CCallIdNotifyMessage {
	CCallIdInterruptMessage();
};

// Call id notify
struct CCallIdStartMessage: public CCallIdNotifyMessage {
	CCallIdStartMessage();
};

// Call id notify
struct CCallIdThreadSleepStartMessage: public CCallIdNotifyMessage {
	CCallIdThreadSleepStartMessage();
	virtual void *operator>>(void *buf);
	virtual size_t getLength() const;

};

// Call id notify
struct CCallIdThreadSleepEndMessage: public CCallIdNotifyMessage {
	CCallIdThreadSleepEndMessage();
	virtual void *operator>>(void *buf);
	virtual size_t getLength() const;

};

} /*namespace JPIAgent*/ } /* namespace Martini */

#endif
