/*****************************************************************************
 * Copyright (c) 1997, 2008 Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/
//
#ifndef MRTE_JIE
#define MRTE_JIE

#ifndef MARTINI_JIE 
#define MARTINI_JIE
#endif

#include "MRTEResults.h"
#include "MRTETypes.h"
#include "MieAPI.h"
#include "OSA.h"
#include "MIterator.h"

// Illegal Java class Constant Pool entry
#define ILLEGAL_CP_ENTRY ((TConstantPoolIndex) 0)

#ifdef JIE_EXPORTS
#define JIE_API API_EXPORT
#else
#define JIE_API API_IMPORT 
#endif

/**
 * @file
 * @brief Martini Instrumentation Engine Java Definitions
 *
 * Defines the Martini Instrumentation Engine API.
 * For more information, refer to the @ref page_mie "Martini Instrumentation Engine Interface" 
 * page.
 */

namespace Martini { 
/**
 * @brief <b> JIE </b> namespace for Java Instrumentation Engine
 */ 
namespace JIE
{
    
   /**
    * @brief <b> EConstantPoolEntryType </b> constant pool entry type
    *
    * This enum describes the type of a value in a constant pool entry. It is used with the 
    * SConstantPoolEntryValue struct.
    */
    enum EConstantPoolEntryType
    {
        CPT_NONE = 0,
        CPT_INTEGER,
        CPT_FLOAT,
        CPT_LONG,
        CPT_DOUBLE,
        CPT_STRING,
        CPT_CLASS,
        CPT_FIELD,
        CPT_METHOD,
        CPT_INTERFACE_METHOD,
        CPT_LAST,
    };

   /**
    * @brief <b> SFieldRefInfo </b> description of a field in the constant pool
    *
    * This structs defines a field type (CPT_FIELD) in a constant pool entry. It is used in 
    * the SConstantPoolEntryValue struct.
    * A field is used to describe a class variable member.
    *
    */
    struct SFieldRefInfo
    {
        MIE::SWideString className; //!< name of the declaring class in which the field is defined
        MIE::SWideString name;      //!< name of the field
        MIE::SWideString signature; //!< signature of the field
    };

    typedef SFieldRefInfo SMethodRefInfo;
    typedef SFieldRefInfo SInterfaceMethodRefInfo;
    
   /**
    * @brief <b> SConstantPoolEntryValue </b> a struct defining an entry in the constant pool
    *
    * This struct describes the type and value of an entry in the constant pool. It is used 
    * when adding a new entry to the constant pool, using IJavaClass::AddEntryToCP(), and when
    * retrieving an existing constant pool entry using IJavaClass::GetValueFromCPEntry()
    */
    struct SConstantPoolEntryValue
    {
        EConstantPoolEntryType type;    //!< type of entry in the constant pool 
        union
        {
            int                    iVal;                        //!< CPT_INTEGER entry
            float                  fVal;                        //!< CPT_FLOAT entry
            S64                    s64Val;                      //!< CPT_LONG entry
            double                 dVal;                        //!< CPT_DOUBLE entry
            MIE::SWideString       strVal;                      //!< CPT_STRING entry
            MIE::SWideString       className;                   //!< CPT_CLASS entry
            SFieldRefInfo          fieldVal;                    //!< CPT_FIELD entry
            SMethodRefInfo         methodVal;                   //!< CPT_METHOD entry
            SInterfaceMethodRefInfo interfaceMethodVal;         //!< CPT_INTERFACE_METHOD entry
        } u;
    };

   /**
    * @brief <b> EVarType </b> type of variable.
    *
    * This enum defines the type of a variable: local variable or class variable.
    *
    */
    enum EVarType
    {
        VT_NONE,
        VT_BYTE,        //!< byte
        VT_SHORT,       //!< short
        VT_INT,         //!< integer
        VT_LONG,        //!< long 
        VT_FLOAT,       //!< float
        VT_DOUBLE,      //!< double
        VT_CHAR,        //!< character
        VT_BOOLEAN,     //!< boolean
        VT_REFERENCE,   //!< reference
        VT_LAST,
    };
    

   /**
    * @brief <b> SVarValue </b> value of a local or class variable
    *
    * This struct represents a value of a variable.  It is used with 
    * IJavaClass::AddField() in order to add a static primitive field to a class
    *  
    */
    struct SVarValue
    {
        EVarType    type;       //!< type of variable
        union
        {
            S8      byteVal;    //!< VT_BYTE type
            S16     shortVal;   //!< VT_SHORT type
            S32     intVal;     //!< VT_INT type
            S64     longVal;    //!< VT_LONG type
            float   floatVal;   //!< VT_FLOAT type
            double  doubleVal;  //!< VT_DOUBLE type
            char    charVal;    //!< VT_CHAR type
            bool    boolVal;    //!< VT_BOOLEAN type
            U32     refVal;     //!< VT_REFERENCE
        };
    };

   /**
    * @brief <b> EFieldAttributeFlags </b> Field attribute flags
    *
    * Field Attribute flags, including access and property flags. It is used with 
    * IJavaClass::AddField() API as bitwise OR'ed flags
    *    
    */
    enum EFieldAttributeFlags
    {
        FAF_PUBLIC       = 0x0001, //!< may be accessed from outside its package
        FAF_PRIVATE      = 0x0002, //!< usable only within the defining class
        FAF_PROTECTED    = 0x0004, //!< may be accessed within subclasses
        FAF_STATIC       = 0x0008, //!< static field
        FAF_FINAL        = 0x0010, //!< no further assignment after initialization
        FAF_VOLATILE     = 0x0040, //!< cannot be cached
        FAF_TRANSIENT    = 0x0080  //!< not written or read by a persistent object manager
    };

   /**
    * @brief <b> EMethodAttributeFlags </b> Method attribute flags
    *
    * Method Attribute flags, including access and property flags. It is used with the 
    * SJavaMethodInfo struct as bitwise OR'ed flags
    *    
    */
    enum EMethodAttributeFlags
    {
        MAF_PUBLIC       = 0x0001, //!< may be accessed from outside its package
        MAF_PRIVATE      = 0x0002, //!< accessible only within the defining class
        MAF_PROTECTED    = 0x0004, //!< may be accessed within subclasses
        MAF_STATIC       = 0x0008, //!< static method
        MAF_FINAL        = 0x0010, //!< may not be overridden
        MAF_SYNCHRONIZED = 0x0020, //!< invocation is wrapped in a monitor block
        MAF_NATIVE       = 0x0100, //!< implemented in a language other than Java
        MAF_ABSTRACT     = 0x0400, //!< no implementation of the method is provided
        MAF_STRICT       = 0x0800, //!< floating-point mode is FP-strict
    };
    
   /**
    * @brief <b> EClassAttributeFlags </b> Class attribute flags
    *
    * Class Attribute flags, including access and property flags. It is used with the 
    * SClassMethodInfo struct as bitwise OR'ed flags
    *    
    */
    enum EClassAttributeFlags
    {
        CAF_PUBLIC       = 0x0001, //!< may be accessed from outside its package
        CAF_FINAL        = 0x0010, //!< no subclasses allowed
        CAF_INTERFACE    = 0x0200, //!< is an interface, not a class
        CAF_ABSTRACT     = 0x0400, //!< class is abstract, may not be instantiated
        CAF_ANNOTATION   = 0x2000, //!< class is declared as an annotation type
        CAF_ENUM         = 0x4000, //!< class is declared as an enum type
    };
    
    
   /**
    * @brief <b> SJavaClassInfo </b> class related information  
    *
    * Class related information. All the SWideString arguments are allocated by 
    * the caller.
    *
    **/
    struct SJavaClassInfo
    {
        MIE::SWideString  className;         //!< class name
        MIE::SWideString  superClassName;    //!< super class name
        MIE::SWideString  sourceFileName;    //!< source file name
        unsigned int uiAttributeFlags;       //!< class access and property flags. Used as bitwise  
                                             //   flags, as defined by EClassAttributeFlags
    };

   /**
    * @brief <b> SJavaMethodInfo </b> method related information 
    *
    * Method related information. The name and signature buffers are allocated by the caller.
    *
    **/
    struct SJavaMethodInfo
    {
        MIE::SWideString      className;        //!< class name
        MIE::SWideString      methodName;       //!< method name
        MIE::SWideString      methodSignature;  //!< method signature
        unsigned int      uiAttributeFlags;     //!< method access and property flags. Used as 
                                                //   bitwise OR'ed flags, as defined by 
                                                //   EMethodAttributeFlags
    };

   /**
    * @brief Defines field information
    *
    * Defines field related information. The name and signature buffers are allocated 
    * by the caller.
    *
    **/
    struct SJavaFieldInfo
    {
        MIE::SWideString      className;        //!< class name
        MIE::SWideString      fieldName;        //!< field name
        MIE::SWideString      fieldSignature;   //!< field signature/descriptor
        unsigned int          uiAttributeFlags; //!< @brief access and property flags. Used as 
                                                //   bitwise OR'ed flags, as defined by 
                                                //   EFieldAttributeFlags
    };


   /**
    * @brief <b> IJavaMethod Interface Class </b> An interface for a Java method
    * 
    */
    class IJavaMethod : public MIE::IMethod
    {
    public:
        virtual ~IJavaMethod() {}
        
        /**
         * @brief <b> GetMethodInfo </b> Provides method related info
         *
         * This API fills the SJavaMethodInfo struct with the method information
         *
         * @param pMethodInfo [out] - an SJavaMethodInfo structure including internal 
         *                            buffers allocated by the caller. Contains method 
         *                            info upon successful return
         *
         * @returns - MRTE_RESULT_OK                success
         * @returns - MRTE_ERROR_UNEXPECTED         failed to retrieve info
         * @returns - MRTE_ERROR_BUFFER_TOO_SHORT   one of the SWideString buffers in 
         *                                          SJavaMethodInfo are not big enough.
         *                                          uiActualSize is updated with the required
         *                                          size
         */
        virtual TResult GetMethodInfo(SJavaMethodInfo* pMethodInfo) const = 0;

        /**
         * @brief <b> AddLocalVariable </b> adds a local variable to the method
         *
         * This API adds a local variable to the method and returns the variable ID
         *
         * @param pVariableId [out]  - the allocated variable id, in case return value is 
         *                             success
         * @param variableType [in]  - indicates the type of the new local variable
         *
         * @returns - MRTE_RESULT_OK                success
         * @returns - MRTE_ERROR_ILLEGAL_ARGUMENT   variable id pointer is NULL, or illegal
         *                                          variableType
         * @returns - MRTE_ERROR_FAILURE            failed to allocate a new local variable
         *                  
         */
        virtual TResult AddLocalVariable(MIE::TVariableID *pVariableId, EVarType variableType) = 0;

        virtual TConstantPoolIndex GetIndex() const = 0;
    };


   /**
    * @brief <b> TJavaMethodIterator </b> an Iterator over a collection of IJavaMethod
    * objects.
    *
    * This type defines a forward-only iterator over a class' methods, 
    * Methods are returned in no particular order.
    */
    typedef RTUtil::IIterator<IJavaMethod*> TJavaMethodIterator;
    
    class IJavaField
    {
    public:
        virtual ~IJavaField() {}
        
        /**
         * @brief Provides field related info
         *
         * This API fills the SJavaFieldInfo struct with the field information
         *
         * @param[out] pFieldInfo   an SJavaFieldInfo structure including internal 
         *                          buffers allocated by the caller. Contains field 
         *                          info upon successful return
         *
         * @returns - MRTE_RESULT_OK                success
         * @returns - MRTE_ERROR_UNEXPECTED         failed to retrieve info
         * @returns - MRTE_ERROR_BUFFER_TOO_SHORT   one of the SWideString buffers in 
         *                                          SJavaFieldInfo are not big enough.
         *                                          uiActualSize is updated with the required
         *                                          size
         */
        virtual TResult GetFieldInfo(SJavaFieldInfo* pFieldInfo) const = 0;

        virtual TConstantPoolIndex GetIndex() const = 0;
    };

    /**
     * @brief Defines an iterator over a collection of IJavaField objects
     *
     * This type defines a forward-only iterator over a class' fields. Fields are 
     * returned in no particular order.
     */
    typedef RTUtil::IIterator<IJavaField*> TJavaFieldIterator;

   /**
    * @brief <b> IJavaClass Interface Class </b> An interface for a Java class
    *
    * This interfaces provides APIs for getting class information, iterating on all its 
    * methods, and retrieveing\adding entries from\to the constant pool
    */    
    class IJavaClass
    {
    public:
        virtual ~IJavaClass() {}

        /**
         * @brief <b> GetName </b> gets the class name
         *
         * This API fills the pClassName argument with the class name. pClassName buffer
         * is allocated by the caller.
         *
         * @param pClassName [out] - the buffer to fill with the class name. In case the buffer 
         *                           is not large enough pClassName->uiActualSize is updated
         *                           with the required buffer size.
         *
         * @returns - MRTE_RESULT_OK                success
         * @returns - MRTE_ERROR_ILLEGAL_ARGUMENT   pClassName is NULL
         * @returns - MRTE_ERROR_BUFFER_TOO_SHORT   pClassName buffer is not large enough
         */
        virtual TResult GetName(MIE::SWideString *pClassName) const = 0;

        /**
         * @brief <b> GetInfo </b> gets class information, as described in SJavaClassInfo
         *
         * This API fills the pClassInfo argument with class information. pClassInfo, 
         * including the internal buffers it contains, is allocated by the caller.
         *
         * @param pClassInfo [out] - the buffer to fill with the class info. In case one or 
         *                           more buffer fields in pClassInfo is not large enough,
         *                           uiActualSize in the too short buffer is updated with
         *                           the required buffer size.
         *
         * @returns - MRTE_RESULT_OK                success
         * @returns - MRTE_ERROR_ILLEGAL_ARGUMENT   pClassInfo is NULL
         * @returns - MRTE_ERROR_BUFFER_TOO_SHORT   a buffer in pClassInfo is not large enough
         */
        virtual TResult GetInfo(SJavaClassInfo *pClassInfo) const = 0;

        /**
         * @brief <b> GetNumberOfMethods </b> returns the number of methods in the class.
         *
         * This API returns the number of methods in the class.
         *
         * @returns - unsigned int      the number of methods in the class.
         */
        virtual unsigned int GetNumberOfMethods() const = 0;

        virtual unsigned int GetNumberOfFields() const = 0;

        /**
         * @brief <b> GetMethodIterator </b> returns an iterator over the class' methods
         *
         * This API initializes and returns a new iterator over the class' methods.
         * The iterator points to objects of type IJavaMethod*.
         * 
         * * NOTE:
         * When done using the iterator, call its Free method to deallocate the iterator.
         * Do not use the delete operator.
         *
         * @returns - an iterator over the class' methods.
         */
        virtual TJavaMethodIterator *GetMethodIterator() = 0;

        /**
         * @brief Returns an iterator over the class' fields
         *
         * This API initializes and returns a new iterator over the class' fields.
         * The iterator points to objects of type IJavaField*.
         * 
         * * NOTE:
         * When done using the iterator, call its Free method to deallocate the iterator.
         * Do not use the delete operator.
         *
         * @return an iterator over the class' fields
         */
        virtual TJavaFieldIterator *GetFieldIterator() = 0;

        /**
         * @brief Adds a field to the class
         *
         * This API adds an instance field or a static field to the class. 
         * The field will receive an initial value, as specified in the 
         * "Java Language Specification". Use this API to create primitive-type fields.
         * Use the AddReferenceField API to create reference fields.
         *
         * @param[in] szName            name of the new field
         * @param[in] type              type of the new field
         * @param[in] uiAttributeFlags  variable access and property flags. Used as bitwise 
         *                              OR'ed flags, as defined by EFieldAttributeFlags
         *
         * @return  an interface to the newly created field, or NULL if an error occured
         */
        virtual IJavaField* AddField(const MIE::SWideString& szName, 
                                     EVarType type, 
                                     unsigned int uiAttributeFlags) = 0;

        virtual IJavaField* AddReferenceField(const MIE::SWideString& szName, 
                                              const MIE::SWideString& szSignature,
                                              int uiAttributeFlags) = 0;

        /**
         * @brief Adds a new method to the class
         *
         * Adds a new method to the Java class. The method is added with no code.
         *
         * @param[in]   szName              method name
         * @param[in]   szSignature         method signature
         * @param[in]   uiAttributeFlags    access flags. Used as bitwise OR'ed flags,
         *                                  as defined by EMethodAttributeFlags.
         *
         * @return an interface to the newly created method, or NULL if an error occured
         */
        virtual IJavaMethod* AddMethod(const MIE::SWideString& szName,
                                       const MIE::SWideString& szSignature,
                                       unsigned int uiAttributeFlags) = 0;

        /** 
         * @brief <b> RegisterRecorderCallback </b> Registers a managed callback in the
         *                                                 class
         *
         * This API registers a managed callback, that can be called by the instrumented 
         * methods in this class.
         *
         * @param szClassname    [in]   - containing class name of the callback method
         * @param szMethodName   [in]   - callback method name
         * @param szMethodSig    [in]   - callback method signature
         *
         * @returns - a valid constant pool index for the new callback function in case of 
         *            success
         * @returns - ILLEGAL_CP_ENTRY in case of a failure
         */
        virtual TConstantPoolIndex RegisterRecorderCallback(const MIE::SWideString& szClassName, 
                                                            const MIE::SWideString& szMethodName,
                                                            const MIE::SWideString& szMethodSig) = 0;

        /** 
         * @brief <b> GetValueFromCPEntry </b> Retrieve a constant pool entry value
         *
         * This API retrieves the value of an entry in the constant pool
         *
         * @param cpIndex [in]  - constant pool index for the requested entry to retrieve
         * @param pValue  [out] - a buffer, allocated by the caller. It is filled with the 
         *                        constant pool entry value in case of success
         *
         * @returns - MRTE_RESULT_OK                success
         * @returns - MRTE_ERROR_ILLEGAL_ARGUMENT   invalid cpIndex, or pValue is NULL
         * @returns - MRTE_ERROR_BUFFER_TOO_SHORT   one of the SWideString fields in pValue  
         *                                          structure is not big enough
         */
        virtual TResult GetValueFromCPEntry(TConstantPoolIndex cpIndex, 
                                            SConstantPoolEntryValue* pValue) const = 0;

        /** 
         * @brief <b> AddEntryToCP </b> Add an entry to the constant pool 
         *
         * This API adds an entry to the constant pool.
         *
         * @param value [in] - the new value to be added to the constant pool. 
         *
         * @returns  - a valid constant pool index for the new constant pool entry
         * @returns  - ILLEGAL_CP_ENTRY in case of a failure
         */
        virtual TConstantPoolIndex AddEntryToCP(const SConstantPoolEntryValue& value) = 0;

        /** 
         * @brief <b> WriteBack </b> writes the instrumented class
         * 
         * This API writes the instrumented class in the format specified by the 
         * "Java Virtual Machine Specification". The class is written to pBuffer
         *
         * @param pBuffer [out] - an unallocated buffer to be filled with the 
         *                        instrumented classfile.
         *                        If the API executes successfully, pBuffer will contain
         *                        the instrumented class file and uiBufferSize and
         *                        uiActualSize will be set to the buffer size in bytes.
         *                        If the API terminates with an error, pBuffer will be
         *                        NULL and other members are undefined.
         *                        The client is responsible for deallocating the buffer.
         *
         * @param funcAllocator [in] - a function to be used for allocating memory
         *                             for pBuffer.
         *
         * @returns - MRTE_RESULT_OK                success
         * @returns - MRTE_ERROR_ILLEGAL_ARGUMENT   pBuffer or funcAllocator are NULL
         * @returns - MRTE_ERROR_OUT_OF_MEMORY      cannot allocate memory
         */
        virtual TResult WriteBack(MIE::SBuffer *pBuffer, 
                                  TMemoryAllocatorFunc funcAllocator) = 0;

        /** 
         * @brief <b> Free </b> deallocates the object and all its resources
         * 
         * This API deallocates the object and all its resources.
         * Call this method when the object is no longer needed.
         */
        virtual void Free() = 0;
    };

   /**
    * @brief <b> IJIE Interface Class </b> Java Instrumentor Engine interface
    *
    */    
    class IJIE : public MIE::IMIE
    {
    public:
        /** 
         * @brief <b> GetClassInterface </b> Get a class interface of an about to be loaded 
         *                                   class
         * 
         * Given a valid memory-mapped classfile, this API creates a class interface,
         * which enables getting class and method information, as well as instrumenting 
         * the class methods and constant pool 
         *
         * @param pucClassFile [in] - a buffer containing a valid memory mapped classfile,
         *                            in the format specified by the "Java Virtual 
         *                            Machine Specification" 
         * @param uiClassSize  [in] - size in bytes of the class file 
         *
         * @returns - a valid class interface upon success
         * @returns - NULL upon failure
         */
        virtual IJavaClass *GetClassInterface(const unsigned char *pucClassFile,
                                              unsigned int uiClassSize) const = 0;


        /**
         * @brief enable the creation of the StackMapTable method
         * attribute when instrumenting Java classes with version >= 50.0 
         * (Java 6 and above)
         */
        virtual void EnableStackMapCalculation() const = 0;

    };

    /**
     * @brief <b> GetJIEInstance </b> An API for getting IJIE interface
     */ 
     extern "C" JIE_API IJIE *GetJIEInstance();

}} // end of namespace JIE


#endif // #define MRTE_JIE
