/*****************************************************************************
 * Copyright (c) 1997-2007, Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/

#ifndef __MHASH_H__
#define __MHASH_H__

#include "MRTEResults.h"
#include "OSA.h"
#include "MString.h"
#include <memory.h>
#include <stdio.h>

/**
 * @file MHash.h
 * @brief <b>Martini Hash classes</b>
 */


/** 
 * @page page_hash Martini Hash Classes
 *
 * @section intro Introduction
 *
 * The Martini hash classes provide a complete set of templated hash classes for primitive 
 * and string key types with a synchronized and non-synchronized versions.
 * The hash classes provide fast access, in O(1), for setting a new element, getting a stored 
 * element and for deleting an element.
 * The data types in the hash can be a primitive type or a pointer to an object, the hash 
 * classes do not accept a class as a template parameter.
 *
 */

namespace Martini { namespace RTUtil 
{
    
    /**
    * @brief <b>HASH_DEFAULT_INITIAL_SIZE</b> defines the default size of the hash.
    **/
    static const unsigned int HASH_DEFAULT_INITIAL_SIZE = 0x10000;
            
    /** 
    * @brief <b>MHash</b> interface for Martini hash classes.
    * 
    * @param KEY - The type of the hash key. Must be a primitive type.
    * @param T - The type of the data stored in the hash. Can be a primitive type or a pointer 
    *            to an object. A class type will cause a compilation error.
    **/
    // THIS INTERFACE IS ONLY FOR DOXYGEN PURPOSE - IT IS NOT INHERITED!!
    template<class KEY, class T>
    class MHash
    {
        /**
        * @brief <b>THashIterationCallBack</b> callback type definition for hash iteration.
        *
        * When using the Iterate() function the user should provide a callback function that  
        * adheres to this type definition. The Iterate() function will call the provided  
        * function for every data item in the hash.
        * @param pParameter - A void pointer provided by the user when calls to Iterate() and 
        *                     passed to the callback transparently.
        * @param key - The key for the current data item.
        * @param data - The data item from the hash. 
        *
        * @returns - true to continue the iteration to the next item
        * @returns - false to stop the iteration and return from the Iterate() function
        **/
        typedef bool (*THashIterationCallBack)(void *pParameter, const KEY key, T data);

    public:
        /**
        * @brief <b>MHash</b> Constructor for the MHash class.
        *
        * The constructor does not accept parameters and does not perform any action, after 
        * the creation of the hash the user must call the Init() function in order to 
        * initialize the data structure.
        **/
        MHash() {}
    
        /**
        * @brief <b>~MHash</b> Destructor for the MHash class.
        *
        * The destructor only deletes the hash internal data items and does not delete the  
        * user data. If the data part is a pointer to some user allocated object it is the 
        * user's responsibility to delete his/her data items using the Iterate() function and  
        * supplying an iteration callback that will normally handle this.
        **/
        virtual ~MHash() {}
    
        /**
        * @brief <b>Init</b> Initialize and configure the hash table. All arguments have 
        *                    default values.
        *
        * @param bReplaceIfExist - Chooses whether to replace an existing data item with the 
        *                          same key or keep the old one when calling the set() 
        *                          function. default is false.
        * @param errorValue - Defines the value to return in case of error or not existing 
        *                     item in the Set(), Get() and Del() functions. This parameter 
        *                     should be used when NULL or zero are legal values in the context  
        *                     of the data stored in the hash.
        * @param uiSize - Defines the size of the hash array. The size of the array must be a 
        *                 power of 2. It will be rounded up to the nearest power of 2 in case
        *                 of failure to satisfy this demand.
        * @param uiShift - Sets the number of least significant bits to right shift in the 
        *                  hash function.
        *
        * @returns - MRTE_RESULT_OK upon success
        * @returns - MRTE_ERROR_OUT_OF_MEMORY out of memory
        * @returns - MRTE_ERROR_ILLEGAL_ARGUMENT setting shift for string hash
        **/
        virtual TResult Init(bool bReplaceIfExist = false, T errorValue = 0, 
            unsigned int uiSize = HASH_DEFAULT_INITIAL_SIZE, unsigned int uiShift = 0) = 0;
    
        /**
        * @brief <b>Set</b> Adds/Sets a data item into the hash. If the key already exists it 
        *                   may be replaced or not, depending on the value of bReplaceIfExist  
        *                   defined in the Init() function.
        *
        * @param key - The key to the data item.
        * @param data - The data item to store in the hash.
        * @param pOldData - Pointer to a variable for retrieving the old data item if Set 
        *                   replaced an existing entry with the new data. This is an optional 
        *                   parameter with default value of NULL.
        *
        * @returns - MRTE_RESULT_OK upon success
        * @returns - MRTE_RESULT_KEY_ALREADY_EXISTS when an element with the same key was 
        *                                           found in the hash. Data is either replaced 
        *                                           or not per bReplaceIfExist.  
        * @returns - MRTE_ERROR_OUT_OF_MEMORY  out of memory
        * @returns - MRTE_ERROR_NOT_YET_INITIALIZED  called before Init()
        * @returns - MRTE_ERROR_IN_ITERATOR  called while iterating over the hash
        **/
        virtual TResult Set(const KEY key, T data, T* pOldData = NULL) = 0;

        /**
        * @brief <b>Get</b> Gets a data item from the hash. 
        *
        * @param key - The key to the data item.
        *
        * @returns - data upon success
        * @returns - errorValue when item does not exists or if called while iterating over 
        *            the hash or if called before Init()

        **/
        virtual T Get(const KEY key) = 0;

        /**
        * @brief <b>Del</b> Deletes a data item from the hash. 
        *
        * @param key - The key to the data item.
        *
        * @returns - data upon success
        * @returns - errorValue when item does not exists or if called while iterating over 
        *            the hash or if called before Init()
        **/
        virtual T Del(const KEY key) = 0;
    
        /**
        * @brief <b>Empty</b> Clears the hash table. This function does not delete any data 
        *                     items pointed from the hash elements. It is the user's  
        *                     responsibility to delete them or to keep them for further use.
        *
        * @returns - MRTE_ERROR_NOT_YET_INITIALIZED  called before Init()
        * @returns - MRTE_ERROR_IN_ITERATOR  called while iterating over the hash
        **/
        virtual TResult Empty() = 0;

        /**
        * @brief <b>Count</b> Returns the number of data items in the hash. 
        *
        * @returns - the number of data items in the hash
        **/
        virtual unsigned int Count() = 0;
    
        /**
        * @brief <b>Iterate</b> Iterates on the data items and calls the user's callback for 
        *                       every item. 
        *
        * @param pCallback - Pointer to a user defined callback of the type 
        *                    THashIterationCallBack to be called for every data item.
        * @param pParameter - A void pointer to be passed to the user's callback.
        *
        * @returns - MRTE_RESULT_OK upon success and iteration over the entire hash data items
        * @returns - MRTE_RESULT_ITERATION_STOPPED when the iteration callback returned false
        *                                          that caused the iteration to stop.
        * @returns - MRTE_ERROR_ILLEGAL_ARGUMENT in case of a NULL in the pCallback parameter
        **/
        virtual TResult Iterate(THashIterationCallBack pCallback, void *pParameter) = 0;
    };

    #define MHASH_HASH_FUNC(xxx, size, shift) (((xxx) >> shift) & (size-1))

    /** 
    * @brief <b>MHashNoSync</b> hash table for primitive type keys with no synchronization.
    * 
    * This class should be used when no synchronization is required or when the user of this 
    * class prefers to synchronize his/her code before accessing.
    *
    * @param KEY - The type of the hash key. Must be a primitive type.
    * @param T - The type of the data stored in the hash. Can be a primitive type or a pointer 
    *            to an object. Class type will cause a compilation error.
    **/
    template<class KEY, class T>
    class MHashNoSync 
    {
    public:
        typedef bool (*THashIterationCallBack)(void *pParameter, const KEY key, T data);

        MHashNoSync();
        virtual ~MHashNoSync();
    
        virtual TResult Init(bool bReplaceIfExist = false, T errorValue = (T)0, 
            unsigned int uiSize = HASH_DEFAULT_INITIAL_SIZE, unsigned int uiShift = 0);
    
        virtual TResult Set(const KEY key, T data, T* pOldData = NULL);
        virtual T Get(const KEY key);
        virtual T Del(const KEY key);
    
        virtual TResult Empty();
        virtual unsigned int Count();

        virtual TResult Iterate(THashIterationCallBack pCallback, void *pParameter);
        
    private:
        // Internal struct for the hash maintenance array
        struct SHashCell
        {
            SHashCell *next;
            T data;
            KEY key;
        };

        MHashNoSync(MHashNoSync<KEY, T>&) {} // preventing copy constructor

        SHashCell **m_ppHashArray;
        unsigned int m_uiCount;
        unsigned int m_uiShift;
        unsigned int m_uiSize;
        bool m_bReplaceIfExist;
        bool m_bInIterator;
        bool m_bWasInit;
        
        T m_errorValue;

#ifdef _DEBUG
        // debug statistics
        unsigned int m_uiSumAccessSteps;
        unsigned int m_uiAccessesCount;
        double m_dAccessAverage;
        unsigned int m_uiMaxStepsToAccess;
        
        void DebugStatistics(unsigned int uiCount)
        {
            if (m_uiMaxStepsToAccess < uiCount)
            {
                m_uiMaxStepsToAccess = uiCount;
            }
            
            m_uiAccessesCount++;
            m_uiSumAccessSteps += uiCount;
            m_dAccessAverage = (double)m_uiSumAccessSteps / m_uiAccessesCount;
        }

    public:
        void PrintDebugStatistics(FILE* pOutFile)
        {
            fprintf(pOutFile, "Number of item: %d, Total size: %d, Accesses: %d\n", 
                m_uiCount, m_uiSize, m_uiAccessesCount);
            fprintf(pOutFile, "Max steps to access item: %d, Average: %.3f Item/Size: %.3f\n", 
                m_uiMaxStepsToAccess, m_dAccessAverage, (double)m_uiCount / m_uiSize);
        }
#endif //_DEBUG
    };

    /** 
    * @brief <b>MHashSync</b> hash table for primitive type keys with synchronization on 
    *                           access.
    *
    * This class should be used when synchronization is required by the user on accessing the 
    * hash.
    *
    * @param KEY - The type of the hash key. Must be a primitive type.
    * @param T - The type of the data stored in the hash. Can be a primitive type or a pointer 
    *            to an object. Class type will cause a compilation error.
    */
    template<class KEY, class T>
    class MHashSync 
    {
    public:
        typedef bool (*THashIterationCallBack)(void *pParameter, const KEY key, T data);

        MHashSync();
        virtual ~MHashSync();
        
        virtual TResult Init(bool bReplaceIfExist = false, T errorValue = (T)0, 
            unsigned int uiSize = HASH_DEFAULT_INITIAL_SIZE, unsigned int uiShift = 0);
        
        virtual TResult Set(const KEY key, T data, T* pOldData = NULL);
        virtual T Get(const KEY key);
        virtual T Del(const KEY key);
        
        virtual TResult Empty();
        virtual unsigned int Count();
        
        virtual TResult Iterate(THashIterationCallBack pCallback, void *pParameter);
    private:
        MHashSync(MHashSync<KEY, T>&) {} // preventing copy constructor

        MHashNoSync<KEY, T> m_Hash;
        Martini::OSA::IThreadSync *m_pThreadSync;

#ifdef _DEBUG   
    public:
        void PrintDebugStatistics(FILE* pOutFile)
        {
            m_Hash.PrintDebugStatistics(pOutFile);
        }
#endif //_DEBUG
    };


    /** 
    * @brief <b>MStrHashNoSync</b> hash table for string keys with no synchronization.
    * 
    * This class should be used when no synchronization is required or when the user of this 
    * class prefers to synchronize his/her code before accessing.
    *
    * @param CHR - The type of the string in the hash key. Must be char or wchar_t.
    * @param T - The type of the data stored in the hash. Can be a primitive type or a pointer 
    *            to an object. Class type will cause a compilation error.
    */
    template <class CHR, class T>
    class MStrHashNoSync 
    {
    public:
        typedef bool (*THashIterationCallBack)(void *pParameter, const CHR* key, T data);

        MStrHashNoSync();
        virtual ~MStrHashNoSync();
        
        virtual TResult Init(bool bReplaceIfExist = false, T errorValue = (T)0, 
            unsigned int uiSize = HASH_DEFAULT_INITIAL_SIZE);
        
        virtual TResult Set(const CHR* key, T data, T* pOldData = NULL);
        virtual T Get(const CHR* key);
        virtual T Del(const CHR* key);
        
        virtual TResult Empty();
        virtual unsigned int Count();
        
        virtual TResult Iterate(THashIterationCallBack pCallback, void *pParameter);
    private:
        // Internal struct for the hash maintenance array
        struct SHashCell
        {
            SHashCell *next;
            T data;
            MString<CHR> key;
        };
        
        MStrHashNoSync(MStrHashNoSync<CHR, T>&) {} // preventing copy constructor

        unsigned int HashValue(const CHR* szKey);
        
        SHashCell **m_ppHashArray;
        unsigned int m_uiCount;
        unsigned int m_uiSize;
        unsigned int m_uiKeySize;
        bool m_bReplaceIfExist;
        bool m_bInIterator;
        bool m_bWasInit;
        
        T m_errorValue;

#ifdef _DEBUG
        // debug statistics
        unsigned int m_uiSumAccessSteps;
        unsigned int m_uiAccessesCount;
        double m_dAccessAverage;
        unsigned int m_uiMaxStepsToAccess;
        
        void DebugStatistics(unsigned int uiCount)
        {
            if (m_uiMaxStepsToAccess < uiCount)
            {
                m_uiMaxStepsToAccess = uiCount;
            }
            
            m_uiAccessesCount++;
            m_uiSumAccessSteps += uiCount;
            m_dAccessAverage = (double)m_uiSumAccessSteps / m_uiAccessesCount;
        }
        
    public:
        void PrintDebugStatistics(FILE* pOutFile)
        {
            fprintf(pOutFile, "Number of item: %d, Total size: %d, Accesses: %d\n", 
                m_uiCount, m_uiSize, m_uiAccessesCount);
            fprintf(pOutFile, "Max steps to access item: %d, Average: %.3f Item/Size: %.3f\n", 
                m_uiMaxStepsToAccess, m_dAccessAverage, (double)m_uiCount / m_uiSize);
        }
#endif //_DEBUG
    };


    /** 
    * @brief <b>MStrHashSync</b> hash table for string keys with synchronization on access.
    * 
    * This class should be used when synchronization is required by the user on accessing the 
    * hash.
    *
    * @param CHR - The type of the string in the hash key. Must be char or wchar_t.
    * @param T - The type of the data stored in the hash. Can be a primitive type or a pointer 
    *            to an object. Class type will cause a compilation error.
    */
    template <class CHR, class T>
    class MStrHashSync 
    {
    public:
        typedef bool (*THashIterationCallBack)(void *pParameter, const CHR* key, T data);

        MStrHashSync();
        virtual ~MStrHashSync();
        
        virtual TResult Init(bool bReplaceIfExist = false, T errorValue = (T)0, 
            unsigned int uiSize = HASH_DEFAULT_INITIAL_SIZE);
        
        virtual TResult Set(const CHR* key, T data, T* pOldData = NULL);
        virtual T Get(const CHR* key);
        virtual T Del(const CHR* key);
        
        virtual TResult Empty();
        virtual unsigned int Count();
        
        virtual TResult Iterate(THashIterationCallBack pCallback, void *pParameter);

    private:
        MStrHashSync(MStrHashSync<CHR, T>&) {} // preventing copy constructor
        
        MStrHashNoSync<CHR, T> m_Hash;
        Martini::OSA::IThreadSync *m_pThreadSync;

#ifdef _DEBUG   
    public:
        void PrintDebugStatistics(FILE* pOutFile)
        {
            m_Hash.PrintDebugStatistics(pOutFile);
        }
#endif //_DEBUG
    };


/************************************************************************/
/*                     MHashNoSync implementation                       */
/************************************************************************/

    template<class KEY, class T>
    MHashNoSync<KEY, T>::MHashNoSync() : m_ppHashArray(0), m_uiCount(0), m_bInIterator(false),
    m_bWasInit(false)
    {
#ifdef _DEBUG
        m_uiAccessesCount = 0;
        m_dAccessAverage = 0;
        m_uiMaxStepsToAccess = 0;
        m_uiSumAccessSteps = 0;
#endif //_DEBUG
    }

    template<class KEY, class T>
    MHashNoSync<KEY, T>::~MHashNoSync() 
    {
        Empty();
        if (m_ppHashArray)
        {
            delete [] m_ppHashArray;
        }
    }
    
    template<class KEY, class T>
    TResult MHashNoSync<KEY, T>::Init(bool bReplaceIfExist, T errorValue, 
        unsigned int uiSize, unsigned int uiShift)
    {
        if (m_bWasInit)
        {
            return MRTE_ERROR_ALREADY_INITIALIZED;
        }
        m_bReplaceIfExist = bReplaceIfExist;
        m_errorValue = errorValue;
        m_uiShift = uiShift;
        
        if (uiSize & (uiSize - 1)) // not a power of two
        {
            int iLastBit = 0;
            int iNumberOfBits = sizeof(unsigned int) * 8;
            int i = 0;
            for (i = 0; i < iNumberOfBits; ++i)
            {
                if (uiSize & 0x00000001)
                {
                    iLastBit = i;
                }
                uiSize >>= 1;
            }
            
            uiSize = 0x00000001;
            for (i = 0; i < iLastBit + 1; ++i)
            {
                uiSize <<= 1;
            }
        }

        m_uiSize = uiSize;

        m_ppHashArray = new SHashCell*[m_uiSize];
        if (!m_ppHashArray)
        {
            return MRTE_ERROR_OUT_OF_MEMORY;
        }

        memset(m_ppHashArray, 0, m_uiSize * sizeof(SHashCell*));
        m_bWasInit = true;
        return MRTE_RESULT_OK;
    }

    template<class KEY, class T>
    TResult MHashNoSync<KEY, T>::Set(const KEY key, T data, T* pOldData)
    {
        if (m_bInIterator)
        {
            return MRTE_ERROR_IN_ITERATOR;
        }
        if (!m_bWasInit)
        {
            return MRTE_ERROR_NOT_YET_INITIALIZED;
        }

        unsigned int uiPlace = (unsigned int)MHASH_HASH_FUNC(key, m_uiSize, m_uiShift);
        SHashCell *pCurrent = new SHashCell;
        if (!pCurrent)
        {
            return MRTE_ERROR_OUT_OF_MEMORY;
        }
        
        pCurrent->key = key;
        pCurrent->data = data;
        pCurrent->next = 0;
        TResult retVal = MRTE_RESULT_OK;
#ifdef _DEBUG
        unsigned int uiStepCount = 1;
#endif //_DEBUG
        
        if (m_ppHashArray[uiPlace] == NULL)
        {
            m_ppHashArray[uiPlace] = pCurrent;
        }
        else if (pCurrent->key < m_ppHashArray[uiPlace]->key)
        {
            pCurrent->next = m_ppHashArray[uiPlace];
            m_ppHashArray[uiPlace] = pCurrent;
        }
        else
        {
            SHashCell *pPrev, *pTemp = m_ppHashArray[uiPlace];
            while (pTemp && (pTemp->key < pCurrent->key))
            { 
                pPrev = pTemp;
                pTemp = pTemp->next;
#ifdef _DEBUG
                uiStepCount++;
#endif //_DEBUG
            }
            
            if (pTemp && pCurrent->key == pTemp->key)
            {
                if (m_bReplaceIfExist)
                {
                    if (pOldData)
                    {
                        *pOldData = pTemp->data;
                    }
                    pTemp->data = data;
                }
                retVal = MRTE_RESULT_KEY_ALREADY_EXISTS;
                
                delete pCurrent;
                m_uiCount--; // there will be an addition at the end
            }
            else
            {
                pPrev->next = pCurrent;
                pCurrent->next = pTemp;
            }
        }
        
        m_uiCount++;  
#ifdef _DEBUG
        DebugStatistics(uiStepCount);
#endif //_DEBUG
        return retVal;
    }

    template<class KEY, class T>
    T MHashNoSync<KEY, T>::Get(const KEY key)
    {
        if (m_bInIterator || !m_bWasInit)
        {
            return m_errorValue;
        }

#ifdef _DEBUG
        unsigned int uiStepCount = 1;
#endif //_DEBUG
        unsigned int uiPlace = (unsigned int)MHASH_HASH_FUNC(key, m_uiSize, m_uiShift);
        SHashCell *pCurrent = m_ppHashArray[uiPlace];
        
        if (pCurrent == 0)
        {
            return m_errorValue;
        }
        
        while (pCurrent && pCurrent->key < key) 
        {
            pCurrent = pCurrent->next;
#ifdef _DEBUG
            uiStepCount++;
#endif //_DEBUG
        }
        
#ifdef _DEBUG
        DebugStatistics(uiStepCount);
#endif //_DEBUG
        if (pCurrent == 0 || pCurrent->key != key)
        {
            return m_errorValue;
        }
        
        return pCurrent->data;
    }

    template<class KEY, class T>
    T MHashNoSync<KEY, T>::Del(const KEY key)
    {
        T retVat = m_errorValue;
        if (m_bInIterator || !m_bWasInit)
        {
            return retVat;
        }
        
        unsigned int uiPlace = (unsigned int)MHASH_HASH_FUNC(key, m_uiSize, m_uiShift);
        SHashCell *pCurrent = m_ppHashArray[uiPlace];
                        
        if (m_ppHashArray[uiPlace] == 0)
        {
            return retVat;
        }
        
        if (m_ppHashArray[uiPlace]->key == key)
        {
            m_ppHashArray[uiPlace] = m_ppHashArray[uiPlace]->next;
        }
        else
        {
            SHashCell *prev = 0;
            while (pCurrent && pCurrent->key < key) 
            {
                prev = pCurrent;
                pCurrent = pCurrent->next;
            }
            
            if (pCurrent && pCurrent->key == key)
            {
                prev->next = pCurrent->next;
            }
        }
        
        if (pCurrent == 0 || pCurrent->key != key) 
        {
            return retVat;
        }
        
        retVat = pCurrent->data;
        delete pCurrent;
        
        m_uiCount--;
        return retVat;
    }
        
    template<class KEY, class T>
    TResult MHashNoSync<KEY, T>::Empty()
    {
        if (m_bInIterator)
        {
            return MRTE_ERROR_IN_ITERATOR;
        }
        if (!m_bWasInit)
        {
            return MRTE_ERROR_NOT_YET_INITIALIZED;
        }

        if (m_ppHashArray)
        {
            for (unsigned int i = 0; i < m_uiSize; ++i)
            {
                SHashCell *pCurrent = m_ppHashArray[i];
                while (pCurrent)
                {
                    SHashCell *tmp = pCurrent;
                    pCurrent = pCurrent->next;
                    delete tmp;
                }
                m_ppHashArray[i] = NULL;
            }
        }
        m_uiCount = 0;
#ifdef _DEBUG
        m_uiAccessesCount = 0;
        m_dAccessAverage = 0;
        m_uiMaxStepsToAccess = 0;
        m_uiSumAccessSteps = 0;
#endif //_DEBUG
        return MRTE_RESULT_OK;
    }

    template<class KEY, class T>
    unsigned int MHashNoSync<KEY, T>::Count()
    {
        return m_uiCount;
    }
    
    template<class KEY, class T>
    TResult MHashNoSync<KEY, T>::Iterate(THashIterationCallBack pCallback, void *pParameter)
    {
        if (pCallback == NULL)
        {
            return MRTE_ERROR_ILLEGAL_ARGUMENT;
        }
                
        m_bInIterator = true;
        
        for (unsigned int i = 0; i < m_uiSize; ++i)
        {
            SHashCell *pCurrent = m_ppHashArray[i];
            while (pCurrent) 
            {
                if (!pCallback(pParameter, pCurrent->key, pCurrent->data))
                {
                    return MRTE_RESULT_ITERATION_STOPPED;
                }
                pCurrent = pCurrent->next;
            }
        }
        
        m_bInIterator = false;

        return MRTE_RESULT_OK;
    }


/************************************************************************/
/*                       MHashSync implementation                       */
/************************************************************************/

    template<class KEY, class T>
    MHashSync<KEY, T>::MHashSync() : m_pThreadSync(NULL)
    {}
    
    template<class KEY, class T> 
    MHashSync<KEY, T>::~MHashSync() 
    {
        if (m_pThreadSync)
        {
            m_pThreadSync->Destroy();
            m_pThreadSync = NULL;
        }
    }
    
    template<class KEY, class T>
    TResult MHashSync<KEY, T>::Init(bool bReplaceIfExist, T errorValue, 
        unsigned int uiSize, unsigned int uiShift)
    {
        if (m_pThreadSync)
        {
            return MRTE_ERROR_ALREADY_INITIALIZED;
        }

        m_pThreadSync = Martini::OSA::CreateThreadSync();
        if (!m_pThreadSync)
        {
            return MRTE_ERROR_OSA_FAILURE;
        }

        return m_Hash.Init(bReplaceIfExist, errorValue, uiSize, uiShift);
    }

    template<class KEY, class T>
    TResult MHashSync<KEY, T>::Set(const KEY key, T data, T* pOldData)
    {
        m_pThreadSync->Enter();
        TResult retVal = m_Hash.Set(key, data, pOldData);
        m_pThreadSync->Leave();
        return retVal;
    }

    template<class KEY, class T>
    T MHashSync<KEY, T>::Get(const KEY key)
    {
        m_pThreadSync->Enter();
        T retVal = m_Hash.Get(key);
        m_pThreadSync->Leave();
        return retVal;
    }

    template<class KEY, class T>
    T MHashSync<KEY, T>::Del(const KEY key)
    {
        m_pThreadSync->Enter();
        T retVal = m_Hash.Del(key);
        m_pThreadSync->Leave();
        return retVal;
    }
        
    template<class KEY, class T>
    TResult MHashSync<KEY, T>::Empty()
    {
        m_pThreadSync->Enter();
        TResult retVal = m_Hash.Empty();
        m_pThreadSync->Leave();
        return retVal;
    }

    template<class KEY, class T>
    unsigned int MHashSync<KEY, T>::Count()
    {
        m_pThreadSync->Enter();
        unsigned int uiCount = m_Hash.Count();
        m_pThreadSync->Leave();
        return uiCount;
    }
    
    template<class KEY, class T>
    TResult MHashSync<KEY, T>::Iterate(THashIterationCallBack pCallback, void *pParameter)
    {
        m_pThreadSync->Enter();
        TResult retVal = m_Hash.Iterate(pCallback, pParameter);
        m_pThreadSync->Leave();
        return retVal;
    }

/************************************************************************/
/*                     MStrHashNoSync implementation                       */
/************************************************************************/

    template<class CHR, class T>
    MStrHashNoSync<CHR, T>::MStrHashNoSync() : m_ppHashArray(0), m_uiCount(0), 
    m_bInIterator(false), m_bWasInit(false)
    {
#ifdef _DEBUG
        m_uiAccessesCount = 0;
        m_dAccessAverage = 0;
        m_uiMaxStepsToAccess = 0;
        m_uiSumAccessSteps = 0;
#endif //_DEBUG
    }

    template<class CHR, class T> 
    MStrHashNoSync<CHR, T>::~MStrHashNoSync() 
    {
        Empty();
        if (m_ppHashArray)
        {
            delete [] m_ppHashArray;
        }
    }
    
    template<class CHR, class T>
    TResult MStrHashNoSync<CHR, T>::Init(bool bReplaceIfExist, T errorValue, 
        unsigned int uiSize)
    {
        if (m_bWasInit)
        {
            return MRTE_ERROR_ALREADY_INITIALIZED;
        }
        m_bReplaceIfExist = bReplaceIfExist;
        m_errorValue = errorValue;
        
        bool bPowerOfTwo = true;
        if (uiSize & (uiSize - 1)) // not a power of two
        {
            bPowerOfTwo = false;
        }

        m_uiKeySize = 0;
        unsigned int uiNumberOfBits = sizeof(unsigned int) * 8;
        unsigned int i = 0;
        for (i = 0; i < uiNumberOfBits; ++i)
        {
            if (uiSize & 0x00000001)
            {
                m_uiKeySize = i;
            }
            uiSize >>= 1;
        }
        
        if (!bPowerOfTwo)
        {
            m_uiKeySize++;
        }
        
        m_uiSize = 0x00000001;
        for (i = 0; i < m_uiKeySize; ++i)
        {
            m_uiSize <<= 1;
        }

        m_ppHashArray = new SHashCell*[m_uiSize];
        if (!m_ppHashArray)
        {
            return MRTE_ERROR_OUT_OF_MEMORY;
        }

        memset(m_ppHashArray, 0, m_uiSize * sizeof(SHashCell*));
        m_bWasInit = true;
        return MRTE_RESULT_OK;
    }

    template<class CHR, class T>
    TResult MStrHashNoSync<CHR, T>::Set(const CHR* key, T data, T* pOldData)
    {
        if (m_bInIterator)
        {
            return MRTE_ERROR_IN_ITERATOR;
        }

        if (!m_bWasInit)
        {
            return MRTE_ERROR_NOT_YET_INITIALIZED;
        }

        unsigned int uiPlace = HashValue(key);
        SHashCell *pCurrent = new SHashCell;
        if (!pCurrent)
        {
            return MRTE_ERROR_OUT_OF_MEMORY;
        }
        
        pCurrent->key.Set(key);
        pCurrent->data = data;
        pCurrent->next = 0;
        TResult retVal = MRTE_RESULT_OK;        
#ifdef _DEBUG
        unsigned int uiStepCount = 1;
#endif //_DEBUG

        SHashCell *pTemp = m_ppHashArray[uiPlace];
        while (pTemp && (!pTemp->key.IsEqual(pCurrent->key)))
        { 
            pTemp = pTemp->next;
#ifdef _DEBUG
            uiStepCount++;
#endif //_DEBUG
        }

        if (pTemp) // the value is in the hash
        {
            if (m_bReplaceIfExist)
            {
                if (pOldData)
                {
                    *pOldData = pTemp->data;
                }
                pTemp->data = data;
            }
            retVal = MRTE_RESULT_KEY_ALREADY_EXISTS;
            
            delete pCurrent;
        }
        else // not in the hash - add to the beginning
        {
            pCurrent->next = m_ppHashArray[uiPlace];
            m_ppHashArray[uiPlace] = pCurrent;
            m_uiCount++;  
        }

#ifdef _DEBUG
        DebugStatistics(uiStepCount);
#endif //_DEBUG

        return retVal;
    }

    template<class CHR, class T>
    T MStrHashNoSync<CHR, T>::Get(const CHR* key)
    {
        if (m_bInIterator || !m_bWasInit)
        {
            return m_errorValue;
        }

        unsigned int uiPlace = HashValue(key);
        SHashCell *pCurrent = m_ppHashArray[uiPlace];
        
#ifdef _DEBUG
        unsigned int uiStepCount = 1;
#endif //_DEBUG
        while (pCurrent != NULL && !pCurrent->key.IsEqual(key)) 
        {
            pCurrent = pCurrent->next;
#ifdef _DEBUG
            uiStepCount++;
#endif //_DEBUG
        }
        
#ifdef _DEBUG
        DebugStatistics(uiStepCount);
#endif //_DEBUG
        if (pCurrent == NULL)
        {
            return m_errorValue;
        }
        
        return pCurrent->data;
    }

    template<class CHR, class T>
    T MStrHashNoSync<CHR, T>::Del(const CHR* key)
    {
        T retVat = m_errorValue;
        if (m_bInIterator || !m_bWasInit)
        {
            return retVat;
        }
        
        unsigned int uiPlace = HashValue(key);
        SHashCell *pCurrent = m_ppHashArray[uiPlace];
                        
        if (m_ppHashArray[uiPlace] == 0)
        {
            return retVat;
        }
        
        if (m_ppHashArray[uiPlace]->key.IsEqual(key))
        {
            m_ppHashArray[uiPlace] = m_ppHashArray[uiPlace]->next;
        }
        else
        {
            SHashCell *prev = 0;
            while (pCurrent != NULL && !pCurrent->key.IsEqual(key)) 
            {
                prev = pCurrent;
                pCurrent = pCurrent->next;
            }
            
            if (pCurrent != NULL)
            {
                prev->next = pCurrent->next;
            }
        }
        
        if (pCurrent == NULL) 
        {
            return retVat;
        }
        
        retVat = pCurrent->data;
        delete pCurrent;
        
        m_uiCount--;
        
        return retVat;
    }
        
    template<class CHR, class T>
    TResult MStrHashNoSync<CHR, T>::Empty()
    {
        if (m_bInIterator)
        {
            return MRTE_ERROR_IN_ITERATOR;
        }
        if (!m_bWasInit)
        {
            return MRTE_ERROR_NOT_YET_INITIALIZED;
        }

        if (m_ppHashArray)
        {
            for (unsigned int i = 0; i < m_uiSize; ++i)
            {
                SHashCell *pCurrent = m_ppHashArray[i];
                while (pCurrent)
                {
                    SHashCell *tmp = pCurrent;
                    pCurrent = pCurrent->next;
                    delete tmp;
                }
                m_ppHashArray[i] = NULL;
            }
        }
        m_uiCount = 0;
#ifdef _DEBUG
        m_uiAccessesCount = 0;
        m_dAccessAverage = 0;
        m_uiMaxStepsToAccess = 0;
        m_uiSumAccessSteps = 0;
#endif //_DEBUG
        return MRTE_RESULT_OK;
    }

    template<class CHR, class T>
    unsigned int MStrHashNoSync<CHR, T>::Count()
    {
        return m_uiCount;
    }
    
    template<class CHR, class T>
    TResult MStrHashNoSync<CHR, T>::Iterate(THashIterationCallBack pCallback, void *pParameter)
    {
        if (pCallback == NULL)
        {
            return MRTE_ERROR_ILLEGAL_ARGUMENT;
        }
                
        m_bInIterator = true;
        
        for (unsigned int i = 0; i < m_uiSize; ++i)
        {
            SHashCell *pCurrent = m_ppHashArray[i];
            while (pCurrent) 
            {
                if (!pCallback(pParameter, pCurrent->key.Get(), pCurrent->data))
                {
                    return MRTE_RESULT_ITERATION_STOPPED;
                }
                pCurrent = pCurrent->next;
            }
        }
        
        m_bInIterator = false;

        return MRTE_RESULT_OK;
    }

    template<class CHR, class T>
    unsigned int MStrHashNoSync<CHR, T>::HashValue(const CHR* szKey)
    {
        if (!szKey)
        {
            return 0;
        }

        // djb2 algorithm, see legal notice in the CVS legal directory
        unsigned int uiHashVal = 5381;

        while (*szKey)
        {
            uiHashVal = ((uiHashVal << 5) + uiHashVal) ^ *szKey++; // uiHashVal * 33 ^ c
        }
        
        return (uiHashVal & (m_uiSize - 1)); 
    }

/************************************************************************/
/*                       MStrHashSync implementation                       */
/************************************************************************/
    
    template<class CHR, class T>
    MStrHashSync<CHR, T>::MStrHashSync() : m_pThreadSync(NULL)
    {}
    
    template<class CHR, class T> 
    MStrHashSync<CHR, T>::~MStrHashSync() 
    {
        if (m_pThreadSync)
        {
            m_pThreadSync->Destroy();
            m_pThreadSync = NULL;
        }
    }
    
    template<class CHR, class T>
    TResult MStrHashSync<CHR, T>::Init(bool bReplaceIfExist, T errorValue, 
        unsigned int uiSize)
    {
        if (m_pThreadSync)
        {
            return MRTE_ERROR_ALREADY_INITIALIZED;
        }
        
        m_pThreadSync = Martini::OSA::CreateThreadSync();
        if (!m_pThreadSync)
        {
            return MRTE_ERROR_OSA_FAILURE;
        }
        
        return m_Hash.Init(bReplaceIfExist, errorValue, uiSize);
    }
    
    template<class CHR, class T>
    TResult MStrHashSync<CHR, T>::Set(const CHR* key, T data, T* pOldData)
    {
        m_pThreadSync->Enter();
        TResult retVal = m_Hash.Set(key, data, pOldData);
        m_pThreadSync->Leave();
        return retVal;
    }
    
    template<class CHR, class T>
    T MStrHashSync<CHR, T>::Get(const CHR* key)
    {
        m_pThreadSync->Enter();
        T retVal = m_Hash.Get(key);
        m_pThreadSync->Leave();
        return retVal;
    }
    
    template<class CHR, class T>
    T MStrHashSync<CHR, T>::Del(const CHR* key)
    {
        m_pThreadSync->Enter();
        T retVal = m_Hash.Del(key);
        m_pThreadSync->Leave();
        return retVal;
    }
    
    template<class CHR, class T>
    TResult MStrHashSync<CHR, T>::Empty()
    {
        m_pThreadSync->Enter();
        TResult retVal = m_Hash.Empty();
        m_pThreadSync->Leave();
        return retVal;
    }
    
    template<class CHR, class T>
    unsigned int MStrHashSync<CHR, T>::Count()
    {
        m_pThreadSync->Enter();
        unsigned int uiCount = m_Hash.Count();
        m_pThreadSync->Leave();
        return uiCount;
    }
    
    template<class CHR, class T>
    TResult MStrHashSync<CHR, T>::Iterate(THashIterationCallBack pCallback, void *pParameter)
    {
        m_pThreadSync->Enter();
        TResult retVal = m_Hash.Iterate(pCallback, pParameter);
        m_pThreadSync->Leave();
        return retVal;
    }

	    /** 
    * @brief <b>MByteArrHashNoSync</b> hash table for string keys with no syncharonization.
    * 
    * This class should be used when no syncharonization is required or when the user of this 
    * class prefers to syncharonize his/her code before accessing.
    *
    * @param char - The type of the string in the hash key. Must be char or wchar_t.
    * @param T - The type of the data stored in the hash. Can be a primitive type or a pointer 
    *            to an object. Class type will cause a compilation error.
    */
    template <class T>
    class MByteArrHashNoSync 
    {
    public:
        typedef bool (*THashIterationCallBack)(void *pParameter, const char* key, 
            unsigned int uiKeySize, T data);

        MByteArrHashNoSync();
        virtual ~MByteArrHashNoSync();
        
        virtual TResult Init(bool bReplaceIfExist = false, T errorValue = 0, 
            unsigned int uiSize = HASH_DEFAULT_INITIAL_SIZE);
        
        virtual TResult Set(const char* key, unsigned int uiKeySize, T data, T* pOldData);
        virtual T Get(const char* key, unsigned int uiKeySize);
        virtual T Del(const char* key, unsigned int uiKeySize);
        
        virtual TResult Empty();
        virtual unsigned int Count();
        
        virtual TResult Iterate(THashIterationCallBack pCallback, void *pParameter);
    private:
        // Internal struct for the hash maintenance array
        struct SHashCell
        {
            SHashCell *next;
            T data;
            char* key;
            unsigned int uiKeySize;
        };
        
        MByteArrHashNoSync(MByteArrHashNoSync<T>&) {} // preventing copy constructor

        unsigned int HashValue(const char* key, unsigned int keySize);
        
        SHashCell **m_ppHashArray;
        unsigned int m_uiCount;
        unsigned int m_uiSize;
        unsigned int m_uiKeySize;
        bool m_bReplaceIfExist;
        bool m_bInIterator;
        bool m_bWasInit;
        
        T m_errorValue;
    };


    /** 
    * @brief <b>MByteArrHashSync</b> hash table for byte array keys with synchronization on access.
    * 
    * This class should be used when synchronization is required by the user on accessing the 
    * hash.
    *
    * @param char - The type of the string in the hash key. Must be char or wchar_t.
    * @param T - The type of the data stored in the hash. Can be a primitive type or a pointer 
    *            to an object. Class type will cause a compilation error.
    */
    template <class T>
    class MByteArrHashSync 
    {
    public:
        typedef bool (*THashIterationCallBack)(void *pParameter, const char* key, 
            unsigned int uiKeySize, T data);

        MByteArrHashSync();
        virtual ~MByteArrHashSync();
        
        virtual TResult Init(bool bReplaceIfExist = false, T errorValue = 0, 
            unsigned int uiSize = HASH_DEFAULT_INITIAL_SIZE);
        
        virtual TResult Set(const char* key, unsigned int uiKeySize, T data, T* pOldData = NULL);
        virtual T Get(const char* key, unsigned int uiKeySize);
        virtual T Del(const char* key, unsigned int uiKeySize);
        
        virtual TResult Empty();
        virtual unsigned int Count();
        
        virtual TResult Iterate(THashIterationCallBack pCallback, void *pParameter);

    private:
        MByteArrHashSync(MByteArrHashSync<T>&) {} // preventing copy constructor
        
        MByteArrHashNoSync<T> m_Hash;
        //CRITICAL_SECTION       m_cs;
		Martini::OSA::IThreadSync *m_cs;
    };

/************************************************************************/
/*                     MByteArrHashNoSync implementation                       */
/************************************************************************/

    template<class T>
    MByteArrHashNoSync<T>::MByteArrHashNoSync() : m_ppHashArray(0), m_uiCount(0), 
    m_bInIterator(false), m_bWasInit(false)
    {
    }

    template<class T> 
    MByteArrHashNoSync<T>::~MByteArrHashNoSync() 
    {
        Empty();
        if (m_ppHashArray)
        {
            delete [] m_ppHashArray;
        }
    }
    
    template<class T>
    TResult MByteArrHashNoSync<T>::Init(bool bReplaceIfExist, T errorValue, 
        unsigned int uiSize)
    {
        if (m_bWasInit)
        {
            return MRTE_ERROR_ALREADY_INITIALIZED;
        }
        m_bReplaceIfExist = bReplaceIfExist;
        m_errorValue = errorValue;
        
        bool bPowerOfTwo = true;
        if (uiSize & (uiSize - 1)) // not a power of two
        {
            bPowerOfTwo = false;
        }

        m_uiKeySize = 0;
        unsigned int uiNumberOfBits = sizeof(unsigned int) * 8;
        unsigned int i = 0;
        for (i = 0; i < uiNumberOfBits; ++i)
        {
            if (uiSize & 0x00000001)
            {
                m_uiKeySize = i;
            }
            uiSize >>= 1;
        }
        
        if (!bPowerOfTwo)
        {
            m_uiKeySize++;
        }
        
        m_uiSize = 0x00000001;
        for (i = 0; i < m_uiKeySize; ++i)
        {
            m_uiSize <<= 1;
        }

        m_ppHashArray = new SHashCell*[m_uiSize];
        if (!m_ppHashArray)
        {
            return MRTE_ERROR_OUT_OF_MEMORY;
        }

        memset(m_ppHashArray, 0, m_uiSize * sizeof(SHashCell*));
        
        m_bWasInit = true;
        return MRTE_RESULT_OK;
    }

    template<class T>
    TResult MByteArrHashNoSync<T>::Set(const char* key, unsigned int uiKeySize, T data, T* pOldData)
    {
        if (m_bInIterator)
        {
            return MRTE_ERROR_IN_ITERATOR;
        }

        if (!m_bWasInit)
        {
            return MRTE_ERROR_NOT_YET_INITIALIZED;
        }

        unsigned int uiPlace = HashValue(key, uiKeySize);
        SHashCell *pCurrent = new SHashCell;
        if (!pCurrent)
        {
            return MRTE_ERROR_OUT_OF_MEMORY;
        }
        
        pCurrent->key = new char[uiKeySize];
        memcpy(pCurrent->key, key, uiKeySize);
        pCurrent->uiKeySize = uiKeySize;
        pCurrent->data = data;
        pCurrent->next = 0;
        TResult retVal = MRTE_RESULT_OK;        

        SHashCell *pTemp = m_ppHashArray[uiPlace];
		while (pTemp && ((pTemp->uiKeySize != pCurrent->uiKeySize) ||
                         memcmp(pTemp->key, pCurrent->key, pTemp->uiKeySize)))
        { 
            pTemp = pTemp->next;
        }

        if (pTemp) // the value is in the hash
        {
            if (m_bReplaceIfExist)
            {
                if (pOldData)
                {
                    *pOldData = pTemp->data;
                }
                pTemp->data = data;
            }
            retVal = MRTE_RESULT_KEY_ALREADY_EXISTS;
            
            delete pCurrent;
        }
        else // not in the hash - add to the beginning
        {
            pCurrent->next = m_ppHashArray[uiPlace];
            m_ppHashArray[uiPlace] = pCurrent;
            m_uiCount++; 
        }

        return retVal;
    }

    template<class T>
    T MByteArrHashNoSync<T>::Get(const char* key, unsigned int uiKeySize)
    {
        if (m_bInIterator || !m_bWasInit)
        {
            return m_errorValue;
        }

        unsigned int uiPlace = HashValue(key, uiKeySize);
        SHashCell *pCurrent = m_ppHashArray[uiPlace];

		while (pCurrent != NULL && ((pCurrent->uiKeySize != uiKeySize) ||
                                    memcmp(pCurrent->key, key, pCurrent->uiKeySize))) 
        {
			pCurrent = pCurrent->next;
        } 
        
		if (pCurrent == NULL)
        {
            return m_errorValue;
        }
        
        return pCurrent->data;
    }

    template<class T>
    T MByteArrHashNoSync<T>::Del(const char* key, unsigned int uiKeySize)
    {
        T retVat = m_errorValue;
        if (m_bInIterator || !m_bWasInit)
        {
            return retVat;
        }
        
        unsigned int uiPlace = HashValue(key, uiKeySize);
        SHashCell *pCurrent = m_ppHashArray[uiPlace];
                        
        if (m_ppHashArray[uiPlace] == 0)
        {
            return retVat;
        }
        
        if ((m_ppHashArray[uiPlace]->uiKeySize == uiKeySize) &&
            memcmp(m_ppHashArray[uiPlace]->key, key, uiKeySize))
        {
            m_ppHashArray[uiPlace] = m_ppHashArray[uiPlace]->next;
        }
        else
        {
            SHashCell *prev = 0;
            while (pCurrent != NULL && 
                ((pCurrent->uiKeySize != uiKeySize) || ! memcmp(pCurrent->key, key, uiKeySize))) 
            {
                prev = pCurrent;
                pCurrent = pCurrent->next;
            }
            
            if (pCurrent != NULL)
            {
                prev->next = pCurrent->next;
            }
        }
        
        if (pCurrent == NULL) 
        {
            return retVat;
        }
        
        retVat = pCurrent->data;
        delete pCurrent;
        
        m_uiCount--;
        
        return retVat;
    }
        
    template<class T>
    TResult MByteArrHashNoSync<T>::Empty()
    {
        if (m_bInIterator)
        {
            return MRTE_ERROR_IN_ITERATOR;
        }
        if (!m_bWasInit)
        {
            return MRTE_ERROR_NOT_YET_INITIALIZED;
        }

        if (m_ppHashArray)
        {
            for (unsigned int i = 0; i < m_uiSize; ++i)
            {
                SHashCell *pCurrent = m_ppHashArray[i];
                while (pCurrent)
                {
                    SHashCell *tmp = pCurrent;
                    pCurrent = pCurrent->next;
                    delete tmp;
                }
                m_ppHashArray[i] = NULL;
            }
        }
        m_uiCount = 0;
        return MRTE_RESULT_OK;
    }

    template<class T>
    unsigned int MByteArrHashNoSync<T>::Count()
    {
        return m_uiCount;
    }
    
    template<class T>
    TResult MByteArrHashNoSync<T>::Iterate(THashIterationCallBack pCallback, void *pParameter)
    {
        if (pCallback == NULL)
        {
            return MRTE_ERROR_ILLEGAL_ARGUMENT;
        }
                
        m_bInIterator = true;
        
        for (unsigned int i = 0; i < m_uiSize; ++i)
        {
            SHashCell *pCurrent = m_ppHashArray[i];
            while (pCurrent) 
            {
                if (!pCallback(pParameter, pCurrent->key, pCurrent->uiKeySize, pCurrent->data))
                {
                    return MRTE_RESULT_ITERATION_STOPPED;
                }
                pCurrent = pCurrent->next;
            }
        }
        
        m_bInIterator = false;

        return MRTE_RESULT_OK;
    }

    template<class T>
    unsigned int MByteArrHashNoSync<T>::HashValue(const char* key, unsigned int uiKeySize)
    {
        // djb2 algorithm, see legal notice in the CVS legal directory
        unsigned int uiHashVal = 5381, i;

        for (i = 0; i < uiKeySize; i++)
        {
            uiHashVal = ((uiHashVal << 5) + uiHashVal) ^ *key++; // uiHashVal * 33 ^ c
        }
        
        return (uiHashVal & (m_uiSize - 1)); 
    }

/************************************************************************/
/*                       MByteArrHashSync implementation                       */
/************************************************************************/
    
    template<class T>
    MByteArrHashSync<T>::MByteArrHashSync()
    {m_cs=NULL;}
    
    template<class T> 
    MByteArrHashSync<T>::~MByteArrHashSync() 
    {
		if (m_cs)
		{
            m_cs->Destroy ();
			m_cs = NULL;
		}
    }
    
    template<class T>
    TResult MByteArrHashSync<T>::Init(bool bReplaceIfExist, T errorValue, 
        unsigned int uiSize)
    {

		if (m_cs)
        {
            return MRTE_ERROR_ALREADY_INITIALIZED;
        }

		m_cs = Martini::OSA::CreateThreadSync();
        if (!m_cs)
        {
            return MRTE_ERROR_OSA_FAILURE;
        }
        
        return m_Hash.Init(bReplaceIfExist, errorValue, uiSize);
    }
    
    template<class T>
    TResult MByteArrHashSync<T>::Set(const char* key, unsigned int uiKeySize, T data, T* pOldData)
    {
        m_cs->Enter();
        TResult retVal = m_Hash.Set(key, uiKeySize, data, pOldData);
        m_cs->Leave();
        return retVal;
    }
    
    template<class T>
    T MByteArrHashSync<T>::Get(const char* key, unsigned int uiKeySize)
    {
        m_cs->Enter();
        T retVal = m_Hash.Get(key, uiKeySize);
        m_cs->Leave();
        return retVal;
    }
    
    template<class T>
    T MByteArrHashSync<T>::Del(const char* key, unsigned int uiKeySize)
    {
        m_cs->Enter();
        T retVal = m_Hash.Del(key, uiKeySize);
        m_cs->Leave();
        return retVal;
    }
    
    template<class T>
    TResult MByteArrHashSync<T>::Empty()
    {
        m_cs->Enter();
        TResult retVal = m_Hash.Empty();
        m_cs->Leave();
        return retVal;
    }
    
    template<class T>
    unsigned int MByteArrHashSync<T>::Count()
    {
        m_cs->Enter();
        unsigned int uiCount = m_Hash.Count();
        m_cs->Leave();
        return uiCount;
    }
    
    template<class T>
    TResult MByteArrHashSync<T>::Iterate(THashIterationCallBack pCallback, void *pParameter)
    {
        m_cs->Enter();
        TResult retVal = m_Hash.Iterate(pCallback, pParameter);
        m_cs->Leave();
        return retVal;
    }


} }
#endif // __MHASH_H__

