/*****************************************************************************
 * Copyright (c) 1997-2007, Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/

#ifndef MRTE_RTUTIL_MLIST
#define MRTE_RTUTIL_MLIST

#pragma warning(disable:4786)

#include <stdio.h>

#include "MRTEResults.h"
#include "MRTETypes.h"	
#include "MIterator.h"

namespace Martini { namespace RTUtil {

// forward declaration
template<class T> class MList;
template<class T> class MListNode;
template<class T> class MListIterator;
template<class T> class IterListNode;

// global functions

template<class T>
class MList
{
public:
	MList();
	MList(const T notFoundRetVal);
    ~MList();
	const TMRTEHandle		GetFirst() const;
	const TMRTEHandle		GetLast() const;	
	TMRTEHandle				GetNext(const TMRTEHandle handle) const;
	TMRTEHandle				GetPrev(const TMRTEHandle handle) const;
	const T					GetData(const TMRTEHandle handle) const;
	T						SetData(const TMRTEHandle handle, const T data);
	TMRTEHandle				InsertBefore(TMRTEHandle handle, const T data);
	TMRTEHandle				InsertAfter(TMRTEHandle handle, const T data);
    T                       DeleteItem(TMRTEHandle handle);
    IListIterator<T>*       GetListIterator();
    unsigned int            Count() const;

private:
    friend class MListIterator<T>;
    void FreeIterator(IterListNode<T> *pIterNode);

	MList(MList<T>&);
	MListNode<T>*           m_pHead;            // list head pointer
	MListNode<T>*		    m_pTail;            // list tail pointer
	T					    m_notFoundRetVal;
    IterListNode<T>*        m_pIterListHead;    // head of iterator list
    IterListNode<T>*        m_pIterListTail;    // tail of the iterator list
    unsigned int            m_uiItemCount;      // number of items in the list
};
		
template<class T>
class IterListNode
{
private:
    friend class MList<T>;

    IterListNode(IListIterator<T> *pIter): m_iter(pIter), m_pNext(NULL), m_pPrev(NULL) {}
    ~IterListNode() {}
    IListIterator<T> *m_iter;
    IterListNode *m_pNext;
    IterListNode *m_pPrev;
};

template<class T>
class MListNode
{
private:
	friend class MList<T>;
    friend class MListIterator<T>;

	MListNode();
	MListNode(const T data);
	~MListNode();
	T				m_data;
	MListNode<T>*	m_pPrev;
	MListNode<T>*	m_pNext;
};

template<class T>
class MListIterator : public IListIterator<T>
{
public:
    // IIterator methods
    virtual bool HasNext();
    virtual T GetNext();
    virtual void Free();

    // IListIterator methods
    virtual T GetFirst();
    virtual T GetLast();
    virtual bool HasPrev();
    virtual T GetPrev();
    virtual IListIterator<T>* Clone();

private:
    friend class MList<T>;

    MList<T> &m_list;               // the list on which the iterator iterates
    TMRTEHandle m_handle;           // the "next" item in the iteration
    IterListNode<T> *m_pIterNode;   // the node of the iterator in the list that owns it.
                                    // used when freeing the iterator

    // prevent manual construction. iterator objects can only be created using
    // the MList::GetListIterator() method
    MListIterator(MList<T> &theList);

};
	
/*********  global functions implementation  *********/ 

template<class T>
void FreeListItems(MList<T> &theList)
{

    TMRTEHandle h = theList.GetFirst();
    while (h != NULL)
    {
        T item = theList.GetData(h);
        delete item;
        h = theList.GetNext(h);
    }
}

/*********  MListIterator Implementation  *********/ 


template<class T>
MListIterator<T>::MListIterator(MList<T> &theList)
    : m_list(theList), m_pIterNode(NULL)
{
    m_handle = m_list.GetFirst();
}

template<class T>
IListIterator<T>* MListIterator<T>::Clone()
{
    IListIterator<T>* clone = new MListIterator<T>(m_list);
    ((MListIterator<T> *)clone)->m_handle = m_handle;
    return (clone);
}

template<class T>
bool MListIterator<T>::HasNext()
{
    return (m_handle != NULL);
}

template<class T>
T MListIterator<T>::GetNext()
{
    // get the current list item and advance the iterator to the next
    T item = m_list.GetData(m_handle);
    m_handle = m_list.GetNext(m_handle);
    return item;
}

template<class T>
void MListIterator<T>::Free()
{
    m_list.FreeIterator(m_pIterNode);
}

template<class T>
T MListIterator<T>::GetFirst()
{
    // return the first item. Note that we do not move the iterator to the next item
    // because GetNext takes care of this
    m_handle = m_list.GetFirst();
    T item = GetNext();
    return item;
}

template<class T>
T MListIterator<T>::GetLast()
{
    // return the last item. Note that we do not move the iterator to the previous item
    // because GetPrev takes care of this
    m_handle = m_list.GetLast();
    T item = m_list.GetData(m_handle);
    return item;
}

template<class T>
bool MListIterator<T>::HasPrev()
{
    // a previous item exists if the iterator does not point to the beginning of the list
    TMRTEHandle hFirstItem = m_list.GetFirst();
    return !(m_handle == hFirstItem);
}

template<class T>
T MListIterator<T>::GetPrev() 
{
    // implementation notes:
    // 1. m_handle always points to the list item that will be returned when calling
    //    GetNext. 
    // 2. to correctly implement GetPrev, we need to move the iterator
    //    to the previous item before returning the item it points to.
    // 3. first, we check whether a previous item exists. If not, we return NULL.
    // 4. if a previous item exists and m_handle is not null, we return the previous item
    // 5. if a previous item exists and m_handle is NULL, it means that the iterator
    //    has passed the end of the list. In this case we return the last item.
    bool bHasPrev = HasPrev();
    if (!bHasPrev)
    {
        return NULL;
    }

    T item;
    if (NULL == m_handle)
    {
        item = GetLast();
    }
    else
    {
        m_handle = m_list.GetPrev(m_handle);
        item = m_list.GetData(m_handle);
    }
    return item;
}

/*********  MList Implementation  *********/ 

/*
 *	MList - Constructor
 */
template<class T>
MList<T>::MList(const T notFoundRetVal)
: m_pHead(0), m_pTail(0), m_notFoundRetVal(notFoundRetVal),
    m_pIterListHead(0), m_pIterListTail(0), m_uiItemCount(0)
{}

/*
 *	MList - Default constructor
 */
template<class T>
MList<T>::MList()
: m_pHead(0), m_pTail(0), m_notFoundRetVal(0),
    m_pIterListHead(0), m_pIterListTail(0), m_uiItemCount(0)
{}

/*
 *	MList - Copy Constructor 
 *			The copy constructor is declared as private and does nothing. 
 *			Its purpose is to prevent the user from copying a list by mistake  
 */
template<class T>
MList<T>::MList(MList<T>&)
{}

/*
 *	~MList - Destructor
 *			 Go over the list nodes and free each node's memory space,
 *           and also free all list iterators.
 *
 */
template<class T>
MList<T>::~MList()
{
    // free list items
	MListNode<T>* pNode = m_pHead;
	MListNode<T>* pRemoveNode;
	while (pNode != NULL)
	{
		pRemoveNode = pNode;
		pNode = pNode->m_pNext;
		delete pRemoveNode;
	}

    // free list iterators
    IterListNode<T>* pIterNode = m_pIterListHead;
    IterListNode<T>* pIterToRemove;
    while (pIterNode != NULL)
    {
        pIterToRemove = pIterNode;
        pIterNode = pIterNode->m_pNext;
        FreeIterator(pIterToRemove);
    }
}

/*
 * FreeIterator - Deletes the given iterator.
 *                This function is called when the list is being destructed, or when
 *                an iterator which is owned by the list is being freed by the client.
 */
template<class T>
void MList<T>::FreeIterator(IterListNode<T> *pIterNode)
{
    // delete the iterator
    delete pIterNode->m_iter;
    // remove the iterator node from the iterator list
    IterListNode<T> *pNodeToRemove = pIterNode;
    if (pNodeToRemove == m_pIterListHead)
    {
        // removing the first iterator in the list
        m_pIterListHead = pNodeToRemove->m_pNext;
        if (m_pIterListHead != NULL)
        {
            m_pIterListHead->m_pPrev = NULL;
        }
    }
    else if (pNodeToRemove == m_pIterListTail)
    {
        // removing the last iterator in the list
        m_pIterListTail = pNodeToRemove->m_pPrev;
        if (m_pIterListTail != NULL)
        {
            m_pIterListTail->m_pNext = NULL;
        }
    }
    else
    {
        // removing an iterator from the middle of the list
        pNodeToRemove->m_pPrev->m_pNext = pNodeToRemove->m_pNext;
        pNodeToRemove->m_pNext->m_pPrev = pNodeToRemove->m_pPrev;
    }
    delete pNodeToRemove;
}

template<class T>
unsigned int MList<T>::Count() const
{
    return m_uiItemCount;
}

/*
 *	GetFirst - Returns a handle to the first element in the list.
 *			   If the list is empty, NULL is returned
 */
template<class T>
const TMRTEHandle MList<T>::GetFirst() const
{
	return (TMRTEHandle)m_pHead;
}

/*
 *	GetLast - Returns a handle to the last element in the list.
 *			  If the list is empty, NULL is returned
 */
template<class T>
const TMRTEHandle MList<T>::GetLast() const
{
	return (TMRTEHandle)m_pTail;
}

/*
 *	GetNext - Returns a handle to the next element in the list 
 *			  relative to the given handle.
 *			  If the handle is NULL - returns NULL
 */
template<class T>
TMRTEHandle MList<T>::GetNext(const TMRTEHandle handle) const
{
	if (handle == NULL)
	{
		return (TMRTEHandle)handle;
	}
	else
	{
		MListNode<T>* pCurrNode = (MListNode<T>*)handle;
		MListNode<T>* pNextNode = pCurrNode->m_pNext;
		return (TMRTEHandle)pNextNode; 
	}
}

/*
 *	GetPrev - Returns a handle to the previous element in the list 
 *			  relative to the given handle.
 *			  If the handle is NULL - returns NULL
 */
template<class T>
TMRTEHandle MList<T>::GetPrev(const TMRTEHandle handle) const
{
	if (handle == NULL)
	{
		return (TMRTEHandle)handle;
	}
	else
	{
		MListNode<T>* pCurrNode = (MListNode<T>*)handle;
		MListNode<T>* pPrevNode = pCurrNode->m_pPrev;
		return (TMRTEHandle)pPrevNode; 
	}
}

/*
 *	GetData - Returns the data associated with the given handle
 *			  If the handle is NULL - returns NULL
 */
template<class T>
const T MList<T>::GetData(const TMRTEHandle handle) const
{
	if (handle == NULL)
	{
		return m_notFoundRetVal;
	}
	else
	{
		MListNode<T>* pNode = (MListNode<T>*)handle;
		T data = pNode->m_data;
		return data; 
	}
}

/*
 *	SetData - Sets new data to the element represented by the given handle.
 *			  handle - the element which data is to be updated
 *			  data	 - the new data to be set
 *			  Returns the old data.
 *			  If handle is NULL - returns the "not found" return value
 */
template<class T>
T MList<T>::SetData(const TMRTEHandle handle, const T data)
{
	T oldValue = m_notFoundRetVal;
	if (handle != NULL)
	{
		MListNode<T>* pNode = (MListNode<T>*)handle;
		oldValue = pNode->m_data;
		pNode->m_data = data; 
	}
	return oldValue;
}

/*
 *	InsertAfter - Inserts an element to the list after a given elemnt
 *				  handle - the element after which the new element should be added
 *				  data - the new added element data
 *				  If handle is NULL and the list is empty adds the new and only 
 *				  element to the list.
 *				  If handle is NULL and the list is not empty returns NULL
 */
template<class T>
TMRTEHandle MList<T>::InsertAfter(TMRTEHandle handle, const T data)
{
	MListNode<T>* pCurrNode = (MListNode<T>*)handle;
	MListNode<T>* pNewNode	= new MListNode<T>(data);
	if (pNewNode == NULL)
	{
        m_uiItemCount++;
		return (TMRTEHandle)pNewNode;
	}
	// pCurrNode == NULL => Insert to end of list
	if (pCurrNode == NULL)
	{
		// empty list
		if (m_pTail == NULL)
		{
			m_pHead = m_pTail = pNewNode;
		}
		// non empty list
		else
		{
			return NULL;
		}
		
	}
	// insert element after pCurrNode
	else
	{
		pNewNode->m_pNext	= pCurrNode->m_pNext;
		pNewNode->m_pPrev	= pCurrNode;
		if (pCurrNode != m_pTail)
		{
			pCurrNode->m_pNext->m_pPrev = pNewNode;
		}		
		else
		{
			m_pTail = pNewNode;
		}
		pCurrNode->m_pNext	= pNewNode;
	}
    m_uiItemCount++;
	return (TMRTEHandle)pNewNode;
}
 
/*
 *	InsertBefore - Inserts an element to the list before a given element
 *				   handle - the element before which the new element should be added
 *				   data - the new added element data
 *				   If handle is NULL and the list is empty adds the new and only 
 *				   element to the list.
 *				   If handle is NULL and the list is not empty returns NULL
 */
template<class T>
TMRTEHandle MList<T>::InsertBefore(TMRTEHandle handle, const T data)
{
	MListNode<T>* pCurrNode = (MListNode<T>*)handle;
	MListNode<T>* pNewNode	= new MListNode<T>(data);
	if (pNewNode == NULL)
	{
        m_uiItemCount++;
		return (TMRTEHandle)pNewNode;
	}
	// pCurrNode == NULL => Insert to head of list
	if (pCurrNode == NULL)
	{
		// empty list
		if (m_pHead == NULL)
		{
			m_pHead = m_pTail = pNewNode;
		}
		// non empty list
		else
		{
			return NULL;
		}
		
	}
	// insert element after pCurrNode
	else
	{
		pNewNode->m_pPrev	= pCurrNode->m_pPrev;
		pNewNode->m_pNext	= pCurrNode;
		if (pCurrNode != m_pHead)
		{
			pCurrNode->m_pPrev->m_pNext = pNewNode;
		}		
		else
		{
			m_pHead = pNewNode;
		}
		pCurrNode->m_pPrev	= pNewNode;
	}
    m_uiItemCount++;
	return pNewNode;
}

/*
 *	GetListIterator - Creates and returns a list iterator.
 *                    The iterator object is owned by the list and will be freed
 *                    when the list is freed. Do not use delete with the returned
 *                    iterator.
 */
template<class T>
IListIterator<T>* MList<T>::GetListIterator()
{
    MListIterator<T> *pIter = new MListIterator<T>(*this);
    // store the iterator in an internal list so it can be freed when the list is freed
    IterListNode<T> *pNewNode = new IterListNode<T>(pIter);
    pIter->m_pIterNode = pNewNode;
    if (NULL == m_pIterListHead)
    {
        // first iterator in the list
        m_pIterListHead = pNewNode;
        m_pIterListTail = m_pIterListHead;
    }
    else
    {
        // add iterator to the end of the list
        pNewNode->m_pPrev = m_pIterListTail;
        m_pIterListTail->m_pNext = pNewNode;
        m_pIterListTail = pNewNode;
    }
    return pIter;
}

template<class T>
T MList<T>::DeleteItem(TMRTEHandle handle)
{
    //TODO: need to invalidate any iterator which points to the item being deleted
    MListNode<T>* pCurrNode = (MListNode<T>*)handle;
    T data = pCurrNode->m_data;

    if (pCurrNode->m_pNext)
    {
        pCurrNode->m_pNext->m_pPrev = pCurrNode->m_pPrev;
    }
    if (pCurrNode->m_pPrev)
    {
        pCurrNode->m_pPrev->m_pNext = pCurrNode->m_pNext;
    }

    if (m_pHead == pCurrNode)
    {
        m_pHead = pCurrNode->m_pNext;
    }
    if (m_pTail == pCurrNode)
    {
        m_pTail = pCurrNode->m_pPrev;
    }

    delete pCurrNode;
    m_uiItemCount--;
    return data;
}


/*********  MListNode Implementation  *********/ 

/*
 *	MListNode - Default Constructor.
 *				The default constructor is declared as private and does nothing. 
 *				Its purpose is enable only the MList objects (frind class) to access it
 */
template<class T>
MListNode<T>::MListNode()
: m_pPrev(0), m_pNext(0)
{}

/*
 *	MListNode - Constructor
 *				Generates a new MListNode object and initiate its data 
 */
template<class T>
MListNode<T>::MListNode(const T data)
: m_pPrev(0), m_pNext(0), m_data(data)
{}

/*
 *	MListNode - Destructor
 */
template<class T>
MListNode<T>::~MListNode()
{}

} /* namespace Martini */} /* namespace RTUtil */

#endif // MRTE_RTUTIL_MLIST





















