/*****************************************************************************
 * Copyright (c) 1997-2007, Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/
//
#ifndef MRTE_RTUTILS_MSTRING
#define MRTE_RTUTILS_MSTRING

#include "MRTEResults.h"
#include <memory.h>
#include <stdio.h>


namespace Martini { namespace RTUtil 
{

// MString - exception free, auto expanding string template.
//    designed to support both 1 and 2 bytes strings.
//    conversion between strings will be provides via RTUtils class
//    return values for TResult methods:
//       MRTE_ERROR_ILLEGAL_ARGUMENT - illegal argument
//       MRTE_ERROR_NO_MEM - if fails to allocate required memory
//       MRTE_RESULT_OK - when all is well
// API -
//    Set - initializes the string with [str]
//    Append - appends a [str] to the current string
//    Get - returns a const version of the internal string
//    Len - returns the length of the string in T units
//    Find - looks for a [str] in the MString starting from a [start](defaulted to 0). 
//           returns the offset in units of T from the beginning of the string 
//           (-1 if not found)
//    StartsWith - checks if the MString starts with a given substring ([str]).
//    IsEqual - compares the MString to a given string ([str]). the comparison is done using
//              T's 'operator='
//    ReplaceAll - replace all occurances of [str1] by [str2]. the MString might grow or
//                 shrink as needed (if len([str1]) != len([str2]))
//    Empty - will release the internal string
//    Erase - will delete [size] units of T starting from [start] (zero based). 
//            the MString will shrink.
template<class T>
class MString
{
public:
    MString<T>();
    MString<T>(const MString<T> &str);
    explicit MString(const T* sz);
    ~MString<T>();
    TResult          Set (const T* str);
    TResult          Set (const MString<T>& str);
    TResult          Append (const T* str);
    TResult          Append (const MString<T>& str);
    const T*            Get () const;
    unsigned int        Len () const;
    int                 Find (const T* str, unsigned int start = 0) const;
    int                 Find (const MString<T>& str, unsigned int start = 0) const;
	int                 FindReverse (const T chr, unsigned int startPos = (unsigned int)(-1)) const;
    bool                StartsWith (const T* str) const;
    bool                StartsWith (const MString<T>& str) const;
    bool                IsEqual (const T* str) const;
    bool                IsEqual (const MString<T>& str) const;
    TResult             ReplaceAll (const T* str1, const T* str2, int *numReplacements = NULL);
    void                Empty ();
    TResult             Erase (unsigned int start, unsigned int size);
	void                Truncate (int pos);
	void                ReInit();
private:
    unsigned int        m_uiLen;
	unsigned int        m_uiAllocated;
    T*                  m_pStr;

    unsigned int        StringLen (const T*) const;
    void                Free ();
    TResult             _Set (const T* str, unsigned int len);
    TResult             _Append (const T* str, unsigned int len);
    bool                _StartsWith (const T* str, unsigned int len) const;
    bool                _IsEqual (T* str1, unsigned int len1, 
                                  const T* str2, unsigned int len2) const;
    inline int          _Find (const T* str, unsigned int len, unsigned int start) const;
};

typedef MString<char> MCString;
typedef MString<unsigned short> MWString;

// public functions:

//
// Constructor
// Construct an empty MString object
//
template<class T>
MString<T>::MString() : m_uiLen(0), m_uiAllocated(0), m_pStr(NULL)
{
}

//
// Constructor
// Constructs an MString object from a regular string
//
template<class T>
MString<T>::MString(const T* sz) : m_uiLen(0), m_uiAllocated(0), m_pStr(NULL)
{
    Set(sz);
}

//
// Copy constructor. Allows MString to be passed by value
//
template <class T>
MString<T>::MString(const MString<T> &str) : m_uiLen(0), m_uiAllocated(0), m_pStr(NULL)
{
    Set(str);
}

//
// Destructor
//
template<class T>
MString<T>::~MString()
{
    Free();
}

template<class T>
TResult MString<T>::Set (const T* str)
{
    unsigned int size = StringLen (str) + 1;
    TResult res = _Set (str, size);
    return res;
}

template<class T>
TResult MString<T>::Set (const MString<T>& str)
{
    TResult res = _Set (str.m_pStr, str.Len() + 1);
    return res;
}

template<class T>
TResult MString<T>::Append (const T* str)
{
    unsigned int len = StringLen (str);
    TResult res = _Append (str, len);
    return res;
}

template<class T>
TResult MString<T>::Append (const MString<T>& str)
{
    unsigned int len = str.Len();
    TResult res = _Append (str.Get(), len);
    return res;
}

template<class T>
const T* MString<T>::Get() const
{
    return m_pStr;
}

template<class T>
unsigned int MString<T>::Len () const
{
    return m_uiLen;
}

template<class T>
int MString<T>::Find (const T* str, unsigned int start) const
{
    if (!str)
    {
        return -1;
    }
    unsigned int len = StringLen(str);
    return _Find(str, len, start);
}



template<class T>
int MString<T>::FindReverse (const T chr, unsigned int startPos) const
{
    int i;
    
	if (startPos==(unsigned int)-1)
	{
        startPos = m_uiLen;
	}
	if (m_uiLen==0 || startPos>m_uiLen)
	{
		return -1;
	}
	for (i = startPos; i>=0; i--)
	{
		if (m_pStr[i] == chr)
		{
			return (i);
		}
	}
	return -1;
}

template<class T>
int MString<T>::Find (const MString<T>& str, unsigned int start) const
{
    if (! str.Get())
    {
        return -1;
    }
    unsigned int len = str.Len();
    return _Find (str.Get(), len, start);
}

template<class T>
bool MString<T>::StartsWith (const T* str) const
{
    unsigned int len = StringLen (str);
    return _StartsWith (str, len);
}

template<class T>
bool MString<T>::StartsWith (const MString<T>& str) const
{
    unsigned int len = str.Len();
    return _StartsWith (str.Get(), len);
}

template<class T>
bool MString<T>::IsEqual (const T* str) const
{
    unsigned int len = StringLen (str);
    return _IsEqual (m_pStr, m_uiLen, str, len);
}

template<class T>
bool MString<T>::IsEqual (const MString<T>& str) const
{
    unsigned int len = str.Len();
    return _IsEqual (m_pStr, m_uiLen, str.Get(), len);
}

template<class T>
TResult MString<T>::ReplaceAll (const T* str1, const T* str2, int *numReplacements)
{
	if (numReplacements)
    {
        *numReplacements = 0;
    }

    unsigned int len1 = StringLen (str1);
    if (len1 == 0)
    {
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }
    unsigned int len2 = StringLen (str2);
    if (len1 != len2)
    {
        unsigned int pos, count = 0;
        for (pos = 0; pos < m_uiLen - len1; )
        {
            if (_IsEqual (&m_pStr[pos], len1, str1, len1))
            {
                count++;
                pos += len1;
                if (numReplacements)
                {
				    (*numReplacements)++;
                }
            }
            else
            {
                pos++;
            }
        }
        if (numReplacements)
        {
		    if ((*numReplacements) == 0)
		    {
			    return MRTE_RESULT_OK;
		    }
        }
        unsigned int uiNewSize = m_uiLen - count * (len1 - len2) + 1;
        T* tmp = new T[uiNewSize];
        if (!tmp)
        {
            return MRTE_ERROR_OUT_OF_MEMORY;
        }
        unsigned int newPos;
        for (pos = 0, newPos = 0; pos < m_uiLen; )
        {
            if ((pos<(m_uiLen-len1)) && _IsEqual (&m_pStr[pos], len1, str1, len1))
            {
                memcpy(&tmp[newPos], str2, len2);
                pos += len1;
                newPos += len2;
            }
            else
            {
                tmp[newPos] = m_pStr[pos];
                pos++;
                newPos++;
            }
        }
		tmp[newPos] = 0;
        Free();
        m_pStr = tmp;
        m_uiLen = uiNewSize - 1;
		m_uiAllocated = uiNewSize;
    }
    else // can use same string
    {
        unsigned int pos;
        for (pos = 0; pos < m_uiLen - len1; )
        {
            if (_IsEqual (&m_pStr[pos], len1, str1, len1))
            {
                memcpy(&m_pStr[pos], str2, len2);
                pos += len1;
                if (numReplacements)
                {
				    (*numReplacements)++;
                }
            }
            else
            {
                pos++;
            }
        }
    }
    return MRTE_RESULT_OK;
}

template<class T>
void MString<T>::Empty()
{
    Free();
}

template<class T>
TResult MString<T>::Erase(unsigned int start, unsigned int size)
{
    if (start + size > m_uiLen)
    {
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }
    if (start == 0 && size == m_uiLen)
    {
        Free();
        return MRTE_RESULT_OK;
    }
    unsigned int uiNewSize = m_uiLen - size;
    T* tmp = new T[uiNewSize + 1];
    if(!tmp)
    {
        return MRTE_ERROR_OUT_OF_MEMORY;
    }
    memcpy(tmp, m_pStr, sizeof(T) * start);
    memcpy(&tmp[start], &m_pStr[start+size], sizeof(T) * (m_uiLen - start - size + 1));
    Free();
    m_pStr = tmp;
    m_uiLen = uiNewSize;
	m_uiAllocated = uiNewSize + 1;
    return MRTE_RESULT_OK;
}

template<class T>
void MString<T>::Truncate (int pos)
{
	m_pStr[pos] = 0;
	m_uiLen = StringLen(m_pStr);
}


/*
ReInit used when there is a structure copy of a structure that has
MString as a field - the structure copy is done as a memcpy
and it is desired for the destination MString to be the same string as the source
but not point at the same string
*/
template<class T>
void MString<T>::ReInit ()
{
	m_pStr = NULL;
	m_uiLen = m_uiLen = 0;
}

/////////////////////////////////////////////////////
//          private MString methods                //
/////////////////////////////////////////////////////
template<class T>
unsigned int MString<T>::StringLen (const T* str) const
{
    if (!str)
    {
        return 0;
    }
    unsigned int res = 0;
    while(str[res] != 0)
    {
        res++;
    }
    return res;
}

template<class T>
void MString<T>::Free()
{
    if (m_pStr)
    {
        delete[] m_pStr;
    }
    m_pStr = NULL;
    m_uiLen = m_uiAllocated = 0;
}

template<class T>
inline TResult MString<T>::_Set (const T* str, unsigned int size)
{
    if (!str)
    {
        Free();
        return MRTE_RESULT_OK;
    }
    T* tmp = new T[size];
    if (!tmp)
    {
        return MRTE_ERROR_OUT_OF_MEMORY;
    }
    Free();
    memcpy(tmp, str, size * sizeof(T));
    m_pStr = tmp;
    m_uiLen = size - 1;
	m_uiAllocated = size;
    return MRTE_RESULT_OK;
}

template<class T>
TResult MString<T>::_Append (const T* str, unsigned int len)
{
    if (len == 0)
    {
        return MRTE_RESULT_OK;
    }
    unsigned int uiNewSize = m_uiLen + len + 1;
	if (uiNewSize < m_uiAllocated)
    {
        memcpy (&m_pStr[m_uiLen], str, len * sizeof(T));
		m_uiLen += len;
		m_pStr[m_uiLen] = 0;
		return MRTE_RESULT_OK;
    }
    T* tmp = new T[uiNewSize*2];//so that next append(s) have chance to be done without re-allocation
    if (!tmp)
    {
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }
    memcpy (tmp, m_pStr, m_uiLen * sizeof(T));
    memcpy (&tmp[m_uiLen], str, len * sizeof(T));
    tmp[uiNewSize-1] = 0;
    Free();
    m_pStr = tmp;
    m_uiLen = uiNewSize - 1;
	m_uiAllocated = uiNewSize*2;
    return MRTE_RESULT_OK;
}

template<class T>
inline int MString<T>::_Find (const T* str, unsigned int len, unsigned int start) const
{
    if (len > m_uiLen - start)
    {
        return -1;
    }
    unsigned int pos;
    for (pos = start; pos <= m_uiLen - len; pos++)
    {
        if (_IsEqual(&m_pStr[pos], len, str, len))
        {
            return pos;
        }
    }
    return -1;
}

template<class T>
inline bool   MString<T>::_StartsWith (const T* str, unsigned int len) const
{
    if (len == 0)
    {
        return true;
    }
    if (len > m_uiLen)
    {
        return false;
    }
    unsigned int pos;
    for (pos = 0; pos < len; pos++)
    {
        if (m_pStr[pos] != str[pos])
        {
            return false;
        }
    }
    return true;
}


template<class T>
inline bool   MString<T>::_IsEqual (T* str1, unsigned int len1,
                                    const T* str2, unsigned int len2) const
{
    if (len1 != len2)
    {
        return false;
    }
    unsigned int pos;
    for (pos=0; pos < len1; pos++)
    {
        if (str1[pos] != str2[pos])
        {
            return false;
        }
    }
    return true;
}




} }
#endif //    MRTE_RTUTILS_MSTRING
