/*****************************************************************************
 * Copyright (c) 1997-2007, Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/
//

#ifndef MRTE_RTUTIL_MVECTOR
#define MRTE_RTUTIL_MVECTOR

#include "MRTEResults.h"
#include <memory.h>
#include <stdio.h>

namespace Martini { namespace RTUtil 
{

// Vector - exception free, auto expanding T array. use with basic types 
// API - 
//      ctor - sets the limits of expanding the array: the exponential increment limit
//             is the point when we stop multiplying the vector size by 2 each time push 
//             needs more space, and linear increment size is the size we add to the array
//             from after we pass the exponential increment limit.
//             both shold be a power of 2 and will be set to MSB of its input
//      Reserve - tries to allocate [size] units of T and if successfull reallocate the
//                array to its new location. the reserved space will be set to NULL.
//      Push - adds an element to the end of the vector (resizing if needed)
//      GetAt - will return the element at [index]
//      SetAt - sets entry [index] to [elem]. valid [index] values are 0 to [m_uiCapacity], 
//              and [m_uiSize] is upadtes if [index] > [m_uiSize]
//      Capacity - returns the allocated size of the vector
//      Size - returns the number of elements in the vector
//      GetVector - returns a const version of the internal array for fast traversal
//      Clear - Erases the elements of the vector and sets its size to 0. Memory is preserved
//              and the capcity is not changed.
//      DeleteElements - goes over the vector and frees each element for 0 to m_uiSize

template<class T>
class MVector
{
public:
    MVector<T>(unsigned int VectorExponentialIncrementLimit = 0x0100000, 
               unsigned int VectorLinearIncrementSize = 0x0100000);
    ~MVector<T>();
    TResult         Reserve(unsigned int size);
    TResult         Push(const T& elem);
    T               Pop();
    TResult         SafeGetAt(unsigned int index, T* elem) const;
    T&              GetAt(unsigned int index) const;
    TResult         SetAt(unsigned int index, const T& elem);
    unsigned int    Capacity() const;
    unsigned int    Size() const;
    const T*        GetVector() const;
    TResult         Clear();
//    void            DeleteElements();
private:
    T*              m_pArray;
    unsigned int    m_uiSize;
    unsigned int    m_uiCapacity;
    unsigned int    m_uiVectorExponentialIncrementLimit;
    unsigned int    m_uiVectorLinearIncrementSize;
};


template<class T>
MVector<T>::MVector(unsigned int VectorExponentialIncrementLimit, 
                    unsigned int VectorLinearIncrementSize)
{
     m_pArray =NULL;
     m_uiSize =0;
     m_uiCapacity =0;
     m_uiVectorExponentialIncrementLimit = VectorExponentialIncrementLimit;
     unsigned int left;
     for(left = 0x80000000; left > 0; left = left >> 1)
     {
         if(VectorExponentialIncrementLimit & left)
         {
             m_uiVectorExponentialIncrementLimit = left;
             break;
         }
     }
     for(left = 0x80000000; left > 0; left = left >> 1)
     {
         if(VectorLinearIncrementSize & left)
         {
             m_uiVectorLinearIncrementSize = left;
             break;
         }
     }
     if(m_uiVectorExponentialIncrementLimit > 0x0800000)
     {
         m_uiVectorExponentialIncrementLimit = 0x800000;
     }
     if(m_uiVectorLinearIncrementSize > 0x0800000)
     {
         m_uiVectorLinearIncrementSize = 0x0800000;
     }
     if(m_uiVectorLinearIncrementSize < m_uiVectorExponentialIncrementLimit >> 0x8)
     {
         m_uiVectorLinearIncrementSize = m_uiVectorExponentialIncrementLimit >> 0x8;
     }
}

//template<class T>
//void MVector<T>::DeleteElements()
//{
//    unsigned int i;
//    for(i=0;i<m_uiSize;i++)
//    {
//        delete m_pArray[i];
//    }
//    memset(m_pArray, 0, m_uiSize * sizeof(T));
//}
//
template<class T>
MVector<T>::~MVector()
{
    if(m_pArray)
    {
        delete[] m_pArray;
    }
}

template<class T>
TResult MVector<T>::Reserve(unsigned int size)
{
    if(size <= m_uiCapacity)
    {
        return MRTE_RESULT_OK;
    }
    T* tmp = new T[size];
    if(!tmp)
    {
        return MRTE_ERROR_OUT_OF_MEMORY;
    }
    memset(tmp, 0, sizeof(T) * size);
    unsigned int i;
    for(i=0;i<m_uiSize;i++)
    {
        tmp[i] = m_pArray[i];
    }
    delete[] m_pArray;
    m_pArray = tmp;
    m_uiCapacity = size;
    return MRTE_RESULT_OK;
}

template<class T>
TResult      MVector<T>::Push(const T& elem)
{
    TResult res;
    if(m_uiSize >= m_uiCapacity)
    {
        if(m_uiCapacity == 0)
        {
            res = Reserve(16);
        }
        else
        {
            if(m_uiCapacity < m_uiVectorExponentialIncrementLimit)
            {
                res = Reserve(m_uiCapacity * 2);
            }
            else
            {
                res = Reserve(m_uiCapacity + m_uiVectorLinearIncrementSize);
            }
        }
        if(res != MRTE_RESULT_OK)
        {
            return res;
        }
    }
    m_pArray[m_uiSize] = elem;
    m_uiSize++;
    return MRTE_RESULT_OK;
}

template<class T>
T MVector<T>::Pop()
{
    //TODO: bug fix: Calling Pop on an empty array (m_uiSize == 0) will cause an error!
    T element = m_pArray[m_uiSize - 1];
    m_uiSize--;
    
    return element;
}

template<class T>
T& MVector<T>::GetAt(unsigned int index) const
{
    return m_pArray[index];
}

template<class T>
TResult MVector<T>::SafeGetAt(unsigned int index, T* elem) const
{
    if(index >= m_uiSize)
    {
        elem = NULL;
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }
    *elem =  m_pArray[index];
    return MRTE_RESULT_OK;
}

template<class T>
TResult MVector<T>::SetAt(unsigned int index, const T& elem)
{
    if(index >= m_uiCapacity)
    {
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }
    m_pArray[index] = elem;
    if(index > m_uiSize)
    {
        m_uiSize = index + 1;
    }
    return MRTE_RESULT_OK;
}

template<class T>
unsigned int    MVector<T>::Capacity() const
{
    return m_uiCapacity;
}

template<class T>
unsigned int    MVector<T>::Size() const
{
    return m_uiSize;
}

template<class T>
const T* MVector<T>::GetVector() const
{
    return m_pArray;
}

template<class T>
TResult MVector<T>::Clear()
{
    if (m_uiCapacity > 0)
    {
        memset(m_pArray, 0, sizeof(T) * m_uiCapacity);
    }
    m_uiSize = 0;
    return MRTE_RESULT_OK;
}

} }

#endif //   MRTE_RTUTIL_MVECTOR
