/*****************************************************************************
 * Copyright (c) 1997, 2010 Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/

#include "CGAdaptor.h"
#include "WideStringUtils.h"
#include "CGProxy.h"
#include "HeapProxy.h"
#include "ThreadProxy.h"
#include "ValidityChecks.h"
#include "MRTEInfrastructureDefinitions.h"

#include <assert.h>
#include <string.h>

#define JIE_FIRST_METHOD_ID 0x10000 // first ids are reserved for threads in the prf resolution

using namespace Martini;
using namespace CGAdaptor;
using namespace JIE;
using namespace MIE;
using namespace Infrastructure;
using namespace OSA;
using namespace RTUtil;

struct TProblematicMethodInfo
{
    const char *szClassName;
    const char *szMethodName;
    TProblematicMethodInfo(const char *i_szClassName, const char *i_szMethodName)
        : szClassName(i_szClassName), szMethodName(i_szMethodName) {}
};

extern "C" CGADAPTOR_API IInstrumentationAdaptor* GetInstrumentationAdaptor()
{
    static CCGAdaptor s_CGAdaptor;
    return (IInstrumentationAdaptor*)&s_CGAdaptor;
}

//////////////////////////////////////////////////////////////////////////
// class CCGAdaptor implementation

//
// Default constructor
//
CCGAdaptor::CCGAdaptor()
    : CInstrumentationAdaptorBase(), m_pFilter(NULL), m_pMethodIdAllocator(NULL)
{
}

//
// Destructor
//
CCGAdaptor::~CCGAdaptor()
{
    if (m_pMethodIdAllocator)
    {
        delete m_pMethodIdAllocator;
    }
}

TResult CCGAdaptor::Init(ILogAssert *pLogger, 
                         MPI::IEventFilter* pFilter,
                         MPI::BitSet configFlags)
{
    // Call common initializer
    TResult res = CInstrumentationAdaptorBase::Init(pLogger, configFlags);
    if (MRTE_FAILED(res))
    {
        return MRTE_ERROR_FAIL;
    }

    // Do CGAdaptor-specific initialization
    m_pMethodIdAllocator = new CIdAllocator(JIE_FIRST_METHOD_ID);
    m_pFilter = (MPI::ICallGraphFilter*)pFilter;
    return MRTE_RESULT_OK;
}

TResult CCGAdaptor::ModifyClass(MPI::TId classId, 
                                const SClassFile &classToInstrument,
                                TMemoryAllocatorFunc funcAllocator,
                                SClassFile *pInstrumentedClass,
                                IInstrumentationContext *pContext)
{
    // Validate arguments
    if (NULL == m_pJIE)
    {
        return MRTE_ERROR_MODULE_NOT_INITIALIZED;
    }
    if (NULL == pInstrumentedClass || NULL == funcAllocator || NULL == pContext)
    {
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }
    if (NULL == classToInstrument.pClassFile || 0 == classToInstrument.uiSize)
    {
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }

    InvalidateInstrumentedClassBuffer(pInstrumentedClass);

    // Parse the class file
    IJavaClass *pJavaClass = m_pJIE->GetClassInterface(classToInstrument.pClassFile,
        classToInstrument.uiSize);
    if (NULL == pJavaClass)
    {
        // Unable to parse class file
        return MRTE_ERROR_ILL_CLASS_FILE;
    }

    SJavaClassInfo jci;
    TResult res = GetJavaClassInfo(&jci, pJavaClass);
    if (MRTE_RESULT_OK != res)
    {
        return MRTE_ERROR_FAIL;
    }

    SAdaptorClassInfo classInfo;
    SetClassInfo(&classInfo, jci, classId);

    res = CanInstrumentClass(classInfo);
    if (MRTE_RESULT_OK == res)
    {
        // Apply basic instrumentation to the class
        res = AddStaticFieldsForMethods(pJavaClass, &classInfo);
        if (MRTE_RESULT_OK == res)
        {
            // Write the instrumented class file to pInstrumentedClass
            SBuffer buffer;
            res = pJavaClass->WriteBack(&buffer, funcAllocator);
            if (MRTE_RESULT_OK == res)
            {
                pInstrumentedClass->pClassFile = buffer.pBuffer;
                pInstrumentedClass->uiSize = buffer.uiActualSize;
            }
        }
    }

    CContext *pCGContext = (CContext*)pContext;
    TResult copyRes = CopyInstrumentedClassInfo(&(pCGContext->classInfo), classInfo, 
        funcAllocator);
    if (MRTE_FAILED(copyRes))
    {
        res = copyRes;
    }

    // Clean-up
    pJavaClass->Free();
    DeallocateJavaClassInfo(&jci);
    DeallocateAdaptorClassInfo(&classInfo);

    return res; 
}

TResult CCGAdaptor::ModifyByteCodes(MPI::TId classId, 
                                    const SClassFile &classToInstrument,
                                    TMemoryAllocatorFunc funcAllocator,
                                    SClassFile *pInstrumentedClass,
                                    IInstrumentationContext *pContext)
{
    // validate arguments
    if (NULL == m_pJIE)
    {
        return MRTE_ERROR_MODULE_NOT_INITIALIZED;
    }
    if (NULL == pInstrumentedClass || NULL == funcAllocator)
    {
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }
    if (NULL == classToInstrument.pClassFile || 0 == classToInstrument.uiSize)
    {
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }

    InvalidateInstrumentedClassBuffer(pInstrumentedClass);

    // parse the class file
    IJavaClass *pJavaClass = m_pJIE->GetClassInterface(classToInstrument.pClassFile,
        classToInstrument.uiSize);
    if (NULL == pJavaClass)
    {
        // unable to parse class file
        return MRTE_ERROR_ILL_CLASS_FILE;
    }

    CContext *pCGContext = (CContext*)pContext;
    TResult res = CanInstrumentClass(pCGContext->classInfo);
    if (MRTE_RESULT_OK == res)
    {
        // Apply call-graph instrumentation to the class
        res = DoCallGraphInstrumentation(pJavaClass, pCGContext->classInfo, m_bJVMInitDone);
        if (MRTE_RESULT_OK == res)
        {
            // Write the instrumented class file to pInstrumentedClass
            SBuffer buffer;
            res = pJavaClass->WriteBack(&buffer, funcAllocator);
            if (MRTE_RESULT_OK == res)
            {
                pInstrumentedClass->pClassFile = buffer.pBuffer;
                pInstrumentedClass->uiSize = buffer.uiActualSize;
            }
        }
    }

    // clean-up
    pJavaClass->Free();

    return res; 
}

//
// Add call-graph instrumentation to the class
//
// Parameters:
//      pJavaClass      [in,out]    : Java class to instrument
//      classInfo       [in]        : Java class information, including an array of methods
//                                    that should be instrumented.
//      bJVMInitDone    [in]        : Whether the JVM has finished initialization or not
//
// Returns:
//      MRTE_RESULT_OK              : success
//      MRTE_ERROR_FAIL             : instrumentation failure
//
TResult CCGAdaptor::DoCallGraphInstrumentation(IJavaClass *pJavaClass,
                                               const SAdaptorClassInfo &classInfo,
                                               bool bJVMInitDone)
{
    unsigned int uiMethodCount = pJavaClass->GetNumberOfMethods();
    if (0 == uiMethodCount)
    {
        // class contains no methods. This is an error
        return MRTE_ERROR_FAIL;
    }
    if (0 == classInfo.uiNumMethods)
    {
        // no methods to instrument. We are done
        return MRTE_RESULT_OK;
    }

    // Determine the CGProxy callback functions to use during instrumentation, 
    // based on whether the JVM has finished initialization or not

    // Bugzilla 179354: a temporary workaround requires always using the "early" versions.
    const char *szMethodEntryCallback = CGADAPTOR_CALLBACK_EARLY_PROLOG_NAME;
    const char *szMethodLeaveCallback = CGADAPTOR_CALLBACK_EARLY_EPILOG_NAME;
/*
    const char *szMethodEntryCallback;
    const char *szMethodLeaveCallback;
    if (bJVMInitDone)
    {
        szMethodEntryCallback = CGADAPTOR_CALLBACK_PROLOG_NAME;
        szMethodLeaveCallback = CGADAPTOR_CALLBACK_EPILOG_NAME;
    }
    else
    {
        szMethodEntryCallback = CGADAPTOR_CALLBACK_EARLY_PROLOG_NAME;
        szMethodLeaveCallback = CGADAPTOR_CALLBACK_EARLY_EPILOG_NAME;
    }
*/

    // register the callbacks by updating the class' constant pool
    SWideString wzCallbackClassName;
    CWideStringUtils::InitWideString(wzCallbackClassName, CGADAPTOR_CALLBACK_CLASS_NAME);
    SWideString wzMethodEntryCallbackName;
    CWideStringUtils::InitWideString(wzMethodEntryCallbackName, szMethodEntryCallback);
    SWideString wzMethodEntryCallbackSig;
    CWideStringUtils::InitWideString(wzMethodEntryCallbackSig, 
        CGADAPTOR_CALLBACK_PROLOG_SIG);
    SWideString wzMethodLeaveCallbackName;
    CWideStringUtils::InitWideString(wzMethodLeaveCallbackName, szMethodLeaveCallback);
    SWideString wzMethodLeaveCallbackSig;
    CWideStringUtils::InitWideString(wzMethodLeaveCallbackSig, 
        CGADAPTOR_CALLBACK_EPILOG_SIG);

    TConstantPoolIndex cpEntryCallback = 
        pJavaClass->RegisterRecorderCallback(wzCallbackClassName, 
            wzMethodEntryCallbackName, wzMethodEntryCallbackSig);
    TConstantPoolIndex cpLeaveCallback = 
        pJavaClass->RegisterRecorderCallback(wzCallbackClassName, 
            wzMethodLeaveCallbackName, wzMethodLeaveCallbackSig);

    // register the CGProxy.IsJVMInit() method
    SWideString wzIsJVMInitName;
    CWideStringUtils::InitWideString(wzIsJVMInitName, CGADAPTOR_IS_JVM_INIT_NAME);
    SWideString wzIsJVMInitSig;
    CWideStringUtils::InitWideString(wzIsJVMInitSig, CGADAPTOR_IS_JVM_INIT_TYPE);
    TConstantPoolIndex cpIsJVMInit = 
        pJavaClass->RegisterRecorderCallback(wzCallbackClassName, 
            wzIsJVMInitName, wzIsJVMInitSig);

    TResult res = MRTE_RESULT_OK;
    IJavaMethod *pMethod;
    TJavaMethodIterator *pMethIter = NULL;
    SJavaMethodInfo *pJavaMethodInfo;
    size_t instrMethodCount = 0;

    if (ILLEGAL_CP_ENTRY != cpEntryCallback
        && ILLEGAL_CP_ENTRY != cpLeaveCallback
        && ILLEGAL_CP_ENTRY != cpIsJVMInit)
    {
        // Prepare a hash table of the methods that can be instrumented
        // (key is name + signature). This will allow for efficient look-ups
        TMethodNameToInfoMap mapMethodsToInstrument;
        mapMethodsToInstrument.Init();
        InitMethodMap(&mapMethodsToInstrument, classInfo);

        //
        // Inject the call-graph callbacks to all methods that can be instrumented and
        // pass the selectivity criterion.
        //

        pMethIter = pJavaClass->GetMethodIterator();
        while (pMethIter->HasNext())
        {
            pMethod = pMethIter->GetNext();
            pJavaMethodInfo = new SJavaMethodInfo();
            res = GetJavaMethodInfo(pJavaMethodInfo, pMethod);
            if (MRTE_RESULT_OK != res)
            {
                DeallocateJavaMethodInfo(pJavaMethodInfo);
                delete pJavaMethodInfo;
                break;
            }
            SAdaptorMethodInfo *pMethodInstrumentationInfo = 
                GetInfoFromMethodMap(mapMethodsToInstrument, *pJavaMethodInfo);
            if (NULL != pMethodInstrumentationInfo)
            {
                bool bCanInstrument = InvokeSelectivityCallback(classInfo, 
                    *pMethodInstrumentationInfo);
                if (bCanInstrument)
                {
                    res = InstrumentMethod(pJavaClass, pMethod, pMethodInstrumentationInfo, 
                        cpEntryCallback, cpLeaveCallback, cpIsJVMInit, bJVMInitDone);
                    if (MRTE_RESULT_OK != res)
                    {
                        DeallocateJavaMethodInfo(pJavaMethodInfo);
                        delete pJavaMethodInfo;
                        LOG_INFORMATIVE4("CGAdaptor", 0, false, 
                            "Error instrumenting '%s.%s(%s)'. Reason: %x", 
                            classInfo.szClassName, pMethodInstrumentationInfo->szName, 
                            pMethodInstrumentationInfo->szSignature, res);
                        break;
                    }
                    instrMethodCount++;
                    LOG_INFORMATIVE3("CGAdaptor", 5, false, "Instrumented: %s.%s(%s)", 
                        classInfo.szClassName, pMethodInstrumentationInfo->szName, 
                        pMethodInstrumentationInfo->szSignature);
                } // if (method passed selectivity criterion)
            } 
            // deallocate the method information
            DeallocateJavaMethodInfo(pJavaMethodInfo);
            delete pJavaMethodInfo;
        } // end of while
        pMethIter->Free();
    } // end if (recorder callbacks successfully added to constant pool)
    else
    {
        // unable to add recorder callbacks to the constant pool
        LOG_INFORMATIVE1("CGAdaptor", 0, false, "Unable to add recorder callbacks to class %s", 
            classInfo.szClassName);
        res = MRTE_ERROR_FAIL;
    }

    // Clean-up
    CWideStringUtils::FreeWideString(wzCallbackClassName);
    CWideStringUtils::FreeWideString(wzMethodEntryCallbackName);
    CWideStringUtils::FreeWideString(wzMethodEntryCallbackSig);
    CWideStringUtils::FreeWideString(wzMethodLeaveCallbackName);
    CWideStringUtils::FreeWideString(wzMethodLeaveCallbackSig);
    CWideStringUtils::FreeWideString(wzIsJVMInitName);    
    CWideStringUtils::FreeWideString(wzIsJVMInitSig);    

    // If no methods were instrumented, modify the return code to indicate this
    if (MRTE_SUCCEEDED(res) && instrMethodCount == 0)
    {
        LOG_INFORMATIVE1("CGAdaptor", 3, false, 
            "Class %s methods not instrumented because all its methods were excluded (likely because they matched a user-defined or profiler-defined filter)",
            classInfo.szClassName);
        res = MRTE_ERROR_INSTRUMENTATION_NOT_NEEDED;
    }
    return res;
}

//
// Checks if the class can be instrumented.
//
// Parameters:
//      info    : class information
//
// Returns:
//      MRTE_RESULT_OK                          : class can be instrumented
//      MRTE_ERROR_INSTRUMENTATION_NOT_NEEDED   : the class was not instrumented
//                                                at all. This value is returned
//                                                for interface classes.
//      MRTE_ERROR_UNABLE_TO_INSTRUMENT         : the class cannot be instrumented.
//                                                This value is returned for classes
//                                                the CG Adaptor cannot
//                                                instrument.
//
TResult CCGAdaptor::CanInstrumentClass(const SAdaptorClassInfo &info)
{
    // Note:
    // These classes are not instrumented here. To support reattach in bug 194081, similar filtering is
    // added into CProfEnv::IsExcludedClassMethod referring to this. When change this 
    // instrument filtering function, CProfEnv::IsExcludedClassMethod must be changed at the same time.
    
    //TODO: document the reason for each excluded class/package
    static char* ExcludedClassPatterns[] = 
    {
        CGADAPTOR_CALLBACK_CLASS_NAME,
        HEAPADAPTOR_CALLBACK_CLASS_NAME,
        THREADADAPTOR_CALLBACK_CLASS_NAME,
        "java/lang/Object",
        "sun/reflect/Generated",
        "jrockit/", //TODO: check why
        "java/lang/J9VM", //TODO: check why
        "com/Intel/VTune/VTuneAPI",
        "org/eclipse/hyades/collection/profiler",
        "CGLIB$$" // TPTP Bugzilla 152579 - CG Instrumentation conflicts with Hibernate
    };

    // Can't instrument interface classes as they contain no code
    if (CAF_INTERFACE & info.uiAttributeFlags)
    {
        return MRTE_ERROR_INSTRUMENTATION_NOT_NEEDED;
    }

    // Check if the class appear in the excluded class patterns list
    const char *szClassName = info.szClassName;
    unsigned int i;
    for (i = 0; i < SIZE_OF_ARRAY(ExcludedClassPatterns); ++i)
    {
        char *szExcludedPattern = ExcludedClassPatterns[i];
        if (strstr(szClassName, szExcludedPattern) != NULL)
        {
            LOG_INFORMATIVE1("CCGAdaptor", 5, false, "Class '%s' will not be instrumented",
                szClassName);
            return MRTE_ERROR_UNABLE_TO_INSTRUMENT;
        }
    }

    // The class can be instrumented
    return MRTE_RESULT_OK;
}

//
// Add an 'already invoked' boolean static field for each method that can be instrumented
//
// Parameters:
//      pJavaClass      [in,out]    : Java class to instrument
//      pClassInstrInfo [out]       : upon successful instrumentation, the pMethods
//                                    member will be filled with the methods that can be
//                                    instrumented and their MPI IDs.
//
// Returns:
//      MRTE_RESULT_OK              : success
//      MRTE_ERROR_FAIL             : instrumentation failure
//
TResult CCGAdaptor::AddStaticFieldsForMethods(IJavaClass *pJavaClass, 
                                              SAdaptorClassInfo *pClassInstrInfo)
{
    unsigned int uiMethodCount = pJavaClass->GetNumberOfMethods();
    if (0 == uiMethodCount)
    {
        return MRTE_ERROR_FAIL;
    }

    //
    // Allocate IDs for all methods in the class.
    // Note that we also allocate IDs for methods which will never be instrumented 
    // (such as native or abstract methods), so we might "lose" some IDs in the process. 
    // This, however, is not a problem.
    //
    MPI::TId nextMethodMPIId = m_pMethodIdAllocator->AllocateIds(uiMethodCount);

    TResult res;
    TMethodInfoVector vecInstrMethods;
    IJavaMethod *pMethod;
    TJavaMethodIterator *pMethIter = NULL;
    bool bCanInstrument;
    SJavaMethodInfo *pJavaMethodInfo;

    //
    // Create a list of all methods that can be instrumented.
    // For each of these methods, add an "already invoked" static field to the class.
    //
    pMethIter = pJavaClass->GetMethodIterator();
    while (pMethIter->HasNext())
    {
        pMethod = pMethIter->GetNext();
        pJavaMethodInfo = new SJavaMethodInfo();
        res = GetJavaMethodInfo(pJavaMethodInfo, pMethod);
        if (MRTE_RESULT_OK != res)
        {
            DeallocateJavaMethodInfo(pJavaMethodInfo);
            delete pJavaMethodInfo;
            break;
        }
        bCanInstrument = CanInstrumentMethod(*pJavaMethodInfo);
        if (bCanInstrument)
        {
            // add a static field for the method
            MPI::TId mpiId = nextMethodMPIId++;
            TConstantPoolIndex cpiFieldAlreadyInvoked;
            res = AddAlreadyInvokedFieldForMethod(&cpiFieldAlreadyInvoked, 
                pJavaClass, mpiId);
            if (MRTE_RESULT_OK != res)
            {
                DeallocateJavaMethodInfo(pJavaMethodInfo);
                delete pJavaMethodInfo;
                break;
            }

            // add the method to a list of instrumented methods
            SAdaptorMethodInfo *pMethodInstrInfo = new SAdaptorMethodInfo();
            pMethodInstrInfo->id = mpiId;
            pMethodInstrInfo->cpAlreadyInvokedFlag = cpiFieldAlreadyInvoked;
            pMethodInstrInfo->szName = CWideStringUtils::WideStringToCString(
                pJavaMethodInfo->methodName);
            pMethodInstrInfo->szSignature = CWideStringUtils::WideStringToCString(
                pJavaMethodInfo->methodSignature);
            pMethodInstrInfo->uiAttributeFlags = pJavaMethodInfo->uiAttributeFlags;
            vecInstrMethods.Push(pMethodInstrInfo);
        } 
        DeallocateJavaMethodInfo(pJavaMethodInfo);
        delete pJavaMethodInfo;
    } // end of while
    pMethIter->Free();

    if (MRTE_RESULT_OK == res)
    {
        // all methods were instrumented successfully.
        // add information of all instrumented methods to pClassInstrInfo
        SetMethodInfo(pClassInstrInfo, vecInstrMethods);
    }

    // deallocate the list of instrumented methods
    FreeMethodsVectorItems(vecInstrMethods);
    
    return res;
}

//
// Checks if the method can be instrumented
//
// Parameters:
//      methodInfo  : method information
//
// Returns:
//      true    : the method can be instrumented
//      false   : the method cannot be instrumented
//
bool CCGAdaptor::CanInstrumentMethod(const SJavaMethodInfo &methodInfo)
{
    //TODO: log methods that are not instrumented
    if (methodInfo.uiAttributeFlags & MAF_NATIVE ||
        methodInfo.uiAttributeFlags & MAF_ABSTRACT)
    {
        // can't instrument native and abstract methods as they have no code
        return false;
    }

    char *szClassName = CWideStringUtils::WideStringToCString(methodInfo.className);
    char *szMethodName = CWideStringUtils::WideStringToCString(methodInfo.methodName);
    bool bIsProblematic = IsProblematicMethod(szClassName, szMethodName);
    delete [] szClassName;
    delete [] szMethodName;

    return (!bIsProblematic);
}

//
// Execute the selectivity callback function to check whether a method
// should be instrumented
//
// Parameters:
//      classInfo   : class information
//      methodInfo  : method information
//
// Returns:
//      true    : the method passes the selectivity criterion
//      false   : the method does not pass the selectivity criterion
//
bool CCGAdaptor::InvokeSelectivityCallback(const SAdaptorClassInfo &classInfo,
                                           const SAdaptorMethodInfo &methodInfo)
{
    if (NULL == m_pFilter)
    {
        // No selectivity callback is defined. All methods should be instrumented
        return true;
    }

    // Calculate the method "access type"
    MPI::EMethodAccessType methodAccessType;
    bool bIsMethodPublic = MAF_PUBLIC & methodInfo.uiAttributeFlags;
    if (bIsMethodPublic)
    {
        if (classInfo.uiAttributeFlags & CAF_PUBLIC)
        {
            methodAccessType = MPI::MAT_PACKAGE_API;
        }
        else
        {
            methodAccessType = MPI::MAT_PACKAGE_INTERNAL;
        }
    }
    else
    {
        methodAccessType = MPI::MAT_CLASS_INTERNAL;
    }
    
    // Invoke the selectivity callback
    MPI::SCallGraphFilterData callGraphFilter;
    callGraphFilter.methodAccessType = methodAccessType;
    callGraphFilter.szClassName = classInfo.szClassName;
    callGraphFilter.szMethodName = methodInfo.szName;
    callGraphFilter.szMethodSignature = methodInfo.szSignature;
    bool bShouldInstrument = m_pFilter->ShouldNotify(callGraphFilter);

    return bShouldInstrument;
}

//
// Instruments the specified method
//
// Parameters:
//      pJavaClass          [in,out]    : Java class to instrument
//      pInfo               [in]        : information about the method to instrument
//      cpEntryCallback     [in]        : constant-pool index of the "Entry" callback
//      cpLeaveCallback     [in]        : constant-pool index of the "Leave" callback
//      cpIsJVMInit         [in]        : constant-pool index of the "IsJVMInit" method
//      bJVMInitDone        [in]        : whether the JVM has initialized
//
// Returns:
//      MRTE_RESULT_OK              : success
//      MRTE_ERROR_FAIL             : instrumentation error
//
TResult CCGAdaptor::InstrumentMethod(IJavaClass *pJavaClass,
                                     IJavaMethod *pMethod,
                                     SAdaptorMethodInfo *pInfo, 
                                     TConstantPoolIndex cpEntryCallback, 
                                     TConstantPoolIndex cpLeaveCallback,
                                     TConstantPoolIndex cpIsJVMInit,
                                     bool bJVMInitDone)
{
    TResult res;

    // add the method ID to the constant pool
    SConstantPoolEntryValue cpvMethodId;
    cpvMethodId.type = CPT_INTEGER;

    // assume the method id can fit in a 32-bit constant
    // update this if it is not true
    int u32MethodId = SAFE_U64_TO_U32(pInfo->id);
    cpvMethodId.u.iVal = u32MethodId;
    
    TConstantPoolIndex cpMethodId;
    cpMethodId = pJavaClass->AddEntryToCP(cpvMethodId);
    if (ILLEGAL_CP_ENTRY == cpMethodId)
    {
        return MRTE_ERROR_FAIL;
    }
    
    // obtain an instruction iterator. This will create the method's flow-graph
    TInstructionListIterator *pInstructionsIter =
        pMethod->GetInstructionListIterator(IET_ALL_INSTRUCTIONS);
    pInstructionsIter->GetNext(); // skip .START
    IInstruction *pOriginalFirstInstr = pInstructionsIter->GetNext();

    // Modify to support empty methods instrumentation for bug 284836
    // Do not instrument empty (compiler-generated) methods
    //if (MNM_RETURN == pOriginalFirstInstr->GetMnemonic())
    //{
    //    return MRTE_RESULT_OK;
    //}

    // If StackMapTable calculation (Java 6) is enabled, do not instrument constructors.
    // This is a temporary workaround for https://bugs.eclipse.org/170075 until a better
    // solution is implemented.
    if (IsStackMapCalcEnabled())
    {
        if (IsConstructor(pInfo->szName))
        {
            return MRTE_RESULT_OK;
        }
    }
    

    TVariableID isJVMInitVar = (TVariableID)-1;
    res = InjectPrologCode(pMethod, pInstructionsIter, pInfo->cpAlreadyInvokedFlag,
        cpMethodId, cpEntryCallback, cpIsJVMInit, bJVMInitDone, &isJVMInitVar);
    if (MRTE_RESULT_OK != res)
    {
        return res;
    }

    res = InjectEpilogCode(pMethod, pInstructionsIter, pOriginalFirstInstr, isJVMInitVar, 
        cpMethodId, cpLeaveCallback, bJVMInitDone,
        IsConstructor(pInfo->szName));

    if (MRTE_RESULT_OK != res)
    {
        return res;
    }

    pInstructionsIter->Free();
    res = pMethod->ApplyInstrumentation();

    return res;
}

//
// Injects the "prolog" instrumentation to the given method
//
// Parameters:
//      pMethod             [in,out]    : the method to instrument
//      pInstructionsIter   [in]        : iterator of the method instructions 
//      cpiFieldAlreadyInvoked [in]     : constant pool index of the field used to check
//                                      : whether the method was invoked for the first time
//      cpiMethodId         [in]        : constant pool entry of the MPI method id
//      cpiEntryCallback    [in]        : MethodEntry callback of the recorder class
//      cpiIsJVMInit        [in]        : constant pool entry of the CGProxy.IsJVMInit method
//      bJVMInit            [in]        : whether the JVM has initialized
//      pIsJVMInitVar       [out]       : the index of the local variable created to
//                                      : store the result of CGProxy.IsJVMInit
//
// Returns:
//      MRTE_RESULT_OK                  : success
//      MRTE_ERROR_FAIL                 : failure
//         
TResult CCGAdaptor::InjectPrologCode(IJavaMethod *pMethod, 
                                     TInstructionListIterator *pInstructionsIter,
                                     TConstantPoolIndex cpiFieldAlreadyInvoked,
                                     TConstantPoolIndex cpiMethodId,
                                     TConstantPoolIndex cpiEntryCallback,
                                     TConstantPoolIndex cpiIsJVMInit,
                                     bool bJVMInitDone,
                                     TVariableID *pIsJVMInitVar)
{
    TResult res;

    IInstruction *pInst = pInstructionsIter->GetFirst();
    assert(MNM_PSEUDO_START == pInst->GetMnemonic());

    IInstruction *pFirstMethodInst = pInstructionsIter->GetNext();
    // if the first instruction is a pseudo-instruction (probably .TRY) - add a 'nop'
    // and use it as the first instruction. This is done because JIE does not permit
    // using pseudo-instructions except .TARGET as branch targets.
    EMnemonic firstMethodMnem = pFirstMethodInst->GetMnemonic();
    if (firstMethodMnem >= MNM_FIRST_PSEUDO && firstMethodMnem != MNM_PSEUDO_TARGET)
    {
        pFirstMethodInst->AddBefore(&pFirstMethodInst, MNM_NOP);
    }

    SOperand op1;

    if (!bJVMInitDone)
    {
        //
        // The JVM has not been initialized yet.
        // Create a local variable that will store the result returned from
        // CGProxy.IsJVMInit()
        //
        res = pMethod->AddLocalVariable(pIsJVMInitVar, VT_INT);
        if (MRTE_RESULT_OK != res)
        {
            return MRTE_ERROR_FAIL;
        }

        //
        // Inject code that verifies that the EarlyMethodEntry callback will be called 
        // only after the JVM has initialized:
        //
        // invokestatic cpiIsJVMInit
        // dup
        // istore *pIsJVMInitVar
        // ifeq [jump to the first instruction of the method]
        //
        op1.type = OT_JAVA_CP_INDEX;
        op1.val.cpIndex = cpiIsJVMInit;
        pInst->AddAfter(&pInst, MNM_INVOKESTATIC, &op1);

        pInst->AddAfter(&pInst, MNM_DUP);
        
        op1.type = OT_VAR_ID;
        op1.val.varID = *pIsJVMInitVar;
        pInst->AddAfter(&pInst, MNM_ISTORE, &op1);

        op1.type = OT_TARGET;
        op1.val.pTarget = pFirstMethodInst;
        pInst->AddAfter(&pInst, MNM_IFEQ, &op1);
    } // end if (!bJVMInitDone)

    //
    // Inject code that calls the MethodEnter or EarlyMethodEnter callbacks
    //
    // getstatic cpiFieldAlreadyInvoked
    // ldc_w cpMethodId
    // invokestatic cpiEntryCallback
    // getstatic cpiFieldAlreadyInvoked
    // ifne [jump to the first instruction of the method]
    // iconst_1
    // putstatic cpiFieldAlreadyInvoked
    //

    op1.type = OT_JAVA_CP_INDEX;
    op1.val.cpIndex = cpiFieldAlreadyInvoked;
    pInst->AddAfter(&pInst, MNM_GETSTATIC, &op1);

    op1.type = OT_JAVA_CP_INDEX;
    op1.val.cpIndex = cpiMethodId;
    pInst->AddAfter(&pInst, MNM_LDC_W, &op1);


    op1.type = OT_JAVA_CP_INDEX;
    op1.val.cpIndex = cpiEntryCallback;
    pInst->AddAfter(&pInst, MNM_INVOKESTATIC, &op1);

    op1.type = OT_JAVA_CP_INDEX;
    op1.val.cpIndex = cpiFieldAlreadyInvoked;
    pInst->AddAfter(&pInst, MNM_GETSTATIC, &op1);

    op1.type = OT_TARGET;
    op1.val.pTarget = pFirstMethodInst;
    pInst->AddAfter(&pInst, MNM_IFNE, &op1);

    pInst->AddAfter(&pInst, MNM_ICONST_1);

    op1.type = OT_JAVA_CP_INDEX;
    op1.val.cpIndex = cpiFieldAlreadyInvoked;
    pInst->AddAfter(&pInst, MNM_PUTSTATIC, &op1);
    
    return MRTE_RESULT_OK;
}

//
// Injects the "epilog" instrumentation to the given method
//
// Parameters:
//      pMethod             [in,out]    : the method to instrument
//      pInstructionsIter   [in]        : iterator of the method instructions 
//      pOriginalFirstMethodInst [in]   : the original first instruction of the method
//                                      : (before adding the prolog code)
//      isJVMInitVar        [in]        : index of the local variable storing the result
//                                      : of calling to CGProxy.IsJVMInit()
//      cpiMethodId         [in]        : constant pool entry of the MPI method id
//      cpiLeaveCallback    [in]        : MethodLeave callback of the recorder class
//      bJVMInit            [in]        : whether the JVM has initialized
//
// Returns:
//      MRTE_RESULT_OK                  : success
//      MRTE_ERROR_FAIL                 : failure
//         
TResult CCGAdaptor::InjectEpilogCode(IJavaMethod *pMethod, 
                                     TInstructionListIterator *pInstructionsIter,
                                     IInstruction *pOriginalFirstMethodInst,
                                     TVariableID isJVMInitVar,
                                     TConstantPoolIndex cpiMethodId,
                                     TConstantPoolIndex cpiLeaveCallback,
                                     bool bJVMInitDone,
                                     bool bIsConstructor)
{
    TResult res;

    IInstruction *pInst = pInstructionsIter->GetLast();
    assert(MNM_PSEUDO_END == pInst->GetMnemonic());

    pInst = pInstructionsIter->GetPrev();
    IInstruction *pLastMethodInstruction = pInst;

    // add a "nop" as the last instruction. This "nop" will serve as a jump
    // target when we need to skip the MethodLeave callback
    IInstruction *pLastInstInFinally;
    pInst->AddAfter(&pLastInstInFinally, MNM_NOP);

    IInstruction *pFirstInstInFinally = NULL;
    SOperand op1;

    // add epilog code at the end of the method

    if (!bJVMInitDone)
    {
        //
        // The JVM has not been initialized yet.
        // Inject code that verifies that the EarlyMethodLeave callback will be called 
        // only after the JVM has initialized:
        //
        // iload isJVMInitVar
        // ifeq [skip invocation of EarlyMethodLeave]
        //
        op1.type = OT_VAR_ID;
        op1.val.varID = isJVMInitVar;
        pInst->AddAfter(&pInst, MNM_ILOAD, &op1);
        pFirstInstInFinally = pInst;

        op1.type = OT_TARGET;
        op1.val.pTarget = pLastInstInFinally;
        pInst->AddAfter(&pInst, MNM_IFEQ, &op1);

    } // end if (!bJVMInitDone)

    //
    // Inject code that calls the MethodLeave or EarlyMethodLeave callbacks
    //
    // ldc_w cpMethodId
    // invokestatic cpiLeaveCallback
    //
    op1.type = OT_JAVA_CP_INDEX;
    op1.val.cpIndex = cpiMethodId;
    pInst->AddAfter(&pInst, MNM_LDC_W, &op1);
    if (NULL == pFirstInstInFinally)
    {
        pFirstInstInFinally = pInst;
    }
    op1.val.cpIndex = cpiLeaveCallback;
    pInst->AddAfter(&pInst, MNM_INVOKESTATIC, &op1);
    
    // Surround the entire method code with a protected block and put the epilog
    // inside a "finally" block. This will ensure that the epilog will be executed
    // whether the method terminates normally or abnormally

    IInstruction *pStartTry;
    pInstructionsIter->GetFirst();
    pStartTry = pOriginalFirstMethodInst;
    res = pMethod->BindTryFinallyBlock(pStartTry, pLastMethodInstruction,
        pFirstInstInFinally, pLastInstInFinally);

    return res;
}

//
// Adds a static private boolean field to the specified Java class
// which is used to determine whether a method was called for the first time.
// The method is identified by methodMPIIid.
//
// * NOTE: the new field is initialized to 'false' by the JVM. The instrumentation
//         added to the method's prolog flips this field to 'true' after the method
//         was called.
//
// Parameters:
//      pcpiField       [out]   : the constant-pool index of the new field
//      pJavaClass      [in]    : the Java class to which to add the field
//      methodMPIIid    [in]    : the method for which the field is added
//
// Returns:
//      MRTE_RESULT_OK      : success
//      MRTE_ERROR_FAILURE  : failure
//      
TResult CCGAdaptor::AddAlreadyInvokedFieldForMethod(TConstantPoolIndex *pcpiField,
                                                    IJavaClass *pJavaClass,
                                                    MPI_NS::TId methodMPIId)
{
    const char *FIRST_TIME_FIELD_NAME = "sm_bAlreadyInvoked%d";
    char szFirstTimeFieldName[100];

	// 32-bit big-endian platforms require an explicit cast to int, versus the defacto one that exists here for little endian platforms
	// Using "%d" with a 64-bit type will take only the 4 bytes of that type. This works on LE, but not 32-bit BE. 
#if (defined(__linux__) && defined(__s390__) && !defined(__s390x__)) || defined(__sparc) || defined(_AIX) || defined(MVS)
    sprintf(szFirstTimeFieldName, FIRST_TIME_FIELD_NAME, (int)methodMPIId);
#else
    sprintf(szFirstTimeFieldName, FIRST_TIME_FIELD_NAME, methodMPIId);
#endif

    SWideString wzFirstTimeFieldName;
    CWideStringUtils::InitWideString(wzFirstTimeFieldName, szFirstTimeFieldName);
    IJavaField *pNewJavaField = pJavaClass->AddField(wzFirstTimeFieldName, VT_BOOLEAN,  FAF_PRIVATE | FAF_STATIC);
	
    if (pNewJavaField)
    {
        *pcpiField = pNewJavaField->GetIndex();
    }
    else
    {
        *pcpiField = ILLEGAL_CP_ENTRY;
    }
    
    CWideStringUtils::FreeWideString(wzFirstTimeFieldName);
    if (ILLEGAL_CP_ENTRY == *pcpiField)
    {
        return MRTE_ERROR_FAIL;
    }
    return MRTE_RESULT_OK;
}

bool CCGAdaptor::IsConstructor(const char *szMethodName)
{
    bool bIsCtor;
    if (strcmp(szMethodName, JAVA_INSTANCE_CONSTRUCTOR_NAME) == 0)
    {
        bIsCtor = true;
    }
    else
    {
        bIsCtor = false;
    }
    return bIsCtor;
}

bool CCGAdaptor::IsProblematicMethod(const char *szClassName, 
                                     const char *szMethodName)
{
    //TODO: document why we exclude each function
/*
    static TProblematicMethodInfo ProblematicMethodsClasses[] = 
    {
        TProblematicMethodInfo("sun/misc/Launcher$AppClassLoader", "loadClass"),
        TProblematicMethodInfo("java/lang/System", "getSecurityManager"),
        TProblematicMethodInfo("java/lang/Thread", "<init>"),
        TProblematicMethodInfo("java/lang/Thread", "init"),
        TProblematicMethodInfo("java/lang/Thread", "isDaemon"),
        TProblematicMethodInfo("java/lang/Thread", "getPriority"),
        TProblematicMethodInfo("java/lang/ClassLoader", "loadClassFromNative"),
        TProblematicMethodInfo("java/lang/ThreadGroup", "remove"),
        TProblematicMethodInfo("java/lang/ThreadGroup", "destroyIfEmptyDaemon"),
        TProblematicMethodInfo("java/lang/ThreadGroup", "checkAccess"),
        TProblematicMethodInfo("java/lang/ThreadGroup", "addUnstarted"),
        TProblematicMethodInfo("java/lang/ThreadGroup", "checkNewThread"),
        TProblematicMethodInfo("java/lang/ClassLoader", "loadClass"),
        TProblematicMethodInfo("java/lang/ClassLoader", "getSystemClassLoader"),
        TProblematicMethodInfo("com/ibm/oti/vm/BootstrapClassLoader", "loadClass"),
        TProblematicMethodInfo("com/ibm/oti/vm/AbstractClassLoader", "getPackageName"),
        TProblematicMethodInfo("java/security/AccessController", "getContext"),
        TProblematicMethodInfo("java/security/AccessController", "toArrayOfProtectionDomains"),
        TProblematicMethodInfo("java/lang/Class", "getName")

    };
    unsigned int i;
    for (i = 0; i < SIZE_OF_ARRAY(ProblematicMethodsClasses); ++i)
    {
        TProblematicMethodInfo problematicMethodInfo = ProblematicMethodsClasses[i];
        if (strcmp(szClassName, problematicMethodInfo.szClassName) == 0)
        {
            if (strncmp(szMethodName, problematicMethodInfo.szMethodName,
                strlen(problematicMethodInfo.szMethodName)) == 0)
            {
                return true;
            }
        }
    }
*/
    return false;
}

/*
void CCGAdaptor::LogError(const char* szFormatMsg, ...)
{
    if (NULL == m_pLogger)
    {
        return;
    }
    va_list argptr;
    va_start(argptr, szFormatMsg);
    m_pLogger->Error("CGAdaptor", szFormatMsg, argptr);
    va_end(argptr);
}

void CCGAdaptor::LogInformative(int iLevel, bool bExposeToEC, const char* szFormatMsg, ...)
{
    va_list argptr;
    va_start(argptr, szFormatMsg);
    char szMsg[MAX_STRING];
    vsnprintf(szMsg, MAX_STRING, szFormatMsg, argptr);
    szMsg[MAX_STRING - 1] = 0;
    m_pLogger->Informative("CGAdaptor", iLevel, bExposeToEC, szMsg);
    va_end(argptr);
}
*/
