/*****************************************************************************
 * Copyright (c) 1997, 2010 Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/

#include "HeapAdaptor.h"
#include "WideStringUtils.h"
#include "HeapProxy.h"
#include "CGProxy.h"
#include "ThreadProxy.h"
#include "ValidityChecks.h"

#include <assert.h>
#include <string.h>

#include <list>

#define JIE_FIRST_METHOD_ID 0x10000 // first ids are reserved for threads in the prf resolution

using namespace std;

using namespace Martini;
using namespace HeapAdaptor;
using namespace JIE;
using namespace MIE;
using namespace Infrastructure;
using namespace OSA;
using namespace RTUtil;

struct TProblematicMethodInfo
{
    const char *szClassName;
    const char *szMethodName;
    TProblematicMethodInfo(const char *i_szClassName, const char *i_szMethodName)
        : szClassName(i_szClassName), szMethodName(i_szMethodName) {}
};

struct TNewObjectInfo
{
    TNewObjectInfo(char *i_szClass, TVariableID i_varId)
        : szClass(i_szClass), varId(i_varId) {}

    char *szClass;
    TVariableID varId;
};

extern "C" HEAPADAPTOR_API IInstrumentationAdaptor* GetInstrumentationAdaptor()
{
    static CHeapAdaptor s_HeapAdaptor;
    return (IInstrumentationAdaptor*)&s_HeapAdaptor;
}

//////////////////////////////////////////////////////////////////////////
// class CHeapAdaptor implementation

//
// Default constructor
//
CHeapAdaptor::CHeapAdaptor()
    : CInstrumentationAdaptorBase(), m_pFilter(NULL), m_pMethodIdAllocator(NULL)
{
}

//
// Destructor
//
CHeapAdaptor::~CHeapAdaptor()
{
    if (m_pMethodIdAllocator)
    {
        delete m_pMethodIdAllocator;
    }
}

TResult CHeapAdaptor::Init(ILogAssert *pLogger, 
                           MPI::IEventFilter* pFilter,
                           MPI::BitSet configFlags)
{
    // Call common initializer
    TResult res = CInstrumentationAdaptorBase::Init(pLogger, configFlags);
    if (MRTE_FAILED(res))
    {
        return MRTE_ERROR_FAIL;
    }

    // Do HeapAdaptor-specific initialization
    m_pMethodIdAllocator = new CIdAllocator(JIE_FIRST_METHOD_ID);
    m_pFilter = (MPI::IHeapFilter*)pFilter; // pFilter is ignored for now. In this version, the selectivity
                      // callback is invoked on each allocation event (in JPI)
    return MRTE_RESULT_OK;
}

TResult CHeapAdaptor::ModifyClass(MPI::TId classId, 
                                  const SClassFile &classToInstrument,
                                  TMemoryAllocatorFunc funcAllocator,
                                  SClassFile *pInstrumentedClass,
                                  IInstrumentationContext *pContext)
{
    // Validate arguments
    if (NULL == m_pJIE)
    {
        return MRTE_ERROR_MODULE_NOT_INITIALIZED;
    }
    if (NULL == pInstrumentedClass || NULL == funcAllocator)
    {
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }
    if (NULL == classToInstrument.pClassFile || 0 == classToInstrument.uiSize)
    {
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }

    InvalidateInstrumentedClassBuffer(pInstrumentedClass);

    // Parse the class file
    IJavaClass *pJavaClass = m_pJIE->GetClassInterface(classToInstrument.pClassFile,
        classToInstrument.uiSize);

    if (NULL == pJavaClass)
    {
        // Unable to parse class file
        return MRTE_ERROR_ILL_CLASS_FILE;
    }

    SJavaClassInfo jci;
    TResult res = GetJavaClassInfo(&jci, pJavaClass);
    if (MRTE_RESULT_OK != res)
    {
        return MRTE_ERROR_FAIL;
    }

    SAdaptorClassInfo classInfo;
    SetClassInfo(&classInfo, jci, classId);

	if(classInfo.szClassName != NULL) {
		LOG_INFORMATIVE1("CHeapAdaptor", 0, false, "Parsing class file in ModifyBytes(), class is %s", classInfo.szClassName);
	}
    res = CanInstrumentClass(classInfo);
    if (MRTE_RESULT_OK == res)
    {
        unsigned int uiMethodCount = pJavaClass->GetNumberOfMethods();
        if (uiMethodCount > 0)
        {
            //
            // Allocate IDs for all methods in the class.
            // Note that we also allocate IDs for methods which will never be instrumented 
            // (such as native or abstract methods), so we might "lose" some IDs in the process. 
            // This, however, is not a problem.
            //
            MPI::TId nextMethodMPIId = m_pMethodIdAllocator->AllocateIds(uiMethodCount);

            TMethodInfoVector vecInstrMethods;
            IJavaMethod *pMethod;
            TJavaMethodIterator *pMethIter = NULL;
            bool bCanInstrument;
            SJavaMethodInfo *pJavaMethodInfo;

            //
            // Create a list of all methods that can be instrumented.
            // For each of these methods, add an "already invoked" static field to the class.
            //
            pMethIter = pJavaClass->GetMethodIterator();
            while (pMethIter->HasNext())
            {
                pMethod = pMethIter->GetNext();
                pJavaMethodInfo = new SJavaMethodInfo();
                res = GetJavaMethodInfo(pJavaMethodInfo, pMethod);
                if (MRTE_RESULT_OK != res)
                {
                    DeallocateJavaMethodInfo(pJavaMethodInfo);
                    delete pJavaMethodInfo;
                    break;
                }
                bCanInstrument = CanInstrumentMethod(*pJavaMethodInfo);
                if (bCanInstrument)
                {
                    MPI::TId mpiId = nextMethodMPIId++;
                    SAdaptorMethodInfo *pMethodInstrInfo = new SAdaptorMethodInfo();
                    pMethodInstrInfo->id = mpiId;
                    pMethodInstrInfo->szName = CWideStringUtils::WideStringToCString(
                        pJavaMethodInfo->methodName);
                    pMethodInstrInfo->szSignature = CWideStringUtils::WideStringToCString(
                        pJavaMethodInfo->methodSignature);
                    pMethodInstrInfo->uiAttributeFlags = pJavaMethodInfo->uiAttributeFlags;
                    vecInstrMethods.Push(pMethodInstrInfo);
                } 
                DeallocateJavaMethodInfo(pJavaMethodInfo);
                delete pJavaMethodInfo;
            } // end of while

            pMethIter->Free();

            if (MRTE_RESULT_OK == res)
            {
                // Add method information to pClassInstrInfo
                SetMethodInfo(&classInfo, vecInstrMethods);
            }

            // Deallocate the list of instrumented methods
            FreeMethodsVectorItems(vecInstrMethods);
        } // if (uiMethodCound > 0)
        
        // No class-level changes are needed. Simply copy classToInstrument to 
        // pInstrumentedClass
        TResult res = CopyClassFile(pInstrumentedClass, classToInstrument, funcAllocator);
    }

    CContext *pHeapContext = (CContext*)pContext;
    TResult copyRes = CopyInstrumentedClassInfo(&(pHeapContext->classInfo), classInfo, 
        funcAllocator);
    if (MRTE_FAILED(copyRes))
    {
        res = copyRes;
    }

    // Clean-up
    pJavaClass->Free();
    DeallocateJavaClassInfo(&jci);
    DeallocateAdaptorClassInfo(&classInfo);

    return res; 
}

//
// Copies a class file buffer
//
// Parameters:
//      pDest           [out]   : destination buffer
//      src             [in]    : source buffer
//      funcAllocator   [in]    : a function to use for allocating memory for pDest buffers
//
// Returns
//      MRTE_RESULT_OK              : success
//      MRTE_ERROR_OUT_OF_MEMORY    : not enough memory to complete the operation
//
TResult CHeapAdaptor::CopyClassFile(SClassFile *pDest, 
                                    const SClassFile& src,
                                    TMemoryAllocatorFunc funcAllocator)
{
    pDest->uiSize = src.uiSize;
    pDest->pClassFile = (unsigned char *)funcAllocator(src.uiSize);
    if (NULL == pDest->pClassFile)
    {
        return MRTE_ERROR_OUT_OF_MEMORY;
    }
    memcpy(pDest->pClassFile, src.pClassFile, src.uiSize);
    return MRTE_RESULT_OK;
}


TResult CHeapAdaptor::ModifyByteCodes(MPI::TId classId, 
                                      const SClassFile &classToInstrument,
                                      TMemoryAllocatorFunc funcAllocator,
                                      SClassFile *pInstrumentedClass,
                                      IInstrumentationContext *pContext)
{
    // Validate arguments
    if (NULL == m_pJIE)
    {
        return MRTE_ERROR_MODULE_NOT_INITIALIZED;
    }
    if (NULL == pInstrumentedClass || NULL == funcAllocator)
    {
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }
    if (NULL == classToInstrument.pClassFile || 0 == classToInstrument.uiSize)
    {
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }

    InvalidateInstrumentedClassBuffer(pInstrumentedClass);

    // Parse the class file
    IJavaClass *pJavaClass = m_pJIE->GetClassInterface(classToInstrument.pClassFile,
        classToInstrument.uiSize);
    if (NULL == pJavaClass)
    {
        // unable to parse class file
        return MRTE_ERROR_ILL_CLASS_FILE;
    }

    CContext *pHeapContext = (CContext*)pContext;

	if(pHeapContext->classInfo.szClassName != NULL) {
		LOG_INFORMATIVE1("CHeapAdaptor", 0, false, "[ModifyByteCodes] ModifyByteCodes called on class %s", pHeapContext->classInfo.szClassName);
	}

	// Bypass the map database used by heap instance data collection so that information is not collected about it.
	// If the ConcurrentHashMap and WeakHashMap are included in the filters, this will cause a stackoverflow error to be generated.
	// This is caused by the Object being added to the Heap Instance data map which causes an allocation. The allocation
	// results in instrumentation to occur.
	if((pHeapContext->classInfo.szClassName != NULL) && ((strcmp(pHeapContext->classInfo.szClassName, "java/util/concurrent/ConcurrentHashMap") == 0) || (strcmp(pHeapContext->classInfo.szClassName, "java/util/WeakHashMap") == 0))) {
		return  MRTE_ERROR_INSTRUMENTATION_NOT_NEEDED;
	}

    if(!CheckFilter(pHeapContext->classInfo)) {
		return  MRTE_ERROR_INSTRUMENTATION_NOT_NEEDED;
    }

    TResult res = CanInstrumentClass(pHeapContext->classInfo);
    if (MRTE_RESULT_OK == res)
    {
        // Apply heap instrumentation to the class
        res = DoHeapInstrumentation(pJavaClass, pHeapContext->classInfo, m_bJVMInitDone);
        if (MRTE_RESULT_OK == res)
        {
            // Write the instrumented class file to pInstrumentedClass
            SBuffer buffer;
            res = pJavaClass->WriteBack(&buffer, funcAllocator);
            if (MRTE_RESULT_OK == res)
            {
                pInstrumentedClass->pClassFile = buffer.pBuffer;
                pInstrumentedClass->uiSize = buffer.uiActualSize;
            }
        }
    }

    // Clean-up
    pJavaClass->Free();

    return res; 
}

//
// Add heap instrumentation to the class
//
// Parameters:
//      pJavaClass      [in,out]    : Java class to instrument
//      classInfo       [in]        : Java class information, including an array of methods
//                                    that should be instrumented.
//      bJVMInitDone    [in]        : Whether the JVM has finished initialization or not
//
// Returns:
//      MRTE_RESULT_OK              : success
//      MRTE_ERROR_FAIL             : instrumentation failure
//
TResult CHeapAdaptor::DoHeapInstrumentation(IJavaClass *pJavaClass,
                                            const SAdaptorClassInfo &classInfo,
                                            bool bJVMInitDone)
{
    unsigned int uiMethodCount = pJavaClass->GetNumberOfMethods();
    if (0 == uiMethodCount)
    {
        // Class contains no methods. This is an error
        return MRTE_ERROR_FAIL;
    }

    const char *szClassName = classInfo.szClassName;

    // Always use the "early" versions of the callbacks. They retrieve the object class
    const char *szObjectAllocCallback = HEAPADAPTOR_CALLBACK_EARLY_OBJECT_ALLOC_NAME;
    const char *szObjectAllocCallbackSig = HEAPADAPTOR_CALLBACK_EARLY_OBJECT_ALLOC_SIG;
    const char *szArrayAllocCallback = HEAPADAPTOR_CALLBACK_EARLY_ARRAY_ALLOC_NAME;
    const char *szArrayAllocCallbackSig = HEAPADAPTOR_CALLBACK_EARLY_ARRAY_ALLOC_SIG;

/*
    // Determine the HeapProxy callback functions to use during instrumentation, 
    // based on whether the JVM has finished initialization or not

    const char *szObjectAllocCallback;
    const char *szArrayAllocCallback;
    if (bJVMInitDone)
    {
        szObjectAllocCallback = HEAPADAPTOR_CALLBACK_OBJECT_ALLOC_NAME;
        szArrayAllocCallback = HEAPADAPTOR_CALLBACK_ARRAY_ALLOC_NAME;
    }
    else
    {
        szObjectAllocCallback = HEAPADAPTOR_CALLBACK_EARLY_OBJECT_ALLOC_NAME;
        szArrayAllocCallback = HEAPADAPTOR_CALLBACK_EARLY_ARRAY_ALLOC_NAME;
    }
*/

    // Register the callbacks by updating the class' constant pool
    SWideString wzCallbackClassName;
    CWideStringUtils::InitWideString(wzCallbackClassName, HEAPADAPTOR_CALLBACK_CLASS_NAME);
    SWideString wzObjectAllocCallbackName;
    CWideStringUtils::InitWideString(wzObjectAllocCallbackName, szObjectAllocCallback);
    SWideString wzObjectAllocCallbackSig;
    CWideStringUtils::InitWideString(wzObjectAllocCallbackSig, szObjectAllocCallbackSig);

    SWideString wzArrayAllocCallbackName;
    CWideStringUtils::InitWideString(wzArrayAllocCallbackName, szArrayAllocCallback);
    SWideString wzArrayAllocCallbackSig;
    CWideStringUtils::InitWideString(wzArrayAllocCallbackSig, szArrayAllocCallbackSig);

    TConstantPoolIndex cpObjAllocCallback = pJavaClass->RegisterRecorderCallback(
        wzCallbackClassName, wzObjectAllocCallbackName, wzObjectAllocCallbackSig);
    TConstantPoolIndex cpArrayAllocCallback = pJavaClass->RegisterRecorderCallback(
        wzCallbackClassName, wzArrayAllocCallbackName, wzArrayAllocCallbackSig);

    TResult res;

    if (ILLEGAL_CP_ENTRY != cpObjAllocCallback
        && ILLEGAL_CP_ENTRY != cpArrayAllocCallback)
    {
/*
        if (strcmp(szClassName, "java/lang/Object") == 0)
        {
            res = InstrumentObjectConstructor(pJavaClass, cpObjAllocCallback);
        }
        else
        {
            res = InstrumentArrayAllocSites(pJavaClass, cpArrayAllocCallback);
        }
*/
        res = InstrumentAllocSites(pJavaClass, cpObjAllocCallback, cpArrayAllocCallback,
            classInfo);
        if (MRTE_ERROR_FAIL == res)
        {
            LOG_INFORMATIVE1("CHeapAdaptor", 0, false, "Internal error when instrumenting "
                "class %s. Class will not be instrumented", szClassName);
        }
    }
    else
    {
        // Unable to add recorder callbacks to the constant pool
        LOG_INFORMATIVE1("CHeapAdaptor", 0, false, "Failed to register HeapProxy callbacks "
            "in class %s. Class will not be instrumented", szClassName);
        res = MRTE_ERROR_FAIL;
    }

    // Clean-up
    CWideStringUtils::FreeWideString(wzCallbackClassName);
    CWideStringUtils::FreeWideString(wzObjectAllocCallbackName);
    CWideStringUtils::FreeWideString(wzObjectAllocCallbackSig);
    CWideStringUtils::FreeWideString(wzArrayAllocCallbackName);
    CWideStringUtils::FreeWideString(wzArrayAllocCallbackSig);

    return res;
}

//
// Checks if the class can be instrumented.
//
// Parameters:
//      info    : class information
//
// Returns:
//      MRTE_RESULT_OK                          : class can be instrumented
//      MRTE_ERROR_INSTRUMENTATION_NOT_NEEDED   : the class will not be instrumented
//                                                at all. This value is returned
//                                                for interface classes.
//      MRTE_ERROR_UNABLE_TO_INSTRUMENT         : the class cannot be instrumented.
//                                                This value is returned for classes
//                                                the Heap Adaptor cannot
//                                                instrument.
//
TResult CHeapAdaptor::CanInstrumentClass(const SAdaptorClassInfo &info)
{
    static char* ExcludedClassPatterns[] = 
    {
        CGADAPTOR_CALLBACK_CLASS_NAME,              // CGAdaptor recorder class
        HEAPADAPTOR_CALLBACK_CLASS_NAME,            // HeapAdaptor recorder class
        THREADADAPTOR_CALLBACK_CLASS_NAME,          // ThreadAdaptor recorder class
        "jrockit/",                                 // Internal JRockit classes. 
                                                    // Can't force-load some of them
                                                    // during attach/detach
        "java/lang/J9VM",                           // IBM Internal classes (J9 VM)
        "java/lang/Object",                         // Do not instrument j.l.Object
        "com/ibm/crypto",                           // Cause verification errors. //TODO: fix
        "sun/reflect/Generated",                    // Classes generated during runtime to 
                                                    // support reflection
        "com/Intel/VTune/VTuneAPI",                 // VTune API Proxy class
        "java/lang/",                               // Patch to IBM redefinition problem
        "java/security/",
        "java/awt/",
        "javax/swing/"
        "com/ibm/jvm/",
        "com/ibm/misc/",
        "com/ibm/oti/",
        "sun/security/",
        "sun/misc/",
        "sun/awt/",
        "sun/java2d/",
        "org/eclipse/hyades/collection/profiler"    // TPTP Profiler API Proxy class
    };

    // Can't instrument interface classes as they contain no code
    if (CAF_INTERFACE & info.uiAttributeFlags)
    {
        return MRTE_ERROR_INSTRUMENTATION_NOT_NEEDED;
    }

    // Check if the class appear in the excluded class patterns list
    const char *szClassName = info.szClassName;
    unsigned int i;
    for (i = 0; i < SIZE_OF_ARRAY(ExcludedClassPatterns); ++i)
    {
        char *szExcludedPattern = ExcludedClassPatterns[i];
        if (strstr(szClassName, szExcludedPattern) != NULL)
        {
            LOG_INFORMATIVE1("CHeapAdaptor", 5, false, "Class '%s' will not be instrumented",
                szClassName);
            return MRTE_ERROR_UNABLE_TO_INSTRUMENT;
        }
    }

    // The class can be instrumented
    return MRTE_RESULT_OK;
}

//
// Checks if the method can be instrumented
//
// Parameters:
//      methodInfo  : method information
//
// Returns:
//      true    : the method can be instrumented
//      false   : the method cannot be instrumented
//
bool CHeapAdaptor::CanInstrumentMethod(const SJavaMethodInfo &methodInfo)
{
    //TODO: log methods that are not instrumented
    if (methodInfo.uiAttributeFlags & MAF_NATIVE ||
        methodInfo.uiAttributeFlags & MAF_ABSTRACT)
    {
        // can't instrument native and abstract methods as they have no code
        return false;
    }

    char *szClassName = CWideStringUtils::WideStringToCString(methodInfo.className);
    char *szMethodName = CWideStringUtils::WideStringToCString(methodInfo.methodName);
    bool bIsProblematic = IsProblematicMethod(szClassName, szMethodName);
    delete [] szClassName;
    delete [] szMethodName;

    return (!bIsProblematic);
}

bool CHeapAdaptor::IsProblematicMethod(const char *szClassName, 
                                       const char *szMethodName)
{
    return false;
}

//
// Add code to java.lang.Object default constructor to pass the initialized object to the
// HeapProxy.ObjectAlloc callback
//
// Parameters:
//      pJavaClass  : class to instrument. Must be java.lang.Object
//      cpCallback  : constant-pool entry of the call-back function to invoke
//
// Returns:
//      MRTE_RESULT_OK  : success
//      MRTE_ERROR_FAIL : failure
//
TResult CHeapAdaptor::InstrumentObjectConstructor(IJavaClass *pJavaClass, 
                                                  TConstantPoolIndex cpCallback)
{
    TResult res = MRTE_ERROR_FAIL;
    IJavaMethod *pMethod = NULL;
    SJavaMethodInfo javaMethodInfo;

    // Search for the (single) default constructor of java.lang.Object
    TJavaMethodIterator *pMethIter = pJavaClass->GetMethodIterator();
    while (pMethIter->HasNext())
    {
        pMethod = pMethIter->GetNext();
        res = GetJavaMethodInfo(&javaMethodInfo, pMethod);
        if (MRTE_RESULT_OK != res)
        {
            DeallocateJavaMethodInfo(&javaMethodInfo);
            break;
        }

        char *szMethodName = CWideStringUtils::WideStringToCString(javaMethodInfo.methodName);
        if (strcmp(szMethodName, "<init>") == 0)
        {
            // Inject the following byte-codes at the beginning of the method:
            // aload 0                  -> load 'this' pointer to the operand stack
            // invokestatic cpCallback  -> invoke the recorder callback function

            TInstructionListIterator *pInstIter = pMethod->GetInstructionListIterator(
                IET_ALL_INSTRUCTIONS);
            IInstruction *pInst = pInstIter->GetFirst();

            pInst->AddAfter(&pInst, MNM_ALOAD_0);

            SOperand op;
            op.type = OT_JAVA_CP_INDEX;
            op.val.cpIndex = cpCallback;
            pInst->AddAfter(&pInst, MNM_INVOKESTATIC, &op);

            // Finalize
            pInstIter->Free();
            res = pMethod->ApplyInstrumentation();

            DeallocateJavaMethodInfo(&javaMethodInfo);
            delete [] szMethodName;
            break;
        }

        DeallocateJavaMethodInfo(&javaMethodInfo);
        delete [] szMethodName;

    } // end of while
    pMethIter->Free();

    return res;
}

//
// Add code to pass newly allocated array objects to the HeapProxy.ArrayAlloc callback
//
// Parameters:
//      pJavaClass  : class to instrument. Must *not* be java.lang.Object
//      cpCallback  : constant-pool entry of the call-back function to invoke
//
// Returns:
//      MRTE_RESULT_OK  : success
//      MRTE_ERROR_FAIL : failure
//
TResult CHeapAdaptor::InstrumentArrayAllocSites(IJavaClass *pJavaClass, 
                                                TConstantPoolIndex cpCallback)
{
    TResult res = MRTE_ERROR_FAIL;
    IJavaMethod *pMethod = NULL;
    SJavaMethodInfo javaMethodInfo;

    TJavaMethodIterator *pMethIter = pJavaClass->GetMethodIterator();
    while (pMethIter->HasNext())
    {
        pMethod = pMethIter->GetNext();
        res = GetJavaMethodInfo(&javaMethodInfo, pMethod);
        if (MRTE_RESULT_OK != res)
        {
            DeallocateJavaMethodInfo(&javaMethodInfo);
            break;
        }

        if (CanInstrumentMethod(javaMethodInfo))
        {
            TInstructionListIterator *pInstIter = pMethod->GetInstructionListIterator(
                IET_ALL_INSTRUCTIONS);
            IInstruction *pInst = NULL;

            // Search for any of the array-allocation instructions:
            // newarray, anewarray, multianewarray
            while (pInstIter->HasNext())
            {
                pInst = pInstIter->GetNext();
                if (pInst->GetMnemonic() == MNM_NEWARRAY
                    || pInst->GetMnemonic() == MNM_ANEWARRAY
                    || pInst->GetMnemonic() == MNM_MULTIANEWARRAY)
                {
                    // Inject the following byte-codes after the array allocation instruction:
                    // dup                      -> duplicate the array reference on stack
                    // invokestatic cpCallback  -> invoke the recorder callback function

                    pInst->AddAfter(&pInst, MNM_DUP);

                    SOperand op;
                    op.type = OT_JAVA_CP_INDEX;
                    op.val.cpIndex = cpCallback;
                    pInst->AddAfter(&pInst, MNM_INVOKESTATIC, &op);
                }
            }

            // Finalize
            pInstIter->Free();
            res = pMethod->ApplyInstrumentation();
        }

        DeallocateJavaMethodInfo(&javaMethodInfo);
    } // end of while

    pMethIter->Free();

    return res;
}

static bool IsAllocObjectInstruction(const char* className, const char* methodName)
{
    if (0 == strcmp(methodName, "clone"))
    {
        return true;
    }

    if ( (0 == strcmp(className, "java/lang/Class") || 
          0 == strcmp(className, "java/lang/reflect/Constructor")) &&
          0 == strcmp(methodName, "newInstance") )
    {
        return true;
    }

    return false;
}

//
// Add code to track newly allocated objects and arrays
//
// Parameters:
//      pJavaClass  : class to instrument. Must *not* be java.lang.Object
//      cpObjAllocCallback  : constant-pool entry of the call-back function to invoke for
//                            object allocations
//      cpArrayAllocCallback : constant-pool entry of the call-back function to invoke for
//                             array allocations
//
// Returns:
//      MRTE_RESULT_OK  : success
//      MRTE_ERROR_FAIL : failure
//
TResult CHeapAdaptor::InstrumentAllocSites(IJavaClass *pJavaClass, 
                                           TConstantPoolIndex cpObjAllocCallback,
                                           TConstantPoolIndex cpArrayAllocCallback,
                                           const SAdaptorClassInfo &classInfo)
{
    TResult res = MRTE_ERROR_FAIL;
    IJavaMethod *pMethod = NULL;
    SJavaMethodInfo javaMethodInfo;
    list<TNewObjectInfo> lstNewObjects;

    const unsigned int file_name_len_limit = 1024;

    SConstantPoolEntryValue cpClassValue;
    cpClassValue.type = CPT_CLASS;
    CWideStringUtils::AllocateWideString(cpClassValue.u.className, file_name_len_limit);
    if (NULL == cpClassValue.u.className.pStr)
    {
         return MRTE_ERROR_FAIL;
    }

    SConstantPoolEntryValue cpMethodValue;
    cpMethodValue.type = CPT_METHOD;
    CWideStringUtils::AllocateWideString(cpMethodValue.u.methodVal.className, file_name_len_limit);
    CWideStringUtils::AllocateWideString(cpMethodValue.u.methodVal.name, file_name_len_limit);
    CWideStringUtils::AllocateWideString(cpMethodValue.u.methodVal.signature, file_name_len_limit);

    // Prepare a hash table of the methods that can be instrumented
    // (key is name + signature). This will allow for efficient look-ups
    TMethodNameToInfoMap mapMethodsToInstrument;
    mapMethodsToInstrument.Init();
    InitMethodMap(&mapMethodsToInstrument, classInfo);

    TJavaMethodIterator *pMethIter = pJavaClass->GetMethodIterator();
    while (pMethIter->HasNext())
    {
        pMethod = pMethIter->GetNext();
        res = GetJavaMethodInfo(&javaMethodInfo, pMethod);
        if (MRTE_RESULT_OK != res)
        {
            DeallocateJavaMethodInfo(&javaMethodInfo);
            break;
        }

        SAdaptorMethodInfo *pMethodInstrumentationInfo = GetInfoFromMethodMap(
            mapMethodsToInstrument, javaMethodInfo);
        if (pMethodInstrumentationInfo != NULL)
        {
            // Add the method id to the constant pool.
            // Assume the method id can fit in a 32-bit constant (update this if wrong assumption)
            SConstantPoolEntryValue cpvMethodId;
            cpvMethodId.type = CPT_INTEGER;
            U32 u32MethodId = SAFE_U64_TO_U32(pMethodInstrumentationInfo->id);
            cpvMethodId.u.iVal = u32MethodId;
            TConstantPoolIndex cpMethodId;
            cpMethodId = pJavaClass->AddEntryToCP(cpvMethodId);
            if (ILLEGAL_CP_ENTRY == cpMethodId)
            {
                LOG_INFORMATIVE3("CHeapAdaptor", 0, false, 
                    "Failed to add constant pool entry for the method id. Method %s.%s(%s)", 
                    classInfo.szClassName,
                    pMethodInstrumentationInfo->szName,
                    pMethodInstrumentationInfo->szSignature);
                continue;
            }

            // Process method instructions
            TInstructionListIterator *pInstIter = pMethod->GetInstructionListIterator(
                IET_ALL_INSTRUCTIONS);
            IInstruction *pInst = NULL;

            while (pInstIter->HasNext())
            {
                pInst = pInstIter->GetNext();

                EMnemonic mneomic = pInst->GetMnemonic();
                char *szInvokeSpecialMethod = NULL;
                char *szInvokevirtualMethod = NULL;
                char *szInvokevirtualClass = NULL;
                IInstruction *pOffsetPoint = NULL;
                
                switch(mneomic)
                {
                case MNM_NEW:
                	// Get 'new' instruction operand
                    SOperand op;
                    res = pInst->GetOperand(&op);
                    if (MRTE_RESULT_OK != res)
                    {
                        LOG_INFORMATIVE3("CHeapAdaptor", 0, false,
                            "Failed to get 'new' instruction operand."
                            " Method %s.%s(%s)",
                            classInfo.szClassName,
                            pMethodInstrumentationInfo->szName,
                            pMethodInstrumentationInfo->szSignature);
                        continue;
                    }
                    if (op.type != OT_JAVA_CP_INDEX)
                    {
                        LOG_INFORMATIVE3("CHeapAdaptor", 0, false, 
                            "Found 'new' instruction with an invalid "
                            "operand. Method %s.%s(%s)", 
                            classInfo.szClassName,
                            pMethodInstrumentationInfo->szName,
                            pMethodInstrumentationInfo->szSignature);
                        continue;
                    }

                    // Get 'new' instruction operand class value
                    res = pJavaClass->GetValueFromCPEntry(op.val.cpIndex, &cpClassValue);
                    if (MRTE_RESULT_OK != res)
                    {
                        LOG_INFORMATIVE3("CHeapAdaptor", 0, false,
                            "Failed to get 'new' instruction operand value."
                            " Method %s.%s(%s)",
                            classInfo.szClassName,
                            pMethodInstrumentationInfo->szName,
                            pMethodInstrumentationInfo->szSignature);
                        continue;
                    }
                    if (cpClassValue.type != CPT_CLASS)
                    {
                        LOG_INFORMATIVE3("CHeapAdaptor", 0, false, 
                            "Found 'new' instruction with an invalid "
                            "class data. Method %s.%s(%s)", 
                            classInfo.szClassName,
                            pMethodInstrumentationInfo->szName,
                            pMethodInstrumentationInfo->szSignature);
                        continue;
                    }

                    // Store a copy of the uninitialized reference in a new local variable.
                    // TODO: only needed if the instantiated class is not filtered-out
                    op.val.varID = 0xFFFF;  // Init it with an invalid varId
                    op.type = OT_VAR_ID;
                    res = pMethod->AddLocalVariable(&(op.val.varID), VT_REFERENCE);
                    if (MRTE_FAILED(res))
                    {
                        LOG_INFORMATIVE3("CHeapAdaptor", 0, false, 
                            "Failed to create a local variable for "
                            "storing uninitialized object reference. Method %s.%s(%s)", 
                            classInfo.szClassName,
                            pMethodInstrumentationInfo->szName,
                            pMethodInstrumentationInfo->szSignature);
                        continue;
                    }
                    pInst->AddAfter(&pInst, MNM_DUP);
                    pInst->AddAfter(&pInst, MNM_ASTORE, &op);

                    // Store the instantiated class type and variable id, as it is needed for
                    // finding the matching 'invokespecial' instruction.
                    lstNewObjects.push_front(TNewObjectInfo(CWideStringUtils::WideStringToCString(
                        cpClassValue.u.className), op.val.varID));
                    break;

                case MNM_INVOKESPECIAL:
                    // Get 'invokespecial' instruction operand
                    SOperand opInvokeSpecial;
                    res = pInst->GetOperand(&opInvokeSpecial);
                    if (MRTE_RESULT_OK != res)
                    {
                        LOG_INFORMATIVE3("CHeapAdaptor", 0, false,
                            "Failed to get 'invokespecial' instruction operand."
                            " Method %s.%s(%s)",
                            classInfo.szClassName,
                            pMethodInstrumentationInfo->szName,
                            pMethodInstrumentationInfo->szSignature);
                        continue;
                    }
                    if (opInvokeSpecial.type != OT_JAVA_CP_INDEX)
                    {
                        LOG_INFORMATIVE3("CHeapAdaptor", 0, false, 
                            "Found 'invokespecial' instruction with an invalid "
                            "operand. Method %s.%s(%s)", 
                            classInfo.szClassName,
                            pMethodInstrumentationInfo->szName,
                            pMethodInstrumentationInfo->szSignature);
                        continue;
                    }

                    // Get 'invokespecial' instruction operand method value
                    res = pJavaClass->GetValueFromCPEntry(opInvokeSpecial.val.cpIndex, &cpMethodValue);
                    if (MRTE_RESULT_OK != res)
                    {
                        LOG_INFORMATIVE3("CHeapAdaptor", 0, false,
                            "Failed to get 'invokespecial' instruction operand value."
                            " Method %s.%s(%s)",
                            classInfo.szClassName,
                            pMethodInstrumentationInfo->szName,
                            pMethodInstrumentationInfo->szSignature);
                        continue;
                    }
                    if (cpMethodValue.type != CPT_METHOD)
                    {
                        LOG_INFORMATIVE3("CHeapAdaptor", 0, false, 
                            "Found 'invokespecial' instruction with an invalid "
                            "method data. Method %s.%s(%s)", 
                            classInfo.szClassName,
                            pMethodInstrumentationInfo->szName,
                            pMethodInstrumentationInfo->szSignature);
                        continue;
                    }

                    // Insert call back for <init>
                    szInvokeSpecialMethod = CWideStringUtils::WideStringToCString(
                        cpMethodValue.u.methodVal.name);
                    if (0 == strcmp(szInvokeSpecialMethod, "<init>"))
                    {
                        if (!lstNewObjects.empty())
                        {
                            TNewObjectInfo newObjInfo = lstNewObjects.front();
                            SOperand opVarId;
                            opVarId.type = OT_VAR_ID;
                            opVarId.val.varID = newObjInfo.varId;
                            lstNewObjects.pop_front();

                            char *szInvokespecialClass = CWideStringUtils::WideStringToCString(
                                cpMethodValue.u.methodVal.className);

                            if (strcmp(newObjInfo.szClass, szInvokespecialClass) == 0)
                            {
                                // Load the uninitialized object reference to the stack
                                pInst->AddAfter(&pInst, MNM_ALOAD, &opVarId);

                                // Inject code to invoke the recorder callback function
                                InvokeHeapCallback(pInst, cpObjAllocCallback, cpMethodId,
                                    pInst);
                            }
                            else
                            {
                                LOG_INFORMATIVE3("CHeapAdaptor", 0, false, 
                                    "Found unexpected sequence of 'new' "
                                    " and 'invokespecial' instructions. Method %s.%s(%s)", 
                                    classInfo.szClassName,
                                    pMethodInstrumentationInfo->szName,
                                    pMethodInstrumentationInfo->szSignature);
                            }

                            delete [] newObjInfo.szClass;
                            delete [] szInvokespecialClass;
                        }
                    }
                    delete []szInvokeSpecialMethod;
                    szInvokeSpecialMethod = NULL;
                    break;

                case MNM_INVOKEVIRTUAL:
                    // Get 'invokevirtual' instruction operand
                    SOperand opInvokeVirtual;
                    res = pInst->GetOperand(&opInvokeVirtual);
                    if (MRTE_RESULT_OK != res)
                    {
                        LOG_INFORMATIVE3("CHeapAdaptor", 0, false,
                            "Failed to get 'invokevirtual' instruction operand."
                            " Method %s.%s(%s)",
                            classInfo.szClassName,
                            pMethodInstrumentationInfo->szName,
                            pMethodInstrumentationInfo->szSignature);
                        continue;
                    }
                    if (opInvokeVirtual.type != OT_JAVA_CP_INDEX)
                    {
                        LOG_INFORMATIVE3("CHeapAdaptor", 0, false,
                            "Found 'invokevirtual' instruction with an invalid "
                            "operand. Method %s.%s(%s)",
                            classInfo.szClassName,
                            pMethodInstrumentationInfo->szName,
                            pMethodInstrumentationInfo->szSignature);
                        continue;
                    }

                    // Get 'invokevirtual' instruction operand method value
                    res = pJavaClass->GetValueFromCPEntry(opInvokeVirtual.val.cpIndex, &cpMethodValue);
                    if (MRTE_RESULT_OK != res)
                    {
                        LOG_INFORMATIVE3("CHeapAdaptor", 0, false,
                            "Failed to get 'invokevirtual' instruction operand value."
                            " Method %s.%s(%s)",
                            classInfo.szClassName,
                            pMethodInstrumentationInfo->szName,
                            pMethodInstrumentationInfo->szSignature);
                        continue;
                    }
                    if (cpMethodValue.type != CPT_METHOD)
                    {
                        LOG_INFORMATIVE3("CHeapAdaptor", 0, false,
                            "Found 'invokevirtual' instruction with an invalid "
                            "method data. Method %s.%s(%s)",
                            classInfo.szClassName,
                            pMethodInstrumentationInfo->szName,
                            pMethodInstrumentationInfo->szSignature);
                        continue;
                    }

                    // Insert call back for reflection, Object clone and Object serilization
                    szInvokevirtualMethod = CWideStringUtils::WideStringToCString(
                        cpMethodValue.u.methodVal.name);
                    szInvokevirtualClass = CWideStringUtils::WideStringToCString(
                                cpMethodValue.u.methodVal.className);
                    
                    if (IsAllocObjectInstruction(szInvokevirtualClass, szInvokevirtualMethod))
                    {
                        pOffsetPoint = pInst;
                        pInst->AddAfter(&pInst, MNM_DUP);
                        InvokeHeapCallback(pInst, cpObjAllocCallback, cpMethodId, pOffsetPoint);
                    }
                    delete []szInvokevirtualMethod;
                    szInvokevirtualMethod = NULL;
                    delete []szInvokevirtualClass;
                    szInvokevirtualClass = NULL;

                    break;

                case MNM_NEWARRAY:
                case MNM_ANEWARRAY:
                case MNM_MULTIANEWARRAY:
                    // InsertAllocArrayCallback();
                    // Inject the following byte-codes after the array allocation instruction:
                    // dup                      -> duplicate the array reference on stack
                    // invokestatic cpCallback  -> invoke the recorder callback function
                    pOffsetPoint = pInst;
                    pInst->AddAfter(&pInst, MNM_DUP);
                    InvokeHeapCallback(pInst, cpArrayAllocCallback, cpMethodId, pOffsetPoint);
                    break;

                default:
                	break;
                }
            } // end of while (more instructions to process)

            if (!lstNewObjects.empty())
            {
                // We have unmatched 'new' and 'invokespecial' instructions.
                // This is a fatal error!
                LOG_INFORMATIVE3("CHeapAdaptor", 0, false, 
                    "Found 'new' without matching "
                    "'invokespecial' instructions. Method %s.%s(%s)", 
                    classInfo.szClassName,
                    pMethodInstrumentationInfo->szName,
                    pMethodInstrumentationInfo->szSignature);
            }

            // Finalize
            pInstIter->Free();
            res = pMethod->ApplyInstrumentation();
        } // if (method information available)

        DeallocateJavaMethodInfo(&javaMethodInfo);
        if (res == MRTE_ERROR_FAIL)
        {
            break;
        }
    } // end of while (more methods to process)
    pMethIter->Free();

    CWideStringUtils::FreeWideString(cpClassValue.u.className);
    CWideStringUtils::FreeWideString(cpMethodValue.u.methodVal.className);
    CWideStringUtils::FreeWideString(cpMethodValue.u.methodVal.name);
    CWideStringUtils::FreeWideString(cpMethodValue.u.methodVal.signature);

    return res;
}

//
// Inject code after pInjectionPoint to invoke a heap callback (EarlyArrayAlloc or 
// EarlyObjectAlloc). Assume that a reference to the allocated object is already
// at the top of the operand stack.
// 
// The code:
// ldc_w cpMethodId                 // load method id
// MIE_loadinstoffset pOffsetPoint  // load offset of pOffsetPoint
// invokestatic cpCallback  // invoke HeapProxy.Early[Object|Array]Alloc
//

void CHeapAdaptor::InvokeHeapCallback(IInstruction *pInjectionPoint,
                                      TConstantPoolIndex cpCallback,
                                      TConstantPoolIndex cpMethodId,
                                      IInstruction *pOffsetPoint)
{
    SOperand op1;
    IInstruction *pInst = pInjectionPoint;

    op1.type = OT_JAVA_CP_INDEX;
    op1.val.cpIndex = cpMethodId;
    pInst->AddAfter(&pInst, MNM_LDC_W, &op1);

    op1.type = OT_TARGET;
    op1.val.pTarget = pOffsetPoint;
    pInst->AddAfter(&pInst, MNM_MIE_LOAD_INST_OFFSET, &op1);

    op1.type = OT_JAVA_CP_INDEX;
    op1.val.cpIndex = cpCallback;
    pInst->AddAfter(&pInst, MNM_INVOKESTATIC, &op1);
}

/** Returns FALSE if the object SHOULD be filtered,
 * returns TRUE if the object SHOULD NOT be filtered.
 * The result is equivalent to: should the JPI layer
 * be notified about this class.*/
bool CHeapAdaptor::CheckFilter(const SAdaptorClassInfo &classInfo)
{
    MCString strClassName;
	bool notified;

    if (NULL == m_pFilter)
    {
        // No selectivity callback is defined. All methods should be instrumented
        return true;
    }

    strClassName.Set(classInfo.szClassName);
    strClassName.ReplaceAll("/", ".");

	// Invoke the selectivity callback
	MPI::SHeapFilterData heapInteractionFilter;
	
	heapInteractionFilter.szClassName = strClassName.Get();

    notified = m_pFilter->ShouldNotify(heapInteractionFilter);

    return notified;
}

