/*****************************************************************************
 * Copyright (c) 2010 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    IBM Corporation - Initial API and implementation
 *
 * $Id: HeapObjectData.java,v 1.3 2010/12/14 22:22:26 jcayne Exp $ 
 *****************************************************************************/

package org.eclipse.tptp.martini.analysis;

import java.lang.ref.WeakReference;
import java.lang.reflect.Array;
import java.lang.reflect.Field;
import java.util.Iterator;
import java.util.Map;
import java.util.WeakHashMap;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Used to store and analyze the Object to obtain instance data information.
 */
public class HeapObjectData {
	/**
	 * Stores the reference to the Object using the TPTP ID (TId) as the key. Uses WeakReference
	 * to prevent holding onto the Object.
	 */
	private static ConcurrentHashMap<Long, WeakReference<Object>> tidInstDataMap;
	
	/**
	 * Prevents adding the map during initialization by initializing the map in the static block.
	 */
	private static boolean initialized;
	
	/**
	 * Tags used when transferring information about the Object.
	 */
	private static final String PRIMITIVE_TAG = "primitiveResult";
	private static final String ARRAY_COMPLEX_TAG = "complexArrayList";
	private static final String ARRAY_PRIMITIVE_TAG = "primitiveArrayList";
	private static final String COMPLEX_TYPE_TAG = "complexType";
	private static final String COMPLEX_MEMBER_TAG = "complexMember";
	private static final String COMPLEX_PRIMITIVE_MEMBER_TAG = "primitiveComplexMember";
	private static final String VERSION_TAG = "version";
	
	/**
	 * The agent version.
	 */
	private static final String VERSION = "4.7.0";
	
	/**
	 * Stores the time of the previous run to prevent garbage collection from happening too often.
	 */
	private static long previousRunTime = 0;
	
	/**
	 * Determines if Debug information should be printed.
	 */
	private static final boolean IS_DEBUG_ENABLED = showDebug();
	
	static {
		initialized = false;
		tidInstDataMap = new ConcurrentHashMap<Long, WeakReference<Object>>();
		initialized = true;
	}
	
	/**
	 * Stores an Object with the TPTP ID (TId) in the map.
	 * @param TId The TId of the Object.
	 * @param obj The Object to store in the map.
	 */
	public static void addToInstDataMap(long TId, Object obj)
	{
		// We don't want to record ourself
		if( !initialized || HeapObjectData.class == obj )
			return;
		
		Long tptpId = new Long(TId);
		
		// Synchronization handled by the map.
		tidInstDataMap.put(tptpId, new WeakReference<Object>(obj));
	}
	
	/**
	 * Returns the Object referenced by TId stored in the map.
	 * @param TId The TPTP id of the Object to lookup.
	 * @return The Object with the associated TId.
	 */
	public static Object getObjectFromInstDataMap(long TId)
	{
		// Synchronization handled by the map.
		Long tptpId = new Long(TId);
		if(tidInstDataMap == null)
			return null;

		WeakReference<Object> obj = tidInstDataMap.get(tptpId);
		if(obj == null)
			return null;
		Object objTId = obj.get();
		return objTId;
	}
	
	/**
	 * Returns the TId referenced by the Object.
	 * @param obj The Object to obtain the TId for.
	 * @return The TId associated with the Object.
	 */
	public static native long getObjectTId(Object obj);
	
	/**
	 * Returns the TId referenced by the Object stored in the map.
	 * @param obj The Object to lookup.
	 * @return The TId associated with the Object.
	 * 
	 * @deprecated Use {@link HeapObjectData#getObjectTId(Object) instead.
	 */
	public static Long getObjectFromInstDataMap(Object obj)
	{
		long TId = getObjectTId(obj);
		// The Object is not tagged or there is no TId
		if(TId == 0)
			return null;
		return new Long(TId);
	}
	
	/**
	 * Analyzes the Object with TId and returns the results.
	 * @param TId The TId associated with the Object to lookup.
	 * @return A String representation of the Object's contents.
	 */
	public static String getObjectValuesFromInstDataMap(long TId)
	{
		// Synchronization handled by the map.
		Long tptpId = new Long(TId);
		Object objTId = null;
		// Skips Objects with a TId of 0
		if(tidInstDataMap == null || TId == 0)
			return new String();
		WeakReference<Object> obj = tidInstDataMap.get(tptpId);
		String results = new String();
		if(obj == null)
			return results;
		objTId = obj.get();
		if(objTId != null)
			results = analyzeObject(objTId);
		return results;
	}
	
	/* Not in use.
	 * Removes the Object and TId entries from the maps.
	 * @param TId The TId of the Object to remove from the map. 
	public static void removeObjectFromInstDataMap(long TId)
	{
		Long tptpId = new Long(TId);
		Object obj = getObjectFromInstDataMap(tptpId);
		if(tidInstDataMap != null)
			tidInstDataMap.remove(tptpId);
	}
	*/
	
	/**
	 * Used to cleanup Objects from the map that have been deallocated during garbage collection.
	 */
	public static void removeUnusedObjectsFromInstDataMap() {
		// Synchronization handled by the map.
		if(tidInstDataMap == null)
			return;
		// Run the cleanup at most once per minute
		long currentRunTime = System.currentTimeMillis();
		if((currentRunTime - previousRunTime) > 60000) {
			previousRunTime = currentRunTime;
			for(Iterator<Map.Entry<Long, WeakReference<Object>>> iter = tidInstDataMap.entrySet().iterator(); iter.hasNext(); ) {
				Map.Entry<Long, WeakReference<Object>> tid = iter.next();
				if(tid.getKey() != null && tid.getValue() != null && tid.getValue().get() == null) {
					iter.remove();
				}
			}
		}
	}
	
	/**
	 * Analyzes the Object to return information about the instance.
	 * @param obj The Object to gather information about.
	 * @return Details about the Object.
	 */
	private static String analyzeObject(Object obj) {
		// Append the version and version tag
		StringBuilder result = new StringBuilder("<"); result.append(VERSION_TAG); result.append(">");	
		result.append(VERSION);	
		result.append("</"); result.append(VERSION_TAG); result.append(">");
		
		if(obj == null)
			return result.toString();
		
		Class<?> objClass = obj.getClass();
				
		// For a primitive object, just return the result
		if(isPrimitiveObject(obj)) {
			result.append("<"); result.append(PRIMITIVE_TAG); result.append(">");
			result.append(obj.getClass().getSimpleName()); result.append(":"); result.append(obj.toString());
			result.append("</"); result.append(PRIMITIVE_TAG); result.append(">");
		}
		
		// For an array, return the list of TIds
		else if(objClass.isArray()) {
			int objArraySize = Array.getLength(obj);
			// For arrays with a size, otherwise skip
			if(objArraySize > 0) {
				// Get the Objects from the Array
				Object arrayElementObj = Array.get(obj,0);
				if(arrayElementObj == null) { 
					// Skip null entries
				} else if(isPrimitiveObject(arrayElementObj)) {
					// Primitives can be printed
					result.append("<"); result.append(ARRAY_PRIMITIVE_TAG); result.append(">");
					// Add the type of the array
					result.append(arrayElementObj.getClass().getSimpleName()); result.append(":");
					// Traverse the array to get the Objects
					for(int i = 0; i < objArraySize; i++) {
						Object workingObj = Array.get(obj, i);
						if(workingObj != null) {
							// Append results to the list.
							if(i != 0)
								result.append(",");
							result.append(workingObj.toString());
						}
					}
					result.append("</"); result.append(ARRAY_PRIMITIVE_TAG); result.append(">");
				}
				// Complex types have their TIDs returned
				else {
					result.append("<"); result.append(ARRAY_COMPLEX_TAG); result.append(">");
					// Go through the array to get the objects
					for(int i = 0; i < objArraySize; i++) {
						// Append results to the list.
						if(i != 0)
							result.append(",");
						result.append(getObjectFromInstDataMap(Array.get(obj, i)));
					}
					result.append("</"); result.append(ARRAY_COMPLEX_TAG); result.append(">");
				}
			}
		} else if (obj != null) {
			// It's a complex type
			Field[] fieldz = objClass.getDeclaredFields();
			result.append("<"); result.append(COMPLEX_TYPE_TAG); result.append(">");
			for (int i = 0; i < fieldz.length; i++) {
				if(fieldz[i] != null) {
					Object memberObj = null;
					// Get access to all the fields
					fieldz[i].setAccessible(true);
					try {
						// Get the value of the member for the current Object
						memberObj = fieldz[i].get(obj);
					}
					catch (IllegalArgumentException err) {
						if(IS_DEBUG_ENABLED) {
							System.err.println("Failed to obtain the member.");
							err.printStackTrace();
						}
					} catch (IllegalAccessException err) {
						if(IS_DEBUG_ENABLED) {
							System.err.println("Unable to access the member.");
							err.printStackTrace();
						}
					}
					// Check the values of the member Object.
					if(memberObj != null) {
						if(isPrimitiveObject(memberObj)) {
							result.append("<"); result.append(COMPLEX_PRIMITIVE_MEMBER_TAG); result.append(">");
							result.append(memberObj.getClass().getSimpleName()); result.append(":"); 
							result.append(fieldz[i].getName()); result.append("="); result.append(memberObj.toString()); 
							result.append("</"); result.append(COMPLEX_PRIMITIVE_MEMBER_TAG); result.append(">");
						} else {
							// Add the TId of the field with it's name from the map.
							Long TIdvalue = getObjectFromInstDataMap(memberObj);
							
							result.append("<"); result.append(COMPLEX_MEMBER_TAG); result.append(">");
							result.append(memberObj.getClass().getSimpleName()); result.append(":");
							result.append(fieldz[i].getName()); result.append("="); result.append(TIdvalue); 
							result.append("</"); result.append(COMPLEX_MEMBER_TAG); result.append(">");
						}
					}
				}
			}
			result.append("</"); result.append(COMPLEX_TYPE_TAG); result.append(">");
		}
		else {
			if(IS_DEBUG_ENABLED)
				System.err.println("Unrecongized type passed");
		}
		return result.toString();
	}
	
	/**
	 * Determines if the passed object is of a primitive type.
	 * @param obj The Object to check.
	 * @return true if the Object is a primitive, false otherwise.
	 */
	private static boolean isPrimitiveObject(Object obj) {
		if(obj == null)
			return false;
		if(obj.getClass().isPrimitive())
			return true;
		if(obj instanceof Boolean || obj instanceof Byte || obj instanceof Character 
			|| obj instanceof Short || obj instanceof Integer || obj instanceof Long 
			|| obj instanceof Float || obj instanceof Double || obj instanceof String) {
			return true;
		}
		return false;
	}
	
	/**
	 * When the Environment Variable HEAP_OBJ_DATA_DEBUG is set to true, this will enable
	 * debug information to be printed.
	 * @return true when HEAP_OBJ_DATA_DEBUG is "true".
	 */
	private static boolean showDebug() {
		String debugEnv = System.getenv("HEAP_OBJ_DATA_DEBUG");
		if(debugEnv != null && debugEnv.equalsIgnoreCase("true"))
			return true;
		return false;
	}
}
