/*****************************************************************************
 * Copyright (c) 1997, 2008 Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/
//
#ifndef MRTE_CGADAPTORIMPL
#define MRTE_CGADAPTORIMPL

#include "InstrumentationAdaptorBase.h"
#include "MVector.h"
#include "MHash.h"
#include "MString.h"
#include "IdAllocator.h"

#ifdef CGADAPTOR_EXPORTS
#define CGADAPTOR_API API_EXPORT
#else
#define CGADAPTOR_API API_IMPORT
#endif

#define CGADAPTOR_DLL_NAME "CGAdaptor"

namespace Martini { namespace CGAdaptor {
    
    /**
     * @brief Function prototype for the MethodLeave native method of the CG Recorder Java 
     *        class
     *
     * This is a prototype for a function that provides an implementation of the 
     * MethodLeave native method of CGProxy.java.
     */
    typedef void  (JNICALL *TMethodLeaveHandler)(JNIEnv *env, jobject self, jint id);

    /**
     * @brief Function prototype for the MethodEnter native method of the CG Recorder Java 
     *        class
     *
     * This is a prototype for a function that provides an implementation of the 
     * MethodEnter native method of CGProxy.java.
     */
    typedef void  (JNICALL *TMethodEnterHandler)(JNIEnv *env, jobject self, 
                                                 bool bIsFirst, jint id);
    
    /**
     * @brief CGAdaptor class
     */
    class CCGAdaptor : public Infrastructure::CInstrumentationAdaptorBase
    {
    public:
        CCGAdaptor();
        ~CCGAdaptor();
        
        // IInstrumentationAdaptor methods

        virtual TResult Init(Infrastructure::ILogAssert *pLogger = NULL,
                             MPI::IEventFilter* pFilter = NULL,
                             MPI::BitSet configFlags = MPI::CF_NONE);

        virtual TResult ModifyClass(MPI::TId classId, 
                                    const Infrastructure::SClassFile &classToInstrument,
                                    TMemoryAllocatorFunc funcAllocator,
                                    Infrastructure::SClassFile *pInstrumentedClass,
                                    Infrastructure::IInstrumentationContext *pContext = NULL);

        virtual TResult ModifyByteCodes(MPI::TId classId, 
                                        const Infrastructure::SClassFile &classToInstrument,
                                        TMemoryAllocatorFunc funcAllocator,
                                        Infrastructure::SClassFile *pInstrumentedClass,
                                        Infrastructure::IInstrumentationContext *pContext = NULL);

    private:
        MPI::ICallGraphFilter* m_pFilter;       // Callback function for method-level selectivity
        Infrastructure::CIdAllocator *m_pMethodIdAllocator; // Method ID allocator

        // Check if the class can be instrumented
        TResult CanInstrumentClass(const Infrastructure::SAdaptorClassInfo &info);

        // Check if the method can be instrumented
        bool CanInstrumentMethod(const JIE::SJavaMethodInfo &methodInfo);

        // Add the 'already invoked' static fields to the specified Java class
        TResult AddStaticFieldsForMethods(JIE::IJavaClass *pJavaClass, 
                                          Infrastructure::SAdaptorClassInfo *pClassInstrInfo);

        // Add call-graph instrumentation to the specified class file
        TResult DoCallGraphInstrumentation(JIE::IJavaClass *pJavaClass,
                                           const Infrastructure::SAdaptorClassInfo &classInfo,
                                           bool bJVMInitDone);
    
        // Execute the selectivity callback function to check whether a method
        // should be instrumented
        bool InvokeSelectivityCallback(const Infrastructure::SAdaptorClassInfo &classInfo,
                                       const Infrastructure::SAdaptorMethodInfo &methodInfo);
    
        // Instruments the specified method
        TResult InstrumentMethod(JIE::IJavaClass *pJavaClass,
                                 JIE::IJavaMethod *pMethod,
                                 Infrastructure::SAdaptorMethodInfo *pInfo, 
                                 JIE::TConstantPoolIndex cpEntryCallback, 
                                 JIE::TConstantPoolIndex cpLeaveCallback,
                                 JIE::TConstantPoolIndex cpIsJVMInit,
                                 bool bJVMInitDone);

        // Injects the "prolog" instrumentation to the given method
        TResult InjectPrologCode(JIE::IJavaMethod *pMethod, 
                                 MIE::TInstructionListIterator *pInstructionsIter,
                                 JIE::TConstantPoolIndex cpiFieldAlreadyInvoked,
                                 JIE::TConstantPoolIndex cpiMethodId,
                                 JIE::TConstantPoolIndex cpiEntryCallback,
                                 JIE::TConstantPoolIndex cpiIsJVMInit,
                                 bool bJVMInitDone,
                                 MIE::TVariableID *pIsJVMInitVar);

        // Injects the "epilog" instrumentation to the given method
        TResult InjectEpilogCode(JIE::IJavaMethod *pMethod, 
                                 MIE::TInstructionListIterator *pInstructionsIter,
                                 MIE::IInstruction *pOriginalFirstMethodInst,
                                 MIE::TVariableID isJVMInitVar,
                                 JIE::TConstantPoolIndex cpiMethodId,
                                 JIE::TConstantPoolIndex cpiLeaveCallback,
                                 bool bJVMInitDone,
                                 bool bIsConstructor);

        // Adds a static private boolean field to the specified Java class
        // which is used to determine whether a method was called for the first time
        TResult AddAlreadyInvokedFieldForMethod(JIE::TConstantPoolIndex *pcpiField,
                                                JIE::IJavaClass *pJavaClass,
                                                MPI::TId methodMPIId);

        bool IsProblematicMethod(const char *szClassName, const char *szMethodName);
        bool IsConstructor(const char *szMethodName);

    }; // class CCGAdaptor

    /**
     * @brief An API for getting the Instrumentation Adaptor interface
     *
     * @return IInstrumentationAdaptor*     A pointer to the adaptor's singleton interface
     */ 
     extern "C" CGADAPTOR_API 
     Infrastructure::IInstrumentationAdaptor* GetInstrumentationAdaptor();

}} // namespace Martini::CGAdaptor

#endif // #define MRTE_CGADAPTORIMPL
