/*****************************************************************************
 * Copyright (c) 1997, 2010 Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/
//
#ifndef MRTE_HEAPADAPTORIMPL
#define MRTE_HEAPADAPTORIMPL

#include "InstrumentationAdaptorBase.h"
#include "MVector.h"
#include "MHash.h"
#include "MString.h"
#include "IdAllocator.h"

#ifdef HEAPADAPTOR_EXPORTS
#define HEAPADAPTOR_API API_EXPORT
#else
#define HEAPADAPTOR_API API_IMPORT
#endif

#define HEAPADAPTOR_DLL_NAME "HeapAdaptor"

namespace Martini { namespace HeapAdaptor {
    
    /**
     * @brief Function prototype for the ObjectAlloc native method of the Heap Recorder Java 
     *        class
     *
     * This is a prototype for a function that provides an implementation of the 
     * ObjectAlloc native method of HeapProxy.java.
     */
    typedef void (JNICALL *TObjectAllocHandler)(JNIEnv *env, jclass self, 
                                                jobject obj, 
                                                jclass objClass, 
                                                jint methodId, 
                                                jlong loc);

    /**
     * @brief Function prototype for the ArrayAlloc native method of the Heap Recorder Java 
     *        class
     *
     * This is a prototype for a function that provides an implementation of the 
     * ArrayAlloc native method of HeapProxy.java.
     */
    typedef void (JNICALL *TArrayAllocHandler)(JNIEnv *env, jclass self, 
                                               jobject obj, 
                                               jclass objClass, 
                                               jint methodId, 
                                               jlong loc);


	/**
	 * @brief Function prototype for the GetObjTId native method of the HeapObjectData Java class.
	 * 
	 * This is a prototype for a function that provides an implementation of the 
	 * getObjectTId native method of HeapObjectData.java. The result is the TId 
	 * for the passed Object.
	 */
    typedef jlong (JNICALL *TGetObjTIdHandler)(JNIEnv *env, jclass self, jobject obj);

    /**
     * @brief HeapAdaptor class
     */
    class CHeapAdaptor : public Infrastructure::CInstrumentationAdaptorBase
    {
    public:
        CHeapAdaptor();
        ~CHeapAdaptor();
        
        // IInstrumentationAdaptor methods

        virtual TResult Init(Infrastructure::ILogAssert *pLogger = NULL,
                             MPI::IEventFilter* pFilter = NULL,
                             MPI::BitSet configFlags = MPI::CF_NONE);

        virtual TResult ModifyClass(MPI::TId classId, 
                                    const Infrastructure::SClassFile &classToInstrument,
                                    TMemoryAllocatorFunc funcAllocator,
                                    Infrastructure::SClassFile *pInstrumentedClass,
                                    Infrastructure::IInstrumentationContext *pContext = NULL);

        virtual TResult ModifyByteCodes(MPI::TId classId, 
                                        const Infrastructure::SClassFile &classToInstrument,
                                        TMemoryAllocatorFunc funcAllocator,
                                        Infrastructure::SClassFile *pInstrumentedClass,
                                        Infrastructure::IInstrumentationContext *pContext = NULL);

    private:
        MPI::IHeapFilter* m_pFilter;            // Callback function for heap-selectivity
        Infrastructure::CIdAllocator *m_pMethodIdAllocator; // Method ID allocator

        // Check if the class can be instrumented
        TResult CanInstrumentClass(const Infrastructure::SAdaptorClassInfo &info);

        // Check if the method can be instrumented
        bool CanInstrumentMethod(const JIE::SJavaMethodInfo &methodInfo);

        // Add Heap instrumentation to the specified class file
        TResult DoHeapInstrumentation(JIE::IJavaClass *pJavaClass,
                                      const Infrastructure::SAdaptorClassInfo &classInfo,
                                      bool bJVMInitDone);
    
        bool IsProblematicMethod(const char *szClassName, const char *szMethodName);

        // Copies a class file buffer
        TResult CopyClassFile(Infrastructure::SClassFile *pDest, 
                              const Infrastructure::SClassFile& src,
                              TMemoryAllocatorFunc funcAllocator);

        // Apply BCI for java.lang.Object constructor
        TResult InstrumentObjectConstructor(JIE::IJavaClass *pJavaClass, 
                                            JIE::TConstantPoolIndex cpCallback);

        // Apply BCI for all Java classes other than java.lang.Object
        TResult InstrumentArrayAllocSites(JIE::IJavaClass *pJavaClass, 
                                          JIE::TConstantPoolIndex cpCallback);

        // Apply BCI for all Java classes
        TResult InstrumentAllocSites(JIE::IJavaClass *pJavaClass, 
                                     JIE::TConstantPoolIndex cpObjAllocCallback,
                                     JIE::TConstantPoolIndex cpArrayAllocCallback,
                                     const Infrastructure::SAdaptorClassInfo &classInfo);
        
        void InvokeHeapCallback(MIE::IInstruction *pInjectionPoint,
                                JIE::TConstantPoolIndex cpCallback,
                                JIE::TConstantPoolIndex cpMethodId,
                                MIE::IInstruction *pOffsetPoint);

        bool CheckFilter(const Infrastructure::SAdaptorClassInfo &classInfo);
    }; // class CHeapAdaptor

    /**
     * @brief An API for getting the Instrumentation Adaptor interface
     *
     * @return IInstrumentationAdaptor*     A pointer to the adaptor's singleton interface
     */ 
     extern "C" HEAPADAPTOR_API 
     Infrastructure::IInstrumentationAdaptor* GetInstrumentationAdaptor();

}} // namespace Martini::HeapAdaptor

#endif // #define MRTE_HEAPADAPTORIMPL
