/*****************************************************************************
 * Copyright (c) 1997, 2008 Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/

#ifndef _MRTE_INSTRUMENTATIONDAPTOR_H
#define _MRTE_INSTRUMENTATIONDAPTOR_H

#include "OSA.h"
#include "MpiAPI.h"
#include "MieAPI.h"

#include "ILogAssert.h"

#define GET_INSTRUMENTATION_ADAPTOR_NAME "GetInstrumentationAdaptor"

namespace Martini { 
    
namespace Infrastructure 
{

    //TODO: Add an intro section that explains what's this header file is all about

    /**
     * @brief Defines a Java class file buffer
     */
    struct SClassFile
    {
        unsigned char *pClassFile;  //!< Pointer to the buffer containing the Java class file
        unsigned int   uiSize;      //!< Class file size in bytes
    };

    /**
     * @brief Instrumentation Adaptor Context Interface
     *
     * Provides a base class for instrumentation information that needs to be shared between
     * the IInstrumentationAdaptor::ModifyClass and IInstrumentationAdaptor::ModifyByteCodes
     * operations. Each instrumentation adaptor can define a specific implementation
     * of this interface to store the instrumentation context information it needs.
     */
    class IInstrumentationContext
    {
    public:

        virtual ~IInstrumentationContext() {}
    };

    /**
     * @brief Instrumentation Adaptor Interface
     *
     * This interface defines the base operations of all Instrumentation Adaptors.
     */    
    class IInstrumentationAdaptor
    {
    public:

        virtual ~IInstrumentationAdaptor() {}

        /**
         * @brief Initializes the Instrumentation Adaptor module.
         *
         * This function initializes the Instrumentation Adaptor. It is called
         * prior to any other Adaptor function.
         * 
         * @param[in] pLogger       : A pointer to an initialized Logger object, to be used
         *                            by the instrumentation adaptor for logging messages and 
         *                            reporting fatal errors.
         * @param[in] pFilter       : An optional pointer to an object implementing the
         *                            IEventFilter interface. The Adaptor uses this 
         *                            object to determine which elements of the class file
         *                            should be instrumented.
         * @param[in] configFlags   : Optional configuration flags passed to the Martini tool
         *                            during its initialization (see MPI::EConfigurationFlags)
         *
         * @retval MRTE_RESULT_OK                   Success
         * @retval MRTE_ERROR_FAIL                  Initialization error
         * @retval MRTE_ERROR_UNSUPPORTED_FILTER    The filter type is not supported
         *                                          by the adaptor module
         */
        virtual TResult Init(Infrastructure::ILogAssert *pLogger = NULL,
                             MPI::IEventFilter* pFilter = NULL,
                             MPI::BitSet configFlags = MPI::CF_NONE) = 0;

        /**
         * @brief Modifies a Java class file
         *
         * This function modifies the given class file and adds static fields and constant pool 
         * entries required for the instrumentation. 
         * The function does not modify the code attribute of any method. Method code is
         * modified by the IInstrumentationAdaptor::ModifyByteCodes function.
         *
         * The function is typically called upon an instrumentation event 
         * ("class file load hook" event) in order to apply class-level changes which are
         * only allowed during class load.
         *
         * Information about the instrumented class, in addition to the new class file itself,
         * can be reported by the adaptor through the optional pContext output parameter.
         *
         * @param[in] classId                   The unique id of this class
         *
         * @param[in] classToInstrument         A client-allocated buffer containing the 
         *                                      the class file to instrument. 
         *                                      The uiActual size member must contain 
         *                                      the exact size of the class file buffer
         *
         * @param[in] funcAllocator             A function for allocating memory for both the 
         *                                      instrumented class file and for relevant
         *                                      member variables of the instrumentation context
         *
         * @param[out] pInstrumentedClass       A buffer to which the instrumented class
         *                                      is written. This buffer is allocated by 
         *                                      the Adaptor using the supplied
         *                                      allocator function (funcAllocator).
         *                                      If the class was successfully instrumented,
         *                                      the pClassFile member will hold the new
         *                                      class file and the uiSize member
         *                                      will be set to the class file buffer size.
         *                                      In case of an instrumentation error, or
         *                                      when the class cannot or should not be 
         *                                      instrumented, the pClassFile member will be
         *                                      set to NULL and the uiSize member
         *                                      will be set to 0.
         *
         * @param[out] pContext                 An optional initialized pointer to a structure 
         *                                      that will contain additional instrumentation 
         *                                      information. This information is specific
         *                                      to each Adaptor implementation. 
         *                                      Memory for the structure members
         *                                      is allocated using the memory allocation 
         *                                      function supplied in the funcAllocator 
         *                                      parameter.
         *                                      The caller is responsible for deallocating
         *                                      this structure.
         *                                      In case of an error, this argument will be
         *                                      left untouched.
         *
         * @retval MRTE_RESULT_OK                   Success
         * @retval MRTE_ERROR_INVALID_ARGUMENT      One of the required arguments is NULL
         * @retval MRTE_ERROR_OUT_OF_MEMORY         Not enough memory to complete the
         *                                          operation
         *
         * @retval MRTE_ERROR_INSTRUMENTATION_NOT_NEEDED   The class was not instrumented
         *                                                 at all. This value is returned
         *                                                 for interface classes.
         *
         * @retval MRTE_ERROR_UNABLE_TO_INSTRUMENT    The class cannot be instrumented.
         *                                            This value is returned for classes
         *                                            the Adaptor chooses not to instrument
         *                                            
         * @retval MRTE_ERROR_MODULE_NOT_INITIALIZED  The Adaptor was not yet initialized.
         *                                            Call the Init API to initialize
         *                                            the Adaptor.
         *
         * @retval MRTE_ERROR_ILL_CLASS_FILE          Unable to parse the class file.
         * @retval MRTE_ERROR_FAIL                    Unknown failure
         */
        virtual TResult ModifyClass(MPI::TId classId, 
                                    const SClassFile &classToInstrument,
                                    TMemoryAllocatorFunc funcAllocator,
                                    SClassFile *pInstrumentedClass,
                                    IInstrumentationContext *pContext = NULL) = 0;

        /**
         * @brief Modify the byte codes of the class' methods
         *
         * This function applies byte code instrumentation to the methods of the given class
         * file. During its execution, the function may use the IEventFilter interface supplied
         * during Adaptor initialization to determine which methods it needs to instrument.
         * 
         * This function can be called either upon instrumentation event 
         * ("class file load hook" event) or when certain MPI events (such as call-graph 
         * events) are enabled during execution and cause the redefinition of loaded classes.
         * Therefore, this function is only allowed to change method code attributes and 
         * the class constant pool, and must note change the class definition itself 
         * (for example, by adding new fields). Otherwise, the modified class file produced
         * by this method may cause the class redefinition process to fail.
         *
         * For some adaptor implementations (such as CG Adaptor), this function can be called
         * only for classes that were previously instrumented with the
         * the IInstrumentationAdaptor::ModifyClass function. In these cases, the 
         * instrumentation context information must be supplied in the 'context' parameter.
         * 
         * @param[in] classId                   The unique id of this class
         *
         * @param[in] classToInstrument         A client-allocated buffer containing the 
         *                                      the class file to instrument. 
         *                                      The uiActual size member must contain 
         *                                      the exact size of the class file buffer
         *
         * @param[in] funcAllocator             A function for allocating memory for the 
         *                                      instrumented class file
         *
         * @param[out] pInstrumentedClass       A buffer to which the instrumented class
         *                                      is written. This buffer is allocated by 
         *                                      the Adaptor using the supplied
         *                                      allocator function (funcAllocator).
         *                                      If the class was successfully instrumented,
         *                                      the pClassFile member will hold the new
         *                                      class file and the uiSize member
         *                                      will be set to the class file buffer size.
         *                                      In case of an instrumentation error, or
         *                                      when the class cannot or should not be 
         *                                      instrumented, the pClassFile member will be
         *                                      set to NULL and the uiSize member
         *                                      will be set to 0.
         *
         * @param[in] pContext                  An optional pointer to an instrumentation
         *                                      context structure generated by a call to the
         *                                      IInstrumentationAdaptor::ModifyClass function.
         *                                      This argument must contain an exact copy of the
         *                                      data generated by the ModifyClass function.
         *
         * @retval MRTE_RESULT_OK                   Success
         * @retval MRTE_ERROR_INVALID_ARGUMENT      One of the required arguments is NULL
         * @retval MRTE_ERROR_OUT_OF_MEMORY         Not enough memory to complete the
         *                                          operation
         *
         * @retval MRTE_ERROR_INSTRUMENTATION_NOT_NEEDED   The class was not instrumented
         *                                                 at all. This value is returned
         *                                                 for interface classes or when all 
         *                                                 methods were filtered-out
         *
         * @retval MRTE_ERROR_UNABLE_TO_INSTRUMENT    The class cannot be instrumented.
         *                                            This value is returned for classes
         *                                            the Adaptor chooses not to instrument
         *                                            
         * @retval MRTE_ERROR_MODULE_NOT_INITIALIZED  The Adaptor was not yet initialized.
         *                                            Call the Init API to initialize
         *                                            the Adaptor.
         *
         * @retval MRTE_ERROR_ILL_CLASS_FILE          Unable to parse the class file.
         * @retval MRTE_ERROR_FAIL                    Unknown failure
         */
        virtual TResult ModifyByteCodes(MPI::TId classId, 
                                        const SClassFile &classToInstrument,
                                        TMemoryAllocatorFunc funcAllocator,
                                        SClassFile *pInstrumentedClass,
                                        IInstrumentationContext *pContext = NULL) = 0;
        /**
         * @brief Signals the Instrumentation Adaptor module when the JVM has 
         * finished initialization.
         *
         * This function is used to notify the Adaptor that the JVM has 
         * finished its initialization. Typically, this API will be invoked from the
         * "JVM Init Done" event handler. Sometimes, the adaptor applies different 
         * instrumentation depending on whether the JVM completed its initialization or not.
         */
        virtual void JvmInitDone() = 0;

    };

    /**
     * @brief Prototype of Instrumentation Adaptor initialization function
     *
     * The instrumentation adaptor must implement an initialization function 
     * whose name is GetInstrumentationAdaptor and whose type is TGetInstrumentationAdaptor.
     *
     * This function must be exported from the adaptor library as an "extern C" function.
     *
     * The function is used by JPI to instantiate the adaptor and obtain a pointer to its 
     * IInstrumentationAdaptor interface.
     *
     * @return a pointer to the IInstrumentationAdaptor interface
     */ 
    typedef IInstrumentationAdaptor* (*TGetInstrumentationAdaptor)();


}};



#endif // _MRTE_INSTRUMENTATIONDAPTOR_H
