/*****************************************************************************
 * Copyright (c) 1997, 2010 Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/
//
#ifndef MRTE_INSTRUMENTATION_ADAPTOR_BASE
#define MRTE_INSTRUMENTATION_ADAPTOR_BASE

#include "InstrumentationAdaptor.h"
#include "JIE.h"
#include "MVector.h"
#include "MHash.h"
#include "MString.h"
#include "IdAllocator.h"

#include <jni.h>

#define JPI_NS Martini::JPI
#define MPI_NS Martini::MPI

#define SIZE_OF_ARRAY(x) sizeof(x)/sizeof(*x)

U32 SAFE_U64_TO_U32(U64 u64Val);

namespace Martini { namespace Infrastructure {
    
   /**
    * @brief Method information
    *
    * This struct defines method information generated by the CG and Heap Adaptors after 
    * applying class-level instrumentation
    */
    struct SAdaptorMethodInfo
    {
		const char *szName;                             //!< method name
		const char *szSignature;                        //!< method signature
        MPI::TId id;                                    //!< method MPI ID
        JIE::TConstantPoolIndex cpAlreadyInvokedFlag;   //!< constant pool index of the 
                                                        //!  'already invoked' static field
                                                        //!  associated with the method
        unsigned int uiAttributeFlags;                  //!< method access and property flags. 
                                                        //   Used as bitwise OR'ed flags, as 
                                                        //   defined by 
                                                        //   JIE::EMethodAttributeFlags
    };

    /**
     * @brief class information
     *
     * This struct defines class information generated by the CG and Heap Adaptors after
     * applying class-level instrumentation
     */
    struct SAdaptorClassInfo
    {
        const char *szClassName;            //!< class name
        const char *szSourceFileName;       //!< source file name
        unsigned int uiAttributeFlags;      //!< class access and property flags. Used as bitwise  
                                            //   flags, as defined by JIE::EClassAttributeFlags        
        SAdaptorMethodInfo *pMethods;       //!< array of class' methods
        unsigned int uiNumMethods;          //!< number of methods in the methods array
        MPI::TId classId;                   //!< class MPI ID
    };
    
    typedef Martini::RTUtil::MVector<SAdaptorMethodInfo*> TMethodInfoVector;

    typedef Martini::RTUtil::MStrHashNoSync<char, SAdaptorMethodInfo*> TMethodNameToInfoMap;

    /**
     * @brief Heap and CG Adaptor instrumentation context information
     */
    class CContext : public Infrastructure::IInstrumentationContext
    {
    public:
        CContext()
        {
            memset(&classInfo, 0, sizeof(SAdaptorClassInfo));
        }
        SAdaptorClassInfo classInfo;
    };

    /**
     * @brief CInstrumentationAdaptorBase class
     */
    class CInstrumentationAdaptorBase : public IInstrumentationAdaptor
    {
    public:
        CInstrumentationAdaptorBase();
        virtual ~CInstrumentationAdaptorBase();
        
        // IInstrumentationAdaptor methods

        virtual void JvmInitDone();

    protected:

        // Common adaptor functionality

        /**
         * @brief Initializes the instrumentation adaptor
         *
         * Implements initialization common to all adaptors. Adaptor implementations should
         * call this function during initialization.
         *
         * @param[in] pLogger       : A pointer to an initialized Logger object, to be used
         *                            by the instrumentation adaptor for logging messages and 
         *                            reporting fatal errors.
         * @param[in] configFlags   : Optional configuration flags passed to the Martini tool
         *                            during its initialization (see MPI::EConfigurationFlags)
         *
         * @retval MRTE_RESULT_OK                   Success
         * @retval MRTE_ERROR_FAIL                  Initialization error
         */
        TResult Init(Infrastructure::ILogAssert *pLogger = NULL,
                     MPI::BitSet configFlags = MPI::CF_NONE);

        /**
         * @brief Retrieves Java class information
         *
         * @param[out] pInfo    struct to which to write class information. pInfo is assumed
         *                      to be a valid pointer
         * @param[in]  pClass   Java class from which to extract information
         *
         * @retval MRTE_RESULT_OK   success
         * @retval other            error
         */
        TResult GetJavaClassInfo(JIE::SJavaClassInfo *pInfo, JIE::IJavaClass *pClass);

        /**
         * @brief Retrieves Java method information
         *
         * @param[out] pInfo    struct to which to write method information. pInfo is assumed
         *                      to be a valid pointer
         * @param[in]  pClass   Java method from which to extract information
         *
         * @retval MRTE_RESULT_OK   success
         * @retval other            error
         */
        TResult GetJavaMethodInfo(JIE::SJavaMethodInfo *pInfo, JIE::IJavaMethod *pMethod);

        /**
         * @brief Sets pClassBuffer internal members to NULL
         *
         * @param[in] pClassBuffer      class file buffer to reset
         */
        void InvalidateInstrumentedClassBuffer(Infrastructure::SClassFile *pClassBuffer);

        /**
         * @brief Deallocates memory of internal JavaClassInfo members
         *
         * @param[in] pClassInfo    class info to deallocate
         */ 
        void DeallocateJavaClassInfo(JIE::SJavaClassInfo *pClassInfo);

        /**
         * @brief Deallocates memory of internal JavaMethodInfo members
         *
         * @param[in] pMethodInfo   method info to deallocate
         */
        void DeallocateJavaMethodInfo(JIE::SJavaMethodInfo *pMethodInfo);

        void SetClassInfo(SAdaptorClassInfo *pClassInfo,
                          const JIE::SJavaClassInfo &jieInfo, 
                          const MPI_NS::TId uiClassMPIId);
        
        TResult CopyInstrumentedClassInfo(SAdaptorClassInfo *pDest, 
                                          const SAdaptorClassInfo &src,
                                          TMemoryAllocatorFunc funcAllocator);

        void DeallocateAdaptorClassInfo(SAdaptorClassInfo *pInstrClassInfo);

        void DeallocateAdaptorMethodInfo(SAdaptorMethodInfo *pInstrMethodInfo);

        void SetMethodInfo(SAdaptorClassInfo *pClassInstrInfo, 
                           TMethodInfoVector &vecMethods);

        void FreeMethodsVectorItems(TMethodInfoVector &vecMethods);

        // Initializes a map of method names to information
        void InitMethodMap(TMethodNameToInfoMap *pMap, 
                           const SAdaptorClassInfo &classInfo);

        // Retrieves method information from the map
        SAdaptorMethodInfo* GetInfoFromMethodMap(TMethodNameToInfoMap &theMap, 
                                                 const JIE::SJavaMethodInfo &javaMethodInfo);

        // Returns true if StackMapTable attribute calculation (Java 6) is enabled
        bool IsStackMapCalcEnabled();

        // Common adaptor fields

        bool m_bJVMInitDone;                    // Indicates whether the JVM has initialized
        JIE::IJIE* m_pJIE;                      // Java Instrumentation Engine instance
        Infrastructure::ILogAssert *m_pLogger;  // Logger instance
        MPI::BitSet m_configFlags;              // Configuration flags
        
    private:

        /**
         * @brief Loads and initializes JIE
         *
         * @returns a pointer to the JIE interface, or null if JIE could not be initialized
         */
        JIE::IJIE* InitJIE();

    }; // class CInstrumentationAdaptorBase


}} // namespace Martini::Infrastructure

// Logging macros

#define LOG_INFORMATIVE(module, iLvl, bEC, szMsg) \
    if (m_pLogger) m_pLogger->Informative(module, iLvl, bEC, szMsg)
#define LOG_INFORMATIVE1(module, iLvl, bEC, szMsg, param1) \
    if (m_pLogger) m_pLogger->Informative(module, iLvl, bEC, szMsg, param1)
#define LOG_INFORMATIVE2(module, iLvl, bEC, szMsg, param1, param2) \
    if (m_pLogger) m_pLogger->Informative(module, iLvl, bEC, szMsg, param1, param2)
#define LOG_INFORMATIVE3(module, iLvl, bEC, szMsg, param1, param2, param3) \
    if (m_pLogger) m_pLogger->Informative(module, iLvl, bEC, szMsg, param1, param2, param3)
#define LOG_INFORMATIVE4(module, iLvl, bEC, szMsg, param1, param2, param3, param4) \
    if (m_pLogger) m_pLogger->Informative(module, iLvl, bEC, szMsg, param1, param2, param3, param4)

#define LOG_ERROR(module, szMsg) \
    if (m_pLogger) m_pLogger->Error(module, szMsg)
#define LOG_ERROR2(module, szMsg, param1, param2) \
    if (m_pLogger) m_pLogger->Error(module, szMsg, param1, param2)
#define LOG_ERROR3(module, szMsg, param1, param2, param3) \
    if (m_pLogger) m_pLogger->Error(module, szMsg, param1, param2, param3)

#endif // #define MRTE_INSTRUMENTATION_ADAPTOR_BASE
