/*****************************************************************************
 * Copyright (c) 1997-2007, Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/

#ifndef __LOGASSERT_H__
#define __LOGASSERT_H__

#include "MpiAPI.h"
#include "ECAgent.h"
#include "Logger.h"
#include "OSA.h"
#include <stdarg.h>
#include "ILogAssert.h"

namespace Martini { namespace Infrastructure
{
    class CLogAssert : public ILogAssert 
    {
    public:
        // ILogAssert methods

        /**
         * @see ILogAssert::Error
         */
        void Error(const char* szClient, const char* szFormatMsg, ...);

        /**
         * @see ILogAssert::Informative
         */
        void Informative(const char* szClient, int iLevel, bool bExposeToEC, const char* szFormatMsg, ...);

        /**
         * @see ILogAssert::Debug
         */
	    void Debug(const char* szClient, int iLevel, const char* szFormatMsg, ...);

        // CLogAssert methods

        virtual ~CLogAssert();

        /**
         * @brief Returns the singleton instance of the logger
         *
         * @return the singleton instance of the logger
         */
        static CLogAssert *GetInstance();

        /**
         * @brief Initializes the logger
         *
         * @param[in] pEC       a pointer to the External Control Agent. Used for forwarding
         *                      messages to the External Controller.
         * @param[in] bInitDone whether initialization completed. If false, it means that log
         *                      messages should be buffered in memory as the log file output
         *                      was not yet defined.
         *
         * @retval  MRTE_RESULT_OK              success
         * @retval  MRTE_ERROR_OUT_OF_MEMORY    not enough memory to initialize the logger
         * @retval  MRTE_ERROR_OSA_FAILURE      failed to create logger critical section
         * @retval  MRTE_FILE_FAILURE           failed to create the log file. Most likely
         *                                      the directory does not exist or no write 
         *                                      permissions.
         * @retval  MRTE_ERROR_FAIL             other failure
         */
        TResult Initialize(CECAgent* pEC, bool bInitDone);

        virtual void HandleEvent(MPI::SSetOutputDirEventData &data);
        virtual unsigned int EventDataTypes();
 
        CLogger* GetLoggerInstance();
    private:
        CLogAssert();
        CLogger* m_pLogger;
        CECAgent* m_pEC;

        char* m_szDirectory;
        int m_iLogLevel;
        unsigned int m_uiConsoleOutput;
        unsigned int m_uiReplaceIfExist;
        bool m_bInitDone;

        OSA::IThreadSync* m_csDirectoryChangeSync;
        OSA::IThreadSync* m_csLogAssertCall;
    };
    
}}


#define MARTINI_ERROR(module, szMsg) \
    CLogAssert::GetInstance()->Error(module, szMsg)
#define MARTINI_ERROR1(module, szMsg, param1) \
    CLogAssert::GetInstance()->Error(module, szMsg, param1)
#define MARTINI_ERROR2(module, szMsg, param1, param2) \
    CLogAssert::GetInstance()->Error(module, szMsg, param1, param2)
#define MARTINI_ERROR3(module, szMsg, param1, param2, param3) \
    CLogAssert::GetInstance()->Error(module, szMsg, param1, param2, param3)
#define MARTINI_ERROR4(module, szMsg, param1, param2, param3, param4) \
    CLogAssert::GetInstance()->Error(module, szMsg, param1, param2, param3, param4)

#define MARTINI_INFORMATIVE(module, iLvl, bEC, szMsg) \
    CLogAssert::GetInstance()->Informative(module, iLvl, bEC, szMsg)
#define MARTINI_INFORMATIVE1(module, iLvl, bEC, szMsg, param1) \
    CLogAssert::GetInstance()->Informative(module, iLvl, bEC, szMsg, param1)
#define MARTINI_INFORMATIVE2(module, iLvl, bEC, szMsg, param1, param2) \
    CLogAssert::GetInstance()->Informative(module, iLvl, bEC, szMsg, param1, param2)
#define MARTINI_INFORMATIVE3(module, iLvl, bEC, szMsg, param1, param2, param3) \
    CLogAssert::GetInstance()->Informative(module, iLvl, bEC, szMsg, param1, param2, param3)
#define MARTINI_INFORMATIVE4(module, iLvl, bEC, szMsg, param1, param2, param3, param4) \
    CLogAssert::GetInstance()->Informative(module, iLvl, bEC, szMsg, param1, param2, param3, param4)

#ifdef _DEBUG
    #define MARTINI_DEBUG(module, iLvl, szMsg) \
        CLogAssert::GetInstance()->Debug(module, iLvl, szMsg)
    #define MARTINI_DEBUG1(module, iLvl, szMsg, param1) \
        CLogAssert::GetInstance()->Debug(module, iLvl, szMsg, param1)
    #define MARTINI_DEBUG2(module, iLvl, szMsg, param1, param2) \
        CLogAssert::GetInstance()->Debug(module, iLvl, szMsg, param1, param2)
    #define MARTINI_DEBUG3(module, iLvl, szMsg, param1, param2, param3) \
        CLogAssert::GetInstance()->Debug(module, iLvl, szMsg, param1, param2, param3)
    #define MARTINI_DEBUG4(module, iLvl, szMsg, param1, param2, param3, param4) \
        CLogAssert::GetInstance()->Debug(module, iLvl, szMsg, param1, param2, param3, param4)
    #define MARTINI_DEBUG5(module, iLvl, szMsg, param1, param2, param3, param4, param5) \
        CLogAssert::GetInstance()->Debug(module, iLvl, szMsg, param1, param2, param3, param4, param5)
#else   //_DEBUG
    #define MARTINI_DEBUG(module, iLvl, szMsg) 
    #define MARTINI_DEBUG1(module, iLvl, szMsg, param1) 
    #define MARTINI_DEBUG2(module, iLvl, szMsg, param1, param2) 
    #define MARTINI_DEBUG3(module, iLvl, szMsg, param1, param2, param3) 
    #define MARTINI_DEBUG4(module, iLvl, szMsg, param1, param2, param3, param4) 
    #define MARTINI_DEBUG5(module, iLvl, szMsg, param1, param2, param3, param4, param5) 
#endif  //_DEBUG


#define MARTINI_ASSERT(module, cond, szMsg) \
    if (!(cond)) CLogAssert::GetInstance()->Error(module, \
    "File: %s (Line: %d) condition failed: %s %s", __FILE__, __LINE__, #cond, szMsg)


#endif 
 

