/*****************************************************************************
 * Copyright (c) 1997-2007, Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/
//

#include "BCIToMIEInstructionMapper.h"
#include "MIEInstructionFactory.h"
#include "ValidityChecks.h"

#include "JVMInsSet.h"
#include "CommonDef.h"

#include <assert.h>

#define SIZE_OF_ARRAY(x) sizeof(x)/sizeof(*x)

using namespace Martini::MIE;

//////////////////////////////////////////////////////////////////////////
// class CBCIToMIEInstructionMapper implementation

// BCI instruction descriptors (vector)
static CInsSetJ BCIInsSet;

// Mnemonics used for the "wide" version of load/store operations
static struct {
	BYTE opcode;
	const char* mnemonic;
    SemTag_t semtag;
} wide_load_store_mnemonic[] = {
	{ JVMI_iload, "wide iload", SEM_GEN },
	{ JVMI_aload, "wide aload", SEM_GEN },
	{ JVMI_dload, "wide dload", SEM_GEN },
	{ JVMI_fload, "wide fload", SEM_GEN },
	{ JVMI_lload, "wide lload", SEM_GEN },
	{ JVMI_istore, "wide istore", SEM_GEN },
	{ JVMI_astore, "wide astore", SEM_GEN },
	{ JVMI_dstore, "wide dstore", SEM_GEN },
	{ JVMI_fstore, "wide fstore", SEM_GEN },
	{ JVMI_lstore, "wide lstore", SEM_GEN },
	{ JVMI_ret, "wide ret", SEM_RETSR }
};

CBCIToMIEInstructionMapper::CBCIToMIEInstructionMapper()
{
    static CNoOperandsMapper no_op_mappers[] = 
    {
        CNoOperandsMapper(MNM_NOP), // nop
        CNoOperandsMapper(MNM_ACONST_NULL), // aconst_null
        CNoOperandsMapper(MNM_ICONST_M1), // iconst_m1
        CNoOperandsMapper(MNM_ICONST_0), // iconst_0
        CNoOperandsMapper(MNM_ICONST_1), // iconst_1
        CNoOperandsMapper(MNM_ICONST_2), // iconst_2
        CNoOperandsMapper(MNM_ICONST_3), // iconst_3
        CNoOperandsMapper(MNM_ICONST_4), // iconst_4
        CNoOperandsMapper(MNM_ICONST_5), // iconst_5
        CNoOperandsMapper(MNM_LCONST_0), // lconst_0
        CNoOperandsMapper(MNM_LCONST_1), // lconst_1
        CNoOperandsMapper(MNM_FCONST_0), // fconst_0
        CNoOperandsMapper(MNM_FCONST_1), // fconst_1
        CNoOperandsMapper(MNM_FCONST_2), // fconst_2
        CNoOperandsMapper(MNM_DCONST_0), // dconst_0
        CNoOperandsMapper(MNM_DCONST_1), // dconst_1
        CNoOperandsMapper(MNM_ILOAD_0), // iload_0
        CNoOperandsMapper(MNM_ILOAD_1), // iload_1
        CNoOperandsMapper(MNM_ILOAD_2), // iload_2
        CNoOperandsMapper(MNM_ILOAD_3), // iload_3
        CNoOperandsMapper(MNM_LLOAD_0), // lload_0
        CNoOperandsMapper(MNM_LLOAD_1), // lload_1
        CNoOperandsMapper(MNM_LLOAD_2), // lload_2
        CNoOperandsMapper(MNM_LLOAD_3), // lload_3
        CNoOperandsMapper(MNM_FLOAD_0), // fload_0
        CNoOperandsMapper(MNM_FLOAD_1), // fload_1
        CNoOperandsMapper(MNM_FLOAD_2), // fload_2
        CNoOperandsMapper(MNM_FLOAD_3), // fload_3
        CNoOperandsMapper(MNM_DLOAD_0), // dload_0
        CNoOperandsMapper(MNM_DLOAD_1), // dload_1
        CNoOperandsMapper(MNM_DLOAD_2), // dload_2
        CNoOperandsMapper(MNM_DLOAD_3), // dload_3
        CNoOperandsMapper(MNM_ALOAD_0), // aload_0
        CNoOperandsMapper(MNM_ALOAD_1), // aload_1
        CNoOperandsMapper(MNM_ALOAD_2), // aload_2
        CNoOperandsMapper(MNM_ALOAD_3), // aload_3
        CNoOperandsMapper(MNM_IALOAD), // iaload
        CNoOperandsMapper(MNM_LALOAD), // laload
        CNoOperandsMapper(MNM_FALOAD), // faload
        CNoOperandsMapper(MNM_DALOAD), // daload
        CNoOperandsMapper(MNM_AALOAD), // aaload
        CNoOperandsMapper(MNM_BALOAD), // baload
        CNoOperandsMapper(MNM_CALOAD), // caload
        CNoOperandsMapper(MNM_SALOAD), // saload
        CNoOperandsMapper(MNM_ISTORE_0), // istore_0
        CNoOperandsMapper(MNM_ISTORE_1), // istore_1
        CNoOperandsMapper(MNM_ISTORE_2), // istore_2
        CNoOperandsMapper(MNM_ISTORE_3), // istore_3
        CNoOperandsMapper(MNM_LSTORE_0), // lstore_0
        CNoOperandsMapper(MNM_LSTORE_1), // lstore_1
        CNoOperandsMapper(MNM_LSTORE_2), // lstore_2
        CNoOperandsMapper(MNM_LSTORE_3), // lstore_3
        CNoOperandsMapper(MNM_FSTORE_0), // fstore_0
        CNoOperandsMapper(MNM_FSTORE_1), // fstore_1
        CNoOperandsMapper(MNM_FSTORE_2), // fstore_2
        CNoOperandsMapper(MNM_FSTORE_3), // fstore_3
        CNoOperandsMapper(MNM_DSTORE_0), // dstore_0
        CNoOperandsMapper(MNM_DSTORE_1), // dstore_1
        CNoOperandsMapper(MNM_DSTORE_2), // dstore_2
        CNoOperandsMapper(MNM_DSTORE_3), // dstore_3
        CNoOperandsMapper(MNM_ASTORE_0), // astore_0
        CNoOperandsMapper(MNM_ASTORE_1), // astore_1
        CNoOperandsMapper(MNM_ASTORE_2), // astore_2
        CNoOperandsMapper(MNM_ASTORE_3), // astore_3
        CNoOperandsMapper(MNM_IASTORE), // iastore
        CNoOperandsMapper(MNM_LASTORE), // lastore
        CNoOperandsMapper(MNM_FASTORE), // fastore
        CNoOperandsMapper(MNM_DASTORE), // dastore
        CNoOperandsMapper(MNM_AASTORE), // aastore
        CNoOperandsMapper(MNM_BASTORE), // bastore
        CNoOperandsMapper(MNM_CASTORE), // castore
        CNoOperandsMapper(MNM_SASTORE), // sastore
        CNoOperandsMapper(MNM_POP), // pop
        CNoOperandsMapper(MNM_POP2), // pop2
        CNoOperandsMapper(MNM_DUP), // dup
        CNoOperandsMapper(MNM_DUP_X1), // dup_x1
        CNoOperandsMapper(MNM_DUP_X2), // dup_x2
        CNoOperandsMapper(MNM_DUP2), // dup2
        CNoOperandsMapper(MNM_DUP2_X1), // dup2_x1
        CNoOperandsMapper(MNM_DUP2_X2), // dup2_x2
        CNoOperandsMapper(MNM_SWAP), // swap
        CNoOperandsMapper(MNM_IADD), // iadd
        CNoOperandsMapper(MNM_LADD), // ladd
        CNoOperandsMapper(MNM_FADD), // fadd
        CNoOperandsMapper(MNM_DADD), // dadd
        CNoOperandsMapper(MNM_ISUB), // isub
        CNoOperandsMapper(MNM_LSUB), // lsub
        CNoOperandsMapper(MNM_FSUB), // fsub
        CNoOperandsMapper(MNM_DSUB), // dsub
        CNoOperandsMapper(MNM_IMUL), // imul
        CNoOperandsMapper(MNM_LMUL), // lmul
        CNoOperandsMapper(MNM_FMUL), // fmul
        CNoOperandsMapper(MNM_DMUL), // dmul
        CNoOperandsMapper(MNM_IDIV), // idiv
        CNoOperandsMapper(MNM_LDIV), // ldiv
        CNoOperandsMapper(MNM_FDIV), // fdiv
        CNoOperandsMapper(MNM_DDIV), // ddiv
        CNoOperandsMapper(MNM_IREM), // irem
        CNoOperandsMapper(MNM_LREM), // lrem
        CNoOperandsMapper(MNM_FREM), // frem
        CNoOperandsMapper(MNM_DREM), // drem
        CNoOperandsMapper(MNM_INEG), // ineg
        CNoOperandsMapper(MNM_LNEG), // lneg
        CNoOperandsMapper(MNM_FNEG), // fneg
        CNoOperandsMapper(MNM_DNEG), // dneg
        CNoOperandsMapper(MNM_ISHL), // ishl
        CNoOperandsMapper(MNM_LSHL), // lshl
        CNoOperandsMapper(MNM_ISHR), // ishr
        CNoOperandsMapper(MNM_LSHR), // lshr
        CNoOperandsMapper(MNM_IUSHR), // iushr
        CNoOperandsMapper(MNM_LUSHR), // lushr
        CNoOperandsMapper(MNM_IAND), // iand
        CNoOperandsMapper(MNM_LAND), // land
        CNoOperandsMapper(MNM_IOR), // ior
        CNoOperandsMapper(MNM_LOR), // lor
        CNoOperandsMapper(MNM_IXOR), // ixor
        CNoOperandsMapper(MNM_LXOR), // lxor
        CNoOperandsMapper(MNM_I2L), // i2l
        CNoOperandsMapper(MNM_I2F), // i2f
        CNoOperandsMapper(MNM_I2D), // i2d
        CNoOperandsMapper(MNM_L2I), // l2i
        CNoOperandsMapper(MNM_L2F), // l2f
        CNoOperandsMapper(MNM_L2D), // l2d
        CNoOperandsMapper(MNM_F2I), // f2i
        CNoOperandsMapper(MNM_F2L), // f2l
        CNoOperandsMapper(MNM_F2D), // f2d
        CNoOperandsMapper(MNM_D2I), // d2i
        CNoOperandsMapper(MNM_D2L), // d2l
        CNoOperandsMapper(MNM_D2F), // d2f
        CNoOperandsMapper(MNM_I2B), // i2b
        CNoOperandsMapper(MNM_I2C), // i2c
        CNoOperandsMapper(MNM_I2S), // i2s
        CNoOperandsMapper(MNM_LCMP), // lcmp
        CNoOperandsMapper(MNM_FCMPL), // fcmpl
        CNoOperandsMapper(MNM_FCMPG), // fcmpg
        CNoOperandsMapper(MNM_DCMPL), // dcmpl
        CNoOperandsMapper(MNM_DCMPG), // dcmpg
        CNoOperandsMapper(MNM_IRETURN), // ireturn
        CNoOperandsMapper(MNM_LRETURN), // lreturn
        CNoOperandsMapper(MNM_FRETURN), // freturn
        CNoOperandsMapper(MNM_DRETURN), // dreturn
        CNoOperandsMapper(MNM_ARETURN), // areturn
        CNoOperandsMapper(MNM_RETURN), // return
        CNoOperandsMapper(MNM_XXXUNUSEDXXX), // xxxunusedxxx
        CNoOperandsMapper(MNM_ARRAYLENGTH), // arraylength
        CNoOperandsMapper(MNM_ATHROW), // athrow
        CNoOperandsMapper(MNM_MONITORENTER), // monitorenter
        CNoOperandsMapper(MNM_MONITOREXIT), // monitorexit
    };
    static CCPIndexOperandMapper cpindex_op_mappers[] = 
    {
        CCPIndexOperandMapper(MNM_LDC_W), // ldc_w
        CCPIndexOperandMapper(MNM_LDC2_W), // ldc2_w
        CCPIndexOperandMapper(MNM_GETSTATIC), // getstatic
        CCPIndexOperandMapper(MNM_PUTSTATIC), // putstatic
        CCPIndexOperandMapper(MNM_GETFIELD), // getfield
        CCPIndexOperandMapper(MNM_PUTFIELD), // putfield
        CCPIndexOperandMapper(MNM_INVOKEVIRTUAL), // invokevirtual
        CCPIndexOperandMapper(MNM_INVOKESPECIAL), // invokespecial
        CCPIndexOperandMapper(MNM_INVOKESTATIC), // invokestatic
        CCPIndexOperandMapper(MNM_NEW), // new
        CCPIndexOperandMapper(MNM_ANEWARRAY), // anewarray
        CCPIndexOperandMapper(MNM_CHECKCAST), // checkcast
        CCPIndexOperandMapper(MNM_INSTANCEOF), // instanceof
    };
    static CLocalVarOperandMapper localvar_op_mappers[] = 
    {
        CLocalVarOperandMapper(MNM_ILOAD), // iload
        CLocalVarOperandMapper(MNM_LLOAD), // lload
        CLocalVarOperandMapper(MNM_FLOAD), // fload
        CLocalVarOperandMapper(MNM_DLOAD), // dload
        CLocalVarOperandMapper(MNM_ALOAD), // aload
        CLocalVarOperandMapper(MNM_ISTORE), // istore
        CLocalVarOperandMapper(MNM_LSTORE), // lstore
        CLocalVarOperandMapper(MNM_FSTORE), // fstore
        CLocalVarOperandMapper(MNM_DSTORE), // dstore
        CLocalVarOperandMapper(MNM_ASTORE), // astore
        CLocalVarOperandMapper(MNM_RET), // ret
    };
    static CWordOffsetOperandMapper word_op_mappers[] = 
    {
        CWordOffsetOperandMapper(MNM_IFEQ), // ifeq
        CWordOffsetOperandMapper(MNM_IFNE), // ifne
        CWordOffsetOperandMapper(MNM_IFLT), // iflt
        CWordOffsetOperandMapper(MNM_IFGE), // ifge
        CWordOffsetOperandMapper(MNM_IFGT), // ifgt
        CWordOffsetOperandMapper(MNM_IFLE), // ifle
        CWordOffsetOperandMapper(MNM_IF_ICMPEQ), // if_icmpeq
        CWordOffsetOperandMapper(MNM_IF_ICMPNE), // if_icmpne
        CWordOffsetOperandMapper(MNM_IF_ICMPLT), // if_icmplt
        CWordOffsetOperandMapper(MNM_IF_ICMPGE), // if_icmpge
        CWordOffsetOperandMapper(MNM_IF_ICMPGT), // if_icmpgt
        CWordOffsetOperandMapper(MNM_IF_ICMPLE), // if_icmple
        CWordOffsetOperandMapper(MNM_IF_ACMPEQ), // if_acmpeq
        CWordOffsetOperandMapper(MNM_IF_ACMPNE), // if_acmpne
        CWordOffsetOperandMapper(MNM_GOTO), // goto
        CWordOffsetOperandMapper(MNM_JSR), // jsr
        CWordOffsetOperandMapper(MNM_IFNULL), // ifnull
        CWordOffsetOperandMapper(MNM_IFNONNULL), // ifnonnull
    };
    static CLongOffsetOperandMapper long_op_mappers[] = 
    {
        CLongOffsetOperandMapper(MNM_GOTO_W), // goto_w
        CLongOffsetOperandMapper(MNM_JSR_W) // jsr_w
    };
    static CByteOperandMapper byte_op_mapper = CByteOperandMapper(MNM_BIPUSH);
    static CShortOperandMapper short_op_mapper = CShortOperandMapper(MNM_SIPUSH);
    static CLdcOperandMapper ldc_op_mapper = CLdcOperandMapper(MNM_LDC);
    static CTableswitchOperandMapper tableswitch_op_mapper = CTableswitchOperandMapper(MNM_TABLESWITCH);
    static CLookupswitchOperandMapper lookupswitch_op_mapper = CLookupswitchOperandMapper(MNM_LOOKUPSWITCH);
    static CInvokeinterfaceOperandMapper invokeinterface_op_mapper = CInvokeinterfaceOperandMapper(MNM_INVOKEINTERFACE);
    static CNewarrayOperandMapper newarray_op_mapper = CNewarrayOperandMapper(MNM_NEWARRAY);
    static CWideOperandMapper wide_op_mapper = CWideOperandMapper(MNM_WIDE);
    static CMultianewarrayOperandMapper multianewarray_op_mapper = CMultianewarrayOperandMapper(MNM_MULTIANEWARRAY);
    static CIincOperandMapper iinc_op_mapper = CIincOperandMapper(MNM_IINC);

    // merge mapper array into one array    
    m_uiMappersCount = 202;
    m_vMappers = new COperandMapper*[m_uiMappersCount];
    int i;
    for (i = 0; i < SIZE_OF_ARRAY(no_op_mappers); ++i)
    {
        m_vMappers[no_op_mappers[i].GetMnemonic()] = &(no_op_mappers[i]);
    }
    for (i = 0; i < SIZE_OF_ARRAY(cpindex_op_mappers); ++i)
    {
        m_vMappers[cpindex_op_mappers[i].GetMnemonic()] = &(cpindex_op_mappers[i]);
    }
    for (i = 0; i < SIZE_OF_ARRAY(localvar_op_mappers); ++i)
    {
        m_vMappers[localvar_op_mappers[i].GetMnemonic()] = &(localvar_op_mappers[i]);
    }
    for (i = 0; i < SIZE_OF_ARRAY(word_op_mappers); ++i)
    {
        m_vMappers[word_op_mappers[i].GetMnemonic()] = &(word_op_mappers[i]);
    }
    for (i = 0; i < SIZE_OF_ARRAY(long_op_mappers); ++i)
    {
        m_vMappers[long_op_mappers[i].GetMnemonic()] = &(long_op_mappers[i]);
    }
    m_vMappers[byte_op_mapper.GetMnemonic()] = &byte_op_mapper;
    m_vMappers[short_op_mapper.GetMnemonic()] = &short_op_mapper;
    m_vMappers[ldc_op_mapper.GetMnemonic()] = &ldc_op_mapper;
    m_vMappers[tableswitch_op_mapper.GetMnemonic()] = &tableswitch_op_mapper;
    m_vMappers[lookupswitch_op_mapper.GetMnemonic()] = &lookupswitch_op_mapper;
    m_vMappers[invokeinterface_op_mapper.GetMnemonic()] = &invokeinterface_op_mapper;
    m_vMappers[newarray_op_mapper.GetMnemonic()] = &newarray_op_mapper;
    m_vMappers[wide_op_mapper.GetMnemonic()] = &wide_op_mapper;
    m_vMappers[multianewarray_op_mapper.GetMnemonic()] = &multianewarray_op_mapper;
    m_vMappers[iinc_op_mapper.GetMnemonic()] = &iinc_op_mapper;

    // initialize the special instruction encoders array
    static CLoadInstOffsetEncoder load_inst_offset_encoder;
    m_uiEncodersSize = 1;
    m_vEncoders = new CSpecialInstEncoder*[m_uiEncodersSize];
    m_vEncoders[0] = &load_inst_offset_encoder;

}

CBCIToMIEInstructionMapper::~CBCIToMIEInstructionMapper()
{
    delete [] m_vMappers;
    delete [] m_vEncoders;
}

CMIEInstruction* CBCIToMIEInstructionMapper::BCI2MIE(CInstruction *pBCIInst)
{
    unsigned int uiOpCode = pBCIInst->GetOpCode();
    if (uiOpCode >= m_uiMappersCount)
    {
        return NULL;
    }
    // extract the correct mapper according to the op-code and use it
    COperandMapper *pMapper = (COperandMapper*)m_vMappers[uiOpCode];
    CMIEInstruction *pMIEInst = pMapper->BCI2MIE(pBCIInst);
    return pMIEInst;
}

CInstruction* CBCIToMIEInstructionMapper::MIE2BCI(CMIEInstruction *pMIEInst)
{
    unsigned int uiOpCode = pMIEInst->GetMnemonic();
    if (uiOpCode >= m_uiMappersCount)
    {
        return NULL;
    }
    // extract the correct mapper according to the op-code and use it
    COperandMapper *pMapper = (COperandMapper*)m_vMappers[uiOpCode];
    CInstruction *pBCIInst = pMapper->MIE2BCI(pMIEInst);
    return pBCIInst;
}

void CBCIToMIEInstructionMapper::EncodeSpecialInstruction(TBciInstructionList *pBciInstList,
                                                          CMIEInstruction *pMieSpecialIsnt,
                                                          CJClassBuilder *pBciClassBuilder)
{
    EMnemonic mnem = pMieSpecialIsnt->GetMnemonic();
    if (mnem < MNM_FIRST_MIE_SPECIAL || mnem >= MNM_LAST_MIE_SPECIAL)
    {
        return;
    }
    m_vEncoders[mnem - MNM_FIRST_MIE_SPECIAL]->Encode(pBciInstList, pMieSpecialIsnt,
        pBciClassBuilder);
}

//////////////////////////////////////////////////////////////////////////
// class COperandMapper implementation

void COperandMapper::InitMIEInstruction(CMIEInstruction *pInstToInit, 
                                     const CInstruction *pBCIInst)
{
    pInstToInit->SetOriginalOffset(pBCIInst->GetIP());
    pInstToInit->SetStatus(CMIEInstruction::IS_ORIGINAL);
}

//////////////////////////////////////////////////////////////////////////
// class CNoOperandsMapper implementation

CMIEInstruction* CNoOperandsMapper::BCI2MIE(CInstruction *pBCIInst)
{
    // create an MIE instruction object with no operands
    CMIEInstruction* pMIEInst = CMIEInstructionFactory::CreateInstruction(
        pBCIInst->GetOpCode());
    InitMIEInstruction(pMIEInst, pBCIInst);
    return pMIEInst;
}

CInstruction* CNoOperandsMapper::MIE2BCI(CMIEInstruction *pMIEInst)
{
    EMnemonic mnem = pMIEInst->GetMnemonic();
    JVMI_t bciMnem = (JVMI_t)mnem;
    CInstruction *pBCIInst = CInsSetJ::Create_simple(bciMnem);
    return pBCIInst;
}

//////////////////////////////////////////////////////////////////////////
// class CByteOperandMapper implementation

CMIEInstruction* CByteOperandMapper::BCI2MIE(CInstruction *pBCIInst)
{
    CMIEInstruction* pMIEInst = CMIEInstructionFactory::CreateInstruction(
        pBCIInst->GetOpCode());
    InitMIEInstruction(pMIEInst, pBCIInst);
    SOperand op;
    op.type = OT_INT8;
    op.val.s8Op = pBCIInst->GetCode()[1];
    pMIEInst->InternalSetOperand(&op);
    return pMIEInst;
}

CInstruction* CByteOperandMapper::MIE2BCI(CMIEInstruction *pMIEInst)
{
    SOperand op;
    TResult res = pMIEInst->GetOperand(&op);
    if (res != MRTE_RESULT_OK)
    {
        return NULL;
    }
    if (op.type != OT_INT8)
    {
        // 1st operand is not a Byte
        return NULL;
    }
    EMnemonic mnem = pMIEInst->GetMnemonic();
    JVMI_t bciMnem = (JVMI_t)mnem;
	CInsDescr* pDescr = BCIInsSet[bciMnem];
	unsigned char code[2] = {bciMnem};
	BYTE arg = (BYTE)op.val.s8Op;
	*(BYTE*)&code[1] = arg; // no need of memcpy as this is a byte
	CInstruction *pBCIInst = new CInstruction(pDescr->GetMnem(), 
	    pDescr->GetSemTag(), 
	    code, SIZE_OF(code), pDescr->GetStack());
    return pBCIInst;
}

//////////////////////////////////////////////////////////////////////////
// class CShortOperandMapper implementation

CMIEInstruction* CShortOperandMapper::BCI2MIE(CInstruction *pBCIInst)
{
    CMIEInstruction* pMIEInst = CMIEInstructionFactory::CreateInstruction(
        pBCIInst->GetOpCode());
    InitMIEInstruction(pMIEInst, pBCIInst);
    SOperand op;
    op.type = OT_INT16;
    S16 shortVal = (pBCIInst->GetCode()[1] << 8) | pBCIInst->GetCode()[2];
    op.val.s16Op = shortVal;
    pMIEInst->InternalSetOperand(&op);
    return pMIEInst;
}

CInstruction* CShortOperandMapper::MIE2BCI(CMIEInstruction *pMIEInst)
{
    SOperand op;
    TResult res = pMIEInst->GetOperand(&op);
    if (res != MRTE_RESULT_OK)
    {
        return NULL;
    }
    if (op.type != OT_INT16)
    {
        // 1st operand is not a Short
        return NULL;
    }
    EMnemonic mnem = pMIEInst->GetMnemonic();
    JVMI_t bciMnem = (JVMI_t)mnem;
	BYTE code[3] = {bciMnem};
	short scratch = LE_WORD(op.val.s16Op);
	MEMCPY2(&code[1], &scratch);
	CInsDescr* pDescr = BCIInsSet[bciMnem];
	CInstruction *pBCIInst = new CInstruction(pDescr->GetMnem(), 
		pDescr->GetSemTag(), code, SIZE_OF(code),
		pDescr->GetStack());
    return pBCIInst;
}

//////////////////////////////////////////////////////////////////////////
// class CLdcOperandMapper implementation

CMIEInstruction* CLdcOperandMapper::BCI2MIE(CInstruction *pBCIInst)
{
    CMIEInstruction* pMIEInst = CMIEInstructionFactory::CreateInstruction(
        pBCIInst->GetOpCode());
    InitMIEInstruction(pMIEInst, pBCIInst);
    SOperand op;
    op.type = OT_JAVA_CP_INDEX;
    op.val.cpIndex = pBCIInst->GetCode()[1];
    pMIEInst->InternalSetOperand(&op);
    return pMIEInst;
}

CInstruction* CLdcOperandMapper::MIE2BCI(CMIEInstruction *pMIEInst)
{
    SOperand op;
    TResult res = pMIEInst->GetOperand(&op);
    if (res != MRTE_RESULT_OK)
    {
        return NULL;
    }
    if (op.type != OT_JAVA_CP_INDEX)
    {
        // 1st operand is not a constant-pool index
        return NULL;
    }
    EMnemonic mnem = pMIEInst->GetMnemonic();
    JVMI_t bciMnem = (JVMI_t)mnem;
    CInstruction *pBCIInst = CInsSetJ::Create_ldc(op.val.cpIndex);
    return pBCIInst;
}

//////////////////////////////////////////////////////////////////////////
// class CCPIndexOperandMapper implementation

CMIEInstruction* CCPIndexOperandMapper::BCI2MIE(CInstruction *pBCIInst)
{
    CMIEInstruction* pMIEInst = CMIEInstructionFactory::CreateInstruction(
        pBCIInst->GetOpCode());
    InitMIEInstruction(pMIEInst, pBCIInst);
    SOperand op;
    op.type = OT_JAVA_CP_INDEX;
    op.val.cpIndex = (pBCIInst->GetCode()[1] << 8) | pBCIInst->GetCode()[2];
    pMIEInst->InternalSetOperand(&op);
    return pMIEInst;
}

CInstruction* CCPIndexOperandMapper::MIE2BCI(CMIEInstruction *pMIEInst)
{
    SOperand op;
    TResult res = pMIEInst->GetOperand(&op);
    if (res != MRTE_RESULT_OK)
    {
        return NULL;
    }
    if (op.type != OT_JAVA_CP_INDEX)
    {
        // 1st operand is not a constant-pool index
        return NULL;
    }
    EMnemonic mnem = pMIEInst->GetMnemonic();
    JVMI_t bciMnem = (JVMI_t)mnem;
	BYTE code[3] = {bciMnem};
	short scratch = LE_WORD(op.val.cpIndex);
	MEMCPY2(&code[1], &scratch);
	CInsDescr* pDescr = BCIInsSet[bciMnem];
	CInstruction *pBCIInst;
    if (bciMnem >= JVMI_getstatic && bciMnem <= JVMI_putfield)
    {
        pBCIInst = new CInstruction_GetPutJ(pDescr->GetMnem(), code, SIZE_OF(code),
		    pDescr->GetStack());
    }
    else if (pDescr->GetSemTag() == SEM_CALL)
    {
        pBCIInst = new CInstruction_InvokeJ(pDescr->GetMnem(), code, SIZE_OF(code),
		    pDescr->GetStack());
    }
    else
    {
        pBCIInst = new CInstruction(pDescr->GetMnem(), pDescr->GetSemTag(), code, 
            SIZE_OF(code), pDescr->GetStack());
    }
    return pBCIInst;
}

//////////////////////////////////////////////////////////////////////////
// class CLocalVarOperandMapper implementation

CMIEInstruction* CLocalVarOperandMapper::BCI2MIE(CInstruction *pBCIInst)
{
    //TODO: verify code - how does it behave for "wide" instructions??? does BCI create "wide" for wide instruction or regular opcodes?
    CMIEInstruction* pMIEInst = CMIEInstructionFactory::CreateInstruction(
        pBCIInst->GetOpCode());
    InitMIEInstruction(pMIEInst, pBCIInst);
    SOperand op;
    op.type = OT_VAR_ID;
    op.val.varID = pBCIInst->GetCode()[1];
    pMIEInst->InternalSetOperand(&op);
    return pMIEInst;
}

CInstruction* CLocalVarOperandMapper::MIE2BCI(CMIEInstruction *pMIEInst)
{
    //TEST: both regular and "wide" versions
    SOperand op;
    TResult res = pMIEInst->GetOperand(&op);
    if (res != MRTE_RESULT_OK)
    {
        return NULL;
    }
    if (op.type != OT_VAR_ID)
    {
        // 1st operand is not a variable index
        return NULL;
    }
    EMnemonic mnem = pMIEInst->GetMnemonic();
    JVMI_t bciMnem = (JVMI_t)mnem;
	BYTE code[4];
	int codeSize;
	CInsDescr* pDescr = BCIInsSet[bciMnem];
    const char *szMnem;
    SemTag_t semtag;
	if (op.val.varID > 255)
    {
		// Use wide form
		code[0] = JVMI_wide;
		code[1] = bciMnem;
		short scratch = LE_WORD(op.val.varID);
		MEMCPY2(&code[2], &scratch);

		// get the mnemonic from wide_load_store_mnemonic since it's wide
		for (int i = 0; ; i++)
        {
			if (wide_load_store_mnemonic[i].opcode == bciMnem)
            {
				szMnem = wide_load_store_mnemonic[i].mnemonic;
                semtag = wide_load_store_mnemonic[i].semtag;
				break;
			}
		}
        CInstruction *pBCIInst = new CInstruction(szMnem, semtag, code, 4, 
            pDescr->GetStack());
		return pBCIInst;
	}
	else
    {
		// Use standard form
		code[0] = bciMnem;
		code[1] = static_cast<BYTE>(op.val.varID);
		codeSize = 2;
        semtag = bciMnem == JVMI_ret ? SEM_RETSR : SEM_GEN;
	}
	CInstruction *pBCIInst = new CInstruction(pDescr->GetMnem(), semtag, code, codeSize,
        pDescr->GetStack());
    return pBCIInst;
}

//////////////////////////////////////////////////////////////////////////
// class CIincOperandMapper implementation

CMIEInstruction* CIincOperandMapper::BCI2MIE(CInstruction *pBCIInst)
{
    //TODO: check "wide" version. may need to adapt this code to handle the "wide" version as well
    // iinc operands:
    // index: an index into the local variable table
    // const: a signed-byte constant by which to increment the variable specified by index
    CMIEInstruction* pMIEInst = CMIEInstructionFactory::CreateInstruction(
        pBCIInst->GetOpCode());
    InitMIEInstruction(pMIEInst, pBCIInst);
    SOperand op1;
    op1.type = OT_VAR_ID;
    op1.val.varID = pBCIInst->GetCode()[1];
    SOperand op2;
    op2.type = OT_INT8;
    op2.val.s8Op = pBCIInst->GetCode()[2];
    pMIEInst->InternalSetOperand(&op1, &op2);
    return pMIEInst;
}

CInstruction* CIincOperandMapper::MIE2BCI(CMIEInstruction *pMIEInst)
{
    //TEST: both regular and "wide" versions
    SOperand op1;
    SOperand op2;
    TResult res = pMIEInst->GetOperand(&op1, &op2);
    if (res != MRTE_RESULT_OK)
    {
        return NULL;
    }
    if (op1.type != OT_VAR_ID)
    {
        // 1st operand is not a local variable index
        return NULL;
    }
    if (op2.type != OT_INT8 && op2.type != OT_INT16)
    {
        // 2nd operand is not a numeric constant
        return NULL;
    }

    EMnemonic mnem = pMIEInst->GetMnemonic();
    JVMI_t bciMnem = (JVMI_t)mnem;
	BYTE code[6];
	int codeSize;
	CInsDescr* pDescr = BCIInsSet[bciMnem];
    short constVal = OT_INT8 == op2.type ? op2.val.s8Op : op2.val.s16Op;
	if (op1.val.varID > 255 || constVal > 127 || constVal < -128)
    {
		// Use wide form
		code[0] = JVMI_wide;
		code[1] = JVMI_iinc;
		short scratch = LE_WORD(op1.val.varID);
		MEMCPY2(&code[2], &scratch);
		scratch = LE_WORD(constVal);
		MEMCPY2(&code[4], &scratch);
        CInstruction *pBCIInst = new CInstruction("wide iinc", SEM_GEN, code, 6, 
            pDescr->GetStack());
		return pBCIInst;
	}
	else
    {
		// Use standard form
		code[0] = bciMnem;
		code[1] = static_cast<BYTE>(op1.val.varID);
        code[2] = static_cast<BYTE>(constVal);
		codeSize = 3;
	}
	CInstruction *pBCIInst = new CInstruction(pDescr->GetMnem(), SEM_GEN, code, codeSize,
        pDescr->GetStack());
    return pBCIInst;
}

//////////////////////////////////////////////////////////////////////////
// class CWordOffsetOperandMapper implementation

CMIEInstruction* CWordOffsetOperandMapper::BCI2MIE(CInstruction *pBCIInst)
{
    // branch instructions in which the first (and only) operand is a 16-bit offset
    // into the code array
    CMIEInstruction *pMIEInst = CMIEInstructionFactory::CreateInstruction(
        pBCIInst->GetOpCode());
    InitMIEInstruction(pMIEInst, pBCIInst);
    S16 s16Offset = (pBCIInst->GetCode()[1] << 8) | pBCIInst->GetCode()[2];
    // set the absolute offset of the branch target from the beginning of the method
    pMIEInst->SetBranchAbsoluteOffset(pBCIInst->GetIP() + s16Offset);
    SOperand op;
    op.type = OT_TARGET;
    // the target instruction is not known at this stage, so we set it to NULL.
    // this will be fixed after the flow-graph is constructed
    op.val.pTarget = NULL;
    pMIEInst->InternalSetOperand(&op);
    return pMIEInst;
}

CInstruction* CWordOffsetOperandMapper::MIE2BCI(CMIEInstruction *pMIEInst)
{
    SOperand op;
    TResult res = pMIEInst->GetOperand(&op);
    if (res != MRTE_RESULT_OK)
    {
        return NULL;
    }
    if (op.type != OT_TARGET)
    {
        // 1st operand is not a jump target
        return NULL;
    }
    CMIEInstruction *pTargetInst = (CMIEInstruction*)op.val.pTarget;
    if (NULL == pTargetInst)
    {
        // branch target is invalid
        return NULL;
    }
    if (MNM_PSEUDO_TARGET != pTargetInst->GetMnemonic())
    {
        // branch target is invalid
        return NULL;
    }

    EMnemonic mnem = pMIEInst->GetMnemonic();
    JVMI_t bciMnem = (JVMI_t)mnem;
	BYTE code[3] = {bciMnem, 0, 0};
    unsigned int uiTarget = pTargetInst->GetActualOffset();
	short scratch = LE_WORD(uiTarget);
	MEMCPY2(&code[1], &scratch);
	CInsDescr* pDescr = BCIInsSet[bciMnem];
    CInstruction *pBCIInst = new CInstruction_BranchJ(
		pDescr->GetMnem(), pDescr->GetSemTag(), 
		code, sizeof(code), pDescr->GetStack(),
		uiTarget);
    return pBCIInst;
}

//////////////////////////////////////////////////////////////////////////
// class CTableswitchOperandMapper implementation

CMIEInstruction* CTableswitchOperandMapper::BCI2MIE(CInstruction *pBCIInst)
{
    //TEST

    // tableswitch operands:
    // [padding - ignored]
    // default: a 32 bit signed offset - switch default target
    // low: a 32 bit unsigned value - target table lower index
    // high: a 32 bit unsigned value - target table upper index
    // target_offsets[]: an array of 32 bit signed offset - jump targets
    if (pBCIInst->GetOpCode() != JVMI_tableswitch)
    {
        // unexpected error. The BCI instruction is not tableswitch
        return NULL;
    }
    CMIEInstruction *pSwitchInst = CMIEInstructionFactory::CreateInstruction(
        pBCIInst->GetOpCode());
    InitMIEInstruction(pSwitchInst, pBCIInst);
    CInstruction_TableSwitchJ *pBCISwitch = (CInstruction_TableSwitchJ*)pBCIInst;
    unsigned int uiTargetsCount = pBCISwitch->GetTargetsCount();
    // number of targets must be at least 1, because the default target is stored
    // as the first target in the table. If the number is less than 1, this is an error
    if (uiTargetsCount < 1)
    {
        return NULL;
    }
    uiTargetsCount--;   // adjust the targets count so it will not include the default

    // update the internal state of the MIE instruction
    CInstruction_Switch::iterator iterTargets = pBCISwitch->begin();
    CInstruction_Switch::iterator iterTargetsEnd = pBCISwitch->end();
    // first target is the default target
    pSwitchInst->SetBranchAbsoluteOffset(*iterTargets);
    ++iterTargets;
    // other targets goes into the offsets array
    pSwitchInst->SetSwitchOffsetsCount(uiTargetsCount);
    unsigned int *vuiTargetOffsets = new unsigned int[uiTargetsCount];
    int ind = 0;
    for (; iterTargets != iterTargetsEnd; ++iterTargets)
    {
        vuiTargetOffsets[ind] = *iterTargets;
        ind++;
    }
    pSwitchInst->SetSwitchAbsoluteOffsets(vuiTargetOffsets);
    pSwitchInst->SetLow(pBCISwitch->GetLow());
    pSwitchInst->SetHigh(pBCISwitch->GetHigh());

    // create the instruction's operand
    SOperand op;
    op.type = OT_TARGETS_ARRAY;
    // initialize the targets array but do not fill it, because the target instructions
    // are not known at this stage. This will be fixed after the flow-graph is constructed
    op.val.targetsArray.uiEntries = uiTargetsCount;
    op.val.targetsArray.uiUsedEntries = uiTargetsCount;
    op.val.targetsArray.pTargetsArray = new SCaseTarget[uiTargetsCount];
    for (unsigned int i = 0; i < uiTargetsCount; ++i)
    {
        SCaseTarget caseTarget;
        caseTarget.caseValue = 0;  // this value is ignored in the tableswitch instruction
        caseTarget.pTarget = NULL; // will be updated after the flow-graph is constructed
        op.val.targetsArray.pTargetsArray[i] = caseTarget;
    }
    op.val.targetsArray.pDefaultTarget = NULL; // will be updated after the flow-graph is constructed
    pSwitchInst->InternalSetOperand(&op);

    delete [] op.val.targetsArray.pTargetsArray;
    return pSwitchInst;
}

CInstruction* CTableswitchOperandMapper::MIE2BCI(CMIEInstruction *pMIEInst)
{
    //TEST
    SOperand op;
    op.type = OT_TARGETS_ARRAY;
    unsigned int uiSwitchOffsetsCount = pMIEInst->GetSwitchOffsetsCount();
    op.val.targetsArray.pTargetsArray = new SCaseTarget[uiSwitchOffsetsCount];
    op.val.targetsArray.uiEntries = uiSwitchOffsetsCount;
    op.val.targetsArray.uiUsedEntries = uiSwitchOffsetsCount;
    TResult res = pMIEInst->GetOperand(&op);
    if (res != MRTE_RESULT_OK)
    {
        return NULL;
    }
    if (op.type != OT_TARGETS_ARRAY)
    {
        // 1st operand is not a switch-targets array
        return NULL;
    }
    // create the switch target table
    vector<BrTarget_t> vecTargets;
    CMIEInstruction *pTargetInst;
    pTargetInst = (CMIEInstruction*)op.val.targetsArray.pDefaultTarget;
    // add default target
    vecTargets.push_back(pTargetInst->GetActualOffset());
    // add all other targets
    unsigned int i;
    for (i = 0; i < op.val.targetsArray.uiUsedEntries; ++i)
    {
        pTargetInst = (CMIEInstruction*)op.val.targetsArray.pTargetsArray[i].pTarget;
        vecTargets.push_back(pTargetInst->GetActualOffset());
    }
    CInstruction *pBCIInst = CInsSetJ::Create_tableswitch(
        pMIEInst->GetLow(), pMIEInst->GetHigh(), vecTargets);

    delete [] op.val.targetsArray.pTargetsArray;
    return pBCIInst;
}

//////////////////////////////////////////////////////////////////////////
// class CLookupswitchOperandMapper implementation

CMIEInstruction* CLookupswitchOperandMapper::BCI2MIE(CInstruction *pBCIInst)
{
    //TEST

    //TODO: refactor. the code is very similar to tableswitch mapper, and can reuse most of it
    
    // lookupswitch operands:
    // [padding - ignored]
    // default: a 32 bit signed offset - switch default target
    // npairs: a 32 bit signed value - number of <key,offset> pairs in the switch table
    // target_offsets[]: an array of <key,offset> pairs
    if (pBCIInst->GetOpCode() != JVMI_lookupswitch)
    {
        // unexpected error. The BCI instruction is not lookupswitch
        return NULL;
    }
    CMIEInstruction *pSwitchInst = CMIEInstructionFactory::CreateInstruction(
        pBCIInst->GetOpCode());
    InitMIEInstruction(pSwitchInst, pBCIInst);
    CInstruction_LookupSwitchJ *pBCISwitch = (CInstruction_LookupSwitchJ*)pBCIInst;
    unsigned int uiTargetsCount = pBCISwitch->GetTargetsCount();
    if (uiTargetsCount != pBCISwitch->GetLookupTable().size())
    {
        return NULL;
    }
    // number of targets must be at least 1, because the default target is stored
    // as the first target in the table. If the number is less than 1, this is an error
    if (uiTargetsCount < 1)
    {
        return NULL;
    }
    uiTargetsCount--;   // adjust the targets count so it will not include the default

    // update the internal state of the MIE instruction
    CInstruction_Switch::iterator iterTargets = pBCISwitch->begin();
    CInstruction_Switch::iterator iterTargetsEnd = pBCISwitch->end();
    // first target is the default target
    pSwitchInst->SetBranchAbsoluteOffset(*iterTargets);
    ++iterTargets;
    // other targets goes into the offsets array
    pSwitchInst->SetSwitchOffsetsCount(uiTargetsCount);
    unsigned int *vuiTargetOffsets = new unsigned int[uiTargetsCount];
    int ind = 0;
    for (; iterTargets != iterTargetsEnd; ++iterTargets)
    {
        vuiTargetOffsets[ind] = *iterTargets;
        ind++;
    }
    pSwitchInst->SetSwitchAbsoluteOffsets(vuiTargetOffsets);

    // create the instruction's operand
    SOperand op;
    op.type = OT_TARGETS_ARRAY;
    // initialize the targets array but do not fill it, because the target instructions
    // are not known at this stage. This will be fixed after the flow-graph is constructed
    op.val.targetsArray.uiEntries = uiTargetsCount;
    op.val.targetsArray.uiUsedEntries = uiTargetsCount;
    op.val.targetsArray.pTargetsArray = new SCaseTarget[uiTargetsCount];
    CInstruction_LookupSwitchJ::LookupTable_t lookupTable = pBCISwitch->GetLookupTable();
    CInstruction_LookupSwitchJ::LookupTable_t::iterator iterLookup = lookupTable.begin();
    ++iterLookup; // skip the first entry. It is unused.
    for (unsigned int i = 0; i < uiTargetsCount; ++i)
    {
        SCaseTarget caseTarget;
        caseTarget.caseValue = *iterLookup;
        ++iterLookup;
        caseTarget.pTarget = NULL; // will be updated after the flow-graph is constructed
        op.val.targetsArray.pTargetsArray[i] = caseTarget;
    }
    op.val.targetsArray.pDefaultTarget = NULL; // will be updated after the flow-graph is constructed
    pSwitchInst->InternalSetOperand(&op);

    delete [] op.val.targetsArray.pTargetsArray;
    return pSwitchInst;
}

CInstruction* CLookupswitchOperandMapper::MIE2BCI(CMIEInstruction *pMIEInst)
{
    SOperand op;
    op.type = OT_TARGETS_ARRAY;
    unsigned int uiSwitchOffsetsCount = pMIEInst->GetSwitchOffsetsCount();
    op.val.targetsArray.pTargetsArray = new SCaseTarget[uiSwitchOffsetsCount];
    op.val.targetsArray.uiEntries = uiSwitchOffsetsCount;
    op.val.targetsArray.uiUsedEntries = uiSwitchOffsetsCount;
    TResult res = pMIEInst->GetOperand(&op);
    if (res != MRTE_RESULT_OK)
    {
        return NULL;
    }
    if (op.type != OT_TARGETS_ARRAY)
    {
        // 1st operand is not a switch-targets array
        return NULL;
    }
    // create the switch target table and lookup table
    vector<BrTarget_t> vecTargets;
    vector<unsigned> vecLookup;
    CMIEInstruction *pTargetInst;
    pTargetInst = (CMIEInstruction*)op.val.targetsArray.pDefaultTarget;
    // add default target
    vecTargets.push_back(pTargetInst->GetActualOffset());
    vecLookup.push_back(0);             // push a dummy value. It is not used anyway
    // add all other targets
    unsigned int i;
    for (i = 0; i < op.val.targetsArray.uiUsedEntries; ++i)
    {
        pTargetInst = (CMIEInstruction*)op.val.targetsArray.pTargetsArray[i].pTarget;
        vecTargets.push_back(pTargetInst->GetActualOffset());
        vecLookup.push_back(op.val.targetsArray.pTargetsArray[i].caseValue);
    }
    CInstruction *pBCIInst = CInsSetJ::Create_lookupswitch(vecTargets, vecLookup);

    delete [] op.val.targetsArray.pTargetsArray;
    return pBCIInst;
}

//////////////////////////////////////////////////////////////////////////
// class CNewarrayOperandMapper implementation

CMIEInstruction* CNewarrayOperandMapper::BCI2MIE(CInstruction *pBCIInst)
{
    CMIEInstruction* pMIEInst = CMIEInstructionFactory::CreateInstruction(
        pBCIInst->GetOpCode());
    InitMIEInstruction(pMIEInst, pBCIInst);
    SOperand op;
    op.type = OT_UINT8;
    op.val.ui8Op = pBCIInst->GetCode()[1];
    pMIEInst->InternalSetOperand(&op);
    return pMIEInst;
}

CInstruction* CNewarrayOperandMapper::MIE2BCI(CMIEInstruction *pMIEInst)
{
    //TEST
    SOperand op;
    TResult res = pMIEInst->GetOperand(&op);
    if (res != MRTE_RESULT_OK)
    {
        return NULL;
    }
    if (op.type != OT_UINT8)
    {
        // 1st operand is not an unsigned Byte
        return NULL;
    }
    if (op.val.ui8Op < 4 || op.val.ui8Op > 11)
    {
        // 1st operand is not a valid type (according to VM Spec, 2nd Edition)
        return NULL;
    }
    CInstruction *pBCIInst = CInsSetJ::Create_newarray(op.val.ui8Op);
    return pBCIInst;
}

//////////////////////////////////////////////////////////////////////////
// class CWideOperandMapper implementation

CMIEInstruction* CWideOperandMapper::BCI2MIE(CInstruction *pBCIInst)
{
    //TEST: test all "wide" flavors

    //
    // JIE converts the Java "wide" instruction to the actual instruction it "widens".
    // The fact that the created instruction is a "wide" version is reflected in the 
    // instruction operands
    //
    BYTE opcode = pBCIInst->GetCode()[1];
    CMIEInstruction* pMIEInst = CMIEInstructionFactory::CreateInstruction(opcode);
    CHECK_RETURN_NULL(pMIEInst);
    InitMIEInstruction(pMIEInst, pBCIInst);

    // 1st operand is a 16-bit unsigned index to the local variable table
    SOperand op1;
    op1.type = OT_VAR_ID;
    op1.val.varID = (pBCIInst->GetCode()[2] << 8) | pBCIInst->GetCode()[3];

    // special case for wide iinc
    SOperand op2;
    if (MNM_IINC == opcode)
    {
        // 2nd operand is a 16-bit signed constant value
        op2.type = OT_INT16;
        op2.val.s16Op = (pBCIInst->GetCode()[4] << 8) | pBCIInst->GetCode()[5];
    }
    else
    {
        // no 2nd operand
        op2.type = OT_NO_OPERAND;
    }
    pMIEInst->InternalSetOperand(&op1, &op2);
    return pMIEInst;
}

CInstruction* CWideOperandMapper::MIE2BCI(CMIEInstruction *pMIEInst)
{
    // unexpected error. JIE does support the creation of the "wide" Java instruction.
    // this instruction is implicitly created for other Java instructions based on the
    // size of their operands.
    assert(false);
    return NULL;
}

//////////////////////////////////////////////////////////////////////////
// class CMultianewarrayOperandMapper implementation

CMIEInstruction* CMultianewarrayOperandMapper::BCI2MIE(CInstruction *pBCIInst)
{
    // multianewarray operands:
    // index: a 16-bit index into the constant pool
    // dimensions: unsigned byte - array dimensions
    CMIEInstruction* pMIEInst = CMIEInstructionFactory::CreateInstruction(
        pBCIInst->GetOpCode());
    InitMIEInstruction(pMIEInst, pBCIInst);
    SOperand op1;
    op1.type = OT_JAVA_CP_INDEX;
    op1.val.cpIndex = (pBCIInst->GetCode()[1] << 8) | pBCIInst->GetCode()[2];
    SOperand op2;
    op2.type = OT_UINT8;
    op2.val.ui8Op = pBCIInst->GetCode()[3];
    pMIEInst->InternalSetOperand(&op1, &op2);
    return pMIEInst;
}

CInstruction* CMultianewarrayOperandMapper::MIE2BCI(CMIEInstruction *pMIEInst)
{
    //TEST
    SOperand op1;
    SOperand op2;
    TResult res = pMIEInst->GetOperand(&op1, &op2);
    if (res != MRTE_RESULT_OK)
    {
        return NULL;
    }
    if (op1.type != OT_JAVA_CP_INDEX)
    {
        // 1st operand is not a constant-pool index
        return NULL;
    }
    if (op2.type != OT_UINT8)
    {
        // 2nd operand is not an unsigned Byte
        return NULL;
    }
    CInstruction *pBCIInst = CInsSetJ::Create_multianewarray(op1.val.cpIndex, 
        op2.val.ui8Op);
    return pBCIInst;
}

//////////////////////////////////////////////////////////////////////////
// class CLongOffsetOperandMapper implementation

CMIEInstruction* CLongOffsetOperandMapper::BCI2MIE(CInstruction *pBCIInst)
{
    // wide branch instructions in which the first (and only) operand is a 32-bit offset
    // into the code array
    CMIEInstruction *pMIEBranchInst = CMIEInstructionFactory::CreateInstruction(
        pBCIInst->GetOpCode());
    InitMIEInstruction(pMIEBranchInst, pBCIInst);
    S32 s32Offset = (pBCIInst->GetCode()[1] << 24) | (pBCIInst->GetCode()[2] << 16) |
        (pBCIInst->GetCode()[3] << 8) | pBCIInst->GetCode()[4];
    // set the absolute offset of the branch target from the beginning of the method
    pMIEBranchInst->SetBranchAbsoluteOffset(pBCIInst->GetIP() + s32Offset);
    SOperand pOp;
    pOp.type = OT_TARGET;
    // the target instruction is not known at this stage, so we set it to NULL.
    // this will be fixed after the flow-graph is constructed
    pOp.val.pTarget = NULL;
    pMIEBranchInst->InternalSetOperand(&pOp);
    return pMIEBranchInst;
}

CInstruction* CLongOffsetOperandMapper::MIE2BCI(CMIEInstruction *pMIEInst)
{
    //TEST
    SOperand op;
    TResult res = pMIEInst->GetOperand(&op);
    if (res != MRTE_RESULT_OK)
    {
        return NULL;
    }
    if (op.type != OT_TARGET)
    {
        // 1st operand is not a jump target
        return NULL;
    }
    CMIEInstruction *pTargetInst = (CMIEInstruction*)op.val.pTarget;
    if (NULL == pTargetInst)
    {
        // branch target is invalid
        return NULL;
    }
    if (MNM_PSEUDO_TARGET != pTargetInst->GetMnemonic())
    {
        // branch target is invalid
        return NULL;
    }

    EMnemonic mnem = pMIEInst->GetMnemonic();
    JVMI_t bciMnem = (JVMI_t)mnem;
	BYTE code[5] = {bciMnem, 0, 0, 0, 0};
    unsigned int uiTarget = pTargetInst->GetActualOffset();
	S32 scratch = LE_DWORD(uiTarget);
	MEMCPY4(&code[1], &scratch);
	CInsDescr* pDescr = BCIInsSet[bciMnem];
    CInstruction *pBCIInst = new CInstruction_Branch_wJ(
		pDescr->GetMnem(), pDescr->GetSemTag(), code, sizeof(code), pDescr->GetStack(),
        uiTarget);
    return pBCIInst;
}

//////////////////////////////////////////////////////////////////////////
// class CInvokeinterfaceOperandMapper implementation

CMIEInstruction* CInvokeinterfaceOperandMapper::BCI2MIE(CInstruction *pBCIInst)
{
    // invokeinterface operands:
    // index: a 16-bit index into the constant pool
    // count: an unsigned byte that must not be zero
    // 0 : an unsigned byte that must be 0
    CMIEInstruction* pMIEInst = CMIEInstructionFactory::CreateInstruction(
        pBCIInst->GetOpCode());
    InitMIEInstruction(pMIEInst, pBCIInst);
    SOperand op1;
    op1.type = OT_JAVA_CP_INDEX;
    op1.val.cpIndex = (pBCIInst->GetCode()[1] << 8) | pBCIInst->GetCode()[2];
    SOperand op2;
    op2.type = OT_UINT8;
    op2.val.ui8Op = pBCIInst->GetCode()[3];
    SOperand op3;
    op3.type = OT_UINT8;
    op3.val.ui8Op = 0;
    pMIEInst->InternalSetOperand(&op1, &op2, &op3);
    return pMIEInst;
}

CInstruction* CInvokeinterfaceOperandMapper::MIE2BCI(CMIEInstruction *pMIEInst)
{
    //TEST
    SOperand op1;
    SOperand op2;
    SOperand op3;
    TResult res = pMIEInst->GetOperand(&op1, &op2, &op3);
    if (res != MRTE_RESULT_OK)
    {
        return NULL;
    }
    if (op1.type != OT_JAVA_CP_INDEX)
    {
        // 1st operand is not a constant-pool index
        return NULL;
    }
    if (op2.type != OT_UINT8)
    {
        // 2nd operand is not an unsigned byte
        return NULL;
    }
    if (op3.type != OT_UINT8)
    {
        // 3rd operand is not an unsigned byte
        return NULL;
    }
    CInstruction *pBCIInst = CInsSetJ::Create_invokeinterface(op1.val.cpIndex, 
        op2.val.ui8Op);
    return pBCIInst;
}

//////////////////////////////////////////////////////////////////////////
// class CLoadInstOffsetEncoder implementation

void CLoadInstOffsetEncoder::Encode(TBciInstructionList *pBciInstList, 
                                    CMIEInstruction *pMieSpecialIsnt,
                                    CJClassBuilder *pBciClassBuilder)
{
    SOperand op;
    TResult res = pMieSpecialIsnt->GetOperand(&op);
    if (MRTE_FAILED(res))
    {
        return;
    }
    if (op.type != OT_TARGET)
    {
        return;
    }
    CMIEInstruction *pTargetInst = (CMIEInstruction*)op.val.pTarget;

    // Create a CP entry to store the instruction offset (int constant)
    CCPLongInfo *pcpInt = pBciClassBuilder->CreateLongConstant(
        pTargetInst->GetActualOffset());

    // Create an ldc_w instruction that loads the instruction offset from the CP
    JVMI_t bciMnem = JVMI_ldc2_w;
	BYTE code[3] = {bciMnem};
	short scratch = LE_WORD(pcpInt->GetCpIndex());
	MEMCPY2(&code[1], &scratch);
	CInsDescr* pDescr = BCIInsSet[bciMnem];
	CInstruction *pBciInst = new CInstruction(pDescr->GetMnem(), pDescr->GetSemTag(), code, 
        SIZE_OF(code), pDescr->GetStack());

    TMRTEHandle hLast = pBciInstList->GetLast();
    pBciInstList->InsertAfter(hLast, pBciInst);
}
