/*****************************************************************************
 * Copyright (c) 1997-2007, Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/
//

#ifndef MRTE_BCITOMIEINSTRUCTIONMAPPER
#define MRTE_BCITOMIEINSTRUCTIONMAPPER

#pragma warning (disable : 4786)

#include "MieAPI.h"
#include "MIEInstruction.h"
#include "InsSet.h"
#include "JClassBuilder.h"

#include "MList.h"

namespace Martini { namespace MIE {

typedef RTUtil::MList<CInstruction*> TBciInstructionList;

//
// Abstract mapper. Serves as the base class for all mappers.
//
// Mappers are used to map BCI Instruction operands to MIE instruction operands.
//
class COperandMapper
{
public:
    COperandMapper(EMnemonic opcode)
    {
        m_mnem = opcode;
    }
    virtual ~COperandMapper() { }

    // convert a BCI instruction object to an MIE instruction object
  	virtual CMIEInstruction* BCI2MIE(CInstruction *pBCIInst) = 0;

    // convert an MIE instruction object to a BCI instruction object
	virtual CInstruction* MIE2BCI(CMIEInstruction *pMIEInst) = 0;

    EMnemonic GetMnemonic() 
    {
        return m_mnem;
    }

protected:
    void InitMIEInstruction(CMIEInstruction *pInstToInit, 
                            const CInstruction *pBCIInst);

private:
    EMnemonic m_mnem;
};

//
// A mapper for instructions that have no operands
//
class CNoOperandsMapper : public COperandMapper
{
public:
    CNoOperandsMapper(EMnemonic opcode) : COperandMapper(opcode) {}
  	virtual CMIEInstruction* BCI2MIE(CInstruction *pBCIInst);
	virtual CInstruction* MIE2BCI(CMIEInstruction *pMIEInst);
};

//
// A mapper for instructions that have one Byte operand
//
class CByteOperandMapper : public COperandMapper
{
public:
    CByteOperandMapper(EMnemonic opcode) : COperandMapper(opcode) {}
  	virtual CMIEInstruction* BCI2MIE(CInstruction *pBCIInst);
	virtual CInstruction* MIE2BCI(CMIEInstruction *pMIEInst);
};

//
// A mapper for instructions that have one Short operand
//
class CShortOperandMapper : public COperandMapper
{
public:
    CShortOperandMapper(EMnemonic opcode) : COperandMapper(opcode) {}
  	virtual CMIEInstruction* BCI2MIE(CInstruction *pBCIInst);
	virtual CInstruction* MIE2BCI(CMIEInstruction *pMIEInst);
};

//
// A mapper for the ldc instruction
//
class CLdcOperandMapper : public COperandMapper
{
public:
    CLdcOperandMapper(EMnemonic opcode) : COperandMapper(opcode) {}
  	virtual CMIEInstruction* BCI2MIE(CInstruction *pBCIInst);
	virtual CInstruction* MIE2BCI(CMIEInstruction *pMIEInst);
};

//
// A mapper for instructions that have one Constant Pool index operand
//
class CCPIndexOperandMapper : public COperandMapper
{
public:
    CCPIndexOperandMapper(EMnemonic opcode) : COperandMapper(opcode) {}
  	virtual CMIEInstruction* BCI2MIE(CInstruction *pBCIInst);
	virtual CInstruction* MIE2BCI(CMIEInstruction *pMIEInst);
};

//
// A mapper for instructions that have one Local Variable index operand
//
class CLocalVarOperandMapper : public COperandMapper
{
public:
    CLocalVarOperandMapper(EMnemonic opcode) : COperandMapper(opcode) {}
  	virtual CMIEInstruction* BCI2MIE(CInstruction *pBCIInst);
	virtual CInstruction* MIE2BCI(CMIEInstruction *pMIEInst);
};

//
// A mapper for the iinc instruction
//
class CIincOperandMapper : public COperandMapper
{
public:
    CIincOperandMapper(EMnemonic opcode) : COperandMapper(opcode) {}
  	virtual CMIEInstruction* BCI2MIE(CInstruction *pBCIInst);
	virtual CInstruction* MIE2BCI(CMIEInstruction *pMIEInst);
};

//
// A mapper for instructions that have one Word "target offset" operand
//
class CWordOffsetOperandMapper : public COperandMapper
{
public:
    CWordOffsetOperandMapper(EMnemonic opcode) : COperandMapper(opcode) {}
  	virtual CMIEInstruction* BCI2MIE(CInstruction *pBCIInst);
	virtual CInstruction* MIE2BCI(CMIEInstruction *pMIEInst);
};

//
// A mapper for the tableswitch instruction
//
class CTableswitchOperandMapper : public COperandMapper
{
public:
    CTableswitchOperandMapper(EMnemonic opcode) : COperandMapper(opcode) {}
  	virtual CMIEInstruction* BCI2MIE(CInstruction *pBCIInst);
	virtual CInstruction* MIE2BCI(CMIEInstruction *pMIEInst);
};

//
// A mapper for the lookupswitch instruction
//
class CLookupswitchOperandMapper : public COperandMapper
{
public:
    CLookupswitchOperandMapper(EMnemonic opcode) : COperandMapper(opcode) {}
  	virtual CMIEInstruction* BCI2MIE(CInstruction *pBCIInst);
	virtual CInstruction* MIE2BCI(CMIEInstruction *pMIEInst);
};

//
// A mapper for the newarray instruction
//
class CNewarrayOperandMapper : public COperandMapper
{
public:
    CNewarrayOperandMapper(EMnemonic opcode) : COperandMapper(opcode) {}
  	virtual CMIEInstruction* BCI2MIE(CInstruction *pBCIInst);
	virtual CInstruction* MIE2BCI(CMIEInstruction *pMIEInst);
};

//
// A mapper for the "wide" instruction
//
class CWideOperandMapper : public COperandMapper
{
public:
    CWideOperandMapper(EMnemonic opcode) : COperandMapper(opcode) {}
  	virtual CMIEInstruction* BCI2MIE(CInstruction *pBCIInst);
	virtual CInstruction* MIE2BCI(CMIEInstruction *pMIEInst);
};

//
// A mapper for the multianewarray instruction
//
class CMultianewarrayOperandMapper : public COperandMapper
{
public:
    CMultianewarrayOperandMapper(EMnemonic opcode) : COperandMapper(opcode) {}
  	virtual CMIEInstruction* BCI2MIE(CInstruction *pBCIInst);
	virtual CInstruction* MIE2BCI(CMIEInstruction *pMIEInst);
};

//
// A mapper for instructions that have one Long "target offset" operand
//
class CLongOffsetOperandMapper : public COperandMapper
{
public:
    CLongOffsetOperandMapper(EMnemonic opcode) : COperandMapper(opcode) {}
  	virtual CMIEInstruction* BCI2MIE(CInstruction *pBCIInst);
	virtual CInstruction* MIE2BCI(CMIEInstruction *pMIEInst);
};

//
// A mapper for the invokeinterface instruction
//
class CInvokeinterfaceOperandMapper : public COperandMapper
{
public:
    CInvokeinterfaceOperandMapper(EMnemonic opcode) : COperandMapper(opcode) {}
  	virtual CMIEInstruction* BCI2MIE(CInstruction *pBCIInst);
	virtual CInstruction* MIE2BCI(CMIEInstruction *pMIEInst);
};

/**
 * @brief Abstract Special Instruction Encoder.
 *
 * Encoder classes are used to encode special MIE instructions as BCI instructions.
 */
class CSpecialInstEncoder
{
public:
    virtual ~CSpecialInstEncoder() {}
    
    /**
     * @brief Encodes a special MIE instructions
     *
     * @param[out]  pBciInstList    a list of BCI instructions implementing the special
     *                              MIE instruction
     * @param[in]   pMieSpecialInst The special MIE instruction to encode
     */
    virtual void Encode(TBciInstructionList *pBciInstList,
                        CMIEInstruction *pMieSpecialIsnt,
                        CJClassBuilder *pBciClassBuilder) = 0;
    
};

/**
 * @brief Encoder for the MIE_loadinstoffset special instruction
 */
class CLoadInstOffsetEncoder : public CSpecialInstEncoder
{
public:
    virtual void Encode(TBciInstructionList *pBciInstList,
                        CMIEInstruction *pMieSpecialIsnt,
                        CJClassBuilder *pBciClassBuilder);
};

//
// Maps BCI instructions to MIE instructions.
// Internally uses one of the *Mapper classes according to the opcode of the given
// instruction.
//
class CBCIToMIEInstructionMapper  
{
public:
	CBCIToMIEInstructionMapper();
	virtual ~CBCIToMIEInstructionMapper();

  	CMIEInstruction* BCI2MIE(CInstruction *pBCIInst);
	CInstruction* MIE2BCI(CMIEInstruction *pMIEInst);
    void EncodeSpecialInstruction(TBciInstructionList *pBciInstList,
                                  CMIEInstruction *pMieSpecialIsnt,
                                  CJClassBuilder *pBciClassBuilder);

private:
    COperandMapper** m_vMappers;        // Operand mappers array
    unsigned int m_uiMappersCount;      // Operand mappers array size
    CSpecialInstEncoder** m_vEncoders;  // Special Instruction Encoders array
    unsigned int m_uiEncodersSize;      // Special Instruction Encoders array size
};

}}

#endif // MRTE_BCITOMIEINSTRUCTIONMAPPER
