/*****************************************************************************
 * Copyright (c) 1997-2007, Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/
//

#include "FlowGraph.h"
#include "ValidityChecks.h"
#include "MIEInstructionFactory.h"

#include <stdlib.h>

using namespace Martini::MIE;
using namespace Martini::RTUtil;

//////////////////////////////////////////////////////////////////////////
// class CFlowGraph implementation
CFlowGraph::CFlowGraph()
{
}

CFlowGraph::~CFlowGraph()
{
    ReleaseAll();
}

//
// Walks over the flow graph and releases the memory allocated for the nodes.
//
void CFlowGraph::ReleaseAll()
{
    FreeListItems(m_lstIterators);
    FreeListItems(m_lstNodes);
    FreeListItems(m_lstClonedInstructions);
    m_lookupTable.Empty();
}

//
// Constructs a flow graph with 2 pseudo instructions: '.start' and '.end'
//
// Returns:
//      MRTE_RESULT_OK : success
//      MRTE_ERROR_OUT_OF_MEMORY : if any of the allocations fails
//
TResult CFlowGraph::Init(CMethodInstrumentor *methodInstrumentor)
{
    ReleaseAll();
    m_lookupTable.Init(true, NULL, 200);
    m_pInstrumentor = methodInstrumentor;
    // create .start inst
    CMIEInstruction* pStartInst = CMIEInstructionFactory::CreateInstruction(
        MNM_PSEUDO_START, m_pInstrumentor);
    CHECK_ALLOC(pStartInst);
    pStartInst->SetStatus(CMIEInstruction::IS_ORIGINAL);
    // create .end inst
    CMIEInstruction* pEndInst = CMIEInstructionFactory::CreateInstruction(
        MNM_PSEUDO_END, m_pInstrumentor);
    CHECK_ALLOC(pEndInst)
    pEndInst->SetStatus(CMIEInstruction::IS_ORIGINAL);
    // add .start to graph and set m_hCurInst to it
    CFlowGraphNode *pStartNode = new CFlowGraphNode(pStartInst);
    CHECK_ALLOC(pStartNode);
    m_hStartNode = m_lstNodes.InsertAfter(NULL, pStartNode);
    CHECK_ALLOC(m_hStartNode)
    pStartNode->m_handle = m_hStartNode;
    // add .end to graph
    CFlowGraphNode *pEndNode = new CFlowGraphNode(pEndInst);
    CHECK_ALLOC(pEndNode);
    m_hEndNode = m_lstNodes.InsertAfter(m_hStartNode, pEndNode);
    CHECK_ALLOC(m_hEndNode)
    pEndNode->m_handle = m_hEndNode;
    return MRTE_RESULT_OK;
}

// 
// Adds an instruction to end of the flow-graph (but before the .end pseudo-instruction),
// and stores a mapping between the instruction object and its offset.
// Use this method to add the original method's instructions to the flow-graph.
// 
// NOTE: do not use this method to add pseudo-instructions, or new instructions added
//       during instrumentation.
//
// Parameters:
//      - pInst [in] : the instruction to add
//      - uiOffset [in] : the instruction's offset from the beginning of the method
//
// Returns:
//      - MRTE_RESULT_OK : success
//
TResult CFlowGraph::AddDecodedInstruction(CMIEInstruction *pInst, 
                                          unsigned int uiOffset)
{
    CFlowGraphNode *pNode = new CFlowGraphNode(pInst);
    CHECK_ALLOC(pNode);
    pNode->m_handle = m_lstNodes.InsertBefore(m_hEndNode, pNode);
    CHECK_ALLOC(pNode->m_handle);
    UpdateFlowGraphNodePointers(pNode);

    // add the instruction to the offset->node lookup table
    m_lookupTable.Set(uiOffset, pNode);
    return MRTE_RESULT_OK;
}

TResult CFlowGraph::AddInstructionBefore(CFlowGraphNode *pNode,
                                         CMIEInstruction *pInst)
{
    CFlowGraphNode *pNewNode = new CFlowGraphNode(pInst);
    CHECK_ALLOC(pNewNode);
    pNewNode->m_handle = m_lstNodes.InsertBefore(pNode->m_handle, pNewNode);
    CHECK_ALLOC(pNewNode->m_handle);
    if (pInst->GetStatus() == CMIEInstruction::IS_ORIGINAL)
    {
        UpdateFlowGraphNodePointers(pNewNode);
    }
    return MRTE_RESULT_OK;
}

TResult CFlowGraph::AddInstructionAfter(CFlowGraphNode *pNode,
                                        CMIEInstruction *pInst)
{
    CFlowGraphNode *pNewNode = new CFlowGraphNode(pInst);
    CHECK_ALLOC(pNewNode);
    pNewNode->m_handle = m_lstNodes.InsertAfter(pNode->m_handle, pNewNode);
    CHECK_ALLOC(pNewNode->m_handle);
    if (pInst->GetStatus() == CMIEInstruction::IS_ORIGINAL)
    {
        UpdateFlowGraphNodePointers(pNewNode);
    }
    return MRTE_RESULT_OK;
}

TInstructionListIterator* CFlowGraph::GetInstructionListIterator(
    EInstructionIteratorType type)
{
    TInstructionListIterator *pinstIter;
    if (IET_ORIGINAL_INSTRUCTIONS == type)
    {
        pinstIter = new COrgInstructionsListIterator(*this);
    }
    else
    {
        TFlowGraphNodeIterator *pfgnIter = m_lstNodes.GetListIterator();
        pinstIter = new CAllInstructionsListIterator(*this, *pfgnIter);
    }
    // add the iterator to a list so it can be freed later
    TMRTEHandle h = m_lstIterators.GetLast();
    m_lstIterators.InsertBefore(h, pinstIter);
    return pinstIter;
}

//
// Looks up an instruction at the specified offset.
//
// Parameters:
//      - uiOffset: the instruction's offset
//
// Returns:
//      CMIEInstruction* - the instruction object located at the specified offset
//      NULL             - no instruction exists at the specified offset
//
CMIEInstruction* CFlowGraph::GetInstructionAtOffset(unsigned int uiOffset)
{
    CFlowGraphNode *pfgNode = m_lookupTable.Get(uiOffset);
    if (NULL == pfgNode)
    {
        return NULL;
    }
    CMIEInstruction *pInst = pfgNode->m_pInst;
    return pInst;
}

//
// Returns the instruction preceding the specified instruction.
//
// Parameters:
//      pInst - the instruction
//
// Returns:
//      CMIEInstruction* - the instruction that precedes pInst in the flow graph
//      NULL - error, or no preceding instruction in the graph.
//
CMIEInstruction* CFlowGraph::GetPrecedingInstruction(CMIEInstruction *pInst)
{
    CHECK_RETURN_NULL(pInst)
    CFlowGraphNode *pfgNode = pInst->GetFlowGraphNode();
    CHECK_RETURN_NULL(pfgNode);
    TMRTEHandle hNode = pfgNode->m_handle;
    TMRTEHandle hPredNode = m_lstNodes.GetPrev(hNode);
    CFlowGraphNode *pfgPredNode = m_lstNodes.GetData(hPredNode);
    CHECK_RETURN_NULL(pfgPredNode);
    return pfgPredNode->GetInstruction();
}

//
// Returns the first real (non-pseudo) instruction that precedes the specified 
// instruction.
//
// Parameters:
//      pInst - the instruction
//
// Returns:
//      CMIEInstruction* - the first real instruction that precedes pInst in the flow
//                         graph
//      NULL - error, or no preceding instruction in the graph.
//
CMIEInstruction* CFlowGraph::GetPrecedingRealInstruction(CMIEInstruction *pInst)
{
    CMIEInstruction *pPrevInst = GetPrecedingInstruction(pInst);
    CHECK_RETURN_NULL(pPrevInst);
    while (pPrevInst->IsPseudo())
    {
        pPrevInst = GetPrecedingInstruction(pPrevInst);
        CHECK_RETURN_NULL(pPrevInst);
    }
    return pPrevInst;
}

//
// Returns the instruction succeeding the specified instruction.
//
// Parameters:
//      pInst - the instruction
//
// Returns:
//      CMIEInstruction* - the instruction that succeeds pInst in the flow graph
//      NULL - error, or no succeeding instruction in the graph.
//
CMIEInstruction* CFlowGraph::GetSucceedingInstruction(CMIEInstruction *pInst)
{
    CHECK_RETURN_NULL(pInst)
    CFlowGraphNode *pfgNode = pInst->GetFlowGraphNode();
    CHECK_RETURN_NULL(pfgNode);
    TMRTEHandle hNode = pfgNode->m_handle;
    TMRTEHandle hPredNode = m_lstNodes.GetNext(hNode);
    CFlowGraphNode *pfgPredNode = m_lstNodes.GetData(hPredNode);
    CHECK_RETURN_NULL(pfgPredNode);
    return pfgPredNode->GetInstruction();
}

//
// Returns the first real (non-pseudo) instruction that succeeds the specified 
// instruction.
//
// Parameters:
//      pInst - the instruction
//
// Returns:
//      CMIEInstruction* - the first real instruction that succeeds pInst in the flow 
//                         graph
//      NULL - error, or no succeeding instruction in the graph.
//
CMIEInstruction* CFlowGraph::GetSucceedingRealInstruction(CMIEInstruction *pInst)
{
    CMIEInstruction *pNextInst = GetSucceedingInstruction(pInst);
    CHECK_RETURN_NULL(pNextInst);
    while (pNextInst->IsPseudo())
    {
        pNextInst = GetPrecedingInstruction(pNextInst);
        CHECK_RETURN_NULL(pNextInst);
    }
    return pNextInst;
}

//
// Implements a "copy-on-write" mechanism for instructions.
// This function is called before modifying an instruction object. If needed,
// a copy of the instruction is added to the flow-graph to allow clients to
// use "original instruction" iterator for inspecting the original flow-graph
// (i.e. without seeing modifications made by the client).
//
// Parameters:
//      pInstToModify   - the instruction being modified (and should be copied)
// 
// Returns:
//      true    - success
//      false   - an error occured (probably memory allocation error)
//
bool CFlowGraph::CopyOnModification(CMIEInstruction *pInstToModify)
{
    if (pInstToModify->GetStatus() != CMIEInstruction::IS_ORIGINAL)
    {
        // the instruction is not an original instruction so there is nothing to do
        return true;
    }
    CMIEInstruction *pClonedInst = pInstToModify->Clone();
    if (NULL == pClonedInst)
    {
        return false;
    }

    // add the cloned instruction to the flow-graph
    CFlowGraphNode *pfgnClonedInst = new CFlowGraphNode(pClonedInst);
    if (NULL == pfgnClonedInst)
    {
        delete pClonedInst;
        return false;
    }
    TMRTEHandle h = m_lstClonedInstructions.GetLast();
    m_lstClonedInstructions.InsertBefore(h, pfgnClonedInst);

    // update the links between the flow-graph nodes that are affected by this change
    CMIEInstruction *pPrevInst = GetPrecedingInstruction(pInstToModify);
    CMIEInstruction *pNextInst = GetSucceedingInstruction(pInstToModify);
    pPrevInst->GetFlowGraphNode()->m_pfgnNextOriginal = pfgnClonedInst;
    pNextInst->GetFlowGraphNode()->m_pfgnPrevOriginal = pfgnClonedInst;
    pfgnClonedInst->m_pfgnNextOriginal = pNextInst->GetFlowGraphNode();
    pfgnClonedInst->m_pfgnPrevOriginal = pPrevInst->GetFlowGraphNode();
    pInstToModify->GetFlowGraphNode()->m_pfgnNextOriginal = NULL;
    pInstToModify->GetFlowGraphNode()->m_pfgnPrevOriginal = NULL;
    return true;
}

void CFlowGraph::UpdateFlowGraphNodePointers(CFlowGraphNode *pNodeToUpdate)
{
    CMIEInstruction *pNextInst = GetSucceedingInstruction(pNodeToUpdate->GetInstruction());
    CMIEInstruction *pPrevInst = GetPrecedingInstruction(pNodeToUpdate->GetInstruction());
    pNodeToUpdate->m_pfgnNextOriginal = pNextInst->GetFlowGraphNode();
    pNextInst->GetFlowGraphNode()->m_pfgnPrevOriginal = pNodeToUpdate;
    pNodeToUpdate->m_pfgnPrevOriginal = pPrevInst->GetFlowGraphNode();
    pPrevInst->GetFlowGraphNode()->m_pfgnNextOriginal = pNodeToUpdate;
}

bool bDebugFileOpened = false;
TResult CFlowGraph::DebugPrint(bool bToResolveNames, char *fileName)
{
    static char* StatusName[] = 
    {
        "Original",
        "New",
        "Modified",
        "CodeGen"
    };

    FILE *fout;
    if (!fileName)
    {
        fout = stdout;
    }
    else
    {
        if (bDebugFileOpened)
        {
            fout = fopen (fileName, "a");
        }
        else
        {
            fout = fopen (fileName, "w");
            bDebugFileOpened = true;
        }
        if (fout == NULL)
        {
            fout = stdout;
        }
    }
    char *szFormat = "%-8s %-7s %-7s %-15s %s\n";
    // print header

    fprintf(fout, szFormat, "Status", "Org Off", "Act Off", "Mnemonic", "Operands");
    // print method instructions
    char szOpValues[1000];
    char szOffset[10];
    char szActualOffset[10];
    unsigned int uiOffset;
    TInstructionListIterator *iterInst = GetInstructionListIterator(IET_ALL_INSTRUCTIONS);
    while (iterInst->HasNext())
    {
        CMIEInstruction *pInst = (CMIEInstruction*)iterInst->GetNext();
        pInst->GetOperatorDebugString(bToResolveNames, szOpValues);
        if (pInst->GetSemantics() == CMIEInstruction::SEM_PSEUDO)
        {
            sprintf(szOffset, "(L%d)", pInst->GetLabel());
            szActualOffset[0] = 0;
        }
        else if (pInst->GetStatus() != CMIEInstruction::IS_ORIGINAL)
        {
            szOffset[0] = 0;
            uiOffset = pInst->GetActualOffset();
			sprintf(szActualOffset, "%d", uiOffset);
        }
        else 
        {
            uiOffset = pInst->InternalGetOriginalOffset();
			sprintf(szOffset, "%d", uiOffset);
            uiOffset = pInst->GetActualOffset();
			sprintf(szActualOffset, "%d", uiOffset);
        }
        fprintf(fout, szFormat, StatusName[pInst->GetStatus()], szOffset, 
            szActualOffset, pInst->GetName(), szOpValues);
    }
    iterInst->Free();
    if( fout != stdout)
    {
        fclose (fout);
    }
    return MRTE_RESULT_OK;
}

void CFlowGraph::RecordIterator (TInstructionListIterator * iterator)
{
    TMRTEHandle h = m_lstIterators.GetLast();
    m_lstIterators.InsertBefore(h, iterator);
}
