/*****************************************************************************
 * Copyright (c) 1997-2007, Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/
//
#ifndef MRTE_FLOWGRAPH
#define MRTE_FLOWGRAPH

#include "MList.h"
#include "MHash.h"

#include "FlowGraphNode.h"

namespace Martini { namespace MIE {

typedef RTUtil::IListIterator<CFlowGraphNode*> TFlowGraphNodeIterator;

class CMethodInstrumentor;          // forward declaration
class COrgInstructionsListIterator; // forward declaration

class CFlowGraph  
{
public:
	CFlowGraph();
	virtual ~CFlowGraph();

    TResult Init(CMethodInstrumentor *methodInstrumentor);
    TResult AddDecodedInstruction(CMIEInstruction *pInst, unsigned int uiOffset);
    TResult AddInstructionBefore(CFlowGraphNode *pNode,
                                 CMIEInstruction *pInst);
    TResult AddInstructionAfter(CFlowGraphNode *pNode,
                                CMIEInstruction *pInst);
    TInstructionListIterator* GetInstructionListIterator(EInstructionIteratorType type);
    CMIEInstruction* GetInstructionAtOffset(unsigned int uiOffset);
    CMIEInstruction* GetPrecedingInstruction(CMIEInstruction *pInst);
    CMIEInstruction* GetPrecedingRealInstruction(CMIEInstruction *pInst);
    CMIEInstruction* GetSucceedingInstruction(CMIEInstruction *pInst);
    CMIEInstruction* GetSucceedingRealInstruction(CMIEInstruction *pInst);
    bool CopyOnModification(CMIEInstruction *pInstToModify);
    TResult DebugPrint(bool bToResolveNames, char *fileName = NULL);
    void RecordIterator (TInstructionListIterator * iterator);

private:
    friend class CMethodInstrumentor;
    friend class COrgInstructionsListIterator;

    CMethodInstrumentor *m_pInstrumentor;

    RTUtil::MList<CFlowGraphNode*> m_lstNodes;                          // graph nodes
    RTUtil::MHashNoSync<unsigned int, CFlowGraphNode*> m_lookupTable;   // offset->node mapping
    TMRTEHandle m_hStartNode;   // graph node for the .start instruction
    TMRTEHandle m_hEndNode;     // graph node for the .end instruction
    RTUtil::MList<TInstructionListIterator*> m_lstIterators;            // list for tracking
                                                                        // allocated iterators
    RTUtil::MList<CFlowGraphNode*> m_lstClonedInstructions;             // list for tracking
                                                                        // cloned instructions

    void ReleaseAll();
    void UpdateFlowGraphNodePointers(CFlowGraphNode *pNodeToUpdate);
};

class CAllInstructionsListIterator : public TInstructionListIterator
{
public:
    TInstructionListIterator* Clone () 
    {

        CAllInstructionsListIterator *cloneOfIter 
            = new CAllInstructionsListIterator (m_flowGraph, m_fgnIter);
        cloneOfIter->m_fgnIter = m_fgnIter;
        // add the iterator to a list so it can be freed later
        m_flowGraph.RecordIterator (cloneOfIter);
        return (cloneOfIter);
       

    }
    bool HasNext()
    {
        return m_fgnIter.HasNext(); 
    }
    IInstruction* GetNext()
    {
        CFlowGraphNode *pNode = m_fgnIter.GetNext();
        if (NULL == pNode)
        {
            return NULL;
        }
        IInstruction *pInst = pNode->GetInstruction();
        return pInst;
    }
    void Free()
    {
        //TODO: implement free (not urgent. iterator will be freed eventually by its container)
    }
    IInstruction* GetFirst()
    {
        CFlowGraphNode *pNode = m_fgnIter.GetFirst();
        IInstruction *pInst = pNode->GetInstruction();
        return pInst;
    }
    IInstruction* GetLast()
    {
        CFlowGraphNode *pNode = m_fgnIter.GetLast();
        IInstruction *pInst = pNode->GetInstruction();
        return pInst;
    }
    bool HasPrev() 
    {
        return m_fgnIter.HasPrev(); 
    }
    IInstruction* GetPrev()
    {
        CFlowGraphNode *pNode = m_fgnIter.GetPrev();
        if (NULL == pNode)
        {
            return NULL;
        }
        IInstruction *pInst = pNode->GetInstruction();
        return pInst;
    }

private:
    friend class CFlowGraph;

    CFlowGraph &m_flowGraph;        // the flow-graph on which the iterator iterates
    TFlowGraphNodeIterator &m_fgnIter;
    CAllInstructionsListIterator(CFlowGraph &flowGraph, TFlowGraphNodeIterator &fgnIter) 
        : m_flowGraph (flowGraph), m_fgnIter(fgnIter) {}
}; // class CAllInstructionsListIterator

class COrgInstructionsListIterator : public TInstructionListIterator
{
public:
    TInstructionListIterator* Clone () 
        {

            COrgInstructionsListIterator *cloneOfIter 
                = new COrgInstructionsListIterator (m_flowGraph);
            cloneOfIter->m_pfgnNext = m_pfgnNext;
            // add the iterator to a list so it can be freed later
            m_flowGraph.RecordIterator (cloneOfIter);
            return (cloneOfIter);

        }
    bool HasNext()
    {
        return (m_pfgnNext != NULL);
    }
    IInstruction* GetNext()
    {
        if (!HasNext())
        {
            return NULL;
        }
        // get the current item and advance the iterator to the next original instruction
        IInstruction *pInst = m_pfgnNext->GetInstruction()->GetReadOnlyWrapper();
        m_pfgnNext = m_pfgnNext->m_pfgnNextOriginal;
        return pInst;
    }
    void Free()
    {
        //TODO: implement free (not urgent. iterator will be freed eventually by its container)
    }
    IInstruction* GetFirst()
    {
        // return the first item. Note that we do not move the iterator to the next item
        // because GetNext takes care of this
        m_pfgnNext = m_flowGraph.m_lstNodes.GetData(m_flowGraph.m_hStartNode);
        IInstruction *pInst = GetNext();
        return pInst;
    }
    IInstruction* GetLast()
    {
        // return the last item. Note that we do not move the iterator to the previous item
        // because GetPrev takes care of this
        m_pfgnNext = m_flowGraph.m_lstNodes.GetData(m_flowGraph.m_hEndNode);
        IInstruction *pInst = m_pfgnNext->GetInstruction()->GetReadOnlyWrapper();
        return pInst;
    }
    bool HasPrev() 
    {
        // a previous item exists if the iterator does not point to first item
        CFlowGraphNode *pfgnFirst = m_flowGraph.m_lstNodes.GetData(m_flowGraph.m_hStartNode);
        return !(m_pfgnNext == pfgnFirst);
    }
    IInstruction* GetPrev()
    {
        bool bHasPrev = HasPrev();
        if (!bHasPrev)
        {
            return NULL;
        }

        IInstruction *pInst;
        if (NULL == m_pfgnNext)
        {
            pInst = GetLast();
        }
        else
        {
            m_pfgnNext = m_pfgnNext->m_pfgnPrevOriginal;
            pInst = m_pfgnNext->GetInstruction()->GetReadOnlyWrapper();
        }
        return pInst;
    }
private:
    friend class CFlowGraph;

    CFlowGraphNode *m_pfgnNext;     // the next node in the iteration
    CFlowGraph &m_flowGraph;        // the flow-graph on which the iterator iterates

    COrgInstructionsListIterator(CFlowGraph &flowGraph) : m_flowGraph(flowGraph)
    {
        m_pfgnNext = m_flowGraph.m_lstNodes.GetData(m_flowGraph.m_hStartNode);
    }
}; // class COrgInstructionsListIterator

}}

#endif // MRTE_FLOWGRAPH
