/*****************************************************************************
 * Copyright (c) 1997, 2008 Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/
//

#include <assert.h>

#include "JIEInstructionCodec.h"
#include "JClassFile.h"
#include "MRTEInfrastructureDefinitions.h"
#include "ValidityChecks.h"
#include "MIEInstructionFactory.h"


using namespace Martini::MIE;
using namespace Martini::Infrastructure;
using namespace Martini::RTUtil;

//
// A structure for storing try..finally block information
// 
struct STryFinallyInfo
{
    STryFinallyInfo(CMIEInstruction *i_pTry, CMIEInstruction *i_pEndTry,
        CMIEInstruction *i_pFinally, CMIEInstruction *i_pEndFinally):
        pTry(i_pTry), pEndTry(i_pEndTry), pFinally(i_pFinally), 
        pEndFinally(i_pEndFinally) {}
    CMIEInstruction *pTry;
    CMIEInstruction *pEndTry;
    CMIEInstruction *pFinally;
    CMIEInstruction *pEndFinally;
};

//////////////////////////////////////////////////////////////////////////
// class CJIEInstructionCodec implementation
CBCIToMIEInstructionMapper CJIEInstructionCodec::sm_BCI2MIEMapper;

CJIEInstructionCodec::CJIEInstructionCodec(CMethodJ *pBciMethod, CJClassBuilder *pClassBuilder)
: m_pBCIMethod(pBciMethod), m_pBCIClassBuilder(pClassBuilder)
{

}

CJIEInstructionCodec::~CJIEInstructionCodec()
{

}


//
// Decode the BCI Java method and construct the flow graph
//
//TODO: define return codes and revise code as necessary
TResult CJIEInstructionCodec::Decode(CFlowGraph *pFlowGraph)
{
    //TODO: add error handling
    
    // add all method's instructions to the flow graph
    CInsBlocks *pBlocks = m_pBCIMethod->GetInsBlocks();
    CInsBlocks::iterator iterBlks = pBlocks->begin();
    CInsBlocks::iterator iterBlksEnd = pBlocks->end();
    for (; iterBlks != iterBlksEnd; ++iterBlks)
    {
        CInsBlock *pBlk = *iterBlks;
        CInstructions *pInstructions = pBlk->GetInstructions();
        CInstructions::iterator iterInsts = pInstructions->begin();
        CInstructions::iterator iterInstsEnd = pInstructions->end();
        for (; iterInsts != iterInstsEnd; ++iterInsts)
        {
            CInstruction *pInst = *iterInsts;
            // add instruction to the flow graph
            CMIEInstruction *pMieInst = sm_BCI2MIEMapper.BCI2MIE(pInst);
            if (NULL == pMieInst)
            {
                return MRTE_ERROR_FAIL;
            }
            pMieInst->SetMethodInstrumentor(m_pInstrumentor);
            pFlowGraph->AddDecodedInstruction(pMieInst, pInst->GetIP());
        }
    }

    TResult res = UpdateBranchTargets(pFlowGraph);
    CHECK_TRESULT(res);
    // decode protected blocks (exception table) and add to flow-graph
    TProtectedBlocksList lstBlocks;
    res = DecodeProtectedBlocks(&lstBlocks);
    if (res != MRTE_RESULT_OK)
    {
        RTUtil::FreeListItems(lstBlocks);
        return res;
    }
    res = AddProtectedBlocksToGraph(pFlowGraph, lstBlocks);
    RTUtil::FreeListItems(lstBlocks);
    CHECK_TRESULT(res);
    return res;
}

//
// Generate the BCI method code from the flow-graph
//
TResult CJIEInstructionCodec::Encode(CFlowGraph &flowGraph)
{
    //TODO: implement - generate goto/goto_w and jsr/jsr_w according to operands size
    TResult res;
    res = ProcessTryFinallyBlocksNoJsr(&flowGraph);

    CHECK_TRESULT(res);
    UpdateInstructionOffsets(flowGraph);
    GenerateBCIMethodCode(flowGraph);
    return MRTE_RESULT_OK;
}

static void BuildFinallyBlockList(
    Martini::RTUtil::MList<STryFinallyInfo*> *pBlockList, 
    CFlowGraph *pFlowGraph)
{
    // build a list of "finally" blocks
    TMRTEHandle h = NULL;
    CMIEInstruction *pHandler;
    CMIEInstruction *pEndFinally;
    CMIEInstruction *pTry;
    CMIEInstruction *pMIEInst;
    SOperand *pOp;

    TInstructionListIterator *pIterInsts = 
        pFlowGraph->GetInstructionListIterator(IET_ALL_INSTRUCTIONS);
    while (pIterInsts->HasNext())
    {
        pMIEInst = (CMIEInstruction*)pIterInsts->GetNext();
        EMnemonic mnem = pMIEInst->GetMnemonic();
        if (MNM_PSEUDO_END_TRY == mnem)
        {
            // add the block to the list only if it is a try..finally block
            pOp = pMIEInst->InternalGetOperand(2);
            assert(OT_TARGET == pOp->type);
            pHandler = (CMIEInstruction*)pOp->val.pTarget;
            if (MNM_PSEUDO_FINALLY == pHandler->GetMnemonic())
            {
                pOp = pMIEInst->InternalGetOperand(1);
                assert(OT_TARGET == pOp->type);
                pTry = (CMIEInstruction*)pOp->val.pTarget;

                pOp = pHandler->InternalGetOperand(1);
                assert(OT_TARGET == pOp->type);
                pEndFinally = (CMIEInstruction*)pOp->val.pTarget;

                h = pBlockList->InsertAfter(h, new STryFinallyInfo(pTry, pMIEInst, pHandler, 
                    pEndFinally));
            }
        } // of if (current instruction is .END_TRY)
    } // of while
}

//
// Processes try..finally blocks in the flow-graph and adds the necessary code
// for implementing the "finally" logic
//
TResult CJIEInstructionCodec::ProcessTryFinallyBlocks(CFlowGraph *pFlowGraph)
{
    RTUtil::MList<STryFinallyInfo*> blockList;
    BuildFinallyBlockList(&blockList, pFlowGraph);
    
    TMRTEHandle h = blockList.GetFirst();
    if (NULL == h)
    {
        // no finally blocks
        return MRTE_RESULT_OK;
    }

    STryFinallyInfo *pBlockInfo;
    unsigned int uiRetVarInd;   // a local variable index to store the address
                                // to which the 'ret' instruction should return

    unsigned int uiExVarInd;    // a local variable index to store the exception received
                                // in the "catch all" block
    CMIEInstruction *pNop;
    CMIEInstruction *pGoto;
    CMIEInstruction *pStore;
    CMIEInstruction *pRet;
    CMIEInstruction *pLoad;
    CMIEInstruction *pJsr;
    CMIEInstruction *pAThrow;
    CMIEInstruction *pCatch;
    CMIEInstruction *pEndCatch;

    CMIEInstruction *pFinallyFirstInst;
    CMIEInstruction *pCatchFirstInst;
    CMIEInstruction *pCatchLastInst;

    RTUtil::MHashNoSync<UIOP, bool> processedInstMap;   // Maps an instruction to a boolean
                                                        // Used for flagging 'return' instructions

    processedInstMap.Init(true, false, 10);     // Set-up the map to return false for 
                                                // non-existing items

    CCodeAttribute *pCodeAttr = m_pBCIMethod->GetCodeAttribute();
    assert(NULL != pCodeAttr);
    
    // process the finally blocks from innermost to outermost
    while (NULL != h)
    {
        pBlockInfo = blockList.GetData(h);
        h = blockList.GetNext(h);

        //
        // If the finally block is not the last block in the flow-graph, 
        // add code to skip the instructions in the finally block.
        // This is done by adding a 'nop' after the .END_FINALLY and adding a 'goto'
        // to this 'nop' before the .FINALLY.
        //
        CMIEInstruction *pInstAfterFinally = pFlowGraph->GetSucceedingInstruction(
            pBlockInfo->pEndFinally);
        if (MNM_PSEUDO_END != pInstAfterFinally->GetMnemonic())
        {
            pNop = CMIEInstructionFactory::CreateInstruction(MNM_NOP, m_pInstrumentor);
            AddCodeGenInstAfter(pFlowGraph, pBlockInfo->pEndFinally, pNop);
            pGoto = CreateGoto(pNop);
            AddCodeGenInstBefore(pFlowGraph, pBlockInfo->pFinally, pGoto);
        }

        //
        // Modify the "finally" block so it will behave like a true Java "finally" block.
        // Execution will reach this block through a 'jsr' call. Therefore we need
        // to store the return address into a local variable at the beginning of the 
        // block and use 'ret' at the end of this block.
        //

        // Note: we assume that we will not reach the local variable limitation.
        // This is checked during the VerifyCode() API.
        
        uiRetVarInd = pCodeAttr->GetMaxLocals();
        pCodeAttr->SetMaxLocals(uiRetVarInd + 1);

        pStore = CreateLoadStore(MNM_ASTORE, uiRetVarInd);
        AddCodeGenInstAfter(pFlowGraph, pBlockInfo->pFinally, pStore);
        pFinallyFirstInst = pStore;

        pRet = CreateRet(uiRetVarInd);
        AddCodeGenInstBefore(pFlowGraph, pBlockInfo->pEndFinally, pRet);

        // 
        // Add a "catch all" block that will serve as a generic exception handler
        // for all instructions between the .TRY and .END_TRY. This handler will
        // transfer control to the "finally" block by using a 'jsr', and will then
        // re-throw the exception received. This catch block is added immediately after
        // the .END_TRY instruction.
        //
        pCatch = CMIEInstructionFactory::CreateInstruction(MNM_PSEUDO_CATCH, 
            m_pInstrumentor);
        AddCodeGenInstAfter(pFlowGraph, pBlockInfo->pEndTry, pCatch);
        pEndCatch = CMIEInstructionFactory::CreateInstruction(MNM_PSEUDO_END_CATCH, 
            m_pInstrumentor);
        AddCodeGenInstAfter(pFlowGraph, pCatch, pEndCatch);

        uiExVarInd = pCodeAttr->GetMaxLocals();
        pCodeAttr->SetMaxLocals(uiExVarInd + 1);

        pStore = CreateLoadStore(MNM_ASTORE, uiExVarInd);
        AddCodeGenInstAfter(pFlowGraph, pCatch, pStore);
        pCatchFirstInst = pStore;
        pJsr = CreateJsr(pFinallyFirstInst);
        AddCodeGenInstAfter(pFlowGraph, pStore, pJsr);
        pLoad = CreateLoadStore(MNM_ALOAD, uiExVarInd);
        AddCodeGenInstAfter(pFlowGraph, pJsr, pLoad);
        pAThrow = CMIEInstructionFactory::CreateInstruction(MNM_ATHROW, m_pInstrumentor);
        AddCodeGenInstAfter(pFlowGraph, pLoad, pAThrow);
        pCatchLastInst = pAThrow;

        //
        // If the "try" block does not falls-through to the next basic block (it does not
        // end with 'return' or 'athrow'), we need to add code to skip the new "catch" block,
        // because we want the catch block to execute only is response to an exception.
        //
        CMIEInstruction *pLastTryBlockInst = pFlowGraph->GetPrecedingRealInstruction(
            pBlockInfo->pEndTry);
        if (CMIEInstruction::SEM_RET != pLastTryBlockInst->GetSemantics()
            && CMIEInstruction::SEM_THROW != pLastTryBlockInst->GetSemantics())
        {
            pNop = CMIEInstructionFactory::CreateInstruction(MNM_NOP, m_pInstrumentor);
            AddCodeGenInstAfter(pFlowGraph, pEndCatch, pNop);
            pGoto = CreateGoto(pNop);
            AddCodeGenInstBefore(pFlowGraph, pCatch, pGoto);
        }

        //
        // Modify the operands of the .TRY, END_TRY, .CATCH, .END_CATCH instructions
        // and associate them with each other.
        // This is needed for the encoder to generate the correct entries in the 
        // exception table.
        //
        SOperand opCatch;
        opCatch.type = OT_TARGET;
        opCatch.val.pTarget = pCatch;

        SOperand opEndCatch;
        opEndCatch.type = OT_TARGET;
        opEndCatch.val.pTarget = pEndCatch;

        SOperand opTry;
        opTry.type = OT_TARGET;
        opTry.val.pTarget = pBlockInfo->pTry;

        SOperand opHandlerType;
        opHandlerType.type = OT_JAVA_CP_INDEX;
        opHandlerType.val.cpIndex = 0;

        SOperand opCatchRealInst;
        opCatchRealInst.type = OT_TARGET;
        opCatchRealInst.val.pTarget = pCatchFirstInst;
        
        SOperand opEndCatchRealInst;
        opEndCatchRealInst.type = OT_TARGET;
        opEndCatchRealInst.val.pTarget = pCatchLastInst;

        pBlockInfo->pTry->InternalSetOperand(2, &opCatch);
        pBlockInfo->pEndTry->InternalSetOperand(2, &opCatch);
        pCatch->InternalSetOperand(&opCatchRealInst, &opEndCatch, &opTry, &opHandlerType);
        pEndCatch->InternalSetOperand(&opEndCatchRealInst, &opCatch);

        //
        // Modify the code of the protected block (between .TRY and .END_TRY):
        // 1. If the block does not end with a 'return', 'athrow' or 'jsr', 
        //    add a 'jsr' that jumps to the "finally" block.
        // 2. Insert a 'jsr' that jumps to the "finally" block before any 'return' 
        //    instruction.
        //
        SOperand *pOpEndTryRealInst = pBlockInfo->pEndTry->InternalGetOperand(0);
        CMIEInstruction *pTryLastInst = (CMIEInstruction*)pOpEndTryRealInst->val.pTarget;
        if (CMIEInstruction::SEM_RET != pTryLastInst->GetSemantics()
            && CMIEInstruction::SEM_RETSR != pTryLastInst->GetSemantics()
            && CMIEInstruction::SEM_THROW != pTryLastInst->GetSemantics())
        {
            pJsr = CreateJsr(pFinallyFirstInst);
            AddCodeGenInstBefore(pFlowGraph, pBlockInfo->pEndTry, pJsr);
        }

        CMIEInstruction *pInst = pBlockInfo->pTry;
        while (pInst != pBlockInfo->pEndTry)
        {
            if (CMIEInstruction::SEM_RET == pInst->GetSemantics())
            {
                bool bProcessed = processedInstMap.Get((UIOP)pInst);
                if (!bProcessed)
                {
                    AddJsrBeforeReturn(pInst, pFinallyFirstInst, pFlowGraph);
                    processedInstMap.Set((UIOP)pInst, true);
                }
            }
            pInst = pFlowGraph->GetSucceedingInstruction(pInst);
        }

    }

    RTUtil::FreeListItems(blockList);
    return MRTE_RESULT_OK;
}

//
// Adds 'jsr' instruction before the specified 'return' instruction.
// The 'jsr' is wrapped with instructions for backing-up the return value
// (top-of-stack) in a local variable and for restoring it after 'jsr'
// returns.
//
// The method assume the input is valid.
//
void CJIEInstructionCodec::AddJsrBeforeReturn(CMIEInstruction *pReturn, 
                                              CMIEInstruction* pJsrTarget,
                                              CFlowGraph *pFlowGraph)
{
    EMnemonic mnem = pReturn->GetMnemonic();
    CMIEInstruction *pJsr;
    if (MNM_RETURN == mnem)
    {
        // regular 'return', so there is no return value on the stack
        pJsr = CreateJsr(pJsrTarget);
        AddCodeGenInstBefore(pFlowGraph, pReturn, pJsr);
        return;
    }

    // allocate a new local variable (we assume this is possible)
    CCodeAttribute *pCodeAttr = m_pBCIMethod->GetCodeAttribute();
    unsigned int uiRetValVarInd = pCodeAttr->GetMaxLocals();
    if (MNM_LRETURN == mnem || MNM_DRETURN == mnem)
    {
        pCodeAttr->SetMaxLocals(uiRetValVarInd + 2);
    }
    else
    {
        pCodeAttr->SetMaxLocals(uiRetValVarInd + 1);
    }

    // generate code
    CMIEInstruction *pLoad;
    CMIEInstruction *pStore;
    pStore = CreateLoadStore(GET_STORE_OPCODE_FOR_RETURN(mnem), uiRetValVarInd);
    AddCodeGenInstBefore(pFlowGraph, pReturn, pStore);
    pJsr = CreateJsr(pJsrTarget);
    AddCodeGenInstAfter(pFlowGraph, pStore, pJsr);
    pLoad = CreateLoadStore(GET_LOAD_OPCODE_FOR_RETURN(mnem), uiRetValVarInd);
    AddCodeGenInstAfter(pFlowGraph, pJsr, pLoad);
}

//
// Update the "actual offset" member of each instruction
//
void CJIEInstructionCodec::UpdateInstructionOffsets(CFlowGraph &flowGraph)
{
    TInstructionListIterator *pIterInsts = 
        flowGraph.GetInstructionListIterator(IET_ALL_INSTRUCTIONS);
    CMIEInstruction *pMIEInst;      // current MIE instruction being processed
    unsigned int uiIP = 0;
    while (pIterInsts->HasNext())
    {
        pMIEInst = (CMIEInstruction*)pIterInsts->GetNext();
        pMIEInst->SetActualOffset(uiIP);
        if (!pMIEInst->IsPseudo())
        {
            // IP is incremented only for real instructions
            uiIP += pMIEInst->GetSize(uiIP);
        }
        

    } // end while
    pIterInsts->Free();
}

void CJIEInstructionCodec::GenerateBCIMethodCode(CFlowGraph &flowGraph)
{
    // backup the local variable table (if exists)
    CLocalVariableTable orgLocalVarTable;
    CLocalVariableTypeTable orgLocalVarTypeTable;
    unsigned int uiOriginalCodeLength = 0;
    CCodeAttribute *pCodeAttr = m_pBCIMethod->GetCodeAttribute();
    if (pCodeAttr)
    {
        // backup the original code length. This is needed later for fixing the local
        // variables table
        uiOriginalCodeLength = pCodeAttr->GetCodeLength();
        CLocalVariableTableAttribute *pLocalVarAttr = pCodeAttr->GetLocalVariables();
        if (pLocalVarAttr)
        {
            orgLocalVarTable = pLocalVarAttr->GetLocalVariableTable();
        }
        CLocalVariableTypeTableAttribute *pLocalVarTypeAttr = 
            pCodeAttr->GetLocalVariableTypes();
        if (pLocalVarTypeAttr)
        {
            orgLocalVarTypeTable = pLocalVarTypeAttr->GetLocalVariableTypeTable();
        }
    }

    // delete the instruction objects of the BCI method
    CInsBlocks *pblks = m_pBCIMethod->GetInsBlocks();
    CInsBlocks::iterator iterBlks;
	for (iterBlks = pblks->begin(); iterBlks < pblks->end(); ++iterBlks)
	{
		delete *iterBlks;
	}
    pblks->clear();

    // clear the label table of the BCI method
    CLabels *pLabels = m_pBCIMethod->GetLabels();
    pLabels->clear();

    // clear the line numbers map of the BCI method.
    // this will cause the BCI method emission process not to emit the line numbers
    // attribute. We will take care of this later (after emission).
    CLineNumbers *pBCILineNumbersMap = m_pBCIMethod->GetLineNumbers();
    if (NULL != pBCILineNumbersMap)
    {
        pBCILineNumbersMap->clear();
    }

    unsigned int uiIP = 0;
	int blockNum = 0;
	CInsBlock *pBlock = new CInsBlock(uiIP, blockNum);
	bool isEmptyBlock = true;
    CMIEInstruction *pMIEInst;      // current MIE instruction being processed
    CInstruction *pBCIInst;         // BCI instruction being created
    EMnemonic mnem;                 // mnemonic of the MIE instruction being processed
    
    TProtectedBlocksList blockList;
    CMIEProtectedBlock *pProtectedBlock;
    TMRTEHandle h = NULL;

    // traverse the flow-graph and create BCI blocks and instructions
    TInstructionListIterator *pIterInsts = 
        flowGraph.GetInstructionListIterator(IET_ALL_INSTRUCTIONS);
    while (pIterInsts->HasNext())
    {
        pMIEInst = (CMIEInstruction*)pIterInsts->GetNext();
        mnem = pMIEInst->GetMnemonic();
        switch (mnem)
        {
        case MNM_PSEUDO_END_TRY:
            // add a protected block
            pProtectedBlock = CreateProtectedBlock(flowGraph, pMIEInst);
            // mark the end of the protected block with a label
            pLabels->AddLabel(pProtectedBlock->uiHandlerEndOffset);
            assert(pProtectedBlock != NULL);
            h = blockList.InsertAfter(h, pProtectedBlock);
            // fall-through
        case MNM_PSEUDO_TRY:
        case MNM_PSEUDO_FINALLY:
        case MNM_PSEUDO_END_FINALLY:
        case MNM_PSEUDO_CATCH:
        case MNM_PSEUDO_END_CATCH:
        case MNM_PSEUDO_TARGET : 
            // create a new basic block if needed
            if (!isEmptyBlock)
            {
                pblks->push_back(pBlock);
                blockNum++;
                pBlock = new CInsBlock(uiIP, blockNum);
                isEmptyBlock = true;
            }
            // add a label to the BCI method
            // (BCI uses labels for locating branch targets and marking protected blocks)
            pLabels->AddLabel(uiIP);
        	break;
        } // end switch
        if (!pMIEInst->IsPseudo())
        {
            if (pMIEInst->GetSemantics() == CMIEInstruction::SEM_SPECIAL)
            {
                TBciInstructionList lstBciInstructions;
                sm_BCI2MIEMapper.EncodeSpecialInstruction(&lstBciInstructions, pMIEInst,
                    m_pBCIClassBuilder);
                assert(lstBciInstructions.Count() > 0);
                TMRTEHandle hBciInst = lstBciInstructions.GetFirst();
                while (hBciInst != NULL)
                {
                    pBCIInst = lstBciInstructions.GetData(hBciInst);
                    pBCIInst->SetIP(uiIP);
                    pBlock->AddInstruction(pBCIInst);
                    isEmptyBlock = false;
                    uiIP += pBCIInst->GetSize(uiIP);
                    hBciInst = lstBciInstructions.GetNext(hBciInst);
                }
            }
            else
            {
                // create a BCI instruction and add it to the current basic block
                pBCIInst = sm_BCI2MIEMapper.MIE2BCI(pMIEInst);
                assert(pBCIInst != NULL);
                pBCIInst->SetIP(uiIP);
                pBlock->AddInstruction(pBCIInst);
                isEmptyBlock = false;
                uiIP += pBCIInst->GetSize(uiIP);
            }
        } // end if (not pseudo instruction)
        
    } // end while
    pIterInsts->Free();

    // add the last block to the method (if it is not empty)
    if (!isEmptyBlock)
    {
        pblks->push_back(pBlock);
    }
    else
    {
        delete pBlock;
    }
    
    //TODO: need to update the m_nextUniqueIP member for good behavior... need to add a public setter for this

    // update the BCI method exception table
    GenerateBCIMethodExceptionTable(blockList);

    // generate the method code into the code attribute
    //TODO: duplicating some of the emit logic instead of invoking emit() may save some redundant instruction processing
    m_pBCIMethod->Emit();
    
    // update the method line number attribute
    UpdateBCIMethodLineNumberTable(flowGraph);

    // update the local variable table
    if (!orgLocalVarTable.empty())
    {
        UpdateBCIMethodLocalVariableTable(flowGraph, orgLocalVarTable,
            orgLocalVarTypeTable, uiOriginalCodeLength);
    }

    // clean-up
    RTUtil::FreeListItems(blockList);
}

void CJIEInstructionCodec::GenerateBCIMethodExceptionTable(
    const TProtectedBlocksList& blockList)
{
    // clear the current exception table
    CMtdExTable *pMtdExTable = m_pBCIMethod->GetExTable();
    CMtdExTable::iterator iterMtdExTable = pMtdExTable->begin();
    for (; iterMtdExTable != pMtdExTable->end(); ++iterMtdExTable)
    {
        delete *iterMtdExTable;
    }
    pMtdExTable->clear();

    TMRTEHandle h = blockList.GetFirst();
    if (NULL == h)
    {
        // there are no exceptions
        return;
    }

    // rebuild the table
    while (h != NULL)
	{
        CMIEProtectedBlock *pProtectedBlock = blockList.GetData(h);
        h = blockList.GetNext(h);
        // find the basic block that contains the handler instructions
		CInsBlock* pblkHandler = m_pBCIMethod->FindBlock(
            pProtectedBlock->uiHandlerStartOffset);
        // add the exception
		m_pBCIMethod->AddException(new CMethodExceptionJ(
            pProtectedBlock->uiHandlerType, pProtectedBlock->uiTryStartOffset,
            pProtectedBlock->uiTryEndOffset, pblkHandler));
    } // end while (more protected blocks to process)
    
}

void CJIEInstructionCodec::UpdateBCIMethodLineNumberTable(CFlowGraph &flowGraph)
{
    CCodeAttribute *pCodeAttr = m_pBCIMethod->GetCodeAttribute();
    if (NULL == pCodeAttr)
    {
        // no code. nothing to do
        return;
    }
	CLineNumberTableAttribute* pLineNums = pCodeAttr->GetLineNumbers();
    if (NULL == pLineNums)
    {
        // no line number table. nothing to do
        return;
    }
	CLineNumberTable& table = pLineNums->GetLineNumberTable();
	CLineNumberTable::iterator iterTable;
	CLineNumbers::iterator itrLines;

	for(iterTable = table.begin(); iterTable != table.end(); ++iterTable)
	{
		
		IP_t ip = (*iterTable)->GetStartPC();
        CMIEInstruction* pInst = flowGraph.GetInstructionAtOffset(ip);
        if (NULL != pInst)
        {
            unsigned int uiNewIP = pInst->GetActualOffset();
			(*iterTable)->SetStartPC(uiNewIP);
        }
	}
    
}

//
// Updates the local variable table of the BCI method
//
// NOTE: call this function only after the BCI method was emitted, as it requires
// the new size of the code array (after instrumentation)
//
//
// Parameters:
//      flowGraph               [in]    : the flow graph of the instrumented method
//      originalLocalVarTable   [in]    : the original local variable table
//                                        (before instrumentation)
//      originalLocalVarTypeTable [in]  : the original local variable type table (LVTT)
//                                        (before instrumentation)
//      uiOriginalCodeLength    [in]    : the original size of the code array
//                                        (before instrumentation)
//
void CJIEInstructionCodec::UpdateBCIMethodLocalVariableTable(CFlowGraph &flowGraph,
    CLocalVariableTable &originalLocalVarTable,
    CLocalVariableTypeTable &originalLocalVarTypeTable,
    unsigned int uiOriginalCodeLength)
{
    CCodeAttribute *pCodeAttr = m_pBCIMethod->GetCodeAttribute();
    if (NULL == pCodeAttr)
    {
        // no code. nothing to do
        return;
    }
	CLocalVariableTableAttribute *pLocalVarAttr = pCodeAttr->GetLocalVariables();
    if (NULL == pLocalVarAttr)
    {
        // no local variables. nothing to do
        return;
    }
    CLocalVariableTable& localVarTable = pLocalVarAttr->GetLocalVariableTable();
    CLocalVariableTable::iterator iterOrgTable = originalLocalVarTable.begin();
    CLocalVariableTable::iterator iterTable;
	CLocalVariableTable::iterator iterTableEnd = localVarTable.end();

	CLocalVariableTypeTableAttribute *pLocalVarTypeAttr = pCodeAttr->GetLocalVariableTypes();
    CLocalVariableTypeTable lvttEmpty;
    CLocalVariableTypeTable& localVarTypeTable = 
        pLocalVarTypeAttr?pLocalVarTypeAttr->GetLocalVariableTypeTable():lvttEmpty;
    CLocalVariableTypeTable::iterator iterOrgTypeTable = originalLocalVarTypeTable.begin();
    CLocalVariableTypeTable::iterator iterTypeTable;
	CLocalVariableTypeTable::iterator iterTypeTableEnd = localVarTypeTable.end();

    // update the "start_pc" and "length" fields of each local variable in the
    // Local Variable Table
    for(iterTable = localVarTable.begin(); iterTable != iterTableEnd; ++iterTable, ++iterOrgTable)
	{
		CLocalVariableInfo *pOrgVarInfo = *iterOrgTable;
        CMIEInstruction *pInst;
        unsigned int uiNewIP;
        pInst = flowGraph.GetInstructionAtOffset(pOrgVarInfo->GetStartPC());
        if (pInst)
        {
            uiNewIP = pInst->GetActualOffset();
            (*iterTable)->SetStartPC(uiNewIP);
        }
        pInst = flowGraph.GetInstructionAtOffset(pOrgVarInfo->GetStartPC() + 
            pOrgVarInfo->GetLength());
        if (pInst)
        {
            uiNewIP = pInst->GetActualOffset();
            (*iterTable)->SetLength(uiNewIP - (*iterTable)->GetStartPC());
        }
    } // for

    // do the same for the Local Variable Type Table
    for(iterTypeTable = localVarTypeTable.begin(); iterTypeTable != iterTypeTableEnd;
        ++iterTypeTable, ++iterOrgTypeTable)
	{
		CLocalVariableTypeInfo *pOrgVarTypeInfo = *iterOrgTypeTable;
        CMIEInstruction *pInst;
        unsigned int uiNewIP;
        pInst = flowGraph.GetInstructionAtOffset(pOrgVarTypeInfo->GetStartPC());
        if (pInst)
        {
            uiNewIP = pInst->GetActualOffset();
            (*iterTypeTable)->SetStartPC(uiNewIP);
        }
        pInst = flowGraph.GetInstructionAtOffset(pOrgVarTypeInfo->GetStartPC() + 
            pOrgVarTypeInfo->GetLength());
        if (pInst)
        {
            uiNewIP = pInst->GetActualOffset();
            (*iterTypeTable)->SetLength(uiNewIP - (*iterTypeTable)->GetStartPC());
        }
    } // for

    // update the "length" field of all variables for which
    // start_pc + length pointed to the end of the original method
    iterOrgTable = originalLocalVarTable.begin();
    unsigned int uiActualCodeLength = pCodeAttr->GetCodeLength();
    for(iterTable = localVarTable.begin(); iterTable != iterTableEnd; ++iterTable, ++iterOrgTable)
	{
		CLocalVariableInfo *pOrgVarInfo = *iterOrgTable;
        if (pOrgVarInfo->GetStartPC() + pOrgVarInfo->GetLength() == uiOriginalCodeLength)
        {
            (*iterTable)->SetLength(uiActualCodeLength - (*iterTable)->GetStartPC());
        }
    } // for

    // do the same for the Local Variable Type Table
    iterOrgTypeTable = originalLocalVarTypeTable.begin();
    for(iterTypeTable = localVarTypeTable.begin(); iterTypeTable != iterTypeTableEnd;
        ++iterTypeTable, ++iterOrgTypeTable)
	{
		CLocalVariableTypeInfo *pOrgVarTypeInfo = *iterOrgTypeTable;
        if (pOrgVarTypeInfo->GetStartPC() + pOrgVarTypeInfo->GetLength() == uiOriginalCodeLength)
        {
            (*iterTypeTable)->SetLength(uiActualCodeLength - (*iterTypeTable)->GetStartPC());
        }
    } // for

}

//
// Create protected block information
//
// Parameters:
//      pFlowGraph  : the flow graph.
//      pEndTry     : the pseudo .endtry instruction from which the extract
//                    the protected block information
//
// Returns:
//      CMIEProtectedBlock* :   the created protected block
//      NULL                :   error
//
CMIEProtectedBlock* CJIEInstructionCodec::CreateProtectedBlock(CFlowGraph &flowGraph,
                                                               CMIEInstruction *pEndTry)
{
    //
    // pEndTry points to the end of the protected block. Its offset is the offset
    // of the instruction that immediately follows the last instruction in the protected
    // block. When creating the protected block structure, we patch the "end try" pointer
    // to point to the last real instruction in the protected block, because
    // this is what the IBM BCI engine expects to get.
    //
    if (pEndTry->GetMnemonic() != MNM_PSEUDO_END_TRY)
    {
        return NULL;
    }
    CMIEProtectedBlock *pProtectedBlock = new CMIEProtectedBlock();
    pProtectedBlock->type = CMIEProtectedBlock::PBT_JAVA_EXCEPTION;

    // Bugzilla 179653:
    // "end try" should point to the last real instruction that precedes the .endtry 
    // instruction, because new instructions might have been added at the end of the
    // original protected block
    CMIEInstruction *pLastInstInBlock = flowGraph.GetPrecedingRealInstruction(pEndTry);
    assert(pLastInstInBlock != NULL && !pLastInstInBlock->IsPseudo());
    pProtectedBlock->uiTryEndOffset = pLastInstInBlock->GetActualOffset();

    SOperand *pOpTryStart = pEndTry->InternalGetOperand(1);
    SOperand *pOpHandlerStart = pEndTry->InternalGetOperand(2);
    pProtectedBlock->uiTryStartOffset = 
        ((CMIEInstruction*)pOpTryStart->val.pTarget)->GetActualOffset();

    CMIEInstruction *pHandler = (CMIEInstruction*)pOpHandlerStart->val.pTarget;
    pProtectedBlock->uiHandlerStartOffset = pHandler->GetActualOffset();

    SOperand *pOpHandlerType = pHandler->InternalGetOperand(3);
    pProtectedBlock->uiHandlerType = pOpHandlerType->val.cpIndex;

#ifdef MARTINI_CIE
    // .NET only: update the location where the protected block ends
    //TODO: check if we need to patch the "end" pointer as well (like in Java)
    SOperand *pOpHandlerEnd = pHandler->InternalGetOperand(1);
    pProtectedBlock->uiHandlerEndOffset = 
        ((CMIEInstruction*)pOpHandlerEnd->val.pTarget)->GetActualOffset();
#endif // MARTINI_CIE

    return pProtectedBlock;
}
//
// Replaces branch-target offsets with pointers to the target instruction.
// Also adds pseudo .target instructions before each target.
//
TResult CJIEInstructionCodec::UpdateBranchTargets(CFlowGraph *pFlowGraph)
{
    CMIEInstruction *pTargetInst;
    CMIEInstruction *pDefaultTargetInst;
    unsigned int *vuiOffsets;
    unsigned int uiTargetsCount;

    TInstructionListIterator *pInstIter = 
        pFlowGraph->GetInstructionListIterator(IET_ALL_INSTRUCTIONS);
    while (pInstIter->HasNext())
    {
        CMIEInstruction *pInst = (CMIEInstruction*)pInstIter->GetNext();
        CMIEInstruction::ESemanticTag semTag = pInst->GetSemantics();
        switch (semTag)
        {
        case CMIEInstruction::SEM_BRANCH:       
        case CMIEInstruction::SEM_CONDBRANCH:   
        case CMIEInstruction::SEM_JSR:  
            // the instruction is a branch. Obtain its target
            pTargetInst = GetTargetInstructionFromOffset(pFlowGraph,
                pInst->GetBranchAbsoluteOffset());
            if (NULL == pTargetInst)
            {
                return MRTE_ERROR_UNEXPECTED;
            }
            // update the target of the branch instruction
            pInst->InternalSetBranchTarget(pTargetInst);
            break;
        case CMIEInstruction::SEM_SWITCH:
            vuiOffsets = pInst->GetSwitchAbsoluteOffsets();
            uiTargetsCount = pInst->GetSwitchOffsetsCount();
            for (unsigned int i = 0; i < uiTargetsCount; ++i)
            {
                pTargetInst = GetTargetInstructionFromOffset(pFlowGraph,
                    vuiOffsets[i]);
                if (NULL == pTargetInst)
                {
                    return MRTE_ERROR_UNEXPECTED;
                }
                // update the target of the branch instruction
                pInst->InternalSetSwitchTarget(i, pTargetInst);
            }
            // update the default target
            pDefaultTargetInst = GetTargetInstructionFromOffset(pFlowGraph,
                pInst->GetBranchAbsoluteOffset());
            if (NULL == pDefaultTargetInst)
            {
                return MRTE_ERROR_UNEXPECTED;
            }
            pInst->InternalSetSwitchDefaultTarget(pDefaultTargetInst);
            break;
        }
    }
    return MRTE_RESULT_OK;
}

//
// Decodes the method's exception table and build a list of protected blocks.
// The order of blocks in the list is the same as in the internal exception table
// (i.e., handlers are sorted from inner to outer).
//
// Parameters:
//      - lstBlocks [out] : an initialized list object that will store the blocks
//
// Returns:
//      - MRTE_RESULT_OK : success
//      - MRTE_ERROR_OUT_OF_MEMORY : error allocating memory for protected block
//
TResult CJIEInstructionCodec::DecodeProtectedBlocks(TProtectedBlocksList *lstBlocks)
{
    CMtdExTableJ *pExTableJ = (CMtdExTableJ*)m_pBCIMethod->GetExTable();
    if (NULL == pExTableJ)
    {
        // method contains no protected blocks
        return MRTE_RESULT_OK;
    }
    if (pExTableJ->size() == 0)
    {
        // method contains no protected blocks
        return MRTE_RESULT_OK;
    }
    CMtdExTableJ::iterator iterEx = pExTableJ->begin();
    CMtdExTableJ::iterator iterExEnd = pExTableJ->end();
    CMIEProtectedBlock *pb;
    TMRTEHandle h = NULL;
    for (; iterEx != iterExEnd; ++iterEx)
    {
        CMethodExceptionJ *pExJ = (CMethodExceptionJ*)*iterEx;
        pb = new CMIEProtectedBlock();
        CHECK_ALLOC(pb);
        pb->type = CMIEProtectedBlock::PBT_JAVA_EXCEPTION;
        pb->uiTryStartOffset = pExJ->GetStart();
        pb->uiTryEndOffset = pExJ->GetEnd();
        pb->uiHandlerStartOffset = pExJ->GetHandler()->GetOrigIP();
        pb->uiHandlerType = pExJ->GetType();
        h = lstBlocks->InsertAfter(h, pb);
    }
    return MRTE_RESULT_OK;
}

//
// Add exception-related pseudo-instructions to the flow-graph based on the given
// list of protected blocks.
//
// Assumptions:
// 1. The protected blocks list is in the same order as in the Java/.NET method, 
//    i.e. from innermost to outermost.
// 2. The "end try" and "end handler" offsets point to the last instruction in the 
//    protected block, NOT to the instruction immediately after it.
//
// Parameters:
//      pFlowGraph [out] : flow-graph
//      lstBlocks [in] : list of protected blocks.
//
// Returns:
//      MRTE_RESULT_OK : success
//TODO: finalize list of return values.
//      MRTE_ERROR_FAIL : unexpected error
//
TResult CJIEInstructionCodec::AddProtectedBlocksToGraph(CFlowGraph* pFlowGraph,
                                                        const TProtectedBlocksList &lstBlocks)
{
    // blocks processing algorithm:
    // traverse the list backwards (from outermost block to the innermost).
    // for each protected block:
    // 1. add .try pseudo-instruction before the first instruction in the protected block
    // 2. add .endtry pseudo-instruction after the last instruction in the protected block
    // 3. add .starthandler pseudo-instruction before the first instruction in the 
    //    handler block
    // 4. (.NET only): add .endhandler pseudo-instruction after the last instruction in 
    //    the handler block
    TMRTEHandle h = lstBlocks.GetLast();
    if (NULL == h)
    {
        // no protected blocks to process
        return MRTE_RESULT_OK;
    }
    
    CMIEInstruction *pStartTry;
    CMIEInstruction *pEndTry;
    CMIEInstruction *pStartHandler;
    CMIEInstruction *pEndHandler;
    CMIEInstruction *pPseudoTry;
    CMIEInstruction *pPseudoEndTry;
    CMIEInstruction *pPseudoHandler;
    CMIEInstruction *pPseudoEndHandler;
    CMIEInstruction *pPrevInst;
    EMnemonic handlerStartMnemonic;
    EMnemonic handlerEndMnemonic;
    SOperand opTryRealInst;
    opTryRealInst.type = OT_TARGET;
    SOperand opEndTryRealInst;
    opEndTryRealInst.type = OT_TARGET;
    SOperand opHandlerRealInst;
    opHandlerRealInst.type = OT_TARGET;
    SOperand opEndHandlerRealInst;
    opEndHandlerRealInst.type = OT_TARGET;
    TResult res;
    while (h != NULL)
    {
        CMIEProtectedBlock *pb = lstBlocks.GetData(h);
        h = lstBlocks.GetPrev(h);
        switch (pb->type) 
        {
        case CMIEProtectedBlock::PBT_JAVA_EXCEPTION:
            handlerStartMnemonic = MNM_PSEUDO_CATCH;
            handlerEndMnemonic = MNM_PSEUDO_ILLEGAL; // no "end handler" in Java
    	    break;
        case CMIEProtectedBlock::PBT_TRY_CATCH:
            handlerStartMnemonic = MNM_PSEUDO_CATCH;
            handlerEndMnemonic = MNM_PSEUDO_END_CATCH;
    	    break;
        case CMIEProtectedBlock::PBT_TRY_FAULT:
            handlerStartMnemonic = MNM_PSEUDO_FAULT;
            handlerEndMnemonic = MNM_PSEUDO_END_FAULT;
    	    break;
        case CMIEProtectedBlock::PBT_TRY_FILTER:
            handlerStartMnemonic = MNM_PSEUDO_FILTER;
            handlerEndMnemonic = MNM_PSEUDO_END_FILTER;
    	    break;
        case CMIEProtectedBlock::PBT_TRY_FINALLY:
            handlerStartMnemonic = MNM_PSEUDO_FINALLY;
            handlerEndMnemonic = MNM_PSEUDO_END_FINALLY;
    	    break;
        default:
            return MRTE_ERROR_FAIL;
        }
        // add .try
        pStartTry = pFlowGraph->GetInstructionAtOffset(pb->uiTryStartOffset);
        CHECK_NULL_RETURN_ERROR(pStartTry);
        pPseudoTry = CMIEInstructionFactory::CreateInstruction(
            MNM_PSEUDO_TRY, m_pInstrumentor);
        CHECK_ALLOC(pPseudoTry);
        pPseudoTry->SetStatus(CMIEInstruction::IS_ORIGINAL);
        res = pFlowGraph->AddInstructionBefore(pStartTry->GetFlowGraphNode(), pPseudoTry);
        CHECK_TRESULT(res);
        opTryRealInst.val.pTarget = pStartTry;

        // add .endtry
        pEndTry = pFlowGraph->GetInstructionAtOffset(pb->uiTryEndOffset);
        CHECK_NULL_RETURN_ERROR(pEndTry);
        pPseudoEndTry = CMIEInstructionFactory::CreateInstruction(
            MNM_PSEUDO_END_TRY, m_pInstrumentor);
        pPseudoEndTry->SetStatus(CMIEInstruction::IS_ORIGINAL);
        CHECK_ALLOC(pPseudoEndTry);
        res = pFlowGraph->AddInstructionAfter(pEndTry->GetFlowGraphNode(), pPseudoEndTry);
        CHECK_TRESULT(res);
        opEndTryRealInst.val.pTarget = pEndTry;
        
        // add .starthandler
        pStartHandler = pFlowGraph->GetInstructionAtOffset(pb->uiHandlerStartOffset);
        CHECK_NULL_RETURN_ERROR(pStartHandler);
        // if the preceding instruction is already a .starthandler, use it instead.
        // this is a special handling for Java "finally" blocks, in which the handler
        // may be shared by different protected blocks
        pPrevInst = pFlowGraph->GetPrecedingInstruction(pStartHandler);
        CHECK_NULL_RETURN_ERROR(pPrevInst);
        if (pPrevInst->GetMnemonic() == MNM_PSEUDO_FINALLY)
        {
            pPseudoHandler = pPrevInst;
        }
        else
        {
            pPseudoHandler = CMIEInstructionFactory::CreateInstruction(
                handlerStartMnemonic, m_pInstrumentor);
            CHECK_ALLOC(pPseudoHandler);
            pPseudoHandler->SetStatus(CMIEInstruction::IS_ORIGINAL);
            res = pFlowGraph->AddInstructionBefore(pStartHandler->GetFlowGraphNode(), 
                pPseudoHandler);
            CHECK_TRESULT(res);
            opHandlerRealInst.val.pTarget = pStartHandler;
        }
        
        if (pb->type != CMIEProtectedBlock::PBT_JAVA_EXCEPTION)
        {
            // add .endhandler (.NET only)
            pEndHandler = pFlowGraph->GetInstructionAtOffset(pb->uiHandlerEndOffset);
            CHECK_NULL_RETURN_ERROR(pEndHandler);
            CHECK_NULL_RETURN_ERROR(pEndHandler);
            pPseudoEndHandler = CMIEInstructionFactory::CreateInstruction(
                handlerEndMnemonic, m_pInstrumentor);
            CHECK_ALLOC(pPseudoEndHandler);
            pPseudoEndHandler->SetStatus(CMIEInstruction::IS_ORIGINAL);
            res = pFlowGraph->AddInstructionAfter(pEndHandler->GetFlowGraphNode(), 
                pPseudoEndHandler);
            CHECK_TRESULT(res);
            opEndHandlerRealInst.val.pTarget = pEndHandler;
        }
        else
        {
            // .endhandler is not available in Java
            pPseudoEndHandler = NULL;
        }
        
        // update pseudo-instruction operands
        SOperand op2;
        op2.type = OT_TARGET;
        SOperand op3;
        op3.type = OT_TARGET;
        SOperand op4;
        op4.type = OT_NO_OPERAND;

        // .try operands: first real instruction, .endtry, .handler
        op2.val.pTarget = pPseudoEndTry;
        op3.val.pTarget = pPseudoHandler;
        pPseudoTry->InternalSetOperand(&opTryRealInst, &op2, &op3);

        // .endtry operands: last real instruction, .try, .handler
        op2.val.pTarget = pPseudoTry;
        op3.val.pTarget = pPseudoHandler;
        pPseudoEndTry->InternalSetOperand(&opEndTryRealInst, &op2, &op3);

        // .handler operands: first real instruction, .endhandler, .try, exception type
        op2.val.pTarget = pPseudoEndHandler;
        op3.val.pTarget = pPseudoTry;
        if (CMIEProtectedBlock::PBT_JAVA_EXCEPTION == pb->type)
        {
            // Java: exception type is a constant-pool index
#ifdef MARTINI_JIE
            op4.type = OT_JAVA_CP_INDEX;
            op4.val.cpIndex = pb->uiHandlerType;
#endif
        }
        else
        {
            // .NET: exception type is a token
#ifdef MARTINI_CIE
            op4.type = OT_DOT_NET_TOKEN;
            op4.val.token = pb->uiHandlerType;
#endif
        }
        pPseudoHandler->InternalSetOperand(&opHandlerRealInst, &op2, &op3, &op4);

        // .NET only: .endhandler operands: last real instruction, .handler
        if (pb->type != CMIEProtectedBlock::PBT_JAVA_EXCEPTION)
        {
            op2.val.pTarget = pPseudoHandler;
            pPseudoEndHandler->InternalSetOperand(&opEndHandlerRealInst, &op2);
        }
    } // while (more protected blocks to process)
    return MRTE_RESULT_OK;
}

CMIEInstruction* CJIEInstructionCodec::GetTargetInstructionFromOffset(
    CFlowGraph *pFlowGraph, unsigned int uiOffset)
{
    //TEST: when uiOffset already points to .target (seems impossible....)

    //
    // The method works as follows: 
    // Look-up the target instruction in the flow-graph.
    // If the target is a pseudo .target, it is returned.
    // If not, and the instruction preceding it is a pseudo .target
    // this .target will be returned.
    // Otherwise, a new .target instruction will be added just before the target 
    // instruction and will be returned.
    //
    CMIEInstruction *pTargetInst = pFlowGraph->GetInstructionAtOffset(uiOffset);
    CHECK_RETURN_NULL(pTargetInst);
    if (pTargetInst->GetMnemonic() == MNM_PSEUDO_TARGET)
    {
        return pTargetInst;
    }

    // if we reached here, pTargetInst must mot be a pseudo instruction
    if (pTargetInst->GetMnemonic() >= MNM_FIRST_PSEUDO)
    {
        return NULL;
    }

    CMIEInstruction *pInstBeforeTarget = pFlowGraph->GetPrecedingInstruction(
        pTargetInst);
    if (pInstBeforeTarget->GetMnemonic() == MNM_PSEUDO_TARGET)
    {
        return pInstBeforeTarget;
    }

    // create a new pseudo .target instruction and update its operands
    CMIEInstruction *pPseudoTarget = CMIEInstructionFactory::CreateInstruction(
        MNM_PSEUDO_TARGET, m_pInstrumentor);
    CHECK_RETURN_NULL(pPseudoTarget);
    TResult res = pFlowGraph->AddInstructionBefore(pTargetInst->GetFlowGraphNode(),
        pPseudoTarget);
    pPseudoTarget->SetStatus(CMIEInstruction::IS_ORIGINAL);
    if (res != MRTE_RESULT_OK)
    {
        return NULL;
    }
    SOperand op;
    op.type = OT_TARGET;
    op.val.pTarget = pTargetInst;
    pPseudoTarget->InternalSetOperand(&op);
    return pPseudoTarget;
    
}

CMIEInstruction* CJIEInstructionCodec::CreateGoto(CMIEInstruction *pTarget)
{
    CMIEInstruction *pNewInst = CMIEInstructionFactory::CreateInstruction(
        MNM_GOTO_W, m_pInstrumentor);
    pNewInst->SetBranchTarget(pTarget);
    return pNewInst;
}

CMIEInstruction* CJIEInstructionCodec::CreateJsr(CMIEInstruction *pTarget)
{
    CMIEInstruction *pNewInst = CMIEInstructionFactory::CreateInstruction(
        MNM_JSR_W, m_pInstrumentor);
    pNewInst->SetBranchTarget(pTarget);
    return pNewInst;
}

CMIEInstruction* CJIEInstructionCodec::CreateLoadStore(EMnemonic mnem, 
                                                       unsigned int uiVarInd)
{
    CMIEInstruction *pNewInst = CMIEInstructionFactory::CreateInstruction(
        mnem, m_pInstrumentor);
    SOperand op;
    op.type = OT_VAR_ID;
    op.val.varID = uiVarInd;
    pNewInst->InternalSetOperand(&op);
    return pNewInst;
}

CMIEInstruction* CJIEInstructionCodec::CreateRet(unsigned int uiVarInd)
{
    CMIEInstruction *pNewInst = CMIEInstructionFactory::CreateInstruction(
        MNM_RET, m_pInstrumentor);
    SOperand op;
    op.type = OT_VAR_ID;
    op.val.varID = uiVarInd;
    pNewInst->InternalSetOperand(&op);
    return pNewInst;
}

TResult CJIEInstructionCodec::AddCodeGenInstBefore(CFlowGraph *pFlowGraph,
                                                   CMIEInstruction *pMarkerInst,
                                                   CMIEInstruction *pNewInst)
{
    pNewInst->SetStatus(CMIEInstruction::IS_CODEGEN);
    TResult res;
    res = pFlowGraph->AddInstructionBefore(pMarkerInst->GetFlowGraphNode(), pNewInst);
    return res;
}

TResult CJIEInstructionCodec::AddCodeGenInstAfter(CFlowGraph *pFlowGraph,
                                                  CMIEInstruction *pMarkerInst,
                                                  CMIEInstruction *pNewInst)
{
    pNewInst->SetStatus(CMIEInstruction::IS_CODEGEN);
    TResult res;
    res = pFlowGraph->AddInstructionAfter(pMarkerInst->GetFlowGraphNode(), pNewInst);
    return res;
}

//
// Processes try..finally blocks in the flow-graph and adds the necessary code
// for implementing the "finally" logic.
// This is a special implementation of the BindTryFinallyBlock API for constructors.
// 
// **** IMPORTANT ***
// This implementation does not use JSR at all, and assumes the following:
// 1. The entire original code of the constructor is bound inside the "try..end_try" 
//    block. The result of using this function on an arbitrary ranges of instructions 
//    may lead to unexpected results. 
// 2. Only 1 try..finally block was defined on the constructor. The operation will
//    fail if multiple blocks were defined on the constructor's code.
// 3. The instructions between .FINALLY and .END_FINALLY are physically located
//    after the .END_TRY instruction in the flow-graph
// 4. The .FINALLY instruction appears immediately after .END_TRY
//
// A more robust implementation based on a technique used by IBM Probekit
// is currently pending legal confirmation and may be used in the future.
//
//TODO: reimplement this function differently when we solve all legal issues
//
TResult CJIEInstructionCodec::ProcessTryFinallyBlocksForCtors(CFlowGraph *pFlowGraph)
{
    RTUtil::MList<STryFinallyInfo*> blockList;
    BuildFinallyBlockList(&blockList, pFlowGraph);
    
    TMRTEHandle h = blockList.GetFirst();
    if (NULL == h)
    {
        // no finally blocks
        return MRTE_RESULT_OK;
    }
    assert(1 == blockList.Count());
    if (blockList.Count() != 1)
    {
        // only one try..finally block is supported for constructors at this time
        return MRTE_ERROR_FAIL;
    }

    STryFinallyInfo *pBlockInfo;
    pBlockInfo = blockList.GetData(h);

    // verify that the try..end_try block spans the entire original constructor code
    CMIEInstruction *pInst;
    pInst = pFlowGraph->GetSucceedingInstruction(pBlockInfo->pTry);
    if (!pInst->IsOriginalInst())
    {
        return MRTE_ERROR_FAIL;
    }
    pInst = pFlowGraph->GetPrecedingInstruction(pBlockInfo->pEndTry);
    if (!pInst->IsOriginalInst())
    {
        return MRTE_ERROR_FAIL;
    }
    pInst = pFlowGraph->GetPrecedingInstruction(pBlockInfo->pTry);
    if (!MNM_PSEUDO_START == pInst->GetMnemonic() &&
        pInst->IsOriginalInst())
    {
        return MRTE_ERROR_FAIL;
    }
    pInst = pFlowGraph->GetSucceedingInstruction(pBlockInfo->pEndTry);
    if (!MNM_PSEUDO_FINALLY == pInst->GetMnemonic() &&
        pInst->IsOriginalInst())
    {
        return MRTE_ERROR_FAIL;
    }

    // verify that the .END_FINALLY instruction is the last instruction in the flow-graph
    pInst = pFlowGraph->GetSucceedingInstruction(pBlockInfo->pEndFinally);
    if (MNM_PSEUDO_END != pInst->GetMnemonic())
    {
        return MRTE_ERROR_FAIL;
    }

    // verify that .FINALLY comes right after .END_TRY
    pInst = pFlowGraph->GetSucceedingInstruction(pBlockInfo->pEndTry);
    if (pInst != pBlockInfo->pFinally)
    {
        return MRTE_ERROR_FAIL;
    }

    // define local variables
/*
    unsigned int uiExVarInd;    // a local variable index to store the exception received
                                // in the "catch all" block
    CMIEInstruction *pStore;
    CMIEInstruction *pLoad;
*/
    CMIEInstruction *pAThrow;
    CMIEInstruction *pCatch;
    CMIEInstruction *pEndCatch;
    CMIEInstruction *pNewInst;
    CMIEInstruction *pInstToCopy;
    CMIEInstruction *pNop;

    CMIEInstruction *pCatchFirstInst;
    CMIEInstruction *pCatchLastInst;

    CCodeAttribute *pCodeAttr = m_pBCIMethod->GetCodeAttribute();
    assert(NULL != pCodeAttr);
    
    // process the first (and only) finally block
    CMIEInstruction *pFirstInstInFinally = pFlowGraph->GetSucceedingInstruction(
        pBlockInfo->pFinally);
    //
    // The "try..finally" block is implemented as follows:
    // - Add a "catch all" block after the "finally" block, copy the code from the
    //   "finally" block to the "catch all" block, and then use 'athrow' to re-throw
    //   the exception.
    // - Add a 'nop' at the end of the "try..end_try" block. Scan all instructions
    //   between .TRY and .END_TRY and replace all 'return' instructions with a 
    //   'goto' to the new 'nop' (e.g. to the end of the "try..end_try" block.
    // - Add a 'return' instruction at the end of the finally block 
    //   (before .END_FINALLY)
    // The result of this process is that the code in the "finally" block will be 
    // executed whether the code in the "try..end_try" block terminates normally or
    // abnormally
    //

    //
    // Create the "catch all" block
    //
    pCatch = CMIEInstructionFactory::CreateInstruction(MNM_PSEUDO_CATCH, 
        m_pInstrumentor);
    AddCodeGenInstAfter(pFlowGraph, pBlockInfo->pEndFinally, pCatch);
    pEndCatch = CMIEInstructionFactory::CreateInstruction(MNM_PSEUDO_END_CATCH, 
        m_pInstrumentor);
    AddCodeGenInstAfter(pFlowGraph, pCatch, pEndCatch);

    // back-up exception object in a local variable
/*
    uiExVarInd = pCodeAttr->GetMaxLocals();
    pCodeAttr->SetMaxLocals(uiExVarInd + 1);

    pStore = CreateLoadStore(MNM_ASTORE, uiExVarInd);
    AddCodeGenInstAfter(pFlowGraph, pCatch, pStore);
*/

    // Copy all "finally" block instructions:
    // First pass: copy instructions and create a source->target map
    // Second pass: for each cloned branch instruction - fix its target by
    // consulting the map from the first pass
    MHashNoSync<UIOP, UIOP> mapSourceToClone;
    mapSourceToClone.Init(false, 0, 100);
    pInstToCopy = pFlowGraph->GetSucceedingInstruction(pBlockInfo->pFinally);
    while (pInstToCopy != pBlockInfo->pEndFinally)
    {
        pNewInst = pInstToCopy->Clone();
        pNewInst->SetOriginalOffset(0);
        AddCodeGenInstBefore(pFlowGraph, pEndCatch, pNewInst);
        mapSourceToClone.Set((UIOP)pInstToCopy, (UIOP)pNewInst);
        pInstToCopy = pFlowGraph->GetSucceedingInstruction(pInstToCopy);
    }

    // Second pass
    CMIEInstruction *pSourceInst;
    CMIEInstruction *pSourceInstTarget; 
    CMIEInstruction *pClonedInst;
    CMIEInstruction *pClonedInstTarget;
    pSourceInst = pFlowGraph->GetSucceedingInstruction(pBlockInfo->pFinally);
    while (pSourceInst != pBlockInfo->pEndFinally)
    {
        if (pSourceInst->IsBranch())
        {
            pSourceInstTarget = (CMIEInstruction*)pSourceInst->GetBranchTarget();
            assert(pSourceInstTarget);
            pClonedInst = (CMIEInstruction*)mapSourceToClone.Get((UIOP)pSourceInst);
            assert(pClonedInst);
            pClonedInstTarget = (CMIEInstruction*)mapSourceToClone.Get((UIOP)pSourceInstTarget);
            assert(pClonedInstTarget);
            pClonedInst->SetBranchTarget(pClonedInstTarget);
        }
        pSourceInst = pFlowGraph->GetSucceedingInstruction(pSourceInst);
    }

/*
    // reload the exception on the stack
    pLoad = CreateLoadStore(MNM_ALOAD, uiExVarInd);
    AddCodeGenInstBefore(pFlowGraph, pEndCatch, pLoad);
*/

    // re-throw
    pAThrow = CMIEInstructionFactory::CreateInstruction(MNM_ATHROW, m_pInstrumentor);
    AddCodeGenInstBefore(pFlowGraph, pEndCatch, pAThrow);
    
    pCatchLastInst = pAThrow;
    pCatchFirstInst = pFlowGraph->GetSucceedingInstruction(pCatch);


    //
    // Modify the code of the protected block (between .TRY and .END_TRY):
    // 1. Add a 'nop' at the end of the protected block
    // 2. Replace all 'return' instructions with 'goto' to the 'nop'
    //

    pNop = CMIEInstructionFactory::CreateInstruction(MNM_NOP, m_pInstrumentor);
    AddCodeGenInstBefore(pFlowGraph, pBlockInfo->pEndTry, pNop);
    CMIEInstruction *pEndTryRealInst = pNop;

    EMnemonic returnMnem = MNM_RETURN;

    pInst = pBlockInfo->pTry;
    while (pInst != pBlockInfo->pEndTry)
    {
        if (CMIEInstruction::SEM_RET == pInst->GetSemantics())
        {
            returnMnem = pInst->GetMnemonic();
            // change the 'return' to 'goto .FINALLY'
            TResult res;
            res = pInst->SetMnemonic(MNM_GOTO_W);
            CHECK_TRESULT(res);
            IInstruction *pGotoTarget;
            pGotoTarget = pInst->SetBranchTarget(pNop);
            CHECK_NULL_RETURN_ERROR(pGotoTarget);
        }
        pInst = pFlowGraph->GetSucceedingInstruction(pInst);
    }

    // Add 'return' at the end of the "finally" block
    pInst = CMIEInstructionFactory::CreateInstruction(returnMnem, m_pInstrumentor);
    AddCodeGenInstBefore(pFlowGraph, pBlockInfo->pEndFinally, pInst);

    //
    // Modify the operands of the .TRY, END_TRY, .CATCH, .END_CATCH instructions
    // and associate them with each other.
    // This is needed for the encoder to generate the correct entries in the 
    // exception table.
    //
    SOperand opCatch;
    opCatch.type = OT_TARGET;
    opCatch.val.pTarget = pCatch;

    SOperand opEndCatch;
    opEndCatch.type = OT_TARGET;
    opEndCatch.val.pTarget = pEndCatch;

    SOperand opTry;
    opTry.type = OT_TARGET;
    opTry.val.pTarget = pBlockInfo->pTry;

    SOperand opHandlerType;
    opHandlerType.type = OT_JAVA_CP_INDEX;
    opHandlerType.val.cpIndex = 0;

    SOperand opCatchRealInst;
    opCatchRealInst.type = OT_TARGET;
    opCatchRealInst.val.pTarget = pCatchFirstInst;
    
    SOperand opEndCatchRealInst;
    opEndCatchRealInst.type = OT_TARGET;
    opEndCatchRealInst.val.pTarget = pCatchLastInst;

    SOperand opEndTryRealInst;
    opEndTryRealInst.type = OT_TARGET;
    opEndTryRealInst.val.pTarget = pEndTryRealInst;

    pBlockInfo->pTry->InternalSetOperand(2, &opCatch);
    pBlockInfo->pEndTry->InternalSetOperand(2, &opCatch);
    pBlockInfo->pEndTry->InternalSetOperand((unsigned int)0, &opEndTryRealInst);
    pCatch->InternalSetOperand(&opCatchRealInst, &opEndCatch, &opTry, &opHandlerType);
    pEndCatch->InternalSetOperand(&opEndCatchRealInst, &opCatch);

    RTUtil::FreeListItems(blockList);
    return MRTE_RESULT_OK;
}

//
// Processes try..finally blocks in the flow-graph and adds the necessary code
// for implementing the "finally" logic.
// 
// This implementation is designed to support the new Java 6 verifier. 
// It does use JSR and RET instruction which are not allowed in Java 6 class files
//
// The following is supported:
//
// - Nested try..finally blocks
// - Multiple try..finally blocks
//
// **** IMPORTANT ***
// This implementation relies on the following assumptions:
//
// - try..finally blocks are constructed correctly (according the the Java language semantics)
// - Execution inside a protected block cannot jump outside the protected block, except for 
//   normal completion (return or fall-through) or an exception.
// - The instructions between .FINALLY and .END_FINALLY are physically located
//   after the .END_TRY instruction in the flow-graph. In other words, the .FINALLY
//   instruction must follow the .END_TRY instruction of the protected block.
//
TResult CJIEInstructionCodec::ProcessTryFinallyBlocksNoJsr(CFlowGraph *pFlowGraph)
{
    RTUtil::MList<STryFinallyInfo*> blockList;
    BuildFinallyBlockList(&blockList, pFlowGraph);
    
    TMRTEHandle h = blockList.GetFirst();
    if (NULL == h)
    {
        // No finally blocks
        return MRTE_RESULT_OK;
    }

    STryFinallyInfo *pBlockInfo;
    CMIEInstruction *pInst;

    CMIEInstruction *pAThrow;
    CMIEInstruction *pCatch;
    CMIEInstruction *pEndCatch;
    CMIEInstruction *pNewInst;
    CMIEInstruction *pInstToCopy;

    CMIEInstruction *pCatchFirstInst;
    CMIEInstruction *pCatchLastInst;

    CCodeAttribute *pCodeAttr = m_pBCIMethod->GetCodeAttribute();
    assert(NULL != pCodeAttr);
    
    // Process the finally blocks from innermost to outermost
    while (NULL != h)
    {
        pBlockInfo = blockList.GetData(h);
        h = blockList.GetNext(h);

        // Verify that .FINALLY comes right after .END_TRY
        pInst = pFlowGraph->GetSucceedingInstruction(pBlockInfo->pEndTry);
        if (pInst != pBlockInfo->pFinally)
        {
            return MRTE_ERROR_FAIL;
        }

        // The "try..finally" block is implemented as follows:
        //
        // - Add a "catch all" block after the "finally" block, copy the code from the
        //   "finally" block to the "catch all" block, and then use 'athrow' to re-throw
        //   the exception.
        // - Scan all instructions between .TRY and .END_TRY and replace all 'return' 
        //   instructions with a 'goto' to the first instruction of the "finally" block
        //   (after .FINALLY)
        // - Add a 'return' instruction at the end of the finally block 
        //   (before .END_FINALLY)
        //
        // The result of this process: is that the code in the "finally" block will be 
        // executed whether the code in the "try..end_try" block terminates normally or
        // abnormally

        CMIEInstruction *pFirstInstInFinally = pFlowGraph->GetSucceedingInstruction(
            pBlockInfo->pFinally);

        // Create the "catch all" block
        pCatch = CMIEInstructionFactory::CreateInstruction(MNM_PSEUDO_CATCH, 
            m_pInstrumentor);
        AddCodeGenInstAfter(pFlowGraph, pBlockInfo->pEndFinally, pCatch);
        pEndCatch = CMIEInstructionFactory::CreateInstruction(MNM_PSEUDO_END_CATCH, 
            m_pInstrumentor);
        AddCodeGenInstAfter(pFlowGraph, pCatch, pEndCatch);

        // Copy all "finally" block instructions:
        // 
        // First pass: copy instructions and create a source->target map
        // Second pass: for each cloned branch instruction - fix its target by
        // consulting the map from the first pass
        MHashNoSync<UIOP, UIOP> mapSourceToClone;
        mapSourceToClone.Init(false, 0, 100);
        pInstToCopy = pFlowGraph->GetSucceedingInstruction(pBlockInfo->pFinally);
        while (pInstToCopy != pBlockInfo->pEndFinally)
        {
            pNewInst = pInstToCopy->Clone();
            pNewInst->SetOriginalOffset(0);
            AddCodeGenInstBefore(pFlowGraph, pEndCatch, pNewInst);
            mapSourceToClone.Set((UIOP)pInstToCopy, (UIOP)pNewInst);
            pInstToCopy = pFlowGraph->GetSucceedingInstruction(pInstToCopy);
        }

        // Second pass
        CMIEInstruction *pSourceInst;
        CMIEInstruction *pSourceInstTarget; 
        CMIEInstruction *pClonedInst;
        CMIEInstruction *pClonedInstTarget;
        pSourceInst = pFlowGraph->GetSucceedingInstruction(pBlockInfo->pFinally);
        while (pSourceInst != pBlockInfo->pEndFinally)
        {
            if (pSourceInst->IsBranch())
            {
                pSourceInstTarget = (CMIEInstruction*)pSourceInst->GetBranchTarget();
                assert(pSourceInstTarget);
                pClonedInst = (CMIEInstruction*)mapSourceToClone.Get((UIOP)pSourceInst);
                assert(pClonedInst);
                pClonedInstTarget = (CMIEInstruction*)mapSourceToClone.Get((UIOP)pSourceInstTarget);
                assert(pClonedInstTarget);
                pClonedInst->SetBranchTarget(pClonedInstTarget);
            }
            pSourceInst = pFlowGraph->GetSucceedingInstruction(pSourceInst);
        }

        // re-throw
        pAThrow = CMIEInstructionFactory::CreateInstruction(MNM_ATHROW, m_pInstrumentor);
        AddCodeGenInstBefore(pFlowGraph, pEndCatch, pAThrow);
        
        pCatchLastInst = pAThrow;
        pCatchFirstInst = pFlowGraph->GetSucceedingInstruction(pCatch);


        // Modify the code of the protected block (between .TRY and .END_TRY):
        // 1. Remove the .END_TRY instruction (change to 'nop')
        // 2. Replace all 'return' instructions with 'goto' to the first instruction
        //    in the "finally" block
        // 3. Add .END_TRY at the end of the "finally" block
        // 4. Associate .TRY with the new .END_TRY
        //
        // Note: we assume that all 'return' instructions use the same opcode, so we capture
        //       this opcode so we'll know which 'return' to use at the end fo the "finally"
        //       block.

        pBlockInfo->pEndTry->InternalSetMnemonic(MNM_PSEUDO_ILLEGAL);
        pInst = pBlockInfo->pTry;
        EMnemonic mnmReturn = MNM_RETURN;

        while (pInst != pBlockInfo->pEndTry)
        {
            if (CMIEInstruction::SEM_RET == pInst->GetSemantics())
            {
                mnmReturn = pInst->GetMnemonic();
                // Change the 'return' to 'goto .FINALLY'
                TResult res;
                res = pInst->SetMnemonic(MNM_GOTO_W);
                CHECK_TRESULT(res);
                IInstruction *pGotoTarget;
                pGotoTarget = pInst->SetBranchTarget(pFirstInstInFinally);
                CHECK_NULL_RETURN_ERROR(pGotoTarget);
            }
            pInst = pFlowGraph->GetSucceedingInstruction(pInst);
        }

        CMIEInstruction *pEndTryRealInst = pFlowGraph->GetPrecedingInstruction(
            pBlockInfo->pEndFinally);
        CMIEInstruction *pNewEndTry = CMIEInstructionFactory::CreateInstruction(
            MNM_PSEUDO_END_TRY, m_pInstrumentor);
        AddCodeGenInstBefore(pFlowGraph, pBlockInfo->pEndFinally, pNewEndTry);

        // Associate the new .END_TRY with the .TRY
        SOperand opTry;
        opTry.type = OT_TARGET;
        opTry.val.pTarget = pBlockInfo->pTry;
        SOperand opEndTry;
        opEndTry.type = OT_TARGET;
        opEndTry.val.pTarget = pNewEndTry;

        pBlockInfo->pTry->InternalSetOperand(1, &opEndTry);
        pNewEndTry->InternalSetOperand(1, &opTry);
        pBlockInfo->pEndTry = pNewEndTry;

        // Add 'return' at the end of the "finally" block.
        // Note: we use the same opcode of the last return instruction used inside the
        // protected block

        pInst = CMIEInstructionFactory::CreateInstruction(mnmReturn, m_pInstrumentor);
        AddCodeGenInstBefore(pFlowGraph, pBlockInfo->pEndFinally, pInst);

        // Modify the operands of the .TRY, END_TRY, .CATCH, .END_CATCH instructions
        // and associate them with each other.
        // This is needed for the encoder to generate the correct entries in the 
        // exception table.

        SOperand opCatch;
        opCatch.type = OT_TARGET;
        opCatch.val.pTarget = pCatch;

        SOperand opEndCatch;
        opEndCatch.type = OT_TARGET;
        opEndCatch.val.pTarget = pEndCatch;

        SOperand opHandlerType;
        opHandlerType.type = OT_JAVA_CP_INDEX;
        opHandlerType.val.cpIndex = 0;

        SOperand opCatchRealInst;
        opCatchRealInst.type = OT_TARGET;
        opCatchRealInst.val.pTarget = pCatchFirstInst;
        
        SOperand opEndCatchRealInst;
        opEndCatchRealInst.type = OT_TARGET;
        opEndCatchRealInst.val.pTarget = pCatchLastInst;

        SOperand opEndTryRealInst;
        opEndTryRealInst.type = OT_TARGET;
        opEndTryRealInst.val.pTarget = pEndTryRealInst;

        pBlockInfo->pTry->InternalSetOperand(2, &opCatch);
        pBlockInfo->pEndTry->InternalSetOperand(2, &opCatch);
        pBlockInfo->pEndTry->InternalSetOperand((unsigned int)0, &opEndTryRealInst);
        pCatch->InternalSetOperand(&opCatchRealInst, &opEndCatch, &opTry, &opHandlerType);
        pEndCatch->InternalSetOperand(&opEndCatchRealInst, &opCatch);
    }

    RTUtil::FreeListItems(blockList);

    return MRTE_RESULT_OK;
}

#if 0
//
// Processes try..finally blocks in the flow-graph and adds the necessary code
// for implementing the "finally" logic.
// This is a special implementation of the BindTryFinallyBlock API for constructors.
// 
// **** IMPORTANT ***
// This implementation does not use JSR at all, and assumes the following:
// 1. The entire original code of the constructor is bound inside the "try..end_try" 
//    block. The result of using this function on an arbitrary ranges of instructions 
//    may lead to unexpected results. 
// 2. Only 1 try..finally block was defined on the constructor. The operation will
//    fail if multiple blocks were defined on the constructor's code.
// 3. The instructions between .FINALLY and .END_FINALLY are physically located
//    after the .END_TRY instruction in the flow-graph
// 4. The .FINALLY instruction appears immediately after .END_TRY
//
// A more robust implementation based on a technique used by IBM Probekit
// is currently pending legal confirmation and may be used in the future.
//
//TODO: reimplement this function differently when we solve all legal issues
//
TResult CJIEInstructionCodec::ProcessTryFinallyBlocksForCtors(CFlowGraph *pFlowGraph)
{
    RTUtil::MList<STryFinallyInfo*> blockList;
    BuildFinallyBlockList(&blockList, pFlowGraph);
    
    TMRTEHandle h = blockList.GetFirst();
    if (NULL == h)
    {
        // no finally blocks
        return MRTE_RESULT_OK;
    }
    assert(1 == blockList.Count());
    if (blockList.Count() != 1)
    {
        // only one try..finally block is supported for constructors at this time
        return MRTE_ERROR_FAIL;
    }

    STryFinallyInfo *pBlockInfo;
    CMIEInstruction *pInst;
/*
    pBlockInfo = blockList.GetData(h);

    // verify that the try..end_try block spans the entire original constructor code
    pInst = pFlowGraph->GetSucceedingInstruction(pBlockInfo->pTry);
    if (!pInst->IsOriginalInst())
    {
        return MRTE_ERROR_FAIL;
    }
    pInst = pFlowGraph->GetPrecedingInstruction(pBlockInfo->pEndTry);
    if (!pInst->IsOriginalInst())
    {
        return MRTE_ERROR_FAIL;
    }
    pInst = pFlowGraph->GetPrecedingInstruction(pBlockInfo->pTry);
    if (!MNM_PSEUDO_START == pInst->GetMnemonic() &&
        pInst->IsOriginalInst())
    {
        return MRTE_ERROR_FAIL;
    }
    pInst = pFlowGraph->GetSucceedingInstruction(pBlockInfo->pEndTry);
    if (!MNM_PSEUDO_FINALLY == pInst->GetMnemonic() &&
        pInst->IsOriginalInst())
    {
        return MRTE_ERROR_FAIL;
    }

    // verify that the .END_FINALLY instruction is the last instruction in the flow-graph
    pInst = pFlowGraph->GetSucceedingInstruction(pBlockInfo->pEndFinally);
    if (MNM_PSEUDO_END != pInst->GetMnemonic())
    {
        return MRTE_ERROR_FAIL;
    }

    // verify that .FINALLY comes right after .END_TRY
    pInst = pFlowGraph->GetSucceedingInstruction(pBlockInfo->pEndTry);
    if (pInst != pBlockInfo->pFinally)
    {
        return MRTE_ERROR_FAIL;
    }
*/

    // define local variables
    unsigned int uiExVarInd;    // a local variable index to store the exception received
                                // in the "catch all" block
    CMIEInstruction *pStore;
    CMIEInstruction *pLoad;
    CMIEInstruction *pAThrow;
    CMIEInstruction *pCatch;
    CMIEInstruction *pEndCatch;
    CMIEInstruction *pNewInst;
    CMIEInstruction *pInstToCopy;
    CMIEInstruction *pNop;

    CMIEInstruction *pCatchFirstInst;
    CMIEInstruction *pCatchLastInst;

    CCodeAttribute *pCodeAttr = m_pBCIMethod->GetCodeAttribute();
    assert(NULL != pCodeAttr);
    
    // process the finally blocks from innermost to outermost
    while (NULL != h)
    {
        pBlockInfo = blockList.GetData(h);
        h = blockList.GetNext(h);
        CMIEInstruction *pFirstInstInFinally = pFlowGraph->GetSucceedingInstruction(
            pBlockInfo->pFinally);
        //
        //TODO: explain algorithm here!
        //

        //
        // Create the "catch all" block
        //
        pCatch = CMIEInstructionFactory::CreateInstruction(MNM_PSEUDO_CATCH, 
            m_pInstrumentor);
        AddCodeGenInstAfter(pFlowGraph, pBlockInfo->pEndFinally, pCatch);
        pEndCatch = CMIEInstructionFactory::CreateInstruction(MNM_PSEUDO_END_CATCH, 
            m_pInstrumentor);
        AddCodeGenInstAfter(pFlowGraph, pCatch, pEndCatch);

/*
        // back-up exception object in a local variable
        uiExVarInd = pCodeAttr->GetMaxLocals();
        pCodeAttr->SetMaxLocals(uiExVarInd + 1);

        pStore = CreateLoadStore(MNM_ASTORE, uiExVarInd);
        AddCodeGenInstAfter(pFlowGraph, pCatch, pStore);
*/

        // copy all "finally" block instructions
        pInstToCopy = pFlowGraph->GetSucceedingInstruction(pBlockInfo->pFinally);
        while (pInstToCopy != pBlockInfo->pEndFinally)
        {
            pNewInst = pInstToCopy->Clone();
            pNewInst->SetOriginalOffset(0);
            AddCodeGenInstBefore(pFlowGraph, pEndCatch, pNewInst);
            pInstToCopy = pFlowGraph->GetSucceedingInstruction(pInstToCopy);
        }

/*
        // reload the exception on the stack
        pLoad = CreateLoadStore(MNM_ALOAD, uiExVarInd);
        AddCodeGenInstBefore(pFlowGraph, pEndCatch, pLoad);
*/

        // re-throw
        pAThrow = CMIEInstructionFactory::CreateInstruction(MNM_ATHROW, m_pInstrumentor);
        AddCodeGenInstBefore(pFlowGraph, pEndCatch, pAThrow);
        
        pCatchLastInst = pAThrow;
        pCatchFirstInst = pFlowGraph->GetSucceedingInstruction(pCatch);


        //
        // Modify the code of the protected block (between .TRY and .END_TRY):
        // 1. Remove the .END_TRY instruction (change to 'nop')
        // 2. Replace all 'return' instructions with 'goto' to the first instruction
        //    in the "finally" block
        // 3. Add .END_TRY at the end of the "finally" block
        // 4. Associate .TRY with the new .END_TRY
        //

        pBlockInfo->pEndTry->InternalSetMnemonic(MNM_NOP);
        pInst = pBlockInfo->pTry;
        while (pInst != pBlockInfo->pEndTry)
        {
            if (CMIEInstruction::SEM_RET == pInst->GetSemantics())
            {
                assert(MNM_RETURN == pInst->GetMnemonic()); // constructors always return 'void'
                // change the 'return' to 'goto .FINALLY'
                TResult res;
                res = pInst->SetMnemonic(MNM_GOTO_W);
                CHECK_TRESULT(res);
                IInstruction *pGotoTarget;
                pGotoTarget = pInst->SetBranchTarget(pFirstInstInFinally);
                CHECK_NULL_RETURN_ERROR(pGotoTarget);
            }
            pInst = pFlowGraph->GetSucceedingInstruction(pInst);
        }

        CMIEInstruction *pEndTryRealInst = pFlowGraph->GetPrecedingInstruction(
            pBlockInfo->pEndFinally);
        CMIEInstruction *pNewEndTry = CMIEInstructionFactory::CreateInstruction(
            MNM_PSEUDO_END_TRY, m_pInstrumentor);
        AddCodeGenInstBefore(pFlowGraph, pBlockInfo->pEndFinally, pNewEndTry);

        // associate the new .END_TRY with the .TRY
        SOperand opTry;
        opTry.type = OT_TARGET;
        opTry.val.pTarget = pBlockInfo->pTry;
        SOperand opEndTry;
        opEndTry.type = OT_TARGET;
        opEndTry.val.pTarget = pNewEndTry;

        pBlockInfo->pTry->InternalSetOperand(1, &opEndTry);
        pNewEndTry->InternalSetOperand(1, &opTry);
        pBlockInfo->pEndTry = pNewEndTry;

        //
        // Add 'return' at the end of the "finally" block
        // Note: we can safely use the regular 'return' opcode since constructors
        // always return 'void'
        //
        pInst = CMIEInstructionFactory::CreateInstruction(MNM_RETURN, m_pInstrumentor);
        AddCodeGenInstBefore(pFlowGraph, pBlockInfo->pEndFinally, pInst);

        //TODO: cleanup: remove the .FINALLY and .END_FINALLY instructions from the flow-graph

        //
        // Modify the operands of the .TRY, END_TRY, .CATCH, .END_CATCH instructions
        // and associate them with each other.
        // This is needed for the encoder to generate the correct entries in the 
        // exception table.
        //
        SOperand opCatch;
        opCatch.type = OT_TARGET;
        opCatch.val.pTarget = pCatch;

        SOperand opEndCatch;
        opEndCatch.type = OT_TARGET;
        opEndCatch.val.pTarget = pEndCatch;

        SOperand opHandlerType;
        opHandlerType.type = OT_JAVA_CP_INDEX;
        opHandlerType.val.cpIndex = 0;

        SOperand opCatchRealInst;
        opCatchRealInst.type = OT_TARGET;
        opCatchRealInst.val.pTarget = pCatchFirstInst;
        
        SOperand opEndCatchRealInst;
        opEndCatchRealInst.type = OT_TARGET;
        opEndCatchRealInst.val.pTarget = pCatchLastInst;

        SOperand opEndTryRealInst;
        opEndTryRealInst.type = OT_TARGET;
        opEndTryRealInst.val.pTarget = pEndTryRealInst;

        pBlockInfo->pTry->InternalSetOperand(2, &opCatch);
        pBlockInfo->pEndTry->InternalSetOperand(2, &opCatch);
        pBlockInfo->pEndTry->InternalSetOperand((unsigned int)0, &opEndTryRealInst);
        pCatch->InternalSetOperand(&opCatchRealInst, &opEndCatch, &opTry, &opHandlerType);
        pEndCatch->InternalSetOperand(&opEndCatchRealInst, &opCatch);


    }

    RTUtil::FreeListItems(blockList);

    return MRTE_RESULT_OK;
}
#endif

