/*****************************************************************************
 * Copyright (c) 1997, 2008 Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/
//
#pragma warning (disable:4786)

#include <stdlib.h>

#ifdef _DEBUG

    #include <crtdbg.h>

#endif

#include "MRTEInfrastructureDefinitions.h"
#include "JavaClass.h"
#include "JavaMethod.h"
#include "JavaField.h"
#include "JIEUtils.h"
#include "JStream.h"
#include "JMemStream.h"
#include "JVMInsSet.h"
#include "MIEInstructionFactory.h"
#include "BciUtils.h"

using namespace Martini::JIE;
using namespace Martini::MIE;
using namespace Martini::RTUtil;
using namespace Martini::Infrastructure;

//////////////////////////////////////////////////////////////////////////
// class CJavaClass implementation

static void* g_pJniEnv = NULL;

static void *
get_vm_pointer(void **pJniEnv) 
{
    *pJniEnv = g_pJniEnv;
    return g_pJniEnv;
}

// constructs a CJavaClass object
CJavaClass::CJavaClass(const unsigned char *pubClassFile, unsigned int uiClassSize)
    : m_pStaticInitializer(NULL), m_bModuleParsed(false), m_bMethodListCreated(false),
    m_bFieldListCreated(false), m_bShouldEmitStaticInitializer(false)
{
#ifdef _DEBUG
    //_CrtSetDbgFlag ( _CRTDBG_ALLOC_MEM_DF | _CRTDBG_LEAK_CHECK_DF );
    //_CrtSetBreakAlloc(2189);
#endif

    if (CJieGlobals::Instance()->isStackMapCalcEnabled)
    {
        TResult res = CJieGlobals::Instance()->pfnJPI_AttachCurrentThread(
            &m_pJniEnv, &m_isThreadAttached);
        if (MRTE_SUCCEEDED(res))
        {
            g_pJniEnv = m_pJniEnv;
            initialize_dynamic(get_vm_pointer);
        }
        else
        {
            CJieGlobals::Instance()->isStackMapCalcEnabled = false;
            m_isThreadAttached = false;
            m_pJniEnv = NULL;
        }
    }

	CJMemStream		memStream;					// Memory stream
	CJStream		jsIn(&memStream);		    // Java input stream
    // read class file
	memStream.Open(const_cast<unsigned char *>(pubClassFile), uiClassSize);
	m_pClassBuilder = new CJClassBuilder();
	m_pClassBuilder->Read(jsIn);
	m_pModule = new CModuleJ;
	// open the module and configure it to "own" the class builder object (hence 'true' in
    // the second argument). The class builder reference will be deleted when the module
    // is deleted.
	m_pModule->Open(m_pClassBuilder, true);
	m_pModule->SetAccessFlags(m_pClassBuilder->GetAccessFlags());
	m_pModule->UseStackMapCalculation(CJieGlobals::Instance()->isStackMapCalcEnabled);
}

// destructor
CJavaClass::~CJavaClass()
{
    // nothing to do here. Java class resources are freed when calling Free()
}

// deallocates resources associated with the JavaClass object
void CJavaClass::Free()
{
    // free memory allocated for the method objects in the method list
    TMRTEHandle h = m_lstJavaMethods.GetFirst();
    IJavaMethod *pJavaMethod;
    while (h != NULL)
    {
        pJavaMethod = m_lstJavaMethods.GetData(h);
        delete pJavaMethod;
        h = m_lstJavaMethods.GetNext(h);
    }

    // free memory allocated for the field objects in the field list
    h = m_lstJavaFields.GetFirst();
    IJavaField *pJavaField;
    while (h != NULL)
    {
        pJavaField = m_lstJavaFields.GetData(h);
        delete pJavaField;
        h = m_lstJavaFields.GetNext(h);
    }

    // delete the BCI "ModuleJ" object associated with the Java class.
    // this will free all resources, including the class builder object.
    delete m_pModule;

    // deallocate the instruction factory
    CMIEInstructionFactory::Free();

    // detach the current thread from the JVM (if needed)
    if (CJieGlobals::Instance()->isStackMapCalcEnabled && m_isThreadAttached)
    {
        CJieGlobals::Instance()->pfnJPI_DetachCurrentThread();
    }

    // deallocate the JavaClass object
    delete this;
}

TResult CJavaClass::GetName(SWideString *pClassName) const
{
    if (NULL == pClassName)
    {
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }
    if (NULL == pClassName->pStr)
    {
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }

    string strClassName = m_pClassBuilder->GetThisClassName();
    const char *szClassName = strClassName.c_str();
    size_t cbClassName = strlen(szClassName);
    pClassName->uiActualSize = cbClassName;

    // check if the buffer we got has enough space
    if (cbClassName > pClassName->uiBufferSize)
    {
        return MRTE_ERROR_BUFFER_TOO_SHORT;
    }

    CWideStringUtils::AssignText(*pClassName, szClassName);
    return MRTE_RESULT_OK;
}

TResult CJavaClass::GetInfo(SJavaClassInfo *pClassInfo) const
{
    if (NULL == pClassInfo)
    {
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }
    // get class name
    string strClassName = m_pClassBuilder->GetThisClassName();
    const char *szClassName = strClassName.c_str();
    pClassInfo->className.uiActualSize = strlen(szClassName);

  	// get super class string
	u2 u2SuperInd = m_pClassBuilder->GetSuperClass();
    CConstPool &cp = (*m_pClassBuilder->GetConstPool());
	string strSuperClass;
    CCPClassInfo *pCI;
    CCPUtf8Info *pUtf8;
	if(u2SuperInd > 0)
	{
		pCI = (CCPClassInfo*)cp[u2SuperInd];
		pUtf8 = (CCPUtf8Info*)cp[pCI->GetClassInd()];
		strSuperClass = string((CSTR)pUtf8->GetBytes(), pUtf8->GetLength());
	}
    const char *szSuperClass = strSuperClass.c_str();
    pClassInfo->superClassName.uiActualSize = strlen(szSuperClass);

    // get class source file
	CJAttribs* pattribs = m_pClassBuilder->GetAttribs();
	// Get known attributes
	string strSourceFile;
	for(CJAttribs::iterator iter = pattribs->begin(); iter < pattribs->end(); iter++)
	{
		if(*(*iter)->GetName() == "SourceFile")
		{
			CSourceFileAttribute* psfa = (CSourceFileAttribute*)*iter;
			CCPUtf8Info *putf8 = psfa->GetValue();
			strSourceFile = (string)*putf8;
            break;
		}
	}
    const char *szSourceFile = strSourceFile.c_str();
    pClassInfo->sourceFileName.uiActualSize = strlen(szSourceFile);

    // check if the buffers we got have enough space
    if (pClassInfo->className.uiActualSize > pClassInfo->className.uiBufferSize ||
        pClassInfo->superClassName.uiActualSize > pClassInfo->superClassName.uiBufferSize ||
        pClassInfo->sourceFileName.uiActualSize > pClassInfo->sourceFileName.uiBufferSize)
    {
        return MRTE_ERROR_BUFFER_TOO_SHORT;
    }

    // copy results to pClassInfo
    pClassInfo->uiAttributeFlags = m_pClassBuilder->GetAccessFlags();
    CWideStringUtils::AssignText(pClassInfo->className, szClassName);
    CWideStringUtils::AssignText(pClassInfo->superClassName, szSuperClass);
    CWideStringUtils::AssignText(pClassInfo->sourceFileName, szSourceFile);
    return MRTE_RESULT_OK;
}

TJavaMethodIterator* CJavaClass::GetMethodIterator()
{
    InitMethodList();
    return m_lstJavaMethods.GetListIterator();
}


unsigned int CJavaClass::GetNumberOfMethods() const
{
    // use the ClassBuilder member instead of the m_lstJavaMethods member
    // because populating the method list is a costly process
    unsigned int uiMethodCount = m_pClassBuilder->GetMethods()->size();
    return uiMethodCount;
}

TJavaFieldIterator* CJavaClass::GetFieldIterator()
{
    InitFieldList();
    return m_lstJavaFields.GetListIterator();
}


unsigned int CJavaClass::GetNumberOfFields() const
{
    // use the ClassBuilder member instead of the m_lstJavaFields member
    // because populating the field list is a costly process
    unsigned int uiFieldCount = m_pClassBuilder->GetFields()->size();
    return uiFieldCount;
}

IJavaField* CJavaClass::AddReferenceField(const SWideString& szName, 
                                          const SWideString& szSignature,
                                          int uiAttributeFlags)
{
    char *szFieldName = CWideStringUtils::WideStringToCString(szName);
    char *szFieldSignature = CWideStringUtils::WideStringToCString(szSignature);
    CJavaType jtFieldType(CJavaType::J_CLASS, 0, szFieldSignature);

    IJavaField *pJieField = InternalAddField(uiAttributeFlags, szFieldName, jtFieldType);
    
    // cleanup
    delete [] szFieldName;
    delete [] szFieldSignature;

    return pJieField;
}

IJavaField* CJavaClass::AddField(
    const SWideString& szName, EVarType type, 
    unsigned int uiAttributeFlags)
{
    char *szFieldName = CWideStringUtils::WideStringToCString(szName);
    CJavaType jtFieldType(VarTypeToJavaType[type]); // convert JIE field type to BCI type

    IJavaField *pJieField = InternalAddField(uiAttributeFlags, szFieldName, jtFieldType);
    
    // cleanup
    delete [] szFieldName;

    return pJieField;
}

IJavaField* CJavaClass::InternalAddField(u2 uiFlags, const char *szName, CJavaType fieldType)
{
	CFieldInfo* pfi = m_pClassBuilder->CreateField(uiFlags, szName, fieldType);
	CCPFieldrefInfo* pfri = m_pClassBuilder->CreateFieldRef(pfi);
    IJavaField *pJieField = new CJavaField(this, pfi, pfri);
    m_lstJavaFields.InsertAfter(NULL, pJieField);
    return pJieField;
}

TConstantPoolIndex CJavaClass::RegisterRecorderCallback(
    const SWideString& szClassName, 
    const SWideString& szMethodName,
    const SWideString& szMethodSig)
{
    //TODO: need to convert all wide strings to UTF8 (see commented code).
    //      this should be done after adding UTF8 support to BCI (class CCPUtf8Info)
/*
    size_t cbMethod = wcslen(szMethodName.pStr);
    size_t cbSignature = wcslen(szMethodSig.pStr);
    size_t cbClass = wcslen(szClassName.pStr);
    size_t cbBuf = (__max(cbMethod, __max(cbSignature, cbClass)) + 1) * 2;
    unsigned char *szBuf = new unsigned char[cbBuf];
    // convert method name to UTF-8
    U16 u16Size = WCharToUtf8(szBuf, cbBuf, szMethodName.pStr);
    char *szMethod = new char[u16Size];
    strncpy(szMethod, reinterpret_cast<char*>(szBuf), u16Size);
    // convert method signature to UTF-8
    u16Size = WCharToUtf8(szBuf, cbBuf, szMethodSig.pStr);
    char *szSignature = new char[u16Size];
    strncpy(szSignature, reinterpret_cast<char*>(szBuf), u16Size);
    // convert class name to UTF-8
    u16Size = WCharToUtf8(szBuf, cbBuf, szClassName.pStr);
    char *szClass = new char[u16Size];
    strncpy(szClass, reinterpret_cast<char*>(szBuf), u16Size);
*/
    char *szMethod = CWideStringUtils::WideStringToCString(szMethodName);
    char *szSignature = CWideStringUtils::WideStringToCString(szMethodSig);
    char *szClass = CWideStringUtils::WideStringToCString(szClassName);
    // add a new method ref to the Java class
    TConstantPoolIndex cpIndex;
    //TODO: remove exception handling
    try
    {
        CCPMethodrefInfo *pMethInfo = m_pClassBuilder->CreateExtMethodRef(szMethod, 
            szSignature, szClass);
        cpIndex = pMethInfo->GetCpIndex();
    }
    catch (...)
    {
        cpIndex = ILLEGAL_CP_ENTRY;
    }

    // cleanup
/*
    delete szBuf;
*/
    delete [] szMethod;
    delete [] szSignature;
    delete [] szClass;
    return cpIndex;
}

TResult CJavaClass::GetValueFromCPEntry(
    TConstantPoolIndex cpIndex, 
    SConstantPoolEntryValue* pValue) const
{
    //TODO: remove exceptions
    if (NULL == pValue)
    {
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }
    CConstPool &cp = *(m_pClassBuilder->GetConstPool());
    // verify cpIndex is a valid constant pool index
    CCPInfo *pCPInfo;
    try
    {
        pCPInfo = cp[cpIndex];
    }
    catch (...)
    {
        // invalid constant pool index
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }

    CCPUtf8Info *pUtf8Info;
    TResult res;
    CCPFieldrefInfo *pFieldInfo;
    CCPClassInfo *pClassInfo;
    CCPNameAndTypeInfo *pNameAndTypeInfo;
    CCPUtf8Info *pUtf8ClassName;
    CCPUtf8Info *pUtf8MemberName;
    CCPUtf8Info *pUtf8MemberSig;
    CCPMethodrefInfo *pMethodInfo;
    CCPInterfaceMethodrefInfo *pInterfaceMethodInfo;


    switch (pValue->type) {
    case CPT_INTEGER:
        pValue->u.iVal = ((CCPIntegerInfo*)pCPInfo)->GetBytes();
    	break;
    case CPT_FLOAT:
        pValue->u.fVal = ((CCPFloatInfo*)pCPInfo)->GetFloat();
    	break;
    case CPT_DOUBLE:
        pValue->u.dVal = ((CCPDoubleInfo*)pCPInfo)->GetDouble();
    	break;
    case CPT_LONG:
        pValue->u.s64Val = ((CCPLongInfo*)pCPInfo)->GetLong();
    	break;
    case CPT_STRING:
        try
        {
            pUtf8Info = (CCPUtf8Info*)cp[((CCPStringInfo*)pCPInfo)->GetStringInd()];
        }
        catch (...)
        {
            return MRTE_ERROR_ILLEGAL_ARGUMENT;
        }
        res = CBciUtils::CCPUtf8InfoToWideString(pValue->u.strVal, pUtf8Info);
        if (res != MRTE_RESULT_OK)
        {
            return res;
        }
        break;
    case CPT_CLASS:
        try
        {
            pUtf8Info = (CCPUtf8Info*)cp[((CCPClassInfo*)pCPInfo)->GetClassInd()];
        }
        catch (...)
        {
            return MRTE_ERROR_ILLEGAL_ARGUMENT;
        }
        res = CBciUtils::CCPUtf8InfoToWideString(pValue->u.className, pUtf8Info);
        if (res != MRTE_RESULT_OK)
        {
            return res;
        }
        break;
    case CPT_FIELD:
        try
        {
            pFieldInfo = (CCPFieldrefInfo*)pCPInfo;
            // extract class name
            pClassInfo = (CCPClassInfo*)cp[pFieldInfo->GetClassInd()];
            pUtf8ClassName = (CCPUtf8Info*)cp[pClassInfo->GetClassInd()];
            // extract name and signature
            pNameAndTypeInfo = (CCPNameAndTypeInfo*)cp[pFieldInfo->GetNameAndTypeInd()];
            pUtf8MemberName = (CCPUtf8Info*)cp[pNameAndTypeInfo->GetNameInd()];
            pUtf8MemberSig = (CCPUtf8Info*)cp[pNameAndTypeInfo->GetDescriptorInd()];
        }
        catch (...)
        {
            return MRTE_ERROR_ILLEGAL_ARGUMENT;
        }
        res = CBciUtils::CCPUtf8InfoToWideString(pValue->u.fieldVal.className, pUtf8ClassName);
        if (res != MRTE_RESULT_OK)
        {
            return res;
        }
        res = CBciUtils::CCPUtf8InfoToWideString(pValue->u.fieldVal.name, pUtf8MemberName);
        if (res != MRTE_RESULT_OK)
        {
            return res;
        }
        res = CBciUtils::CCPUtf8InfoToWideString(pValue->u.fieldVal.signature, pUtf8MemberSig);
        if (res != MRTE_RESULT_OK)
        {
            return res;
        }
        break;
    case CPT_METHOD:
        try
        {
            pMethodInfo = (CCPMethodrefInfo*)pCPInfo;
            // extract class name
            pClassInfo = (CCPClassInfo*)cp[pMethodInfo->GetClassInd()];
            pUtf8ClassName = (CCPUtf8Info*)cp[pClassInfo->GetClassInd()];
            // extract name and signature
            pNameAndTypeInfo = (CCPNameAndTypeInfo*)cp[pMethodInfo->GetNameAndTypeInd()];
            pUtf8MemberName = (CCPUtf8Info*)cp[pNameAndTypeInfo->GetNameInd()];
            pUtf8MemberSig = (CCPUtf8Info*)cp[pNameAndTypeInfo->GetDescriptorInd()];
        }
        catch (...)
        {
            return MRTE_ERROR_ILLEGAL_ARGUMENT;
        }
        res = CBciUtils::CCPUtf8InfoToWideString(pValue->u.methodVal.className, pUtf8ClassName);
        if (res != MRTE_RESULT_OK)
        {
            return res;
        }
        res = CBciUtils::CCPUtf8InfoToWideString(pValue->u.methodVal.name, pUtf8MemberName);
        if (res != MRTE_RESULT_OK)
        {
            return res;
        }
        res = CBciUtils::CCPUtf8InfoToWideString(pValue->u.methodVal.signature, pUtf8MemberSig);
        if (res != MRTE_RESULT_OK)
        {
            return res;
        }
        break;
    case CPT_INTERFACE_METHOD:
        try
        {
            pInterfaceMethodInfo = (CCPInterfaceMethodrefInfo*)pCPInfo;
            // extract class name
            pClassInfo = (CCPClassInfo*)cp[pInterfaceMethodInfo->GetClassInd()];
            pUtf8ClassName = (CCPUtf8Info*)cp[pClassInfo->GetClassInd()];
            // extract name and signature
            pNameAndTypeInfo = (CCPNameAndTypeInfo*)cp[pInterfaceMethodInfo->GetNameAndTypeInd()];
            pUtf8MemberName = (CCPUtf8Info*)cp[pNameAndTypeInfo->GetNameInd()];
            pUtf8MemberSig = (CCPUtf8Info*)cp[pNameAndTypeInfo->GetDescriptorInd()];
        }
        catch (...)
        {
            return MRTE_ERROR_ILLEGAL_ARGUMENT;
        }
        res = CBciUtils::CCPUtf8InfoToWideString(pValue->u.interfaceMethodVal.className, 
            pUtf8ClassName);
        if (res != MRTE_RESULT_OK)
        {
            return res;
        }
        res = CBciUtils::CCPUtf8InfoToWideString(pValue->u.interfaceMethodVal.name, 
            pUtf8MemberName);
        if (res != MRTE_RESULT_OK)
        {
            return res;
        }
        res = CBciUtils::CCPUtf8InfoToWideString(pValue->u.interfaceMethodVal.signature, 
            pUtf8MemberSig);
        if (res != MRTE_RESULT_OK)
        {
            return res;
        }
        break;
    default:
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }
    return MRTE_RESULT_OK;
}

TConstantPoolIndex CJavaClass::AddEntryToCP(const SConstantPoolEntryValue& value)
{
    TConstantPoolIndex cpiResult;
    CCPInfo *pCPInfo;
    char *szString;
    char *szClass;
    char *szSignature;
    CJavaType jtFieldType;

    switch (value.type) {
    case CPT_INTEGER:
        // create an Integer constant
        pCPInfo = m_pClassBuilder->CreateIntegerConstant(value.u.iVal);
        if (NULL == pCPInfo)
        {
            cpiResult = ILLEGAL_CP_ENTRY;
        }
        else
        {
            cpiResult = pCPInfo->GetCpIndex();
        }
    	break;
    case CPT_LONG:
        // create a Long constant
        pCPInfo = m_pClassBuilder->CreateLongConstant(value.u.s64Val);
        if (NULL == pCPInfo)
        {
            cpiResult = ILLEGAL_CP_ENTRY;
        }
        else
        {
            cpiResult = pCPInfo->GetCpIndex();
        }
    	break;
    case CPT_DOUBLE:
        // create a Double constant
        pCPInfo = m_pClassBuilder->CreateDoubleConstant(value.u.dVal);
        if (NULL == pCPInfo)
        {
            cpiResult = ILLEGAL_CP_ENTRY;
        }
        else
        {
            cpiResult = pCPInfo->GetCpIndex();
        }
    	break;
    case CPT_FLOAT:
        // create a Float constant
        pCPInfo = m_pClassBuilder->CreateFloatConstant(value.u.fVal);
        if (NULL == pCPInfo)
        {
            cpiResult = ILLEGAL_CP_ENTRY;
        }
        else
        {
            cpiResult = pCPInfo->GetCpIndex();
        }
    	break;
    case CPT_STRING:
        // create a String constant
        szString = CWideStringUtils::WideStringToCString(value.u.strVal); //TODO: convert to UTF-8
        pCPInfo = m_pClassBuilder->CreateStringConstant(szString); 
        if (NULL == pCPInfo)
        {
            cpiResult = ILLEGAL_CP_ENTRY;
        }
        else
        {
            cpiResult = pCPInfo->GetCpIndex();
        }
        delete [] szString;
    	break;
    case CPT_CLASS:
        // create a Class constant
        szClass = CWideStringUtils::WideStringToCString(value.u.className); //TODO: convert to UTF-8
        pCPInfo = CreateClassConstant(szClass);
        if (NULL == pCPInfo)
        {
            cpiResult = ILLEGAL_CP_ENTRY;
        }
        else
        {
            cpiResult = pCPInfo->GetCpIndex();
        }
        delete [] szClass;
    	break;
    case CPT_FIELD:
        // create a Field constant
        szString = CWideStringUtils::WideStringToCString(value.u.fieldVal.name);
        szClass = CWideStringUtils::WideStringToCString(value.u.fieldVal.className);
        szSignature = CWideStringUtils::WideStringToCString(value.u.fieldVal.signature);
        jtFieldType.Parse(szSignature);
        pCPInfo = m_pClassBuilder->CreateExtFieldRef(szString, jtFieldType , szClass);
        if (NULL == pCPInfo)
        {
            cpiResult = ILLEGAL_CP_ENTRY;
        }
        else
        {
            cpiResult = pCPInfo->GetCpIndex();
        }
        delete [] szString;
        delete [] szClass;
        delete [] szSignature;
        break;
    case CPT_METHOD:
        // create a Method constant
        szString = CWideStringUtils::WideStringToCString(value.u.methodVal.name);
        szClass = CWideStringUtils::WideStringToCString(value.u.methodVal.className);
        szSignature = CWideStringUtils::WideStringToCString(value.u.methodVal.signature);
        pCPInfo = m_pClassBuilder->CreateExtMethodRef(szString, szSignature , szClass);
        if (NULL == pCPInfo)
        {
            cpiResult = ILLEGAL_CP_ENTRY;
        }
        else
        {
            cpiResult = pCPInfo->GetCpIndex();
        }
        delete [] szString;
        delete [] szClass;
        delete [] szSignature;
        break;
    case CPT_INTERFACE_METHOD:
        // create an InterfaceMethod constant
        szString = CWideStringUtils::WideStringToCString(value.u.interfaceMethodVal.name);
        szClass = CWideStringUtils::WideStringToCString(value.u.interfaceMethodVal.className);
        szSignature = CWideStringUtils::WideStringToCString(value.u.interfaceMethodVal.signature);
        pCPInfo = m_pClassBuilder->CreateInterfaceMethodRef(szString, szSignature, szClass);
        if (NULL == pCPInfo)
        {
            cpiResult = ILLEGAL_CP_ENTRY;
        }
        else
        {
            cpiResult = pCPInfo->GetCpIndex();
        }
        delete [] szString;
        delete [] szClass;
        delete [] szSignature;
        break;
    default:
        cpiResult = ILLEGAL_CP_ENTRY;
    }
    return cpiResult;
}

TResult CJavaClass::WriteBack(SBuffer *pBuffer,
                              TMemoryAllocatorFunc funcAllocator)
{
    if (NULL == pBuffer || NULL == funcAllocator)
    {
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }

    // emit the static initializer if needed
    if (m_bShouldEmitStaticInitializer)
    {
        m_pStaticInitializer->Emit();
    }

    // allocate memory for pBuffer
    u4 cbClassSize = m_pClassBuilder->GetSize();
    pBuffer->pBuffer = (unsigned char*)funcAllocator(cbClassSize);
    if (NULL == pBuffer->pBuffer)
    {
        return MRTE_ERROR_OUT_OF_MEMORY;
    }
    pBuffer->uiBufferSize = cbClassSize;
    pBuffer->uiActualSize = cbClassSize;

    // write the instrumented class
	CJMemStream		memStream;                      // Memory stream
	memStream.Open(pBuffer->pBuffer, pBuffer->uiActualSize);
	CJStream jsOut(&memStream);		                // Java output stream
	m_pClassBuilder->Write(jsOut);
    return MRTE_RESULT_OK;
}

//TODO: move CreateClassConstant to BCI's JClassBuilder class
// Create a "Class" entry in the constant pool
//
// Parameters:
//      - szClassName [in]: class name
// 
// Returns:
//      CCPClassInfo* : pointer to the class info in the constant pool
//
CCPClassInfo* CJavaClass::CreateClassConstant(const char *szClassName)
{
    CConstPool *pCP = m_pClassBuilder->GetConstPool();
	u2 u2ClassName = pCP->Add(new CCPUtf8Info(szClassName));
	u2 u2ClassRef  = pCP->Add(new CCPClassInfo(u2ClassName));
	CCPClassInfo* pcpClass = (CCPClassInfo*)((*pCP)[u2ClassRef]);
	return pcpClass;
}

//
// Initialize a static field
// 
// Parameters:
//      - fieldRef [in] : the field to initialize
//      - initValue [in]: initial value
//
// Returns:
//      MRTE_RESULT_OK              : success
//      MRTE_ERROR_ILLEGAL_ARGUMENT : unsupported initial value type
//
TResult CJavaClass::InitializeStaticField(const CCPFieldrefInfo *fieldRef, 
                                          const SVarValue *initValue)

{
    //TODO: fix. Need to add serialVersionUID field if it is needed and does not already
    // exist, in order not to break the Java serialization mechanism
    CMethodJ *pStaticInitializer = GetStaticInitializerMethod();
    CInsBlocks *pInsBlocks = pStaticInitializer->GetInsBlocks();
    CInsBlock *pNewBlock = new CInsBlock((unsigned int)-1);
    CCPInfo *pCpInfo;
    CInstruction *pInitInst;

    // generate byte-code to push the field's value on the stack
    switch (initValue->type) {
    case VT_BYTE:
        pInitInst = CInsSetJ::Create_push_constant(initValue->byteVal);
    	break;
    case VT_SHORT:
        pInitInst = CInsSetJ::Create_push_constant(initValue->shortVal);
    	break;
    case VT_INT:
        if (initValue->intVal >= -32768 && initValue->intVal <= 32767)
        {
            // the value is a 16-bit value and can be pushed directly to the stack
            pInitInst = CInsSetJ::Create_push((short)initValue->intVal);
        }
        else
        {
            // the value is larger than 16-bit and requires a constant pool entry
            pCpInfo = m_pClassBuilder->CreateIntegerConstant(initValue->intVal);
            pInitInst = CInsSetJ::Create_ldc(pCpInfo->GetCpIndex());
        }
        break;
    case VT_LONG:
        pCpInfo = m_pClassBuilder->CreateLongConstant(initValue->longVal);
        pInitInst = CInsSetJ::Create_ldc(pCpInfo->GetCpIndex());
    	break;
    case VT_FLOAT:
        pCpInfo = m_pClassBuilder->CreateFloatConstant(initValue->floatVal);
        pInitInst = CInsSetJ::Create_ldc(pCpInfo->GetCpIndex());
    	break;
    case VT_DOUBLE:
        pCpInfo = m_pClassBuilder->CreateDoubleConstant(initValue->doubleVal);
        pInitInst = CInsSetJ::Create_ldc(pCpInfo->GetCpIndex());
    	break;
    case VT_CHAR:
        pInitInst = CInsSetJ::Create_push((char)initValue->charVal);
    	break;
    case VT_BOOLEAN:
        pInitInst = CInsSetJ::Create_push_constant(initValue->boolVal?1:0);
    	break;
    default:
        // unsupported type
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }
    pNewBlock->AddInstruction(pInitInst);
    
    // generate byte-code to initialize the field:
    //      putstatic #fieldCpIndex
    pNewBlock->AddInstruction(CInsSetJ::Create_putstatic(fieldRef->GetCpIndex()));

    // add the new block to the head of the <clinit> method
    pInsBlocks->insert(pInsBlocks->begin(), pNewBlock);

    // Do not call pStaticInitializer->Emit() here!
    // Reason: the client may add more fields later, and IBM BCI allows only one
    // call to Emit() for each instrumented method. To overcome this,
    // we remember that we have instrumented the static initializer and defer the 
    // call to Emit() to WriteBack time.
    m_bShouldEmitStaticInitializer = true;

    return MRTE_RESULT_OK;
}

//
// Gets the class static initializer method (<clinit>), creating it if necessary
//
// Returns:
//      CMethodJ* : a pointer to the <clinit> method
//
CMethodJ* CJavaClass::GetStaticInitializerMethod()
{
    if (m_pStaticInitializer != NULL)
    {
        return m_pStaticInitializer;
    }

    m_pStaticInitializer = LookupMethod(JAVA_CLASS_CONSTRUCTOR_NAME);
    if (m_pStaticInitializer != NULL)
    {
        return m_pStaticInitializer;
    }

    // the method <clinit> does not exist. create it
    CreateClassStaticInitializer();
    return m_pStaticInitializer;
}

//
// Looks up a method by name
//
// Parameters:
//      szName : method name
//
// Returns:
//      CMethodJ* : a pointer to the method, or NULL if the method does not exist
//
CMethodJ* CJavaClass::LookupMethod(const char *szName)
{
    ParseInternalModule();
    CMethods* pmeths = m_pModule->GetMethods();
    for (int i = 0; i < pmeths->size(); i++) {
        CMethod* pmeth = (*pmeths)[i];
        const char* mname = pmeth->GetName();
        if (strcmp(mname, szName) == 0) {
            return (CMethodJ*)pmeth;
        }
    }
    // method does not exist
    return NULL;
}

//
// Creates a class static initializer
//
// Returns:
//      CMethodJ* : a pointer to the class static initializer method
//
CMethodJ* CJavaClass::CreateClassStaticInitializer()
{
    // create a method named "<clinit>"
    m_pStaticInitializer = m_pModule->CreateMethod(ACC_STATIC,
        JAVA_CLASS_CONSTRUCTOR_NAME, "()V");

    // add a "return" instruction as the method body
    CInsBlocks* pInsBlocks = m_pStaticInitializer ->GetInsBlocks();
    CInsBlock* pBlk = new CInsBlock((unsigned int)-1);
    pBlk->AddInstruction(CInsSetJ::Create_simple(JVMI_return));
    pInsBlocks->push_back(pBlk);
    return m_pStaticInitializer;
}

//
// parses the internal module only if necessary
//
void CJavaClass::ParseInternalModule()
{
    if (!m_bModuleParsed)
    {
        m_pModule->Parse();
        m_bModuleParsed = true;
    }
}

//
// Init the class' method list
//
void CJavaClass::InitMethodList() 
{
    if (!m_bMethodListCreated)
    {
        // initialize the class' methods list (m_lstJavaMethods)
        ParseInternalModule();
        CMethods* pmeths = m_pModule->GetMethods();
        TMRTEHandle h = NULL;
        for (int i = 0; i < pmeths->size(); i++) {
            CMethodJ* pmethj = (CMethodJ*)(*pmeths)[i];
            CJavaMethod *pJieMethod = new CJavaMethod(this, pmethj);
            h = m_lstJavaMethods.InsertAfter(h, pJieMethod);
        }
        m_bMethodListCreated = true;
    }
}

//
// Init the class' field list
//
void CJavaClass::InitFieldList() 
{
    if (!m_bFieldListCreated)
    {
        ParseInternalModule();
        CJFields* pFields = m_pClassBuilder->GetFields();
        TMRTEHandle h = NULL;
        for (int i = 0; i < pFields->size(); i++) {
            CFieldInfo* pFieldInfo = (CFieldInfo*)(*pFields)[i];
            char *szFieldName = CBciUtils::CCPUtf8InfoToCString(pFieldInfo->GetName());
            CCPFieldrefInfo *pCpFieldRef = m_pClassBuilder->FindFieldRef(szFieldName);
            delete [] szFieldName;
            CJavaField *pJieField = new CJavaField(this, pFieldInfo, pCpFieldRef);
            h = m_lstJavaFields.InsertAfter(h, pJieField);
        }
        m_bFieldListCreated = true;
    }
}

IJavaMethod* CJavaClass::AddMethod(const SWideString& szName,
                                   const SWideString& szSignature,
                                   unsigned int uiAttributeFlags)
{
    char *szMethodName = CWideStringUtils::WideStringToCString(szName);
    char *szMethodSig = CWideStringUtils::WideStringToCString(szSignature);

    // create the method
    CMethodJ *pNewMethod = m_pModule->CreateMethod(uiAttributeFlags, szMethodName, 
        szMethodSig);

    CJavaMethod *pJieMethod = new CJavaMethod(this, pNewMethod);
    m_lstJavaMethods.InsertAfter(NULL, pJieMethod);

    // clean-up
    delete [] szMethodName;
    delete [] szMethodSig;

    return pJieMethod;
    
}
