/*****************************************************************************
 * Copyright (c) 1997, 2008 Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/
//

#include "MRTEInfrastructureDefinitions.h"
#include "JavaMethod.h"
#include "JIEInstructionCodec.h"
#include "WideStringUtils.h"
#include "JIEUtils.h"

using namespace Martini::MIE;
using namespace Martini::JIE;
using namespace Martini::RTUtil;
using namespace Martini::Infrastructure;

//////////////////////////////////////////////////////////////////////////
// class CJavaMethod implementation

CJavaMethod::CJavaMethod(CJavaClass *pOwner, CMethodJ *pmethj) 
    : m_pBCIMethod(pmethj), m_pOwnerClass(pOwner)
{
    // initialize the method instrumentor
    CJIEInstructionCodec *pCodec = new CJIEInstructionCodec(m_pBCIMethod, 
        m_pOwnerClass->GetBciClassBuilder());
    m_instrumentor.Init(pCodec);
}

CJavaMethod::~CJavaMethod()
{
}

TInstructionListIterator* CJavaMethod::GetInstructionListIterator(
    const EInstructionIteratorType iteratorType)
{
    TInstructionListIterator *pIter = 
        m_instrumentor.GetInstructionListIterator(iteratorType);
    return pIter;
}

TResult CJavaMethod::BindTryFinallyBlock(IInstruction* iTry, 
                                         IInstruction* iEndTry, 
                                         IInstruction* iFinally, 
                                         IInstruction* iEndFinally)
{
    if (NULL == iTry
        || NULL == iEndTry
        || NULL == iFinally
        || NULL == iEndFinally)
    {
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }
    CMIEInstruction *pTry = (CMIEInstruction*)iTry;
    CMIEInstruction *pEndTry = (CMIEInstruction*)iEndTry;
    CMIEInstruction *pFinally = (CMIEInstruction*)iFinally;
    CMIEInstruction *pEndFinally = (CMIEInstruction*)iEndFinally;
    TResult res = m_instrumentor.BindTryFinallyBlock(pTry, pEndTry, pFinally, pEndFinally);    
    return res;
}

TResult CJavaMethod::VerifyCode(SInstrumentationErrorsArray* pErrorsArray) const
{
    //TODO: implement - operands count match mnemonic
    //TODO: implement - operands types match mnemonic
    //TODO: implement - correct nesting of try..finally blocks
    //TODO: implement - enough local variables available for implementing "try..finally" blocks
    return MRTE_ERROR_FAIL;
}

TResult CJavaMethod::ApplyInstrumentation()
{
    // PATCH: if the method is the class static initializer (<clinit>), flag it
    // so that it won't be emitted again during JavaClass::WriteBack.
    // This is OK for CG Adaptor, but does not support a more generic usage model.
    
    //TODO: BUGFIX: the current patch will cause IJavaClass::AddField to do nothing
    // if it was called after <clinit> was instrumented through an IJavaMethod instance.

    if (IsStaticInitializer())
    {
        m_pOwnerClass->StaticInitializerEmitted();
    }

    TResult res;
    try
    {
        res = m_instrumentor.ApplyInstrumentation();
    }
    catch (CModuleException e)
    {
    	//TODO: log an error to the log file
        res = MRTE_ERROR_FAIL;
    }
    return res;
}

TResult CJavaMethod::DebugPrint(bool bToResolveNames, char* fileName)
{
    // hack: create an iterator to force the flow-graph to initialize
    TInstructionListIterator *pIter = GetInstructionListIterator(IET_ALL_INSTRUCTIONS);
    pIter->Free();

    TResult res = m_instrumentor.GetFlowGraph().DebugPrint(bToResolveNames, fileName);
    return res;
}

TResult CJavaMethod::GetMethodInfo(SJavaMethodInfo* pMethodInfo) const
{
    if (NULL == m_pBCIMethod)
    {
        return MRTE_ERROR_UNEXPECTED;
    }

    // get class name
    TResult res = m_pOwnerClass->GetName(&(pMethodInfo->className));
    if (MRTE_FAILED(res))
    {
        return res;
    }

    const char *sz;
    // get method name
    sz = m_pBCIMethod->GetName();
    pMethodInfo->methodName.uiActualSize = strlen(sz);
    if (pMethodInfo->methodName.uiActualSize > pMethodInfo->methodName.uiBufferSize)
    {
        return MRTE_ERROR_BUFFER_TOO_SHORT;
    }
    CWideStringUtils::AssignText(pMethodInfo->methodName, sz);
    // get method signature
    sz = m_pBCIMethod->GetSignature();
    pMethodInfo->methodSignature.uiActualSize = strlen(sz);
    if (pMethodInfo->methodSignature.uiActualSize > pMethodInfo->methodSignature.uiBufferSize)
    {
        return MRTE_ERROR_BUFFER_TOO_SHORT;
    }
    CWideStringUtils::AssignText(pMethodInfo->methodSignature, sz);
    // get method's access flags
    pMethodInfo->uiAttributeFlags = m_pBCIMethod->GetAccessFlags();
    return MRTE_RESULT_OK; 
}

TResult CJavaMethod::AddLocalVariable(TVariableID *pVariableId, 
                                      EVarType variableType) 
{
    // validate arguments
    if (NULL == pVariableId)
    {
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }
    if (variableType <= VT_NONE || variableType >= VT_LAST)
    {
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }
    // add the local variable
    CCodeAttribute *pCodeAttr = m_pBCIMethod->GetCodeAttribute();
    CJavaType jtVarType = VarTypeToJavaType[variableType];
    if (!_AddLocalVariable(pCodeAttr, jtVarType, *pVariableId))
    {
        return MRTE_ERROR_FAIL;
    }
    return MRTE_RESULT_OK;
}

//
// Adds a local variable of the specified type.
// 
// Parameters:
//      - pCodeAttr [in]: the code attribute to which to add the variable
//      - jtType [in]: variable type
//      - varIndex [out]: the index of the newly allocated variable
//
// Returns:
//      - true if the variable was successfully added, false otherwise
//
bool CJavaMethod::_AddLocalVariable(CCodeAttribute *pCodeAttr, const CJavaType &jtType,
                                    u2 &varIndex)
{
    //TODO: move this method to CCodeAttribute (after coordinating with IBM/BCI owner)
    const u2 MAX_LOCAL_VARIABLES = 65535;
    u2 u2NewVarInd = pCodeAttr->GetMaxLocals();
    if (u2NewVarInd >= MAX_LOCAL_VARIABLES)
    {
        // the maximum number of local variables per method was reached
        return false;
    }
    u2 u2LocalVarCount = u2NewVarInd;
    if (jtType.GetType() == CJavaType::J_LONG || jtType.GetType() == CJavaType::J_DOUBLE)
    {
        // LONG and DOUBLE variables require 2 consecutive entries in the local 
        // variable table
        u2LocalVarCount += 2;
    }
    else
    {
        u2LocalVarCount++;
    }
    pCodeAttr->SetMaxLocals(u2LocalVarCount);
    varIndex = u2NewVarInd;
    return true;
}

inline bool CJavaMethod::IsStaticInitializer() const
{
    if (strcmp(JAVA_CLASS_CONSTRUCTOR_NAME, m_pBCIMethod->GetName()) == 0)
    {
        return true;
    }
    return false;
}

TConstantPoolIndex CJavaMethod::GetIndex() const
{
    const char *szMethodName = m_pBCIMethod->GetName();
    const char *szMethodSig = m_pBCIMethod->GetSignature();

    CCPMethodrefInfo *pcpm = 
        m_pOwnerClass->GetBciClassBuilder()->FindMethodRef(szMethodName, szMethodSig);
    if (pcpm != NULL)
    {
        return pcpm->GetCpIndex();
    }
    
    // the method is not registered in the constant pool
    return ILLEGAL_CP_ENTRY;
}
