/*****************************************************************************
 * Copyright (c) 1997, 2009, Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/
//

#include <stdlib.h>
#include <string.h>

#include "MIEInstruction.h"
#include "MethodInstrumentor.h"
#include "ValidityChecks.h"
#include "MIEInstructionFactory.h"

#define IS_PSEUDO_MNEM(mnem) (mnem >= MNM_FIRST_PSEUDO && mnem < MNM_LAST_PSEUDO)

using namespace Martini::MIE;

//////////////////////////////////////////////////////////////////////////
// class CMIEInstruction implementation

CMIEInstruction::CMIEInstruction(CMethodInstrumentor* pInstrumentor)
: m_pInstrumentor(pInstrumentor), m_pfgNode(NULL), m_vpOperands(NULL), m_uiOperandsCount(0),
    m_uiTargetOffset(0), m_uiOriginalOffset(0), m_uiTargetsCount(0), 
    m_vuiTargetOffsets(NULL), m_pReadOnlyInst(NULL), m_uiActualOffset(0),
    m_uiLabel(0)
{
}

CMIEInstruction::~CMIEInstruction()
{
    unsigned int uiOpCount = GetOperandsCount();
    if (uiOpCount > 0)
    {
        // free operands
        unsigned int i;
        for (i = 0; i < uiOpCount; ++i)
        {
            DeallocateOperand(m_vpOperands[i]);
            delete m_vpOperands[i];
        }
        delete [] m_vpOperands;
    }
    // free offsets array of switch instructions
    if (m_uiTargetsCount > 0)
    {
        delete [] m_vuiTargetOffsets;
    }
    // free read-only wrapper
    if (m_pReadOnlyInst != NULL)
    {
        delete m_pReadOnlyInst;
    }
}

const EMnemonic CMIEInstruction::GetMnemonic() const
{
    return m_mnem;
}

TResult CMIEInstruction::SetMnemonic(const EMnemonic newMnemonic)
{
    //TEST
    if (SEM_PSEUDO == m_semTag)
    {
        return MRTE_ERROR_NOT_SUPPORTED;
    }
    if (newMnemonic < 0 || newMnemonic >= MNM_LAST_PSEUDO)
    {
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }
    TResult res = InternalSetMnemonic(newMnemonic);
    return res;
}

TResult CMIEInstruction::InternalSetMnemonic(const EMnemonic newMnemonic)
{
    CMIEInstructionDescr &descr = CMIEInstructionFactory::GetDescriptor(newMnemonic);
/*
    if (NULL == pDescr)
    {
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }
*/
    CopyOnModification();
    m_mnem = newMnemonic;
    m_szName = descr.GetName();
    m_semTag = descr.GetSemantics();
    if (descr.GetOpCount() != m_uiOperandsCount)
    {
        // reallocate space for operands (old operands are discarded)
        unsigned int i;
        for (i = 0; i < m_uiOperandsCount; ++i)
        {
            delete m_vpOperands[i];
        }
        delete [] m_vpOperands;
        AllocateOperands(descr.GetOpCount());        
    }
    m_uiOperandsCount = descr.GetOpCount();
    return MRTE_RESULT_OK;
}
//
// Returns the operand(s) of the instruction
//
// Parameters:
//      - pOperand1 [out] - a caller allocated SOperand structure
//      - pOperand2 [out] - a caller allocated SOperand structure
//      - pOperand3 [out] - a caller allocated SOperand structure
//      - pOperand4 [out] - a caller allocated SOperand structure
//
// Returns:
//      MRTE_RESULT_OK : success
//      MRTE_ERROR_BUFFER_TOO_SHORT : the STargetsArray member in the 
//                                    caller's SOperand is not big enough.
//      MRTE_ERROR_ILLEGAL_ARGUMENT : at least one of the instruction operands 
//                                    cannot be filled. The reason can be: its 
//                                    pOperand buffer is NULL, or its internal
//                                    STargetArray is not allocated (for switch
//                                    instruction).
//      MRTE_RESULT_NO_OPERAND : instruction has no operand or not as many
//                               operands as the number of arguments
//
TResult CMIEInstruction::GetOperand(SOperand *pOperand1, 
                                    SOperand *pOperand2,
                                    SOperand *pOperand3,
                                    SOperand *pOperand4) const
{
    //TEST
    unsigned int uiOpCount = GetOperandsCount();
    if (0 == uiOpCount)
    {
        return MRTE_RESULT_NO_OPERAND;
    }
    if (SEM_PSEUDO == m_semTag)
    {
        return MRTE_ERROR_NOT_SUPPORTED;
    }
    // copy operands into the appropriate buffers
    TResult res;
    res = CopyOperand(pOperand1, m_vpOperands[0], false);
    CHECK_TRESULT(res);
    if (uiOpCount >= 2)
    {
        res = CopyOperand(pOperand2, m_vpOperands[1], false);
        CHECK_TRESULT(res);
    }
    if (uiOpCount >= 3)
    {
        res = CopyOperand(pOperand3, m_vpOperands[2], false);
        CHECK_TRESULT(res);
    }
    if (uiOpCount >= 4)
    {
        res = CopyOperand(pOperand4, m_vpOperands[3], false);
        CHECK_TRESULT(res);
    }
    return MRTE_RESULT_OK;
}

//
// Returns a pointers (not a copy) to the operand at the specified index.
//
// Parameters:
//      uiIndex :   operand index (0-based)
//
// Returns:
//      SOperand* : a pointer to the operand
//      NULL      : when no such operand exists
//
SOperand* CMIEInstruction::InternalGetOperand(unsigned int uiIndex) const
{
    if (uiIndex >= GetOperandsCount())
    {
        return NULL;
    }
    return m_vpOperands[uiIndex];
}

//
// Sets or replaces the operands of the instruction with new ones. The 
// caller has to make sure the new operand types match the instruction mnemonics
// or change the mnemonics.
//
// Parameters:
//      pOperand1 [in] - the caller allocated SOperand 
//                       structure for the 1st operand.
//      pOperand2 [in] - the caller allocated SOperand 
//                       structure for the 2nd operand.
//      pOperand3 [in] - the caller allocated SOperand 
//                       structure for the 3rd operand.
//      pOperand4 [in] - the caller allocated SOperand 
//                       structure for the 4th operand.
//
// NOTE: for branch instruction, pOperand1 must contain a pointer to the IInstruction
//       object which is the branch target. If the target is not a pseudo 
//       .target, the pseudo .target is automatically
//       added just before the targeted instruction.
//
// Returns:
//      MRTE_RESULT_OK               : success
//      MRTE_ERROR_ILLEGAL_ARGUMENT  : Operand type is invalid
//      MRTE_ERROR_ARGUMENT_MISMATCH : Operand type and operand value mismatch.
//                                     e.g. operand type is OT_TARGET or 
//                                     OT_TARGETS_ARRAY and the operand value 
//                                     contains invalid IInstruction pointer(s), 
//                                     operand type is OT_VAR_ID and the value
//                                     contains an invalid variable id
//
TResult CMIEInstruction::SetOperand(SOperand *pOperand1, 
                                    SOperand *pOperand2, 
                                    SOperand *pOperand3,
                                    SOperand *pOperand4)
{
    //TEST
    if (SEM_PSEUDO == m_semTag)
    {
        return MRTE_ERROR_NOT_SUPPORTED;
    }
    if (NULL == pOperand1)
    {
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }
    if (IsBranch())
    {
        // verify that the first operand is a valid branch target
        if (pOperand1->type != OT_TARGET)
        {
            return MRTE_ERROR_ILLEGAL_ARGUMENT;
        }
        IInstruction *pTarget = SetBranchTarget(pOperand1->val.pTarget);
        if (NULL == pTarget)
        {
            return MRTE_ERROR_ILLEGAL_ARGUMENT;
        }
        else
        {
            // branch target succesfully updated
            return MRTE_RESULT_OK;
        }
    } // if (IsBranch)
    CopyOnModification();
    InternalSetOperand(pOperand1, pOperand2, pOperand3, pOperand4);
    return MRTE_RESULT_OK;
}

TResult CMIEInstruction::AddBefore(IInstruction   **ppInstruction,
                                   const EMnemonic newInstMnemonic,
                                   const SOperand *pNewInstOperand1,
                                   const SOperand *pNewInstOperand2,
                                   const SOperand *pNewInstOperand3)
{
    //TEST
    if (NULL == m_pInstrumentor)
    {
        return MRTE_ERROR_FAIL;
    }
    if (IS_PSEUDO_MNEM(newInstMnemonic))
    {
        // pseudo-instructions cannot be added with this API
        return MRTE_ERROR_ILLEGAL_OPERATION;

    }
    if (MNM_PSEUDO_START == m_mnem)
    {
        // instructions cannot be added before .START
        return MRTE_ERROR_ILLEGAL_OPERATION;
    }
    CMIEInstruction *pNewInst;
    TResult res = m_pInstrumentor->AddInstructionBefore(&pNewInst,
        m_pfgNode, newInstMnemonic, pNewInstOperand1, pNewInstOperand2,
        pNewInstOperand3);
    *ppInstruction = pNewInst;
    return res;
}

TResult CMIEInstruction::AddAfter(IInstruction   **ppInstruction,
                                  const EMnemonic newInstMnemonic, 
                                  const SOperand *pNewInstOperand1,
                                  const SOperand *pNewInstOperand2,
                                  const SOperand *pNewInstOperand3)
{
    //TEST
    if (NULL == m_pInstrumentor)
    {
        return MRTE_ERROR_FAIL;
    }
    if (IS_PSEUDO_MNEM(newInstMnemonic))
    {
        // pseudo-instructions cannot be added with this API
        return MRTE_ERROR_ILLEGAL_OPERATION;

    }
    if (MNM_PSEUDO_END == m_mnem)
    {
        // instructions cannot be added after .END
        return MRTE_ERROR_ILLEGAL_OPERATION;
    }
    CMIEInstruction *pNewInst;
    TResult res = m_pInstrumentor->AddInstructionAfter(&pNewInst,
        m_pfgNode, newInstMnemonic, pNewInstOperand1, pNewInstOperand2,
        pNewInstOperand3);
    *ppInstruction = pNewInst;
    return res;
}

bool CMIEInstruction::IsOriginalInst() const
{
    return (IS_ORIGINAL == m_status);
}

TResult CMIEInstruction::GetOriginalOffset(unsigned int* puiOriginalOffset) const
{
    //TEST
    if (m_status == IS_NEW)
    {
        return MRTE_RESULT_ADDED_INSTRUCTION;
    }
    *puiOriginalOffset = m_uiOriginalOffset;
    return MRTE_RESULT_OK;
}

//
// Sets the branch target of this branch instruction.
//
// Parameters:
//      - pTargetInst [in] : the target instruction.
//                           If pTargetInst is not a pseudo .target, a pseudo .target
//                           will be automatically added just before pTargetInst.
//
// Returns:
//      - IInstruction* - the pseudo .target which is the branch target
//      - NULL - failure
//
IInstruction* CMIEInstruction::SetBranchTarget(IInstruction *pTargetInst)
{
    //TEST
    if (!IsBranch())
    {
        // the instruction object is not a branch
        return NULL;
    }
    if (NULL == pTargetInst)
    {
        return NULL;
    }
    if (GetOperandsCount() < 1)
    {
        return NULL;
    }

    CMIEInstruction *pMieTargetInst = (CMIEInstruction*)pTargetInst;
    CMIEInstruction *pPseudoTarget;
    if (pMieTargetInst->GetMnemonic() == MNM_PSEUDO_TARGET)
    {
        pPseudoTarget = pMieTargetInst;
    }
    else
    {
        // if we reached here, pTargetInst must mot be a pseudo instruction
        if (IS_PSEUDO_MNEM(pMieTargetInst->GetMnemonic()))
        {
            return NULL;
        }

        // if the instruction that precedes pTargetInst is a .target, use it
        CMIEInstruction *pInstBeforeTarget = 
            m_pInstrumentor->GetFlowGraph().GetPrecedingInstruction(pMieTargetInst);
        if (pInstBeforeTarget->GetMnemonic() == MNM_PSEUDO_TARGET)
        {
            pPseudoTarget = pInstBeforeTarget;
        }
        else
        {
            // create a new pseudo .target instruction
            TResult res = m_pInstrumentor->AddInstructionBefore(&pPseudoTarget,
                pMieTargetInst->GetFlowGraphNode(), MNM_PSEUDO_TARGET, NULL, NULL, NULL);
            if (res != MRTE_RESULT_OK)
            {
                return NULL;
            }
            SOperand opRealTargetInst;
            opRealTargetInst.type = OT_TARGET;
            opRealTargetInst.val.pTarget = pMieTargetInst;
            pPseudoTarget->InternalSetOperand(&opRealTargetInst);
        }
        
    }
    // update operand
    CopyOnModification();
    m_vpOperands[0]->type = OT_TARGET;
    m_vpOperands[0]->val.pTarget = pPseudoTarget;

    return pPseudoTarget;
}

IInstruction* CMIEInstruction::GetBranchTarget() const
{
    //TEST
    if (!IsBranch())
    {
        // the instruction object is not a branch
        return NULL;
    }
    if (GetOperandsCount() < 1)
    {
        return NULL;
    }
    if (m_vpOperands[0]->type != OT_TARGET)
    {
        return NULL;
    }
    IInstruction *pTargetInst = m_vpOperands[0]->val.pTarget;
    return pTargetInst;
}

//
// Copies the specified operands to the internal operands array of the CMIEInstruction
// object. The function does not perform any validity checks and assume that the 
// operands array is already initialized with the correct number of operands.
//
void CMIEInstruction::InternalSetOperand(const SOperand *pOperand1, 
                                         const SOperand *pOperand2, 
                                         const SOperand *pOperand3,
                                         const SOperand *pOperand4)
{
    // here we assume that the m_vpOperands array is correctly initialized.
    // this is done during CMIEInstruction construction (see CMIEInstructionFactory)
    if (m_uiOperandsCount >= 1)
    {
        CopyOperand(m_vpOperands[0], pOperand1, true);
    }
    if (m_uiOperandsCount >= 2)
    {
        CopyOperand(m_vpOperands[1], pOperand2, true);
    }
    if (m_uiOperandsCount >= 3)
    {
        CopyOperand(m_vpOperands[2], pOperand3, true);
    }
    if (m_uiOperandsCount >= 4)
    {
        CopyOperand(m_vpOperands[3], pOperand4, true);
    }
}

//
// Copies the specified operand to the specified location in the internal operands array
// of the CMIEInstruction object. The function does not perform any validity checks and 
// assume that the operands array is already initialized with the correct number of 
// operands.
//
void CMIEInstruction::InternalSetOperand(unsigned int uiOpInd,
                                         const SOperand *pOp)
{
    // here we assume that the m_vpOperands array is correctly initialized.
    // this is done during CMIEInstruction construction (see CMIEInstructionFactory)
    CopyOperand(m_vpOperands[uiOpInd], pOp, true);
}

void CMIEInstruction::GetOperatorDebugString(bool bToResolveNames,
                                             char *szOpValues)
{
    char szOpVal[1000];
    szOpValues[0] = 0;
    unsigned int i;
    for (i = 0; i < GetOperandsCount(); ++i)
    {
        OperandToString(szOpVal, m_vpOperands[i], bToResolveNames);
        strcat(szOpValues, szOpVal);
        strcat(szOpValues, " ");
    }
}

void CMIEInstruction::OperandToString(char *szOp, SOperand *pOp, bool bToResolveNames)
{
    if (NULL == pOp)
    {
        szOp[0] = 0;
        return;
    }

    unsigned int uiOffset;
    char szTemp[100];
    CMIEInstruction *pInst;
    unsigned int i;
	
    switch (pOp->type) {
    case OT_INT8:
		sprintf(szOp, "%d", pOp->val.s8Op);
        break;
    case OT_UINT8:
		sprintf(szOp, "%u", pOp->val.ui8Op);
        break;
    case OT_INT16:
		sprintf(szOp, "%d", pOp->val.s16Op);
        break;
    case OT_UINT16:
		sprintf(szOp, "%u", pOp->val.ui16Op);
        break;
    case OT_INT32:
		sprintf(szOp, "%d", pOp->val.s32Op);
        break;
    case OT_UINT32:
		sprintf(szOp, "%u", pOp->val.ui32Op);
        break;
    case OT_INT64:
		sprintf(szOp, "%I64d", pOp->val.s64Op);
        break;
    case OT_UINT64:
		sprintf(szOp, "%I64u", pOp->val.ui64Op);
        break;
    case OT_FLOAT32:
		sprintf(szOp, "%f", pOp->val.f32Op);
        break;
    case OT_FLOAT64:
		sprintf(szOp, "%f", pOp->val.d64Op);
        break;
    case OT_TARGET:
        pInst = (CMIEInstruction*)pOp->val.pTarget;
        if (NULL == pInst)
        {
            strcpy(szOp, "N/A");
        }
        else
        {
            if (pInst->IsPseudo())
            {
                sprintf(szOp, "(L%d)", pInst->GetLabel());
            }
            else
            {
                uiOffset = pInst->InternalGetOriginalOffset();
				sprintf(szOp, "%u", uiOffset);
            }
        }
        break;
    case OT_TARGETS_ARRAY:
        // process default target
        pInst = (CMIEInstruction*)pOp->val.targetsArray.pDefaultTarget;
        if (pInst != NULL)
        {
            uiOffset = pInst->InternalGetOriginalOffset();
			sprintf(szOp, "%u", uiOffset);
        }
        else
        {
            strcpy(szOp, "N/A");
        }
        // process other targets
        for (i = 0; i < pOp->val.targetsArray.uiUsedEntries; ++i)
        {
            pInst = (CMIEInstruction*)pOp->val.targetsArray.pTargetsArray[i].pTarget;
            if (pInst != NULL)
            {
                uiOffset = ((CMIEInstruction*)pInst)->InternalGetOriginalOffset();
			    sprintf(szTemp, "%u", uiOffset);
                strcat(szOp, " ");
                strcat(szOp, szTemp);
            }
            else
            {
                strcpy(szOp, "N/A");
            }
        }
        break;
    case OT_VAR_ID:
		sprintf(szOp, "%d", pOp->val.varID);
        break;
    case OT_JAVA_CP_INDEX:
    	sprintf(szOp, "0x%04x", pOp->val.cpIndex);
        if (bToResolveNames)
        {
            //TODO: implement CP item resolution
        }
        break;
    default:
        szOp[0] = 0;
    }
}

//
// Copies pSourceOp to pDestOp.
//
// Parameters:
//      pDestOp             [out] : a buffer allocated by the caller, that will contain the data
//                                  in pSourceOp
//      pSourceOp           [in]  : operand to copy
//      bAllocSwitchTargets [in]  : whether to allocate the targets array member when 
//                                  the operand is a OT_TARGETS_ARRAY and the targets
//                                  array is not big enough
//
// Returns:
//      MRTE_RESULT_OK : success
//      MRTE_ERROR_BUFFER_TOO_SHORT : the STargetsArray member in the 
//                                    pDestOp is not big enough.
//      MRTE_ERROR_ILLEGAL_ARGUMENT : a required member of pDestOp is not properly
//                                    allocated, or pDestOp is NULL.
//      MRTE_RESULT_NO_OPERAND      : pSourceOp is NULL and pDestOp is not NULL.
//
TResult CMIEInstruction::CopyOperand(SOperand *pDestOp, 
                                     const SOperand *pSourceOp,
                                     bool bAllocSwitchTargets) const
{
    if (NULL == pDestOp && pSourceOp != NULL)
    {
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }
    if (NULL == pSourceOp && pDestOp != NULL)
    {
        return MRTE_RESULT_NO_OPERAND;
    }
    if (NULL == pSourceOp)
    {
        // both operands are NULL. this is ok. nothing to do.
        return MRTE_RESULT_OK;
    }
    // update operand type
    pDestOp->type = pSourceOp->type;
    // update operand data
    unsigned int i;
    switch (pSourceOp->type)
    {
    case OT_NO_OPERAND:
    case OT_INT8:      
    case OT_UINT8:     
    case OT_INT16:     
    case OT_UINT16:    
    case OT_INT32:     
    case OT_UINT32:    
    case OT_INT64:     
    case OT_UINT64:    
    case OT_FLOAT32:   
    case OT_FLOAT64:   
    case OT_TARGET:    
    case OT_VAR_ID:    
    case OT_DOT_NET_TOKEN:
    case OT_JAVA_CP_INDEX:
        pDestOp->val = pSourceOp->val;
        break;
    case OT_TARGETS_ARRAY:
        if (NULL == pSourceOp->val.targetsArray.pTargetsArray
            && 0 != pSourceOp->val.targetsArray.uiUsedEntries)
        {
            return MRTE_ERROR_ILLEGAL_ARGUMENT;
        }
        if (pDestOp->val.targetsArray.uiEntries < pSourceOp->val.targetsArray.uiUsedEntries)
        {
            if (bAllocSwitchTargets)
            {
                pDestOp->val.targetsArray.uiEntries = 
                    pSourceOp->val.targetsArray.uiUsedEntries;
                pDestOp->val.targetsArray.uiUsedEntries = 
                    pSourceOp->val.targetsArray.uiUsedEntries;
                pDestOp->val.targetsArray.pTargetsArray = new SCaseTarget[
                    pDestOp->val.targetsArray.uiUsedEntries];
            }
            else
            {
                pDestOp->val.targetsArray.uiUsedEntries = 
                    pSourceOp->val.targetsArray.uiUsedEntries;
                return MRTE_ERROR_BUFFER_TOO_SHORT;
            }
        }
        // copy array members
        pDestOp->val.targetsArray.uiUsedEntries = pSourceOp->val.targetsArray.uiUsedEntries;
        pDestOp->val.targetsArray.pDefaultTarget = pSourceOp->val.targetsArray.pDefaultTarget;
        for (i = 0 ; i < pSourceOp->val.targetsArray.uiUsedEntries; ++i)
        {
            pDestOp->val.targetsArray.pTargetsArray[i] = 
                pSourceOp->val.targetsArray.pTargetsArray[i];

        }
        break;
    default:
        // unknown operand type
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    } // switch (pSourceOp->type)
    return MRTE_RESULT_OK;
}

//
// Updates the instruction's operand with the branch target.
//
// Parameters:
//      - pTargetInst : target instruction. Must be a pseudo .target
//
// Returns:
//      - MRTE_RESULT_OK : success
//      - MRTE_ERROR_ILLEGAL_ARGUMENT: pTargetInst is NULL or not a pseudo .target
//      - MRTE_ERROR_FAIL : unknown failure
//
TResult CMIEInstruction::InternalSetBranchTarget(CMIEInstruction *pTargetInst)
{
    if (NULL == pTargetInst)
    {
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }
    if (pTargetInst->GetMnemonic() != MNM_PSEUDO_TARGET)
    {
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }
    // update the instruction's operand
    if (GetOperandsCount() < 1)
    {
        return MRTE_ERROR_FAIL;
    }
    m_vpOperands[0]->type = OT_TARGET;
    m_vpOperands[0]->val.pTarget = pTargetInst;
    return MRTE_RESULT_OK;
}

//
// Updates the instruction's operand with the branch target.
//
// Parameters:
//      - uiTargetIndex : the target to update
//      - pTargetInst : target instruction. Must be a pseudo .target
//
// Returns:
//      - MRTE_RESULT_OK : success
//      - MRTE_ERROR_ILLEGAL_ARGUMENT: pTargetInst is NULL or not a pseudo .target
//      - MRTE_ERROR_FAIL : unknown failure
//
TResult CMIEInstruction::InternalSetSwitchTarget(unsigned int uiTargetIndex,
                                                 CMIEInstruction *pTargetInst)
{
    if (NULL == pTargetInst)
    {
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }
    if (pTargetInst->GetMnemonic() != MNM_PSEUDO_TARGET)
    {
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }
    // update the instruction's operand
    if (GetOperandsCount() < 1)
    {
        return MRTE_ERROR_FAIL;
    }
    m_vpOperands[0]->type = OT_TARGETS_ARRAY;
    m_vpOperands[0]->val.targetsArray.pTargetsArray[uiTargetIndex].pTarget = pTargetInst;
    return MRTE_RESULT_OK;
}

TResult CMIEInstruction::InternalSetSwitchDefaultTarget(CMIEInstruction *pTargetInst)
{
    if (NULL == pTargetInst)
    {
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }
    if (pTargetInst->GetMnemonic() != MNM_PSEUDO_TARGET)
    {
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }
    // update the instruction's operand
    if (GetOperandsCount() < 1)
    {
        return MRTE_ERROR_FAIL;
    }
    m_vpOperands[0]->type = OT_TARGETS_ARRAY;
    m_vpOperands[0]->val.targetsArray.pDefaultTarget = pTargetInst;
    return MRTE_RESULT_OK;
}

//
// Allocates and initializes the internal operands array
//
void CMIEInstruction::AllocateOperands(unsigned int uiOpCount)
{
    m_uiOperandsCount = uiOpCount;
    if (0 == uiOpCount) 
    {
        // no operands
        return;
    }
    m_vpOperands = new SOperand*[uiOpCount];
    unsigned int i;
    for (i = 0; i < uiOpCount; ++i)
    {
        SOperand *pNewOP = new SOperand();
        pNewOP->type = OT_NO_OPERAND; // default type
        pNewOP->val.targetsArray.uiEntries = 0;
        pNewOP->val.targetsArray.uiUsedEntries = 0;
        pNewOP->val.targetsArray.pTargetsArray = NULL;
        m_vpOperands[i] = pNewOP;
    }
}

CMIEReadOnlyInstruction* CMIEInstruction::GetReadOnlyWrapper()
{
    if (NULL == m_pReadOnlyInst)
    {
        m_pReadOnlyInst = new CMIEReadOnlyInstruction(this);
    }
    return m_pReadOnlyInst;
}

bool CMIEInstruction::CopyOnModification()
{
    CFlowGraph &flowGraph = m_pInstrumentor->GetFlowGraph();
    bool res = flowGraph.CopyOnModification(this);
    if (!res)
    {
        return false;
    }
    // update the status of the current instruction
    if (IS_ORIGINAL == m_status)
    {
        m_status = IS_MODIFIED;
    }
    return true;
}

//
// Creates and returns an exact copy of the instruction object.
//
// Returns:
//      CMIEInstruction*    - an exact copy of the instruction object
//      NULL                - error (memory allocation problem)
//
CMIEInstruction* CMIEInstruction::Clone()
{
    CMIEInstruction *pClonedInst = 
        CMIEInstructionFactory::CreateInstruction(m_mnem, m_pInstrumentor);
    CHECK_RETURN_NULL(pClonedInst);
    pClonedInst->m_status = m_status;
    pClonedInst->m_semTag = m_semTag;
    pClonedInst->m_szName = m_szName;
    pClonedInst->m_uiOriginalOffset = m_uiOriginalOffset;
    pClonedInst->m_uiTargetOffset = m_uiTargetOffset;
    pClonedInst->m_uiTargetsCount = m_uiTargetsCount;
    // copy targets array (for switch instructions)
    unsigned int i;
    if (m_uiTargetsCount > 0)
    {
        pClonedInst->m_vuiTargetOffsets = new unsigned int[m_uiTargetsCount];
        for (i = 0; i < m_uiTargetsCount; ++i)
        {
            pClonedInst->m_vuiTargetOffsets[i] = m_vuiTargetOffsets[i];
        }
    }
    // copy operands
    for (i = 0; i < m_uiOperandsCount; ++i)
    {
        CopyOperand(pClonedInst->m_vpOperands[i], m_vpOperands[i], true);
    }
    return pClonedInst;
}

bool CMIEInstruction::IsOpCountValid(const SOperand *pOp1,
                                     const SOperand *pOp2,
                                     const SOperand *pOp3,
                                     const SOperand *pOp4)
{
    unsigned int uiOpCount = 0;
    if (NULL != pOp1)
    {
        uiOpCount++;
    }
    if (NULL != pOp2)
    {
        uiOpCount++;
    }
    if (NULL != pOp3)
    {
        uiOpCount++;
    }
    if (NULL != pOp4)
    {
        uiOpCount++;
    }
    return uiOpCount == m_uiOperandsCount;
}

// Calculate padding bytes for switch instructions
inline static unsigned CalcPad(unsigned int ip)	
{
	return ++ip % 4 ? 4 - ip % 4 : 0;
}

//
// Calculates and returns the instruction size in bytes.
//
// Parameters:
//      uiCurrentIP :   the current instruction pointer. This is required because
//                      it may affect the instruction size. For example, the Java
//                      tableswitch and lookupswitch instructions use padding
//                      to align their jump tables.
//
unsigned int CMIEInstruction::GetSize(unsigned int uiCurrentIP)
{
    if (IsPseudo())
    {
        return 0;
    }

    unsigned int uiSize;
    int align;
    SOperand *pOp1;
    SOperand *pOp2;
    short constVal;
    unsigned int i;

    switch (m_mnem)
    {
    case MNM_TABLESWITCH:
        // calculate the size of the tableswitch instruction
	    align = CalcPad(uiCurrentIP);
	    uiSize =  1 + align + 12 + (m_uiHigh - m_uiLow + 1) * 4;
    	break;
    case MNM_LOOKUPSWITCH:
        // calculate the size of the lookupswitch instruction
        align = CalcPad(uiCurrentIP);
        uiSize = 1 + align + 8 + m_uiTargetsCount * 8;
    	break;
    case MNM_ILOAD:
    case MNM_LLOAD:
    case MNM_ALOAD:
    case MNM_DLOAD:
    case MNM_FLOAD:
    case MNM_ISTORE:
    case MNM_LSTORE:
    case MNM_ASTORE:
    case MNM_DSTORE:
    case MNM_FSTORE:
    case MNM_RET:
        // the size depends on whether the regular version or the wide version is used
        pOp1 = InternalGetOperand(0);
        assert(OT_VAR_ID == pOp1->type);
	    if (pOp1->val.varID > 255)
        {
            // wide version
            uiSize = 4;
        }
        else
        {
            // regular version
            uiSize = 2;
        }
        break;
    case MNM_IINC:
        // the size depends on whether the regular version or the wide version is used
        pOp1 = InternalGetOperand(0);
        pOp2 = InternalGetOperand(1);
        assert(OT_VAR_ID == pOp1->type);
        assert(OT_INT8 == pOp2->type || OT_INT16 == pOp2->type);

       	#if defined(_SOLARIS) || defined(_SOLARISX86)
       		// These lines on Solaris, and the line below, should produce exactly the same result. However, these
       		// produce different results even when values are explicitly casted and extra parentheses are liberally applied. - jgw
       		//
	        if(OT_INT8 == pOp2->type) {
	        	constVal = (pOp2->val.s8Op);
	        } else {
	        	constVal = (pOp2->val.s16Op);
	        }
        #else
        	constVal = OT_INT8 == pOp2->type ? pOp2->val.s8Op : pOp2->val.s16Op;
        #endif


	    if (pOp1->val.varID > 255 || constVal > 127 || constVal < -128)
        {
            // wide version
            uiSize = 6;
        }
        else
        {
            // regular version
            uiSize = 3;
        }
        break;
    case MNM_GOTO_W:
    case MNM_JSR_W:
        uiSize = 5;
        break;
    case MNM_LDC:
        uiSize = 2;
        break;
    case MNM_MIE_LOAD_INST_OFFSET:
        // loadinstoffset is implemented as ldc2_w
        uiSize = 3;
        break;
    default:
        // generic calculation based on the instruction operands
        uiSize = 1;
        for (i = 0; i < m_uiOperandsCount; ++i)
        {
            pOp1 = InternalGetOperand(i);
            switch (pOp1->type)
            {
            case OT_NO_OPERAND:
                uiSize += 0;
    	        break;
            case OT_INT8:
            case OT_UINT8:
                uiSize += 1;
                break;
            case OT_INT16:
            case OT_UINT16:
            case OT_TARGET:
                // by default, target is a 16 bit value (wide goto and jsr are
                // handled separately)
            case OT_JAVA_CP_INDEX:
                uiSize += 2;
                break;
            case OT_INT32:
            case OT_UINT32:
            case OT_FLOAT32:
                uiSize += 4;
                break;
            case OT_INT64:
            case OT_UINT64:
            case OT_FLOAT64:
                uiSize += 8;
                break;
            } // of switch (operand type)
        } // for
    } // end switch
    return uiSize;
}

void CMIEInstruction::DeallocateOperand(SOperand *pOp)
{
    if (OT_TARGETS_ARRAY == pOp->type)
    {
        delete [] pOp->val.targetsArray.pTargetsArray;
    }
}

