/*****************************************************************************
 * Copyright (c) 1997-2007, Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/
//

#ifndef MRTE_MIEINSTRUCTION
#define MRTE_MIEINSTRUCTION

#include "MieAPI.h"

#include <assert.h>

namespace Martini { namespace MIE {

class CFlowGraph;               // forward declaration
class CFlowGraphNode;           // forward declaration
class CMethodInstrumentor;      // forward declaration
class CMIEInstructionDescr;     // forward declaration
class CMIEReadOnlyInstruction;  // forward declaration

class CMIEInstruction : public IInstruction
{
public:
    enum EInstStatus
    {
        IS_ORIGINAL = 0,    // original instruction
        IS_NEW,             // instruction added during instrumentation
        IS_MODIFIED,        // an original instruction that was modified during instrumentation
        IS_CODEGEN,         // instruction added during code generation (for implementing try..finally blocks)
        IS_LAST             // sentinel
    };
    enum ESemanticTag
    {
	    SEM_GEN,			// Generic
	    SEM_BRANCH,		    // Branch
	    SEM_CONDBRANCH,		// Conditional branch
	    SEM_SWITCH,			// Switch
	    SEM_CALL,			// Call
	    SEM_JSR,			// Jump to local subroutine
	    SEM_RET,			// Return
	    SEM_RETSR,			// Return from local subroutine 
	    SEM_THROW,			// Throw
	    SEM_INVALID,		// Invalid byte code (causes an exception)
	    SEM_PLACEHOLDER,	// Synthetic, placeholder instruction for the end of the function (zero size)
        SEM_META,           // Meta (.NET)
        SEM_BREAK,          // Break (.NET)
        SEM_PSEUDO,         // Pseudo-instruction
        SEM_SPECIAL,        // MIE special instruction
	    SEM_LAST			// Sentinel
    };
	virtual ~CMIEInstruction();

    // IInstruction methods
    virtual const EMnemonic GetMnemonic() const;
    virtual TResult SetMnemonic(const EMnemonic newMnemonic);
    virtual TResult GetOperand(SOperand* pOperand1, 
                               SOperand *pOperand2 = NULL, 
                               SOperand *pOperand3 = NULL,
                               SOperand *pOperand4 = NULL) const;
    virtual TResult SetOperand(SOperand *pOperand1, 
                               SOperand *pOperand2 = NULL, 
                               SOperand *pOperand3 = NULL,
                               SOperand *pOperand4 = NULL);
    virtual TResult AddBefore(IInstruction   **ppInstruction,
                              const EMnemonic newInstMnemonic,
                              const SOperand *pNewInstOperand1 = NULL,
                              const SOperand *pNewInstOperand2 = NULL,
                              const SOperand *pNewInstOperand3 = NULL);
    virtual TResult AddAfter(IInstruction   **ppInstruction,
                             const EMnemonic newInstMnemonic, 
                             const SOperand *pNewInstOperand1 = NULL,
                             const SOperand *pNewInstOperand2 = NULL,
                             const SOperand *pNewInstOperand3 = NULL);
    virtual bool IsOriginalInst() const;
    virtual TResult GetOriginalOffset(unsigned int* puiOriginalOffset) const;
    virtual IInstruction* GetBranchTarget() const;
    virtual IInstruction* SetBranchTarget(IInstruction *pTargetInst);

    // internal methods
    void InternalSetOperand(const SOperand *pOperand1, 
                            const SOperand *pOperand2 = NULL, 
                            const SOperand *pOperand3 = NULL,
                            const SOperand *pOperand4 = NULL);
    void InternalSetOperand(unsigned int uiOpInd, const SOperand *pOp);
    unsigned int GetOperandsCount() const { return m_uiOperandsCount; }
    void AllocateOperands(unsigned int uiOpCount);
    SOperand* InternalGetOperand(unsigned int uiIndex) const;
    bool IsOpCountValid(const SOperand *pOp1 = NULL,
                        const SOperand *pOp2 = NULL,
                        const SOperand *pOp3 = NULL,
                        const SOperand *pOp4 = NULL);
    TResult InternalSetMnemonic(const EMnemonic newMnemonic);

    void SetFlowGraphNode(CFlowGraphNode *pNode) { m_pfgNode = pNode; }
    CFlowGraphNode* GetFlowGraphNode() { return m_pfgNode; }
    unsigned int InternalGetOriginalOffset() { return m_uiOriginalOffset; }
    void SetOriginalOffset(unsigned int uiOriginalOffset)
    {
        m_uiOriginalOffset = uiOriginalOffset;
    }
    EInstStatus GetStatus() const { return m_status; }
    void SetStatus(EInstStatus status) { m_status = status; }
    ESemanticTag GetSemantics() const { return m_semTag; }
    void SetSemantics(ESemanticTag semtag) { m_semTag = semtag; }
    void SetMethodInstrumentor(CMethodInstrumentor *pInstrumentor)
    {
        m_pInstrumentor = pInstrumentor;
    }
    const char* GetName() const { return m_szName; }
    void SetName(const char *szName) { m_szName = szName; }
    void GetOperatorDebugString(bool bToResolveNames,
                                char *szOpValues);
    bool IsBranch() const
    {
        return (SEM_BRANCH == m_semTag || SEM_CONDBRANCH == m_semTag || SEM_JSR == m_semTag); 
    }
    bool IsPseudo() const { return SEM_PSEUDO == m_semTag; }
    CMIEReadOnlyInstruction* GetReadOnlyWrapper();
    bool IsReadOnly() { return false; }
    CMIEInstruction* Clone();
    unsigned int GetActualOffset() const { return m_uiActualOffset; }
    void SetActualOffset(unsigned int uiOffset) { m_uiActualOffset = uiOffset; }
    unsigned int GetSize(unsigned int uiCurrentIP);

    // branch instructions specific operations 
    void SetBranchAbsoluteOffset(unsigned int uiOffset)
    {
        m_uiTargetOffset = uiOffset;
    }
    unsigned int GetBranchAbsoluteOffset() { return m_uiTargetOffset; }
    TResult InternalSetBranchTarget(CMIEInstruction *pTargetInst);

    // switch instructions specific operations
    unsigned int* GetSwitchAbsoluteOffsets() { return m_vuiTargetOffsets; }
    void SetSwitchAbsoluteOffsets(unsigned int *vuiOffsets)
    {
        m_vuiTargetOffsets = vuiOffsets;
    }
    unsigned int GetSwitchOffsetsCount() { return m_uiTargetsCount; }
    void SetSwitchOffsetsCount(unsigned int uiCount)
    {
        m_uiTargetsCount = uiCount;
    }
    TResult InternalSetSwitchTarget(unsigned int uiTargetIndex,
                                    CMIEInstruction *pTargetInst);
    TResult InternalSetSwitchDefaultTarget(CMIEInstruction *pTargetInst);
    unsigned int GetLow() { return m_uiLow; }
    void SetLow(unsigned int low) { m_uiLow = low; }
    unsigned int GetHigh() { return m_uiHigh; }
    void SetHigh(unsigned int high) { m_uiHigh = high; }

    // pseudo instructions specific operations
    unsigned int GetLabel()
    {
        assert(IsPseudo());
        return m_uiLabel; 
    }
    void SetLabel(unsigned int uiLabel)
    {
        assert(IsPseudo());
        m_uiLabel = uiLabel; 
    }

private:
    //friend class CFlowGraph;
    friend class CMIEInstructionDescr;
    friend class CMIEReadOnlyInstruction;

    CFlowGraphNode *m_pfgNode;              // the flow graph node the instruction is stored in
    EMnemonic m_mnem;                       // mnemonic
    EInstStatus m_status;                   // instruction status
    unsigned int m_uiOriginalOffset;        // original offset
    ESemanticTag m_semTag;                  // semantics
    const char *m_szName;                   // mnemonic name (for debug)

    CMethodInstrumentor *m_pInstrumentor;   // the method instrumentor of the method to 
                                            // which the instruction belongs
    unsigned int m_uiActualOffset;          // actual offset. Used during code generation

    SOperand **m_vpOperands;                // array of instruction operand pointers
    unsigned int m_uiOperandsCount;         // number of operands in the operands array

    CMIEReadOnlyInstruction *m_pReadOnlyInst; // a "read-only" wrapper for the instruction

    // branch instructions specific fields
    unsigned int m_uiTargetOffset;          // the absolute offset of the branch target
                                            // NOTE: this is NOT the RELATIVE offset!!!

    // switch instructions specific fields
    unsigned int *m_vuiTargetOffsets;       // array of target offsets.
                                            // NOTE: offsets are absolute. NOT relative!

    unsigned int m_uiTargetsCount;          // number of targets in the target array
    unsigned int m_uiLow;                   // "low" tableswitch index
    unsigned int m_uiHigh;                  // "high" tableswitch index

    // pseudo instructions specific fields
    unsigned int m_uiLabel;                 // a "label" for pseudo instructions. Used
                                            // in the "DebugPrint" API to show branch targets

    // private constructor to prevent instantiation. Use MIEInstructionFactory
    CMIEInstruction(CMethodInstrumentor* pInstrumentor = NULL);

    void OperandToString(char *szOp, SOperand *pOp, bool bToResolveNames);
    TResult CopyOperand(SOperand *pDestOp, const SOperand *pSourceOp, 
                        bool bAllocSwitchTargets) const;
    bool CopyOnModification();

    void DeallocateOperand(SOperand *pOp);
    
}; // class MIEInstruction

//
// A CMIEInstruction wrapper that provides read-only access to the instruction it wraps.
//
class CMIEReadOnlyInstruction : public CMIEInstruction
{
public:
    virtual ~CMIEReadOnlyInstruction() {}

    // CMIEInstruction method overrides
    virtual const EMnemonic GetMnemonic() const
    {
        return m_pInst->GetMnemonic();
    }
    virtual TResult SetMnemonic(const EMnemonic newMnemonic)
    {
        return MRTE_ERROR_READ_ONLY;
    }
    virtual TResult GetOperand(SOperand* pOperand1, 
                               SOperand *pOperand2 = NULL, 
                               SOperand *pOperand3 = NULL,
                               SOperand *pOperand4 = NULL) const
    {
        return m_pInst->GetOperand(pOperand1, pOperand2, pOperand3, pOperand4);
    }
    virtual TResult SetOperand(SOperand *pOperand1, 
                               SOperand *pOperand2 = NULL, 
                               SOperand *pOperand3 = NULL,
                               SOperand *pOperand4 = NULL)
    {
        return MRTE_ERROR_READ_ONLY;
    }
    virtual TResult AddBefore(IInstruction   **ppInstruction,
                              const EMnemonic newInstMnemonic,
                              const SOperand *pNewInstOperand1 = NULL,
                              const SOperand *pNewInstOperand2 = NULL,
                              const SOperand *pNewInstOperand3 = NULL)
    {
        return MRTE_ERROR_READ_ONLY;
    }
    virtual TResult AddAfter(IInstruction   **ppInstruction,
                             const EMnemonic newInstMnemonic, 
                             const SOperand *pNewInstOperand1 = NULL,
                             const SOperand *pNewInstOperand2 = NULL,
                             const SOperand *pNewInstOperand3 = NULL)
    {
        return MRTE_ERROR_READ_ONLY;
    }
    virtual bool IsOriginalInst() const
    {
        return m_pInst->IsOriginalInst();
    }
    virtual TResult GetOriginalOffset(unsigned int* puiOriginalOffset) const
    {
        return m_pInst->GetOriginalOffset(puiOriginalOffset);
    }
    virtual IInstruction* GetBranchTarget() const
    {
        return m_pInst->GetBranchTarget();
    }
    virtual IInstruction* SetBranchTarget(IInstruction *pTargetInst)
    {
        return NULL;
    }
    bool IsReadOnly() { return true; }

private:
    friend class CMIEInstruction;

    const IInstruction *m_pInst;            // the wrapped instruction
    // private constructor to prevent instantiation.
    CMIEReadOnlyInstruction(const IInstruction *pInst): m_pInst(pInst) {}


}; // class CMIEReadOnlyInstruction

}}

#endif // MRTE_MIEINSTRUCTION

