/*****************************************************************************
 * Copyright (c) 1997-2007, Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/
//

#include "MethodInstrumentor.h"
#include "ValidityChecks.h"
#include "MIEInstructionCodec.h"
#include "MIEInstructionFactory.h"

using namespace Martini::MIE;

//////////////////////////////////////////////////////////////////////////
// class CMethodInstrumentor implementation

CMethodInstrumentor::CMethodInstrumentor()
: m_bFlowGraphInitialized(false), m_pCodec(NULL)
{

}

CMethodInstrumentor::~CMethodInstrumentor()
{
    if (m_pCodec != NULL)
    {
        delete m_pCodec;
    }
}

TResult CMethodInstrumentor::Init(IMIEInstructionCodec *pInstructionCodec)
{
    m_uiNextLabel = 0;
    TResult res = m_flowGraph.Init(this);
    m_pCodec = pInstructionCodec;
    m_pCodec->Init(this);
    return res;
}

//
// Creates an instruction object and adds it to the flow-graph before the node fgNode
//
TResult CMethodInstrumentor::AddInstructionBefore(CMIEInstruction   **ppInstruction,
                                                  CFlowGraphNode *pfgNode,
                                                  const EMnemonic newInstMnemonic,
                                                  const SOperand *pNewInstOperand1,
                                                  const SOperand *pNewInstOperand2,
                                                  const SOperand *pNewInstOperand3)
{
    *ppInstruction = NULL;
    CMIEInstruction *pNewInst = CMIEInstructionFactory::CreateInstruction(
        newInstMnemonic, this);
    CHECK_ALLOC(pNewInst);
    TResult res;
    res = m_flowGraph.AddInstructionBefore(pfgNode, pNewInst);
    CHECK_TRESULT(res);
    if (pNewInst->IsBranch())
    {
        IInstruction *pTargetInst = pNewInst->SetBranchTarget(
            pNewInstOperand1->val.pTarget);
        if (NULL == pTargetInst)
        {
            return MRTE_ERROR_OPERAND_MISMATCH;
        }
    }
    else
    {
        pNewInst->InternalSetOperand(pNewInstOperand1, pNewInstOperand2, pNewInstOperand3);
    }
    *ppInstruction = pNewInst;
    return MRTE_RESULT_OK;
}

//
// Creates an instruction object and adds it to the flow-graph after the node fgNode
//
TResult CMethodInstrumentor::AddInstructionAfter(CMIEInstruction   **ppInstruction,
                                                 CFlowGraphNode *pfgNode,
                                                 const EMnemonic newInstMnemonic,
                                                 const SOperand *pNewInstOperand1,
                                                 const SOperand *pNewInstOperand2,
                                                 const SOperand *pNewInstOperand3)
{
    *ppInstruction = NULL;
    CMIEInstruction *pNewInst = CMIEInstructionFactory::CreateInstruction(
        newInstMnemonic,this);
    CHECK_ALLOC(pNewInst);
    TResult res;
    res = m_flowGraph.AddInstructionAfter(pfgNode, pNewInst);
    CHECK_TRESULT(res);
    if (pNewInst->IsBranch())
    {
        IInstruction *pTargetInst = pNewInst->SetBranchTarget(
            pNewInstOperand1->val.pTarget);
        if (NULL == pTargetInst)
        {
            return MRTE_ERROR_OPERAND_MISMATCH;
        }
    }
    else
    {
        pNewInst->InternalSetOperand(pNewInstOperand1, pNewInstOperand2, pNewInstOperand3);
    }
    *ppInstruction = pNewInst;
    return MRTE_RESULT_OK;
}

TInstructionListIterator* CMethodInstrumentor::GetInstructionListIterator(
    EInstructionIteratorType type)
{
    TResult res;
    if (!m_bFlowGraphInitialized)
    {
        // build the flow-graph
        res = m_pCodec->Decode(&m_flowGraph);
        if (MRTE_RESULT_OK != res)
        {
            return NULL;
        }
        m_bFlowGraphInitialized = true;
    }
    TInstructionListIterator *iter = m_flowGraph.GetInstructionListIterator(type);
    return iter;
}

//
// Add protected block pseudo-instructions to the flow-graph
//
TResult CMethodInstrumentor::BindTryFinallyBlock(CMIEInstruction* pTry, 
                                                 CMIEInstruction* pEndTry, 
                                                 CMIEInstruction* pFinally, 
                                                 CMIEInstruction* pEndFinally)
{
    //TEST
    if (pTry->IsReadOnly()
        || pEndFinally->IsReadOnly()
        || pFinally ->IsReadOnly()
        || pEndFinally->IsReadOnly())
    {
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }
    TResult res;
    res = ValidateTryFinallyBlockInstOrder(pTry, pEndTry, pFinally, pEndFinally);
    CHECK_TRESULT(res);

    // find the real instructions for which the pseudo-instructions are created
    CMIEInstruction* pRealTry = FindRealInstAfter(pTry);
    CMIEInstruction* pRealEndTry = FindRealInstBefore(pEndTry);
    CMIEInstruction* pRealFinally = FindRealInstAfter(pFinally);
    CMIEInstruction* pRealEndFinally = FindRealInstBefore(pEndFinally);

    // add .try
    CMIEInstruction *pPseudoTry = CMIEInstructionFactory::CreateInstruction(
        MNM_PSEUDO_TRY, this);
    CHECK_ALLOC(pPseudoTry);
    res = m_flowGraph.AddInstructionBefore(pTry->GetFlowGraphNode(), pPseudoTry);
    CHECK_TRESULT(res);
    SOperand opTryRealInst;
    opTryRealInst.type = OT_TARGET;
    opTryRealInst.val.pTarget = pRealTry;

    // add .endtry
    CMIEInstruction *pPseudoEndTry = CMIEInstructionFactory::CreateInstruction(
        MNM_PSEUDO_END_TRY, this);
    CHECK_ALLOC(pPseudoEndTry);
    res = m_flowGraph.AddInstructionAfter(pEndTry->GetFlowGraphNode(), pPseudoEndTry);
    CHECK_TRESULT(res);
    SOperand opEndTryRealInst;
    opEndTryRealInst.type = OT_TARGET;
    opEndTryRealInst.val.pTarget = pRealEndTry;

    // add .finally
    CMIEInstruction *pPseudoFinally = CMIEInstructionFactory::CreateInstruction(
        MNM_PSEUDO_FINALLY, this);
    CHECK_ALLOC(pPseudoFinally);
    res = m_flowGraph.AddInstructionBefore(pFinally->GetFlowGraphNode(), pPseudoFinally);
    CHECK_TRESULT(res);
    SOperand opFinallyRealInst;
    opFinallyRealInst.type = OT_TARGET;
    opFinallyRealInst.val.pTarget = pRealFinally;

    // add .endfinally
    CMIEInstruction *pPseudoEndFinally = CMIEInstructionFactory::CreateInstruction(
        MNM_PSEUDO_END_FINALLY, this);
    CHECK_ALLOC(pPseudoEndFinally);
    res = m_flowGraph.AddInstructionAfter(pEndFinally->GetFlowGraphNode(), pPseudoEndFinally);
    CHECK_TRESULT(res);
    SOperand opEndFinallyRealInst;
    opEndFinallyRealInst.type = OT_TARGET;
    opEndFinallyRealInst.val.pTarget = pRealEndFinally;

    // update pseudo-instruction operands
    SOperand op2;
    op2.type = OT_TARGET;
    SOperand op3;
    op3.type = OT_TARGET;
    SOperand op4;

    // .try operands: first real instruction, .endtry, .handler
    op2.val.pTarget = pPseudoEndTry;
    op3.val.pTarget = pPseudoFinally;
    pPseudoTry->InternalSetOperand(&opTryRealInst, &op2, &op3);

    // .endtry operands: last real instruction, .try, .handler
    op2.val.pTarget = pPseudoTry;
    op3.val.pTarget = pPseudoFinally;
    pPseudoEndTry->InternalSetOperand(&opEndTryRealInst, &op2, &op3);

    // .handler operands: first real instruction, .endhandler, .try, exception type
    op2.val.pTarget = pPseudoEndFinally;
    op3.val.pTarget = pPseudoTry;
    // Java: exception type is a constant-pool index
#ifdef MARTINI_JIE
    op4.type = OT_JAVA_CP_INDEX;
    op4.val.cpIndex = 0; // 0 means "catch all" and used for finally blocks
#endif
#ifdef MARTINI_CIE
    //TODO: insert correct token value for .NET
#error
#endif
    pPseudoFinally->InternalSetOperand(&opFinallyRealInst, &op2, &op3, &op4);

    // .endhandler operands: last real instruction, .handler
    op2.val.pTarget = pPseudoFinally;
    pPseudoEndFinally->InternalSetOperand(&opEndFinallyRealInst, &op2);
    
    return MRTE_RESULT_OK;
}

TResult CMethodInstrumentor::ValidateTryFinallyBlockInstOrder(
    CMIEInstruction* pTry, 
    CMIEInstruction* pEndTry, 
    CMIEInstruction* pFinally, 
    CMIEInstruction* pEndFinally)
{
    // find the ordinal place of each instruction within the flow-graph
    TInstructionListIterator *pIter = m_flowGraph.GetInstructionListIterator(
        IET_ALL_INSTRUCTIONS);
    IInstruction *pInst;
    int i = 0;
    int iTryInd = -1;
    int iEndTryInd = -1;
    int iFinallyInd = 1;
    int iEndFinallyInd = -1;
    while (pIter->HasNext())
    {
        pInst = pIter->GetNext();
        if (pInst == pTry)
        {
            iTryInd = i;
        }
        if (pInst == pEndTry)
        {
            iEndTryInd = i;
        }
        if (pInst == pFinally)
        {
            iFinallyInd = i;
        }
        if (pInst == pEndFinally)
        {
            iEndFinallyInd = i;
        }
        i++;
    }
    if (-1 == iTryInd || -1 == iEndTryInd || -1 == iFinallyInd || -1 == iEndFinallyInd)
    {
        // at least one of the instructions does not exist in the flow-graph
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }
    // verify that pEndTry succeeds pTry and that pEndFinally succeeds pFinally
    if (iEndTryInd < iTryInd || iEndFinallyInd < iFinallyInd)
    {
        return MRTE_ERROR_ILLEGAL_TRY_BLOCK;
    }
    // verify that the try and finally blocks do not cross each other
    if (iEndTryInd > iFinallyInd && iEndTryInd < iEndFinallyInd)
    {
        // the try block is either nested within the finally block or crosses it
        return MRTE_ERROR_ILLEGAL_TRY_BLOCK;
    }
    if (iEndFinallyInd > iTryInd && iEndFinallyInd < iEndTryInd)
    {
        // the finally block is either nested within the try block or crosses it
        return MRTE_ERROR_ILLEGAL_TRY_BLOCK;
    }

    return MRTE_RESULT_OK;
}

TResult CMethodInstrumentor::ApplyInstrumentation()
{
    TResult res = m_pCodec->Encode(m_flowGraph);
    CHECK_TRESULT(res);
    return MRTE_RESULT_OK;
    //TODO: implmenet: destroy the flow-graph and do not allow any inspection/iteration/debug print
}

CMIEInstruction* CMethodInstrumentor::FindRealInstBefore(CMIEInstruction *pInst)
{
    if (!pInst->IsPseudo())
    {
        return pInst;
    }
    CMIEInstruction* pRealInst = m_flowGraph.GetPrecedingRealInstruction(pInst);
    return pRealInst;
}

CMIEInstruction* CMethodInstrumentor::FindRealInstAfter(CMIEInstruction *pInst)
{
    if (!pInst->IsPseudo())
    {
        return pInst;
    }
    CMIEInstruction* pRealInst = m_flowGraph.GetSucceedingRealInstruction(pInst);
    return pRealInst;
}
