/*****************************************************************************
 * Copyright (c) 1997, 2010 Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$
 *****************************************************************************/

#ifdef EM64T_ARCH
#pragma runtime_checks( "", off )
#endif

#include <memory.h>
#include <string.h>

#include "DataManager.h"
#include "DataManagerEvents.h"
#include "EventManager.h"
#include "JavaInstrumentorManager.h"
#include "OSA.h"
#include "LogAssert.h"
#include "JpiGlobals.h"

using namespace Martini;
using namespace Martini::JPI;
using namespace Martini::Infrastructure;
using namespace Martini::RTUtil;

using namespace Martini::OSA;

#define SIZE_OF_ARRAY(x) sizeof(x)/sizeof(*x)

const MPI::TId ILLEGAL_METHOD_ID = 0x80000001;

struct TProblematicMethodInfo
{
    const char *szClassName;
    const char *szMethodName;
    const char *szMethodSig;
    TProblematicMethodInfo(const char *i_szClassName, const char *i_szMethodName,
        const char *i_szMethodSig)
        : szClassName(i_szClassName), szMethodName(i_szMethodName), szMethodSig(i_szMethodSig)
    {}
};

//
// Cosnstructor
//
CDataManager::CDataManager(MPI::TId moduleId)
: m_moduleId(moduleId), m_pClassPrepareEvent(NULL),
    m_pMethodJitLoadedEvent(NULL), m_pThreadStartEvent(NULL), m_pThreadEndEvent(NULL),
    m_pThreadInfoManager(NULL), m_pObjectInfoManager(NULL), m_pVMInitEvent(NULL),
    m_pGCStartEvent(NULL), m_pVMShutdownEvent(NULL), m_pcsClassInfoDb(NULL)
{
    // Set module info array
    m_DatabaseModuleInfo[MODULE_INTERPRETED].szName.Set(MODULE_INTERPRETED_NAME);
    m_DatabaseModuleInfo[MODULE_INTERPRETED].id = MODULE_INTERPRETED;

    m_DatabaseModuleInfo[MODULE_INLINED].szName.Set(MODULE_INLINED_NAME);
    m_DatabaseModuleInfo[MODULE_INLINED].id = MODULE_INLINED;

    char szModuleName[MAX_MRTE_STRING_LENGTH];
    unsigned int uiRequiredLength;
    szModuleName[0] = '\0';
    TResult retVal = GetExecutableFileName(szModuleName, MAX_MRTE_STRING_LENGTH,
        &uiRequiredLength);

    if(MRTE_FAILED(retVal))
    {
        szModuleName[0] = '\0';
    }

    // Initialize databases
    m_DatabaseModuleInfo[MODULE_JIT].szName.Set(szModuleName);
    m_DatabaseModuleInfo[MODULE_JIT].id = MODULE_JIT;
    m_pInitialModule = &m_DatabaseModuleInfo[MODULE_INLINED];

    m_pDatabaseMethodInfo = new TMethodInfoDB;
    m_pDatabaseMethodInfo->Init(false, NULL, MRTE_METHOD_DB_SIZE, MRTE_DB_SHIFT);
    m_pJvmpiToMpiMethodId = new TJvmpiToMpiIdHash;
    m_pJvmpiToMpiMethodId->Init(false, 0, MRTE_METHOD_DB_SIZE, MRTE_DB_SHIFT);
    m_pDatabaseClassInfo = new TClassInfoDB;
    m_pDatabaseClassInfo->Init(false, 0, MRTE_CLASS_DB_SIZE, MRTE_DB_SHIFT);
    m_pClassNameToId = new TStringToIdHash;
    m_pClassNameToId->Init();
    m_pDatabaseClassesWithCompleteInfo = new TStringToIdHash();
    m_pDatabaseClassesWithCompleteInfo->Init();

    // Initialize id allocators
    m_pClassIdAllocator = new CIdAllocator(0x80000001);
    m_pMethodIdAllocator = new CIdAllocator(0x40000002); // Was 0x80000002 but changed
                                                         // so that method ids can fit within
                                                         // a signed int32 (a current limitation
                                                         // of the TPTP UI data models).
    m_constantModuleId = MODULE_UNINITIALIZED;

    // Initialize critical sections
    m_pcsClassInfoDb = CreateThreadSync();
    if (!m_pcsClassInfoDb)
    {
        MARTINI_ERROR("CDataManager", "Unable to initialize class database critical section");
    }
}

//<summary> destructor </summary>
CDataManager::~CDataManager(void)
{
    if (m_pClassPrepareEvent)
    {
        delete m_pClassPrepareEvent;
    }
    if (m_pMethodJitLoadedEvent)
    {
        delete m_pMethodJitLoadedEvent;
    }
    if (m_pClassIdAllocator)
    {
        delete m_pClassIdAllocator;
    }

    if (m_pMethodIdAllocator)
    {
        delete m_pMethodIdAllocator;
    }

    if (m_pJvmpiToMpiMethodId)
    {
        //TODO: iterate over the hash and delete all the allocated data. Since it is done only
        // at the end of the process it is not critical to do so
        delete m_pJvmpiToMpiMethodId;
    }

    if (m_pThreadStartEvent)
    {
        delete m_pThreadStartEvent;
    }

    if (m_pThreadEndEvent)
    {
        delete m_pThreadEndEvent;
    }

    if (m_pVMInitEvent)
    {
        delete m_pVMInitEvent;
    }

    if (m_pVMShutdownEvent)
    {
        delete m_pVMShutdownEvent;
    }

    if (m_pGCStartEvent)
    {
        delete m_pGCStartEvent;
    }

    if (m_pThreadInfoManager)
    {
        delete m_pThreadInfoManager;
    }

    if (m_pObjectInfoManager)
    {
        delete m_pObjectInfoManager;
    }

    if (m_pcsClassInfoDb)
    {
        m_pcsClassInfoDb->Destroy();
    }

}

TResult CDataManager::GetModuleInfo(MPI::TId clientId,
                                    MPI::TId moduleId,
                                    MPI::BitSet requestedDataTypes,
                                    MPI::SModuleInfo *pData)
{
    TResult iRetVal = m_pParamChecker->CheckClientAndDataAndPtr(clientId, requestedDataTypes,
        pData);
    if (MRTE_FAILED(iRetVal))
    {
        return iRetVal;
    }

    pData->validData = MPI::DR_NONE;
    SDMModuleInfo *pModuleData = NULL;
    iRetVal = MRTE_RESULT_OK;

    if (moduleId > MAX_JAVA_MODULE_ID)
    {
        MARTINI_INFORMATIVE("CDataManager", 0, false, "Module data does not exist in database");
        return MRTE_ERROR_FAIL;
    }
    pModuleData = &m_DatabaseModuleInfo[moduleId];
    if (requestedDataTypes & MPI::DR_MODULE_NAME)
    {
        pData->validData |= MPI::DR_MODULE_NAME;
        pData->szModuleName = pModuleData->szName.Get();
    }

    return iRetVal;
}

TResult CDataManager::GetClassInfo(MPI::TId clientId,
                                   MPI::TId classId,
                                   MPI::BitSet requestedDataTypes,
                                   MPI::SClassInfo *pData)
{
    TResult iRetVal = m_pParamChecker->CheckClientAndDataAndPtr(clientId, requestedDataTypes,
        pData);
    if (MRTE_FAILED(iRetVal))
    {
        return iRetVal;
    }

    pData->validData = MPI::DR_NONE;
    SDMClassInfo  *pClassData = NULL;
    iRetVal = MRTE_RESULT_OK;

    pClassData = m_pDatabaseClassInfo->Get(classId);
    if (pClassData == NULL)
    {
        MARTINI_INFORMATIVE1("CDataManager", 0, false,
            "information for class %u does not exist in database", (U32)classId);
        return MRTE_ERROR_FAIL;
    }
    if (requestedDataTypes & MPI::DR_CLASS_NAME)
    {
        pData->validData |= MPI::DR_CLASS_NAME;
        pData->szClassName = pClassData->szName.Get();
    }
    if (requestedDataTypes & MPI::DR_SOURCE_FILE_NAME)
    {
        pData->validData |= MPI::DR_SOURCE_FILE_NAME;
        pData->szSourceFileName = pClassData->szSrcFileName.Get();
    }
    if (requestedDataTypes & MPI::DR_JAVA_NATIVE_CLASS_NAME)
    {
        pData->validData |= MPI::DR_JAVA_NATIVE_CLASS_NAME;
        pData->szJavaClassName = pClassData->szNativeName.Get();
    }
    return iRetVal;
}

TResult CDataManager::GetMethodInfo(MPI::TId clientId,
                                    MPI::TId methodId,
                                    MPI::BitSet requestedDataTypes,
                                    MPI::SMethodInfo *pData)
{
    TResult iRetVal = m_pParamChecker->CheckClientAndDataAndPtr(clientId, requestedDataTypes,
        pData);
    if (MRTE_FAILED(iRetVal))
    {
        return iRetVal;
    }

    pData->validData = MPI::DR_NONE;
    SDMMethodInfo *pMethodData = NULL;
    SDMClassInfo  *pClassData = NULL;
    SDMModuleInfo *pModuleData = NULL;
    iRetVal = MRTE_RESULT_OK;

    pMethodData = m_pDatabaseMethodInfo->Get(methodId);
    if (pMethodData == NULL)
    {
        MARTINI_INFORMATIVE1("CDataManager", 0, false,
            "information for method %u does not exist in database", (U32)methodId);
        return MRTE_ERROR_FAIL;
    }
    pClassData = pMethodData->pClassInfo;
    pModuleData = pMethodData->pModuleInfo;
    MARTINI_ASSERT("CDataManager", pClassData, "Method data does not contain class data");
    if (NULL == pClassData)
    {
        MARTINI_INFORMATIVE1("CDataManager", 0, false,
            "no class information associated with method %u", (U32)methodId);
    }
    MARTINI_ASSERT("CDataManager", pModuleData, "Method data does not contain module data");

    if (requestedDataTypes & MPI::DR_METHOD_NAME)
    {
        pData->validData |= MPI::DR_METHOD_NAME;
        pData->szMethodName = pMethodData->szFriendlyName.Get();
    }
    if (requestedDataTypes & MPI::DR_METHOD_SIGNATURE)
    {
        pData->validData |= MPI::DR_METHOD_SIGNATURE;
        pData->szMethodSig = pMethodData->szFriendlySignature.Get();
    }
    if (requestedDataTypes & MPI::DR_JAVA_NATIVE_METHOD_NAME)
    {
        pData->validData |= MPI::DR_JAVA_NATIVE_METHOD_NAME;
        pData->szJavaMethodName = pMethodData->szMethodOriginalName.Get();
    }
    if (requestedDataTypes & MPI::DR_JAVA_NATIVE_METHOD_SIGNATURE)
    {
        pData->validData |= MPI::DR_JAVA_NATIVE_METHOD_SIGNATURE;
        pData->szJavaMethodSig = pMethodData->szMethodOriginalSignature.Get();
    }
    if (requestedDataTypes & MPI::DR_SOURCE_FILE_NAME)
    {
        pData->validData |= MPI::DR_SOURCE_FILE_NAME;
        pData->szMethodSourceFileName = pClassData->szSrcFileName.Get();
    }
    if (requestedDataTypes & MPI::DR_METHOD_LINE_NUMBERS)
    {
        pData->validData |= MPI::DR_METHOD_LINE_NUMBERS;
        pData->methodLineNumbers = pMethodData->lineNumbers;
    }
    if (requestedDataTypes & MPI::DR_CLASS_ID)
    {
        pData->validData |= MPI::DR_CLASS_ID;
        pData->classId = pClassData->id;
    }
    if (requestedDataTypes & MPI::DR_MODULE_ID)
    {
        pData->validData |= MPI::DR_MODULE_ID;
        pData->moduleId = pModuleData->id;
    }
    if (requestedDataTypes & MPI::DR_MODULE_NAME)
    {
        pData->validData |= MPI::DR_MODULE_NAME;
        pData->szModuleName = pModuleData->szName.Get();
    }
    if (requestedDataTypes & MPI::DR_CLASS_NAME)
    {
        pData->validData |= MPI::DR_CLASS_NAME;
        pData->szClassName = pClassData->szName.Get();
    }
    if (requestedDataTypes & MPI::DR_JAVA_NATIVE_CLASS_NAME)
    {
        pData->validData |= MPI::DR_JAVA_NATIVE_CLASS_NAME;
        pData->szJavaClassName = pClassData->szNativeName.Get();
    }
    if (requestedDataTypes & MPI::DR_MANAGED_TO_SRC_LINE_MAP)
    {
        iRetVal = m_pJavaInterface->GetLineNumberTable(
            &(pData->managedToSrcLineMap), pMethodData->vmId);
        if (MRTE_SUCCEEDED(iRetVal))
        {
            pData->validData |= MPI::DR_MANAGED_TO_SRC_LINE_MAP;
        }
    }

    return iRetVal;
}

TResult CDataManager::GetObjectInfo(MPI::TId clientId,
                                    MPI::TId objectId,
                                    MPI::BitSet requestedDataTypes,
                                    MPI::SObjectDataRequest *pData)
{
    if (!CJpiGlobals::Instance()->IsThreadMpiSafe())
    {
        return MRTE_ERROR_FAIL;
    }

    pData->validData = MPI::DR_NONE;

    bool bAttached;
    JNIEnv *pJNIEnv;

    TResult res = m_pJavaInterface->AttachCurrentThread(&pJNIEnv, &bAttached);
    if (MRTE_FAILED(res))
    {
        MARTINI_INFORMATIVE("CDataManager", 0, false,
            "[GetObjectInfo] failed to attach the calling thread to the VM");
        return MRTE_ERROR_FAIL;
    }

    if (MPI::DR_OBJECT_INFO & requestedDataTypes)
    {
        MPI::SObjectInfo *pObjectInfo = GetObjectInfoManager()->GetInfo(objectId, pJNIEnv);
        if (pObjectInfo != NULL)
        {
            pData->objectInfo = *pObjectInfo;
            pData->validData |= MPI::DR_OBJECT_INFO;
        }
    }

    if ((MPI::DR_MONITOR_OWNER_THREAD_ID | MPI::DR_MONITOR_OWN_WAITERS
        | MPI::DR_MONITOR_NOTIFY_WAITERS) & requestedDataTypes)
    {
        jobject jniRef = GetObjectInfoManager()->GetJniRefFromId(objectId, pJNIEnv);
        if (jniRef != 0)
        {
            res = m_pJavaInterface->GetMonitorUsage(&(pData->monOwnerThreadId),
                &(pData->monOwnWaiters), &(pData->monNotifyWaiters),
                requestedDataTypes, jniRef);
            if (MRTE_SUCCEEDED(res))
            {
                pData->validData |= requestedDataTypes;
            }
        }
    }

    if (bAttached)
    {
        m_pJavaInterface->DetachCurrentThread();
    }

    return MRTE_RESULT_OK;
}

TResult CDataManager::GetObjectReferences(MPI::TId clientId,
                                          MPI::SObjectReferenceInfo *pData)
{
    return MRTE_ERROR_NOT_IMPLEMENTED;
#if 0
    pData->uiActualSize = 0;
    bool bAttached;
    JNIEnv *pJNIEnv;
    if (!CJpiGlobals::Instance()->bJvmInitDone)
    {
        return MRTE_ERROR_PHASE_FAILURE;
    }

    TResult res = m_pJavaInterface->AttachCurrentThread(&pJNIEnv, &bAttached);
    if (MRTE_FAILED(res))
    {
        MARTINI_INFORMATIVE("CDataManager", 0, false,
            "[GetObjectReferences] failed to attach the calling thread to the VM");
        return MRTE_ERROR_FAIL;
    }

    //
    // Suspend the VM, retrieve object reference information, and add the information for
    // these objects to the Object Info Manager. This will make sure that the client can
    // request object information even if the object was freed by the Garbage Collector
    //
    TTagVector tagVector;
    res = m_pJavaInterface->GetObjectReferences(pData, &tagVector);
/*
    if (MRTE_SUCCEEDED(res))
    {
        GetThreadInfoManager()->SuspendAllThreads();
        // Find the objects for which we do not have information
        CObjectInfoManager *pObjectInfoManager = GetObjectInfoManager();
        TTagVector tagsWithoutInfo;
        size_t i;
        for (i = 0; i < tagVector.Size(); ++i)
        {
            if (NULL ==
                pObjectInfoManager->GetTagInfo(tagVector.GetAt(i))->jobjectInfo.pObjectInfo)
            {
                tagsWithoutInfo.Push(tagVector.GetAt(i));
            }
        }

        if (tagsWithoutInfo.Size() > 0)
        {
            // Get information for missing objects
            TTagVector outTagsVector;
            TObjectInfoVector objVector;
            res = m_pJavaInterface->GetObjectInfoFromTags(tagsWithoutInfo, &objVector,
                &outTagsVector);
            if (MRTE_SUCCEEDED(res))
            {
                size_t objCount = objVector.Size();
                for (i = 0; i < objCount; ++i)
                {
                    SJvmtiObjectTag *pTagInfo = pObjectInfoManager->GetTagInfo(
                        outTagsVector.GetAt(i));
                    pTagInfo->jobjectInfo.pObjectInfo = objVector.GetAt(i);
                }
            }
        }
        GetThreadInfoManager()->ResumeAllThreads();
    }

*/
    if (bAttached)
    {
        m_pJavaInterface->DetachCurrentThread();
    }

    return res;
#endif
}

TResult CDataManager::GetAllThreadsInfo(MPI::TId clientId,
                                        U32 maxFrameCount,
                                        MPI::BitSet requestedDataTypes,
                                        MPI::SThreadInfoArray *pData)
{
    pData->uiActualSize = 0;
    bool bAttached;
    JNIEnv *pJNIEnv;
    if (!CJpiGlobals::Instance()->bJvmInitDone)
    {
        return MRTE_ERROR_PHASE_FAILURE;
    }

    TResult res = m_pJavaInterface->AttachCurrentThread(&pJNIEnv, &bAttached);
    if (MRTE_FAILED(res))
    {
        MARTINI_INFORMATIVE("CDataManager", 0, false,
            "[GetAllThreadsInfo] failed to attach the calling thread to the VM");
        return MRTE_ERROR_FAIL;
    }

    jint framesToGet;
    if (requestedDataTypes & MPI::DR_VM_RELATIVE_STACK_TRACE)
    {
        framesToGet =  maxFrameCount > 0 ? maxFrameCount : 1;
    }
    else
    {
        framesToGet = 1;
    }

    // Bugzilla 184425: In JRockit, the GetAllStackTraces JVMTI function has a bug. It does
    // not work correctly when max_frame_count = 0. Therefore, it is always invoked with a
    // frame count of at least 1.

    // Get threads information from the JVM
    jvmtiStackInfo *stackInfoArray;
    jint threadCount;

    res = m_pJavaInterface->GetAllStackTraces(&stackInfoArray, &threadCount, framesToGet);
    if (MRTE_SUCCEEDED(res))
    {
        int threadsToProcess = ((U32)threadCount > pData->uiSize) ? pData->uiSize : threadCount;
        int stackInfoIndex;
        for (stackInfoIndex = 0; stackInfoIndex < threadsToProcess; ++stackInfoIndex)
        {
            jvmtiStackInfo *pStackInfo = &stackInfoArray[stackInfoIndex];
            MPI::TId threadId = GetThreadInfoManager()->GetThreadId(pStackInfo->thread);
            if (threadId != 0)
            {
                if (GetThreadInfoManager()->GetThreadType(threadId) != STlsThreadInfo::JAVA)
                {
                    // Ignore external threads
                    continue;
                }
                int mpiThreadInfoIndex = pData->uiActualSize;
                pData->pEntries[mpiThreadInfoIndex].threadId = threadId;
                pData->uiActualSize++;
                pData->pEntries[mpiThreadInfoIndex].threadInfo.state =
                    m_pJavaInterface->VmThreadStateToMpiThreadState(pStackInfo->state);
                pData->pEntries[mpiThreadInfoIndex].threadInfo.validData |=
                    MPI::DR_THREAD_STATE;
                if (requestedDataTypes & MPI::DR_VM_RELATIVE_STACK_TRACE)
                {
                    ConvertFrameData(pData->pEntries[mpiThreadInfoIndex].threadInfo.vmOffsetStack.pStackEntries,
                        pStackInfo->frame_buffer, pStackInfo->frame_count);
                    pData->pEntries[mpiThreadInfoIndex].threadInfo.vmOffsetStack.uiActualSize =
                        pStackInfo->frame_count;
                    pData->pEntries[mpiThreadInfoIndex].threadInfo.validData |=
                        MPI::DR_VM_RELATIVE_STACK_TRACE;
                }
                if (requestedDataTypes & MPI::DR_THREAD_INFO)
                {
                    MPI::SThreadInfo threadInfo;
                    TResult r = GetThreadInfoManager()->GetThreadInfo(&threadInfo, threadId);
                    if (MRTE_SUCCEEDED(r))
                    {
                        if (requestedDataTypes & MPI::DR_THREAD_NAME)
                        {
                            pData->pEntries[mpiThreadInfoIndex].threadInfo.validData |=
                                MPI::DR_THREAD_NAME;
                            pData->pEntries[mpiThreadInfoIndex].threadInfo.szName =
                                threadInfo.szName;
                        }
                        if (requestedDataTypes & MPI::DR_THREAD_GROUP)
                        {
                            pData->pEntries[mpiThreadInfoIndex].threadInfo.validData |=
                                MPI::DR_THREAD_GROUP;
                            pData->pEntries[mpiThreadInfoIndex].threadInfo.szGroupName =
                                threadInfo.szGroupName;
                        }
                        if (requestedDataTypes & MPI::DR_THREAD_PARENT_GROUP)
                        {
                            pData->pEntries[mpiThreadInfoIndex].threadInfo.validData |=
                                MPI::DR_THREAD_PARENT_GROUP;
                            pData->pEntries[mpiThreadInfoIndex].threadInfo.szParentGroupName =
                                threadInfo.szParentGroupName;
                        }
                    }
                }
                if (requestedDataTypes & MPI::DR_CURRENT_MONITOR)
                {
                    TResult r = GetThreadCurrentMonitor(
                        &(pData->pEntries[mpiThreadInfoIndex].threadInfo.currentMonitorId),
                        pStackInfo->thread);
                    if (MRTE_SUCCEEDED(r))
                    {
                        pData->pEntries[mpiThreadInfoIndex].threadInfo.validData |=
                            MPI::DR_CURRENT_MONITOR;
                    }
                }

                if (requestedDataTypes & MPI::DR_OBJECT_ID)
                {
                    jlong tag;
                    TResult r = GetObjectInfoManager()->GetOrCreateTag(
                        &(pData->pEntries[mpiThreadInfoIndex].threadInfo.threadObjectId), &tag,
                        pStackInfo->thread, pJNIEnv, true /* bSaveInDb */);
                    if (MRTE_SUCCEEDED(r))
                    {
                        pData->pEntries[mpiThreadInfoIndex].threadInfo.validData |=
                            MPI::DR_OBJECT_ID;
                    }
                }

            }
            else
            {
                MARTINI_INFORMATIVE1("CDataManager", 0, false,
                    "[GetAllThreadsInfo] unknown thread %p", pStackInfo->thread);
            }
        }
        m_pJavaInterface->DeallocateVmBuffer((unsigned char *)stackInfoArray);
    }

    if (bAttached)
    {
        m_pJavaInterface->DetachCurrentThread();
    }

    return res;
}

TResult CDataManager::GetThreadInfo(MPI::TId clientId,
                                    MPI::TId threadId,
                                    MPI::BitSet requestedDataTypes,
                                    MPI::SThreadInfo *pData,
                                    MPI::TDotNetUnmanagedStackWalkCallBack pfnStackWalk)
{
    if (!CJpiGlobals::Instance()->IsThreadMpiSafe())
    {
        return MRTE_ERROR_FAIL;
    }

    TResult iRetVal = m_pParamChecker->CheckClientAndDataAndPtr(clientId, requestedDataTypes,
        pData);
    if (MRTE_FAILED(iRetVal))
    {
        return iRetVal;
    }

    pData->validData = MPI::DR_NONE;

    //TODO: need to support "current thread" for DR_THREAD_INFO
    if (requestedDataTypes & MPI::DR_THREAD_INFO)
    {
        MPI::SThreadInfo threadInfo;
        TResult iRes = m_pThreadInfoManager->GetThreadInfo(&threadInfo, threadId);
        if (MRTE_SUCCEEDED(iRes))
        {
            if (requestedDataTypes & MPI::DR_THREAD_NAME)
            {
                pData->validData |= MPI::DR_THREAD_NAME;
                pData->szName = threadInfo.szName;
            }
            if (requestedDataTypes & MPI::DR_THREAD_GROUP)
            {
                pData->validData |= MPI::DR_THREAD_GROUP;
                pData->szGroupName = threadInfo.szGroupName;
            }
            if (requestedDataTypes & MPI::DR_THREAD_PARENT_GROUP)
            {
                pData->validData |= MPI::DR_THREAD_PARENT_GROUP;
                pData->szParentGroupName = threadInfo.szParentGroupName;
            }
        }
    }

    if ((requestedDataTypes & MPI::DR_THREAD_CPU_TIME) && (0 == threadId))
    {
        TResult iRes = m_pJavaInterface->GetCurrentThreadCpuTime(&pData->uiCpuNanos);
        if (MRTE_SUCCEEDED(iRes))
        {
            pData->validData |= MPI::DR_THREAD_CPU_TIME;
        }
    }

    if ((requestedDataTypes & MPI::DR_THREAD_ELAPSED_TIME) && (0 == threadId))
    {
        pData->uiElapsedNanos = GetTimeStamp();
        pData->validData |= MPI::DR_THREAD_ELAPSED_TIME;
    }

    if ((requestedDataTypes & MPI::DR_VM_RELATIVE_STACK_TRACE))
    {
        if (NULL == pData->vmOffsetStack.pStackEntries)
        {
            return MRTE_ERROR_NULL_PTR;
        }

        jint bufferSize = pData->vmOffsetStack.uiSize;
        jvmtiFrameInfo *pJvmFrameBuffer = new jvmtiFrameInfo[bufferSize];
        if (NULL == pJvmFrameBuffer)
        {
            return MRTE_ERROR_OUT_OF_MEMORY;
        }
        jint frameCount = 0;
        TResult iRes = m_pThreadInfoManager->GetStack(pJvmFrameBuffer, &frameCount, threadId,
            bufferSize);
        if (MRTE_FAILED(iRes))
        {
            return iRes;
        }
        ConvertFrameData(pData->vmOffsetStack.pStackEntries, pJvmFrameBuffer, frameCount);
        pData->vmOffsetStack.uiActualSize = frameCount;
        if (frameCount > bufferSize)
        {
            return MRTE_ERROR_BUFFER_TOO_SHORT;
        }

        // Success
        pData->validData |= MPI::DR_VM_RELATIVE_STACK_TRACE;
    }

    if (requestedDataTypes & MPI::DR_THREAD_STATE)
    {
        TResult iRes = m_pThreadInfoManager->GetState(&pData->state, threadId);
        if (MRTE_SUCCEEDED(iRes))
        {
            pData->validData |= MPI::DR_THREAD_STATE;
        }
    }

    if (requestedDataTypes & MPI::DR_CURRENT_MONITOR)
    {
        jthread threadObj;
        TResult iRes = m_pThreadInfoManager->GetThreadObject(&threadObj, threadId);
        if (MRTE_SUCCEEDED(iRes))
        {
            iRes = GetThreadCurrentMonitor(&pData->currentMonitorId, threadObj);
            if (MRTE_SUCCEEDED(iRes))
            {
                pData->validData |= MPI::DR_CURRENT_MONITOR;
            }
        }
    }

    if (requestedDataTypes & MPI::DR_OBJECT_ID)
    {
        jthread threadObj;
        TResult iRes = m_pThreadInfoManager->GetThreadObject(&threadObj, threadId);
        if (MRTE_SUCCEEDED(iRes))
        {

            JNIEnv *pJNIEnv;
            if (MRTE_SUCCEEDED(m_pJavaInterface->GetJNIEnv(&pJNIEnv)))
            {
                jlong tag;
                iRes = GetObjectInfoManager()->GetOrCreateTag(&pData->threadObjectId, &tag,
                    threadObj, pJNIEnv, true /* bSaveInDb */);
                if (MRTE_SUCCEEDED(iRes))
                {
                    pData->validData |= MPI::DR_OBJECT_ID;
                }
            }
            else
            {
                MARTINI_INFORMATIVE("CDataManager", 0, false,
                    "[GetThreadInfo] (DR_OBJECT_ID) failed to get the JNI env of the "
                    "calling thread");
            }
        }
    }

    if (requestedDataTypes == pData->validData)
    {
        // All requested items were retrieved
        iRetVal = MRTE_RESULT_OK;
    }
    else
    {
        // Not all requested items were retrieved
        if (0 != pData->validData)
        {
            iRetVal = MRTE_ERROR_PARTIAL_INFO;
        }
        else
        {
            iRetVal = MRTE_ERROR_FAIL;
        }
    }
    return iRetVal;
}

//
// Receive method information from JVMPI. This function calls JVMPI to request
// a Class Load event. The event should arrive before the request returns.
// This function does nothing when working with JVMTI.
//
// Parameters:
//      methodId    [in]    : the method for which to retrieve information
//
void CDataManager::RequestMethodInfoFromStdPI(jmethodID methodId)
{
    m_pJavaInterface->RequestClassLoadEvent((UIOP)methodId);
    // class_load should had arrived by now
    return;
}


//
// JVMPI: register to Class Load event
// JVMTI: register to both Class Load and Class Prepare events
//
// Returns:
//      MRTE_RESULT_OK  : success
//      MRTE_ERROR_FAIL : failure
//
TResult CDataManager::RegisterClassPrepareEvent()
{
    TResult iRetVal = MRTE_RESULT_OK;
    if (NULL == m_pClassPrepareEvent)
    {
        m_pClassPrepareEvent = new CClassPrepareEventObserver(this);

        iRetVal = m_pEventManager->RegisterEventInternal(m_moduleId, *m_pClassPrepareEvent);
    }

    return iRetVal;
}

TResult CDataManager::ParseClassName(const char *szName, const char *szGenericSig, MCString *pOutName)
{
    if (!szName)
    {
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }

    TResult iRes = MRTE_RESULT_OK;
    int iPlace = 0;
    if (szName[strlen(szName) - 1] == ';')
    {
        iRes = Unmangle(szName, &iPlace, pOutName);
    }
    else
    {
        iRes = pOutName->Set(szName);
        pOutName->ReplaceAll("/", ".");
    }
    if (MRTE_FAILED(iRes))
    {
        return iRes;
    }

    if (!szGenericSig || szGenericSig[0] != '<') // class template is the first element
    {
        return MRTE_RESULT_OK;
    }


    iPlace = 0;
    int iLength = strlen(szGenericSig);

    while (iPlace < iLength)
    {
        char vcNext[2];
        vcNext[0] = szGenericSig[iPlace++];
		vcNext[1] = 0;

        if (vcNext[0] != ':')
        {
            pOutName->Append(vcNext);
        }
        else
        {
            // empty loop to clean the type
            while (szGenericSig[iPlace++] != ';');

            if (szGenericSig[iPlace] != '>')
            {
                pOutName->Append(","); // there is another param so add comma
            }
        }

        if (vcNext[0] == '>')
        {
            break;
        }
    }
    return MRTE_RESULT_OK;
}

TResult CDataManager::ParseMethodName(const char *szName, const char *szGenericSig, MCString *pOutName)
{
    if (!szName)
    {
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }

    pOutName->Append(szName);

    if (!szGenericSig || szGenericSig[0] != '<') // class template is the first element
    {
        return MRTE_RESULT_OK;
    }


    int iPlace = 0;
    int iLength = strlen(szGenericSig);

    while (iPlace < iLength)
    {
        char vcNext[2];
        vcNext[0] = szGenericSig[iPlace++];
        vcNext[1] = 0;

        if (vcNext[0] != ':')
        {
            pOutName->Append(vcNext);
        }
        else
        {
            // empty loop to clean the type
            //while (szGenericSig[iPlace++] != ';');
            MCString genType;

            iPlace--;
            Unmangle(szGenericSig, &iPlace, &genType);
            iPlace++;

            if (genType.StartsWith(" extends ") || genType.StartsWith(" &"))
            {
                pOutName->Append(genType);
            }

            if (szGenericSig[iPlace] != '>' && szGenericSig[iPlace] != ':')
            {
                pOutName->Append(","); // there is another param so add comma
            }
        }

        if (vcNext[0] == '>')
        {
            break;
        }
    }
    return MRTE_RESULT_OK;
}

TResult CDataManager::ParseMethodPrototype(const char *szPrototype, const char *szGenericSig,
            unsigned int uiFlags, MCString *pOutPrototype)
{
    TResult iRes;
    const char *szMangled = szPrototype;
    if (szGenericSig)
    {
        szMangled = szGenericSig;
        while (*szMangled != '(') // skip the method template part
        {
            szMangled++;
        }
    }

    int iOrigLength = (int)strlen(szMangled);
    MCString szArguments;
    MCString szRetVal;
    int iPtr = 0;

    szArguments.Set("(");

    while (iPtr < iOrigLength && szMangled[++iPtr] != ')')
    {
        if (iPtr != 1)
        {
            iRes = szArguments.Append(", ");
            if (MRTE_FAILED(iRes))
            {
                return iRes;
            }
        }
        iRes = Unmangle(szMangled, &iPtr, &szArguments);
        if (MRTE_FAILED(iRes))
        {
            return iRes;
        }
    }

    if (uiFlags & ACC_VARARGS)
    {
        unsigned int uiLen = szArguments.Len();
        iRes = szArguments.Erase(uiLen - 2, 2);
        if (MRTE_FAILED(iRes))
        {
            return iRes;
        }

        iRes = szArguments.Append("...");
        if (MRTE_FAILED(iRes))
        {
            return iRes;
        }
    }

    iRes = szArguments.Append(")");
    if (MRTE_FAILED(iRes))
    {
        return iRes;
    }
    ++iPtr;
    iRes = Unmangle(szMangled, &iPtr, &szRetVal);
    if (MRTE_FAILED(iRes))
    {
        return iRes;
    }
    iRes = pOutPrototype->Set("(");
    if (MRTE_FAILED(iRes))
    {
        return iRes;
    }
    iRes = pOutPrototype->Append(szRetVal);
    if (MRTE_FAILED(iRes))
    {
        return iRes;
    }
    iRes = pOutPrototype->Append(szArguments);
    if (MRTE_FAILED(iRes))
    {
        return iRes;
    }
    iRes = pOutPrototype->Append(")");
    return iRes;
}

//<summary> Change java standard field descriptor character to a readable string </summary>
//<param name='szSrc'> java field descriptor buffer</param>
//<param name='pPlace'> location of relevant character in the szSrc buffer</param>
//<param name='szDest'> buffer for the translated descriptor (output) </param>
//<returns> none </returns>
TResult CDataManager::Unmangle (const char *szSrc, int *pPlace, MCString *szDest)
{
    TResult iRes = MRTE_RESULT_OK;
    switch(szSrc[*pPlace])
	{
	case 'B':
		iRes = szDest->Append("byte");
		break;
	case 'C':
		iRes = szDest->Append("char");
		break;
	case 'D':
		iRes = szDest->Append("double");
		break;
	case 'F':
		iRes = szDest->Append("float");
		break;
	case 'I':
		iRes = szDest->Append("int");
		break;
	case 'J':
		iRes = szDest->Append("long");
		break;
	case 'S':
		iRes = szDest->Append("short");
		break;
	case 'Z':
		iRes = szDest->Append("boolean");
		break;
    case 'V':
        iRes = szDest->Append("void");
        break;
    case '*':
        iRes = szDest->Append("?");
        break;
    case ':':
        (*pPlace)++;
        if (szSrc[*pPlace] == ':')
        {
            iRes = szDest->Append(" extends ");
            (*pPlace)++;
        }
        else if (szSrc[(*pPlace) - 2] == ';')
        {
            iRes = szDest->Append(" & ");
        }
        iRes = Unmangle(szSrc, pPlace, szDest);
        break;
    case '+':
        iRes = szDest->Append("? extends ");
        (*pPlace)++;
		iRes = Unmangle(szSrc, pPlace, szDest);
        break;
    case '-':
        iRes = szDest->Append("? super ");
        (*pPlace)++;
		iRes = Unmangle(szSrc, pPlace, szDest);
        break;
	case 'L':
		(*pPlace)++;
		iRes = UnmangleName(szSrc, pPlace, szDest);
		break;
    case 'T':
        (*pPlace)++;
        iRes = UnmangleName(szSrc, pPlace, szDest);
        break;
	case '[':
		(*pPlace)++;
		iRes = Unmangle(szSrc, pPlace, szDest);
		szDest->Append("[]");
		break;
	}
    return iRes;
}


TResult CDataManager::UnmangleName(const char *szSrc, int *pPlace, MCString *szDest)
{
    TResult iRes = MRTE_RESULT_OK;
    while (szSrc[*pPlace] != ';')
    {
        char vcNext[2];
        vcNext[0] = szSrc[*pPlace];
        vcNext[1] = 0;

        if (szSrc[*pPlace] == '/')
            iRes = szDest->Append(".");
        else if (szSrc[*pPlace] == '<')
        {
            iRes = szDest->Append("<");
            (*pPlace)++;
            while (szSrc[*pPlace] != '>')
            {
                if (szSrc[*pPlace - 1] != '<')
                {
                    iRes = szDest->Append(",");
                }

                Unmangle(szSrc, pPlace, szDest);
                (*pPlace)++;
            }
            (*pPlace)--;
        }
        else
            iRes = szDest->Append(vcNext);
        (*pPlace)++;
    }
    return iRes;
}



//<summary> Set the internal modules, called by the kernel at initialization phase </summary>
//<param name='pJavaInterface'> standard Java interface (JVMPI/JVMTI) </param>
//<param name='pDataManager'> data manager </param>
//<param name='pParamChecker'> parameter checker </param>
//<param name='pKernel'> kernel </param>
void CDataManager::SetInternalModules(CEventManager *pEventManager,
                                      CParamChecker *pParamChecker, CJPIKernel *pKernel,
                                      CJavaInstrumentorManager *pJIM)
{
    m_pJavaInterface = CJpiGlobals::Instance()->pJvmInterface;
    m_pEventManager = pEventManager;
    m_pParamChecker = pParamChecker;
    m_pKernel = pKernel;
    m_pJavaInstrumentorManager = pJIM;

    RegisterClassPrepareEvent();

    // Initialize the Threads Manager object
    if (!m_pThreadInfoManager)
    {
        m_pThreadInfoManager = new CThreadInfoManager();
        m_pThreadInfoManager->Init(pEventManager);
    }

    // Initialize the Object Info Manager
    if (!m_pObjectInfoManager)
    {
        m_pObjectInfoManager = new CObjectInfoManager();
        m_pObjectInfoManager->Init();
    }

    // Internally register to Thread events to correctly manage thread ids
    if (!m_pThreadStartEvent)
    {
        m_pThreadStartEvent = new CThreadStartEventObserver(this);
        m_pEventManager->RegisterEventInternal(m_moduleId, *m_pThreadStartEvent);
    }
    if (!m_pThreadEndEvent)
    {
        m_pThreadEndEvent = new CThreadEndEventObserver(this);
        m_pEventManager->RegisterEventInternal(m_moduleId, *m_pThreadEndEvent);
    }
    if (!m_pVMInitEvent)
    {
        m_pVMInitEvent = new CDMVMInitEventObserver(this);
        m_pEventManager->RegisterEventInternal(m_moduleId, *m_pVMInitEvent);
    }
    if (!m_pVMShutdownEvent)
    {
        m_pVMShutdownEvent = new CDMVMShutdownEventObserver(this);
        m_pEventManager->RegisterEventInternal(m_moduleId, *m_pVMShutdownEvent);
    }
    if (!m_pGCStartEvent)
    {
        m_pGCStartEvent = new CDMGCStartEventObserver(this);
        m_pEventManager->RegisterEventInternal(m_moduleId, *m_pGCStartEvent);
    }
}


// if module id is not needed, the pModuleId can be NULL
TResult CDataManager::GetMpiMethodIdWModuleId(jmethodID jvmpiMethodId,
                                              MPI::TId *pMpiId,
                                              MPI::TId *pModuleId,
                                              bool *pbNewMethod,
                                              bool bIgnoreNotFoundMethod)
{
    if (pbNewMethod)
    {
        *pbNewMethod = false;
    }
    if (pMpiId == NULL)
    {
        return MRTE_ERROR_NULL_PTR;
    }
    int i;
    for (i = 0; i < 2; ++i) // max search twice
    {
        SDMMethodInfo *pMethodInfo = m_pJvmpiToMpiMethodId->Get((UIOP)jvmpiMethodId);
        if (pMethodInfo)
        {
            *pMpiId = pMethodInfo->id;
            if (pModuleId)
            {
                MARTINI_ASSERT("CDataManager", pMethodInfo->pModuleInfo, "No module info");
                *pModuleId = pMethodInfo->pModuleInfo->id;
            }
            if (pbNewMethod && pMethodInfo->bIsFirstTime)
            {
                pMethodInfo->bIsFirstTime = false;
                *pbNewMethod = true;
            }
            // support for workaround of BEA 1.4.2 bug (Windows and Linux IA-32)
            if (*pMpiId == ILLEGAL_METHOD_ID)
            {
                return MRTE_ERROR_FAIL;
            }

            return MRTE_RESULT_OK;
        }
        // if we got here it means means that the method is not in our database because of
        // user defined selectivity
        if (bIgnoreNotFoundMethod)
        {
            return MRTE_ERROR_METHOD_ID_NOT_FOUND;
        }

        // the class was not yet loaded (event did not arrive). This could happen the first
        // time the method is entered, the class was loaded by a different thread but event
        // didn't arrive yet. Simply request the event
        RequestMethodInfoFromStdPI(jvmpiMethodId);
    }
    MARTINI_INFORMATIVE("CDataManager", 0, false,
        "GetMpiMethodIdWModuleId failed to locate ID");
    return MRTE_ERROR_METHOD_ID_NOT_FOUND;
}

MPI::TId CDataManager::GetMethodModuleId(MPI::TId mpiMethodId)
{
    if (m_pDatabaseMethodInfo == NULL)
    {
        return MRTE_ERROR_NULL_PTR;
    }

    SDMMethodInfo *pMethodInfo = m_pDatabaseMethodInfo->Get(mpiMethodId);
    if (pMethodInfo)
    {
        if (pMethodInfo->pModuleInfo)
        {
            return pMethodInfo->pModuleInfo->id;
        }
        MARTINI_INFORMATIVE("CDataManager", 0, false, "GetMethodModuleId - No module ID");
    }
    return 0; // illegal module id
}

MPI::TId CDataManager::GetMethodClassId(MPI::TId methodId)
{
    SDMMethodInfo *pMethodInfo = m_pDatabaseMethodInfo->Get(methodId);
    if (pMethodInfo)
    {
        if (pMethodInfo->pClassInfo)
        {
            return pMethodInfo->pClassInfo->id;
        }
        MARTINI_INFORMATIVE("CDataManager", 0, false, "GetMethodClassId - No class ID");
    }
    return 0;
}

//
// Introduce a new class to the Data Manager
//
// Parameters:
//      classId                 : id of the new class
//      szNativeClassName       : "native" (mangled) class name
//      bFromInstrumentation    : whether the class was discovered during the instrumentation
//                                event (true) or during the ClassPrepare event (false)
//
// Returns:
//      MRTE_RESULT_OK                  : success
//      MRTE_RESULT_KEY_ALREADY_EXISTS  : a class with the same name already exists
//      MRTE_ERROR_ILLEGAL_ARGUMENT     : szNativeClassName is NULL
//      MRTE_ERROR_OUT_OF_MEMORY        : not enough memory to allocate an entry for the new
//                                        class
//
TResult CDataManager::NewClass(const MPI::TId classId,
                               const char *szNativeClassName,
                               const MPI::TId loaderObjectId,
                               bool bFromInstrumentation)
{
    if (NULL == szNativeClassName)
    {
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }

    MCString className;
    TResult iRes = GetClassName(szNativeClassName, className);
    if (MRTE_FAILED(iRes))
    {
        return iRes;
    }

    iRes = MRTE_RESULT_OK;

    m_pcsClassInfoDb->Enter();

    if (GetClassIdFromClassName(szNativeClassName, loaderObjectId) == 0)
    {
        MCString uniqueClassName;
        MakeUniqueClassName(uniqueClassName, className, loaderObjectId);
        m_pClassNameToId->Set(uniqueClassName.Get(), classId);

        // Create a placeholder for additional class information.
        // The information will be filled later by the instrumentation process or when the
        // class is prepared by the VM (Class Prepare event handler)
        SDMClassInfo *pDMClassInfo = new SDMClassInfo();
        if (pDMClassInfo != NULL)
        {
            pDMClassInfo->id = classId;
            pDMClassInfo->szNativeName.Set(szNativeClassName);
            pDMClassInfo->bFromInstrumentation = bFromInstrumentation;
            m_pDatabaseClassInfo->Set(classId, pDMClassInfo);
        }
        else
        {
            // Not enough memory to allocate an entry for this class
            iRes = MRTE_ERROR_OUT_OF_MEMORY;
        }
    }
    else
    {
        // A class with the given name already exists in the database.

        // NOTE: Bugzilla 200815 (https://bugs.eclipse.org/bugs/show_bug.cgi?id=200815):
        // This logic must be changed when resolving 200815.

        MARTINI_INFORMATIVE1("CDataManager", 3, false,
            "[NewClass] Class '%s' already exists in the database", className.Get());
        iRes = MRTE_RESULT_KEY_ALREADY_EXISTS;
    }

    m_pcsClassInfoDb->Leave();

    return iRes;
}

//
// Updates internal databases with information about the instrumentation applied
// to a class
//
TResult CDataManager::SetClassInstrumentationStatus(const Martini::MPI::TId classId,
                                                    bool bEventsEnabled)
{
    // NOTE: The process of updating the class information in m_pDatabaseClassInfo
    //       is not thread-safe. It is assumed that this function will never be called
    //       for the same class from multiple threads.

    SDMClassInfo *pDMClassInfo = m_pDatabaseClassInfo->Get(classId);
    if (NULL == pDMClassInfo)
    {
        // Fatal error. Entry for this class must exist
        MARTINI_ERROR1("CDataManager", "Failed to update instrumentation status "
            "for class id %d. Class information entry does not exist",
            (U32)classId);
        return MRTE_ERROR_FAIL;
    }

    // Update instrumentation status
    pDMClassInfo->bEventsEnabled = bEventsEnabled;

    return MRTE_RESULT_OK;
}

//
// Updates internal databases with information about an instrumented class
// (the class and method IDs assigned during instrumentation).
//
// The internal CDataManager database is updated with the contents of pClassInfo.
//
// SAdaptorClassInfo comes from the ModifyClass() or ModifyByteCodes() functions of one of (Heap/Thread/CG) Adaptors
TResult CDataManager::SetClassInstrumentationInfo(SAdaptorClassInfo *pClassInfo)
{
    // NOTE: The process of updating the class information in m_pDatabaseClassInfo
    //       is not thread-safe. It is assumed that this function will never be called
    //       for the same class from multiple threads.

    if (!pClassInfo)
    {
        return MRTE_ERROR_ILLEGAL_ARGUMENT;
    }

    unsigned int uiNumMethods = pClassInfo->uiNumMethods;

    m_pcsClassInfoDb->Enter();  // Add for bug 260290

    SDMClassInfo *pDMClassInfo = m_pDatabaseClassInfo->Get(pClassInfo->classId);
    if (NULL == pDMClassInfo)
    {
        // Class does not exist. This can happen only in JVMPI, where the class name
        // is not provided by the Class File Load Hook event.
        // Martini for JVMPI does not support multiple loaders for the same class.
        // Therefore, the bootstrap loader is assumed.

        TResult res = NewClass(pClassInfo->classId, pClassInfo->szClassName, 0,
            true /* bFromInstrumentation */);
        if (MRTE_FAILED(res))
        {
            m_pcsClassInfoDb->Leave();  // Add for bug 260290
            MARTINI_ERROR2("CDataManager", "Failed to update instrumentation information "
                "for class %s (id = %d). Class information entry does not exist",
                pClassInfo->szClassName, (U32)pClassInfo->classId);
            // Fatal error. Entry for this class must exist
            return MRTE_ERROR_FAIL;
        }
        pDMClassInfo = m_pDatabaseClassInfo->Get(pClassInfo->classId);
    }
    m_pcsClassInfoDb->Leave();  // Add for bug 260290

    // The following members are left untouched (NULL) and will be updated later:
    //
    // jniGlobalRef  : will be set when a ClassPrepare event arrives for this class
    // pClassBytes   : set by SetClassFileBuffer, which is called by the ClassFileLoadHook
    //                 handler (in JIM) after basic instrumentation was applied to the
    //                 class file
    // szName        : will be set when the ClassPrepare event arrives for this class
    // szSrcFilename : will be set when the ClassPrepare event arrives for this class
    pDMClassInfo->uiNumMethods = uiNumMethods;
    pDMClassInfo->uiAttributeFlags = pClassInfo->uiAttributeFlags;
    pDMClassInfo->methodIds.Reserve(uiNumMethods);
    m_pDatabaseClassInfo->Set(pDMClassInfo->id, pDMClassInfo);

    // Update method information
    for (unsigned int i = 0; i < uiNumMethods; ++i)
    {
        SDMMethodInfo *pDMMethodInfo = new SDMMethodInfo();
        if (!pDMMethodInfo)
        {
            return MRTE_ERROR_OUT_OF_MEMORY;
        }
        pDMMethodInfo->pClassInfo = pDMClassInfo;
        pDMMethodInfo->pModuleInfo = m_pInitialModule;
        pDMMethodInfo->id = pClassInfo->pMethods[i].id;
        pDMMethodInfo->uiAttributeFlags = pClassInfo->pMethods[i].uiAttributeFlags;
        pDMMethodInfo->cpAlreadyInvokedFlag = pClassInfo->pMethods[i].cpAlreadyInvokedFlag;
        pDMMethodInfo->szMethodOriginalName.Set(pClassInfo->pMethods[i].szName);
        pDMMethodInfo->szMethodOriginalSignature.Set(pClassInfo->pMethods[i].szSignature);

        MARTINI_INFORMATIVE3("CDataManager", 5, false,
            "Updated CDataManager internal instrumentation information for method '%s(%s)' (id = %u)",
            pDMMethodInfo->szMethodOriginalName.Get(),
            pDMMethodInfo->szMethodOriginalSignature.Get(),
            (U32)pDMMethodInfo->id);

        m_pDatabaseMethodInfo->Set(pDMMethodInfo->id, pDMMethodInfo);

        pDMClassInfo->methodIds.Push(pDMMethodInfo->id);
    }

    MARTINI_INFORMATIVE2("CDataManager", 5, false,
        "Updated CDataManager internal instrumentation information for class '%s' (id = %d)",
        pClassInfo->szClassName, (U32)pDMClassInfo->id);

    return MRTE_RESULT_OK;
}

//
// Stores a class with basic instrumentation in the Data Manager
//
// NOTE: method is not thread safe. All invocations must be serialized by the caller
//
TResult CDataManager::SetClassFileBuffer(MPI::TId classId, SClassFile *pClassBytes)
{
    SDMClassInfo *pDMClassInfo = m_pDatabaseClassInfo->Get(classId);
    if (pDMClassInfo == NULL)
    {
        // Unknown class
        return MRTE_ERROR_FAIL;
    }

    pDMClassInfo->pClassBytes = (pClassBytes->uiSize == 0) ? NULL : pClassBytes;
    return MRTE_RESULT_OK;
}

//
// ClassLoad (JVMPI)/ClassPrepare (JVMTI) event handler
//
// This event is initiated by the Class Prepare Handler in JVMTIInterface, which
// generates the MPI event and passes it along to event manager, which calls this method. - JGW
TResult CDataManager::ClassPrepareHandler(struct SEmData *pEvent)
{
    MCString szClassName, szGenericClassName;

    TResult res = GetClassName(pEvent->u.pJVMData->u.classPrepare.szClassName, szClassName);
    if (MRTE_FAILED(res))
    {
        return res;
    }

    res = GetGenericClassName(pEvent, szGenericClassName);
    if (MRTE_FAILED(res))
    {
        return res;
    }

    MPI::TId loaderId = 0;
    jlong loaderTag = 0;
    if (pEvent->u.pJVMData->u.classPrepare.jniLoaderLocalRef != NULL)
    {
        res = m_pObjectInfoManager->GetOrCreateTag(&loaderId, &loaderTag,
            pEvent->u.pJVMData->u.classPrepare.jniLoaderLocalRef,
            pEvent->pJniEnv, false);
        if (MRTE_FAILED(res))
        {
            MARTINI_INFORMATIVE1("CDataManager", 3, false,
                "Unable to get object id for class loader of class '%s'. "
                "The bootstrap class loader is assumed", szClassName.Get());
            loaderId = 0;
        }
    }

    // Unique name = szClassName + @ + a U32 of the classsLoader Id (loaderId), stores it in strUniqueClassName
    MCString strUniqueClassName;
    MakeUniqueClassName(strUniqueClassName, szClassName, loaderId);
    if (m_pDatabaseClassesWithCompleteInfo->Get(strUniqueClassName.Get()) != 0)
    {
        // ClassPrepare event already received for this class. Do nothing and exit
        return MRTE_RESULT_OK;
    }

    SDMClassInfo *pClassInfo = NULL;

    m_pcsClassInfoDb->Enter();  // Add for bug 260290

    // Get the class ID for the class+classloaderID combinatoin
    MPI::TId classId = GetClassIdFromClassName(
        pEvent->u.pJVMData->u.classPrepare.szClassName, loaderId);

    if (0 == classId)
    {
        // Unknown class. Add it to the Data Manager

    	// Allocate an ID, and create new class based on class name + loaderID
        classId = m_pClassIdAllocator->AllocateId();
        res = NewClass(classId, pEvent->u.pJVMData->u.classPrepare.szClassName,
            loaderId, false /* bFromInstrumentation */);
        if (MRTE_FAILED(res))
        {
        	m_pcsClassInfoDb->Leave();  // Add for bug 260290
        	MARTINI_INFORMATIVE2("CDataManager", 0, false,
                "[ClassPrepareHandler] Failed to create new definition for class %s (id = %d)",
                pEvent->u.pJVMData->u.classPrepare.szClassName, (U32)classId);
            return res;
        }

        MARTINI_INFORMATIVE2("CDataManager", 5, false,
            "[ClassPrepareHandler] New definition created for class %s (id = %d)",
            pEvent->u.pJVMData->u.classPrepare.szClassName, (U32)classId);

        // Get the new class info from the datamanager DB, and populate it
        pClassInfo = m_pDatabaseClassInfo->Get(classId);
        pClassInfo->szName.Set(szGenericClassName);
        pClassInfo->id = classId;
        pClassInfo->pClassBytes = NULL;
        pClassInfo->uiAttributeFlags = 0;

        // Creates method information (for all the methods of the class) based on the pEvent, and the class info, and adds it to the DB
        FillMethodDB(pEvent, pClassInfo);

        // FillClassDB must be called after FillMethodDB because method ids are allocated
        // only in FillMethodDB
        FillClassDB(pEvent, pClassInfo, classId);
    }

    m_pcsClassInfoDb->Leave();  // Add for bug 260290

    // When we reach here, the class information must exist in the database
    pClassInfo = m_pDatabaseClassInfo->Get(classId);
    if (!pClassInfo)
    {
        MARTINI_ERROR2("CDataManager",
            "[ClassPrepareHandler] Failed to retrieve information for class %s (id = %d)",
            pEvent->u.pJVMData->u.classPrepare.szClassName, (U32) classId);
        return MRTE_ERROR_FAIL;
    }

    if (pClassInfo->bFromInstrumentation)
    {
        m_pJavaInstrumentorManager->InstrumentationLock();

        // Create a global JNI reference for this class so it can be redefined later
        // if needed (e.g., when the user enables/disables call-graph events)
        JNIEnv *pJNIEnv = NULL;
        TResult iRes = m_pJavaInterface->GetJNIEnv(&pJNIEnv);
        if (MRTE_SUCCEEDED(iRes))
        {
            pClassInfo->jniGlobalRef = (jclass)pJNIEnv->NewGlobalRef(
                pEvent->u.pJVMData->u.classPrepare.jniClassLocalRef);
        }

        // Update the databases with additional information supplied by the JVM, such as
        // the JVM IDs of the class and its methods and line numbers
        pClassInfo->szName.Set(szGenericClassName);

        const char *pSourceFileName = pEvent->u.pJVMData->u.classPrepare.szSourceName;
        if (pSourceFileName && pSourceFileName[0] != '\0')
        {
            pClassInfo->szSrcFileName.Set(pSourceFileName);
        }
        else
        {
            pClassInfo->szSrcFileName.Set("<Unknown>");
        }

        MARTINI_ASSERT("CDataManager", pClassInfo, "");
        MARTINI_ASSERT("CDataManager", pEvent->dataType == MPI_VM, "");

        unsigned int uiJVMPINumMethods = pEvent->u.pJVMData->u.classPrepare.uiNumMethods;
        unsigned int uiJIENumMethods = pClassInfo->uiNumMethods;
        unsigned int i;
        for (i = 0; i < uiJIENumMethods; ++i)
        {
            // For each method that was reported by the instrumentor, find the corresponding
            // jmethodID. This will get selectivity for JittedMethodLoaded events,
            // since we're not keeping in the database all methods, just the ones reported by
            // instrumentation
            MPI::TId mpiMethodId = pClassInfo->methodIds.GetAt(i);
            SDMMethodInfo *pMethodInfo = m_pDatabaseMethodInfo->Get(mpiMethodId);
            MARTINI_ASSERT("CDataManager", pMethodInfo, "");

	        unsigned int uiJVMPIMethodIndex;
            for (uiJVMPIMethodIndex = 0; uiJVMPIMethodIndex < uiJVMPINumMethods;
                 ++uiJVMPIMethodIndex)
            {
                const char *szJVMPIMethodName =
                    pEvent->u.pJVMData->u.classPrepare.pMethodInfo[uiJVMPIMethodIndex].szMethodName;
                const char *szJVMPIMethodSignature =
                    pEvent->u.pJVMData->u.classPrepare.pMethodInfo[uiJVMPIMethodIndex].szMethodSignature;
                if (pMethodInfo->szMethodOriginalName.IsEqual(szJVMPIMethodName))
                {
                    if (pMethodInfo->szMethodOriginalSignature.IsEqual(szJVMPIMethodSignature))
                    {
                        UIOP jvmpiMethodId =
                            (UIOP) pEvent->u.pJVMData->u.classPrepare.pMethodInfo[uiJVMPIMethodIndex].methodId;
                        m_pJvmpiToMpiMethodId->Set(jvmpiMethodId, pMethodInfo);
    ;
                        res = SetMethodNameAndSignature(pClassInfo, pMethodInfo,
                            pEvent->u.pJVMData->u.classPrepare.pMethodInfo[uiJVMPIMethodIndex].szMethodName,
                            pEvent->u.pJVMData->u.classPrepare.pMethodInfo[uiJVMPIMethodIndex].szMethodSignature,
                            pEvent->u.pJVMData->u.classPrepare.pMethodInfo[uiJVMPIMethodIndex].szMethodGeneric,
                            pEvent->u.pJVMData->u.classPrepare.pMethodInfo[uiJVMPIMethodIndex].uiAccessFlags);
                        if (MRTE_FAILED(res))
                        {
                            m_pJavaInstrumentorManager->InstrumentationUnlock();
                            return res;
                        }
                        pMethodInfo->lineNumbers.uiStart = GetMethodStartLineNumber(pEvent, uiJVMPIMethodIndex);
                        pMethodInfo->lineNumbers.uiEnd = GetMethodEndLineNumber(pEvent, uiJVMPIMethodIndex);
                        pMethodInfo->vmId = (jmethodID)jvmpiMethodId;

                        break;
                    }
                }

            }

        } // for (class methods)

        m_pJavaInstrumentorManager->InstrumentationUnlock();

        // Query the class instrumentation status to verify if its instrumentation
        // is consistent with the global events status
        if (pClassInfo->pClassBytes != 0)
        {
            bool bEventsEnabledForClass = AreEventsEnabledForClass(classId);
            if (bEventsEnabledForClass != m_pJavaInstrumentorManager->GetInstrGeneratedEventsStatus())
            {
                if (m_pJavaInterface->CanRedefineClass(pClassInfo->jniGlobalRef))
                {
                    TJvmtiClassDefVector vecClassesToRedefine;
                    iRes = m_pJavaInstrumentorManager->InstrumentAndQueueForRedef(
                        &vecClassesToRedefine, classId,
                        MPI::EG_NONE, // argument is ignored, so we can put anything we want here
                        m_pJavaInstrumentorManager->GetInstrGeneratedEventsStatus());
                    if (MRTE_SUCCEEDED(iRes))
                    {
                        MARTINI_INFORMATIVE1("CDataManager", 0, false,
                            "Class '%s' has wrong instrumentation. Queuing class for redefinition",
                            pClassInfo->szNativeName.Get());
                        m_pJavaInstrumentorManager->RedefineMultipleClasses(
                            vecClassesToRedefine, false /* isBlocking */);
                    }
                    else if (MRTE_ERROR_INSTRUMENTATION_NOT_NEEDED == iRes) {
                        MARTINI_INFORMATIVE1("CDataManager", 0, false,
                            "Class '%s' will use basic instrumentation, as InstrumentationNotNeeded for class was set (likely because it was filtered elsewhere)",
                            pClassInfo->szNativeName.Get());
                    }
                    else if (MRTE_ERROR_UNABLE_TO_INSTRUMENT == iRes)
                    {
                        MARTINI_INFORMATIVE1("CDataManager", 0, false,
                            "Class '%s' has wrong instrumentation, but will not be redefined due to user-defined "
                            "filters or an internal reason",
                            pClassInfo->szNativeName.Get());
                    }
                    else
                    {
                        if (bEventsEnabledForClass)
                        {
                            // Class was successfully instrumented before. The instrumentation
                            // failure at this stage is unexpected and we treat this as
                            // a fatal error.
                            MARTINI_ERROR1("CDataManager",
                                "Failed to instrument class '%s'", pClassInfo->szNativeName.Get());
                        }
                        else
                        {
                            // If events are not enabled for the class, it means that it was
                            // never instrumented before. Therefore, we do not consider
                            // an instrumentation failure as a fatal error
                            MARTINI_INFORMATIVE1("CDataManager", 0, false,
                                "Class '%s' has wrong instrumentation, but will not be redefined due to user-defined "
                                "filters or an internal reason",
                                pClassInfo->szNativeName.Get());
                        }
                    }
                }
            }
        }


    } // if (class discovered by instrumentation)


    // Add the class name to the database of classes with complete information
    // so that we will not process it again in case we get another ClassPrepare
    // event for this class
    m_pDatabaseClassesWithCompleteInfo->Set(strUniqueClassName.Get(), classId);

    //Set tag for this class
    if(pEvent->u.pJVMData->u.classPrepare.jniClassLocalRef != NULL){
        m_pJavaInterface->SetClassTag(pEvent->u.pJVMData->u.classPrepare.jniClassLocalRef);
    }
    return MRTE_RESULT_OK;
}

TResult CDataManager::GetUpdatedMethodName(const char *szClassName, const char *szMethodName,
                                           MCString *szUpdatedMethodName)
{
    if (szUpdatedMethodName == NULL)
    {
        return MRTE_ERROR_NULL_PTR;
    }
    TResult iRetVal;
    if (strcmp(szMethodName, JAVA_INSTANCE_CONSTRUCTOR_NAME) == 0)
    {
        const char *pLeafName = strrchr(szClassName, '.');
        if (pLeafName != NULL)
        {
            pLeafName++;
        }
        else
        {
            pLeafName = szClassName;
        }
        iRetVal = szUpdatedMethodName->Set(pLeafName);
    }
    else if (strcmp(szMethodName, JAVA_CLASS_CONSTRUCTOR_NAME) == 0)
    {
        iRetVal = szUpdatedMethodName->Set(STATIC_CONSTRACTOR_NAME);
    }
    else
    {
        iRetVal = szUpdatedMethodName->Set(szMethodName);
    }
    return iRetVal;
}

TResult
CDataManager::SetMethodNameAndSignature(SDMClassInfo *pClassInfo,
                                        SDMMethodInfo *pMethodInfo,
                                        const char *szMethodName,
                                        const char *szPrototype,
                                        const char *szGeneric,
                                        unsigned int uiFlags)
{
    pMethodInfo->szMethodOriginalName.Set(szMethodName);
    pMethodInfo->szMethodOriginalSignature.Set(szPrototype);

    pMethodInfo->szFriendlyName.Empty();
    pMethodInfo->szFriendlySignature.Empty();
    MCString methodName, methodPrototype;

    // Change constructor names from <init> to the actual class name
    TResult iRes = GetUpdatedMethodName(pClassInfo->szName.Get(), szMethodName, &methodName);
    if (MRTE_FAILED(iRes))
    {
        return iRes;
    }

    iRes = ParseMethodName(methodName.Get(), szGeneric, &pMethodInfo->szFriendlyName);
    if (MRTE_FAILED(iRes))
    {
        return iRes;
    }

    iRes = ParseMethodPrototype(szPrototype, szGeneric, uiFlags,
        &pMethodInfo->szFriendlySignature);

    return iRes;
}

/** Creates method based on the pEvent, and the class info, and adds it to the DB */
MPI::TId CDataManager::AddMethodToDB(SDMClassInfo *pClassInfo,
                                     struct SEmData *pEvent,
                                     unsigned int uiIndex)
{
    SDMMethodInfo *pMethodInfo;
    MPI::TId mpiId;

    // Create a new Data Manager Method
    pMethodInfo = new SDMMethodInfo();
    if (!pMethodInfo)
    {
        return 0;
    }

    // Fill in the method info, from the class, and (mostly) from the pEvent JVM data
    pMethodInfo->pClassInfo = pClassInfo;
    pMethodInfo->pModuleInfo = m_pInitialModule;
    pMethodInfo->cpAlreadyInvokedFlag = 0;
    pMethodInfo->vmId = pEvent->u.pJVMData->u.classPrepare.pMethodInfo[uiIndex].methodId;
    pMethodInfo->uiAttributeFlags =
        pEvent->u.pJVMData->u.classPrepare.pMethodInfo[uiIndex].uiAccessFlags;
    TResult iRes = SetMethodNameAndSignature(pClassInfo, pMethodInfo,
        pEvent->u.pJVMData->u.classPrepare.pMethodInfo[uiIndex].szMethodName,
        pEvent->u.pJVMData->u.classPrepare.pMethodInfo[uiIndex].szMethodSignature,
        pEvent->u.pJVMData->u.classPrepare.pMethodInfo[uiIndex].szMethodGeneric,
        pEvent->u.pJVMData->u.classPrepare.pMethodInfo[uiIndex].uiAccessFlags);
    if (MRTE_FAILED(iRes))
    {
        return 0;
    }
    pMethodInfo->lineNumbers.uiStart = GetMethodStartLineNumber(pEvent, uiIndex);
    pMethodInfo->lineNumbers.uiEnd = GetMethodEndLineNumber(pEvent, uiIndex);

    // This gets the method's method ID from event data
    pMethodInfo->id = GetMethodId(pEvent, uiIndex);
    bool bNewJVMPIMethod = false;
    UIOP jvmpiMethodId = 0;
    if (pMethodInfo->id == 0)
    {
        // First time this method was encountered. Need to allocate a method id
        bNewJVMPIMethod = true;

        MARTINI_ASSERT("CDataManager", pEvent->dataType == MPI_VM, "");// JVMPI event
        jvmpiMethodId = (UIOP)pEvent->u.pJVMData->u.classPrepare.pMethodInfo[uiIndex].methodId;
        // patch in order to work-around a BEA 1.4.2 bug in JVMPI enter-leave that results in
        // out of order events, or "leave" events without corresponding "enter" events.
        if (IsProblematicMethod(pEvent->u.pJVMData->u.classPrepare.szClassName,
            pMethodInfo->szFriendlyName.Get(), pMethodInfo->szFriendlySignature.Get()))
        {
            pMethodInfo->id = ILLEGAL_METHOD_ID;
        }
        else
        {
            pMethodInfo->id = m_pMethodIdAllocator->AllocateId();
        }

    }

    mpiId = pMethodInfo->id;
    // update the method database
    TResult res = MRTE_RESULT_OK;
    if (ILLEGAL_METHOD_ID != mpiId)
    {
        res = m_pDatabaseMethodInfo->Set(mpiId, pMethodInfo);
    }
    if (bNewJVMPIMethod && MRTE_RESULT_KEY_ALREADY_EXISTS != res)
    {
        m_pJvmpiToMpiMethodId->Set(jvmpiMethodId, pMethodInfo);
    }
    else
    {
        // method already exists in the databases. cleanup
        delete pMethodInfo;
    }
    return mpiId;
}

// Creates method information (for all the methods of the class) based on the pEvent, and the class info, and adds it to the DB
void CDataManager::FillMethodDB(struct SEmData *pEvent, SDMClassInfo *pClassInfo)
{
    unsigned int uiNumMethods = pEvent->u.pJVMData->u.classPrepare.uiNumMethods;
    for (unsigned int i = 0; i < uiNumMethods; ++i)
    {
        AddMethodToDB(pClassInfo, pEvent, i);
    }
}

/**
 * Given a ClassPrepate event (SEmData *pEvent) and a class ID, fill in the pClassInfo with methods
 * and source name, then update the class database.
 */
void CDataManager::FillClassDB(struct SEmData *pEvent,
                               SDMClassInfo *pClassInfo,
                               MPI::TId classId)
{
    if (!pClassInfo)
    {
        return;
    }
    pClassInfo->id = classId;
    pClassInfo->uiNumMethods = pEvent->u.pJVMData->u.classPrepare.uiNumMethods;
    pClassInfo->methodIds.Reserve(pClassInfo->uiNumMethods);
    unsigned int i;
    for (i = 0; i < pClassInfo->uiNumMethods; ++i)
    {
        MPI::TId methodId = GetMethodId(pEvent, i);
        if (methodId == 0)
        {
            return;
        }
        pClassInfo->methodIds.Push(methodId);
    }

    const char *pSourceFileName = pEvent->u.pJVMData->u.classPrepare.szSourceName;
    if (pSourceFileName && pSourceFileName[0] != '\0')
    {
        pClassInfo->szSrcFileName.Set(pSourceFileName);
    }
    else
    {
        pClassInfo->szSrcFileName.Set("<Unknown>");
    }

    // update the class info database
    m_pDatabaseClassInfo->Set(pClassInfo->id, pClassInfo);
}


TResult CDataManager::GetClassName(const char *szClassName, MCString& name)
{
    if (szClassName == NULL)
    {
        return MRTE_ERROR_FAIL;
    }

    TResult iRes = MRTE_RESULT_OK;
    if (szClassName[strlen(szClassName) - 1] == ';'
        || szClassName[0] == '[')
    {
        int iPlace = 0;
        iRes = Unmangle(szClassName, &iPlace, &name);
    }
    else
    {
        iRes = name.Set(szClassName);
        name.ReplaceAll("/", ".");
    }
    return iRes;
}

TResult CDataManager::GetGenericClassName(struct SEmData *pEvent, MCString& name)
{
    const char *pClassName;
    if (pEvent->dataType == MPI_VM)
    {
        pClassName = pEvent->u.pJVMData->u.classPrepare.szClassName;
    }
    if (pClassName == NULL)
    {
        return MRTE_ERROR_FAIL;
    }

    const char *pGenericSig = NULL;
    if (pEvent->dataType == MPI_VM)
    {
        pGenericSig = pEvent->u.pJVMData->u.classPrepare.szGenericSig;
    }

    // to handle the case of an array of generic type the square parenthesis are added here
    unsigned int uiArrayCount = 0;
    while (*pClassName == '[')
    {
        pClassName++;
        uiArrayCount++;
    }

    TResult res = ParseClassName(pClassName, pGenericSig, &name);

    while (uiArrayCount--)
    {
        name.Append("[]");
    }

    return res;
}

//
// Returns the class id associated with szNativeClassName.
//
// Parameters:
//      szNativeClassName   : native (mangled) class name
//      loaderObjectId      : the object id of the initiating loader of this class.
//                            use 0 for the bootstrap class loader
//
// Returns:
//      Class id, or 0 if the class does not exist
//
MPI::TId CDataManager::GetClassIdFromClassName(const char *szNativeClassName,
                                               MPI::TId loaderObjectId)
{
    MCString className;
    TResult res = GetClassName(szNativeClassName, className);
    if (MRTE_FAILED(res))
    {
        return 0;
    }

    MCString uniqueClassName;
    MakeUniqueClassName(uniqueClassName, className, loaderObjectId);
    MPI::TId classId = m_pClassNameToId->Get(uniqueClassName.Get());
    return classId;
}

//
// Returns the class id associated with szNativeClassName. If not found, allocates and returns
// a new class id.
//
// Parameters:
//      szNativeClassName   : native (mangled) class name
//      loaderObjectId      : the object id of the initiating loader of this class.
//                            use 0 for the bootstrap class loader
// Returns:
//      Class id, or 0 if the operation failed
//
MPI::TId CDataManager::GetOrAllocateClassIdFromClassName(const char *szNativeClassName,
                                                         MPI::TId loaderObjectId)
{
    MCString className;
    TResult res = GetClassName(szNativeClassName, className);
    if (MRTE_FAILED(res))
    {
        return 0;
    }

    m_pcsClassInfoDb->Enter();

    MPI::TId classId = GetClassIdFromClassName(szNativeClassName, loaderObjectId);
    if (0 == classId)
    {
        // Unknown class name. Add the class to the database
        classId = m_pClassIdAllocator->AllocateId();
        res = NewClass(classId, szNativeClassName, loaderObjectId,
            false /* bFromInstrumentation */);
        if (MRTE_FAILED(res))
        {
            m_pcsClassInfoDb->Leave();
            return 0;
        }

        // Update the class name
        SDMClassInfo *pDMClassInfo = m_pDatabaseClassInfo->Get(classId);
        if (pDMClassInfo)
        {
            pDMClassInfo->szName.Set(className.Get());
            pDMClassInfo->szSrcFileName.Set("<Unknown>");
        }

    }

    m_pcsClassInfoDb->Leave();

    return classId;
}

MPI::TId CDataManager::GetMethodId(struct SEmData *pEvent, unsigned int uiIndex)
{
    MPI::TId mpiId = 0;
    if (pEvent->dataType == MPI_VM)
    {
        UIOP jvmMethodId = (UIOP)pEvent->u.pJVMData->u.classPrepare.pMethodInfo[uiIndex].methodId;
        // check if this method is already in the jvmpi-to-mpi-method-id
        SDMMethodInfo *pMethodInfo = m_pJvmpiToMpiMethodId->Get(jvmMethodId);
        if (pMethodInfo)
        {
            mpiId = pMethodInfo->id;
        }
    }
    return mpiId;
}


unsigned int CDataManager::GetMethodStartLineNumber(SEmData *pEvent, unsigned int uiIndex)
{
    int iLineNumber =
        pEvent->u.pJVMData->u.classPrepare.pMethodInfo[uiIndex].lineNumbers.uiStart;
    if (iLineNumber < 0)
    {
        iLineNumber = 0;
    }
    return iLineNumber;
}

unsigned int CDataManager::GetMethodEndLineNumber(SEmData *pEvent, unsigned int uiIndex)
{
    int iLineNumber =
        pEvent->u.pJVMData->u.classPrepare.pMethodInfo[uiIndex].lineNumbers.uiEnd;
    if (iLineNumber < 0)
    {
        iLineNumber = 0;
    }
    return iLineNumber;
}


void CDataManager::NotifyOnProfilerEventRegistration(MPI::TId clientId, MPI::IEventObserver &event)
{
    MPI::BitSet dataRequestType = event.EventDataTypes();

    // Register for internal events as needed
    switch (event.Type())
    {
    case MPI::EV_JITTED_METHOD_LOADED:
        if (!m_pMethodJitLoadedEvent)
        {
            m_pMethodJitLoadedEvent = new CMethodJitEventObserver(this);
            m_pEventManager->RegisterEventInternal(m_moduleId, *m_pMethodJitLoadedEvent);
        }

        break;

    default:
        ;
    }
}

TResult CDataManager::SetMethodJitted(MPI::TId methodId)
{
    SDMMethodInfo *pMethodInfo = m_pDatabaseMethodInfo->Get(methodId);
    if (!pMethodInfo)
    {
        return MRTE_ERROR_FAIL;
    }
    pMethodInfo->pModuleInfo = &m_DatabaseModuleInfo[MODULE_JIT];
    return MRTE_RESULT_OK;
}

MPI::TId CDataManager::GetConstantModuleId()
{
    return m_constantModuleId;
}


//
// Checks whether the given method belongs to a list of methods for which BEA 1.4.2
// JVMPI generates "Leave" events without "Enter" events.
//
// Returns:
//      true    : the method belongs to this group
//      false   : the method does not belong to this group
//
bool CDataManager::IsProblematicMethod(const char *szClassName,
                                       const char *szMethodName,
                                       const char *szMethodSig)
{
    //TODO: clean-up. Always use VM names for classes, methods and signatures, and remove
    //      duplications
    static TProblematicMethodInfo ProblematicMethodsClasses[] =
    {
        TProblematicMethodInfo("jrockit.vm.Threads",
            "setThrownException", "void(java.lang.Thread, java.lang.Throwable)"),
        TProblematicMethodInfo("jrockit/vm/Threads",
            "setThrownException", "void(java.lang.Thread, java.lang.Throwable)"),
        TProblematicMethodInfo("java.lang.Object", "wait", "void()"),
        TProblematicMethodInfo("java/lang/Object", "wait", "void()"),
        TProblematicMethodInfo("java.lang.Object", "wait", "void(long)"),
        TProblematicMethodInfo("java/lang/Object", "wait", "void(long)"),
        TProblematicMethodInfo("java.lang.ref.ReferenceQueue",
            "remove", "java.lang.ref.Reference(long)"),
        TProblematicMethodInfo("java/lang/ref/ReferenceQueue",
            "remove", "java.lang.ref.Reference(long)"),
        TProblematicMethodInfo("java.lang.ref.ReferenceQueue",
            "remove", "java.lang.ref.Reference()"),
        TProblematicMethodInfo("java/lang/ref/ReferenceQueue",
            "remove", "java.lang.ref.Reference()")
    };
    unsigned int i;
    for (i = 0; i < SIZE_OF_ARRAY(ProblematicMethodsClasses); ++i)
    {
        TProblematicMethodInfo problematicMethodInfo = ProblematicMethodsClasses[i];
        if (strcmp(szClassName, problematicMethodInfo.szClassName) == 0)
        {
            if (strcmp(szMethodName, problematicMethodInfo.szMethodName) == 0)
            {
                return true;
            }
        }
    }
    return false;
}

//
// Returns a list of ids of all classes to which basic instrumentation was applied
//
// Parameters:
//      pClassIdList    [out]   : a list to be filled with class ids
//
// Returns:
//      MRTE_RESULT_OK          : success
//      otherwise               : failure
//
TResult CDataManager::GetInstrumentedClasses(TIdList *pClassIdList)
{
    TResult iRes = m_pDatabaseClassInfo->Iterate(ClassInfoDbIterationCallBack, pClassIdList);
    return iRes;
}

//
// Returns the JNI global class reference (jclass) of a class.
//
// Parameters:
//      pClassJniGlobalRef  [out]   : JNI class reference (jclass). Used for redefining the
//                                    class.
//      classId             [in]    : class id
//
// Returns:
//      MRTE_RESULT_OK              : success
//      MRTE_ERROR_FAIL             : class information does not exist
//
TResult CDataManager::GetClassGlobalJniRef(jclass *pClassJniGlobalRef, MPI::TId classId)
{
    SDMClassInfo *pDMClassInfo = m_pDatabaseClassInfo->Get(classId);
    if (NULL == pDMClassInfo)
    {
        return MRTE_ERROR_FAIL;
    }
    *pClassJniGlobalRef = pDMClassInfo->jniGlobalRef;
    return MRTE_RESULT_OK;
}

//
// Returns instrumentation information for the specified class id
//
// Parameters:
//      pStrNativeClassName  [out]  : JVM Internal class name
//      pClassWithBasicInstr [out]  : Class bytes with basic instrumentation.
//      pClassJniGlobalRef   [out]  : JNI class reference (jclass). Used for redefining the
//                                    class.
//      classId               [in]  : class id
//
// Returns:
//      MRTE_RESULT_OK              : Success
//      MRTE_ERROR_FAIL             : Class information does not exist
//
TResult CDataManager::GetClassInstrumentationInfo(MCString *pStrNativeClassName,
                                                  SClassFile *pClassWithBasicInstr,
                                                  jclass *pClassJniGlobalRef,
                                                  MPI::TId classId)
{
    SDMClassInfo *pDMClassInfo = m_pDatabaseClassInfo->Get(classId);
    if (NULL == pDMClassInfo)
    {
        return MRTE_ERROR_FAIL;
    }
    *pClassJniGlobalRef = pDMClassInfo->jniGlobalRef;
    *pClassWithBasicInstr = *(pDMClassInfo->pClassBytes);
    pStrNativeClassName->Set(pDMClassInfo->szNativeName.Get());
    return MRTE_RESULT_OK;
}

//
// Returns instrumentation context information for the specified class id
//
// Parameters:
//      pContext    [out]  : Class instrumentation info to be used by the Instrumentaiton Adaptor.
//                           The pMethods member is allocated by this function and
//                           must be deallocated by the caller.
//
//      classId      [in]  : class id
//
// Returns:
//      MRTE_RESULT_OK              : success
//      MRTE_ERROR_FAIL             : class information does not exist
//
TResult CDataManager::GetClassInstrContext(CContext *pCgContext,
                                           MPI::TId classId)
{
    //TODO: Refactor CgContext out of Data Manager and into the CgAdaptorProxy,
    //      because it is the only component that uses it.

    SDMClassInfo *pDMClassInfo = m_pDatabaseClassInfo->Get(classId);
    if (NULL == pDMClassInfo)
    {
        return MRTE_ERROR_FAIL;
    }

    pCgContext->classInfo.classId = classId;
    pCgContext->classInfo.szClassName = pDMClassInfo->szNativeName.Get();
    pCgContext->classInfo.szSourceFileName = pDMClassInfo->szSrcFileName.Get();
    pCgContext->classInfo.uiAttributeFlags = pDMClassInfo->uiAttributeFlags;
    pCgContext->classInfo.uiNumMethods = pDMClassInfo->uiNumMethods;
    if (pDMClassInfo->uiNumMethods > 0)
    {
        unsigned int i;
        SAdaptorMethodInfo *arrMethods;
        arrMethods = new SAdaptorMethodInfo[pDMClassInfo->methodIds.Size()];
        for (i = 0; i < pDMClassInfo->methodIds.Size(); ++i)
        {
            MPI::TId methodId = pDMClassInfo->methodIds.GetAt(i);
            SDMMethodInfo *pDMMethodInfo = m_pDatabaseMethodInfo->Get(methodId);
            SAdaptorMethodInfo methodInfo;
            if (NULL != pDMMethodInfo)
            {
                methodInfo.id = methodId;
                methodInfo.szName = pDMMethodInfo->szMethodOriginalName.Get();
                methodInfo.szSignature = pDMMethodInfo->szMethodOriginalSignature.Get();
                methodInfo.cpAlreadyInvokedFlag = pDMMethodInfo->cpAlreadyInvokedFlag;
                methodInfo.uiAttributeFlags = pDMMethodInfo->uiAttributeFlags;
            }
            else
            {
                // Should not happen, but we handle this anyway
                MARTINI_INFORMATIVE2("CDataManager", 0, false,
                    "Warning: missing information for method #%d in class %s",
                    i, pCgContext->classInfo.szClassName);
                methodInfo.id = 0;
                methodInfo.szName = NULL;
                methodInfo.szSignature = NULL;
                methodInfo.cpAlreadyInvokedFlag = 0;
                methodInfo.uiAttributeFlags = 0;
            }
            arrMethods[i] = methodInfo;
        } // end of for
        pCgContext->classInfo.pMethods = arrMethods;
    }
    return MRTE_RESULT_OK;
}

TResult CDataManager::SetMethodIdWModuleId(MPI::TId *pMethodId, SEmData *pEmData,
                                           MPI::TId *pModuleId, bool bNeedNewMethodIndication)
{
    TResult retVal = MRTE_RESULT_OK;
    MARTINI_ASSERT("CEventManager", pMethodId, "");
    if (pEmData->dataType == MPI_VM)
    {
        bool bNewMethod = false;
        jmethodID jvmpiMethodId = pEmData->u.pJVMData->u.method.methodId;
        retVal = GetMpiMethodIdWModuleId(jvmpiMethodId, pMethodId, pModuleId,
            &bNewMethod, false /* bIgnoreNotFoundMethod */);
        // retVal may indicate a failure. A failure may happen in BEA 1.4.x if the
        // method is considered problematic (i.e., Exit event for this method is delivered
        // out-of-order). Such methods are ignored by Martini
        if (MRTE_SUCCEEDED(retVal))
        {
            if (bNewMethod && bNeedNewMethodIndication)
            {
                m_pEventManager->NewMethodEvent(*pMethodId, pEmData->pJniEnv);
            }
        }
    }
    else if (pEmData->dataType == MPI_INTERNAL)
    {
        *pMethodId = pEmData->u.pMpiData->methodId;
        if (pModuleId)
        {
            *pModuleId = GetMethodModuleId(*pMethodId);
        }
    }
    else
    {
        retVal = MRTE_ERROR_NOT_SUPPORTED;
    }
    return retVal;
}

MPI::TId CDataManager::AddMethodToDB(jmethodID jvmMethodId)
{
    // Get method information
    MCString strName;
    MCString strSig;
    MCString strGeneric;
    jint uiAccessFlags;
    MPI::SMethodLineNumbers lineNumbers;
    TResult res = m_pJavaInterface->GetMethodInfo(strName, strSig, strGeneric,
        &uiAccessFlags, &lineNumbers, jvmMethodId);
    if (MRTE_FAILED(res))
    {
        return 0;
    }

    // Get method's class information
    MCString strDeclaringClass;

    MPI::TId classLoaderId = 0;
    res = m_pJavaInterface->GetMethodDeclaringClassName(strDeclaringClass,
        &classLoaderId, jvmMethodId);
    if (MRTE_FAILED(res))
    {
        return 0;
    }

    MPI::TId classId = GetOrAllocateClassIdFromClassName(strDeclaringClass.Get(),
        classLoaderId);
    if (0 == classId)
    {
        MARTINI_INFORMATIVE3("CDataManager", 0, false,
            "[AddMethodToDB] failed to get id for declaring class '%s' of method '%s(%s)'",
            strDeclaringClass.Get(), strName.Get(), strSig.Get());
        return 0;
    }

    SDMClassInfo *pClassInfo = m_pDatabaseClassInfo->Get(classId);
    if (0 == pClassInfo)
    {
        MARTINI_INFORMATIVE1("CDataManager", 0, false,
            "[AddMethodToDB] lookup in m_pDatabaseClassInfo database failed for class id %u",
            classId);
        return 0;
    }

    SDMMethodInfo *pMethodInfo = new SDMMethodInfo();
    if (!pMethodInfo)
    {
        MARTINI_INFORMATIVE("CDataManager", 0, false,
            "[AddMethodToDB] error allocating memory for SDMMethodInfo");
        return 0;
    }

    pMethodInfo->pClassInfo = pClassInfo;
    pMethodInfo->pModuleInfo = m_pInitialModule;
    pMethodInfo->cpAlreadyInvokedFlag = 0;
    pMethodInfo->uiAttributeFlags = uiAccessFlags;
    pMethodInfo->lineNumbers = lineNumbers;
    pMethodInfo->vmId = jvmMethodId;

    TResult iRes = SetMethodNameAndSignature(pClassInfo, pMethodInfo,
        strName.Get(), strSig.Get(), strGeneric.Get(), uiAccessFlags);
    if (MRTE_FAILED(iRes))
    {
        return 0;
    }

    SDMMethodInfo *pTempInfo = m_pJvmpiToMpiMethodId->Get((UIOP)jvmMethodId);
    if (pTempInfo)
    {
        pMethodInfo->id = pTempInfo->id;
    }
    else
    {
        pMethodInfo->id = 0;
    }

    if (0 == pMethodInfo->id)
    {
        // Unknown method. Need to allocate a method id
        pMethodInfo->id = m_pMethodIdAllocator->AllocateId();
    }

    iRes = m_pDatabaseMethodInfo->Set(pMethodInfo->id, pMethodInfo);
    if (iRes != MRTE_RESULT_KEY_ALREADY_EXISTS)
    {
        m_pJvmpiToMpiMethodId->Set((UIOP)jvmMethodId, pMethodInfo);
    }
    return pMethodInfo->id;
}

void CDataManager::ConvertFrameData(MPI::SStackEntry *pStackEntries,
                                    jvmtiFrameInfo *pJvmtiFrameBuffer,
                                    jint frameCount)
{
    TResult res;
    jint i;
    MPI::TId mpiMethodId = 0;
    for (i = 0; i < frameCount; ++i)
    {
        bool bGenerateNewMethodEvent = false;
        res = GetMpiMethodIdWModuleId(pJvmtiFrameBuffer[i].method, &mpiMethodId,
            NULL, &bGenerateNewMethodEvent, true /* bIgnoreNotFoundMethod */);
        if (MRTE_FAILED(res))
        {
            mpiMethodId = AddMethodToDB(pJvmtiFrameBuffer[i].method);
            bGenerateNewMethodEvent = true;
        }
        if (0 == mpiMethodId)
        {
            MARTINI_INFORMATIVE1("CDataManager", 5, false,
                "[ConvertFrameData] failed to add method %p to the database",
                pJvmtiFrameBuffer[i].method);
        }
        else
        {
            if (bGenerateNewMethodEvent)
            {
                m_pEventManager->NewMethodEvent(mpiMethodId, NULL);
            }
        }
        pStackEntries[i].methodId = mpiMethodId;
        pStackEntries[i].location = pJvmtiFrameBuffer[i].location;
    }
}

void CDataManager::SetNewMethodReported(MPI::TId methodId)
{
    SDMMethodInfo *pMethodInfo = m_pDatabaseMethodInfo->Get(methodId);
    if (pMethodInfo)
    {
        pMethodInfo->bIsFirstTime = false;
    }
}

bool CDataManager::IsNewMethodReported(MPI::TId methodId)
{
    SDMMethodInfo *pMethodInfo = m_pDatabaseMethodInfo->Get(methodId);
    if (pMethodInfo)
    {
        return !(pMethodInfo->bIsFirstTime);
    }
    return false;
}

TResult CDataManager::GetThreadCurrentMonitor(MPI::TId *pMonId, jthread thread)
{
    jobject monObj;
    TResult res = m_pJavaInterface->GetCurrentContendedMonitor(&monObj, thread);
    if (0 == monObj)
    {
        // The thread does not hold any monitor. This is not an error
        *pMonId = 0;
        return MRTE_RESULT_OK;
    }

    if (MRTE_SUCCEEDED(res))
    {
        JNIEnv *pJniEnv;
        res = m_pJavaInterface->GetJNIEnv(&pJniEnv);
        if (MRTE_SUCCEEDED(res))
        {
            jlong tag;
            res = GetObjectInfoManager()->GetOrCreateTag(pMonId, &tag, monObj,
                pJniEnv, true /* bSaveInDb */);
            if (MRTE_SUCCEEDED(res))
            {
                pJniEnv->DeleteLocalRef(monObj);
            }
            else
            {
                MARTINI_INFORMATIVE1("CDataManager", 0, false,
                    "[GetThreadCurrentMonitor] failed to allocate object id for JNI ref %p",
                    monObj);
            }
        }
        else
        {
            MARTINI_INFORMATIVE("CDataManager", 0, false,
                "[GetThreadCurrentMonitor] failed to get JNI env for calling thread");

        }
    }
    return res;
}

TResult CDataManager::GetClassInstrumentationStatus(const MPI::TId classId,
                                                    bool *pEventsEnabled)
{
    SDMClassInfo *pDMClassInfo = m_pDatabaseClassInfo->Get(classId);
    if (NULL == pDMClassInfo)
    {
        // Fatal error. Entry for this class must exist
        MARTINI_ERROR1("CDataManager", "Failed to get instrumentation status "
            "for class id %d. Class information entry does not exist",
            (U32)classId);
        return MRTE_ERROR_FAIL;
    }

    *pEventsEnabled = pDMClassInfo->bEventsEnabled;

    return MRTE_RESULT_OK;
}

bool CDataManager::AreEventsEnabledForClass(MPI::TId classId)
{
    bool bEventsEnabled = false;
    TResult res = GetClassInstrumentationStatus(classId, &bEventsEnabled);
    if (MRTE_FAILED(res))
    {
        MARTINI_ERROR1("CDataManager", "Unable to get instrumentation status for class id %u",
            (U32)classId);
    }
    return bEventsEnabled;
}

//
// Generates a unique class name for a class
//
// Parameters:
//   strUniqueName [out]     : unique name (name@loaderId)
//   strClassName  [in]      : class friendly/binary name
//   classLoaderId [in]      : class loader id (0 for the bootstrap loader)
//
void CDataManager::MakeUniqueClassName(RTUtil::MCString& strUniqueName,
                                       const RTUtil::MCString& strClassName,
                                       MPI::TId classLoaderId)
{
    char szUniqueName[1000];
    sprintf(szUniqueName, "%s@%u", strClassName.Get(), (U32)classLoaderId);
    strUniqueName.Set(szUniqueName);
}

extern "C"
bool ClassInfoDbIterationCallBack(void *pParameter, const MPI::TId key, SDMClassInfo* data)
{
    TIdList *pList = (TIdList*)pParameter;
    if (NULL != data->pClassBytes && data->pClassBytes->uiSize > 0)
    {
        TMRTEHandle hLast = pList->GetLast();
        pList->InsertAfter(hLast, key);
    }
    return true;
}


#if 0
CUserData *CDataManager::GetObjectUserData(UIOP JVMPIObjId)
{
    MARTINI_ASSERT("CDataManager", m_pObjectAllocEvent, "");

    return m_pObjectAllocEvent->GetObjectUserData(JVMPIObjId);
}
#endif

#ifdef _DEBUG
// DEBUG services
void CDataManager::PrintDebugStatistics()
{
    fprintf(stderr, "Data Manager statistics\n");
    fprintf(stderr, "=======================\n");
    fprintf(stderr, "Class database (m_pDatabaseClassInfo):\n");
    m_pDatabaseClassInfo->PrintDebugStatistics(stderr);
    fprintf(stderr, "\n");
    fprintf(stderr, "Method database (m_pDatabaseMethodInfo):\n");
    m_pDatabaseMethodInfo->PrintDebugStatistics(stderr);
    fprintf(stderr, "\n");
    fprintf(stderr, "Class Name -> ID mapping (m_pClassNameToId):\n");
    m_pClassNameToId->PrintDebugStatistics(stderr);
    fprintf(stderr, "\n");
    fprintf(stderr, "Utility Class database (m_pDatabaseClassesWithCompleteInfo):\n");
    m_pDatabaseClassesWithCompleteInfo->PrintDebugStatistics(stderr);
    fprintf(stderr, "\n");
    fprintf(stderr, "VM ID -> MPI ID mapping (m_pJvmpiToMpiMethodId):\n");
    m_pJvmpiToMpiMethodId->PrintDebugStatistics(stderr);
}

#endif

/**
 * Updates the flag to indicate if instance data collection should be collected.
 */
void CDataManager::SetHeapObjDataCollection(bool heapObjDataCollection) {
	GetObjectInfoManager()->setHeapObjDataCollection(heapObjDataCollection);
}

/**
 * Checks the flag to indicate if instance data should be collected.
 */
bool CDataManager::isHeapObjDataCollectionEnabled() {
	return GetObjectInfoManager()->isHeapObjDataCollectionEnabled();
}


/*
 * Obtains heap data information about the passed Object using the
 * TPTP ID. The Object information is obtained from the database. 
 */
TResult CDataManager::RunHeapObjDataCollection(THeapObjectAnalysis *heapObjectAnalysis) {

	bool bAttached;
	JNIEnv *pJNIEnv;

	CObjectInfoManager *s_pObjectInfoManager = GetObjectInfoManager();
	IJVM * s_pJavaInterface = GetJVMInterface();
	TResult res = s_pJavaInterface->AttachCurrentThread(&pJNIEnv, &bAttached);
	if (MRTE_FAILED(res))
	{
		MARTINI_INFORMATIVE("CJVMTIInterface", 0, false,
			"[RunHeapObjDataCollection] failed to attach the calling thread to the VM for Object data collection.");
		return MRTE_ERROR_FAIL;
	}

	MPI::TId objectId = (MPI::TId)atol(heapObjectAnalysis->TId);
	// Obtains the Object from the data store and analyzes its values.
	jstring stringObjValues = (jstring)s_pObjectInfoManager->GetObjectValuesFromInstDataMap(objectId, pJNIEnv);

	// Convert the String.
	if(stringObjValues != 0) {
		const char *utfString;
		utfString = pJNIEnv->GetStringUTFChars(stringObjValues, NULL);
		if(utfString == NULL) {
			return MRTE_ERROR_FAIL;
		}

		// Gather the size of the Object and append it to the result.
		jobject objectRef = s_pObjectInfoManager->GetObjectFromInstDataMap(objectId, pJNIEnv);
		char sizeString[34] = { '\0' }; // 34 bytes will hold a size up to 2^64 bytes
		if(objectRef) {
			MPI::SObjectInfo objectInfo;
			TResult infoResult = s_pJavaInterface->GetObjectInfo(&objectInfo, objectRef);
			if(infoResult == MRTE_RESULT_OK)
				sprintf(sizeString, "<size>%d</size>", objectInfo.uiSize);
			else
				sprintf(sizeString, "<size>0</size>");
		}

		// Copies the result string to be returned to the client.
		char * result = (char *)malloc(strlen(utfString)+strlen(sizeString)+100);
		sprintf(result, "%s%s", utfString, sizeString);
		
		pJNIEnv->ReleaseStringUTFChars(stringObjValues, utfString);

		// Set the max size for the returned result value to be 128 KB = (131072 * 1 byte).
    	// Matches the maximum value set in ACCollector.
		const int RESULT_VALUE_MAX_SIZE = 131072;
		if(strlen(result) == 0) {
			sprintf((char *)heapObjectAnalysis->resultValue, "<error type=\"empty\" />");
		} else if((strlen(result) + 1) > RESULT_VALUE_MAX_SIZE) {
			sprintf((char *)heapObjectAnalysis->resultValue, "<error type=\"maxSize\" />");
		} else {
			sprintf((char *)heapObjectAnalysis->resultValue, "%s", result);
		}

		free(result);

	} else {
		// Set result value to an error so that it is not printed.
		sprintf((char *)heapObjectAnalysis->resultValue, "<error type=\"conversionFailure\" />");

		if(bAttached)
		{
			s_pJavaInterface->DetachCurrentThread();
		}
		return MRTE_ERROR_FAIL;
	}
	
	if(bAttached)
	{
		s_pJavaInterface->DetachCurrentThread();
	}
	return MRTE_RESULT_OK;
}

