/*****************************************************************************
 * Copyright (c) 1997, 2010 Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/

#ifndef _MRTE_DATA_MANAGER_H
#define _MRTE_DATA_MANAGER_H

#include "MpiAPI.h"
#include "ParamChecker.h"
#include "IdAllocator.h"
#include "MRTEInfrastructureDefinitions.h"
#include "MString.h"
#include "MVector.h"
#include "MHash.h"
#include "MList.h"
#include "IJVM.h"
#include "IDataManager.h"
#include "CGAdaptor.h"
#include "ThreadInfoManager.h"
#include "ObjectInfoManager.h"
#include "OSA.h"

#define RTUTIL_NS Martini::RTUtil

namespace Martini { namespace JPI
{
    class CJPIKernel;
    class CEventManager;
    struct SEMData;
    class CClassPrepareEventObserver;
    class CMethodJitEventObserver;
    class CThreadStartEventObserver;
    class CThreadEndEventObserver;
    class CDMVMInitEventObserver;
    class CDMVMShutdownEventObserver;
    class CDMGCStartEventObserver;
    class CJavaInstrumentorManager;

    typedef RTUtil::MList<MPI::TId> TIdList;

    //TODO: attach/detach: move SDM* structures back to their place (private of DataManager)
    struct SDMModuleInfo
    {
        RTUtil::MCString    szName;    // name 
        MPI::TId            id;        // id
    };

    struct SDMClassInfo
    {
        SDMClassInfo() : uiNumMethods(0), id(0), pClassBytes(NULL), jniGlobalRef(NULL),
            uiAttributeFlags(0), bFromInstrumentation(false), bEventsEnabled(false) {}
        
        RTUtil::MCString        szName;         // class name, including the Generic signature
                                                // for Generic classes (Java 5+).
                                                // available only after the class was prepared.

        RTUtil::MCString        szSrcFileName;  // source file name
        RTUtil::MCString        szNativeName;   // class name extracted from the class file
                                                // during instrumentation.
                                                // available only for instrumented classes.
                                                // does not contain a Generic signature for
                                                // Generic classes.

        RTUtil::MVector<MPI::TId> methodIds;    // class methods (ids only)
        unsigned int            uiNumMethods;   // number of methods in the class
        MPI::TId                id;             // MPI class id
        Infrastructure::SClassFile *pClassBytes;// class bytes after applying basic
                                                // instrumentation

        jclass                  jniGlobalRef;   // jclass global reference. Needed when
                                                // redefining the class

        unsigned int            uiAttributeFlags;       // class attributes. Used as bitwise  
                                                        // flags, as defined by 
                                                        // JIE::EClassAttributeFlags

        bool                    bFromInstrumentation;   // true - information originated from
                                                        // instrumentation
                                                        // false - information originated from
                                                        // ClassPrepare event
        
        bool                    bEventsEnabled;         // true - class methods were 
                                                        // instrumented for generating
                                                        // events (enter-leave or object alloc).
                                                        // false - otherwise
    };
    
    struct SDMMethodInfo
    {
        SDMMethodInfo() : pClassInfo(NULL), pModuleInfo(NULL), bIsFirstTime(true), id(0),
            uiAttributeFlags(0), cpAlreadyInvokedFlag(0), vmId(0) {}
        RTUtil::MCString    szFriendlyName;
        RTUtil::MCString    szFriendlySignature;
        RTUtil::MCString    szMethodOriginalName;
        RTUtil::MCString    szMethodOriginalSignature;
        MPI::SMethodLineNumbers  lineNumbers;
        SDMClassInfo       *pClassInfo;
        SDMModuleInfo      *pModuleInfo;
        bool                bIsFirstTime;
        MPI::TId            id;
        unsigned int        uiAttributeFlags;       // method attributes. Used as bitwise  
                                                    // flags, as defined by 
                                                    // JIE::EMethodAttributeFlags
        JIE::TConstantPoolIndex cpAlreadyInvokedFlag;   // constant pool index of the 
                                                        // 'already invoked' static field
                                                        // associated with the method
        jmethodID           vmId;                   // method's VM id
    };

    // Class that handles all data managements. The data manager purpose is to serve clients' 
    // data retrieval requests
    class CDataManager : public Infrastructure::IDataManager
    {

        typedef RTUTIL_NS::MHashSync<MPI::TId,  SDMMethodInfo*> TMethodInfoDB;
        typedef RTUTIL_NS::MHashSync<MPI::TId, SDMClassInfo*> TClassInfoDB;
        typedef RTUTIL_NS::MStrHashSync<char, MPI::TId> TStringToIdHash;
        typedef RTUTIL_NS::MHashSync<UIOP, SDMMethodInfo*> TJvmpiToMpiIdHash;

    public:
        // IMpi methods

        virtual TResult GetModuleInfo(MPI::TId clientId,
                                      MPI::TId moduleId,
                                      MPI::BitSet requestedDataTypes, 
                                      MPI::SModuleInfo *pData);

        virtual TResult GetClassInfo(MPI::TId clientId,
                                     MPI::TId classId,
                                     MPI::BitSet requestedDataTypes, 
                                     MPI::SClassInfo *pData);

        virtual TResult GetMethodInfo(MPI::TId clientId,
                                      MPI::TId methodId,
                                      MPI::BitSet requestedDataTypes, 
                                      MPI::SMethodInfo *pData);

        virtual TResult GetObjectInfo(MPI::TId clientId,
                                      MPI::TId objectId,
                                      MPI::BitSet requestedDataTypes,
                                      MPI::SObjectDataRequest *pData);

        virtual TResult GetThreadInfo(MPI::TId clientId,
                                      MPI::TId threadId,
                                      MPI::BitSet requestedDataTypes, 
                                      MPI::SThreadInfo *pData,
                                      MPI::TDotNetUnmanagedStackWalkCallBack pfnStackWalk);

        virtual TResult GetAllThreadsInfo(MPI::TId clientId, 
                                          U32 maxFrameCount, 
                                          MPI::BitSet requestedDataTypes, 
                                          MPI::SThreadInfoArray *pData);


        virtual TResult GetObjectReferences(MPI::TId clientId, 
                                            MPI::SObjectReferenceInfo *pData);

        virtual TResult GetCompiledMethodInfo(MPI::TId clientId,
                                              const char* szModuleName,
                                              UIOP uiIpOffset,
                                              MPI::BitSet requestedDataTypes,
                                              MPI::TId *pMethodId,
                                              MPI::SMethodInfo *pData)
        {
            return MRTE_ERROR_NOT_IMPLEMENTED;
        }

		/**
		 * Sets the flag to indicate instance data collection should be collected.
		 */
        virtual void SetHeapObjDataCollection(bool heapObjDataCollection);

		/**
		 * Checks the flag to indicate if instance data should be collected.
		 */
        virtual bool isHeapObjDataCollectionEnabled();

		/**
		 * Runs heap instance data collection to analyze the Object.
		 */
        virtual TResult RunHeapObjDataCollection(THeapObjectAnalysis *tptpID);

        // CData Manager public methods

        void SetInternalModules(CEventManager *pEventManager, 
                                Infrastructure::CParamChecker *pParamChecker, 
                                CJPIKernel *pKernel, CJavaInstrumentorManager *pJIM);
        
        TResult ClassPrepareHandler(struct SEmData *pEvent);
        TResult SetClassInstrumentationInfo(Infrastructure::SAdaptorClassInfo *pClassInfo);
        TResult SetClassFileBuffer(MPI::TId classId, Infrastructure::SClassFile *pClassBytes);
        void NotifyOnProfilerEventRegistration(MPI::TId clientId, MPI::IEventObserver &event);

        TResult SetMethodIdWModuleId(MPI::TId *pMethodId,
                                     SEmData *pEmData, MPI::TId *pModuleId,
                                     bool bNeedNewMethodIndication);
        TResult GetMpiMethodIdWModuleId(jmethodID jvmpiMethodId, MPI::TId *pMpiId,
                                           MPI::TId *pModuleId,
                                           bool *pbNewMethod,
                                           bool bIgnoreNotFoundMethod);
        MPI::TId GetMethodModuleId(MPI::TId mpiMethodId);
        MPI::TId GetMethodClassId(MPI::TId methodId);
        TResult SetMethodJitted(MPI::TId methodId);
        MPI::TId GetConstantModuleId();
        TResult GetUpdatedMethodName(const char *szClassName, const char *szMethodName,
                                     RTUtil::MCString *szUpdatedMethodName);

        TResult GetInstrumentedClasses(TIdList *pClassIdList);

        TResult GetClassInstrumentationInfo(RTUtil::MCString *pStrNativeClassName,
                                            Infrastructure::SClassFile *pClassWithBasicInstr, 
                                            jclass *pClassJniGlobalRef, 
                                            MPI::TId classId);

        TResult GetClassInstrContext(Infrastructure::CContext *pCgContext, MPI::TId classId);

        TResult GetClassGlobalJniRef(jclass *pClassJniGlobalRef, MPI::TId classId);

        TResult GetClassInstrumentationStatus(const MPI::TId classId, bool *pEventsEnabled);

        TResult SetClassInstrumentationStatus(const MPI::TId classId, bool bEventsEnabled);

        bool AreEventsEnabledForClass(MPI::TId classId);

        TResult NewClass(const MPI::TId classId, 
                         const char *szNativeClassName,
                         const MPI::TId loaderObjectId,
                         bool bFromInstrumentation);

        CThreadInfoManager* GetThreadInfoManager() { return m_pThreadInfoManager; }

        CObjectInfoManager* GetObjectInfoManager() { return m_pObjectInfoManager; }

        IJVM* GetJVMInterface() { return m_pJavaInterface; }

        TResult GetClassName(const char *szClassName, RTUtil::MCString& name);

        MPI::TId GetOrAllocateClassIdFromClassName(const char *szNativeClassName,
                                                   MPI::TId loaderObjectId);

        MPI::TId GetClassIdFromClassName(const char *szNativeClassName, 
                                         MPI::TId loaderObjectId);

        void SetNewMethodReported(MPI::TId methodId);

        bool IsNewMethodReported(MPI::TId methodId);

        CDataManager(MPI::TId moduleId);
        ~CDataManager(void);

    private:
        TResult ParseClassName(const char *szName, const char *szGenericSig, 
            RTUtil::MCString *pOutName);
        TResult ParseMethodName(const char *szName, const char *szGenericSig, 
            RTUtil::MCString *pOutName);
        TResult ParseMethodPrototype(const char *szPrototype, const char *szGenericSig, 
            unsigned int uiFlags, RTUtil::MCString *pOutPrototype);

        TResult RegisterClassPrepareEvent();
        TResult Unmangle(const char *szSrc, int *pPlace, RTUtil::MCString *szDest);
        TResult UnmangleName(const char *szSrc, int *pPlace, RTUtil::MCString *szDest);
        void RequestMethodInfoFromStdPI(jmethodID jvmpiMethodId);
        TResult GetGenericClassName(struct SEmData *pEvent, RTUtil::MCString& name);
        MPI::TId GetMethodId(struct SEmData *pEvent, unsigned int uiIndex);
        unsigned int GetMethodStartLineNumber(SEmData *pEvent, unsigned int uiIndex);
        unsigned int GetMethodEndLineNumber(SEmData *pEvent, unsigned int uiIndex);
        unsigned int GetMethodAccessFlags(SEmData *pEvent, unsigned int uiIndex);
        void FillMethodDB(struct SEmData *pEvent, SDMClassInfo *pClassInfo);
        void FillClassDB(struct SEmData *pEvent, SDMClassInfo *pClassInfo, MPI::TId classId);
        
        MPI::TId AddMethodToDB(SDMClassInfo *pClassInfo, 
                               struct SEmData *pEvent, 
                               unsigned int uiIndex);
        MPI::TId AddMethodToDB(jmethodID jvmMethodId);
        TResult SetMethodNameAndSignature(SDMClassInfo *pClassInfo, 
                                          SDMMethodInfo *pMethodInfo,
                                          const char *szMethodName,
                                          const char *szPrototype,
                                          const char *szGeneric,
                                          unsigned int uiFlags);
        bool IsProblematicMethod(const char *szClassName, 
                                 const char *szMethodName,
                                 const char *szMethodSig);
        void ConvertFrameData(MPI::SStackEntry *pStackEntries, 
                              jvmtiFrameInfo *pJvmtiFrameBuffer,
                              jint frameCount);
        
        TResult GetThreadCurrentMonitor(MPI::TId *pMonId, jthread thread);

        void MakeUniqueClassName(RTUtil::MCString& strUniqueName,
                                 const RTUtil::MCString& strClassName,
                                 MPI::TId classLoaderId);

        // a pointer to the JVMPI/JVMTI interface
        IJVM *m_pJavaInterface;

        // a pointer to the event manager
        CEventManager *m_pEventManager;

        // a pointer to the param checker
        Infrastructure::CParamChecker *m_pParamChecker;

        // a pointer to the kernel
        CJPIKernel *m_pKernel;
        
        // a pointer to the java instrumentation manager
        CJavaInstrumentorManager *m_pJavaInstrumentorManager;
        
        // module id
        MPI::TId m_moduleId;

        // method name database. maps between a method MPI ID and its information
        TMethodInfoDB *m_pDatabaseMethodInfo;

        // class name database. maps between a class MPI ID and its information
        TClassInfoDB *m_pDatabaseClassInfo;

        // Maps a class name to its MPI id.
        // Used for keeping track of classes whos IDs were assigned by the instrumentor.
        // The class name is stored in the following format:
        // className@classLoaderObjectId, where:
        // className: is the friendly/binary class name (using . separator)
        // classLoaderObjectId: an MPI id which identified the class loader of the class,
        //                      or 0 for the bootstrap class loader.
        TStringToIdHash *m_pClassNameToId;

        // mapping between class name and its MPI id
        // keeps track of classes for which we have all required information.
        // used to prevent the processing of the same class twice due to multiple
        // ClassLoad (JVMPI)/ClassPrepare (JVMTI) events for the same class
        TStringToIdHash *m_pDatabaseClassesWithCompleteInfo;

        // mapping between JVMPI method id to MPI method id
        TJvmpiToMpiIdHash *m_pJvmpiToMpiMethodId;

        // A pointer to the Threads Manager object that manages thread information
        CThreadInfoManager *m_pThreadInfoManager;

        // A pointer to the Object Info Manager that manages object information
        CObjectInfoManager *m_pObjectInfoManager;

        // JIT module info
        SDMModuleInfo m_DatabaseModuleInfo[Infrastructure::MAX_JAVA_MODULE_ID + 1];        
        SDMModuleInfo *m_pInitialModule; // patch for BEA

        // Internal event observers
        CClassPrepareEventObserver *m_pClassPrepareEvent;
        CMethodJitEventObserver *m_pMethodJitLoadedEvent;
        CThreadStartEventObserver *m_pThreadStartEvent;
        CThreadEndEventObserver *m_pThreadEndEvent;
        CDMVMInitEventObserver *m_pVMInitEvent;
        CDMVMShutdownEventObserver *m_pVMShutdownEvent;
        CDMGCStartEventObserver *m_pGCStartEvent;

        // Class ID allocator
        Infrastructure::CIdAllocator *m_pClassIdAllocator;

        // Method ID allocator
        Infrastructure::CIdAllocator *m_pMethodIdAllocator;

        // Critical Section for synchronizing access to the class database
        OSA::IThreadSync *m_pcsClassInfoDb;

        // when the vendor is BEA, there is a fixed module-id (always initializing to MODULE_JIT)
        MPI::TId m_constantModuleId;

#ifdef _DEBUG
        // DEBUG services
    public:
        void PrintDebugStatistics();
#endif

    }; // class CDataManager

    extern "C"
    bool ClassInfoDbIterationCallBack(void *pParameter, 
                                      const MPI::TId key, 
                                      SDMClassInfo* data);

}}

#endif

