/*****************************************************************************
 * Copyright (c) 1997, 2009 Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/

#include "EventDispatcher.h"
#include "EventManager.h"
#include "DataManager.h"
#include "OSA.h"
#include "LogAssert.h"

using namespace Martini;
using namespace Martini::JPI;
using namespace Martini::MPI;
using namespace Martini::OSA;
using namespace Martini::Infrastructure;

//////////////////////////////////////////////////////////////////////////
// Common functions

TId GetThreadIdFromTls(const JNIEnv *pJniEnv, 
                       CDataManager *pDataManager,
                       jthread thread)
{
    CThreadInfoManager *pThreadInfoManager = pDataManager->GetThreadInfoManager();
    STlsThreadInfo *pTlsThreadInfo = NULL;
    TId threadId;
    // Check TLS data. If it does not contain the thread id, update the TLS.
    // If the thread is not known to the Thread Manager, generate an EV_THREAD_START
    // event
    pTlsThreadInfo = pThreadInfoManager->GetTlsInfo();
    if (NULL != pTlsThreadInfo)
    {
        threadId = pTlsThreadInfo->threadId;
    }
    else
    {
        threadId = pThreadInfoManager->GetThreadIdAndUpdateTls(pJniEnv, thread);
    }
    return threadId;
}

template <class EventData>
void GetEnterLeaveEventData(EventData *pDataOut,
                            SEmData *pDataIn, 
                            BitSet dataItems,
                            CDataManager *pDataManager)
{
    pDataOut->validData = DR_NONE;

    if (DR_CONTAINS_THREAD_ID(dataItems))
    {
        pDataOut->threadId = GetThreadIdFromTls(pDataIn->pJniEnv, pDataManager, NULL);
        pDataOut->validData |= DR_THREAD_ID;
    }

    if (DR_CONTAINS_THREAD_CPU_TIME(dataItems))
    {
        pDataManager->GetJVMInterface()->GetCurrentThreadCpuTime(
            &pDataOut->uiCpuNanos);
        pDataOut->validData |= DR_THREAD_CPU_TIME;
    }

    if (DR_CONTAINS_THREAD_ELAPSED_TIME(dataItems))
    {
        pDataOut->uiElapsedNanos = GetTimeStamp();
        pDataOut->validData |= DR_THREAD_ELAPSED_TIME;
    }
}

template <class EventData>
void GetMonitorEventData(EventData *pDataOut,
                         jobject monitorJniRef,
                         jthread threadJniRef,
                         const JNIEnv *pJniEnv,
                         BitSet dataItems,
                         CDataManager *pDataManager)
{
    pDataOut->validData = DR_NONE;
    if (DR_CONTAINS_OBJECT_ID(dataItems))
    {
        CObjectInfoManager *pObjectInfoManager = pDataManager->GetObjectInfoManager();
        jlong tag;
        TResult res = pObjectInfoManager->GetOrCreateTag(&(pDataOut->objectId), &tag, 
            monitorJniRef, pJniEnv, true /* bSaveInDb */);
        if (MRTE_FAILED(res))
        {
            MARTINI_INFORMATIVE1("GetMonitorEventData", 0, false, 
                "failed to allocate object id for JNI ref %p", monitorJniRef);
            return;
        }

        MARTINI_INFORMATIVE2("GetMonitorEventData", 5, false,
            "monitor JNI ref %p is associated with object id %u",
            monitorJniRef, pDataOut->objectId);
        pDataOut->validData |= DR_OBJECT_ID;
    }

    if (DR_CONTAINS_THREAD_ID(dataItems))
    {
        pDataOut->threadId = pDataManager->GetThreadInfoManager()->GetThreadId(pJniEnv);
        if (pDataOut->threadId != 0)
        {
            pDataOut->validData |= DR_THREAD_ID;
        }
    }
    

}

template <class EventData>
void GetThreadEventData(EventData *pDataOut,
                        const JNIEnv *pJniEnv,
                        jthread threadJniRef,
                        BitSet dataItems,
                        CDataManager *pDataManager)
{
    if (DR_CONTAINS_THREAD_ID(dataItems)) 
    {
        pDataOut->threadId = pDataManager->GetThreadInfoManager()->GetThreadId(pJniEnv);
        if (pDataOut->threadId != 0)
        {
            pDataOut->validData |= DR_THREAD_ID;
        }
    }

    if (DR_CONTAINS_OBJECT_ID(dataItems) || DR_CONTAINS_CLASS_ID(dataItems))
    {
        TId objectId;
        jlong tag;
        CObjectInfoManager *pObjectInfoManager = pDataManager->GetObjectInfoManager();
        TResult res = pObjectInfoManager->GetOrCreateTag(&objectId, &tag, threadJniRef, 
            pJniEnv, true /* bSaveInDb */);
        if (MRTE_FAILED(res))
        {
            MARTINI_INFORMATIVE1("GetThreadEventData", 0, false, 
                "failed to allocate object id for thread %p", threadJniRef);
            return;
        }
        if (DR_CONTAINS_OBJECT_ID(dataItems))
        {
            pDataOut->objectId = objectId;
            pDataOut->validData |= DR_OBJECT_ID;
        }
        if (DR_CONTAINS_CLASS_ID(dataItems))
        {
            SObjectInfo *pObjectInfo = pObjectInfoManager->GetInfo(objectId, pJniEnv);
            pDataOut->classId = pObjectInfo->classId;
            if (pDataOut->classId != 0)
            {
                pDataOut->validData |= DR_CLASS_ID;
            }
        }
    }
}

//////////////////////////////////////////////////////////////////////////
// CEventDispatcher implementation

//
// Destructor
//
CEventDispatcher::~CEventDispatcher()
{
    RTUtil::FreeListItems(m_observers);
}

TResult CEventDispatcher::Register(CObserverInfo *observerInfo)
{
	TMRTEHandle handle = m_observers.GetFirst();    
    
    // Verify that the client is not already registered for this event
    CObserverInfo *pData;
    while (NULL != handle)
    {
        pData = m_observers.GetData(handle);
        if (pData && pData->clientId == observerInfo->clientId)
        {
            return MRTE_ERROR_FAIL; // Client already registered
        }
		handle = m_observers.GetNext(handle);
    }

    
    // Find the place in the clients list to which to add the registered client.
    // This is done according to the client's priority.
    // 
    // Note: the following implementation is not thread safe, but since event registration
    // is currently being allowed only at initialization time, and since all initializations 
    // are done from the same thread, this is ok.
	handle = m_observers.GetFirst();
    while (NULL != handle)
    {
        if ((m_observers.GetData(handle))->priority < observerInfo->priority)
        {
            // Found the correct place for this client
            break;
        }
		handle = m_observers.GetNext(handle);
    }
	if (NULL == handle)
	{
        // Add the observer to the end of the list
		handle = m_observers.GetLast();
		m_observers.InsertAfter(handle, observerInfo);
	}
	else
	{
        // Add the observer to the place we found
		m_observers.InsertBefore(handle, observerInfo);
	}
    return MRTE_RESULT_OK;
}

void CEventDispatcher::NotifyObservers(SEmData *pData)
{
    CObserverInfo *pObserverInfo;

    TMRTEHandle handle = m_observers.GetFirst(); 
    while (handle != NULL)
    {
        pObserverInfo = m_observers.GetData(handle);
        if (pObserverInfo->bIsEnabled)
        {
            if (pObserverInfo->bIsInternal)
            {
                // For internal observers, use their special HandleEvent callback
                //TODO: This "hack" for internal observer is not clean. Consider removing
                //      internal observers entirely and use regular observers.
                //      Note that currently, some internal observers has access to data
                //      that is not available for a regular observer for the same event
                //      (e.g., Thread Start internal observer has access to the JNI Env).
                IInternalEventObserver *pInternalObserver = 
                    (IInternalEventObserver*)pObserverInfo->pObserver;
                pInternalObserver->HandleEvent(pData, NULL);
            }
            else
            {
                // For regular observers, delegate processing to the appropriate dispatcher
                Notify(pData, pObserverInfo->pObserver, pObserverInfo->dataRequestType);
            }
        }
        handle = m_observers.GetNext(handle);
    }
}

void CEventDispatcher::UnregisterAll()
{
    TMRTEHandle handle = m_observers.GetFirst();    
    while (handle != NULL)
    {
        CObserverInfo* pObserverInfo = m_observers.DeleteItem(handle);
        if (pObserverInfo)
        {
            delete pObserverInfo; 
        }
        handle = m_observers.GetFirst();
    }
}

//////////////////////////////////////////////////////////////////////////
// CMethodEnterEventDispatcher

void CMethodEnterEventDispatcher::NotifyMethodEnterObservers(SMethodEnterEventData data)
{
    CObserverInfo *pObserverInfo;
    IMethodEnterEventObserver *pObserver;

    TMRTEHandle handle = m_observers.GetFirst(); 
    while (handle != NULL)
    {
        pObserverInfo = m_observers.GetData(handle);
        if (pObserverInfo->bIsEnabled)
        {
            pObserver = (IMethodEnterEventObserver*)pObserverInfo->pObserver;
            pObserver->HandleEvent(data);
        }
        handle = m_observers.GetNext(handle);
    }
}

void CMethodEnterEventDispatcher::Notify(SEmData *pData, 
                                         IEventObserver *pObserver,
                                         BitSet dataItems)
{
    SMethodEnterEventData methodEnterData;
    methodEnterData.validData = DR_NONE;
    CDataManager *pDataManager = m_pEventManager->GetDM();

    if (DR_CONTAINS_MODULE_ID(dataItems))
    {
        TResult res = pDataManager->SetMethodIdWModuleId(&methodEnterData.methodId, pData, 
            &methodEnterData.moduleId, true /* bNeedNewMethodIndication */);
        if (MRTE_SUCCEEDED(res))
        {
            methodEnterData.validData |= DR_MODULE_ID;
            methodEnterData.validData |= DR_METHOD_ID;
        }
        else
        {
            // Can't find method id. Do not report the event
            return;
        }
    }
    else if (DR_CONTAINS_METHOD_ID(dataItems)) 
    {
        TResult res = pDataManager->SetMethodIdWModuleId(&methodEnterData.methodId, pData, 
            NULL, true /* bNeedNewMethodIndication */);
        if (MRTE_SUCCEEDED(res))
        {
            methodEnterData.validData |= DR_METHOD_ID;
        }
        else
        {
            // Can't find method id. Do not report the event
            return;
        }
    }
    
    GetEnterLeaveEventData(&methodEnterData, pData, dataItems, pDataManager);

    IMethodEnterEventObserver *pConcreteObserver = (IMethodEnterEventObserver*)pObserver;
    pConcreteObserver->HandleEvent(methodEnterData);
}

//////////////////////////////////////////////////////////////////////////
// CMethodLeaveEventDispatcher

void CMethodLeaveEventDispatcher::NotifyMethodLeaveObservers(SMethodLeaveEventData data)
{
    CObserverInfo *pObserverInfo;
    IMethodLeaveEventObserver *pObserver;

    TMRTEHandle handle = m_observers.GetFirst(); 
    while (handle != NULL)
    {
        pObserverInfo = m_observers.GetData(handle);
        if (pObserverInfo->bIsEnabled)
        {
            pObserver = (IMethodLeaveEventObserver*)pObserverInfo->pObserver;
            pObserver->HandleEvent(data);
        }
        handle = m_observers.GetNext(handle);
    }
}

void CMethodLeaveEventDispatcher::Notify(SEmData *pData, 
                                         IEventObserver *pObserver,
                                         BitSet dataItems)
{
    SMethodLeaveEventData methodLeaveData;
    methodLeaveData.validData = DR_NONE;
    CDataManager *pDataManager = m_pEventManager->GetDM();

    if (DR_CONTAINS_METHOD_ID(dataItems)) 
    {
        TResult res = pDataManager->SetMethodIdWModuleId(&methodLeaveData.methodId, pData, 
            NULL, false /* bNeedNewMethodIndication */);
        if (MRTE_SUCCEEDED(res))
        {
            methodLeaveData.validData |= DR_METHOD_ID;
        }
        else
        {
            // Can't find method id. Do not report the event
            return;
        }
    }
    
    GetEnterLeaveEventData(&methodLeaveData, pData, dataItems, pDataManager);

    IMethodLeaveEventObserver *pConcreteObserver = (IMethodLeaveEventObserver*)pObserver;
    pConcreteObserver->HandleEvent(methodLeaveData);
}

//////////////////////////////////////////////////////////////////////////
// CInternalEventDispatcher

void CInternalEventDispatcher::Notify(SEmData *pData, 
                                      IEventObserver *pObserver,
                                      BitSet dataItems)
{
    IInternalEventObserver *pConcreteObserver = (IInternalEventObserver*)pObserver;
    pConcreteObserver->HandleEvent(pData, NULL);
}

//////////////////////////////////////////////////////////////////////////
// CNewMethodEventDispatcher

void CNewMethodEventDispatcher::Notify(SEmData *pData, 
                                       IEventObserver *pObserver,
                                       BitSet dataItems)
{
    SNewMethodEventData newMethodData;
    newMethodData.validData = DR_NONE;
    CDataManager *pDataManager = m_pEventManager->GetDM();

    if (DR_CONTAINS_METHOD_ID(dataItems))
    {
        TResult res = pDataManager->SetMethodIdWModuleId(&newMethodData.methodId, pData, NULL, 
            false /* bNeedNewMethodIndication */);
        if (MRTE_SUCCEEDED(res))
        {
            newMethodData.validData |= DR_METHOD_ID;
        }
        else
        {
            // Can't find method id. Do not report the event
            return;
        }
    }
    
    INewMethodEventObserver *pConcreteObserver = (INewMethodEventObserver*)pObserver;
    pConcreteObserver->HandleEvent(newMethodData);
}

//////////////////////////////////////////////////////////////////////////
// CJittedMethodLoadedEventDispatcher

void CJittedMethodLoadedEventDispatcher::Notify(SEmData *pData, 
                                                IEventObserver *pObserver,
                                                BitSet dataItems)
{
    SJittedMethodLoadedEventData jitData;
    jitData.validData = DR_NONE;
    jitData.modulePrejitInfo.pPrejitModuleBaseAddr = NULL;
    jitData.modulePrejitInfo.szModuleName = NULL;

    CDataManager *pDataManager = m_pEventManager->GetDM();
    SJVMData *pJVMData = pData->u.pJVMData;

    if (DR_CONTAINS_METHOD_ID(dataItems)) 
    {
        jmethodID jvmMethodId = pJVMData->u.methodLoad.methodId;
        TResult res = pDataManager->GetMpiMethodIdWModuleId(jvmMethodId, &jitData.methodId, 
            NULL, NULL, true /* bIgnoreNotFoundMethod */);
        if (MRTE_SUCCEEDED(res))
        {
            jitData.validData |= DR_METHOD_ID;
        }
    }

    if (DR_CONTAINS_METHOD_CODE(dataItems)) 
    {
        jitData.codeInfo.uiActualSize = 1;
        jitData.codeInfo.uiSize = 1;
        jitData.codeInfo.pCodeRegions = new SCodeRegion[1];
        jitData.codeInfo.pCodeRegions[0].pCodeAddr =  pJVMData->u.methodLoad.pCodeAddr;
        jitData.codeInfo.pCodeRegions[0].uiCodeSize = pJVMData->u.methodLoad.uiCodeSize;
        jitData.validData |= DR_METHOD_CODE;
    }

    if (DR_CONTAINS_NATIVE_TO_SRC_LINE_MAP(dataItems))
    {
        jitData.nativeToSrcLineMap = pJVMData->u.methodLoad.nativeToSrcLineMap;
        if (jitData.nativeToSrcLineMap.uiSize != 0)
        {
            jitData.validData |= DR_NATIVE_TO_SRC_LINE_MAP;
        }
    }
    
    if (DR_CONTAINS_NATIVE_TO_MANAGED_LINE_MAP(dataItems))
    {
        jitData.nativeToManagedLineMap = pJVMData->u.methodLoad.nativeToManagedLineMap;
        if (jitData.nativeToManagedLineMap.uiSize != 0)
        {
            jitData.validData |= DR_NATIVE_TO_MANAGED_LINE_MAP;
        }
    }

    if (DR_CONTAINS_MANAGED_TO_SRC_LINE_MAP(dataItems))
    {
        jitData.managedToSrcLineMap = pJVMData->u.methodLoad.managedToSrcLineMap;
        if (jitData.managedToSrcLineMap.uiSize != 0)
        {
            jitData.validData |= DR_MANAGED_TO_SRC_LINE_MAP;
        }
    }

    IJittedMethodLoadedEventObserver *pConcreteObserver = (IJittedMethodLoadedEventObserver*)pObserver;
    pConcreteObserver->HandleEvent(jitData);

    // Clean-up
    delete [] jitData.codeInfo.pCodeRegions;
}

//////////////////////////////////////////////////////////////////////////
// CJittedMethodUnloadedEventDispatcher

void CJittedMethodUnloadedEventDispatcher::Notify(SEmData *pData, 
                                                  IEventObserver *pObserver,
                                                  BitSet dataItems)
{
    SJittedMethodUnloadedEventData jitData;
    jitData.validData = DR_NONE;
    CDataManager *pDataManager = m_pEventManager->GetDM();

    SJVMData *pJVMData = pData->u.pJVMData;

    if (DR_CONTAINS_METHOD_ID(dataItems)) 
    {
        jmethodID jvmMethodId = pJVMData->u.methodLoad.methodId;
        TResult res = pDataManager->GetMpiMethodIdWModuleId(jvmMethodId, &jitData.methodId, 
            NULL, NULL, true /* bIgnoreNotFoundMethod */);
        if (MRTE_SUCCEEDED(res))
        {
            jitData.validData |= DR_METHOD_ID;
        }
    }

    IJittedMethodUnloadedEventObserver *pConcreteObserver = 
        (IJittedMethodUnloadedEventObserver*)pObserver;
    pConcreteObserver->HandleEvent(jitData);
}

//////////////////////////////////////////////////////////////////////////
// CJavaDynamicCodeGeneratedEventDispatcher

void CJavaDynamicCodeGeneratedEventDispatcher::Notify(SEmData *pData, 
                                                      IEventObserver *pObserver,
                                                      BitSet dataItems)
{
    SJavaDynamicCodeEventData dynamicCodeData;
    dynamicCodeData.validData = DR_NONE;

    SJVMData *pJVMData = pData->u.pJVMData;

    if (DR_CONTAINS_JAVA_DYNAMIC_CODE(dataItems)) 
    {
        dynamicCodeData.jvmDynamicCode.pCodeAddr = pJVMData->u.dynamicCode.address;
        dynamicCodeData.jvmDynamicCode.uiCodeSize = pJVMData->u.dynamicCode.length;
        dynamicCodeData.jvmDynamicCode.szCodeName = pJVMData->u.dynamicCode.name;
        dynamicCodeData.validData |= DR_JAVA_DYNAMIC_CODE;
    }

    IJavaDynamicCodeGeneratedEventObserver *pConcreteObserver = 
        (IJavaDynamicCodeGeneratedEventObserver*)pObserver;
    pConcreteObserver->HandleEvent(dynamicCodeData);
}

//////////////////////////////////////////////////////////////////////////
// CEcSetOutputDirEventDispatcher

void CEcSetOutputDirEventDispatcher::Notify(SEmData *pData, 
                                            IEventObserver *pObserver,
                                            BitSet dataItems)
{
    SSetOutputDirEventData outDirData;
    outDirData.validData = DR_NONE;

    if (DR_CONTAINS_ACTIVITY_PATH(dataItems)) 
    {
        outDirData.szActivityPath = pData->u.pECData->szECConfigurationPath;
        outDirData.validData |= DR_ACTIVITY_PATH;
    }

    IEcSetOutputDirEventObserver *pConcreteObserver = 
        (IEcSetOutputDirEventObserver*)pObserver;
    pConcreteObserver->HandleEvent(outDirData);
}

//////////////////////////////////////////////////////////////////////////
// CEcCustomCommandEventDispatcher

void CEcCustomCommandEventDispatcher::Notify(SEmData *pData, 
                                             IEventObserver *pObserver,
                                             BitSet dataItems)
{
    SCustomCommandEventData cmdData;
    cmdData.validData = DR_NONE;

    if (DR_CONTAINS_COMMAND_ID(dataItems)) 
    {
        cmdData.commandId = pData->u.pECData->commandId;
        cmdData.validData |= DR_COMMAND_ID;
    }

    if (DR_CONTAINS_COMMAND_DATA(dataItems)) 
    {
        cmdData.pData = pData->u.pECData->pData;
        cmdData.validData |= DR_COMMAND_DATA;
    }

    IEcCustomCommandEventObserver *pConcreteObserver = 
        (IEcCustomCommandEventObserver*)pObserver;
    pConcreteObserver->HandleEvent(cmdData);
}

//////////////////////////////////////////////////////////////////////////
// CThreadStartEventDispatcher

void CThreadStartEventDispatcher::Notify(SEmData *pData, 
                                         IEventObserver *pObserver,
                                         BitSet dataItems)
{
    SThreadEventData threadData;
    threadData.validData = DR_NONE;
    CDataManager *pDataManager = m_pEventManager->GetDM();

    GetThreadEventData(&threadData, pData->u.pJVMData->u.threadStart.pThreadEnv, 
        pData->u.pJVMData->u.threadStart.threadLocalRef, dataItems, pDataManager);

    IThreadStartEventObserver *pConcreteObserver = (IThreadStartEventObserver*)pObserver;
    pConcreteObserver->HandleEvent(threadData);
}

//////////////////////////////////////////////////////////////////////////
// CThreadEndEventDispatcher

void CThreadEndEventDispatcher::Notify(SEmData *pData, 
                                       IEventObserver *pObserver,
                                       BitSet dataItems)
{
    SThreadEventData threadData;
    threadData.validData = DR_NONE;
    CDataManager *pDataManager = m_pEventManager->GetDM();

    GetThreadEventData(&threadData, pData->u.pJVMData->u.threadEnd.pThreadEnv, 
        pData->u.pJVMData->u.threadEnd.threadLocalRef, dataItems, pDataManager);

    IThreadEndEventObserver *pConcreteObserver = (IThreadEndEventObserver*)pObserver;
    pConcreteObserver->HandleEvent(threadData);
}

//////////////////////////////////////////////////////////////////////////
// CVmInitEventDispatcher

void CVmInitEventDispatcher::Notify(SEmData *pData, 
                                    IEventObserver *pObserver,
                                    BitSet dataItems)
{
    SVmInitEventData vmInitData;
    vmInitData.validData = DR_NONE;

    SJVMData *pJVMEventData = pData->u.pJVMData;
    CDataManager *pDataManager = m_pEventManager->GetDM();
    CThreadInfoManager *pThreadInfoManager = pDataManager->GetThreadInfoManager();

    if (DR_CONTAINS_THREAD_ID(dataItems)) 
    {
        vmInitData.threadId = pThreadInfoManager->GetThreadId(
            pJVMEventData->u.vmEvent.pJniEnv);
        if (0 == vmInitData.threadId)
        {
            pThreadInfoManager->ThreadStart(&vmInitData.threadId, 
                pJVMEventData->u.vmEvent.pJniEnv, 0, false /* bGenerateEvent */,
                NULL, NULL, NULL);

        }
        vmInitData.validData |= DR_THREAD_ID;
    }

    IVmInitEventObserver *pConcreteObserver = (IVmInitEventObserver*)pObserver;
    pConcreteObserver->HandleEvent(vmInitData);
}

//////////////////////////////////////////////////////////////////////////
// CVmShutdownEventDispatcher

void CVmShutdownEventDispatcher::Notify(SEmData *pData, 
                                        IEventObserver *pObserver,
                                        BitSet dataItems)
{
    IVmShutdownEventObserver *pConcreteObserver = (IVmShutdownEventObserver*)pObserver;
    pConcreteObserver->HandleEvent();
}

//////////////////////////////////////////////////////////////////////////
// CObjectAllocEventDispatcher

void CObjectAllocEventDispatcher::Notify(SEmData *pData, 
                                         IEventObserver *pObserver,
                                         BitSet dataItems)
{
    SJvmtiObjectTag *pTagInfo = NULL;
    jobject objJniLocalRef = NULL;
    TId allocMethodId = -1;
    TId allocLocation = -1;
    if (MPI_INTERNAL == pData->dataType)
    {
        // Event triggered by instrumentation
        pTagInfo = pData->u.pMpiData->pObjectTag;
        objJniLocalRef = pData->u.pMpiData->objectJniLocalRef;
        allocMethodId = pData->u.pMpiData->methodId;
        allocLocation = pData->u.pMpiData->allocLocation;
    }
    else
    {
        // Event triggered by the JVMTI VM Object Alloc event
        pTagInfo = pData->u.pJVMData->u.vmObjectAlloc.pTagInfo;
        objJniLocalRef = pData->u.pJVMData->u.vmObjectAlloc.objectJniLocalRef;
    }

    SHeapEventData heapData;
    heapData.validData = DR_OBJECT_ID;
    heapData.objectId = pTagInfo->jobjectInfo.objectId;

    CDataManager *pDataManager = m_pEventManager->GetDM();
    if (DR_CONTAINS_THREAD_ID(dataItems) && pData->pJniEnv != NULL)
    {
        heapData.threadId = GetThreadIdFromTls(pData->pJniEnv, pDataManager, NULL);
        heapData.validData |= DR_THREAD_ID;
    }
    
    if (DR_CONTAINS_OBJECT_INFO(dataItems))
    {

        if(objJniLocalRef != NULL){
            SObjectInfo *pObjectInfo = new SObjectInfo();
            MARTINI_ASSERT("CObjectAllocEventDispatcher", pObjectInfo, "out of memory");
            memset(pObjectInfo, 0, sizeof(SObjectInfo));
            //TODO: optimize: when the event source is JVMTI_EVENT_VM_OBJECT_ALLOC, we already 
            //      know the object's size and it's class, and we don't need to get it again.
            TResult res = m_pEventManager->GetJVMInterface()->GetObjectInfo(pObjectInfo, 
                objJniLocalRef);
            if (MRTE_SUCCEEDED(res))
            {
                // Cache the information in the object's tag
                pTagInfo->jobjectInfo.pObjectInfo = pObjectInfo;

                // Add the requested information to heapData
                heapData.validData |= DR_OBJECT_INFO;
                heapData.pObjectInfo = pObjectInfo;
            }
            else
            {
                heapData.pObjectInfo = NULL;
            }
        }else{
            SObjectInfo *pObjectInfo = pTagInfo->jobjectInfo.pObjectInfo;
            if(pObjectInfo!=NULL && pObjectInfo->classId != 0){
                heapData.pObjectInfo = pObjectInfo;
                heapData.validData |= DR_OBJECT_INFO;
            }else{
                heapData.pObjectInfo = NULL;
            }
        }
    }

    if (DR_CONTAINS_METHOD_ID(dataItems))
    {
        heapData.validData |= DR_METHOD_ID;
        heapData.allocMethodId = allocMethodId;
        // Generate EV_NEW_METHOD if needed
        if (allocMethodId > 0 && allocMethodId !=  (TId (-1)) && !pDataManager->IsNewMethodReported(allocMethodId))
        {
            m_pEventManager->NewMethodEvent(allocMethodId, NULL);
        }

    }
    if (DR_CONTAINS_ALLOC_VM_INSTRUCTION_OFFSET(dataItems))
    {
        heapData.validData |= DR_ALLOC_VM_INSTRUCTION_OFFSET;
        heapData.allocLocation = allocLocation;
    }


    IObjectAllocEventObserver *pConcreteObserver = (IObjectAllocEventObserver*)pObserver;
    pConcreteObserver->HandleEvent(heapData);
}

//////////////////////////////////////////////////////////////////////////
// CObjectFreeEventDispatcher

void CObjectFreeEventDispatcher::Notify(SEmData *pData, 
                                        IEventObserver *pObserver,
                                        BitSet dataItems)
{
    SHeapEventData heapData;
    heapData.validData = DR_NONE;
    CDataManager *pDataManager = m_pEventManager->GetDM();

    if (DR_CONTAINS_OBJECT_ID(dataItems) || DR_CONTAINS_OBJECT_INFO(dataItems) ||
        DR_CONTAINS_OBJECT_AGE(dataItems) )
    {
        CObjectInfoManager *pObjectInfoManager = pDataManager->GetObjectInfoManager();
        SJvmtiObjectTag* pTagInfo = pObjectInfoManager->GetTagInfo(
            pData->u.pJVMData->u.objectFree.tag);
        if (pTagInfo->jobjectInfo.objectId != 0)
        {
            heapData.objectId = pTagInfo->jobjectInfo.objectId;
            heapData.validData |= DR_OBJECT_ID;
        }
        if (pTagInfo->jobjectInfo.pObjectInfo != NULL)
        {
            heapData.pObjectInfo = pTagInfo->jobjectInfo.pObjectInfo;
            heapData.validData |= DR_OBJECT_INFO;
        }
        heapData.objectAge = pObjectInfoManager->GetGcCount() - pTagInfo->uiStartAge;
        heapData.validData |= DR_OBJECT_AGE;

    }

    if (DR_CONTAINS_THREAD_ID(dataItems) && 0 != pData->pJniEnv)
    {
        heapData.threadId = GetThreadIdFromTls(pData->pJniEnv, pDataManager, NULL);
        heapData.validData |= DR_THREAD_ID;
    }
    
    IObjectFreeEventObserver *pConcreteObserver = (IObjectFreeEventObserver*)pObserver;
    pConcreteObserver->HandleEvent(heapData);
}

//////////////////////////////////////////////////////////////////////////
// CGcStartEventDispatcher

void CGcStartEventDispatcher::Notify(SEmData *pData, 
                                     IEventObserver *pObserver,
                                     BitSet dataItems)
{
    IGcStartEventObserver *pConcreteObserver = (IGcStartEventObserver*)pObserver;
    pConcreteObserver->HandleEvent();
}

//////////////////////////////////////////////////////////////////////////
// CGcEndEventDispatcher

void CGcEndEventDispatcher::Notify(SEmData *pData, 
                                   IEventObserver *pObserver,
                                   BitSet dataItems)
{
    IGcEndEventObserver *pConcreteObserver = (IGcEndEventObserver*)pObserver;
    pConcreteObserver->HandleEvent();
}

//////////////////////////////////////////////////////////////////////////
// CEcStartEventDispatcher

void CEcStartEventDispatcher::Notify(SEmData *pData, 
                                     IEventObserver *pObserver,
                                     BitSet dataItems)
{
    IEcStartEventObserver *pConcreteObserver = (IEcStartEventObserver*)pObserver;
    pConcreteObserver->HandleEvent();
}

//////////////////////////////////////////////////////////////////////////
// CEcStopEventDispatcher

void CEcStopEventDispatcher::Notify(SEmData *pData, 
                                    IEventObserver *pObserver,
                                    BitSet dataItems)
{
    IEcStopEventObserver *pConcreteObserver = (IEcStopEventObserver*)pObserver;
    pConcreteObserver->HandleEvent();
}

//////////////////////////////////////////////////////////////////////////
// CVmShutdownEventDispatcher

void CEcAttachEventDispatcher::Notify(SEmData *pData, 
                                      IEventObserver *pObserver,
                                      BitSet dataItems)
{
    IEcAttachEventObserver *pConcreteObserver = (IEcAttachEventObserver*)pObserver;
    pConcreteObserver->HandleEvent();
}

//////////////////////////////////////////////////////////////////////////
// CVmShutdownEventDispatcher

void CEcDetachEventDispatcher::Notify(SEmData *pData, 
                                      IEventObserver *pObserver,
                                      BitSet dataItems)
{
    IEcDetachEventObserver *pConcreteObserver = (IEcDetachEventObserver*)pObserver;
    pConcreteObserver->HandleEvent();
}

//////////////////////////////////////////////////////////////////////////
// CMonitorWaitEventDispatcher

void CMonitorWaitEventDispatcher::Notify(SEmData *pData, 
                                         IEventObserver *pObserver, 
                                         BitSet dataItems)
{
    SJVMData *pJvmData = pData->u.pJVMData;
    SMonitorWaitEventData monWaitData;
    monWaitData.validData = DR_NONE;

    GetMonitorEventData(&monWaitData, pJvmData->u.monitorWait.monitorLocalRef,
        pJvmData->u.monitorWait.threadLocalRef, pData->pJniEnv, dataItems, 
        m_pEventManager->GetDM());

    if (DR_CONTAINS_THREAD_ID(dataItems) &&
        (!(DR_CONTAINS_THREAD_ID(monWaitData.validData))))
    {
        MARTINI_INFORMATIVE2("CMonitorWaitEventDispatcher", 3, false,
            "Cannot obtain thread id. Event will not be dispatched. pJniEnv = %p, thread = %p",
            pData->pJniEnv, pJvmData->u.monitorWait.threadLocalRef);
        return;
    }

    if (DR_CONTAINS_MONITOR_TIMEOUT(dataItems))
    {
        monWaitData.timeoutMillis = pJvmData->u.monitorWait.timeout;
        monWaitData.validData |= DR_MONITOR_TIMEOUT;
    }

    IMonitorWaitEventObserver *pConcreteObserver = (IMonitorWaitEventObserver*)pObserver;
    pConcreteObserver->HandleEvent(monWaitData);
}

//////////////////////////////////////////////////////////////////////////
// CMonitorWaitedEventDispatcher

void CMonitorWaitedEventDispatcher::Notify(SEmData *pData, 
                                           IEventObserver *pObserver, 
                                           BitSet dataItems)
{
    SJVMData *pJvmData = pData->u.pJVMData;
    SMonitorWaitedEventData monWaitedData;
    monWaitedData.validData = DR_NONE;

    GetMonitorEventData(&monWaitedData, pJvmData->u.monitorWaited.monitorLocalRef,
        pJvmData->u.monitorWaited.threadLocalRef, pData->pJniEnv, dataItems, 
        m_pEventManager->GetDM());

    if (DR_CONTAINS_THREAD_ID(dataItems) &&
        (!(DR_CONTAINS_THREAD_ID(monWaitedData.validData))))
    {
        MARTINI_INFORMATIVE2("CMonitorWaitedEventDispatcher", 3, false,
            "Cannot obtain thread id. Event will not be dispatched. pJniEnv = %p, thread = %p",
            pData->pJniEnv, pJvmData->u.monitorWait.threadLocalRef);
        return;
    }

    monWaitedData.isTimedOut  = pJvmData->u.monitorWaited.bTimedOut;
    monWaitedData.validData  |= DR_MONITOR_TIMED_OUT;

    IMonitorWaitedEventObserver *pConcreteObserver = (IMonitorWaitedEventObserver*)pObserver;
    pConcreteObserver->HandleEvent(monWaitedData);
}

//////////////////////////////////////////////////////////////////////////
// CContendedMonitorEnterEventDispatcher

void CContendedMonitorEnterEventDispatcher::Notify(SEmData *pData, 
                                                   IEventObserver *pObserver, 
                                                   BitSet dataItems)
{
    SJVMData *pJvmData = pData->u.pJVMData;
    SContendedMonitorEnterEventData monEnterData;
    monEnterData.validData = DR_NONE;

    GetMonitorEventData(&monEnterData, pJvmData->u.contendedMonitor.monitorLocalRef,
        pJvmData->u.contendedMonitor.threadLocalRef, pData->pJniEnv, dataItems, 
        m_pEventManager->GetDM());

    if (DR_CONTAINS_THREAD_ID(dataItems) &&
        (!(DR_CONTAINS_THREAD_ID(monEnterData.validData))))
    {
        MARTINI_INFORMATIVE2("CMonitorEnterEventDispatcher", 3, false,
            "Cannot obtain thread id. Event will not be dispatched. pJniEnv = %p, thread = %p",
            pData->pJniEnv, pJvmData->u.monitorWait.threadLocalRef);
        return;
    }

    if (DR_CONTAINS_MONITOR_OWNER_THREAD_ID(dataItems))
    {
        TResult res = m_pEventManager->GetJVMInterface()->GetMonitorUsage(
            &monEnterData.ownerThreadId, NULL, NULL, DR_MONITOR_OWNER_THREAD_ID,
            pJvmData->u.contendedMonitor.monitorLocalRef);
        if (MRTE_SUCCEEDED(res))
        {
            monEnterData.validData |= DR_MONITOR_OWNER_THREAD_ID;
        }
    }

    IContendedMonitorEnterEventObserver *pConcreteObserver = 
        (IContendedMonitorEnterEventObserver*)pObserver;
    pConcreteObserver->HandleEvent(monEnterData);
}

//////////////////////////////////////////////////////////////////////////
// CContendedMonitorEnteredEventDispatcher

void CContendedMonitorEnteredEventDispatcher::Notify(SEmData *pData, 
                                                     IEventObserver *pObserver, 
                                                     BitSet dataItems)
{
    SJVMData *pJvmData = pData->u.pJVMData;
    SContendedMonitorEnteredEventData monEnteredData;
    monEnteredData.validData = DR_NONE;

    GetMonitorEventData(&monEnteredData, pJvmData->u.contendedMonitor.monitorLocalRef,
        pJvmData->u.contendedMonitor.threadLocalRef, pData->pJniEnv, dataItems, 
        m_pEventManager->GetDM());

    if (DR_CONTAINS_THREAD_ID(dataItems) &&
        (!(DR_CONTAINS_THREAD_ID(monEnteredData.validData))))
    {
        MARTINI_INFORMATIVE2("CMonitorEnteredEventDispatcher", 3, false,
            "Cannot obtain thread id. Event will not be dispatched. pJniEnv = %p, thread = %p",
            pData->pJniEnv, pJvmData->u.monitorWait.threadLocalRef);
        return;
    }

    IContendedMonitorEnteredEventObserver *pConcreteObserver = 
        (IContendedMonitorEnteredEventObserver*)pObserver;
    pConcreteObserver->HandleEvent(monEnteredData);
}



//////////////////////////////////////////////////////////////////////////
// CThreadInteractionEventDispatcher

void CThreadInteractionEventDispatcher::Notify(SEmData *pData, 
                                               IEventObserver *pObserver,
                                               BitSet dataItems)
{

    // Event triggered by instrumentation
    MARTINI_ASSERT("EventDispatcher", MPI_INTERNAL == pData->dataType, "");

    SThreadInteractionEventData callThreadInteraction;
    callThreadInteraction.validData = DR_NONE;
    CDataManager *pDataManager = m_pEventManager->GetDM();

    if (DR_CONTAINS_OBJECT_ID(dataItems))
    {
        callThreadInteraction.objectId   = pData->u.pMpiData->pObjectTag->jobjectInfo.objectId;
        callThreadInteraction.validData |= DR_OBJECT_ID;
    }
    if (DR_CONTAINS_THREAD_ID(dataItems)) 
    {
        callThreadInteraction.threadId = GetThreadIdFromTls(pData->pJniEnv, pDataManager, NULL);
        callThreadInteraction.validData |= DR_THREAD_ID;
    }
    if (DR_CONTAINS_THREAD_INTERACTION_TYPE(dataItems))
    {
        callThreadInteraction.interactionType = (EThreadInteractionType)pData->u.pMpiData->interactionType;
        callThreadInteraction.validData |= DR_THREAD_INTERACTION_TYPE;
    }


    IThreadInteractionEventObserver *pConcreteObserver = (IThreadInteractionEventObserver*)pObserver;
    pConcreteObserver->HandleEvent(callThreadInteraction);
}


//////////////////////////////////////////////////////////////////////////
// CJavaClassFileLoadHookEventDispatcher

TResult CJavaClassFileLoadHookEventDispatcher::Register(CObserverInfo *observerInfo)
{
/*
    //TODO: generalize check to use the capabilities model, e.g.:

    if (!m_pJavaInterface->GetCapabilities()->Enabled(
        CT_CAN_REGISTER_MULTIPLE_INSTRUMENTATION_EVENT_CLIENTS))
    {
        return MRTE_ERROR_NOT_SUPPORTED;
    }
*/

    // Current slice supports only a single instrumentation client (external or internal)
    if (m_observers.Count() >= 1)
    {
        MARTINI_INFORMATIVE("CJavaClassFileLoadHookEventDispatcher", 0, false, 
            "Multiple ClassFileLoadHook (instrumentation) event clients are not currently "
            "supported");
        return MRTE_ERROR_NOT_SUPPORTED;
    }

    return CEventDispatcher::Register(observerInfo);
}

void CJavaClassFileLoadHookEventDispatcher::Notify(SEmData *pData, 
                                                   IEventObserver *pObserver, 
                                                   BitSet dataItems)
{
    SJVMData *pJvmData = pData->u.pJVMData;
    SClassFileLoadHookEventData classLoadHookData;
    classLoadHookData.className = pJvmData->u.classLoadHook.szClassName;
    classLoadHookData.classData = pJvmData->u.classLoadHook.pucClassData;
    classLoadHookData.classDataLength = pJvmData->u.classLoadHook.classDataLen;

    IJavaClassFileLoadHookEventObserver *pConcreteObserver = 
        (IJavaClassFileLoadHookEventObserver*)pObserver;

    pConcreteObserver->HandleEvent(classLoadHookData, pJvmData->u.classLoadHook.pfnAllocate,
        pJvmData->u.classLoadHook.ppucNewClassData, 
        (S32*)pJvmData->u.classLoadHook.pNewClassDataLen);
}
