/*****************************************************************************
 * Copyright (c) 1997, 2009 Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/

#ifndef _MRTE_EVENT_DISPATCHER_H
#define _MRTE_EVENT_DISPATCHER_H

#include "MpiAPI.h"
#include "MRTEInfrastructureDefinitions.h"
#include "IJVM.h"
#include "MList.h"

namespace Martini { namespace JPI
{
    class CEventManager;

    struct SMpiInternalData
    {
        MPI::TId methodId;                       //!< Method id (EV_NEW_METHOD, EV_OBJECT_ALLOC)
        jobject objectJniLocalRef;               //!< Local reference of the object (EV_OBJECT_ALLOC)
                                                 //!  Valid only in the context of the ObjectAlloc JNI
                                                 //!  Callback (!!!)
        SJvmtiObjectTag *pObjectTag;             //!< JVMTI object tag (EV_OBJECT_ALLOC and EV_THREAD_INTERACTION)
        U64 allocLocation;                       //!< Object allocation byte-code index (EV_OBJECT_ALLOC)

        int interactionType;                     //!< type of interaction between threads (EV_THREAD_INTERACTION)
        MPI::BitSet validData;
    };
    
    struct SECEventData
    {
        char *szECConfigurationPath;    //!< EC Configuration event data (EV_EC_SET_OUTPUT_DIR)
        unsigned int commandId;         //!< Command id (EV_EC_CUSTOM_COMMAND)
        void *pData;                    //!< Command data (EV_EC_CUSTOM_COMMAND)
    };

    struct SEmData
    {
        union
        {
            SJVMData         *pJVMData;              // VM-generated event data
            SMpiInternalData *pMpiData;              // MPI-generated event data
            SECEventData     *pECData;               // External Control generated event data
        } u;
        Infrastructure::EEventImplementationType dataType;
        const JNIEnv *pJniEnv;  // identifies the thread that sent the event. 
                                // 0 if not applicable for the event
    };

    /**
     * @brief Defines information for event observers
     */
    class CObserverInfo             
    {
    public:
        CObserverInfo(MPI::IEventObserver *observer, 
                      MPI::TId theClientId,
                      Infrastructure::EEventPriority prior,
                      bool isInternal,
                      MPI::BitSet dataItems, 
                      bool isEnabled) : pObserver(observer), clientId(theClientId),
                                          priority(prior), bIsInternal(isInternal),
                                          dataRequestType(dataItems), bIsEnabled(isEnabled)
        {
        }

        MPI::IEventObserver *pObserver;          //!< The observer
        MPI::TId clientId;                       //!< Client registering the observer
        Infrastructure::EEventPriority priority; //!< Priority
        bool bIsInternal;                        //!< Whether this is an internal event 
                                                 //!  observer
        MPI::BitSet dataRequestType;             //!< Data items requested for the event
        bool bIsEnabled;                         //!< Whether the event is enabled for
                                                 //!  the client
    };

    typedef RTUtil::MList<CObserverInfo*> TObserverList;

    /**
     * @brief Event dispatcher
     *
     * Implements registration and generic notification for all event observers
     */
    class CEventDispatcher
    {
    public:
        virtual ~CEventDispatcher();

        /**
         * @brief Registers an event observer
         *
         * Registers an event observer with the event dispatcher.
         * The observer is added to the observer's list according to its priority.
         * An MPI Client cannot register to the same event more than once
         *
         * @remark - This method implements a default registration process with general
         *           validity checks. Concrete dispatchers (inherited from CEventDispatcher)
         *           may override this method to provide event-specific registration process..
         *
         * @param[in] observerInfo      Observer information
         *
         * @retval MRTE_RESULT_OK       Success
         * @retval MRTE_ERROR_FAIL      The MPI Client that defined this observer has already
         *                              registered to this event
         */
        virtual TResult Register(CObserverInfo *observerInfo);

        /**
         * @brief Remove all observers registered for the event
         */
        void UnregisterAll();

        /**
         * @brief Dispatches the event to its observers.
         *
         * Dispatches the event to all registered observer. The function internally invokes
         * the Notify template method.
         *
         * @param[in] pData     Event data
         */
        void NotifyObservers(SEmData *pData);

        /**
         * @brief Returns the list of event observers
         *
         * @return the list of observers registered for the event
         */
        TObserverList& GetObservers() { return m_observers; }
    
    protected:

        /**
         * @brief Template method for notifying a concrete observer implementation
         * 
         * Actually dispatches an event to concrete event observers.
         * This function is implemented by concrete event dispatchers. Each concrete
         * dispatcher is responsible for extracting from pData the data relevant for the event
         * observer it notifies
         *
         * @param[in] pData         Event data
         * @param[in] pObserver     The observer to notify
         * @param[in] dataItems     Event data items needed by the observer
         *
         * @remark - The dataItems argument is included in order to save the implementor
         *           the need to do a virtual function call on pObserver to obtain this
         *           information. This improves performance.
         */
        virtual void Notify(SEmData *pData, 
                            MPI::IEventObserver *pObserver,
                            MPI::BitSet dataItems) = 0;

        /**
         * @brief Event observers
         */
        TObserverList m_observers;
    };

    /**
     * @brief Dispatcher for internal event observers
     */
    class CInternalEventDispatcher : public CEventDispatcher
    {
    public:

        virtual void Notify(SEmData *pData, 
                            MPI::IEventObserver *pObserver,
                            MPI::BitSet dataItems);
    };

    /**
     * @brief Dispatcher for the New Method event
     */
    class CNewMethodEventDispatcher : public CEventDispatcher
    {
    public:
        CNewMethodEventDispatcher(CEventManager *pEventManager) 
            : m_pEventManager(pEventManager) {}

        virtual void Notify(SEmData *pData, 
                            MPI::IEventObserver *pObserver,
                            MPI::BitSet dataItems);

    private:
        CEventManager *m_pEventManager;
    };

    /**
     * @brief Method Enter event dispatcher
     *
     * Dispatches the Method Enter event.
     */
    class CMethodEnterEventDispatcher : public CEventDispatcher
    {
    public:

        CMethodEnterEventDispatcher(CEventManager *pEventManager) 
            : m_pEventManager(pEventManager) {}
        /**
         * @brief Dispatches the event to its observers.
         *
         * Called only when JVMTI is used.
         */
        void NotifyMethodEnterObservers(MPI::SMethodEnterEventData data);
        
        /**
         * @brief Dispatches the event.
         *
         * Invoked only when JVMPI is used. In JVMTI, the NotifyMethodEnterObservers
         * method is invoked directly by the Method Enter recorder callback.
         */
        virtual void Notify(SEmData *pData, 
                            MPI::IEventObserver *pObserver,
                            MPI::BitSet dataItems);

    private:
        CEventManager *m_pEventManager;
    };

    /**
     * @brief Method Leave event dispatcher
     */
    class CMethodLeaveEventDispatcher : public CEventDispatcher
    {
    public:

        CMethodLeaveEventDispatcher(CEventManager *pEventManager) 
            : m_pEventManager(pEventManager) {}

        /**
         * @brief @brief Dispatches the event to its observers.
         *
         * Called only when JVMTI is used.
         */
        void NotifyMethodLeaveObservers(MPI::SMethodLeaveEventData data);
        
        /**
         * @brief Dispatches the event.
         *
         * Invoked only when JVMPI is used. In JVMTI, the NotifyMethodLeaveObservers
         * method is invoked directly by the Method Leave recorder callback.
         */
        virtual void Notify(SEmData *pData, 
                            MPI::IEventObserver *pObserver,
                            MPI::BitSet dataItems);

    private:
        CEventManager *m_pEventManager;

    };

    /**
     * @brief Dispatcher for the JITted Method Loaded event
     */
    class CJittedMethodLoadedEventDispatcher : public CEventDispatcher
    {
    public:
        CJittedMethodLoadedEventDispatcher(CEventManager *pEventManager) 
            : m_pEventManager(pEventManager) {}

        virtual void Notify(SEmData *pData, 
                            MPI::IEventObserver *pObserver,
                            MPI::BitSet dataItems);

    private:
        CEventManager *m_pEventManager;
    };

    /**
     * @brief Dispatcher for the JITted Method Unloaded event
     */
    class CJittedMethodUnloadedEventDispatcher : public CEventDispatcher
    {
    public:
        CJittedMethodUnloadedEventDispatcher(CEventManager *pEventManager) 
            : m_pEventManager(pEventManager) {}

        virtual void Notify(SEmData *pData, 
                            MPI::IEventObserver *pObserver,
                            MPI::BitSet dataItems);

    private:
        CEventManager *m_pEventManager;
    };

    /**
     * @brief Dispatcher for the Java Dynamic Code Generated event
     */
    class CJavaDynamicCodeGeneratedEventDispatcher : public CEventDispatcher
    {
    public:

        virtual void Notify(SEmData *pData, 
                            MPI::IEventObserver *pObserver,
                            MPI::BitSet dataItems);

    };

    /**
     * @brief Dispatcher for the Set Output Directory event
     */
    class CEcSetOutputDirEventDispatcher : public CEventDispatcher
    {
    public:

        virtual void Notify(SEmData *pData, 
                            MPI::IEventObserver *pObserver,
                            MPI::BitSet dataItems);

    };

    /**
     * @brief Dispatcher for the Custom Command event
     */
    class CEcCustomCommandEventDispatcher : public CEventDispatcher
    {
    public:

        virtual void Notify(SEmData *pData, 
                            MPI::IEventObserver *pObserver,
                            MPI::BitSet dataItems);

    };

    /**
     * @brief Dispatcher for the Thread Start event
     */
    class CThreadStartEventDispatcher : public CEventDispatcher
    {
    public:
        CThreadStartEventDispatcher(CEventManager *pEventManager) 
            : m_pEventManager(pEventManager) {}

        virtual void Notify(SEmData *pData, 
                            MPI::IEventObserver *pObserver,
                            MPI::BitSet dataItems);

    private:
        CEventManager *m_pEventManager;
    };

    /**
     * @brief Dispatcher for the Thread End event
     */
    class CThreadEndEventDispatcher : public CEventDispatcher
    {
    public:
        CThreadEndEventDispatcher(CEventManager *pEventManager) 
            : m_pEventManager(pEventManager) {}

        virtual void Notify(SEmData *pData, 
                            MPI::IEventObserver *pObserver,
                            MPI::BitSet dataItems);

    private:
        CEventManager *m_pEventManager;
    };

    /**
     * @brief Dispatcher for the VM Init event
     */
    class CVmInitEventDispatcher : public CEventDispatcher
    {
    public:
        CVmInitEventDispatcher(CEventManager *pEventManager) 
            : m_pEventManager(pEventManager) {}

        virtual void Notify(SEmData *pData, 
                            MPI::IEventObserver *pObserver,
                            MPI::BitSet dataItems);

    private:
        CEventManager *m_pEventManager;
    };

    /**
     * @brief Dispatcher for the VM Shutdown event
     */
    class CVmShutdownEventDispatcher : public CEventDispatcher
    {
    public:

        virtual void Notify(SEmData *pData, 
                            MPI::IEventObserver *pObserver,
                            MPI::BitSet dataItems);

    };

    /**
     * @brief Dispatcher for the Monitor Wait event
     */
    class CMonitorWaitEventDispatcher : public CEventDispatcher
    {
    public:
        CMonitorWaitEventDispatcher(CEventManager *pEventManager) 
            : m_pEventManager(pEventManager) {}

        virtual void Notify(SEmData *pData, 
                            MPI::IEventObserver *pObserver,
                            MPI::BitSet dataItems);

    private:
        CEventManager *m_pEventManager;
    };

    /**
     * @brief Dispatcher for the Monitor Waited event
     */
    class CMonitorWaitedEventDispatcher : public CEventDispatcher
    {
    public:
        CMonitorWaitedEventDispatcher(CEventManager *pEventManager) 
            : m_pEventManager(pEventManager) {}

        virtual void Notify(SEmData *pData, 
                            MPI::IEventObserver *pObserver,
                            MPI::BitSet dataItems);

    private:
        CEventManager *m_pEventManager;
    };

    /**
     * @brief Dispatcher for the Contended Monitor Enter event
     */
    class CContendedMonitorEnterEventDispatcher : public CEventDispatcher
    {
    public:
        CContendedMonitorEnterEventDispatcher(CEventManager *pEventManager) 
            : m_pEventManager(pEventManager) {}

        virtual void Notify(SEmData *pData, 
                            MPI::IEventObserver *pObserver,
                            MPI::BitSet dataItems);

    private:
        CEventManager *m_pEventManager;
    };

    /**
     * @brief Dispatcher for the Contended Monitor Entered event
     */
    class CContendedMonitorEnteredEventDispatcher : public CEventDispatcher
    {
    public:
        CContendedMonitorEnteredEventDispatcher(CEventManager *pEventManager) 
            : m_pEventManager(pEventManager) {}

        virtual void Notify(SEmData *pData, 
                            MPI::IEventObserver *pObserver,
                            MPI::BitSet dataItems);

    private:
        CEventManager *m_pEventManager;
    };


    /**
     * @brief Dispatcher for the Thread Interaction event
     */
    class CThreadInteractionEventDispatcher : public CEventDispatcher
    {
    public:
        CThreadInteractionEventDispatcher(CEventManager *pEventManager) 
            : m_pEventManager(pEventManager) {}

        virtual void Notify(SEmData *pData, 
                            MPI::IEventObserver *pObserver,
                            MPI::BitSet dataItems);        

    private:
        CEventManager *m_pEventManager;
    };


    /**
     * @brief Dispatcher for the Object Alloc event
     */
    class CObjectAllocEventDispatcher : public CEventDispatcher
    {
    public:
        CObjectAllocEventDispatcher(CEventManager *pEventManager) 
            : m_pEventManager(pEventManager) {}

        virtual void Notify(SEmData *pData, 
                            MPI::IEventObserver *pObserver,
                            MPI::BitSet dataItems);

    private:
        CEventManager *m_pEventManager;
    };

    /**
     * @brief Dispatcher for the Object Free event
     */
    class CObjectFreeEventDispatcher : public CEventDispatcher
    {
    public:
        CObjectFreeEventDispatcher(CEventManager *pEventManager) 
            : m_pEventManager(pEventManager) {}

        virtual void Notify(SEmData *pData, 
                            MPI::IEventObserver *pObserver,
                            MPI::BitSet dataItems);

    private:
        CEventManager *m_pEventManager;
    };

    /**
     * @brief Dispatcher for the GC Start event
     */
    class CGcStartEventDispatcher : public CEventDispatcher
    {
    public:

        virtual void Notify(SEmData *pData, 
                            MPI::IEventObserver *pObserver,
                            MPI::BitSet dataItems);

    };

    /**
     * @brief Dispatcher for the GC End event
     */
    class CGcEndEventDispatcher : public CEventDispatcher
    {
    public:

        virtual void Notify(SEmData *pData, 
                            MPI::IEventObserver *pObserver,
                            MPI::BitSet dataItems);
    };

    /**
     * @brief Dispatcher for the Java Class File Load Hook event
     */
    class CJavaClassFileLoadHookEventDispatcher : public CEventDispatcher
    {
    public:

        virtual void Notify(SEmData *pData, 
                            MPI::IEventObserver *pObserver,
                            MPI::BitSet dataItems);

        virtual TResult Register(CObserverInfo *observerInfo);
    };

    /**
     * @brief Dispatcher for the EC Start event
     */
    class CEcStartEventDispatcher : public CEventDispatcher
    {
    public:

        virtual void Notify(SEmData *pData, 
                            MPI::IEventObserver *pObserver,
                            MPI::BitSet dataItems);

    };

    /**
     * @brief Dispatcher for the EC Stop event
     */
    class CEcStopEventDispatcher : public CEventDispatcher
    {
    public:

        virtual void Notify(SEmData *pData, 
                            MPI::IEventObserver *pObserver,
                            MPI::BitSet dataItems);

    };

    /**
     * @brief Dispatcher for the EC Attach event
     */
    class CEcAttachEventDispatcher : public CEventDispatcher
    {
    public:

        virtual void Notify(SEmData *pData, 
                            MPI::IEventObserver *pObserver,
                            MPI::BitSet dataItems);

    };

    /**
     * @brief Dispatcher for the EC Detach event
     */
    class CEcDetachEventDispatcher : public CEventDispatcher
    {
    public:

        virtual void Notify(SEmData *pData, 
                            MPI::IEventObserver *pObserver,
                            MPI::BitSet dataItems);

    };

}}

#endif
