/*****************************************************************************
 * Copyright (c) 1997, 2010 Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/
#ifdef EM64T_ARCH
#pragma runtime_checks( "", off )
#endif

#include <memory.h>
#include "EventManager.h"
#include "JPI.H"
#include "ECAgent.h"
#include "LogAssert.h"
#include "ThreadInfoManager.h"
#include "MPIUtils.h"
#include "JpiGlobals.h"

using namespace Martini::MPI;
using namespace Martini::JPI;
using namespace Martini::Infrastructure;
using namespace Martini::CGAdaptor;
using namespace Martini::OSA;

//////////////////////////////////////////////////////////////////////////
// CEventManager implementation


// the instance of the Event Manager. Needed for NotifyStdPIEvent().
static CEventManager *s_pEM = NULL;

CEventManager *CEventManager::GetInstance()
{
    return s_pEM;
}


TResult CEventManager::RegisterEvent(TId clientId, IEventObserver& observer)
{
    if (m_bIsInitialiazationCompleted)
    {
        return MRTE_ERROR_PHASE_FAILURE;
    }
    
    TEventType mpiEvent = observer.Type();
    unsigned int dataRequestTypes = observer.EventDataTypes();
    TResult iRetVal = m_pParamChecker->CheckClientAndEventAndData(clientId, mpiEvent, 
        dataRequestTypes);
    if (MRTE_FAILED(iRetVal))
    {
        MARTINI_INFORMATIVE1("EventManager", 0, false, "failed to register '%s' event. "
            "Not all requested data items are supported", 
            m_vEventInfo[mpiEvent].pDescriptor->Name());
        return MRTE_ERROR_NOT_SUPPORTED;
    }

    if (IsCallGraphEvent(mpiEvent) && m_vEventInfo[mpiEvent].uiNumClients > 0)
    {
        if (!m_pJavaInterface->GetCapabilities()->Enabled(
            CT_CAN_REGISTER_MULTIPLE_CALLGRAPH_EVENT_CLIENTS))
        {
            return MRTE_ERROR_NOT_SUPPORTED;
        }
    }

    EEventGroup eventGroup = GetEventGroup(observer.Type());
    bool bEventGroupStatus = GetEventGroupStatusForClient(clientId, eventGroup);

    // Update event information
    iRetVal = InformInternalClients(clientId, observer);
    if (MRTE_SUCCEEDED(iRetVal))
    {
        iRetVal = UpdateEventInformation(clientId, observer, EVENT_PRIORITY_NORMAL, 
            false /* bIsInternal */, bEventGroupStatus /* bIsEnabled */);
        if (MRTE_FAILED(iRetVal))
        {
            return iRetVal;
        }
    }

    // Check whether this event can be implemented with JVMPI/JVMTI or whether 
    // instrumentation is needed
    unsigned int eventImplementationType = GetEventImplementationType(mpiEvent); //TODO: encapsulate in event descriptor

    // If another client already registered, no need to do it again
    // the check is for 1 since the addition to the list is prior to registration
    if (m_vEventInfo[mpiEvent].uiNumClients == 1 
        || eventImplementationType == MPI_EXTERNAL_CONTROL)
    {
        switch (eventImplementationType)
        {
            case MPI_VM:
                iRetVal = m_pJavaInterface->RegisterEvent(mpiEvent); 
                break;
            case MPI_EXTERNAL_CONTROL:
                iRetVal = m_pECAgent->RegisterEvent(mpiEvent);
                break;
            case MPI_INTERNAL:
                // Nothing to do
                break;

            case MPI_INSTRUMENTATION:
                iRetVal = m_pJavaInstrumentorManager->RegisterEvent(observer, 
                    false /* bIsTentative */, true /* bIsEnabled */);
                break;
            case (MPI_TENTATIVE_INSTRUMENTATION | MPI_VM):
                // We might be able to supply instrumentation events, depending if 
                // class_load_hook event arrives at an early stage (before JVM completed
                // its initialization). Register via VM and via instrumentation engine.
                // One of them will be disabled when the first event arrives.
                iRetVal = m_pJavaInstrumentorManager->RegisterEvent(observer, 
                    true /* IsTentative */,  bEventGroupStatus /* bIsEnabled */);
                iRetVal = m_pJavaInterface->RegisterEvent(mpiEvent); 
                break;
            default:
                return MRTE_ERROR_NOT_SUPPORTED;
        }
    }

    if (MRTE_FAILED(iRetVal))
    {
        RemoveEventInformation(observer);
    }

    return iRetVal;
}

TResult CEventManager::SetEventGroupFilter(TId clientID, 
                                           EEventGroup group,
                                           IEventFilter &filter)
{
    // Check that the filter matches the group
    if (!DoesGroupSupportFilter(group, filter))
    {
        return MRTE_ERROR_FAIL;
    }

    // Check that no other filter is applied to the group (by any client)
    // NOTE: this limitation may be lifted in future versions
    IEventFilter *pExistingFilter = GetEventGroupFilter(group);
    if (NULL != pExistingFilter)
    {
        return MRTE_ERROR_CONFLICT;
    }

    m_vEventGroupInfo[group].pFilter = &filter;
    TResult res = m_pJavaInstrumentorManager->SetEventGroupFilter(group, filter);
    return res;
}

TResult CEventManager::EnableEventGroup(TId clientId, 
                                        EEventGroup eventGroup)
{
    return SetEventGroupEnabled(clientId, eventGroup, true);
}

TResult CEventManager::DisableEventGroup(TId clientId, 
                                         EEventGroup eventGroup)
{
    return SetEventGroupEnabled(clientId, eventGroup, false);
}

// TODO: make this more general. Maybe let internal clients notify the event manager, during
// initialization phase, on which events they want to be notified when an external client 
// registers. Meanwhile this is hard coded :(
TResult CEventManager::InformInternalClients(TId clientId, IEventObserver& event)
{
    // in order to support some events, internal clients must be notified too. for example:
    // data manager must know if anyone registered to EVENT_OBJECT_ALLOC and FREE
    unsigned int dataRequestType = event.EventDataTypes();
/*
    TResult res;
*/
    switch (event.Type())
    {
    case EV_JITTED_METHOD_LOADED:
        m_pDataManager->NotifyOnProfilerEventRegistration(clientId, event);
        break;

    case EV_METHOD_ENTER:
        if (m_methodEnterDataRequestType != DR_NONE &&
            m_methodEnterDataRequestType != dataRequestType)
        {
            return MRTE_ERROR_CONFLICT;
        }
        m_methodEnterDataRequestType = dataRequestType;
        m_pDataManager->NotifyOnProfilerEventRegistration(clientId, event);
        break;
    
    case EV_METHOD_LEAVE:
        if (m_methodLeaveDataRequestType != DR_NONE &&
            m_methodLeaveDataRequestType != dataRequestType)
        {
            return MRTE_ERROR_CONFLICT;
        }
        m_methodLeaveDataRequestType = dataRequestType;
        m_pDataManager->NotifyOnProfilerEventRegistration(clientId, event);
        break;
    
    case EV_NEW_METHOD:
        if (m_newMethodDataRequestType != DR_NONE && 
            m_newMethodDataRequestType != dataRequestType)
        {
            return MRTE_ERROR_CONFLICT;
        }
        m_newMethodDataRequestType = dataRequestType;
        break;
    case EV_THREAD_START:
    case EV_THREAD_END:
    case EV_VM_INIT:
        m_pDataManager->NotifyOnProfilerEventRegistration(clientId, event);
        break;
    case EV_OBJECT_ALLOC:
        m_pJavaInterface->GetCapabilities()->Enable(CT_CAN_GENERATE_OBJECT_ALLOC_EVENTS);
        // Need to register for the JVMTI VM Object Alloc event

        //TODO: code currently disabled because the way we do BCI (instrumenting
        //      java.lang.Object.<init>) seem to catch all allocation.
        //      Need to enable this code once we change the BCI to something more
        //      optimal.
/*
        res = m_pJavaInterface->RegisterEvent(EM_EVENT_INTERNAL_VM_OBJECT_ALLOC);
        if (MRTE_FAILED(res))
        {
            MARTINI_INFORMATIVE("CEventManager", 0, false, "JVM does not support the "
                "JVMTI_EVENT_VM_OBJECT_ALLOC event. Object allocation information may be "
                "incomplete");
        }
*/
        break;
    case EV_THREAD_INTERACTION:
        if (m_threadInteractionDataRequestType != DR_NONE && 
            m_threadInteractionDataRequestType != dataRequestType)
        {
            return MRTE_ERROR_CONFLICT;
        }
        m_threadInteractionDataRequestType = dataRequestType;
        break;
    default:
        ;
    }
    return MRTE_RESULT_OK;
}


unsigned int CEventManager::GetEventImplementationType(TEventType mpiEvent)
{
    unsigned int implementor = m_vEventInfo[mpiEvent].implementor;
    return implementor;
}

TResult CEventManager::RegisterEventInternal(TId moduleId, IInternalEventObserver& jpiEvent)
{
    TEventType mpiEvent = jpiEvent.Type();

    TResult iRetVal = m_pParamChecker->CheckClientAndEvent(moduleId, mpiEvent);
    if (MRTE_FAILED(iRetVal))
    {
        return iRetVal;
    }
    
    if (m_vEventInfo[mpiEvent].implementor != MPI_INSTRUMENTATION) //TODO: use event descriptor to check this
    {
        iRetVal = m_pJavaInterface->RegisterEvent(mpiEvent);
    }

    if (MRTE_SUCCEEDED(iRetVal))
    {
        iRetVal = UpdateEventInformation(moduleId, jpiEvent, jpiEvent.GetPriority(), 
            true /* bIsInternal */, true /* bIsEnabled */);
    }
    return iRetVal;
}

//
// Default Constructor. Disabled (declared private). Use CEventManager(TId) to construct
// 
CEventManager::CEventManager(void)
{
}

//
// Constructor. Called by the kernel.
// 
// Parameters:
//      moduleId - module id
//
CEventManager::CEventManager(TId moduleId)
{
    m_pVMShutdownEvent = NULL;
    m_csEnableDisableEvent = NULL;
    s_pEM = this;
    m_ModuleId = moduleId;

    // Init the event management array
    int i;
    for (i = 0; i < EM_MAX_EVENTS_TOTAL; ++i)
    {
        m_vEventInfo[i].bIsEnabledByClients = false;
        m_vEventInfo[i].uiNumClients = 0;
        m_vEventInfo[i].implementor = MPI_IMPLEMENTOR_NONE;
        m_vEventInfo[i].pDispatcher = CreateEventDispatcher(i);
        m_vEventInfo[i].pDescriptor = NULL;
    }
    InitEventDescriptors();

    // Init the event group management array
    for (i = 0; i < EG_LAST; ++i)
    {
        m_vEventGroupInfo[i].pFilter = NULL;
    }

    m_pJavaInterface = NULL;
    m_pDataManager = NULL;
    m_pParamChecker = NULL;
    m_pECAgent = NULL;

    m_newMethodDataRequestType = DR_NONE;
    m_methodEnterDataRequestType = DR_NONE;
    m_methodLeaveDataRequestType = DR_NONE;
    m_threadInteractionDataRequestType = DR_NONE;

    m_bIsInitialiazationCompleted = false;
    m_bVMShutdownReceived = false;

    SetEventImplementor(EV_NEW_METHOD, MPI_INTERNAL);
}

//
// Destructor
//
CEventManager::~CEventManager(void)
{
    // Deallocate event information
    for (int i = 0; i < EM_MAX_EVENTS_TOTAL; ++i)
    {
        if (m_vEventInfo[i].pDispatcher)
        {
		    delete m_vEventInfo[i].pDispatcher;
        }
    }
}

//
// Creates a concrete dispatcher for the event
//
// Note: this function can be generalized in the future to a DispatcherFactory class
// 
CEventDispatcher* CEventManager::CreateEventDispatcher(TEventType mpiEvent)
{
    CEventDispatcher *pDispatcher = NULL;
    switch (mpiEvent)
    {
    // MPI events
    case EV_NEW_METHOD:
        pDispatcher = new CNewMethodEventDispatcher(this);
        break;
    case EV_METHOD_ENTER:
        pDispatcher = new CMethodEnterEventDispatcher(this);
        break;
    case EV_METHOD_LEAVE:
        pDispatcher = new CMethodLeaveEventDispatcher(this);
        break;
    case EV_JITTED_METHOD_LOADED:
        pDispatcher = new CJittedMethodLoadedEventDispatcher(this);
        break;
    case EV_JITTED_METHOD_UNLOADED: 
        pDispatcher = new CJittedMethodUnloadedEventDispatcher(this);
        break;
    case EV_JAVA_DYNAMIC_CODE_GENERATED:
        pDispatcher = new CJavaDynamicCodeGeneratedEventDispatcher();
        break;
    case EV_THREAD_START:
        pDispatcher = new CThreadStartEventDispatcher(this);
        break;
    case EV_THREAD_END:
        pDispatcher = new CThreadEndEventDispatcher(this);
        break;
    case EV_VM_INIT:
        pDispatcher = new CVmInitEventDispatcher(this);
        break;
    case EV_VM_SHUTDOWN:
        pDispatcher = new CVmShutdownEventDispatcher();
        break;
    case EV_OBJECT_ALLOC:
        pDispatcher = new CObjectAllocEventDispatcher(this);
        break;
    case EV_OBJECT_FREE:
        pDispatcher = new CObjectFreeEventDispatcher(this);
        break;
    case EV_GC_START:
        pDispatcher = new CGcStartEventDispatcher();
        break;
    case EV_GC_END:
        pDispatcher = new CGcEndEventDispatcher();
        break;
    case EV_MONITOR_WAIT:
        pDispatcher = new CMonitorWaitEventDispatcher(this);
        break;
    case EV_MONITOR_WAITED:
        pDispatcher = new CMonitorWaitedEventDispatcher(this);
        break;
    case EV_CONTENDED_MONITOR_ENTER:
        pDispatcher = new CContendedMonitorEnterEventDispatcher(this);
        break;
    case EV_CONTENDED_MONITOR_ENTERED:
        pDispatcher = new CContendedMonitorEnteredEventDispatcher(this);
        break;
    case EV_THREAD_INTERACTION:
        pDispatcher = new CThreadInteractionEventDispatcher(this);
        break;
    case EV_JAVA_CLASS_FILE_LOAD_HOOK:
        pDispatcher = new CJavaClassFileLoadHookEventDispatcher();
        break;
        
    // External Control events
    case EV_EC_SET_OUTPUT_DIRECTORY:
        pDispatcher = new CEcSetOutputDirEventDispatcher();
        break;
    case EV_EC_START:              
        pDispatcher = new CEcStartEventDispatcher();
        break;
    case EV_EC_STOP:               
        pDispatcher = new CEcStopEventDispatcher();
        break;
    case EV_EC_ATTACH:             
        pDispatcher = new CEcAttachEventDispatcher();
        break;
    case EV_EC_DETACH:
        pDispatcher = new CEcDetachEventDispatcher();
        break;
    case EV_EC_CUSTOM_COMMAND:
        pDispatcher = new CEcCustomCommandEventDispatcher();
        break;
        
    // Internal events
    case EM_EVENT_INTERNAL_CLASS_PREPARE:                                                
    case EM_EVENT_INTERNAL_CLASS_UNLOAD:
    case EM_EVENT_INTERNAL_CLASS_LOAD:
    case EM_EVENT_INTERNAL_FUNCTION_UNLOAD:
    case EM_EVENT_INTERNAL_EXCEPTION_UNWIND_ENTER:
    case EM_EVENT_INTERNAL_EXCEPTION_UNWIND_LEAVE:
    case EM_EVENT_INTERNAL_OBJECT_MOVE:
    case EM_EVENT_ARENA_NEW:
    case EM_EVENT_ARENA_DELETE:
    case EM_EVENT_MODULE_LOAD_FINISHED:
    case EM_EVENT_THREAD_NAME_CHANGED:
    case EM_EVENT_INTERNAL_MODULE_UNLOAD:
    case EM_EVENT_INTERNAL_VM_OBJECT_ALLOC:
        pDispatcher = new CInternalEventDispatcher();
        break;
    default:
        // Do nothing. Event is not supported
        break;
    }
    return pDispatcher;
}

//
// Initialize the array of event descriptors
//
void CEventManager::InitEventDescriptors()
{
    m_vEventInfo[EV_NONE].pDescriptor =  
        new CEventDescriptor(EV_NONE, "Not an event");
    m_vEventInfo[EV_METHOD_ENTER].pDescriptor =  
        new CEventDescriptor(EV_METHOD_ENTER, "Method Enter", EG_CALL_GRAPH);
    m_vEventInfo[EV_METHOD_LEAVE].pDescriptor =   
        new CEventDescriptor(EV_METHOD_LEAVE, "Method Leave", EG_CALL_GRAPH);
    m_vEventInfo[EV_NEW_METHOD].pDescriptor =  
        new CEventDescriptor(EV_NEW_METHOD, "New Method");
    m_vEventInfo[EV_JITTED_METHOD_LOADED].pDescriptor =  
        new CEventDescriptor(EV_JITTED_METHOD_LOADED, "Compiled Method Loaded");
    m_vEventInfo[EV_JITTED_METHOD_UNLOADED].pDescriptor =  
        new CEventDescriptor(EV_JITTED_METHOD_UNLOADED, "Compiled Method Unloaded");
    m_vEventInfo[EV_THREAD_START].pDescriptor = 
        new CEventDescriptor(EV_THREAD_START, "Thread Start");
    m_vEventInfo[EV_THREAD_END].pDescriptor = 
        new CEventDescriptor(EV_THREAD_END, "Thread End");
    m_vEventInfo[EV_GC_START].pDescriptor = 
        new CEventDescriptor(EV_GC_START, "Grabage Collection Start");
    m_vEventInfo[EV_GC_END].pDescriptor = 
        new CEventDescriptor(EV_GC_END, "Garbage Collection End");
    m_vEventInfo[EV_VM_INIT].pDescriptor = 
        new CEventDescriptor(EV_VM_INIT, "VM Initialized");
    m_vEventInfo[EV_VM_SHUTDOWN].pDescriptor = 
        new CEventDescriptor(EV_VM_SHUTDOWN, "VM Shutdown");
    m_vEventInfo[EV_OBJECT_ALLOC].pDescriptor = 
        new CEventDescriptor(EV_OBJECT_ALLOC, "Object Allocated", EG_HEAP);
    m_vEventInfo[EV_OBJECT_FREE].pDescriptor = 
        new CEventDescriptor(EV_OBJECT_FREE, "Object Freed", EG_HEAP);
    m_vEventInfo[EV_MONITOR_WAIT].pDescriptor = 
        new CEventDescriptor(EV_MONITOR_WAIT, "Monitor Wait", EG_MONITOR);
    m_vEventInfo[EV_MONITOR_WAITED].pDescriptor =  
        new CEventDescriptor(EV_MONITOR_WAITED, "Monitor Waited", EG_MONITOR);
    m_vEventInfo[EV_CONTENDED_MONITOR_ENTER].pDescriptor = 
        new CEventDescriptor(EV_CONTENDED_MONITOR_ENTER, "Contended Monitor Enter", 
        EG_MONITOR);
    m_vEventInfo[EV_CONTENDED_MONITOR_ENTERED].pDescriptor =  
        new CEventDescriptor(EV_CONTENDED_MONITOR_ENTERED, "Contended Monitor Entered", 
        EG_MONITOR);
    m_vEventInfo[EV_THREAD_INTERACTION].pDescriptor =  
        new CEventDescriptor(EV_THREAD_INTERACTION, "Thread interaction", EG_THREAD_INTERACTION);
    m_vEventInfo[EV_EC_START].pDescriptor = 
        new CEventDescriptor(EV_EC_START, "External Control Start Notification"); 
    m_vEventInfo[EV_EC_STOP].pDescriptor = 
        new CEventDescriptor(EV_EC_STOP, "External Control Stop Notification"); 
    m_vEventInfo[EV_EC_SET_OUTPUT_DIRECTORY].pDescriptor = 
        new CEventDescriptor(EV_EC_SET_OUTPUT_DIRECTORY, 
        "External Control Output Directory Changed"); 
    m_vEventInfo[EV_JAVA_DYNAMIC_CODE_GENERATED].pDescriptor = 
        new CEventDescriptor(EV_JAVA_DYNAMIC_CODE_GENERATED, "Dynamic Code Generated"); 
    m_vEventInfo[EV_JAVA_CLASS_FILE_LOAD_HOOK].pDescriptor = 
        new CEventDescriptor(EV_JAVA_CLASS_FILE_LOAD_HOOK, "Java Class File Load Hook"); 
    m_vEventInfo[EV_EC_ATTACH].pDescriptor = 
        new CEventDescriptor(EV_EC_ATTACH, "External Control Attach Notification"); 
    m_vEventInfo[EV_EC_DETACH].pDescriptor = 
        new CEventDescriptor(EV_EC_DETACH, "External Control Detach Notification");
    m_vEventInfo[EV_EC_CUSTOM_COMMAND].pDescriptor = 
        new CEventDescriptor(EV_EC_CUSTOM_COMMAND, "External Control Custom Command");
    m_vEventInfo[EV_LAST].pDescriptor = 
        new CEventDescriptor(EV_LAST, "End Sentinel for external events (not an event)");
    m_vEventInfo[EM_EVENT_INTERNAL_CLASS_PREPARE].pDescriptor = 
        new CEventDescriptor(EM_EVENT_INTERNAL_CLASS_PREPARE, "Class Prepare (internal)"); 
    m_vEventInfo[EM_EVENT_INTERNAL_CLASS_UNLOAD].pDescriptor = 
        new CEventDescriptor(EM_EVENT_INTERNAL_CLASS_UNLOAD, "Class Unload (internal)");
    m_vEventInfo[EM_EVENT_INTERNAL_CLASS_LOAD].pDescriptor = 
        new CEventDescriptor(EM_EVENT_INTERNAL_CLASS_LOAD, "Class Load (internal)"); 
    m_vEventInfo[EM_EVENT_INTERNAL_FUNCTION_UNLOAD].pDescriptor = 
        new CEventDescriptor(EM_EVENT_INTERNAL_FUNCTION_UNLOAD, "Function Unload (internal)");
    m_vEventInfo[EM_EVENT_INTERNAL_EXCEPTION_UNWIND_ENTER].pDescriptor = 
        new CEventDescriptor(EM_EVENT_INTERNAL_EXCEPTION_UNWIND_ENTER, 
        "Exception Unwind Enter (internal)");
    m_vEventInfo[EM_EVENT_INTERNAL_EXCEPTION_UNWIND_LEAVE].pDescriptor = 
        new CEventDescriptor(EM_EVENT_INTERNAL_EXCEPTION_UNWIND_LEAVE,
        "Exception Unwind Leave (internal)");
    m_vEventInfo[EM_EVENT_INTERNAL_OBJECT_MOVE].pDescriptor = 
        new CEventDescriptor(EM_EVENT_INTERNAL_OBJECT_MOVE, "Object Move (internal)");
    m_vEventInfo[EM_EVENT_ARENA_NEW].pDescriptor = 
        new CEventDescriptor(EM_EVENT_ARENA_NEW, "New Arena (internal)");
    m_vEventInfo[EM_EVENT_ARENA_DELETE].pDescriptor = 
        new CEventDescriptor(EM_EVENT_ARENA_DELETE, "Arena Deleted (internal)");
    m_vEventInfo[EM_EVENT_MODULE_LOAD_FINISHED].pDescriptor = 
        new CEventDescriptor(EM_EVENT_MODULE_LOAD_FINISHED, "Module Load Finish (internal)");
    m_vEventInfo[EM_EVENT_INTERNAL_VM_OBJECT_ALLOC].pDescriptor = 
        new CEventDescriptor(EM_EVENT_INTERNAL_VM_OBJECT_ALLOC, "VM Object Alloc (internal)");
    m_vEventInfo[EM_EVENT_LAST].pDescriptor = 
        new CEventDescriptor(EM_EVENT_LAST, "End Sentinel for internal events (not an event)");
}

//
// Remove information of all registered clients for the specified event
//
TResult CEventManager::RemoveEventInformation(IEventObserver& event)
{
    SEmEventInfo *pEventInfo = &m_vEventInfo[event.Type()];
    pEventInfo->pDispatcher->UnregisterAll();
    return MRTE_RESULT_OK;
}

//
// Update the information of an event upon registration of a new client
//
// Parameters:
//      clientId    : the id of the client that registered to the event
//      event       : the event object being registered
//      bIsInternal : whether this is an internal event (true) or 
//                    a Profiler (MPI) event (false)
//      bIsEnabled  : whether the event is initially enabled (true) or not (false)
//
// Returns
//      MRTE_RESULT_OK      : event successfully updated
//      MRTE_OUT_OF_MEMORY  : memory allocation error
//      MRTE_ERROR_FAIL     : the client has already registered for the event
//
TResult CEventManager::UpdateEventInformation(TId clientId, IEventObserver& event, 
                                              EEventPriority priority, 
                                              bool bIsInternal,
                                              bool bIsEnabled)
{
    // This function should be called after the event parameters were checked.
    // Event parameters are assumed to be correct and are not checked again.
    
    SEmEventInfo *pEventInfo = &m_vEventInfo[event.Type()];

    MARTINI_ASSERT("CEventManager", pEventInfo != NULL, "");
    if (NULL == pEventInfo)
    {
        return MRTE_ERROR_FAIL;
    }

    if (NULL == pEventInfo->pDispatcher)
    {
        return MRTE_ERROR_NOT_SUPPORTED;
    }
    
    CObserverInfo* pNewObserver = new CObserverInfo(&event, clientId, priority, bIsInternal,
        event.EventDataTypes(), bIsEnabled);
    if (NULL == pNewObserver)
    {
        return MRTE_ERROR_OUT_OF_MEMORY;
    }

    TResult res = pEventInfo->pDispatcher->Register(pNewObserver);
    if (MRTE_FAILED(res))
    {
        return res;
    }

    // if the event is enabled, mark it as 'globally enabled' as well
    if (bIsEnabled)
    {
        pEventInfo->bIsEnabledByClients = true;
    }

    // increment the number of registered clients
    ++pEventInfo->uiNumClients;
    
    
    return MRTE_RESULT_OK;
}

//
// Returns the status (enabled/disabled) of an event group for the specified client
//
bool CEventManager::GetEventGroupStatusForClient(TId clientId, EEventGroup eventGroup)
{
    TObserverList eventsList;
    GetRegisteredEventsForClient(&eventsList, clientId);
    TMRTEHandle hEvent = eventsList.GetFirst();
    while (NULL != hEvent)
    {
        CObserverInfo *pObserverInfo = eventsList.GetData(hEvent);
        hEvent = eventsList.GetNext(hEvent);
        TEventType eventType = pObserverInfo->pObserver->Type();
        if (IsEventInGroup(eventType, eventGroup))
        {
            // All events in the group must have the same status, so returning
            // the status of any of these events is ok
            return pObserverInfo->bIsEnabled;
        }
    }
    // If we reached here, it means that the client has not registered for any 
    // events in eventGroup. The group is therefore enabled by default
    return true;
}

//
// Updates the 'enabled' status of all events in the group for the specified client,
// and also the global 'enabled' status of these events.
//
// Parameters:
//      clientId    : the event client
//      eventGroup  : the event group
//      bIsEnabled  : the 'enabled' status to use
//
void CEventManager::SetEventGroupStatusForClient(TId clientId, 
                                                 EEventGroup eventGroup, 
                                                 bool bIsEnabled)
{
    TObserverList eventsList;
    GetRegisteredEventsForClient(&eventsList, clientId);
    TMRTEHandle hEvent = eventsList.GetFirst();
    while (NULL != hEvent)
    {
        CObserverInfo *pObserverInfo = eventsList.GetData(hEvent);
        hEvent = eventsList.GetNext(hEvent);
        TEventType eventType = pObserverInfo->pObserver->Type();
        if (IsEventInGroup(eventType, eventGroup))
        {
            pObserverInfo->bIsEnabled = bIsEnabled;
        }
    }

    // Update the global 'enabled' status of all events in the group
    TEventTypeList eventTypeList;
    GetEventsInGroup(&eventTypeList, eventGroup);
    TMRTEHandle hEventType = eventTypeList.GetFirst();
    while (NULL != hEventType)
    {
        TEventType eventType = eventTypeList.GetData(hEventType);
        hEventType = eventTypeList.GetNext(hEventType);
        UpdateEventGlobalEnabledStatus(eventType);
    }
}

//
// Updates the global 'enabled' status of the specified event
// An event is 'globally enabled' if it is enabled by at least one client.
// An event is 'globally disabled' if it is disabled by all clients
//
// Parameters:
//      eventType   : the event to update
//
void CEventManager::UpdateEventGlobalEnabledStatus(TEventType eventType)
{
    SEmEventInfo *pEventInfo = &m_vEventInfo[eventType];
    if (NULL == pEventInfo)
    {
        return;
    }
    
    bool bNewStatus = false;

    TObserverList &eventObservers = pEventInfo->pDispatcher->GetObservers();
    TMRTEHandle hClient = eventObservers.GetFirst();
    while (NULL != hClient)
    {
        CObserverInfo *pObserverInfo = eventObservers.GetData(hClient);
        hClient = eventObservers.GetNext(hClient);
        if (pObserverInfo->bIsEnabled)
        {
            bNewStatus = true;
            break;
        }
    }

    pEventInfo->bIsEnabledByClients = bNewStatus;
}

//
// Returns the global 'enabled' status of the specified event
//
bool CEventManager::GetEventGlobalEnabledStatus(TEventType eventType)
{
    SEmEventInfo *pEventInfo = &m_vEventInfo[eventType];
    if (NULL == pEventInfo)
    {
        return false;
    }
    return pEventInfo->bIsEnabledByClients;
}

//
// Returns true if at least one of the events in the specified group is globally enabled
//
bool CEventManager::GetEventsGlobalEnabledStatus(EEventGroup eventGroup)
{
    bool bResult = false;
    switch (eventGroup)
    {
    case EG_CALL_GRAPH:
        bResult = GetEventGlobalEnabledStatus(EV_METHOD_ENTER)
            || GetEventGlobalEnabledStatus(EV_METHOD_LEAVE);
        break;
    case EG_HEAP:
        bResult = GetEventGlobalEnabledStatus(EV_OBJECT_ALLOC)
            || GetEventGlobalEnabledStatus(EV_OBJECT_FREE);
        break;
    case EG_MONITOR:
        bResult = GetEventGlobalEnabledStatus(EV_MONITOR_WAIT)
            || GetEventGlobalEnabledStatus(EV_MONITOR_WAITED)
            || GetEventGlobalEnabledStatus(EV_CONTENDED_MONITOR_ENTER)
            || GetEventGlobalEnabledStatus(EV_CONTENDED_MONITOR_ENTERED);
    case EG_THREAD_INTERACTION:
        bResult = GetEventGlobalEnabledStatus(EV_THREAD_INTERACTION);
        break;
    }
    return bResult;
}

// WARNING: All the support for heap profiler in this slice is only for win32!! 
// in order to port it to 64 changes should be done in data manager in order to support it 
//void CEventManager::SetObjInfoWUserData(TEventType mpiEvent, SData *pMpiData, 
//                                        SJVMData *pJVMEventData)
//{
//    UIOP JVMPIobjId; 
//    if (mpiEvent == EV_OBJECT_ALLOC)
//    {
//        JVMPIobjId = (UIOP)pStdEventData->u.obj_alloc.obj_id;
//    }
//    else
//    {
//        assert(mpiEvent == EV_OBJECT_FREE);
//        JVMPIobjId = (UIOP)pStdEventData->u.obj_free.obj_id;
//    }
//    pMpiData->pUserData = m_pDataManager->GetObjectUserData(JVMPIobjId);
//}


//<summary> Set the internal modules, called by the kernel at initialization phase </summary>
//<param name='pJavaInterface'> standard Java interface (JVMPI/JVMTI) </param>
//<param name='pDataManager'> data manager </param>
//<param name='pParamChecker'> parameter checker </param>
//<param name='pKernel'> kernel </param>
//<returns> MRTE_RESULT_OK on success, or an error code in case of a failure </returns>
TResult CEventManager::SetInternalModules(CDataManager *pDataManager,
                                       CParamChecker *pParamChecker, 
                                       CJavaInstrumentorManager *pJIM,
                                       CECAgent *pECAgent, CJPIKernel *pKernel)
{
    MARTINI_ASSERT("CEventManager", pDataManager && pParamChecker && pKernel, "");
    m_pJavaInterface = CJpiGlobals::Instance()->pJvmInterface;
    m_pDataManager = pDataManager;
    m_pKernel = pKernel;
    m_pParamChecker = pParamChecker;
    m_pJavaInstrumentorManager = pJIM;
    m_pECAgent = pECAgent;
    m_pParamChecker->SetEventsLimits(EV_NONE, (TEventType) EM_EVENT_LAST);

    if (!m_csEnableDisableEvent)
    {
        m_csEnableDisableEvent = CreateThreadSync();
        if (!m_csEnableDisableEvent)
        {
            MARTINI_ERROR("CEventManager", "Error creating critical section");
            return MRTE_ERROR_OSA_FAILURE;
        }
    }

    // Register for the VM_SHUTDOWN event
    if (NULL == m_pVMShutdownEvent)
    {
        m_pVMShutdownEvent = new CEMShutdownEventObserver(this);
        RegisterEventInternal(m_ModuleId, *m_pVMShutdownEvent);
    }

    // Initialize JPI global information
    CJpiGlobals *pJpiGlobals = CJpiGlobals::Instance();
    pJpiGlobals->pfnMethodEnterHandler = MethodEnterHandler;
    pJpiGlobals->pfnMethodLeaveHandler = MethodLeaveHandler;
    pJpiGlobals->pfnMethodLeaveWithExcetionHandler = MethodLeaveWithException;
    pJpiGlobals->pfnObjectAllocHandler = ObjectAllocHandler;
    pJpiGlobals->pfnArrayAllocHandler = ArrayAllocHandler;
    pJpiGlobals->pfnGetObjTIdHandler = GetObjTIdHandler;
    pJpiGlobals->pfnThreadInteractionHandler = ThreadInteractionHandler;

    return MRTE_RESULT_OK;
}

void CEventManager::NotifyECEvent(TEventType event, 
                                  char *szData, 
                                  unsigned int commandId, 
                                  void *pData)
{
    SEmData data;
    SECEventData ecData;

    switch (event)
    {
    case EV_EC_SET_OUTPUT_DIRECTORY:
        data.dataType = MPI_EXTERNAL_CONTROL;
        ecData.szECConfigurationPath = szData;
        data.u.pECData = &ecData;
    	break;
    case EV_EC_CUSTOM_COMMAND:
        data.dataType = MPI_EXTERNAL_CONTROL;
        ecData.commandId = commandId;
        ecData.pData = pData;
        data.u.pECData = &ecData;

        break;
    default:
        data.dataType = MPI_IMPLEMENTOR_NONE;
        break;
    }
    
    NotifyMpiEvent(event, &data);
}


//
// Dispatches an MPI event to registered clients.
//
// Parameters
//      mpiEvent    [in] : The event to dispatch
//      pEmData     [in] : Additional event information.
//                         The SEmData.dataType member contains the event information type:
//                         MPI_VM: the event data was obtained from the VM and is stored
//                                 in the pJVMData member.
//                         MPI_INTERNAL: the event data was obtained from internal databases
//                                       and is stored in the pMpiData member. This is used 
//                                       for dispatching internal (non-VM) events such as 
//                                       EV_NEW_METHOD.
//                         MPI_EXTERNAL_CONTROL: the event data was obtained from an External
//                                               Control module and is stored in the 
//                                               szECConfigurationPath member.
//
void CEventManager::NotifyMpiEvent(TEventType mpiEvent, SEmData *pEmData)
{
    // Do not dispatch any event after VM_SHUTDOWN
    if (IsVMShutdownReceived())
    {
        return;
    }
    
    CEventDispatcher *pDispatcher = m_vEventInfo[mpiEvent].pDispatcher;
    if (pDispatcher)
    {
        pDispatcher->NotifyObservers(pEmData);
    }
}


void CEventManager::SetEventImplementor(TEventType mpiEvent, 
                                        EEventImplementationType implementor)
{
    // special treatment to method-enter\leave events
    if (mpiEvent == EV_METHOD_ENTER || mpiEvent == EV_METHOD_LEAVE)
    {
        // if implementor is instrumentor manager, have to make sure it can generate ALL events
        if (implementor == MPI_INSTRUMENTATION)
        {
            if (m_pJavaInterface->CanSupplyClassLoadHookToAllClasses())
            {
                // could run over JVM implementor. This is ok since JIE has a higher priority
                m_vEventInfo[mpiEvent].implementor = implementor;
                return;
            }
            else
            {
                m_vEventInfo[mpiEvent].implementor |= MPI_TENTATIVE_INSTRUMENTATION;
                return;
            }
        }
    }

    m_vEventInfo[mpiEvent].implementor |= implementor;
}

bool CEventManager::UpdateTentativeInstrumentation(bool bIsCapableToInstrumentAllClasses)
{
    TResult iRes = MRTE_RESULT_OK;
    bool bShouldContinueInstrumentation = true;
    if (bIsCapableToInstrumentAllClasses)
    {
        // have to disable method enter-leave events from the VM
        iRes = m_pJavaInterface->DisableEvent(EV_METHOD_ENTER);
        if (MRTE_FAILED(iRes))
        {
            MARTINI_ERROR("CEventManager", "Failed to disable event");
        }
        iRes = m_pJavaInterface->DisableEvent(EV_METHOD_LEAVE);
        bShouldContinueInstrumentation = true;
    }
    else
    {
        // have to disable instrumentation events
        iRes = m_pJavaInstrumentorManager->DisableInstrumentation();
        bShouldContinueInstrumentation = false;
    }
    if (MRTE_FAILED(iRes))
    {
        MARTINI_ERROR("CEventManager", "Failed to disable instrumentation");
    }
    return bShouldContinueInstrumentation;
}

//
// Returns the events registered by the specified client
//
// Parameters:
//      pEventsList     [out]   : a pointer to an initialized TEventsList object to be filled
//                                with the list of events
//      clientId        [in]    : the client id
//
void CEventManager::GetRegisteredEventsForClient(TObserverList *pEventsList, TId clientId)
{
    if (NULL == pEventsList)
    {
        return;
    }

    // fill the list of events the client is registered for
    unsigned int i;
    for (i = 0; i < EM_MAX_EVENTS_TOTAL; ++i)
    {
        SEmEventInfo &eventInfo = m_vEventInfo[i];
        if (eventInfo.uiNumClients > 0)
        {
            TObserverList &clientList = eventInfo.pDispatcher->GetObservers();
            TMRTEHandle hClient = clientList.GetFirst();
            while (NULL != hClient)
            {
                CObserverInfo *pObserverInfo = clientList.GetData(hClient);
                if (NULL != pObserverInfo)
                {
                    if (pObserverInfo->clientId == clientId)
                    {
                        TMRTEHandle hLast = pEventsList->GetLast();
                        pEventsList->InsertAfter(hLast, pObserverInfo);
                    }
                }
                hClient = clientList.GetNext(hClient);
            } // end of while (more registered clients exist)
        } // end of if (event clients exist)
    } // end of for (all supported events)
}

TResult CEventManager::DisableEventInternal(TEventType mpiEvent)
{
    TResult iRes = MRTE_RESULT_OK;
    unsigned int implementor = GetEventImplementationType(mpiEvent);
    switch(implementor) 
    {
    case MPI_VM:
        iRes = m_pJavaInterface->DisableEvent(mpiEvent);
    	break;
    default:
        iRes = MRTE_ERROR_NOT_SUPPORTED;
    }
    return iRes;
}

/** Called by DisableEventGroup or EnableEventGroup functions */
TResult CEventManager::SetEventGroupEnabled(TId clientId, 
                                            EEventGroup eventGroup, 
                                            bool bIsEnabled)
{
    if (!CJpiGlobals::Instance()->IsThreadMpiSafe())
    {
        return MRTE_ERROR_FAIL;
    }

    //
    // The current version has the following limitation:
    // 1. The API is supported only for Java 5.0 (JDK 1.5.0) and above (JVMTI).
    // 2. Only one client can register for a specific event. Therefore, we don't have to worry
    //    about multiple clients enabling/disabling events
    //
    // A note about marking events as enabled/disabled for the specified client:
    // When the event status is being changed during the execution of the profiled Java
    // application, there may be a need to redefine loaded classes and apply different
    // BCI to them, which is a lengthy process. To make sure the client perceives this
    // as an atomic operation, the following logic is being used:
    // When disabling events: the event will be marked as 'disabled' before redefining
    // loaded classes.
    // When enabling events: the event will be marked as 'enabled' after redefining
    // loaded classes.
    //
    
    // Check if the operation is supported for the specified event
    switch (eventGroup)
    {
		case EG_CALL_GRAPH:
		case EG_HEAP:
		case EG_THREAD_INTERACTION:
			if (!m_pJavaInterface->GetCapabilities()->Enabled(
				CT_CAN_ENABLE_OR_DISABLE_BCI_GENERATED_EVENTS))
			{
				return MRTE_ERROR_NOT_SUPPORTED;
			}
			break;
		case EG_MONITOR:
			// No special capabilities are needed as we use only JVMTI events
			break;
		default:
			// Operation not supported for other event groups
			return MRTE_ERROR_NOT_SUPPORTED;
    }
    
    if (!IsClientRegisteredForEventInGroup(clientId, eventGroup))
    {
        return MRTE_ERROR_FAIL;
    }

    if (m_bIsInitialiazationCompleted)
    {
        if (!CJpiGlobals::Instance()->bJvmInitDone)
        {
            return MRTE_ERROR_PHASE_FAILURE;
        }
    }
    
    m_csEnableDisableEvent->Enter();

    // Set a flag to indicate whether the new event status is the same as the global status.
    // We use this flag to determine whether we need to redefine all loaded classes
    bool bNeedToRedefineClasses = (bIsEnabled != GetEventsGlobalEnabledStatus(eventGroup));

    // Redefine loaded classes (if needed)
    TResult iRes = MRTE_RESULT_OK;
    if (bNeedToRedefineClasses)
    {
        if (m_bIsInitialiazationCompleted)
        {
            bool bCurrentStatus = GetEventGroupStatusForClient(clientId, eventGroup);
            if (!bIsEnabled)
            {
                // If events are being disabled, update the events status before redefining
                // classes
                SetEventGroupStatusForClient(clientId, eventGroup, false);
            }

            //TODO: generalize this code to use event implementor information
            // If we are doing monitor contention profiling (EG_MONITOR is out event group)
            if (EG_MONITOR == eventGroup)
            {
                bool bAttached;
                JNIEnv *pJNIEnv;
                TResult iRes = m_pJavaInterface->AttachCurrentThread(&pJNIEnv, &bAttached);
                if (MRTE_SUCCEEDED(iRes))
                {
                    TEventTypeList monitorEventList;
                    GetEventsInGroup(&monitorEventList, EG_MONITOR);
                    TMRTEHandle hMonitorEvent = monitorEventList.GetFirst();

                    // For each of the events in the list
                    while (hMonitorEvent != NULL)
                    {
                        if (bIsEnabled)
                        {
                            m_pJavaInterface->EnableEvent(monitorEventList.GetData(hMonitorEvent));
                        } else  {
                            m_pJavaInterface->DisableEvent(monitorEventList.GetData(hMonitorEvent));
                        }

                        hMonitorEvent = monitorEventList.GetNext(hMonitorEvent);
                    }
                    if (bAttached)
                    {
                        m_pJavaInterface->DetachCurrentThread();
                    }
                }
                else
                {
                    MARTINI_ERROR("CEventManager", "[SetEventGroupEnabled] failed to attach "
                        "the External Control thread to the VM");
                    iRes = MRTE_ERROR_FAIL;
                }
            }
            else
            { // ... else, if we are not doing monitor contention profiling...
                iRes = m_pJavaInstrumentorManager->ApplyInstrForLoadedClasses(eventGroup, 
                    bIsEnabled);
                iRes = m_pJavaInstrumentorManager->SetEventStreamStatus(bIsEnabled);
            }

            if (MRTE_FAILED(iRes))
            {
                // Restore original status in case of an error
                SetEventGroupStatusForClient(clientId, eventGroup, bCurrentStatus);
            }
        }
        else
        {	// ... else if m_bIsInitialiazationCompleted is false...
            //
        	// JPI is still initializing. The Java application is not running yet.
            // Just update the instrumentation status
            m_pJavaInstrumentorManager->SetInstrGeneratedEventsStatus(bIsEnabled);

            //TODO: generalize this code to use event implementor information
            // If we are doing monitor contention profiling (EG_MONITOR is out event group)
            if (EG_MONITOR == eventGroup)
            {
                TEventTypeList monitorEventList;
                GetEventsInGroup(&monitorEventList, EG_MONITOR);
                TMRTEHandle hMonitorEvent = monitorEventList.GetFirst();
                while (hMonitorEvent != NULL)
                {
                    if (bIsEnabled)
                    {
                        m_pJavaInterface->EnableEvent(monitorEventList.GetData(
                            hMonitorEvent));
                    }
                    else
                    {
                        m_pJavaInterface->DisableEvent(monitorEventList.GetData(
                            hMonitorEvent));
                    }
                    hMonitorEvent = monitorEventList.GetNext(hMonitorEvent);
                }
            }
        } // if (m_bIsInitialiazationCompleted)
    }

    if (MRTE_RESULT_OK == iRes)
    {
        // Update the event status for the specified client
        SetEventGroupStatusForClient(clientId, eventGroup, bIsEnabled);
    }

    m_csEnableDisableEvent->Leave();

    return iRes;
}

bool CEventManager::EventExists(TEventType eventType)
{
    SEmEventInfo *pEventInfo = &m_vEventInfo[eventType];
    if (NULL == pEventInfo)
    {
        return false;
    }
    return true;
}

//
// Checks whether the specified client is registered for an event in the specified group
//
// Parameters:
//      clientId    : the client
//      group       : the event group
//
// Returns:
//      true        : the client is registered for an event in the group
//      false       : the client is not registered for an event in the group
//
bool CEventManager::IsClientRegisteredForEventInGroup(TId clientId, EEventGroup group)
{
    // Find the events to which the client registered
    TObserverList observersList;
    GetRegisteredEventsForClient(&observersList, clientId);
    if (observersList.Count() == 0)
    {
        // The client is not registered to any events
        return false;
    }

    // Find the events belonging to the event group
    TEventTypeList eventsInGroup;
    GetEventsInGroup(&eventsInGroup, group);
    if (eventsInGroup.Count() == 0)
    {
        // No events in the group, or unknown group
        return false;
    }

    // Check if the client is registered to any of the events in the group
    TMRTEHandle hEvent = observersList.GetFirst();
    while (NULL != hEvent)
    {
        CObserverInfo *pObserverInfo = observersList.GetData(hEvent);
        hEvent = observersList.GetNext(hEvent);
        if (NULL != pObserverInfo)
        {
            TMRTEHandle hEventInGroup = eventsInGroup.GetFirst();
            while (NULL != hEventInGroup)
            {
                TEventType eventType = eventsInGroup.GetData(hEventInGroup);
                hEventInGroup = eventsInGroup.GetNext(hEventInGroup);
                if (pObserverInfo->pObserver->Type() == eventType)
                {
                    return true;
                }
            }
        }
    }
    return false;
}

bool CEventManager::IsCallGraphEvent(TEventType eventType)
{
    EEventGroup group = m_vEventInfo[eventType].pDescriptor->Group();
    if (EG_CALL_GRAPH == group)
    {
        return true;
    }
    return false;
}

IEventFilter* CEventManager::GetEventGroupFilter(EEventGroup group)
{
    return m_vEventGroupInfo[group].pFilter;    
}

//
// Generates the EV_NEW_METHOD event
//
void CEventManager::NewMethodEvent(TId methodId, const JNIEnv *pJniEnv)
{
    // Dispatch the event
    SMpiInternalData mpiData;
    SEmData data;

    mpiData.validData = m_newMethodDataRequestType;

    mpiData.methodId = methodId;

    data.dataType = MPI_INTERNAL;
    data.u.pMpiData = &mpiData;
    data.pJniEnv = pJniEnv;
    NotifyMpiEvent(EV_NEW_METHOD, &data);

    // Mark that EV_NEW_METHOD was reported for this method
    m_pDataManager->SetNewMethodReported(methodId);
}

void CEventManager::GenMethodEnterEvent(JNIEnv *env, jobject self, bool bIsAlreadyInvoked, jint id)
{
    if (IsVMShutdownReceived())
    {
        return;
    }

    Martini::MPI::SMethodEnterEventData mpiData;
    mpiData.validData = GetMethodEnterDataType();
    mpiData.methodId = id;
    
    if (mpiData.validData & DR_MODULE_ID)
    {
        static Martini::MPI::TId s_ConstantModuleId = m_pDataManager->GetConstantModuleId();
        if (s_ConstantModuleId != MODULE_UNINITIALIZED)
        {
            mpiData.moduleId = s_ConstantModuleId;
        }
        else
        {
            mpiData.moduleId = m_pDataManager->GetMethodModuleId(id); 
            MARTINI_ASSERT("CEventManager", mpiData.moduleId != 0, "");
        }
    }
    
    if (!bIsAlreadyInvoked)
    {
        NewMethodEvent(id, env);
    }
    
    if (mpiData.validData & DR_THREAD_ID)
    {
        static CThreadInfoManager *pThreadInfoManager = m_pDataManager->GetThreadInfoManager();
        // Check TLS data. If it does not contain the thread id, update the TLS.
        // If the thread is not known to the Thread Manager, generate a EV_THREAD_START
        // event
        STlsThreadInfo *pTlsThreadInfo = pThreadInfoManager->GetTlsInfo();
        if (NULL != pTlsThreadInfo)
        {
            mpiData.threadId = pTlsThreadInfo->threadId;
        }
        else
        {
            mpiData.threadId = pThreadInfoManager->GetThreadIdAndUpdateTls(env);
        }
    }

    if (mpiData.validData & DR_THREAD_CPU_TIME)
    {
        m_pJavaInterface->GetCurrentThreadCpuTime(&mpiData.uiCpuNanos);
    }

    if (mpiData.validData & DR_THREAD_ELAPSED_TIME)
    {
        mpiData.uiElapsedNanos = GetTimeStamp();
    }
    
    CMethodEnterEventDispatcher *pDispatcher = 
        (CMethodEnterEventDispatcher*)(m_vEventInfo[EV_METHOD_ENTER].pDispatcher);

    if (NULL != pDispatcher)
    {
        pDispatcher->NotifyMethodEnterObservers(mpiData);
    }
}

void CEventManager::GenMethodLeaveEvent(JNIEnv *env, jobject self, jint id)
{
    if (IsVMShutdownReceived())
    {
        return;
    }
    
    Martini::MPI::SMethodLeaveEventData mpiData;
    mpiData.validData = GetMethodLeaveDataType();
    mpiData.methodId = id;

    if (mpiData.validData & DR_THREAD_ID)
    {
        static CThreadInfoManager *pThreadInfoManager = m_pDataManager->GetThreadInfoManager();
        
        // Check TLS data. If it does not contain the thread id, update the TLS.
        // If the thread is not known to the Thread Manager, generate a EV_THREAD_START
        // event
        STlsThreadInfo *pTlsThreadInfo = pThreadInfoManager->GetTlsInfo();
        if (NULL != pTlsThreadInfo)
        {
            mpiData.threadId = pTlsThreadInfo->threadId;
        }
        else
        {
            mpiData.threadId = pThreadInfoManager->GetThreadIdAndUpdateTls(env);
        }
    }

    if (mpiData.validData & DR_THREAD_CPU_TIME)
    {
        m_pJavaInterface->GetCurrentThreadCpuTime(&mpiData.uiCpuNanos);
    }

    if (mpiData.validData & DR_THREAD_ELAPSED_TIME)
    {
        mpiData.uiElapsedNanos = GetTimeStamp();
    }
    
    CMethodLeaveEventDispatcher *pDispatcher = 
        (CMethodLeaveEventDispatcher*)(m_vEventInfo[EV_METHOD_LEAVE].pDispatcher);

    if (NULL != pDispatcher)
    {
        pDispatcher->NotifyMethodLeaveObservers(mpiData);
    }
}

TResult CEventManager::GenObjectAllocEvent(JNIEnv *pJNIEnv, jobject obj, 
                                                            jlong tag, jint methodId, jlong loc)
{
    static CObjectInfoManager *s_pObjectInfoManager = m_pDataManager->GetObjectInfoManager();
    
    SJvmtiObjectTag *pTagInfo = s_pObjectInfoManager->GetTagInfo(tag);
    
    // Update tag information
    pTagInfo->bNotify = true;

    // Notify the event
    SMpiInternalData mpiData;
    mpiData.pObjectTag = pTagInfo;
    mpiData.objectJniLocalRef = obj;
    mpiData.allocLocation = loc;
    mpiData.methodId = methodId;

    SEmData eventData;
    eventData.dataType = MPI_INTERNAL;
    eventData.pJniEnv = pJNIEnv;
    eventData.u.pMpiData = &mpiData;
    NotifyMpiEvent(Martini::MPI::EV_OBJECT_ALLOC, &eventData);
    return MRTE_RESULT_OK;  
}
//////////////////////////////////////////////////////////////////////////
// CEMShutdownEvent implementation

void CEMShutdownEventObserver::HandleEvent(SEmData *pEvent, 
                                           void *pUserData)
{
    m_pEventManager->VMShutdownReceived();
}

//////////////////////////////////////////////////////////////////////////
// External functions implementation

//
// Callback function called by the instrumented application (through the CGProxy Java
// class) to notify about method-enter event
//
extern "C" void JNICALL MethodEnterHandler(JNIEnv *env, 
                                           jobject self, 
                                           bool bIsAlreadyInvoked, 
                                           jint id)
{
    s_pEM->GenMethodEnterEvent(env, self, bIsAlreadyInvoked, id);
}

//
// Callback function called by the instrumented application (through the CGProxy Java
// class) to notify about method-leave event
//
extern "C" void JNICALL MethodLeaveHandler(JNIEnv *env, jobject self, jint id)
{
    s_pEM->GenMethodLeaveEvent(env, self, id);
}

//
// Callback function called by the instrumented application (through the CGProxy Java
// class) to notify about an abnormal method exit (exception)
// 
// This callback is for future use. It is not being used in the current version.
//
extern "C" void JNICALL MethodLeaveWithException(JNIEnv *env, jobject self, jint id)
{
    s_pEM->GenMethodLeaveEvent(env, self, id);
}

void InternalObjectAllocHandler(JNIEnv *env, jobject self, 
                                jobject obj, 
                                jclass objClass,
                                jint methodId,
                                jlong loc,
                                bool isArray)
{

    if (s_pEM->IsVMShutdownReceived())
    {
        return;
    }

    //TODO: replace static variables with a singleton to improve performance
    static CDataManager *s_pDataManager = s_pEM->GetDM();
    static IJVM *s_pJVMInterface = s_pEM->GetJVMInterface();
    static CObjectInfoManager *s_pObjectInfoManager = s_pDataManager->GetObjectInfoManager();    
        
    // Invoke the selectivity filter (if defined) to determine whether to report an
    // event for this object. This is not optimal, and can be improved by applying the
    // selectivity filter during instrumentation (not supported in this slice).
    bool bNotify = true;
    jlong classTag;
    TResult res = s_pJVMInterface->GetObjectTag(&classTag, objClass);
    SJvmtiObjectTag *pClsTagInfo = s_pObjectInfoManager->GetTagInfo(classTag);
    const char *szClassName = NULL;
    Martini::RTUtil::MCString className;
    Martini::RTUtil::MCString classFriendlyName;

    if(pClsTagInfo != NULL && pClsTagInfo->jclassInfo.szClassName != NULL){
        szClassName = pClsTagInfo->jclassInfo.szClassName;
    }else{
        s_pJVMInterface->GetObjectClassName(className, obj);
        s_pDataManager->GetClassName(className.Get(), classFriendlyName);
        szClassName = classFriendlyName.Get();
    }

    // printf("DEBUG: received alloc event for class %s", szClassName);

    if (!MRTE_FAILED(res))
    {
        IHeapFilter *pHeapFilter = (IHeapFilter*)s_pEM->GetEventGroupFilter(EG_HEAP);
        if (pHeapFilter != NULL )
        {
            SHeapFilterData heapFilterData;
            heapFilterData.szClassName = szClassName;
            bNotify = pHeapFilter->ShouldNotify(heapFilterData);
        }
    }

    if (bNotify)
    {
        // Tag the object
        jlong tag;
        TId objectId;
        if (MRTE_FAILED(s_pObjectInfoManager->CreateTag(&objectId, &tag, obj, 
            env, false /* bSaveInDb */)))
        {
            // Cannot tag the object. Do not report the event
            MARTINI_INFORMATIVE("InternalObjectAllocHandler", 0, false, 
                "failed to allocate object id");
            return;
        }

        SJvmtiObjectTag *pTagInfo = s_pObjectInfoManager->GetTagInfo(tag);

        // Update tag information
        if(pTagInfo != NULL){
            pTagInfo->jobjectInfo.szClassName = szClassName;
        }
        
        s_pEM->GenObjectAllocEvent(env, obj, tag, methodId, loc);
    }
}

//
// Callback function called by the instrumented application (through the HeapProxy Java
// class) to notify about a non-array object allocation
//
extern "C" void JNICALL ObjectAllocHandler(JNIEnv *env, jclass self, 
                                           jobject obj, 
                                           jclass objClass, 
                                           jint methodId, 
                                           jlong loc)
{
    InternalObjectAllocHandler(env, self, obj, objClass, methodId, loc, false /* isArray */);
}

//
// Callback function called by the instrumented application (through the HeapProxy Java
// class) to notify about an array allocation
//
extern "C" void JNICALL ArrayAllocHandler(JNIEnv *env, jclass self, 
                                          jobject obj, 
                                          jclass objClass, 
                                          jint methodId, 
                                          jlong loc)
{
    InternalObjectAllocHandler(env, self, obj, objClass, methodId, loc, true /* isArray */);
}

// Returns the TId for the passed Object, 0 if the Object is not tagged.
extern "C" jlong JNICALL GetObjTIdHandler(JNIEnv *env, jclass self, jobject obj)
{

	// Gain access to JVMTI and ObjectInfoManager
	static CDataManager *s_pDataManager = s_pEM->GetDM();
	static IJVM *s_pJVMInterface = s_pEM->GetJVMInterface();
	static CObjectInfoManager *s_pObjectInfoManager = s_pDataManager->GetObjectInfoManager();

	jlong objTag;
	TResult res = s_pJVMInterface->GetObjectTag(&objTag, obj);

	// The object is not tagged
	if(objTag == 0)
		return (jlong)0;
	// Obtain the TId from the Object
	TId objTId = s_pObjectInfoManager->GetTagInfo(objTag)->jobjectInfo.objectId;
	return (jlong)objTId;
}

// 
// Callback function for handling thread interactions called by instrumentation (through the ThreadProxy Java
// class)
//
extern "C" void JNICALL ThreadInteractionHandler(JNIEnv *pJniEnv, jclass self, jobject obj, jint methodID)
{
    if (s_pEM->IsVMShutdownReceived())
    {
        return;
    }

    static CDataManager *s_pDataManager = s_pEM->GetDM();
	static CObjectInfoManager *s_pObjectInfoManager = s_pDataManager->GetObjectInfoManager();

    // Tag the object
    jlong tag;
    TId objectId;
    if (MRTE_FAILED(s_pObjectInfoManager->GetOrCreateTag(&objectId, &tag, obj,
        pJniEnv, true /* bSaveInDb */)))
    {
        // Cannot tag the object. Do not report the event
        MARTINI_INFORMATIVE("InternalObjectAllocHandler", 0, false, 
            "failed to allocate object id");
        return;
    }

    // Notify the event
    SMpiInternalData mpiData;
    mpiData.pObjectTag = s_pObjectInfoManager->GetTagInfo(tag);
    mpiData.interactionType = methodID;

    SEmData eventData;
    eventData.dataType = MPI_INTERNAL;
    eventData.pJniEnv = pJniEnv;
    eventData.u.pMpiData = &mpiData;
    s_pEM->NotifyMpiEvent(Martini::MPI::EV_THREAD_INTERACTION, &eventData);
}


