/*****************************************************************************
 * Copyright (c) 1997, 2010 Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/

#ifndef _MRTE_EVENT_MANAGER_H
#define _MRTE_EVENT_MANAGER_H

#include "MpiAPI.h"
#include "ParamChecker.h"
#include "JavaInstrumentorManager.h"
#include "JpiEvent.h"
#include "MList.h"
#include "MVector.h"
#include "IEventManager.h"
#include "ECAgent.h"
#include "CGAdaptor.h"
#include "ThreadInfoManager.h"
#include "EventDispatcher.h"

namespace Martini { namespace JPI
{
    /**
     * @brief Event Descriptor
     * 
     * Stores information (e.g., name) about an event.
     * Used for validation and error reporting.
     */
    class CEventDescriptor
    {
    public:
        CEventDescriptor(unsigned int type, 
                         const char *szName, 
                         MPI::EEventGroup group = MPI::EG_NONE) 
            : m_type(type), m_szName(szName), m_group(group) {}
        unsigned int Type() { return m_type; }
        const char* Name() { return m_szName; }
        const MPI::EEventGroup Group() { return m_group; }
    private:
        unsigned int m_type;
        const char *m_szName;
        MPI::EEventGroup m_group;
    };

    class CJPIKernel;
    class CDataManager;
    class CJavaInstrumentorManager;
    class CJpiEvent;

    /**
     * @class CEMShutdownEvent
     * 
     * Internal event handler for the VM_SHUTDOWN event. 
     * This handler will signal Event Manager to stop reporting events to clients,
     * so that clients will not receive any events after VM_SHUTDOWN
     * (TPTP Bugzilla 143437 - https://bugs.eclipse.org/bugs/show_bug.cgi?id=143437)
     */
    class CEMShutdownEventObserver : public IInternalEventObserver
    {
    public:
        CEMShutdownEventObserver(CEventManager *pEventManager)
        {
            m_pEventManager = pEventManager;
        }

        virtual MPI::TEventType Type() { return MPI::EV_VM_SHUTDOWN; }

        virtual Infrastructure::EEventPriority GetPriority() 
        {
            // Since the event handler instructs Event Manager to disable all event 
            // notifications, it should be called last, after all other event handlers
            // were executed
            return Infrastructure::EVENT_PRIORITY_LOW;
        }

        void HandleEvent(SEmData *pEvent, void *pUserData);    

    private:
        CEventManager *m_pEventManager;
    };

    /**
     * @brief Event Manager
     */
    class CEventManager : public Infrastructure::IEventManager
    {
    public:

        /**
         * @brief Information for events managed by the Event Manager
         */
        struct SEmEventInfo
        {
            CEventDispatcher *pDispatcher;  //!< Event dispatcher
            CEventDescriptor *pDescriptor;  //!< Event descriptor
            bool bIsEnabledByClients;       //!< Whether this event is enabled by at least one 
                                            //!  client
            unsigned int uiNumClients;      //!< Number of clients registered for the event
            unsigned int implementor;       //!< Event implementor (JVM, Instrumentation, ...)
        };

        /**
         * @brief Information for event groups managed by the Event Manager
         */
        struct SEmEventGroupInfo
        {
            MPI::IEventFilter *pFilter;     //!< Filter assigned to the group
        };

        TResult RegisterEvent(MPI::TId clientId, 
                              MPI::IEventObserver& observer);
        
        /**
         * @brief Event registration for internal JPI clients
         * 
         * This function is called by JPI internal modules that want to register to a JVMPI
         * or JVMTI event. When the event occurs the jpiEvent.HandleEvent callback function
         * will be called with the JVMPI event data. The event data is shipped as-is and
         * the client is responsible for the interpretation
         *
         * @param[in] moduleId      Id of the requesting JPI module
         * @param[in] jpiEvent      Event observer
         *
         * @retval MRTE_RESULT_OK               Event successfully registered
         * @retval MRTE_ERROR_FAIL              Event registration failed
         * @retval MRTE_ERROR_NOT_SUPPORTED     Event is not supported by the JVM
         * @retval MRTE_ERROR_ILLEGAL_ID        Illegal client id
         */
        TResult RegisterEventInternal(MPI::TId moduleId, IInternalEventObserver& jpiEvent);

        virtual TResult SetEventGroupFilter(MPI::TId clientID, 
                                            MPI::EEventGroup group,
                                            MPI::IEventFilter &filter);

        virtual TResult EnableEventGroup(MPI::TId clientId, 
                                         MPI::EEventGroup eventGroup);

        virtual TResult DisableEventGroup(MPI::TId clientId, 
                                          MPI::EEventGroup eventGroup);
      
        TResult SetInternalModules(CDataManager *pDataManager,
            Infrastructure::CParamChecker *pParamChecker, CJavaInstrumentorManager *pJIM,
            Infrastructure::CECAgent *pECAgent, CJPIKernel *pKernel);

        void NotifyMpiEvent(MPI::TEventType mpiEvent, SEmData *pData);

        void NotifyECEvent(MPI::TEventType event, 
                           char *szData, 
                           unsigned int commandId,
                           void *pData);


        void NewMethodEvent(MPI::TId methodId, const JNIEnv *pJniEnv);
        void GenMethodEnterEvent(JNIEnv *env, jobject self, bool bIsAlreadyInvoked, jint id);
        void GenMethodLeaveEvent(JNIEnv *env, jobject self, jint id);
        TResult GenObjectAllocEvent(JNIEnv *pJNIEnv, jobject obj,
        	jlong pTag, jint methodId, jlong loc);
        
        unsigned int GetMethodEnterDataType() {return m_methodEnterDataRequestType;};
        unsigned int GetMethodLeaveDataType() {return m_methodLeaveDataRequestType;};
        unsigned int GetThreadInteractionDataType() {return m_threadInteractionDataRequestType;};
        CDataManager *GetDM() {return m_pDataManager;};
        IJVM* GetJVMInterface() { return m_pJavaInterface; }
        void InitializationPhaseCompleted(){m_bIsInitialiazationCompleted = true;};

        void VMShutdownReceived() { m_bVMShutdownReceived = true; }

        bool IsVMShutdownReceived() { return m_bVMShutdownReceived; }

        CEventManager(MPI::TId moduleId);
        
        ~CEventManager(void);

        static CEventManager *GetInstance();

        void SetEventImplementor(MPI::TEventType mpiEvent, 
                                 Infrastructure::EEventImplementationType implementor);
        
        bool UpdateTentativeInstrumentation(bool bIsCapableToInstrumentAllClasses);
        
        TResult DisableEventInternal(MPI::TEventType mpiEvent);

        MPI::IEventFilter* GetEventGroupFilter(MPI::EEventGroup group);

    private:

        // Default constructor is disabled (private). Use CEventManager(TId) to construct
        CEventManager(void);

        TResult RemoveEventInformation(MPI::IEventObserver& observer);
        TResult UpdateEventInformation(MPI::TId clientId, 
                                       MPI::IEventObserver& observer, 
                                       Infrastructure::EEventPriority priority, 
                                       bool bIsInternal,
                                       bool bIsEnabled);
        unsigned int GetEventImplementationType(MPI::TEventType mpiEvent);
        TResult InformInternalClients(MPI::TId clientId, MPI::IEventObserver& observer);

        void GetRegisteredEventsForClient(TObserverList *pEventsList, MPI::TId clientId);

        void SetEventGroupStatusForClient(MPI::TId clientId, 
                                          MPI::EEventGroup eventGroup, 
                                          bool bIsEnabled);

        bool GetEventGroupStatusForClient(MPI::TId clientId, MPI::EEventGroup eventGroup);
        
        void UpdateEventGlobalEnabledStatus(MPI::TEventType eventType);
        
        bool GetEventGlobalEnabledStatus(MPI::TEventType eventType);

        bool GetEventsGlobalEnabledStatus(MPI::EEventGroup eventGroup);

        bool EventExists(MPI::TEventType eventType);

        bool IsCallGraphEvent(MPI::TEventType eventType);

        bool IsClientRegisteredForEventInGroup(MPI::TId clientId, MPI::EEventGroup group);
        
        TResult SetEventGroupEnabled(MPI::TId clientId, 
                                     MPI::EEventGroup eventGroup, 
                                     bool bIsEnabled);
        
        void InitEventDescriptors();

        CEventDispatcher* CreateEventDispatcher(MPI::TEventType mpiEvent);
        
        // module id
        MPI::TId m_ModuleId;

        // Event information array. Indexed by event type
        SEmEventInfo m_vEventInfo[EM_MAX_EVENTS_TOTAL];

        // Event Group information array. Indexed by event group type
        SEmEventGroupInfo m_vEventGroupInfo[MPI::EG_LAST];

        // a pointer to the JVMPI/JVMTI interface
        IJVM *m_pJavaInterface;

        // a pointer to the data manager
        CDataManager *m_pDataManager;

        // a pointer to the param checker
        Infrastructure::CParamChecker *m_pParamChecker;

        // a pointer to the java instrumentor manager
        CJavaInstrumentorManager *m_pJavaInstrumentorManager;

        // a pointer to the kernel
        CJPIKernel *m_pKernel;

        // a pointer to the external control
        Infrastructure::CECAgent *m_pECAgent;

        unsigned int m_newMethodDataRequestType;
        unsigned int m_methodEnterDataRequestType;
        unsigned int m_methodLeaveDataRequestType;
        unsigned int m_threadInteractionDataRequestType;

        // flag indicating if profilers intialization phase is complete
        bool m_bIsInitialiazationCompleted;

        // flag indicating whether the VM has terminated (VM_SHUTDOWN event received)
        bool m_bVMShutdownReceived;

        // critical section for synchronizing calls to EnableEvent and DisableEvent
        OSA::IThreadSync *m_csEnableDisableEvent;

        // Event handler for VM_SHUTDOWN event
        CEMShutdownEventObserver *m_pVMShutdownEvent;

/*
        // An array of Event Descriptors. Used for validating event data
        RTUtil::MVector<CEventDescriptor*> m_vecEventDescriptors;
*/
    };

}}

extern "C" void JNICALL MethodEnterHandler(JNIEnv *env, 
                                           jobject self, 
                                           bool bIsFirst, 
                                           jint id);

extern "C" void JNICALL MethodLeaveHandler(JNIEnv *env, jobject self, jint);

extern "C" void JNICALL MethodLeaveWithException(JNIEnv *env, jobject self, jint);

extern "C" void JNICALL ObjectAllocHandler(JNIEnv *env, jclass self, 
                                           jobject obj, 
                                           jclass objClass, 
                                           jint methodId, 
                                           jlong loc);

extern "C" void JNICALL ArrayAllocHandler(JNIEnv *env, jclass self, 
                                          jobject obj, 
                                          jclass objClass, 
                                          jint methodId, 
                                          jlong loc);

extern "C" jlong JNICALL GetObjTIdHandler(JNIEnv *env, jclass self, jobject obj);

extern "C" void JNICALL ThreadInteractionHandler(JNIEnv *env, jclass self,
                                                 jobject obj, jint methodID);

#endif
