/*****************************************************************************
 * Copyright (c) 1997-2007, Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/

#include "IJVM.h"
#include "LogAssert.h"
#include "JpiGlobals.h"

using namespace Martini::JPI;
using namespace Martini::Infrastructure;

//////////////////////////////////////////////////////////////////////
// IJVM implementation

IJVM::IJVM()
: m_pCapManager(NULL), m_ptlsJniEnv(NULL)
{
    m_pCapManager = new CCapabilityManager();

    m_ptlsJniEnv = OSA::CreateThreadLocalStorage();
    if (!m_ptlsJniEnv)
    {
        MARTINI_ERROR("CThreadInfoManager", "Error creating thread local storage");
    }
}

IJVM::~IJVM()
{
    if (m_pCapManager)
    {
        delete m_pCapManager;
    }
    
    if (m_ptlsJniEnv)
    {
        m_ptlsJniEnv->Destroy();
    }
}

//
// Attaches the current thread to the JVM
//
// Parameters:
//      ppJNIEnv   [out]    : the JNI environment of the attached thread, or NULL if the 
//                            operation failed.
//      bAttached  [out]    : 'true' if the thread was attached.
//                            'false' if the thread was already attached
//
// Returns
//      MRTE_RESULT_OK  : success
//      MRTE_ERROR_FAIL : error
//
TResult IJVM::AttachCurrentThread(JNIEnv **ppJNIEnv, bool *bAttached)
{
    *ppJNIEnv = NULL;
    *bAttached = false;

    // Check if the current thread is already associated with a JNI environment
    *ppJNIEnv = (JNIEnv*)m_ptlsJniEnv->GetValue();
    if (0 != *ppJNIEnv)
    {
        return MRTE_RESULT_OK;
    }

    jint iVMErr;
    iVMErr = m_pJVM->GetEnv((void **)ppJNIEnv, JNI_VERSION_1_4);
    if (0 != iVMErr)
    {
        if (JNI_EDETACHED == iVMErr)
        {
            // Attach the current thread
            iVMErr = m_pJVM->AttachCurrentThread((void**)ppJNIEnv, NULL);
            if (0 != iVMErr)
            {
                MARTINI_INFORMATIVE1("IJVM", 0, false, 
                    "failed to attach to VM. Error: %d\n", iVMErr);
                return MRTE_ERROR_FAIL;
            }
            *bAttached = true;

            CDataManager *pDM = CJpiGlobals::Instance()->pDataManager;
            if (pDM)
            {
                pDM->GetThreadInfoManager()->SetThreadType(*ppJNIEnv, 
                    STlsThreadInfo::EXTERNAL);
            }
        }
        else
        {
            MARTINI_ERROR1("IJVM", "AttachCurrentThread: failed to get "
                "JNI_VERSION_1_4 environment. Error: %d\n", iVMErr);
            return MRTE_ERROR_FAIL;
        }
    }

    // Store the new JNI environment in the TLS
    m_ptlsJniEnv->SetValue(*ppJNIEnv);

    return MRTE_RESULT_OK;
}

void IJVM::DetachCurrentThread()
{
    m_pJVM->DetachCurrentThread();
    m_ptlsJniEnv->SetValue(0);
}

jobject IJVM::AllocateJavaThread()
{
    bool bAttached = false;
    JNIEnv *pJniEnv = NULL;
    TResult res = AttachCurrentThread(&pJniEnv, &bAttached);
    if (MRTE_FAILED(res))
    {
        return NULL;
    }

    jclass thrClass = pJniEnv->FindClass("java/lang/Thread");
    if (NULL == thrClass) 
    {
        MARTINI_INFORMATIVE("IJVM", 0, false, 
            "[AllocateJavaThread] cannot load class java.lang.Thread");
        return NULL;
    }
    
    jmethodID ctor = pJniEnv->GetMethodID(thrClass, "<init>", "()V");
    if (NULL == ctor)
    {
        MARTINI_INFORMATIVE("IJVM", 0, false, 
            "[AllocateJavaThread] cannot bind default constructor of java.lang.Thread");
        return NULL;
    }

    jobject newThread = pJniEnv->NewObject(thrClass, ctor);
    return newThread;
}

//////////////////////////////////////////////////////////////////////
// CapabilityManager Implementation

void CCapabilityManager::Initialize(ICapabilityFactory *pCapFactory)
{
    m_capabilities.Reserve(CAP_COUNT);
    m_capabilities.SetAt(CT_CAN_ENABLE_OR_DISABLE_ALL_EVENTS, 
        pCapFactory->CreateCapability(CT_CAN_ENABLE_OR_DISABLE_ALL_EVENTS));
    m_capabilities.SetAt(CT_CAN_ENABLE_OR_DISABLE_BCI_GENERATED_EVENTS, 
        pCapFactory->CreateCapability(CT_CAN_ENABLE_OR_DISABLE_BCI_GENERATED_EVENTS));
    m_capabilities.SetAt(CT_CAN_REGISTER_MULTIPLE_CALLGRAPH_EVENT_CLIENTS, 
        pCapFactory->CreateCapability(CT_CAN_REGISTER_MULTIPLE_CALLGRAPH_EVENT_CLIENTS));
    m_capabilities.SetAt(CT_CAN_GET_THREAD_TIMES, 
        pCapFactory->CreateCapability(CT_CAN_GET_THREAD_TIMES));
    m_capabilities.SetAt(CT_CAN_SUSPEND_RESUME_VM, 
        pCapFactory->CreateCapability(CT_CAN_SUSPEND_RESUME_VM));
    m_capabilities.SetAt(CT_CAN_GENERATE_OBJECT_ALLOC_EVENTS, 
        pCapFactory->CreateCapability(CT_CAN_GENERATE_OBJECT_ALLOC_EVENTS));
}

TResult CCapabilityManager::Enable(ECapabilityType cap)
{
    ICapability *pCap = m_capabilities.GetAt(cap);
    return pCap->Enable();
}

bool CCapabilityManager::Enabled(ECapabilityType cap)
{
    ICapability *pCap = m_capabilities.GetAt(cap);
    return pCap->Enabled();
}

bool CCapabilityManager::Supported(ECapabilityType cap)
{
    ICapability *pCap = m_capabilities.GetAt(cap);
    return pCap->Supported();
}

TResult CCapabilityManager::Disable(ECapabilityType cap)
{
    ICapability *pCap = m_capabilities.GetAt(cap);
    return pCap->Disable();
}

