/*****************************************************************************
 * Copyright (c) 1997, 2010 Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/

#ifndef MRTE_IJVM_H
#define MRTE_IJVM_H

#include "MpiAPI.h"
#include "MRTETypes.h"
#include "OSA.h"
#include "MVector.h"
#include "MString.h"

#include <jni.h>
#include "jvmti.h"

namespace Martini { namespace JPI
{

    struct SJVMMethodInfo
    {
        jmethodID methodId;
        const char *szMethodName;
        const char *szMethodSignature;
        const char *szMethodGeneric;
        unsigned int uiAccessFlags;
        MPI::SMethodLineNumbers lineNumbers;
    };

    enum JVMDataType
    {
        DT_METHOD_LOAD,
        DT_METHOD_UNLOAD,
        DT_CLASS_LOAD,          // JVMTI Class Load event (not supported by JVMPI)
        DT_CLASS_PREPARE,       // JVMTI Class Prepare event / JVMPI Class Load event
        DT_CLASS_LOAD_HOOK,
        DT_METHOD_ENTER,
        DT_METHOD_LEAVE,
        DT_THREAD_START,
        DT_THREAD_END,
        DT_DYNAMIC_CODE,
        DT_VM,
        DT_OBJECT_FREE,
        DT_MONITOR_WAIT,
        DT_MONITOR_WAITED,
        DT_CONTENDED_MONITOR
    };

    enum EInterfaceType
    {
        IT_JVMPI,
        IT_JVMTI
    };
    
    /**
     * @brief Defines properties to be stored in JVMTI object tags
     */
    struct SJvmtiObjectTag
    {
        SJvmtiObjectTag(U32 i_uiStartAge) 
            : bNotify(false), uiStartAge(i_uiStartAge) //, bDeleted(false)
        {
        }

        // Information relevant for all tagged objects (including class objects)
        struct _JObjectInfo {
            _JObjectInfo() : objectId(0), szClassName(NULL), pObjectInfo(NULL) {}

            MPI::TId objectId;              //!< Object MPI id
            const char *szClassName;        //!< Object's class name (friendly name)
                                            //!  This field is required to implement selectivity
                                            //!  and can be removed once selectivity is implemented
                                            //!  as part of the Heap Adaptor
            MPI::SObjectInfo *pObjectInfo;  //!< MPI Object information
        } jobjectInfo;

        // Information relevant only for class objects (jclass)
        struct _JClassInfo {
            _JClassInfo() : classId(0), arrayType(MPI::AT_NONE), 
                szClassName(NULL) {}

            MPI::TId classId;               //!< Class MPI id
            MPI::EArrayType arrayType;      //!< Class array type
            const char *szClassName;        //!< Class friendly name
        } jclassInfo;

        bool bNotify;                   //!< True: object evens should be sent to clients
                                        //!  False: object is tagged for internal purposes

        U32 uiStartAge;

    }; 

    struct SJVMData
    {
        JVMDataType dataType;

        union
        {
            struct 
            {
                jmethodID methodId;
                unsigned int uiCodeSize;
                const void*  pCodeAddr;
                MPI::SLineNumberTable nativeToSrcLineMap;
                MPI::SLineNumberTable nativeToManagedLineMap;
                MPI::SLineNumberTable managedToSrcLineMap;
            } methodLoad;
            
            struct 
            {
                jmethodID methodId;
            } method;

            // JVMTI Class Prepare event / JVMPI Class Load event
            struct 
            {
                jclass jniClassLocalRef;    // jclass local reference.
                                            // Valid only for the duration of the
                                            // ClassLoad/Prepare event. If you wish to keep
                                            // this value, convert it to a global JNI reference
                                            // using JNI's NewGlobalRef API
                const char *szClassName;
                const char *szSourceName;
                const char *szGenericSig;
                unsigned int uiNumMethods;
                SJVMMethodInfo *pMethodInfo;
                jobject jniLoaderLocalRef;  // local reference to the class loader (JVMTI-only).
                                            // Valid only for the duration of the
                                            // ClassLoad/Prepare event. If you wish to keep
                                            // this value, convert it to a global JNI reference
                                            // using JNI's NewGlobalRef API
            } classPrepare;

            // JVMTI Class Load event data (not supported by JVMPI)
            struct 
            {
                jclass jniClassLocalRef;    // jclass local reference.
                                            // Valid only for the duration of the
                                            // ClassLoad event. If you wish to keep
                                            // this value, convert it to a global JNI reference
                                            // using JNI's NewGlobalRef API
                const char *szClassName;
            } classLoad;

            struct 
            {
                const char* szClassName;
                jint classDataLen;
                const unsigned char* pucClassData;
                jint* pNewClassDataLen;
                unsigned char** ppucNewClassData;
                TMemoryAllocatorFunc pfnAllocate;
                jclass classBeingRedefined;
                jobject loader;
            } classLoadHook;
            
            struct 
            {
                const JNIEnv *pThreadEnv;   // JNI environment of the starting thread.
                jthread threadLocalRef;     // Local JNI reference of the starting thread
                const char *szName;
                const char *szGroupName;
                const char *szParentGroupName;
            } threadStart;

            struct 
            {
                const JNIEnv *pThreadEnv;   // JNI environment of the dying thread
                jthread threadLocalRef;     // Local JNI reference of the ending thread
            } threadEnd;

            struct {
                const char* name;
                const void* address;
                jint length;
            } dynamicCode;

            struct
            {
                JNIEnv *pJniEnv;
            } vmEvent;

            // JVMTI Object Free event
            struct  
            {
                jlong tag;
            } objectFree;

            // JVMTI VM Object Alloc event
            struct  
            {
                SJvmtiObjectTag *pTagInfo;  // Object's tag
                jobject objectJniLocalRef;  // Local reference of the object. Valid only in 
                                            // the context of the VMObjectAllocHandler JNI
                                            // Callback (!!!)
                jlong size;                 // Object's size
            } vmObjectAlloc;

            // JVMTI Monitor Wait event
            struct  
            {
                jthread threadLocalRef;     // Local reference of the waiting thread.
                                            // Valid only in the context of the 
                                            // MonitorWaitHandler JNI Callback (!!!)
                jobject monitorLocalRef;    // Local reference of the monitor.
                                            // Valid only in the context of the 
                                            // MonitorWaitHandler JNI Callback (!!!)
                jlong timeout;              // Wait timeout
            } monitorWait;

            // JVMTI Monitor Waited event
            struct  
            {
                jthread threadLocalRef;     // Local reference of the waited thread.
                                            // Valid only in the context of the 
                                            // MonitorWaitedHandler JNI Callback (!!!)
                jobject monitorLocalRef;    // Local reference of the monitor.
                                            // Valid only in the context of the 
                                            // MonitorWaitHandler JNI Callback (!!!)
                jlong timeout;              // Wait timeout
                bool bTimedOut;             // Whether the wait() operation timed out
            } monitorWaited;

            // JVMTI Contended Monitor Enter/Entered events
            struct  
            {
                jthread threadLocalRef;     // Local reference of the enter/entered thread.
                                            // Valid only in the context of the 
                                            // ContendedMonitor JNI Callback (!!!)
                jobject monitorLocalRef;    // Local reference of the monitor.
                                            // Valid only in the context of the 
                                            // MonitorWaitHandler JNI Callback (!!!)
                jlong timeout;              // Wait timeout
            } contendedMonitor;
        } u;
    };

    // definitions of access flags used in JPI
    const unsigned int ACC_VARARGS = 0x0080;

	/**
	 * Capability types
	 */
    enum ECapabilityType
	{
		CT_CAN_ENABLE_OR_DISABLE_BCI_GENERATED_EVENTS,
		CT_CAN_ENABLE_OR_DISABLE_ALL_EVENTS,
		CT_CAN_REGISTER_MULTIPLE_CALLGRAPH_EVENT_CLIENTS,
        CT_CAN_GET_THREAD_TIMES,
        CT_CAN_SUSPEND_RESUME_VM,
        CT_CAN_GENERATE_OBJECT_ALLOC_EVENTS
	};
    const U16 CAP_COUNT = 6;

    /**
     * @class ICapability
     * 
     * @brief Represents a capability
     */
    class ICapability
    {
    public:
        /**
		 * @brief <b> Enable </b> Enables the capability
		 * 
         * @returns MRTE_RESULT_OK           : capability successfully enabled
         * @returns MRTE_ERROR_NOT_SUPPORTED : capability not supported
		 */
		virtual TResult Enable() = 0;

		/**
		 * @brief <b> Enabled </b> Checks whether the capability is enabled
         *
         * @returns true  : the capability is enabled
         * @returns false : the capability is not enabled
		 */
		virtual bool Enabled() = 0;

		/**
		 * @brief <b> Supported </b> Checks whether the capability is supported
		 * 
         * @returns true  : the capability is supported
         * @returns false : the capability is not supported
		 */
        virtual bool Supported() = 0;

        /**
		 * @brief <b> Disable </b> Disables the capability
		 * 
         * @returns MRTE_RESULT_OK           : capability successfully disabled
         * @returns MRTE_ERROR_NOT_SUPPORTED : capability not supported
		 */
		virtual TResult Disable() = 0;
    };

    /**
     * @class CGenericUnsupportedCapability
     * 
     * @brief Represents an unsupported capability
     */
    class CGenericUnsupportedCapability : public ICapability
    {
    public:
        virtual TResult Enable() { return MRTE_ERROR_NOT_SUPPORTED; }
        virtual bool Enabled() { return false; }
        virtual bool Supported() { return false; }
        virtual TResult Disable() { return MRTE_ERROR_NOT_SUPPORTED; }
    };

    /**
     * @class CGenericSupportedCapability
     * 
     * @brief Represents a capability that is implicitly supported by the VM
     */
    class CGenericSupportedCapability : public ICapability
    {
    public:
        virtual TResult Enable() { return MRTE_RESULT_OK; }
        virtual bool Enabled() { return true; }
        virtual bool Supported() { return true; }
        virtual TResult Disable() { return MRTE_ERROR_NOT_SUPPORTED; }
    };

    /**
     * @class CapabilityFactory
     * 
     * @brief Creates capability objects
     */
    class ICapabilityFactory
    {
    public:
        virtual ICapability* CreateCapability(ECapabilityType cap) = 0;
    };

    /**
	 * @class CCapabilityManager
	 *
	 * @brief Manages capabilities of the VM interface
	 */
	class CCapabilityManager
	{
	public:
        CCapabilityManager() {}
        ~CCapabilityManager() {}

        /**
         * @brief <b> Initialize </b> Initializes the Capability Manager with the capabilities
         * of the concrete platform
         * 
         * @param pCapFactory  [in]: a factory for creating Capability objects
         */
        void Initialize(ICapabilityFactory *pCapFactory);

        /**
		 * @brief <b> Enable </b> Enables a capability
		 * 
		 * @param cap  [in]: the capability to enable
         *
         * @returns MRTE_RESULT_OK           : capability successfully enabled
         * @returns MRTE_ERROR_NOT_SUPPORTED : capability not supported
		 */
		TResult Enable(ECapabilityType cap);


		/**
		 * @brief <b> Enabled </b> Checks whether a capability is enabled
		 * 
		 * @param cap  [in]: the capability to check
         *
         * @returns true  : the capability is enabled
         * @returns false : the capability is not enabled
		 */
		bool Enabled(ECapabilityType cap);

		/**
		 * @brief <b> Supported </b> Checks whether a capability is supported
		 * 
		 * @param cap  [in]: the capability to check
         *
         * @returns true  : the capability is supported
         * @returns false : the capability is not supported
		 */
        bool Supported(ECapabilityType cap);

        /**
		 * @brief <b> Disable </b> Disables a capability
		 * 
		 * @param cap  [in]: the capability to disable
         *
         * @returns MRTE_RESULT_OK           : capability successfully disabled
         * @returns MRTE_ERROR_NOT_SUPPORTED : capability not supported
		 */
		TResult Disable(ECapabilityType cap);

    private:
        RTUtil::MVector<ICapability*> m_capabilities;
	};

    /**
     * @brief A Java Raw Monitor
     *
     * Defines an abstract interface for a Java Raw Monitor.
     */
    class CRawMonitor
    {
    public:
        CRawMonitor(const char *szName)
            : m_szName(szName)
        {
        }

        virtual ~CRawMonitor() {}
        
        virtual TResult Enter() = 0;
        virtual TResult Exit() = 0;
        virtual TResult Wait(jlong timeoutMillis) = 0;
        virtual TResult Notify() = 0;
        virtual TResult NotifyAll() = 0;

    private:
        const char * m_szName;
    };

    typedef RTUtil::MVector<jlong> TTagVector;

    typedef RTUtil::MVector<MPI::SObjectInfo*> TObjectInfoVector;

    typedef void (JNICALL *TAgentThreadStartFunction)
        (JNIEnv* pJniEnv, void* arg);

    struct TAgentThreadStartFunctionArgs
    {
        TAgentThreadStartFunctionArgs(TAgentThreadStartFunction _proc, void *_pArgs)
            : entryPoint(_proc), pArgs(_pArgs) {}
        
        TAgentThreadStartFunction entryPoint;
        void* pArgs;
    };

    typedef unsigned int IterHeapOps;
    enum ITERATE_HEAP_OPS{
        ITER_HEAP_NO_OPS = 0,
        ITER_HEAP_BUFF_OBJ = 1,
        ITER_HEAP_GEN_ALLOC_EVENT = 2
    };
    
    // Used to solve redefine performance problem, refer bug 291400
    typedef enum {
        UNDEFINED_JVM,
        IBM_JVM,
        OTHER_JVM
    } JVMVendor;    

	/**
	 * @class IJVM
	 *
	 * JVM Interface
	 */
    class IJVM  
    {
    public:
        IJVM();
        virtual ~IJVM();

        // Interface functions (abstract)
        virtual TResult Initialize(MPI::TId moduleId, JavaVM *pJVM) = 0;
        virtual void ProfilerExit() = 0;
        virtual TResult RegisterEvent(MPI::TEventType mpiEvent) = 0;
        virtual TResult RequestClassLoadEvent(UIOP classId) = 0;
        virtual bool CanSupplyClassLoadHookToAllClasses() = 0;
        virtual TResult DisableEvent(MPI::TEventType mpiEvent) = 0;
        virtual TResult EnableEvent(MPI::TEventType mpiEvent) = 0;
        virtual TResult GetJNIEnv(JNIEnv **ppJNIEnv) = 0;
        virtual EInterfaceType GetInterfaceType() = 0;
        virtual TResult RedefineClasses(jint classCount, 
                                        const jvmtiClassDefinition* classDefinitions) = 0;
        virtual TMemoryAllocatorFunc GetMemoryAllocator() = 0;
        virtual jclass FindClass(const char *szClassName) = 0;
        virtual TResult GetThreadInfo(MPI::SThreadInfo *pThreadInfo, jthread thread) = 0;
        virtual TResult SuspendThread(jthread thread) = 0;
        virtual TResult ResumeThread(jthread thread) = 0;
        virtual TResult GetCurrentThreadCpuTime(MPI::TTimeStamp *pCpu) = 0;
        virtual TResult SetClassTag(jclass cls) = 0;
        virtual TResult SetObjectTag(jobject obj, jlong tag) = 0;
        virtual TResult GetObjectTag(jlong *pTag, jobject obj) = 0;
        virtual TResult GetObjectInfo(MPI::SObjectInfo *pObjectInfo, jobject obj) = 0;
        virtual TResult GetObjectInfoFromTags(TTagVector &inTagVector, 
                                              TObjectInfoVector *pOutInfoVector,
                                              TTagVector *pOutTagVector) = 0;
        virtual TResult GetObjectClassName(RTUtil::MCString &className, jobject obj) = 0;
        virtual TResult GetObjectReferences(MPI::SObjectReferenceInfo *pData, 
                                            TTagVector *pTagVector) = 0;
        virtual TResult GetMonitorUsage(MPI::TId *pOwnerThreadId,
                                        MPI::TIdArray *pOwnWaiters,
                                        MPI::TIdArray *pNotifyWaiters,
                                        MPI::BitSet dataItems, 
                                        jobject monitorJniRef) = 0;
        virtual TResult GetStackTrace(jvmtiFrameInfo *pFrameBuffer, 
                                      jint *pFrameCount,
                                      jthread thread,
                                      jint startDepth,
                                      jint maxFrameCount) = 0;
        virtual TResult GetMethodInfo(RTUtil::MCString &name,
                                      RTUtil::MCString &sig,
                                      RTUtil::MCString &generic,
                                      jint *pAccessFlags,
                                      MPI::SMethodLineNumbers *pLineNumbers,
                                      jmethodID jvmMethodId) = 0;
        virtual TResult GetMethodDeclaringClassName(RTUtil::MCString &className,
                                                    MPI::TId *pClassLoaderObjectId,
                                                    jmethodID jvmMethodId) = 0;
        virtual TResult GetLineNumberTable(MPI::SLineNumberTable *pLineNumberTable,
                                           jmethodID jvmMethodId) = 0;
        virtual TResult GetThreadState(jint *pState, jthread thread) = 0;
        virtual TResult GetCurrentContendedMonitor(jobject *pMonObj, jthread thread) = 0;
        virtual TResult GetAllStackTraces(jvmtiStackInfo **pStackInfoArray, 
                                          jint *pThreadCount,
                                          jint maxFrameCount) = 0;
        virtual MPI::EThreadState VmThreadStateToMpiThreadState(jint vmThreadState) = 0;
        virtual TResult DeallocateVmBuffer(unsigned char *pBuffer) = 0;
        virtual TResult RunGC() = 0;
        virtual TResult IterateLiveHeap(IterHeapOps iterops) = 0;
        virtual bool CanRedefineClass(jclass cls) = 0;
        virtual TResult CreateRawMonitor(const char *szName, CRawMonitor **pNewMonitor) = 0;
        virtual TResult RunAgentThread(jobject thread, 
                                       TAgentThreadStartFunction proc, 
                                       void *arg) = 0;
        // Add for bug 291400
        virtual JVMVendor GetJVMVendor() {return UNDEFINED_JVM;}
   
        // Public functions
        
        CCapabilityManager* GetCapabilities() { return m_pCapManager; }
        TResult AttachCurrentThread(JNIEnv **ppJNIEnv, bool *bAttached);
        void DetachCurrentThread();
        jobject AllocateJavaThread();

        // Thread Local Storage for managing JNI environment for the current thread
        OSA::IThreadLocalStorage *m_ptlsJniEnv;
    
    protected:
        MPI::TId m_moduleId;   

        // Pointer to Capability Manager. Initialized by the Initialize method
        CCapabilityManager *m_pCapManager;

        // Pointer to the Java environment
        JavaVM *m_pJVM;

    };
}};

#endif // #ifndef MRTE_IJVM_H


