/*****************************************************************************
 * Copyright (c) 1997, 2010 Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/
#ifdef EM64T_ARCH
#pragma runtime_checks( "", off )
#endif

#include <memory.h>
#include <iostream>
#include "JPIKernel.h"
#include "JPI.H"
#include "LibraryLoader.h"
#include "MpiAPI.h" 
#include "OSA.h"
#include "JVMTIInterface.h"
#include "JVMPIInterface.h"
#include "JpiGlobals.h"
#include <stdio.h>

using namespace Martini::MPI;
using namespace Martini::JPI;
using namespace Martini::Infrastructure;
using namespace Martini::OSA;
using namespace Martini::RTUtil;

///////////////////////////////////////////////////////////////////////////////////////
// CJPIKernel implementation

CJPIKernel *CJPIKernel::pThis = NULL;

/**
 * @brief <b>GetJPIKernel</b> Returns the singleton instance of the JPI kernel
 * 
 * Get an instance of the singleton JPI kernel. If the kernel was not created yet, this
 * function creates it. If it was already allocated, its instance is returned
 *
 * @returns the single instance to the JPI kernel
 */
CJPIKernel *CJPIKernel::GetJPIKernel() 
{
    if (CJPIKernel::pThis == NULL)
    {
        CJPIKernel::pThis = new CJPIKernel();        
    }
    return CJPIKernel::pThis;
}

/**
 * @brief <b>InstantiateClient</b> Loads an MPI profiler
 *
 * This function is called by any client that wishes to instantiate another client.
 * The requested module is loaded only if it was not previously instantiated by 
 * another client or during initialization (i.e. specified in the command-line)
 *
 * @param clientId              [in] : id of the requesting client
 * @param szRequestedModuleName [in] : name of the client to instantiate
 * @param szOptions             [in] : client options
 *
 * @returns MRTE_RESULT_OK      : success
 * @returns MRTE_PHASE_FAILURE  : wrong phase. Profilers can be instantiated only during
 *                                initialization
 * @returns MRTE_RESULT_PRV_INSTANTIATED : requested profiler was already loaded
 * @returns MRTE_ERROR_FAIL     : general failure
 */
TResult CJPIKernel::InstantiateClient(TId clientId, 
                                      const char *szRequestedModuleName,
                                      const char *szOptions)
{
    TResult retVal = m_pParamChecker->CheckClientAndPtr(clientId, szRequestedModuleName);
    if (MRTE_FAILED(retVal))
    {
        return MRTE_ERROR_FAIL;
    }
    
    if (m_bInitPhaseCompleted)
    {
        return MRTE_ERROR_PHASE_FAILURE;
    }

    SClientInfo *pRequestedClientInfo = 0;
    pRequestedClientInfo = GetClientInfo(szRequestedModuleName);

    if(pRequestedClientInfo != 0)
    {
        return MRTE_RESULT_PRV_INSTANTIATED;
    }
    
    return LoadProfiler(szRequestedModuleName, szOptions);
}

//
// Constructor
//
CJPIKernel::CJPIKernel(void)
{
    m_pJVMInterface = NULL;
    m_pEventManager = NULL;
    m_pDataManager = NULL;
    m_pJavaInstrumentorManager = NULL;
    m_pParamChecker = NULL;
    m_pMpiInterface = NULL;
    m_pECAgent = NULL;
    m_ModuleId = 0;
    m_bInitPhaseCompleted = false;
 
    int i;
    for (i = 0; i < EM_MAX_CLIENTS; i++)
    {
        m_vClientInfo[i].szClientName.Set("");
    }
}

/**
 * @brief <b>Initialize<b> Initialize JPI
 *
 * This function is called by the boot loader in order to initialize JPI and pass it
 * the virtual machine interface and profiler options.
 * 
 * @param pVM               [in] : VM invocation interface
 * @param szOptions         [in] : command line options
 * @param jvmInterfaceType  [in] : JVM Interface type (JVMPI or JVMTI)
 * 
 * @returns MRTE_RESULT_OK              : success
 * @returns MRTE_ERROR_OUT_OF_MEMORY    : memory error
 * @returns MRTE_ERROR_FAIL             : general error
 */
TResult CJPIKernel::Initialize(JavaVM *pVM, 
                               char *szOptions, 
                               EJVMInterfaceType jvmInterfaceType)
{
    TId id;
    TResult retVal;

    m_nextId = 1;
    m_pLogAssert = CLogAssert::GetInstance();
    m_pLogAssert->Initialize(NULL, false);

    SModuleSpec ecSpec;
    TModuleSpecList lstProfilerSpecs(SModuleSpec("unknown", "unknown"));
    retVal = CBootLoaderOptionParser::Parse(&ecSpec, &lstProfilerSpecs, szOptions);
    if (MRTE_FAILED(retVal))
    {
        MARTINI_ERROR1("CJPIKernel", "illegal command line: '%s'", szOptions);
        return retVal;
    }

    m_pParamChecker = new CParamChecker();
    
    IdAllocate(&m_ModuleId);


    IdAllocate(&id);
    m_pEventManager = new CEventManager(id);

    IdAllocate(&id);
    m_pDataManager = new CDataManager(id);

    IdAllocate(&id);
    if (jvmInterfaceType == TYPE_JVMTI)
    {
        m_pJVMInterface = new CJVMTIInterface();
    }
    else
    {
        MARTINI_ASSERT("CEventManager", jvmInterfaceType == TYPE_JVMPI, "");
        m_pJVMInterface = new CJVMPIInterface();
    }
    
    IdAllocate(&id);
    m_pJavaInstrumentorManager = new CJavaInstrumentorManager(id);

    m_pECAgent = CECAgent::GetInstance();
    
    m_pMpiInterface = new CJpiInterface(RT_JAVA, m_pDataManager, m_pEventManager, this, 
        m_pECAgent, m_pLogAssert->GetLoggerInstance()); 


    if ((m_pJVMInterface == NULL)               ||
	    (m_pEventManager == NULL)               ||
		(m_pDataManager  == NULL)               ||
		(m_pJavaInstrumentorManager == NULL)    ||
	    (m_pMpiInterface == NULL)               ||
        (m_pECAgent == NULL)                    ||
	    (m_pParamChecker == NULL) )
    {
        MARTINI_ERROR("CJPIKernel", "unable to instantiate internal classes. Memory error");
        return MRTE_ERROR_OUT_OF_MEMORY;
    }

    retVal = m_pJVMInterface->Initialize(id, pVM);
    if (MRTE_FAILED(retVal))
    {
        MARTINI_ERROR("CJPIKernel", "failed to initialize the JVM interface");
        return retVal;
    }

    CJpiGlobals::Instance()->pJvmInterface = m_pJVMInterface;
    
    retVal = m_pEventManager->SetInternalModules(m_pDataManager, m_pParamChecker, 
        m_pJavaInstrumentorManager, m_pECAgent, this);
    if (MRTE_FAILED(retVal))
    {
        MARTINI_ERROR("CJPIKernel", "failed to initialize Event Manager");
        return retVal;
    }
    
    m_pDataManager->SetInternalModules(m_pEventManager, m_pParamChecker, this,
        m_pJavaInstrumentorManager);

    CJpiGlobals::Instance()->pDataManager = m_pDataManager;

    retVal = m_pJavaInstrumentorManager->Initialize(m_pDataManager, m_pEventManager, 
        m_pParamChecker);
    if (MRTE_FAILED(retVal))
    {
        MARTINI_ERROR("CJPIKernel", "failed to initialize Instrumentation Manager");
        return retVal;
    }
    
    IdAllocate(&id);
    retVal = m_pECAgent->Initialize(id, ecSpec.strName.Get(), m_pEventManager, 
        ecSpec.strOptions.Get());
    if (MRTE_FAILED(retVal))
    {
        MARTINI_ERROR2("CJPIKernel", "failed to initialize EC module: %s with "
            "options='%s'", ecSpec.strName.Get(), ecSpec.strOptions.Get());
        return retVal;
    }
    
    IdAllocate(&id);
    m_pEventManager->RegisterEvent(id, *m_pLogAssert);

    m_pLogAssert->Initialize(m_pECAgent, true);

    retVal = LoadProfilers(lstProfilerSpecs);
    if (MRTE_FAILED(retVal))
    {
        return retVal;
    }
    
    // Notify internal modules that initialization completed successfully
    m_pEventManager->InitializationPhaseCompleted();
    m_pECAgent->InitializationPhaseCompleted();
    
    retVal = m_pJavaInstrumentorManager->InitializationPhaseCompleted();
    if (MRTE_FAILED(retVal))
    {
        return retVal;
    }

    m_bInitPhaseCompleted = true;

    return retVal;
}  

/**
 * @brief <b>LoadProfilers</b> Load profilers
 * 
 * NOTE: the current version loads only the first profiler in the list. The rest are ignored.
 * 
 * @param lstProfilerSpec   [in] : list of profilers to load
 *
 * @returns MRTE_RESULT_OK  : success
 * @returns MRTE_ERROR_FAIL : failure
 */
TResult CJPIKernel::LoadProfilers(const TModuleSpecList &lstProfilerSpecs)
{
    if (lstProfilerSpecs.Count() == 0)
    {
        MARTINI_ERROR("CJPIKernel", "no profilers specified");
        return MRTE_ERROR_FAIL;
    }
    TMRTEHandle hProfSpec = lstProfilerSpecs.GetFirst();
    while (NULL != hProfSpec)
    {
        SModuleSpec profSpec = lstProfilerSpecs.GetData(hProfSpec);
        TResult iRetVal = LoadProfiler(profSpec.strName.Get(), profSpec.strOptions.Get());
        //TODO: add support for loading multiple profilers
        return iRetVal;
    }
    return MRTE_RESULT_OK;
}


/**
 * @brief <b>LoadProfiler</b> Loads and initializes an MPI profiler
 * 
 * @param szProfilerName    [in] : the name of the profiler module to load
 * @param szProfilerOptions [in] : profiler options. Passed to the profiler init function
 * 
 * @returns MRTE_RESULT_OK  : success
 * @returns MRTE_ERROR_FAIL : failure
 */
TResult CJPIKernel::LoadProfiler(const char *szProfilerName, const char *szProfilerOptions)
{
    TResult iRetVal = MRTE_RESULT_OK;
    TClientInstantiate initFunc;    
    ILibraryLoader *pLibrary = LoadBistroLibrary(szProfilerName);
    
    if (!pLibrary)
    {
        MARTINI_ERROR1("CJPIKernel", "unable to load profiler library: %s", szProfilerName);
        return MRTE_ERROR_FAIL;
    }
    
    initFunc = (TClientInstantiate) pLibrary->GetEntry(CLIENT_INSTANTIATE);
    if (initFunc)
    {
        TId id;
        IdAllocate(&id);
        iRetVal = initFunc((IMpi *)m_pMpiInterface, id, szProfilerOptions);
        if (MRTE_FAILED(iRetVal))
        {
            MARTINI_ERROR1("CJPIKernel", "Profiler \"%s\" failed to initialize", 
                szProfilerName);
            return iRetVal;
        }
        m_vClientInfo[id].szClientName.Set(szProfilerName); 
    }
    else 
    {
        MARTINI_ERROR("CJPIKernel", "unable to call the profiler initialize function");
        return MRTE_ERROR_FAIL;
    }
    pLibrary->Destroy();
    
    return iRetVal;
}

//
// Destructor
//
CJPIKernel::~CJPIKernel(void)
{
}

/**
 * @brief <b>IdAllocate</b> Allocate a unique id
 * 
 * @param pId  [out] : a placeholder for the newly allocated id. If the function fails, 
 *                     the content of the pointed variable is not modified.
 *
 * @returns MRTE_RESULT_OK  : success
 * @returns MRTE_ERROR_FAIL : failure
 */
TResult CJPIKernel::IdAllocate(TId *pId)
{
    if (m_nextId < EM_MAX_CLIENTS)
    {
        *pId = m_nextId++;
        
        m_pParamChecker->SetClientIdLimits(0, *pId);
        return MRTE_RESULT_OK;
    }
    return MRTE_ERROR_FAIL;
}

/**
 * @brief <b>GetClientInfo</b> Retrieve client information by id
 *
 * @param id    [in] : client id
 *
 * @returns SClientInfo*    : on success, returns the client information
 * @returns null            : on failure
 */
SClientInfo *CJPIKernel::GetClientInfo(TId id)
{
    TResult retVal = m_pParamChecker->CheckClient(id);
    if (MRTE_SUCCEEDED(retVal))
    {
        return &m_vClientInfo[id];
    }
    return NULL;
}

/**
 * @brief <b>GetClientInfo</b> Retrieve client information by name
 *
 * @param szClientName  [in] : client name
 *
 * @returns SClientInfo*    : on success, returns the client information
 * @returns null            : on failure
 */
SClientInfo *CJPIKernel::GetClientInfo(const char* szClientName)
{
    unsigned int i;

    for (i = 0; i < EM_MAX_CLIENTS; i++)
    {
        if (m_vClientInfo[i].szClientName.IsEqual(szClientName))
        {
            return &m_vClientInfo[i];
        }
    }

    return 0;
}

TResult CJPIKernel::SuspendVM()
{
    if (!m_bInitPhaseCompleted)
    {
        return MRTE_ERROR_PHASE_FAILURE;
    }

    if (!CJpiGlobals::Instance()->IsThreadMpiSafe())
    {
        return MRTE_ERROR_FAIL;
    }

	MARTINI_INFORMATIVE("JPIKernel", 5, false, "[SuspendVM] Suspend VM called");

    JNIEnv *pJniEnv;
    bool bAttached;
    if (MRTE_FAILED(m_pJVMInterface->AttachCurrentThread(&pJniEnv, &bAttached)))
    {
        return MRTE_ERROR_FAIL;
    }
    CThreadInfoManager *pThreadInfoManager = m_pDataManager->GetThreadInfoManager();
    TResult res = MRTE_ERROR_FAIL;
    if (pThreadInfoManager->GetThreadType(pJniEnv) == STlsThreadInfo::EXTERNAL)
    {
        res = pThreadInfoManager->SuspendAllThreads();
    }

    // Do not detach the thread from the VM. This will be done when the VM is resumed
    return res;
}

TResult CJPIKernel::ResumeVM()
{
    if (!m_bInitPhaseCompleted)
    {
        return MRTE_ERROR_PHASE_FAILURE;
    }

    if (!CJpiGlobals::Instance()->IsThreadMpiSafe())
    {
        return MRTE_ERROR_FAIL;
    }

	MARTINI_INFORMATIVE("JPIKernel", 5, false, "[ResumeVM] Resume VM called");

    JNIEnv *pJniEnv;
    if (MRTE_FAILED(m_pJVMInterface->GetJNIEnv(&pJniEnv)))
    {
        return MRTE_ERROR_FAIL;
    }
    CThreadInfoManager *pThreadInfoManager = m_pDataManager->GetThreadInfoManager();
    TResult res = MRTE_ERROR_FAIL;
    if (pThreadInfoManager->GetThreadType(pJniEnv) == STlsThreadInfo::EXTERNAL)
    {
        res = pThreadInfoManager->ResumeAllThreads();
        m_pJVMInterface->DetachCurrentThread();
    }


    return res;
}

TResult CJPIKernel::RunGC()
{
    if (!CJpiGlobals::Instance()->bJvmInitDone)
    {
        return MRTE_ERROR_PHASE_FAILURE;
    }

    if (!CJpiGlobals::Instance()->IsThreadMpiSafe())
    {
        return MRTE_ERROR_FAIL;
    }

    JNIEnv *pJniEnv;
    bool bAttached;
    if (MRTE_FAILED(m_pJVMInterface->AttachCurrentThread(&pJniEnv, &bAttached)))
    {
        return MRTE_ERROR_FAIL;
    }

    TResult res = m_pJVMInterface->RunGC();

    if (bAttached)
    {
        m_pJVMInterface->DetachCurrentThread();
    }

    return res;
}

TResult CJPIKernel::GenerateObjectAllocEventForLiveObjects(TId clientId)
{
    TResult res = m_pJVMInterface->IterateLiveHeap(ITER_HEAP_GEN_ALLOC_EVENT);
    return res;
}

TResult CJPIKernel::Configure(BitSet configuration)
{
    CJpiGlobals::Instance()->configFlags = configuration;
    return MRTE_RESULT_OK;
}
